/////////////////////////////////////////////////////
//
// BLGDefs.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    declaration of constants, structs, and classes
//    used throughout the Base Logic Generator's
//    source files
//
// Portability information:
//    the code in this file should be portable without
//    much trouble, although some constants may need to
//    be defined in order for the BLG to compile on
//    non-Windows platforms
//
/////////////////////////////////////////////////////

// make sure this header can't be included twice
#if !defined(BLG_DEFS_H_INCLUDED)
#define BLG_DEFS_H_INCLUDED

#if defined(WIN32)

// this is a Win32 build of the program

#include <windows.h>

// in Windows, the end of a line in a text file is indicated by a
// carriage return followed by a newline
#define STR_END_OF_LINE "\r\n"

#else // if !defined(WIN32)

// this is not a Win32 build of the program

// these typedefs are defined automatically by the windows header
// files; if this isn't a Windows build of the program, define
// them here
typedef long int LONG;
typedef unsigned long int ULONG;
typedef long long int LONGLONG;
typedef unsigned long long int ULONGLONG;
typedef short int SHORT;
typedef unsigned short int USHORT;
typedef unsigned char BYTE;
typedef long double LONGDOUBLE;

typedef int BOOL;
#define TRUE  1
#define FALSE 0

#if defined(NULL)
#undef NULL
#define NULL reinterpret_cast<void*>(0)
#endif // defined(NULL)

// in Unix, the end of a line is indicated by just a newline; if porting
// to an operating system that does things differently, use preprocessor
// directives to add another possible value for STR_END_OF_LINE; don't
// remove what's already here
#define STR_END_OF_LINE "\n"

#endif // defined(WIN32)

// there are 27 vars reserved for system use by the AGI interpreter
#define RESERVED_VAR_COUNT   27
// there are 16 flags reserved for system use by the AGI interpreter
#define RESERVED_FLAG_COUNT  16
// there's only 1 string reserved for system use by the AGI interpreter
#define RESERVED_STR_COUNT    1
// the constants technically aren't "reserved", but there might as well
// be some consistency in the names; there are 19 of them in the AGI Studio
// template file
#define RESERVED_CONST_COUNT 19
// there's only one reserved object in AGI: ego
#define RESERVED_OBJ_COUNT    1

// define the lengths of the longest default name for each data
// type; this is so we know how big to make the default name
// arrays that are defined in BLGDefs.cpp
#define MAX_RESERVED_VAR_NAME_LEN   23
#define MAX_RESERVED_FLAG_NAME_LEN  24
#define MAX_RESERVED_STR_NAME_LEN   11
#define MAX_RESERVED_CONST_NAME_LEN 12
#define MAX_RESERVED_OBJ_NAME_LEN    3

// this is the maximum length for user-specified define names
// in the base logic generator
#define MAX_DEFINE_NAME_LEN 100

// the following constants give names to the reserved var numbers
// that have some indication of the meaning of the reserved var
#define AGI_VAR_ROOM_NO                  0
#define AGI_VAR_PREV_ROOM_NO             1
#define AGI_VAR_EGO_EDGE_CODE            2
#define AGI_VAR_SCORE                    3
#define AGI_VAR_OBJECT_TOUCHING_EDGE     4
#define AGI_VAR_OBJECT_EDGE_CODE         5
#define AGI_VAR_EGO_DIR                  6
#define AGI_VAR_MAX_SCORE                7
#define AGI_VAR_FREE_MEMORY              8
#define AGI_VAR_UNKNOWN_WORD_NO          9
#define AGI_VAR_CYCLE_DELAY             10
#define AGI_VAR_CLOCK_SECONDS           11
#define AGI_VAR_CLOCK_MINUTES           12
#define AGI_VAR_CLOCK_HOURS             13
#define AGI_VAR_CLOCK_DAYS              14
#define AGI_VAR_JOYSTICK_SENSITIVITY    15
#define AGI_VAR_EGO_VIEW_NO             16
#define AGI_VAR_ERROR_CODE              17
#define AGI_VAR_ERROR_INFORMATION       18
#define AGI_VAR_KEY_PRESSED             19
#define AGI_VAR_COMPUTER_TYPE           20
#define AGI_VAR_WINDOW_CLOSE_TIME       21
#define AGI_VAR_SOUND_TYPE              22
#define AGI_VAR_SOUND_VOLUME            23
#define AGI_VAR_V24                     24
#define AGI_VAR_SELECTED_INVENTORY_ITEM 25
#define AGI_VAR_VIDEO_MODE              26

// the following constants give names to the reserved flag numbers
// that have some indication of the meaning of the reserved flag
#define AGI_FLAG_EGO_ON_WATER              0
#define AGI_FLAG_EGO_HIDDEN                1
#define AGI_FLAG_INPUT_RECEIVED            2
#define AGI_FLAG_EGO_TOUCHING_SIGNAL_LINE  3
#define AGI_FLAG_INPUT_PARSED              4
#define AGI_FLAG_NEW_ROOM                  5
#define AGI_FLAG_GAME_RESTARTED            6
#define AGI_FLAG_SCRIPT_BUFFER_BLOCKED     7
#define AGI_FLAG_JOYSTICK_SENSITIVITY_SET  8
#define AGI_FLAG_SOUND_ON                  9
#define AGI_FLAG_TRACE_ENABLED            10
#define AGI_FLAG_FIRST_LOGIC0_CYCLE       11
#define AGI_FLAG_GAME_RESTORED            12
#define AGI_FLAG_INVENTORY_SELECT_ENABLED 13
#define AGI_FLAG_MENU_ENABLED             14
#define AGI_FLAG_WINDOWS_REMAIN           15

// the AGI_CONST defines are indeces into the array of strings that
// contains the AGI Studio template game default names for these
// constants; they are not the values of the constants themselves
#define AGI_CONST_NO_EDGE       0
#define AGI_CONST_HORIZON_EDGE  1
#define AGI_CONST_RIGHT_EDGE    2
#define AGI_CONST_BOTTOM_EDGE   3
#define AGI_CONST_LEFT_EDGE     4
#define AGI_CONST_STOPPED       5
#define AGI_CONST_UP            6
#define AGI_CONST_UPRIGHT       7
#define AGI_CONST_RIGHT         8
#define AGI_CONST_DOWNRIGHT     9
#define AGI_CONST_DOWN         10
#define AGI_CONST_DOWNLEFT     11
#define AGI_CONST_LEFT         12
#define AGI_CONST_UPLEFT       13
#define AGI_CONST_CGA          14
#define AGI_CONST_RGB          15
#define AGI_CONST_HERCULES     16
#define AGI_CONST_EGA          17
#define AGI_CONST_VGA          18

// give a symbolic name to the object number of ego
#define AGI_OBJ_EGO 0

// the default horizon value for the AGI system is 36, but
// don't hard code that value into the BLG
#define DEFAULT_HORIZON_VALUE 36

// the maximum horizon value for the AGI system is 166; any
// higher than that and the game will lock up
#define MAX_HORIZON_VALUE 166


// use this define name with ego positioning controls to specify or test
// whether the previous room that ego was in should be ignored for that
// positioning control
#define ABSOLUTE_POSITIONING_SRC_ROOM -1

// MAX_DIGITS is the maximum number of digits allowed in an AGI
// number -- since they are ranged 0-255, the maximum is 3
#define MAX_DIGITS 3
// MAX_WORD_GROUP_DIGITS is the maximum number of digits that
// can be used to represent a word group; they are ranged
// 0-65535?? -- "rol" is group 9999, so the actual max. digits may
// be 4 but I can't find anything about it in the AGI Specs
#define MAX_WORD_GROUP_DIGITS 5

// NO_GOTO_ROOM indicates that no room was provided for an edge code control
#define NO_GOTO_ROOM -1

// MSG_NUM_UNASSIGNED indicates that the BLG's mini-compiler hasn't assigned
// a number to a particular message yet
#define MSG_NUM_UNASSIGNED -1

// NO_POS_X and NO_POS_Y are values used to indicate that the user has not
// specified an x or y position, wherever applicable -- essentially, this means
// that these fields should be ignored
#define NO_POS_X   -1
#define NO_POS_Y   -1

// the maximum value of any var in AGI is 255
#define AGI_MAX_VALUE 255

// the following constants give the minimum and maximum allowable positions
// that an object may occupy...these are passed to the AGI command "position"
// and others like it
#define MIN_POS_X   0
#define MAX_POS_X 159
#define MIN_POS_Y   0
#define MAX_POS_Y 167

// MAX_PRINT_MSG_LEN indicates the maximum number of characters that may
// appear in a message that is to be put into an AGI print statement in
// the generated files
#define MAX_PRINT_MSG_LEN 255

#define MAX_LOGIC_TITLE_LEN 255

/////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// CountUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    counts the number of unescaped quotes (quotes
//    without a backslash in front of them) in a
//    string
// Parameter psz:
//    a pointer to the string for which unescaped 
//    quotes should be counted; must not be NULL
// Return value:
//    the total number of unescaped quotes in the 
//    string
// Remarks:
//    this function goes through the entire string
//    and counts ALL the unescaped quotes; it does
//    this so that the caller can use this number
//    to determine if the string buffer is large
//    enough to hold the string if the unescaped
//    quotes were converted to escaped quotes
//
/////////////////////////////////////////////////////

int CountUnescapedQuotes(PSTR psz);

/////////////////////////////////////////////////////
//
// CountUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    counts the number of escaped quotes (quotes
//    with a backslash in front of them) in a
//    string
// Parameter psz:
//    a pointer to the string for which escaped 
//    quotes should be counted; must not be NULL
// Return value:
//    the total number of escaped quotes in the 
//    string
// Remarks:
//    this function goes through the entire string
//    and counts ALL the escaped quotes; it does
//    this to be consistent with the way that
//    CountUnescapedQuotes works
//
/////////////////////////////////////////////////////

int CountEscapedQuotes(PSTR psz);

/////////////////////////////////////////////////////
//
// RemoveUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    removes all unescaped quotes (quotes without
//    backslashes in front of them) from a string
// Parameter psz:
//    a pointer to the string from which to remove all
//    unescaped quotes; must not be NULL
// 
/////////////////////////////////////////////////////

void RemoveUnescapedQuotes(PSTR psz);

/////////////////////////////////////////////////////
//
// ConvertUnescapedQuotesToEscapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    inserts backslashes into a string as necessary
//    to make unescaped quotes (quotes without 
//    backslashes in front of them) into escaped quotes
// Parameter psz:
//    a pointer to the string for which unescaped quotes
//    will be converted to escaped quotes; must not be NULL
// Remarks:
//    before calling this function, the caller should
//    be sure that the buffer is large enough to hold
//    the entire string after conversion, because the
//    string will grow in length if backslashes must
//    be inserted
//
/////////////////////////////////////////////////////

void ConvertUnescapedQuotesToEscapedQuotes(PSTR psz);

/////////////////////////////////////////////////////
//
// ConvertEscapedQuotesToUnescapedQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    removes all escaped quotes (quotes with
//    backslashes in front of them) from a string by
//    removing all backslashes that occur immediately
//    before a quote
// Parameter psz:
//    a pointer to the string from which to remove all
//    escaped quotes; must not be NULL
// 
/////////////////////////////////////////////////////

void ConvertEscapedQuotesToUnescapedQuotes(PSTR psz);

/////////////////////////////////////////////////////
//
// struct EGOPOSITIONCONTROL
//
/////////////////////////////////////////////////////

typedef struct tagEGOPOSITIONCONTROL
{
    int nSrcRoom;
    int nPosX;
    int nPosY;
} EGOPOSITIONCONTROL, *PEGOPOSITIONCONTROL;

/////////////////////////////////////////////////////
//
// class COptions
//
/////////////////////////////////////////////////////

class COptions
{
public:
    COptions();
    virtual ~COptions();
    void BecomeCopy(COptions* poptions);

    int m_nHeaderCommentStyle;
    int m_nOneLineCommentStyle;
    int m_nTabsOrSpaces;
    int m_nIndenterCount;
    int m_nCurlyBraceStyle;
    int m_nUnescapedQuoteResponse;
    int m_nEscapedQuoteResponse;
    BOOL m_bGenerateHeader;
    BOOL m_bGenerateBasicComments;
    BOOL m_bGenerateVerboseComments;
    BOOL m_bGenerateTODOComments;
    BOOL m_bGenerateWarningComments;
    BOOL m_bUseDefineNames;
    BOOL m_bWriteToFile;
    BOOL m_bWriteToClipboard;
    // it may seem superfluous to have a generate source member when
    // its value can be deduced from the various "m_bWriteToWhatever"
    // members, but for modules that just use the options and don't set
    // them, having this member saves them the effort of having to deduce
    // whether a source file should be generated -- the program as a whole
    // is also more easily extensible this way
    BOOL m_bGenerateSource;
    BOOL m_bGenerateCompiledCode;
    BOOL m_bAllowEmptyCompiledCodeBlocks;
};

/////////////////////////////////////////////////////
//
// defines related to the COptions class
//
/////////////////////////////////////////////////////

// values for m_nHeaderCommentStyle
#define OPT_HCS_BOUND_BY_STRING_OF_SLASHES           0
#define OPT_HCS_BOUND_BY_MULTILINE                   1
#define OPT_HCS_BOUND_BY_DOUBLE_SLASH_THEN_ASTERISKS 2
#define OPT_HCS_START_WITH_DOUBLE_SLASH              3
#define OPT_HCS_START_WITH_LEFT_BRACKET              4
#define OPT_HCS_ENCLOSE_WITH_MULTILINE               5
#define OPT_HCS_ONE_LINE_HEADER                      6

// values for m_nOneLineCommentStyle
#define OPT_OLCS_DOUBLE_SLASH 0
#define OPT_OLCS_LEFT_BRACKET 1
#define OPT_OLCS_MULTILINE    2

// values for m_nTabsOrSpaces
#define OPT_TOS_SPACES 0
#define OPT_TOS_TABS   1

// values for m_nCurlyBraceStyle
#define OPT_CBS_NEXT_LINE   0
#define OPT_CBS_END_OF_LINE 1

// values for m_nUnescapedQuoteResponse
#define OPT_UQR_ALWAYS_INCLUDE_QUOTES 0
#define OPT_UQR_ALWAYS_REMOVE_QUOTES  1
#define OPT_UQR_ALWAYS_PROMPT         2

// values for m_nEscapedQuoteResponse
#define OPT_EQR_ALWAYS_REMOVE_BACKSLASH  0
#define OPT_EQR_ALWAYS_INCLUDE_BACKSLASH 1
#define OPT_EQR_ALWAYS_PROMPT            2

/////////////////////////////////////////////////////
//
// class CEdgeCodeInfo
//
/////////////////////////////////////////////////////

class CEdgeCodeInfo
{
public:
    CEdgeCodeInfo();
    virtual ~CEdgeCodeInfo();
    void BecomeCopy(CEdgeCodeInfo* pecinfo);

    int m_nLeftGotoRoom;
    int m_nRightGotoRoom;
    int m_nBottomGotoRoom;
    int m_nHorizonGotoRoom;
    BOOL m_bEmptyLeft;
    BOOL m_bEmptyRight;
    BOOL m_bEmptyBottom;
    BOOL m_bEmptyHorizon;
    char m_szLeftMessage[MAX_PRINT_MSG_LEN + 1];
    char m_szRightMessage[MAX_PRINT_MSG_LEN + 1];
    char m_szBottomMessage[MAX_PRINT_MSG_LEN + 1];
    char m_szHorizonMessage[MAX_PRINT_MSG_LEN + 1];
    int m_nLeftMessageNumber;
    int m_nRightMessageNumber;
    int m_nBottomMessageNumber;
    int m_nHorizonMessageNumber;
};

// the word "look" is in group 20 by default in the AGI Studio template game
#define DEFAULT_LOOK_WORD_GROUP 20

class CEntryLookInfo
{
public:
    CEntryLookInfo();
    virtual ~CEntryLookInfo();
    void BecomeCopy(CEntryLookInfo* pelinfo);

    char m_szEntryMessage[MAX_PRINT_MSG_LEN + 1];
    char m_szLookMessage[MAX_PRINT_MSG_LEN + 1];
    int m_nEntryMessageNumber;
    int m_nLookMessageNumber;
    int m_nLookWordGroup;
};

class CFirstRoomCtrl
{
public:
    CFirstRoomCtrl();
    virtual ~CFirstRoomCtrl();
    void BecomeCopy(CFirstRoomCtrl* pfrctrl);

    int m_nEgoX;
    int m_nEgoY;
    BOOL m_bStatusBarOn;
    BOOL m_bAcceptInput;
};

class CMainLogicOptions
{
public:
    CMainLogicOptions();
    virtual ~CMainLogicOptions();

    int m_nLogicNumber;
    BOOL m_bUseRoomNumForPic;
    int m_nPicNumber;
    int m_nHorizon;
    BOOL m_bDrawEgoInitially;
    BOOL m_bFirstRoom;
    char m_szLogicTitle[MAX_LOGIC_TITLE_LEN + 1];
};

#endif // !defined(BLG_DEFS_H_INCLUDED)