/////////////////////////////////////////////////////
//
// DefineNameList.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implementation of the CDefineNameList class
//
// Portability information:
//    the code in this file should be portable without
//    modification; if it is not, feel free to modify
//    it so that it is portable; do not add non-portable
//    code to this file
//
//    the constant _MAX_PATH will need to be defined
//    for ports of the Base Logic Generator; it should
//    be defined in BLGDefs.h
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cstring>
#include <cassert>
#include "DefineNameList.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in BLGDefs.cpp
extern char g_aszDefaultVarNames[RESERVED_VAR_COUNT]
                                [MAX_RESERVED_VAR_NAME_LEN + 1];
extern char g_aszDefaultFlagNames[RESERVED_FLAG_COUNT]
                                 [MAX_RESERVED_FLAG_NAME_LEN + 1];
extern char g_aszDefaultStrNames[RESERVED_STR_COUNT]
                                [MAX_RESERVED_STR_NAME_LEN + 1];
extern char g_aszDefaultConstNames[RESERVED_CONST_COUNT]
                                  [MAX_RESERVED_CONST_NAME_LEN + 1];
extern char g_aszDefaultObjNames[RESERVED_OBJ_COUNT]
                                [MAX_RESERVED_OBJ_NAME_LEN + 1];
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// constructor
//
/////////////////////////////////////////////////////
//
// Purpose:
//    constructs a CDefineNameList object
// Remarks:
//    when a CDefineNameList object is created, the
//    define names are initially set to the AGI Studio
//    template game's define names
//
/////////////////////////////////////////////////////

CDefineNameList::CDefineNameList()
{
    int i;

    for (i = 0; i < RESERVED_VAR_COUNT; i++)
    {
        strcpy(m_aszVarNames[i], g_aszDefaultVarNames[i]);
    }

    for (i = 0; i < RESERVED_FLAG_COUNT; i++)
    {
        strcpy(m_aszFlagNames[i], g_aszDefaultFlagNames[i]);
    }

    for (i = 0; i < RESERVED_STR_COUNT; i++)
    {
        strcpy(m_aszStrNames[i], g_aszDefaultStrNames[i]);
    }

    for (i = 0; i < RESERVED_CONST_COUNT; i++)
    {
        strcpy(m_aszConstNames[i], g_aszDefaultConstNames[i]);
    }

    for (i = 0; i < RESERVED_OBJ_COUNT; i++)
    {
        strcpy(m_aszObjNames[i], g_aszDefaultObjNames[i]);
    }

    strcpy(m_szDesiredPath, "");
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////
//
// Purpose:
//    performs any necessary cleanup work for a
//    CDefineNameList object before it is destroyed
//
/////////////////////////////////////////////////////

CDefineNameList::~CDefineNameList()
{
}

/////////////////////////////////////////////////////
//
// SetVarName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the define name for one of the AGI
//    reserved variables
// Parameter ulVarNum:
//    number of the reserved variable whose name
//    is to be set; must be less than RESERVED_VAR_COUNT,
//    as defined in BLGDefs.h
// Parameter pszNewName:
//    the new name for the variable; must not be
//    NULL and must not be longer than MAX_DEFINE_NAME_LEN
//    characters
//
/////////////////////////////////////////////////////

void CDefineNameList::SetVarName(ULONG ulVarNum, PCSTR pszNewName)
{
    assert(pszNewName != NULL);
    assert(ulVarNum < RESERVED_VAR_COUNT);
    assert(strlen(pszNewName) <= MAX_DEFINE_NAME_LEN);

    if (pszNewName[0] != '\0')
    {
        strncpy(m_aszVarNames[ulVarNum], pszNewName, 
                MAX_DEFINE_NAME_LEN + 1);
    }
    else
    {
        strcpy(m_aszVarNames[ulVarNum], g_aszDefaultVarNames[ulVarNum]);
    }
}

/////////////////////////////////////////////////////
//
// SetFlagName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the define name for one of the AGI
//    reserved flags
// Parameter ulFlagNum:
//    number of the reserved flag whose name
//    is to be set; must be less than RESERVED_FLAG_COUNT,
//    as defined in BLGDefs.h
// Parameter pszNewName:
//    the new name for the flag; must not be
//    NULL and must not be longer than MAX_DEFINE_NAME_LEN
//    characters
//
/////////////////////////////////////////////////////

void CDefineNameList::SetFlagName(ULONG ulFlagNum, PCSTR pszNewName)
{
    assert(pszNewName != NULL);
    assert(ulFlagNum < RESERVED_FLAG_COUNT);
    assert(strlen(pszNewName) <= MAX_DEFINE_NAME_LEN);

    if (pszNewName[0] != '\0')
    {
        strncpy(m_aszFlagNames[ulFlagNum], pszNewName, 
                MAX_DEFINE_NAME_LEN + 1);
    }
    else
    {
        strcpy(m_aszFlagNames[ulFlagNum], g_aszDefaultFlagNames[ulFlagNum]);
    }
}

/////////////////////////////////////////////////////
//
// SetStrName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the define name for one of the AGI
//    reserved strings
// Parameter ulStrNum:
//    number of the reserved string whose name
//    is to be set; must be less than RESERVED_STR_COUNT,
//    as defined in BLGDefs.h
// Parameter pszNewName:
//    the new name for the string; must not be
//    NULL and must not be longer than MAX_DEFINE_NAME_LEN
//    characters
//
/////////////////////////////////////////////////////

void CDefineNameList::SetStrName(ULONG ulStrNum, PCSTR pszNewName)
{
    assert(pszNewName != NULL);
    assert(ulStrNum < RESERVED_STR_COUNT);
    assert(strlen(pszNewName) <= MAX_DEFINE_NAME_LEN);

    if (pszNewName[0] != '\0')
    {
        strncpy(m_aszStrNames[ulStrNum], pszNewName, 
                MAX_DEFINE_NAME_LEN + 1);
    }
    else
    {
        strcpy(m_aszStrNames[ulStrNum], g_aszDefaultStrNames[ulStrNum]);
    }
}

/////////////////////////////////////////////////////
//
// SetConstName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the define name for one of the constants
//    that is meaningful to the AGI interpreter
// Parameter ulConstID:
//    number of the constant whose name is to be set; 
//    must be less than RESERVED_CONST_COUNT, as defined 
//    in BLGDefs.h; the AGI_CONST... defines in 
//    BLGDefs.h give the numbers corresponding
//    to each AGI constant
// Parameter pszNewName:
//    the new name for the constant; must not be
//    NULL and must not be longer than MAX_DEFINE_NAME_LEN
//    characters
//
/////////////////////////////////////////////////////

void CDefineNameList::SetConstName(ULONG ulConstID, PCSTR pszNewName)
{
    assert(pszNewName != NULL);
    assert(ulConstID < RESERVED_CONST_COUNT);
    assert(strlen(pszNewName) <= MAX_DEFINE_NAME_LEN);

    if (pszNewName[0] != '\0')
    {
        strncpy(m_aszConstNames[ulConstID], pszNewName, 
                MAX_DEFINE_NAME_LEN + 1);
    }
    else
    {
        strcpy(m_aszConstNames[ulConstID], g_aszDefaultConstNames[ulConstID]);
    }
}

/////////////////////////////////////////////////////
//
// SetObjName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the define name for one of the AGI
//    reserved objects
// Parameter ulObjNum:
//    number of the reserved object whose name
//    is to be set; must be less than RESERVED_OBJ_COUNT,
//    as defined in BLGDefs.h
// Parameter pszNewName:
//    the new name for the object; must not be
//    NULL and must not be longer than MAX_DEFINE_NAME_LEN
//    characters
//
/////////////////////////////////////////////////////

void CDefineNameList::SetObjName(ULONG ulObjNum, PCSTR pszNewName)
{
    assert(pszNewName != NULL);
    assert(ulObjNum < RESERVED_OBJ_COUNT);
    assert(strlen(pszNewName) <= MAX_DEFINE_NAME_LEN);

    if (pszNewName[0] != '\0')
    {
        strncpy(m_aszObjNames[ulObjNum], pszNewName,
                MAX_DEFINE_NAME_LEN + 1);
    }
    else
    {
        strcpy(m_aszObjNames[ulObjNum], g_aszDefaultObjNames[ulObjNum]);
    }
}

/////////////////////////////////////////////////////
//
// GetVarName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the define name for one of the AGI
//    reserved variables
// Parameter ulVarNum:
//    number of the reserved variable whose name
//    is to be retrieved; must be less than RESERVED_VAR_COUNT,
//    as defined in BLGDefs.h
// Parameter psz:
//    a pointer to a buffer that will receive the name
//    of the variable; must not be NULL and must be
//    large enough to hold the entire define name
// Parameter nMaxChars:
//    the maximum number of characters of the define
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetVarName(ULONG ulVarNum, PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    assert(ulVarNum < RESERVED_VAR_COUNT);
    strncpy(psz, m_aszVarNames[ulVarNum], nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetFlagName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the define name for one of the AGI
//    reserved flags
// Parameter ulFlagNum:
//    number of the reserved flag whose name
//    is to be retrieved; must be less than RESERVED_FLAG_COUNT,
//    as defined in BLGDefs.h
// Parameter psz:
//    a pointer to a buffer that will receive the name
//    of the flag; must not be NULL and must be
//    large enough to hold the entire define name
// Parameter nMaxChars:
//    the maximum number of characters of the define
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetFlagName(ULONG ulFlagNum, PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    assert(ulFlagNum < RESERVED_FLAG_COUNT);
    strncpy(psz, m_aszFlagNames[ulFlagNum], nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetStrName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the define name for one of the AGI
//    reserved strings
// Parameter ulStrNum:
//    number of the reserved string whose name
//    is to be retrieved; must be less than RESERVED_STR_COUNT,
//    as defined in BLGDefs.h
// Parameter psz:
//    a pointer to a buffer that will receive the name
//    of the string; must not be NULL and must be
//    large enough to hold the entire define name
// Parameter nMaxChars:
//    the maximum number of characters of the define
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetStrName(ULONG ulStrNum, PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    assert(ulStrNum < RESERVED_STR_COUNT);
    strncpy(psz, m_aszStrNames[ulStrNum], nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetConstName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the define name for one of the constants
//    that is meaningful to the AGI interpreter
// Parameter ulConstID:
//    number of the constant whose name is to be retrieved; 
//    must be less than RESERVED_CONST_COUNT, as defined 
//    in BLGDefs.h; the AGI_CONST... defines in 
//    BLGDefs.h give the numbers corresponding
//    to each AGI constant
// Parameter psz:
//    a pointer to a buffer that will receive the name
//    of the constant; must not be NULL and must be
//    large enough to hold the entire define name
// Parameter nMaxChars:
//    the maximum number of characters of the define
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetConstName(ULONG ulConstID, PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    assert(ulConstID < RESERVED_CONST_COUNT);
    strncpy(psz, m_aszConstNames[ulConstID], nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetObjName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the define name for one of the AGI
//    reserved objects
// Parameter ulObjNum:
//    number of the reserved object whose name
//    is to be retrieved; must be less than RESERVED_OBJ_COUNT,
//    as defined in BLGDefs.h
// Parameter psz:
//    a pointer to a buffer that will receive the name
//    of the obj; must not be NULL and must be
//    large enough to hold the entire define name
// Parameter nMaxChars:
//    the maximum number of characters of the define
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetObjName(ULONG ulObjNum, PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    assert(ulObjNum < RESERVED_OBJ_COUNT);
    strncpy(psz, m_aszObjNames[ulObjNum], nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetVarNameLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the define name currently
//    assigned to one of the AGI variables
// Parameter ulVarNum:
//    number of the reserved variable whose name length
//    is to be retrieved; must be less than RESERVED_VAR_COUNT,
//    as defined in BLGDefs.h
// Return value:
//    the number of characters in the variable's define
//    name, not including the null character
//
/////////////////////////////////////////////////////

int CDefineNameList::GetVarNameLength(ULONG ulVarNum)
{
    assert(ulVarNum < RESERVED_VAR_COUNT);
    return strlen(m_aszVarNames[ulVarNum]);
}

/////////////////////////////////////////////////////
//
// GetFlagNameLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the define name currently
//    assigned to one of the AGI flags
// Parameter ulFlagNum:
//    number of the reserved flag whose name length
//    is to be retrieved; must be less than RESERVED_FLAG_COUNT,
//    as defined in BLGDefs.h
// Return value:
//    the number of characters in the flag's define
//    name, not including the null character
//
/////////////////////////////////////////////////////


int CDefineNameList::GetFlagNameLength(ULONG ulFlagNum)
{
    assert(ulFlagNum < RESERVED_FLAG_COUNT);
    return strlen(m_aszFlagNames[ulFlagNum]);
}

/////////////////////////////////////////////////////
//
// GetStrNameLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the define name currently
//    assigned to one of the AGI strings
// Parameter ulStrNum:
//    number of the reserved string whose name length
//    is to be retrieved; must be less than RESERVED_STR_COUNT,
//    as defined in BLGDefs.h
// Return value:
//    the number of characters in the string's define
//    name, not including the null character
//
/////////////////////////////////////////////////////

int CDefineNameList::GetStrNameLength(ULONG ulStrNum)
{
    assert(ulStrNum < RESERVED_STR_COUNT);
    return strlen(m_aszStrNames[ulStrNum]);
}

/////////////////////////////////////////////////////
//
// GetConstNameLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the define name currently
//    assigned to one of the AGI constants
// Parameter ulConstID:
//    number of the constant whose name length is to be 
//    retrieved; must be less than RESERVED_CONST_COUNT, 
//    as defined in BLGDefs.h; the AGI_CONST... defines in 
//    BLGDefs.h give the numbers corresponding
//    to each AGI constant
// Return value:
//    the number of characters in the constant's define
//    name, not including the null character
//
/////////////////////////////////////////////////////

int CDefineNameList::GetConstNameLength(ULONG ulConstID)
{
    assert(ulConstID < RESERVED_CONST_COUNT);
    return strlen(m_aszConstNames[ulConstID]);
}

/////////////////////////////////////////////////////
//
// GetObjNameLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the define name currently
//    assigned to one of the AGI objects
// Parameter ulStrNum:
//    number of the reserved object whose name length
//    is to be retrieved; must be less than RESERVED_OBJ_COUNT,
//    as defined in BLGDefs.h
// Return value:
//    the number of characters in the object's define
//    name, not including the null character
//
/////////////////////////////////////////////////////

int CDefineNameList::GetObjNameLength(ULONG ulObjNum)
{
    assert(ulObjNum < RESERVED_OBJ_COUNT);
    return strlen(m_aszObjNames[ulObjNum]);
}

/////////////////////////////////////////////////////
//
// SetDesiredPath
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the path where this define name object
//    would PREFER to be saved; this value might be
//    ignored
// Parameter pszDesiredPath:
//    a string containing the desired path; must not
//    be NULL and must be less than _MAX_PATH
//
/////////////////////////////////////////////////////

void CDefineNameList::SetDesiredPath(PCSTR pszDesiredPath)
{
    assert(pszDesiredPath != NULL);
    assert(strlen(pszDesiredPath) <= _MAX_PATH);

    strncpy(m_szDesiredPath, pszDesiredPath, _MAX_PATH + 1);
}

/////////////////////////////////////////////////////
//
// GetDesiredPath
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the path where this define name object
//    would prefer to be saved
// Parameter psz:
//    a pointer to a string buffer where the path
//    string will go; must not be NULL and must be
//    big enough to hold the path string
// Parameter nMaxChars:
//    the maximum number of characters of the path
//    name that are to be retrieved, including the
//    null character
//
/////////////////////////////////////////////////////

void CDefineNameList::GetDesiredPath(PSTR psz, int nMaxChars)
{
    assert(psz != NULL);
    strncpy(psz, m_szDesiredPath, nMaxChars);
}

/////////////////////////////////////////////////////
//
// GetDesiredPathLength
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the length of the path where this 
//    define name object would prefer to be saved
// Return value:
//    the number of characters in the path string,
//    not including the null character
//
/////////////////////////////////////////////////////

int CDefineNameList::GetDesiredPathLength()
{
    return strlen(m_szDesiredPath);
}

/////////////////////////////////////////////////////
//
// BecomeCopy
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes this CDefineNameList object into a copy of
//    another CDefineNameList object; similar to a copy
//    constructor, except it's not a constructor
// Parameter deflistSrc:
//    the CDefineNameList object that this CDefineNameList
//    object is to become a copy of; this object is not
//    changed
//
/////////////////////////////////////////////////////

void CDefineNameList::BecomeCopy(CDefineNameList& deflistSrc)
{
    char szBuffer[MAX_DEFINE_NAME_LEN + 1];
    ULONG ul;

    for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
    {
        deflistSrc.GetVarName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
        SetVarName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
    {
        deflistSrc.GetFlagName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
        SetFlagName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
    {
        deflistSrc.GetStrName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
        SetStrName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
    {
        deflistSrc.GetConstName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
        SetConstName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
    {
        deflistSrc.GetObjName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
        SetObjName(ul, szBuffer);
    }

}

/////////////////////////////////////////////////////
//
// SaveToDesiredPath
//
/////////////////////////////////////////////////////
//
// Purpose:
//    instructs this CDefineNameList object to 
//    save a representation of itself to its
//    desired path
// Return value:
//    TRUE if the file is saved successfully;
//    FALSE if not
//
/////////////////////////////////////////////////////

BOOL CDefineNameList::SaveToDesiredPath()
{
    ofstream file;

    if (m_szDesiredPath[0] == '\0')
    {
        return FALSE;
    }

    file.open(m_szDesiredPath);

    if (!file)
    {
        return FALSE;
    }

    file << "AGIBLG DefineNameList 2.0" << endl;

    ULONG ul;

    for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
    {
        file << m_aszVarNames[ul] << endl;
    }

    for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
    {
        file << m_aszFlagNames[ul] << endl;
    }

    for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
    {
        file << m_aszStrNames[ul] << endl;
    }

    for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
    {
        file << m_aszConstNames[ul] << endl;
    }

    for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
    {
        file << m_aszObjNames[ul] << endl;
    }

    file.close();

    return TRUE;
}

/////////////////////////////////////////////////////
//
// LoadFromDesiredPath
//
/////////////////////////////////////////////////////
//
// Purpose:
//    instructs this CDefineNameList object to load
//    its data from its desired path
// Return value:
//    TRUE if the file is successfully loaded;
//    otherwise FALSE
//
/////////////////////////////////////////////////////

BOOL CDefineNameList::LoadFromDesiredPath()
{
    ifstream file;

    if (m_szDesiredPath[0] == '\0')
    {
        return FALSE;
    }

    file.open(m_szDesiredPath);

    if (!file)
    {
        return FALSE;
    }

    char szLoadBuffer[MAX_DEFINE_NAME_LEN + 1];
    PSTR pszTemp = NULL;

    file.getline(szLoadBuffer, MAX_DEFINE_NAME_LEN + 1);

    pszTemp = strtok(szLoadBuffer, " \r\n");

    if (!pszTemp || (strcmp(pszTemp, "AGIBLG") != 0))
    {
        file.close();
        return FALSE;
    }
    else
    {
        pszTemp = strtok(NULL, " \r\n");

        if (!pszTemp || (strcmp(pszTemp, "DefineNameList") != 0))
        {
            file.close();
            return FALSE;
        }
        else
        {
            pszTemp = strtok(NULL, " \r\n");

            if (!pszTemp)
            {
                file.close();
                return FALSE;
            }
            else if (strcmp(pszTemp, "2.0") == 0)
            {
                Load2Pt0DefineNameList(file);
                // it's possible for the version specific
                // load functions to return FALSE which means 
                // that, at best, a partial load of the
                // file was successful -- this is
                // ok; just salvage what's salvageable
                // and leave the rest as it was
            }
            else
            {
                file.close();
                return FALSE;
            }
        }
    }

    file.close();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// Load
//
/////////////////////////////////////////////////////
//
// Purpose:
//    instructs this CDefineNameList object to load
//    its data from a file
// Parameter pszPath:
//    the path of the file from which this CDefineNameList
//    object should load its data; must not be NULL
// Return value:
//    TRUE if the data is loaded successfully;
//    otherwise FALSE
//
/////////////////////////////////////////////////////

BOOL CDefineNameList::Load(PSTR pszPath)
{
    assert(pszPath != NULL);

    ifstream file;

    if (pszPath[0] == '\0')
    {
        return FALSE;
    }

    file.open(pszPath);

    if (!file)
    {
        return FALSE;
    }

    char szLoadBuffer[MAX_DEFINE_NAME_LEN + 1];
    PSTR pszTemp = NULL;

    file.getline(szLoadBuffer, MAX_DEFINE_NAME_LEN + 1);

    pszTemp = strtok(szLoadBuffer, " \r\n");

    if (!pszTemp || (strcmp(pszTemp, "AGIBLG") != 0))
    {
        file.close();
        return FALSE;
    }
    else
    {
        pszTemp = strtok(NULL, " \r\n");

        if (!pszTemp || (strcmp(pszTemp, "DefineNameList") != 0))
        {
            file.close();
            return FALSE;
        }
        else
        {
            pszTemp = strtok(NULL, " \r\n");

            if (!pszTemp)
            {
                file.close();
                return FALSE;
            }
            else if (strcmp(pszTemp, "2.0") == 0)
            {
                Load2Pt0DefineNameList(file);
                // it's possible for the version specific
                // load functions to return FALSE which means 
                // that, at best, a partial load of the
                // file was successful -- this is
                // ok; just salvage what's salvageable
                // and leave the rest as it was
            }
            else
            {
                file.close();
                return FALSE;
            }
        }
    }

    file.close();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// Load2Pt0DefineNameList
//
/////////////////////////////////////////////////////
//
// Purpose:
//    loads a version 2.0 define name list into this
//    CDefineNameList object; if changes to the BLG
//    change the format of the define name list file,
//    then a new version should be defined and another
//    member function should carry out loading it; this
//    function should not be modified in a significant
//    way, so that old define name lists can continue
//    to be loaded even if a new format is defined
// Parameter file:
//    an already open ifstream file from which to load
//    the resource
// Return value:
//    TRUE if the entire file is read successfully;
//    FALSE if the operation fails at any point; note
//    that even if the operation fails, the function
//    salvages as much of the file as it can
//
/////////////////////////////////////////////////////

BOOL CDefineNameList::Load2Pt0DefineNameList(ifstream& file)
{
    ULONG ul;
    char szDefineNameBuffer[MAX_DEFINE_NAME_LEN + 1];

    for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
    {
        file.getline(szDefineNameBuffer, MAX_DEFINE_NAME_LEN + 1);

        if (IsValidDefineName(szDefineNameBuffer))
        {
            SetVarName(ul, szDefineNameBuffer);
        }
        else
        {
            return FALSE;
        }
    }

    for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
    {
        file.getline(szDefineNameBuffer, MAX_DEFINE_NAME_LEN + 1);

        if (IsValidDefineName(szDefineNameBuffer))
        {
            SetFlagName(ul, szDefineNameBuffer);
        }
        else
        {
            return FALSE;
        }
    }

    for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
    {
        file.getline(szDefineNameBuffer, MAX_DEFINE_NAME_LEN + 1);

        if (IsValidDefineName(szDefineNameBuffer))
        {
            SetStrName(ul, szDefineNameBuffer);
        }
        else
        {
            return FALSE;
        }
    }

    for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
    {
        file.getline(szDefineNameBuffer, MAX_DEFINE_NAME_LEN + 1);

        if (IsValidDefineName(szDefineNameBuffer))
        {
            SetConstName(ul, szDefineNameBuffer);
        }
        else
        {
            return FALSE;
        }
    }

    for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
    {
        file.getline(szDefineNameBuffer, MAX_DEFINE_NAME_LEN + 1);

        if (IsValidDefineName(szDefineNameBuffer))
        {
            SetObjName(ul, szDefineNameBuffer);
        }
        else
        {
            return FALSE;
        }
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// IsValidDefineName
//
/////////////////////////////////////////////////////
//
// Purpose:
//    determines if a given string contains a valid
//    AGI define name; a valid AGI define name
//    contains letters, numbers, and the characters
//    '.' and '_'
// Parameter pszDefName:
//    a pointer to the string to be tested; must
//    not be NULL
// Return value:
//    TRUE if the string is a valid AGI define name;
//    FALSE if not
// Remarks:
//    this member function is static; it can be called
//    even if there is no current instance of the
//    CDefineNameList class available
//
/////////////////////////////////////////////////////

BOOL CDefineNameList::IsValidDefineName(PSTR pszDefName)
{
    assert(pszDefName != NULL);

    if (pszDefName[0] == '\0')
    {
        return FALSE;
    }

    for (;;)
    {
        if (pszDefName[0] == '\0')
        {
            return TRUE;
        }
        else if ((pszDefName[0] < 'a' || pszDefName[0] > 'z') && 
            (pszDefName[0] < 'A' || pszDefName[0] > 'Z') &&
            (pszDefName[0] < '0' || pszDefName[0] > '9') &&
            pszDefName[0] != '_' && pszDefName[0] != '.')
        {
            return FALSE;
        }

        pszDefName++;
    }
}
