/////////////////////////////////////////////////////
//
// EntryLookDlgUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    Entry and Looking dialog; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written specifically for
//    a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include "resource.h"
#include "BLGDefs.h"
#include "MessageBoxes.h"
#include "EntryLookDlgUI.h"

/////////////////////////////////////////////////////
//
// globals
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in EntryLookDlgEventHandlers.cpp
extern HWND g_hwndEntryLookDlg;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in EntryLookDlg.cpp
extern CEntryLookInfo g_elinfoTemp;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// OnEntryLookDlgOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles event for the Entry and Looking dialog's
//    OK button (validates entries, etc.)
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEntryLookDlgOK()
{
    char szBuffer[MAX_WORD_GROUP_DIGITS + 1];
    HWND hwndEdit;
    int nUnescapedQuoteCount;
    int nResponse;

    // get the user's entry from the Entry Text edit box
    hwndEdit = GetDlgItem(g_hwndEntryLookDlg, IDC_EDIT_ENTRY_TEXT);
    GetWindowText(hwndEdit, g_elinfoTemp.m_szEntryMessage,
                  MAX_PRINT_MSG_LEN + 1);

    // find out if there are any unescaped quotes in the user's entry
    nUnescapedQuoteCount = CountUnescapedQuotes(g_elinfoTemp.m_szEntryMessage);

    if (nUnescapedQuoteCount != 0)
    {
        // if there are unescaped quotes, ask the user what should be done
        // about them -- (note: this call will not actually display a message
        // box if the user has specified not to be asked again about this
        // issue)
        nResponse = UnescapedQuoteMessageBox(g_hwndEntryLookDlg,
                                             "Room entry print message");

        switch(nResponse)
        {
        case UQMB_INCLUDE_QUOTES:
            // the user has chosen to include the unescaped quotes in the
            // logic -- this means that they must be converted to escaped
            // quotes, otherwise weird problems may occur
            if ((nUnescapedQuoteCount + 
                strlen(g_elinfoTemp.m_szEntryMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEntryLookDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndEdit);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_elinfoTemp.m_szEntryMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            // user has chosen to remove the unescaped quotes from the string;
            // this probably means that they were intended to be string
            // containers instead of literal quotes
            RemoveUnescapedQuotes(g_elinfoTemp.m_szEntryMessage);
            break;
        case UQMB_USER_EDIT:
            // user has chosen to edit their entry to decide what they will do
            // with the unescaped quotes; in this case, cancel the processing
            // of the OK button
            SetFocus(hwndEdit);
            return TRUE;
        }
    }

    // set any modifications to the entry text immediately so that the user
    // doesn't have to repeat the previous step if they cancel at a later
    // point
    SetWindowText(hwndEdit, g_elinfoTemp.m_szEntryMessage);

    // get the user's entry from the Look Text edit box
    hwndEdit = GetDlgItem(g_hwndEntryLookDlg, IDC_EDIT_LOOK_TEXT);
    GetWindowText(hwndEdit, g_elinfoTemp.m_szLookMessage,
                  MAX_PRINT_MSG_LEN + 1);

    // find out if there are any unescaped quotes in the user's entry
    nUnescapedQuoteCount = CountUnescapedQuotes(g_elinfoTemp.m_szLookMessage);

    if (nUnescapedQuoteCount != 0)
    {
        // if there are unescaped quotes, ask the user what should be done
        // about them -- (note: this call will not actually display a message
        // box if the user has specified not to be asked again about this
        // issue)
        nResponse = UnescapedQuoteMessageBox(g_hwndEntryLookDlg,
                                             "\"Look\" response");

        switch(nResponse)
        {
        case UQMB_INCLUDE_QUOTES:
            // the user has chosen to include the unescaped quotes in the
            // logic -- this means that they must be converted to escaped
            // quotes, otherwise weird problems may occur
            if ((nUnescapedQuoteCount + 
                strlen(g_elinfoTemp.m_szLookMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEntryLookDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndEdit);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_elinfoTemp.m_szLookMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            // user has chosen to remove the unescaped quotes from the string;
            // this probably means that they were intended to be string
            // containers instead of literal quotes
            RemoveUnescapedQuotes(g_elinfoTemp.m_szLookMessage);
            break;
        case UQMB_USER_EDIT:
            // user has chosen to edit their entry to decide what they will do
            // with the unescaped quotes; in this case, cancel the processing
            // of the OK button
            SetFocus(hwndEdit);
            return TRUE;
        }
    }

    // set any modifications to the look text immediately so that the user
    // doesn't have to repeat the previous step if they cancel at a later
    // point
    SetWindowText(hwndEdit, g_elinfoTemp.m_szLookMessage);


    // get the user's Look Group entry -- this must be validated as
    // an integer between 0 and 65535
    hwndEdit = GetDlgItem(g_hwndEntryLookDlg, IDC_EDIT_LOOK_GROUP);
    GetWindowText(hwndEdit, szBuffer, 
                  MAX_WORD_GROUP_DIGITS + 1);

    if (szBuffer[0] == '\0')
    {
        if (g_elinfoTemp.m_szLookMessage[0] != '\0')
        {
            // if look text has been specified, then the look word group
            // can't be left blank (otherwise, there's no way to generate
            // a compiled logic)
            nResponse = MessageBox(g_hwndEntryLookDlg,
                                   "If you specify look text, you must also "
                                   "specify a word group for the word "
                                   "\"look\".\n\n Do you want to use the "
                                   "default word group?",
                                   "Need group number", MB_YESNO);

            if (nResponse == IDYES)
            {
                g_elinfoTemp.m_nLookWordGroup = DEFAULT_LOOK_WORD_GROUP;
            }
            else
            {
                SetFocus(hwndEdit);
                return TRUE;
            }
        }

        // don't change the word group from what it was if there's no
        // look text, in case the user accidently deleted it
    }
    else
    {
        g_elinfoTemp.m_nLookWordGroup = atoi(szBuffer);

        if (g_elinfoTemp.m_nLookWordGroup < 0 ||
            g_elinfoTemp.m_nLookWordGroup > 65535)
        {
            MessageBox(g_hwndEntryLookDlg,
                       "The WORDS.TOK group for \"look\" must be between "
                       "0 and 65535 (inclusive).", "Error", MB_OK);
            SetFocus(hwndEdit);
            return TRUE;
        }
    }

    // reached the end with no problems, close the dialog down and return
    // the code IDOK to indicate that the OK button was clicked
    EndDialog(g_hwndEntryLookDlg, IDOK);
    return TRUE;
}