/////////////////////////////////////////////////////
//
// GeneralUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides helper functions to reduce the tedious
//    nature of many common Windows tasks
//
// Portability information:
//    this file contains code written specifically
//    for a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "GeneralUI.h"

/////////////////////////////////////////////////////
//
// SetCheckboxChecked
//
/////////////////////////////////////////////////////
//
// Purpose:
//    checks or unchecks checkbox controls
// Parameter hDlg:
//    the window handle of the checkbox's PARENT
// Parameter nCtrlID:
//    the dialog control id of the checkbox
// Parameter bChecked:
//    specifies whether the checkbox should be set to
//    the checked or the unchecked state -- defaults
//    to TRUE if not provided
//
/////////////////////////////////////////////////////

void SetCheckboxChecked(HWND hDlg, int nCtrlID, BOOL bChecked /* = TRUE */)
{
	SendMessage(GetDlgItem(hDlg, nCtrlID), BM_SETCHECK, 
              static_cast<WPARAM>(bChecked), 0);
}

/////////////////////////////////////////////////////
//
// IsCheckboxChecked
//
/////////////////////////////////////////////////////
//
// Purpose:
//    queries a checkbox and reports whether it is
//    currently checked or unchecked
// Parameter hDlg:
//    the window handle of the checkbox's PARENT
// Parameter nCtrlID:
//    the dialog control id of the checkbox
// Return value:
//    TRUE if the checkbox is currently checked;
//    FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL IsCheckboxChecked(HWND hDlg, int nCtrlID)
{
    return SendMessage(GetDlgItem(hDlg, nCtrlID), BM_GETCHECK, 0, 0);
}

/////////////////////////////////////////////////////
//
// SetRadioButton
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the current selection in a group of radio
//    buttons
// Parameter hDlg:
//    the window handle of the radio buttons' PARENT
// Parameter nCtrlIDButtonToSet:
//    the dialog control id of the radio button which
//    should be selected from the group
// Parameter nFirstCtrlID:
//    the control id of the first radio button in the
//    group of radio buttons
// Parameter nLastCtrlID:
//    the control id of the last radio button in the
//    group of radio buttons
// Remarks:
//    in order for this function to work, the radio
//    buttons in the group must have sequential control
//    ids, and the value of nLastCtrlID must be no
//    smaller than the value of nFirstCtrlID; this
//    function selects one out of a group of radio
//    buttons and deselects all other radio buttons
//    in the group
// 
/////////////////////////////////////////////////////

void SetRadioButton(HWND hDlg, int nCtrlIDButtonToSet,
                    int nFirstCtrlID, int nLastCtrlID)
{
    assert(nLastCtrlID >= nFirstCtrlID);

    for (int i = nFirstCtrlID; i <= nLastCtrlID; i++)
    {
        SendMessage(GetDlgItem(hDlg, nCtrlIDButtonToSet), BM_SETCHECK, 
                    static_cast<WPARAM>(FALSE), 0);
    }

    SendMessage(GetDlgItem(hDlg, nCtrlIDButtonToSet), BM_SETCHECK, 
                static_cast<WPARAM>(TRUE), 0);
}

/////////////////////////////////////////////////////
//
// IsRadioButtonSelected
//
/////////////////////////////////////////////////////
//
// Purpose:
//    queries a particular radio button to determine
//    whether it is selected
// Parameter hDlg:
//    the window handle of the radio buttons' PARENT
// Parameter nCtrlIDButtonToCheck:
//    the dialog control id of the radio button that
//    is to be queried
// Return value:
//    TRUE if the radio button is selected; FALSE
//    otherwise
//
/////////////////////////////////////////////////////

BOOL IsRadioButtonSelected(HWND hDlg, int nCtrlIDButtonToCheck)
{
    return IsDlgButtonChecked(hDlg, nCtrlIDButtonToCheck);
}

/////////////////////////////////////////////////////
//
// ListBoxAddString
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds a string to a list box (encapsulates the
//    LB_ADDSTRING message)
// Parameter hDlg:
//    the window handle of the list box's PARENT
// Parameter nCtrlID:
//    the dialog control id of the list box
// Parameter pszStrToAdd:
//    a pointer to a null-terminated string to add
//    to the list box; must not be NULL
//
/////////////////////////////////////////////////////

void ListBoxAddString(HWND hDlg, int nCtrlID, PCSTR pszStrToAdd)
{
    assert(pszStrToAdd != NULL);

    SendMessage(GetDlgItem(hDlg, nCtrlID), LB_ADDSTRING,
                0, reinterpret_cast<LPARAM>(pszStrToAdd));
}

/////////////////////////////////////////////////////
//
// ComboBoxAddString
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds a string to a combo box (encapsulates the
//    CB_ADDSTRING message)
// Parameter hDlg:
//    the window handle of the combo box's PARENT
// Parameter nCtrlID:
//    the dialog control id of the combo box
// Parameter pszStrToAdd:
//    a pointer to a null-terminated string to add
//    to the combo box; must not be NULL
//
/////////////////////////////////////////////////////

void ComboBoxAddString(HWND hDlg, int nCtrlID, PCSTR pszStrToAdd)
{
    ComboBoxAddString(GetDlgItem(hDlg, nCtrlID), pszStrToAdd);
}

/////////////////////////////////////////////////////
//
// ComboBoxAddString
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds a string to a combo box (encapsulates the
//    CB_ADDSTRING message)
// Parameter hwndCombo:
//    the window handle of the combo box itself
// Parameter pszStrToAdd:
//    a pointer to a null-terminated string to add
//    to the combo box; must not be NULL
//
/////////////////////////////////////////////////////

void ComboBoxAddString(HWND hwndCombo, PCSTR pszStrToAdd)
{
    assert(pszStrToAdd != NULL);

    SendMessage(hwndCombo, CB_ADDSTRING, 0,
                reinterpret_cast<LPARAM>(pszStrToAdd));
}

/////////////////////////////////////////////////////
//
// ComboBoxSetCurSel
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the current selection for a combo box
//    (encapsulates the CB_SETCURSEL message)
// Parameter hDlg:
//    the window handle of the combo box's PARENT
// Parameter nCtrlID:
//    the dialog control id of the combo box
// Parameter nIndex:
//    the 0-based index of the item which will become
//    the combo box's current selection
//
/////////////////////////////////////////////////////

void ComboBoxSetCurSel(HWND hDlg, int nCtrlID, int nIndex)
{
    ComboBoxSetCurSel(GetDlgItem(hDlg, nCtrlID), nIndex);
}

/////////////////////////////////////////////////////
//
// ComboBoxSetCurSel
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the current selection for a combo box
//    (encapsulates the CB_SETCURSEL message)
// Parameter hwndCombo:
//    the window handle of the combo box itself
// Parameter nIndex:
//    the 0-based index of the item which will become
//    the combo box's current selection
//
/////////////////////////////////////////////////////

void ComboBoxSetCurSel(HWND hwndCombo, int nIndex)
{
    SendMessage(hwndCombo, CB_SETCURSEL, static_cast<WPARAM>(nIndex), 0);
}

/////////////////////////////////////////////////////
//
// ComboBoxGetCurSel
//
/////////////////////////////////////////////////////
//
// Purpose:
//    gets the current selection for a combo box
//    (encapsulates the CB_GETCURSEL message)
// Parameter hDlg:
//    the window handle of the combo box's PARENT
// Parameter nCtrlID:
//    the dialog control id of the combo box
// Return value:
//    the 0-based index of the currently selected
//    item in the combo box, or CB_ERR if no item
//    is selected
//
/////////////////////////////////////////////////////

int ComboBoxGetCurSel(HWND hDlg, int nCtrlID)
{
    return ComboBoxGetCurSel(GetDlgItem(hDlg, nCtrlID));
}

/////////////////////////////////////////////////////
//
// ComboBoxGetCurSel
//
/////////////////////////////////////////////////////
//
// Purpose:
//    gets the current selection for a combo box
//    (encapsulates the CB_GETCURSEL message)
// Parameter hwndCombo:
//    the window handle of the combo box itself
// Return value:
//    the 0-based index of the currently selected
//    item in the combo box, or CB_ERR if no item
//    is selected
//
/////////////////////////////////////////////////////

int ComboBoxGetCurSel(HWND hwndCombo)
{
    return SendMessage(hwndCombo, CB_GETCURSEL, 0, 0);
}

/////////////////////////////////////////////////////
//
// QueryRadioButton
//
/////////////////////////////////////////////////////
//
// Purpose:
//    determines which radio button in a group is
//    selected
// Parameter hDlg:
//    the window handle of the radio buttons' PARENT
// Parameter nFirstCtrlID:
//    the control id of the first radio button in the
//    group of radio buttons
// Parameter nLastCtrlID:
//    the control id of the last radio button in the
//    group of radio buttons
// Return value:
//    the dialog control id of the currently selected
//    radio button in the group -- 0 if none of the
//    radio buttons is selected
// Remarks:
//    in order for this function to work, the radio
//    buttons in the group must have sequential control
//    ids, and the value of nLastCtrlID must be no
//    smaller than the value of nFirstCtrlID;
//
/////////////////////////////////////////////////////

int QueryRadioButton(HWND hDlg, int nFirstCtrlID, int nLastCtrlID)
{
    for (int i = nFirstCtrlID; i <= nLastCtrlID; i++)
    {
        if (IsDlgButtonChecked(hDlg, i))
        {
            return i;
        }
    }

    return 0;
}

/////////////////////////////////////////////////////
//
// CenterDialogToParent
//
/////////////////////////////////////////////////////
//
// Purpose:
//    centers a dialog box relative to its parent
//    window
// Parameter hDlg:
//    the window handle of the dialog to be centered
// Remarks:
//    technically, this function will work for any
//    child window, not just dialogs
//
/////////////////////////////////////////////////////

void CenterDialogToParent(HWND hDlg)
{
    int nParentX;
    int nParentY;
    int nParentWidth;
    int nParentHeight;
    int nDlgWidth;
    int nDlgHeight;
	RECT rect;

	GetWindowRect(GetParent(hDlg), &rect);

	nParentX = rect.left;
	nParentY = rect.top;
	nParentWidth = rect.right - rect.left;
	nParentHeight = rect.bottom - rect.top;

	GetWindowRect(hDlg, &rect);

	nDlgWidth = rect.right - rect.left;
	nDlgHeight = rect.bottom - rect.top;

	MoveWindow(hDlg, nParentX + (nParentWidth / 2) - (nDlgWidth / 2), 
               nParentY + (nParentHeight / 2) - (nDlgHeight / 2), 
               nDlgWidth, nDlgHeight, TRUE);
}

/////////////////////////////////////////////////////
//
// CenterWindowToScreen
//
/////////////////////////////////////////////////////
//
// Purpose:
//    centers a window relative to the screen
// Parameter hwnd:
//    the window handle of the window to be centered
//
/////////////////////////////////////////////////////

void CenterWindowToScreen(HWND hwnd)
{
    HDC hdc;
    int nScreenWidth;
    int nScreenHeight;
    int nWndWidth;
    int nWndHeight;
    RECT rect;

    // get the current screen resolution
    hdc = CreateDC("DISPLAY", NULL, NULL, NULL);
    nScreenWidth = GetDeviceCaps(hdc, HORZRES);
    nScreenHeight = GetDeviceCaps(hdc, VERTRES);
    DeleteDC(hdc);

    GetWindowRect(hwnd, &rect);

    nWndWidth = rect.right - rect.left;
    nWndHeight = rect.bottom - rect.top;

    MoveWindow(hwnd, (nScreenWidth / 2) - (nWndWidth / 2),
               (nScreenHeight / 2) - (nWndHeight / 2),
               nWndWidth, nWndHeight, TRUE);
}