/////////////////////////////////////////////////////
//
// LogicGeneration.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    contains functions that generate a compiled logic
//    file and write it to disk
//
// Portability information:
//    the code in this file should be easily portable
//    to other platforms; however, this file does
//    make use of the LOBYTE and HIBYTE macros that
//    are defined in the Windows header files; ports
//    of this code will need to provide their own
//    definitions of LOBYTE and HIBYTE
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include <cstdio>
#include "BLGDefs.h"
#include "EgoPositionControlArray.h"
#include "DefineNameList.h"
#include "SimpleStack.h"
#include "LogicGeneration.h"

/////////////////////////////////////////////////////
//
// defines
//
/////////////////////////////////////////////////////

// the number of characters in the Avis Durgan string
#define AVIS_DURGAN_LEN 11

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in MainDlgUI.cpp
extern CMainLogicOptions g_mlopt;
extern CEntryLookInfo g_elinfo;
extern CFirstRoomCtrl g_frctrl;
extern CEgoPositionControlArray g_aepc;
extern CEdgeCodeInfo g_ecinfo;
extern CDefineNameList g_deflist;
extern COptions g_options;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in BLGDefs.cpp
extern int g_anAGIConstValues[RESERVED_CONST_COUNT];
/////////////////////////////////////////////////////

// the number of the next string in the message section
// of the logic
int g_nNextMessageNum = 1;
// the current position in the Avis Durgan string -- this
// is needed only because I misunderstood how the Avis Durgan
// string was used...I originally thought that each message
// was individually xor'ed against Avis Durgan, but I later
// realized that the message section as a whole was xor'ed
// against Avis Durgan...rather than rewrite the code, I just
// kept track of the current Avis position in a global variable
int g_nAvisPos;

// the Avis Durgan string -- used for weak encryption of the
// message section of the compiled logic
char g_szAvisDurgan[] = "Avis Durgan";

/////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////

// functions used for compiled logic generation
int NewMessageNumber(int& n);
void EncryptString(PSTR psz);
void ResetAvisPos();

/////////////////////////////////////////////////////
//
// GenerateCompiledLogic
//
/////////////////////////////////////////////////////
//
// Purpose:
//    the main function that should be called upon to
//    generate a compiled AGI logic file -- takes care
//    of the details of where the logic should be written
//    based on the options that have already been set
// Parameter pszGameDir:
//    the directory where the file should be written;
//    must not be NULL
//
/////////////////////////////////////////////////////

void GenerateCompiledLogic(PSTR pszGameDir)
{
    // REMEMBER: byte ordering is flipped in AGI

    // first two bytes = offset of message section - 2
    // logic data appears
    // message section:
    //    number of messages (1 byte)
    //    pointer to end of messages (offset from previous byte) (2 bytes)
    //    list of pointers to first byte of each null terminated msg (2 bytes)

    // this is the compiled logic file itself, stored in an array of bytes
    BYTE abyLogicData[MAX_LOGIC_DATA_SIZE];
    // a pointer to the current position in the logic data array
    BYTE* pbyCurPos;
    BYTE* pbyTemp;
    // counts the bytes at the current nesting level so that this information
    // can be reported at the beginning of an if-statement and other places
    // where it is needed
    USHORT usBytesAtCurNestLevel = 0;
    // the total number of bytes in the compiled logic file
    int nTotalBytes = 0;
    // the number of message in the logic file
    BYTE byMsgCount = 0;
    int i;

    // this stack is needed for nested if-statements
    CSimpleStack<USHORT> stackByteCount(100);
    // this stack is needed to store the positions in the logic file
    // where information must be filled out that is not known when these
    // bytes are reached during compilation
    CSimpleStack<BYTE*> stackPlaceholder(100);

    assert(pszGameDir != NULL);

    // make sure the next msg number is reset for each compile, or screwy
    // things will happen
    g_nNextMessageNum = 1;

    // the first two bytes are filled out last, so skip 'em for now
    pbyCurPos = abyLogicData + 2;

    /////////////////////////////////////////////////////
    //
    // generate code for if (new_room)
    //
    /////////////////////////////////////////////////////

    // begin if (new_room)
    pbyCurPos[0] = LOGIC_CMD_START_IF;
    pbyCurPos++;

    // generate isset(new_room)
    pbyCurPos[0] = LOGIC_CMD_ISSET;
    pbyCurPos[1] = AGI_FLAG_NEW_ROOM;

    // advance past the bytes that were just added
    pbyCurPos += 2;
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    // generate the bytes indicating the end of the if test
    pbyCurPos[0] = LOGIC_CMD_END_IF;
    pbyCurPos++;

    // count the if-statement delimiter bytes
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    // entering an if-statement, save the byte count at the previous
    // nesting level and reset the byte count at this level
    stackByteCount.Push(usBytesAtCurNestLevel);
    usBytesAtCurNestLevel = 0;

    // save the position where the byte count for this if-statement
    // must be filled in, then skip past those two bytes
    stackPlaceholder.Push(pbyCurPos);
    pbyCurPos += 2;

    // inside if (new_room)

    if (!g_mlopt.m_bUseRoomNumForPic)
    {
        // if the user specified not to use the room number for the pic
        // resource, load the picture number into a temporary variable
        // (note: this should probably be a variable of the user's
        // choosing)

        // generate v255 = specified pic number;
        pbyCurPos[0] = LOGIC_CMD_ASSIGNN;
        pbyCurPos[1] = 255;
        pbyCurPos[2] = static_cast<BYTE>(g_mlopt.m_nPicNumber);

        // skip past the bytes that were just added
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;
    }
    
    // generate load.pic command
    pbyCurPos[0] = LOGIC_CMD_LOAD_PIC;
    if (g_mlopt.m_bUseRoomNumForPic)
    {
        // generate load.pic(room_no);
        pbyCurPos[1] = AGI_VAR_ROOM_NO;
    }
    else
    {
        // generate load.pic(v255);
        pbyCurPos[1] = 255;
    }

    // skip past the generated bytes
    pbyCurPos += 2;
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    // generate draw.pic command
    pbyCurPos[0] = LOGIC_CMD_DRAW_PIC;
    if (g_mlopt.m_bUseRoomNumForPic)
    {
        // generate draw.pic(room_no);
        pbyCurPos[1] = AGI_VAR_ROOM_NO;
    }
    else
    {
        // generate draw.pic(v255);
        pbyCurPos[1] = 255;
    }

    // skip past the generated bytes
    pbyCurPos += 2;
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    // generate discard.pic command
    pbyCurPos[0] = LOGIC_CMD_DISCARD_PIC;
    if (g_mlopt.m_bUseRoomNumForPic)
    {
        // generate discard.pic(room_no);
        pbyCurPos[1] = AGI_VAR_ROOM_NO;
    }
    else
    {
        // generate discard.pic(v255);
        pbyCurPos[1] = 255;
    }

    // skip past the generated bytes
    pbyCurPos += 2;
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    if (g_mlopt.m_nHorizon != DEFAULT_HORIZON_VALUE)
    {
        // if the horizon value is not the default 36, then
        // generate the set.horizon command
        pbyCurPos[0] = LOGIC_CMD_SET_HORIZON;
        pbyCurPos[1] = g_mlopt.m_nHorizon;

        // skip past the generated bytes
        pbyCurPos += 2;
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }


    /////////////////////////////////////////////////////
    //
    // handle first room case
    //
    /////////////////////////////////////////////////////

    if (g_mlopt.m_bFirstRoom)
    {
        // check to make sure that code relevant to the first room
        // actually needs to be generated in the compiled file
        if ((g_frctrl.m_nEgoX != NO_POS_X && g_frctrl.m_nEgoY != NO_POS_Y) ||
            g_frctrl.m_bAcceptInput ||
            g_frctrl.m_bStatusBarOn ||
            g_options.m_bAllowEmptyCompiledCodeBlocks)
        {
            // generate if ((prev_room_no == 0 || prev_room_no == 1))
            pbyCurPos[0] = LOGIC_CMD_START_IF;
            pbyCurPos++;

            pbyCurPos[0] = LOGIC_CMD_START_OR;
            pbyCurPos++;
    
            // generate prev_room_no == 0
            pbyCurPos[0] = LOGIC_CMD_EQUALN;
            pbyCurPos[1] = AGI_VAR_PREV_ROOM_NO;
            pbyCurPos[2] = 0;

            // skip past prev_room_no == 0
            pbyCurPos += 3;
            usBytesAtCurNestLevel += 3;
            nTotalBytes += 3;

            // generate prev_room_no == 1
            pbyCurPos[0] = LOGIC_CMD_EQUALN;
            pbyCurPos[1] = AGI_VAR_PREV_ROOM_NO;
            pbyCurPos[2] = 1;

            // skip past prev_room_no == 1
            pbyCurPos += 3;
            usBytesAtCurNestLevel += 3;
            nTotalBytes += 3;

            pbyCurPos[0] = LOGIC_CMD_END_OR;
            pbyCurPos++;

            // count the or delimiter bytes
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // end the if condition
            pbyCurPos[0] = LOGIC_CMD_END_IF;
            pbyCurPos++;

            // count the if delimiter bytes
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // entering an if-statement, save the byte count at the previous
            // nesting level and reset the byte count at this level
            stackByteCount.Push(usBytesAtCurNestLevel);
            usBytesAtCurNestLevel = 0;

            // save the position where the byte count for this if-statement
            // must be filled in, then skip past those two bytes
            stackPlaceholder.Push(pbyCurPos);
            pbyCurPos += 2;

            // check to see if a position command needs to be
            // generated
            if (g_frctrl.m_nEgoX != NO_POS_X &&
                g_frctrl.m_nEgoY != NO_POS_Y)
            {
                // generate position command
                pbyCurPos[0] = LOGIC_CMD_POSITION;
                pbyCurPos[1] = AGI_OBJ_EGO;
                pbyCurPos[2] = static_cast<BYTE>(g_frctrl.m_nEgoX);
                pbyCurPos[3] = static_cast<BYTE>(g_frctrl.m_nEgoY);

                // skip past the position command
                pbyCurPos += 4;
                usBytesAtCurNestLevel += 4;
                nTotalBytes += 4;
            }

            if (g_frctrl.m_bStatusBarOn)
            {
                // generate status.line.on command and
                // advance pointers past it
                pbyCurPos[0] = LOGIC_CMD_STATUS_LINE_ON;
                pbyCurPos++;
                usBytesAtCurNestLevel++;
                nTotalBytes++;
            }

            if (g_frctrl.m_bAcceptInput)
            {
                // generate accept.input command and
                // advance pointers past it
                pbyCurPos[0] = LOGIC_CMD_ACCEPT_INPUT;
                pbyCurPos++;
                usBytesAtCurNestLevel++;
                nTotalBytes++;
            }

            // TODO: BLG should generate a warning if code block contains
            //       0 bytes -- these code blocks will not import into
            //       AGI Studio properly (this problem may be corrected
            //       in v.1.36, but v.1.35 and earlier AGI Studio can't
            //       handle 0 length command blocks) -- if a source file
            //       is going to be generated, the empty block may be able
            //       to be generated in the source file but not in the
            //       compiled file, but if no source file is going to be
            //       generated then this is a serious problem

            // fill in the byte count for: 
            // if (prev_room_no == 0 || prev_room_no == 1)
            pbyTemp = stackPlaceholder.Pop();
            pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
            pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

            // restore the byte count from the previous nesting
            // level (and include the byte count from the if-statement
            // that was just generated)
            usBytesAtCurNestLevel += stackByteCount.Pop();
            // count the bracket offset for the last if statement
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }
    }

    /////////////////////////////////////////////////////
    //
    // handle ego positioning
    //
    /////////////////////////////////////////////////////

    // find out how many ego positioning commands there need
    // to be, including the absolute positioning command
    int nNumPosCommands = g_aepc.GetElementCount() +
        ((g_aepc.GetAbsolutePositioningItem() != NULL) ? 1 : 0);

    // need to store the number of commands and decrement it,
    // so make another var
    int nRemainingPosCommands = nNumPosCommands;
    EGOPOSITIONCONTROL epc;

    // generate the code for all the previous-room-dependent
    // positioning commands
    for (i = 0; i < g_aepc.GetElementCount(); i++)
    {
        epc = g_aepc.GetAt(i);

        // generate the if statement for this block
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // generate prev_room_no == epc.nSrcRoom
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_PREV_ROOM_NO;
        pbyCurPos[2] = static_cast<BYTE>(epc.nSrcRoom);

        // skip past prev_room_no == epc.nSrcRoom
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // end the if test
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiter bytes
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering a new nesting level (see above for details)
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        // generate position command
        pbyCurPos[0] = LOGIC_CMD_POSITION;
        pbyCurPos[1] = AGI_OBJ_EGO;
        pbyCurPos[2] = static_cast<BYTE>(epc.nPosX);
        pbyCurPos[3] = static_cast<BYTE>(epc.nPosY);

        // skip past position command
        pbyCurPos += 4;
        usBytesAtCurNestLevel += 4;
        nTotalBytes += 4;

        pbyTemp = stackPlaceholder.Pop();

        // figure out if there's an else branch
        nRemainingPosCommands--;

        if (nRemainingPosCommands > 0)
        {
            // there's going to be an else branch for this if statement,
            // which means an extra three bytes
            usBytesAtCurNestLevel += 3;
            nTotalBytes += 3;
        }

        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        // count the bracket offset for the last if statement
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        if (nRemainingPosCommands > 0)
        {
            // start generating the else
            pbyCurPos[0] = LOGIC_CMD_ELSE;
            pbyCurPos++;

            // IMPORTANT: 
            // don't count the else command or the else's 2 offset
            // bytes; they've already been counted

            stackByteCount.Push(usBytesAtCurNestLevel);
            usBytesAtCurNestLevel = 0;
            
            stackPlaceholder.Push(pbyCurPos);
            pbyCurPos += 2;
        }
    }

    // now generate the absolute positioning item, if it exists
    if (g_aepc.GetAbsolutePositioningItem() != NULL)
    {
        epc = *(g_aepc.GetAbsolutePositioningItem());

        // generate the position command
        pbyCurPos[0] = LOGIC_CMD_POSITION;
        pbyCurPos[1] = AGI_OBJ_EGO;
        pbyCurPos[2] = static_cast<BYTE>(epc.nPosX);
        pbyCurPos[3] = static_cast<BYTE>(epc.nPosY);

        // skip past the position command
        pbyCurPos += 4;
        usBytesAtCurNestLevel += 4;
        nTotalBytes += 4;

    }

    // now go back through and make sure all the else branches have
    // their byte counts correct (again, DO NOT count the bytes for 
    // these things; they've already been counted)
    for (i = 0; i < (nNumPosCommands - 1); i++)
    {
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
    }

    /////////////////////////////////////////////////////
    //
    // do the rest of the new_room section
    //
    /////////////////////////////////////////////////////

    if (g_mlopt.m_bDrawEgoInitially)
    {
        // if ego gets drawn initially, generate draw(ego);
        pbyCurPos[0] = LOGIC_CMD_DRAW;
        pbyCurPos[1] = AGI_OBJ_EGO;

        // skip past draw(ego);
        pbyCurPos += 2;
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    // generate show.pic(); and skip past it
    pbyCurPos[0] = LOGIC_CMD_SHOW_PIC;
    pbyCurPos++;
    usBytesAtCurNestLevel++;
    nTotalBytes++;

    /////////////////////////////////////////////////////
    //
    // handle the room entry message
    //
    /////////////////////////////////////////////////////

    if (g_elinfo.m_szEntryMessage[0] != '\0')
    {
        byMsgCount++;

        // assign a number to this message
        NewMessageNumber(g_elinfo.m_nEntryMessageNumber);

        // generate print(m1); or whatever the number is
        pbyCurPos[0] = LOGIC_CMD_PRINT;
        pbyCurPos[1] = static_cast<BYTE>(g_elinfo.m_nEntryMessageNumber);

        // skip past print() command
        pbyCurPos += 2;
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    // end if (new_room)

    // now fill out the byte count that's required at the beginning of
    // an if block
    pbyTemp = stackPlaceholder.Pop();

    pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
    pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

    usBytesAtCurNestLevel += stackByteCount.Pop();
    // count the two offset bytes for the if (new_room) statement
    usBytesAtCurNestLevel += 2;
    nTotalBytes += 2;

    /////////////////////////////////////////////////////
    //
    // handle left edge code
    //
    /////////////////////////////////////////////////////

    // determine if code needs to be generated
    if (g_ecinfo.m_nLeftGotoRoom != NO_GOTO_ROOM ||
        g_ecinfo.m_szLeftMessage[0] != '\0' ||
        (g_ecinfo.m_bEmptyLeft && g_options.m_bAllowEmptyCompiledCodeBlocks))
    {
        // generate the if-statement for this block
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // generate ego_edge_code == left_edge
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_EGO_EDGE_CODE;
        pbyCurPos[2] = 
            static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_LEFT_EDGE]);

        // skip past ego_edge_code == left_edge
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // end the if test
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiter bytes
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering if block
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        // if there's a message to be printed, generate the code for it
        if (g_ecinfo.m_szLeftMessage[0] != '\0')
        {
            byMsgCount++;

            NewMessageNumber(g_ecinfo.m_nLeftMessageNumber);

            // generate print command
            pbyCurPos[0] = LOGIC_CMD_PRINT;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nLeftMessageNumber);

            // skip past print command
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // if ego's not going to a new room, then stop its motion
            // so that the left message doesn't keep printing forever
            // and ever
            if (g_ecinfo.m_nLeftGotoRoom == NO_GOTO_ROOM)
            {
                // generate ego_dir = stopped
                pbyCurPos[0] = LOGIC_CMD_ASSIGNN;
                pbyCurPos[1] = AGI_VAR_EGO_DIR;
                pbyCurPos[2] = 
                    static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_STOPPED]);

                // skip past ego_dir = stopped
                pbyCurPos += 3;
                usBytesAtCurNestLevel += 3;
                nTotalBytes += 3;
            }
        }

        // if ego is going to a new room, generate the command
        if (g_ecinfo.m_nLeftGotoRoom != NO_GOTO_ROOM)
        {
            // generate new.room
            pbyCurPos[0] = LOGIC_CMD_NEW_ROOM;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nLeftGotoRoom);

            // skip past new.room
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }

        // leaving an if-statement
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    /////////////////////////////////////////////////////
    //
    // handle right edge code
    //
    /////////////////////////////////////////////////////

    // determine if code needs to be generated
    if (g_ecinfo.m_nRightGotoRoom != NO_GOTO_ROOM ||
        g_ecinfo.m_szRightMessage[0] != '\0' ||
        (g_ecinfo.m_bEmptyRight && g_options.m_bAllowEmptyCompiledCodeBlocks))
    {
        // generate the if-statement for this block
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // generate ego_edge_code == right_edge
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_EGO_EDGE_CODE;
        pbyCurPos[2] = 
            static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_RIGHT_EDGE]);

        // skip past ego_edge_code == right_edge
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // end the if test
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiter bytes
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering an if block
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        // if there's a right edge message, generate the code to print it
        if (g_ecinfo.m_szRightMessage[0] != '\0')
        {
            byMsgCount++;

            NewMessageNumber(g_ecinfo.m_nRightMessageNumber);

            // generate print command
            pbyCurPos[0] = LOGIC_CMD_PRINT;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nRightMessageNumber);

            // skip past print command
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // if ego's not going to a new room, stop its motion so the
            // right edge message doesn't keep printing over and over
            if (g_ecinfo.m_nRightGotoRoom == NO_GOTO_ROOM)
            {
                // generate ego_dir = stopped
                pbyCurPos[0] = LOGIC_CMD_ASSIGNN;
                pbyCurPos[1] = AGI_VAR_EGO_DIR;
                pbyCurPos[2] = 
                    static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_STOPPED]);

                // skip past ego_dir = stopped
                pbyCurPos += 3;
                usBytesAtCurNestLevel += 3;
                nTotalBytes += 3;
            }
        }

        // if ego goes to a new room, generate the code
        if (g_ecinfo.m_nRightGotoRoom != NO_GOTO_ROOM)
        {
            // generate new.room
            pbyCurPos[0] = LOGIC_CMD_NEW_ROOM;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nRightGotoRoom);

            // skip past new.room
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }

        // leaving an if-statement
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    /////////////////////////////////////////////////////
    //
    // handle bottom edge code
    //
    /////////////////////////////////////////////////////

    // check to see if code needs to be generated
    if (g_ecinfo.m_nBottomGotoRoom != NO_GOTO_ROOM ||
        g_ecinfo.m_szBottomMessage[0] != '\0' ||
        (g_ecinfo.m_bEmptyBottom && 
         g_options.m_bAllowEmptyCompiledCodeBlocks))
    {
        // start the if test code
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // generate ego_edge_code == bottom_edge
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_EGO_EDGE_CODE;
        pbyCurPos[2] = 
            static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_BOTTOM_EDGE]);

        // skip past ego_edge_code == bottom_edge
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // end the if test
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiters
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering if statement
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        // if there's a bottom edge message, make it print
        if (g_ecinfo.m_szBottomMessage[0] != '\0')
        {
            byMsgCount++;

            NewMessageNumber(g_ecinfo.m_nBottomMessageNumber);

            // generate print command
            pbyCurPos[0] = LOGIC_CMD_PRINT;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nBottomMessageNumber);

            // skip print command
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // stop ego's motion if not going to new room so that bottom
            // message doesn't repeat
            if (g_ecinfo.m_nBottomGotoRoom == NO_GOTO_ROOM)
            {
                // generate ego_dir = stopped
                pbyCurPos[0] = LOGIC_CMD_ASSIGNN;
                pbyCurPos[1] = AGI_VAR_EGO_DIR;
                pbyCurPos[2] = 
                    static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_STOPPED]);

                // skip past ego_dir = stopped
                pbyCurPos += 3;
                usBytesAtCurNestLevel += 3;
                nTotalBytes += 3;
            }
        }

        // if ego goes to a new room, make it happen
        if (g_ecinfo.m_nBottomGotoRoom != NO_GOTO_ROOM)
        {
            // generate new.room
            pbyCurPos[0] = LOGIC_CMD_NEW_ROOM;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nBottomGotoRoom);

            // skip past new.room
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }

        // leaving an if block
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    /////////////////////////////////////////////////////
    //
    // handle horizon edge code
    //
    /////////////////////////////////////////////////////

    // see if code needs to be generated
    if (g_ecinfo.m_nHorizonGotoRoom != NO_GOTO_ROOM ||
        g_ecinfo.m_szHorizonMessage[0] != '\0' ||
        (g_ecinfo.m_bEmptyHorizon && 
         g_options.m_bAllowEmptyCompiledCodeBlocks))
    {
        // start if test
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // generate ego_edge_code == horizon_edge
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_EGO_EDGE_CODE;
        pbyCurPos[2] = 
            static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_HORIZON_EDGE]);

        // skip past ego_edge_code == horizon_edge
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // end if test
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiters
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering if block
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        // if there is a horizon message, make it print
        if (g_ecinfo.m_szHorizonMessage[0] != '\0')
        {
            byMsgCount++;

            NewMessageNumber(g_ecinfo.m_nHorizonMessageNumber);

            // generate print command
            pbyCurPos[0] = LOGIC_CMD_PRINT;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nHorizonMessageNumber);

            // skip past print command
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // if ego isn't going to a new room, stop its motion so the
            // horizon message doesn't repeat
            if (g_ecinfo.m_nHorizonGotoRoom == NO_GOTO_ROOM)
            {
                // generate ego_dir = stopped
                pbyCurPos[0] = LOGIC_CMD_ASSIGNN;
                pbyCurPos[1] = AGI_VAR_EGO_DIR;
                pbyCurPos[2] = 
                    static_cast<BYTE>(g_anAGIConstValues[AGI_CONST_STOPPED]);

                // skip past ego_dir = stopped
                pbyCurPos += 3;
                usBytesAtCurNestLevel += 3;
                nTotalBytes += 3;
            }
        }

        // if ego goes to a new room, make it happen
        if (g_ecinfo.m_nHorizonGotoRoom != NO_GOTO_ROOM)
        {
            // generate new.room
            pbyCurPos[0] = LOGIC_CMD_NEW_ROOM;
            pbyCurPos[1] = static_cast<BYTE>(g_ecinfo.m_nHorizonGotoRoom);

            // skip past new.room
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }

        // leaving if block
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;
    }

    /////////////////////////////////////////////////////
    //
    // begin input parsing block
    //
    /////////////////////////////////////////////////////

    // make sure an input parsing block is needed
    if (g_elinfo.m_szLookMessage[0] != '\0' ||
        g_options.m_bAllowEmptyCompiledCodeBlocks)
    {
        // if (
        pbyCurPos[0] = LOGIC_CMD_START_IF;
        pbyCurPos++;

        // input_received
        pbyCurPos[0] = LOGIC_CMD_ISSET;
        pbyCurPos[1] = AGI_FLAG_INPUT_RECEIVED;
        pbyCurPos += 2;
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // && !input_parsed
        pbyCurPos[0] = LOGIC_CMD_NOT;
        pbyCurPos[1] = LOGIC_CMD_ISSET;
        pbyCurPos[2] = AGI_FLAG_INPUT_PARSED;
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // && unknown_word_no == 0
        pbyCurPos[0] = LOGIC_CMD_EQUALN;
        pbyCurPos[1] = AGI_VAR_UNKNOWN_WORD_NO;
        pbyCurPos[2] = 0;
        pbyCurPos += 3;
        usBytesAtCurNestLevel += 3;
        nTotalBytes += 3;

        // )
        pbyCurPos[0] = LOGIC_CMD_END_IF;
        pbyCurPos++;

        // count the if delimiters
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

        // entering if block
        stackByteCount.Push(usBytesAtCurNestLevel);
        usBytesAtCurNestLevel = 0;

        stackPlaceholder.Push(pbyCurPos);
        pbyCurPos += 2;

        if (g_elinfo.m_szLookMessage[0] != '\0')
        {
            // if (
            pbyCurPos[0] = LOGIC_CMD_START_IF;
            pbyCurPos++;
        
            // said("look")
            pbyCurPos[0] = LOGIC_CMD_SAID;
            // the number of 2-byte word group values used as
            // arguments for said()
            pbyCurPos[1] = 1;
            pbyCurPos[2] = LOBYTE(static_cast<USHORT>(g_elinfo.m_nLookWordGroup));
            pbyCurPos[3] = HIBYTE(static_cast<USHORT>(g_elinfo.m_nLookWordGroup));

            pbyCurPos += 4;
            usBytesAtCurNestLevel += 4;
            nTotalBytes += 4;

            // )
            pbyCurPos[0] = LOGIC_CMD_END_IF;
            pbyCurPos++;

            // count the if delimiters
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // entering if block
            stackByteCount.Push(usBytesAtCurNestLevel);
            usBytesAtCurNestLevel = 0;

            stackPlaceholder.Push(pbyCurPos);
            pbyCurPos += 2;

            byMsgCount++;

            NewMessageNumber(g_elinfo.m_nLookMessageNumber);

            // generate print command
            pbyCurPos[0] = LOGIC_CMD_PRINT;
            pbyCurPos[1] = static_cast<BYTE>(g_elinfo.m_nLookMessageNumber);

            // skip print command
            pbyCurPos += 2;
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;

            // leaving if block
            pbyTemp = stackPlaceholder.Pop();
            pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
            pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

            usBytesAtCurNestLevel += stackByteCount.Pop();
            usBytesAtCurNestLevel += 2;
            nTotalBytes += 2;
        }

        // leaving if block
        pbyTemp = stackPlaceholder.Pop();
        pbyTemp[0] = LOBYTE(usBytesAtCurNestLevel);
        pbyTemp[1] = HIBYTE(usBytesAtCurNestLevel);

        usBytesAtCurNestLevel += stackByteCount.Pop();
        usBytesAtCurNestLevel += 2;
        nTotalBytes += 2;

    }

    // generate return command and skip past it
    pbyCurPos[0] = LOGIC_CMD_RETURN;
    pbyCurPos++;
    usBytesAtCurNestLevel++;
    nTotalBytes++;

    // go back and set offset to message section (first two
    // bytes of the logic resource)
    abyLogicData[0] = LOBYTE(usBytesAtCurNestLevel);
    abyLogicData[1] = HIBYTE(usBytesAtCurNestLevel);
    nTotalBytes += 2;

    /////////////////////////////////////////////////////
    //
    // create message section
    //
    /////////////////////////////////////////////////////

    // store the number of messages in this resource
    pbyCurPos[0] = byMsgCount;
    pbyCurPos++;
    nTotalBytes++;

    USHORT usBytesToEndOfMessages;
    int nStrLen;
    char szBuffer[MAX_PRINT_MSG_LEN + 1];

    // the number of bytes to the end of the messages is always at least 2
    usBytesToEndOfMessages = 2;

    // save where the number of bytes to the end of the message section
    // is stored
    stackPlaceholder.Push(pbyCurPos);
    pbyCurPos += 2;
    nTotalBytes += 2;

    pbyCurPos += (2 * byMsgCount);
    nTotalBytes += (2 * byMsgCount);
    usBytesToEndOfMessages += (2 * byMsgCount);

    // the entire message section (not each individual string) is encrypted
    // with Avis Durgan every 11 bytes; because each individual string
    // is encrypted with the EncryptString function and the current position
    // in the Avis Durgan string is stored in a global variable, it needs
    // to be reset for each compile
    ResetAvisPos();

    // add the Room Entry message to the logic
    if (g_elinfo.m_nEntryMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_elinfo.m_szEntryMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_elinfo.m_nEntryMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // add the Left Edge message to the logic
    if (g_ecinfo.m_nLeftMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_ecinfo.m_szLeftMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_ecinfo.m_nLeftMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // add the Right Edge message to the logic
    if (g_ecinfo.m_nRightMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_ecinfo.m_szRightMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_ecinfo.m_nRightMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // add the Bottom Edge message to the logic
    if (g_ecinfo.m_nBottomMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_ecinfo.m_szBottomMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_ecinfo.m_nBottomMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // add the Horizon Edge message to the logic
    if (g_ecinfo.m_nHorizonMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_ecinfo.m_szHorizonMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_ecinfo.m_nHorizonMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // add the Look message to the logic
    if (g_elinfo.m_nLookMessageNumber != MSG_NUM_UNASSIGNED)
    {
        strcpy(szBuffer, g_elinfo.m_szLookMessage);
        ConvertEscapedQuotesToUnescapedQuotes(szBuffer);
        nStrLen = strlen(szBuffer) + 1;
        EncryptString(szBuffer);

        memcpy(pbyCurPos, szBuffer, nStrLen);

        // write the offset of the message
        pbyTemp = stackPlaceholder.Peek();
        pbyTemp += (2 * g_elinfo.m_nLookMessageNumber);
        pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
        pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

        pbyCurPos += nStrLen;
        usBytesToEndOfMessages += nStrLen;
        nTotalBytes += nStrLen;
    }

    // go back to the beginning of the message section and write
    // how many bytes it is to the end
    pbyTemp = stackPlaceholder.Pop();
    pbyTemp[0] = LOBYTE(usBytesToEndOfMessages);
    pbyTemp[1] = HIBYTE(usBytesToEndOfMessages);

    // write the logic to a file
    char szFileLocation[_MAX_PATH + 1];
    sprintf(szFileLocation, "%slogic.%0.3d", pszGameDir, 
            g_mlopt.m_nLogicNumber);

    FILE* file = fopen(szFileLocation, "wb");

    if (!file)
    {
        MessageBox(NULL, "couldn't open file for compiled logic",
                   "error", MB_OK);
        return;
    }

    fwrite(abyLogicData, 1, nTotalBytes, file);
    fclose(file);
}

/////////////////////////////////////////////////////
//
// NewMessageNumber
//
/////////////////////////////////////////////////////
//
// Purpose:
//    generates the next message number that will
//    be assigned to a message
// Parameter n:
//    output parameter through which the new number
//    is reported
// Return value:
//    the new number
// Remarks:
//    I don't remember why I used both an output
//    parameter and a return value for this function
//
/////////////////////////////////////////////////////

int NewMessageNumber(int& n)
{
    n = g_nNextMessageNum;
    g_nNextMessageNum++;
    return n;
}

/////////////////////////////////////////////////////
//
// EncryptString
//
/////////////////////////////////////////////////////
//
// Purpose:
//    performs the weak encryption of the message
//    section of the compiled logic resource
// Parameter psz:
//    the message to encrypt; must not be NULL
// Remarks:
//    messages must be encrypted in the order that
//    they are going to be stored in the logic file
//
/////////////////////////////////////////////////////

void EncryptString(PSTR psz)
{
    // IMPORTANT NOTE: messages must be encrypted in the order that they
    //                 are going to be stored in the logic file; may be
    //                 better just to change this so that the entire message
    //                 section is encrypted at once
    assert(psz != NULL);

    // the null terminator is included in the encryption, so add 1 to
    // the return value of strlen
    int nStrLen = strlen(psz) + 1;

 
    for (int i = 0; i < nStrLen; i++)
    {
        psz[i] ^= g_szAvisDurgan[g_nAvisPos];

        g_nAvisPos++;
        g_nAvisPos %= AVIS_DURGAN_LEN;
    }
}

/////////////////////////////////////////////////////
//
// ResetAvisPos
//
/////////////////////////////////////////////////////
//
// Purpose:
//    Resets the position in the Avis Durgan string
//    used by the EncryptString function -- needed
//    because of the way that I wrote the encryption
//    due to a misunderstanding of the logic resource
//    format
//
/////////////////////////////////////////////////////

void ResetAvisPos()
{
    g_nAvisPos = 0;
}

