/////////////////////////////////////////////////////
//
// MainDlgEventHandlers.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implementation of event handlers for the main
//    BLG dialog
//
// Portability information:
//    This file contains non-portable code written
//    for a Windows version of the software.
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////
#include "resource.h"
#include "BLGDefs.h"
#include "BLGExceptions.h"
#include "DefineNameList.h"
#include "Persistence.h"
#include "GeneralUI.h"
#include "MainDlgUI.h"
#include "MainDlgEventHandlers.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in AGIBLG.cpp
extern HINSTANCE g_hInstance;
/////////////////////////////////////////////////////

HWND g_hwndMainDlg;

/////////////////////////////////////////////////////
// declared in MainDlgUI.cpp
extern COptions g_options;
extern CDefineNameList g_deflist;
extern CEntryLookInfo g_elinfo;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// OnMainDlgWMCommand
//
/////////////////////////////////////////////////////
//
// Purpose:
//    WM_COMMAND handler for the main BLG dialog
// Parameter hDlg:
//    the window handle of the dialog for which the
//    WM_COMMAND message needs to be handled
// Parameter wParam:
//    the wParam sent with the WM_COMMAND message
// Parameter lParam:
//    the lParam sent with the WM_COMMAND message
//
/////////////////////////////////////////////////////

BOOL OnMainDlgWMCommand(HWND hDlg, WPARAM wParam, LPARAM lParam)
{
    switch(LOWORD(wParam))
    {
    case IDC_EDIT_LOGIC_NUMBER:
        return OnEditLogicNumber(wParam);

    case IDC_CHECK_USE_ROOM_NUM_PIC:
        return OnUseRoomNumPic();

    case IDC_CHECK_FIRST_ROOM:
        return OnFirstRoom();

    case IDC_CHECK_DRAW_EGO_INITIALLY:
        return OnDrawEgoInitially();

    case IDC_BUTTON_ENTER_LOOK:
        return OnButtonEnterLook();

    case IDC_BUTTON_FIRST_ROOM_CONTROLS:
        return OnButtonFirstRoomControls();

    case IDC_BUTTON_OPTIONS:
        return OnButtonOptions();

    case IDC_BUTTON_BROWSE_FOR_GAME_PATH:
        return OnButtonBrowseForGamePath();

    case IDC_LIST_EGO_POS:
        return OnListEgoPos(wParam);

    case IDC_BUTTON_ADD_POS_CONTROL:
        return OnAddPosControl();

    case IDC_BUTTON_EDIT_POS_CONTROL:
        return OnEditPosControl();

    case IDC_BUTTON_DEL_POS_CONTROL:
        return OnDelPosControl();

    case IDC_BUTTON_CLEAR_POS_CONTROLS:
        return OnClearPosControls();

    case IDC_BUTTON_EDGE_CONTROL_ADVANCED:
        return OnEdgeControlAdvanced();

    case IDOK:
        return OnMainDlgOK();

    case IDCANCEL:
        return OnMainDlgCancel();

    }
    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnMainDlgWMInitDialog
//
/////////////////////////////////////////////////////
//
// Purpose:
//    WM_INITDIALOG handler for the main BLG dialog
// Parameter hDlg:
//    the window handle of the dialog for which the
//    WM_INITDIALOG message needs to be handled
// Parameter wParam:
//    the wParam sent with the WM_INITDIALOG message
// Parameter lParam:
//    the lParam sent with the WM_INITDIALOG message
//
/////////////////////////////////////////////////////

BOOL OnMainDlgWMInitDialog(HWND hDlg, WPARAM wParam, LPARAM lParam)
{
    HWND hwndControl;
    char szBuffer[_MAX_PATH + 1];
    HMENU hmenu;
    BOOL bLoadStateInfoProblems = FALSE;

    SetClassLong(hDlg, GCL_HICON, 
                 reinterpret_cast<LONG>(LoadIcon(g_hInstance, 
                                                 MAKEINTRESOURCE(IDI_BLG))));

    // the main BLG window should not be sizeable or maximizeable,
    // but the system menu commands for these actions work, so disable
    // them
    hmenu = GetSystemMenu(hDlg, FALSE);
    EnableMenuItem(hmenu, SC_SIZE, MF_GRAYED);
    EnableMenuItem(hmenu, SC_MAXIMIZE, MF_GRAYED);
    // also add an option for the About box to the system menu
    AppendMenu(hmenu, MF_SEPARATOR, 0, NULL);
    AppendMenu(hmenu, MF_STRING, IDM_SYS_ABOUT, 
        TEXT("&About the BLG"));

    // save the dialog's window handle to a global variable
    g_hwndMainDlg = hDlg;

    CenterWindowToScreen(hDlg);

    try
    {
        LoadBLGStateInfo(BLG_STATE_DEFINE_NAME_LIST, &g_deflist);
    }
    catch(CPersistentDataLoadFailureException)
    {
        bLoadStateInfoProblems = TRUE;
    }

    try
    {
        LoadBLGStateInfo(BLG_STATE_OPTIONS, &g_options);
    }
    catch(CPersistentDataLoadFailureException)
    {
        bLoadStateInfoProblems = TRUE;
    }

    try
    {
        LoadBLGStateInfo(BLG_STATE_LOOK_WORD_GROUP,
                     &g_elinfo.m_nLookWordGroup);
    }
    catch(CPersistentDataLoadFailureException)
    {
        bLoadStateInfoProblems = TRUE;
    }

    try
    {
        strcpy(szBuffer, "");
        LoadBLGStateInfo(BLG_STATE_LAST_GAME_PATH, szBuffer);
        SetWindowText(GetDlgItem(hDlg, IDC_EDIT_GAME_PATH), szBuffer);
    }
    catch(CPersistentDataLoadFailureException)
    {
        bLoadStateInfoProblems = TRUE;
    }

    if (bLoadStateInfoProblems)
    {
        MessageBox(hDlg, "Some BLG settings could not be loaded.\n\n"
                   "This is normal if this is the first time you've "
                   "run the BLG on this computer.\n\nIf it is not, some of "
                   "your settings may be lost.", "Settings",
                   MB_OK);
    }


    /////////////////////////////////////////////////////
    // logic number edit box initialization
    //
    // enabled      : yes
    // initial value:   0
    // min. value   :   0
    // max. value   : 255

    hwndControl = GetDlgItem(hDlg, IDC_EDIT_LOGIC_NUMBER);

    // set the maximum digits for the logic number field
    SendMessage(hwndControl, EM_SETLIMITTEXT, 
                static_cast<WPARAM>(MAX_DIGITS), 0L);

    // set the initial value for the logic number field
    SetWindowText(hwndControl, "0");

    /////////////////////////////////////////////////////
    // pic number edit box initialization
    //
    // enabled      :  no
    // initial value:   0
    // min. value   :   0
    // max. value   : 255

    hwndControl = GetDlgItem(hDlg, IDC_EDIT_PIC_NUMBER);

    // set the maximum digits for the pic number field
    SendMessage(hwndControl, EM_SETLIMITTEXT,
                static_cast<WPARAM>(MAX_DIGITS), 0L);

    // in the average case, the user won't want to have to manually
    // enter the picture number; disable this control to start
    EnableWindow(hwndControl, FALSE);

    // set the initial value for the pic number field
    SetWindowText(hwndControl, "0");

    /////////////////////////////////////////////////////
    // horizon at y edit box initialization
    //
    // enabled      : yes
    // initial value:  36
    // min. value   :   0
    // max. value   : 166 (167 locks the game)

    hwndControl = GetDlgItem(hDlg, IDC_EDIT_HORIZON);

    // set the maximum digits for the horizon at y field
    SendMessage(hwndControl, EM_SETLIMITTEXT, 
                static_cast<WPARAM>(MAX_DIGITS), 0L);

    // set the initial value for the horizon at y field (this
    // value is 36 by default because the horizon defaults to
    // 36 in AGI when the new.room command is issued; a value
    // of 36 here means that no set.horizon command should be
    // included in the generated logic)
    SetWindowText(hwndControl, "36");

    /////////////////////////////////////////////////////
    // left edge goto edit box initialization
    //
    // enabled      :     yes
    // initial value: (empty) 
    // min. value   :       0
    // max. value   :     255

    // set the maximum digits for the left edge goto field
    SendMessage(GetDlgItem(hDlg, IDC_EDIT_LEFT_EDGE_GOTO),
                EM_SETLIMITTEXT, static_cast<WPARAM>(MAX_DIGITS), 0L);

    /////////////////////////////////////////////////////
    // right edge goto edit box initialization
    //
    // enabled      :     yes
    // initial value: (empty) 
    // min. value   :       0
    // max. value   :     255

    // set the maximum digits for the right edge goto field
    SendMessage(GetDlgItem(hDlg, IDC_EDIT_RIGHT_EDGE_GOTO),
                EM_SETLIMITTEXT, static_cast<WPARAM>(MAX_DIGITS), 0L);

    /////////////////////////////////////////////////////
    // bottom edge goto edit box initialization
    //
    // enabled      :     yes
    // initial value: (empty) 
    // min. value   :       0
    // max. value   :     255

    // set the maximum digits for the bottom edge goto field
    SendMessage(GetDlgItem(hDlg, IDC_EDIT_BOTTOM_EDGE_GOTO),
                EM_SETLIMITTEXT, static_cast<WPARAM>(MAX_DIGITS), 0L);

    /////////////////////////////////////////////////////
    // horizon edge goto edit box initialization
    //
    // enabled      :     yes
    // initial value: (empty) 
    // min. value   :       0
    // max. value   :     255

    // set the maximum digits for the horizon edge goto field
    SendMessage(GetDlgItem(hDlg, IDC_EDIT_HORIZON_EDGE_GOTO),
                EM_SETLIMITTEXT, static_cast<WPARAM>(MAX_DIGITS), 0L);

    /////////////////////////////////////////////////////
    // other control initialization

    // the use room_no for pic checkbox should be checked by default;
    // its value indicates whether the pic number edit box is
    // enabled; if this box is checked, then the value of the pic number
    // edit box must be exactly the same as the value of the logic number
    // edit box; otherwise, the value of the pic number edit box is
    // user-editable
    SetCheckboxChecked(hDlg, IDC_CHECK_USE_ROOM_NUM_PIC);    
    
    // a generated logic is not the first room in the game in the average
    // case; don't enable the first room controls button unless this logic
    // actually is for the first room in the game
    EnableWindow(GetDlgItem(hDlg, IDC_BUTTON_FIRST_ROOM_CONTROLS), FALSE);
    
    // to start out with, there is no selection in the Ego Positioning
    // list box, so the Delete and Edit buttons are not applicable; disable
    // them until a selection is made
    EnableWindow(GetDlgItem(hDlg, IDC_BUTTON_DEL_POS_CONTROL), FALSE);
    EnableWindow(GetDlgItem(hDlg, IDC_BUTTON_EDIT_POS_CONTROL), FALSE);

    // to start out with, there are no Ego Positioning control items
    // in the Ego Positioning list box, so Delete All button is inapplicable;
    // disable it until there are some items in the list box
    EnableWindow(GetDlgItem(hDlg, IDC_BUTTON_CLEAR_POS_CONTROLS), FALSE);

    // in the average case, ego will be drawn in a new room as soon as the
    // room loads
    SetCheckboxChecked(hDlg, IDC_CHECK_DRAW_EGO_INITIALLY);

    hwndControl = GetDlgItem(hDlg, IDC_EDIT_LOGIC_TITLE);
    SendMessage(hwndControl, EM_SETLIMITTEXT,
                static_cast<WPARAM>(MAX_LOGIC_TITLE_LEN), 0);

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnMainDlgWMSysCommand
//
/////////////////////////////////////////////////////
//
// Purpose:
//    WM_SYSCOMMAND handler for the main BLG dialog
// Parameter hDlg:
//    the window handle of the dialog for which the
//    WM_SYSCOMMAND message needs to be handled
// Parameter wParam:
//    the wParam sent with the WM_SYSCOMMAND message
// Parameter lParam:
//    the lParam sent with the WM_SYSCOMMAND message
//
/////////////////////////////////////////////////////

BOOL OnMainDlgWMSysCommand(HWND hDlg, WPARAM wParam, LPARAM lParam)
{
    switch(LOWORD(wParam))
    {
    case IDM_SYS_ABOUT:
        return OnMainDlgSysAbout();
    }

    return FALSE;
}