/////////////////////////////////////////////////////
//
// MessageBoxesUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    BLG's custom message boxes; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written for a Windows
//    build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include "resource.h"
#include "BLGDefs.h"
#include "GeneralUI.h"
#include "MessageBoxes.h"
#include "MessageBoxesUI.h"

/////////////////////////////////////////////////////
// declared in MessageBoxesEventHandlers.cpp
extern HWND g_hwndMsgBox;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in MainDlgEventHandlers.cpp
extern COptions g_options;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// functions for Unescaped Quote message boxes
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// OnRadioIncludeQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_INCLUDE_QUOTES
//    radio button for the Unescaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnRadioIncludeQuotes()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_INCLUDE_QUOTES))
    {
        // the Don't Ask Again option is applicable for this selection,
        // so enable the checkbox
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnRadioRemoveQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_REMOVE_QUOTES
//    radio button for the Unescaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnRadioRemoveQuotes()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_REMOVE_QUOTES))
    {
        // the Don't Ask Again option is applicable for this selection,
        // so enable the checkbox
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnRadioEditQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_EDIT_QUOTES
//    radio button for the Unescaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnRadioEditQuotes()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_EDIT_QUOTES))
    {
        // the Don't Ask Again option is not applicable for this selection,
        // so disable the checkbox
        SetCheckboxChecked(g_hwndMsgBox, IDC_CHECK_REMEMBER_QUOTE_RESPONSE,
                           FALSE);
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), FALSE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnUnescapedQuoteMessageBoxOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the OK button for the Unescaped
//    Quotes msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnUnescapedQuoteMessageBoxOK()
{
    // this code is fairly self-explanatory...basically,
    // figure out which selection the user made, set the
    // return value appropriately, and if the user checked
    // the Don't Ask Again box then save their choice so that
    // they won't be asked again

    int nSelection = QueryRadioButton(g_hwndMsgBox, 
                                     IDC_RADIO_INCLUDE_QUOTES,
                                     IDC_RADIO_EDIT_QUOTES);
    int nReturnValue = IDCANCEL;
    BOOL bDontAskAgain = IsCheckboxChecked(g_hwndMsgBox,
                                           IDC_CHECK_REMEMBER_QUOTE_RESPONSE);

    switch(nSelection)
    {
    case IDC_RADIO_INCLUDE_QUOTES:
        nReturnValue = UQMB_INCLUDE_QUOTES;
        if (bDontAskAgain)
        {
            g_options.m_nUnescapedQuoteResponse = 
                OPT_UQR_ALWAYS_INCLUDE_QUOTES;
        }
        break;

    case IDC_RADIO_REMOVE_QUOTES:
        nReturnValue = UQMB_REMOVE_QUOTES;
        if (bDontAskAgain)
        {
            g_options.m_nUnescapedQuoteResponse =
                OPT_UQR_ALWAYS_REMOVE_QUOTES;
        }
        break;

    case IDC_RADIO_EDIT_QUOTES:
        // the Don't Ask Again option doesn't apply for this
        // selection, because the program needs to let the
        // user know if the text they entered needs editing;
        // there's not much point in implementing a separate
        // message box for this
        nReturnValue = UQMB_USER_EDIT;
        break;
    }

    EndDialog(g_hwndMsgBox, nReturnValue);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// functions for Escaped Quote message boxes
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// OnEscapedQuoteMsgBoxRadioRemoveBackslash
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_REMOVE_BACKSLASH
//    radio button for the Escaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEscapedQuoteMsgBoxRadioRemoveBackslash()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_REMOVE_BACKSLASH))
    {
        // the Don't Ask Again option is applicable for this selection,
        // so enable the checkbox
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEscapedQuoteMsgBoxRadioIncludeBackslash
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_INCLUDE_BACKSLASH
//    radio button for the Escaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEscapedQuoteMsgBoxRadioIncludeBackslash()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_INCLUDE_BACKSLASH))
    {
        // the Don't Ask Again option is applicable for this selection,
        // so enable the checkbox
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEscapedQuoteMsgBoxRadioEditQuotes
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_RADIO_EDIT_QUOTES
//    radio button for the Escaped Quote msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEscapedQuoteMsgBoxRadioEditQuotes()
{
    if (IsRadioButtonSelected(g_hwndMsgBox, IDC_RADIO_EDIT_QUOTES))
    {
        // the Don't Ask Again option is not applicable for this selection,
        // so disable the checkbox
        SetCheckboxChecked(g_hwndMsgBox, IDC_CHECK_REMEMBER_QUOTE_RESPONSE,
                           FALSE);
        EnableWindow(GetDlgItem(g_hwndMsgBox, 
                                IDC_CHECK_REMEMBER_QUOTE_RESPONSE), FALSE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEscapedQuoteMessageBoxOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the OK button for the Escaped
//    Quotes msg box
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEscapedQuoteMsgBoxOK()
{
    // this code is fairly self-explanatory...basically,
    // figure out which selection the user made, set the
    // return value appropriately, and if the user checked
    // the Don't Ask Again box then save their choice so that
    // they won't be asked again

    int nSelection = QueryRadioButton(g_hwndMsgBox, 
                                     IDC_RADIO_REMOVE_BACKSLASH,
                                     IDC_RADIO_EDIT_QUOTES);
    int nReturnValue = IDCANCEL;
    BOOL bDontAskAgain = IsCheckboxChecked(g_hwndMsgBox,
                                           IDC_CHECK_REMEMBER_QUOTE_RESPONSE);

    switch(nSelection)
    {
    case IDC_RADIO_REMOVE_BACKSLASH:
        nReturnValue = EQMB_REMOVE_BACKSLASH;
        if (bDontAskAgain)
        {
            g_options.m_nEscapedQuoteResponse =
                OPT_EQR_ALWAYS_REMOVE_BACKSLASH;
        }
        break;

    case IDC_RADIO_INCLUDE_BACKSLASH:
        nReturnValue = EQMB_INCLUDE_BACKSLASH;
        if (bDontAskAgain)
        {
            g_options.m_nEscapedQuoteResponse =
                OPT_EQR_ALWAYS_INCLUDE_BACKSLASH;
        }
        break;

    case IDC_RADIO_EDIT_QUOTES:
        // the Don't Ask Again option doesn't apply for this
        // selection, because the program needs to let the
        // user know if the text they entered needs editing;
        // there's not much point in implementing a separate
        // message box for this
        nReturnValue = EQMB_USER_EDIT;
        break;
    }

    EndDialog(g_hwndMsgBox, nReturnValue);
    return TRUE;
}
