/////////////////////////////////////////////////////
//
// Persistence.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implements the BLG state persistence interface
//    defined in Persistence.h
//
// Portability information:
//    a Win32 build of the program uses the Windows
//    registry to store state information; ports of
//    the BLG will have to provide their own implementation
//    of the state persistence interface
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "BLGExceptions.h"
#include "DefineNameList.h"
#include "Persistence.h"

#if defined(WIN32)

/////////////////////////////////////////////////////
//
// SPECIAL NOTE ABOUT WIN32 BUILDS
//
// if you are making additions for a Win32 build of
// the program, give the program a different version
// number and change the registry key where the options
// are stored for your build of the program; the settings
// in the software currently save to the key
//
// HKEY_CURRENT_USER/Software/VvTGrouchSoftware/AGIBLG/Version2.0
//
// you should change only the version number when you change
// anything at all; when you build a new version, you should
// change the version number, especially if your changes
// to the software include modification, deletion, or
// addition of values stored in a previous version; new
// builds of the software should not conflict with the old;
// users should be able to have more than one version of the
// Base Logic Generator on their machine at once
//
// also, only change the major and minor version numbers,
// don't add build numbers; users don't need Version2.1.345,
// Version2.1.346, Version2.1.347, etc. in their registry,
// or even Version2.11, Version2.12, etc. if it can be avoided
//
/////////////////////////////////////////////////////

#include <windows.h>

// constants that define the names of the registry values
// so that they don't have to be typed out each time they are used
#define REG_ENTRY_DEFINE_NAME_LIST         "DefineNameList"
#define REG_ENTRY_HEADER_COMMENT_STYLE     "HeaderCommentStyle"
#define REG_ENTRY_ONE_LINE_COMMENT_STYLE   "OneLineCommentStyle"
#define REG_ENTRY_TABS_OR_SPACES           "TabsOrSpaces"
#define REG_ENTRY_INDENTER_COUNT           "IndenterCount"
#define REG_ENTRY_CURLY_BRACE_STYLE        "CurlyBraceStyle"
#define REG_ENTRY_UNESCAPED_QUOTE_RESPONSE "UnescapedQuoteResponse"
#define REG_ENTRY_ESCAPED_QUOTE_RESPONSE   "EscapedQuoteReponse"
#define REG_ENTRY_GEN_HEADER               "GenerateHeader"
#define REG_ENTRY_GEN_BASIC_COMMENTS       "GenerateBasicComments"
#define REG_ENTRY_GEN_VERBOSE_COMMENTS     "GenerateVerboseComments"
#define REG_ENTRY_GEN_TODO_COMMENTS        "GenerateTODOComments"
#define REG_ENTRY_GEN_WARNING_COMMENTS     "GenerateWarningComments"
#define REG_ENTRY_USE_DEFINE_NAMES         "UseDefineNames"
#define REG_ENTRY_WRITE_TO_FILE            "WriteToFile"
#define REG_ENTRY_WRITE_TO_CLIPBOARD       "WriteToClipboard"
#define REG_ENTRY_GEN_SOURCE               "GenerateSource"
#define REG_ENTRY_GEN_COMPILED_CODE        "GenerateCompiledCode"
#define REG_ENTRY_ALLOW_EMPTY_CODE_BLOCKS  "AllowEmptyCodeBlocks"
#define REG_ENTRY_LOOK_WORD_GROUP          "LookWordGroup"
#define REG_ENTRY_LAST_GAME_PATH           "LastGamePath"

void SaveOptions(COptions* poptions, HKEY hkey);
void LoadOptions(COptions* poptions, HKEY hkey);

/////////////////////////////////////////////////////
//
// SaveBLGStateInfo (Windows version)
//
/////////////////////////////////////////////////////
//
// Purpose:
//    saves various information regarding the state
//    of the BLG so that the user doesn't have to
//    keep setting their options every time the BLG
//    is run
// Parameter ulInfoType:
//    a ULONG specifying what kind of information is
//    to be stored -- use the constants defined in
//    Persistence.h for this parameter
// Parameter pvData:
//    a pointer to a data type appropriate for the
//    kind of information that is being stored;
//    depending on the value of ulInfoType, must not
//    be NULL; see the list of ulInfoType constants
//    in Persistence.h for a description of what kind
//    of pointer is expected for various values of
//    ulInfoType
//
/////////////////////////////////////////////////////

void SaveBLGStateInfo(ULONG ulInfoType, void* pvData)
{
    // up until the switch(ulInfoType) statement, this function is
    // standard registry stuff -- consult an API reference if you're
    // not familiar with it
    HKEY hkeySoftware;
    HKEY hkeyVvTGrouchSoftware;
    HKEY hkeyAGIBLG;
    HKEY hkeyVersion2Point0;
    LONG lResult;
    char szFileLocation[_MAX_PATH + 1];

    lResult = RegCreateKeyEx(HKEY_CURRENT_USER, "Software", 0, NULL,
                             REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL,
                             &hkeySoftware, NULL);

    if (lResult != ERROR_SUCCESS)
    {
        // TODO: throw exception
        MessageBox(NULL, "couldn't open registry key", "registry error", 
                   MB_OK);
        return;
    }

    lResult = RegCreateKeyEx(hkeySoftware, "VvTGrouchSoftware", 0, NULL, 
                             REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL,
                             &hkeyVvTGrouchSoftware, NULL);



    if (lResult != ERROR_SUCCESS)
    {
        // TODO: throw exception
        MessageBox(NULL, 
                   "couldn't create registry key\n\n"
                   "normal on first run", 
                   "registry error", MB_OK);
        RegCloseKey(hkeySoftware);
        return;
    }

    lResult = RegCreateKeyEx(hkeyVvTGrouchSoftware, "AGIBLG", 0, NULL, 
                             REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL,
                             &hkeyAGIBLG, NULL);

    if (lResult != ERROR_SUCCESS)
    {
        // TODO: throw exception
        MessageBox(NULL, 
                   "couldn't create registry key", 
                   "registry error", MB_OK);
        RegCloseKey(hkeyVvTGrouchSoftware);
        RegCloseKey(hkeySoftware);
        return;
    }

    lResult = RegCreateKeyEx(hkeyAGIBLG, "Version2.0", 0, NULL, 
                             REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL,
                             &hkeyVersion2Point0, NULL);

    if (lResult != ERROR_SUCCESS)
    {
        // TODO: throw exception
        MessageBox(NULL, 
                   "couldn't create registry key", 
                   "registry error", MB_OK);
        RegCloseKey(hkeyAGIBLG);
        RegCloseKey(hkeyVvTGrouchSoftware);
        RegCloseKey(hkeySoftware);
        return;
    }

    switch(ulInfoType)
    {
    case BLG_STATE_DEFINE_NAME_LIST:
        // saving the list of define names -- note: what this actually
        // does is store the define names in a file and save the location
        // of that file in the registry
        assert(pvData != NULL);
        CDefineNameList* pdeflist;

        pdeflist = reinterpret_cast<CDefineNameList*>(pvData);

        pdeflist->GetDesiredPath(szFileLocation, _MAX_PATH + 1);

        if (strlen(szFileLocation) == 0)
        {
            GetCurrentDirectory(_MAX_PATH + 1, szFileLocation);

            int nStrLen = strlen(szFileLocation);
            if (szFileLocation[nStrLen - 1] != '\\')
            {
                strcat(szFileLocation, "\\");
            }

            strcat(szFileLocation, "___deflist.dat");
            pdeflist->SetDesiredPath(szFileLocation);

        }

        lResult = RegSetValueEx(hkeyVersion2Point0, 
                                REG_ENTRY_DEFINE_NAME_LIST,
                                0, REG_SZ, 
                                reinterpret_cast<BYTE*>(szFileLocation),
                                strlen(szFileLocation) + 1);

        if (lResult != ERROR_SUCCESS)
        {
            // TODO: throw exception
            MessageBox(NULL, "couldn't set registry value", 
                       "registry error", MB_OK);
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
            return;
        }

        // the following if statement saves the define names to a file
        if (!pdeflist->SaveToDesiredPath())
        {
            // TODO: throw exception
            MessageBox(NULL, 
                       "couldn't save define names in specified location", 
                       "save error", MB_OK);
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
            return;
        }

        pdeflist = NULL;
        break;

    case BLG_STATE_OPTIONS:
        // saving the options stored in an instance of the COptions class
        assert(pvData != NULL);
        COptions* poptions;
        poptions = reinterpret_cast<COptions*>(pvData);

        // there are a lot of options, so let another function take care
        // of it
        SaveOptions(poptions, hkeyVersion2Point0);

        break;

    case BLG_STATE_LOOK_WORD_GROUP:
        // saving the WORDS.TOK group number for the word "look"
        assert(pvData != NULL);

        int* pnLookWordGroup;
        pnLookWordGroup = reinterpret_cast<int*>(pvData);

        lResult = RegSetValueEx(hkeyVersion2Point0, 
                                REG_ENTRY_LOOK_WORD_GROUP,
                                0, REG_DWORD, 
                                reinterpret_cast<BYTE*>(pnLookWordGroup),
                                sizeof(int));

        if (lResult != ERROR_SUCCESS)
        {
            MessageBox(NULL, "Couldn't set registry value", "Error",
                       MB_OK);
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
            return;
        }

        break;

    case BLG_STATE_LAST_GAME_PATH:
        // saving the game path so that the user doesn't have to enter
        // it every time they generate a file
        assert(pvData != NULL);

        PSTR pszGamePath;
        pszGamePath = reinterpret_cast<PSTR>(pvData);

        lResult = RegSetValueEx(hkeyVersion2Point0, 
                                REG_ENTRY_LAST_GAME_PATH,
                                0, REG_SZ, 
                                reinterpret_cast<BYTE*>(pszGamePath),
                                strlen(pszGamePath) + 1);

        if (lResult != ERROR_SUCCESS)
        {
            MessageBox(NULL, "Couldn't set registry value", "Error",
                       MB_OK);
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
            return;
        }

        break;

    }

    // finished saving the persistent information...close out the registry
    // keys
    lResult = RegCloseKey(hkeyVersion2Point0);

    if (lResult != ERROR_SUCCESS)
    {
        MessageBox(NULL, "Error closing Version2.0 key", "TRACE", MB_OK);
    }

    lResult = RegCloseKey(hkeyAGIBLG);

    if (lResult != ERROR_SUCCESS)
    {
        MessageBox(NULL, "Error closing AGIBLG key", "TRACE", MB_OK);
    }

    lResult = RegCloseKey(hkeyVvTGrouchSoftware);

    if (lResult != ERROR_SUCCESS)
    {
        MessageBox(NULL, "Error closing vvtg key", "TRACE", MB_OK);
    }

    lResult = RegCloseKey(hkeySoftware);

    if (lResult != ERROR_SUCCESS)
    {
        MessageBox(NULL, "Error closing vvtg key", "TRACE", MB_OK);
    }

}

/////////////////////////////////////////////////////
//
// LoadBLGStateInfo (Windows version)
//
/////////////////////////////////////////////////////
//
// Purpose:
//    loads various information regarding the state
//    of the BLG so that the user doesn't have to
//    keep setting their options every time the BLG
//    is run
// Parameter ulInfoType:
//    a ULONG specifying what kind of information is
//    to be loaded -- use the constants defined in
//    Persistence.h for this parameter
// Parameter pvData:
//    a pointer to a data type appropriate for the
//    kind of information that is being loaded;
//    depending on the value of ulInfoType, must not
//    be NULL; see the list of ulInfoType constants
//    in Persistence.h for a description of what kind
//    of pointer is expected for various values of
//    ulInfoType
//
/////////////////////////////////////////////////////

void LoadBLGStateInfo(ULONG ulInfoType, void* pvData)
{
    // up until the switch(ulInfoType) statement, this function is
    // standard registry stuff -- consult an API reference if you're
    // not familiar with it
    HKEY hkeySoftware;
    HKEY hkeyVvTGrouchSoftware;
    HKEY hkeyAGIBLG;
    HKEY hkeyVersion2Point0;
    LONG lResult;
    char szFileLocation[_MAX_PATH + 1];
    DWORD dwDataSize;
    BOOL bKeyOpenFailed = FALSE;

    lResult = RegOpenKeyEx(HKEY_CURRENT_USER, "Software", 0,
                           KEY_READ, &hkeySoftware);

    if (lResult != ERROR_SUCCESS)
    {
        throw CPersistentDataLoadFailureException();
        return;
    }

    lResult = RegOpenKeyEx(hkeySoftware, "VvTGrouchSoftware", 0,
                           KEY_READ, &hkeyVvTGrouchSoftware);

    if (lResult != ERROR_SUCCESS)
    {
        RegCloseKey(hkeySoftware);
        throw CPersistentDataLoadFailureException();
        return;
    }

    lResult = RegOpenKeyEx(hkeyVvTGrouchSoftware, "AGIBLG", 0,
                           KEY_READ, &hkeyAGIBLG);

    if (lResult != ERROR_SUCCESS)
    {
        RegCloseKey(hkeyVvTGrouchSoftware);
        RegCloseKey(hkeySoftware);
        throw CPersistentDataLoadFailureException();
        return;
    }

    lResult = RegOpenKeyEx(hkeyAGIBLG, "Version2.0", 0,
                           KEY_READ, &hkeyVersion2Point0);


    if (lResult != ERROR_SUCCESS)
    {
        RegCloseKey(hkeyAGIBLG);
        RegCloseKey(hkeyVvTGrouchSoftware);
        RegCloseKey(hkeySoftware);
        throw CPersistentDataLoadFailureException();
        return;
    }

    switch(ulInfoType)
    {
    case BLG_STATE_DEFINE_NAME_LIST:
        // retrieve the location of the define name list file and then load
        // the file
        assert(pvData != NULL);

        dwDataSize = sizeof(szFileLocation);

        lResult = RegQueryValueEx(hkeyVersion2Point0,
                                  REG_ENTRY_DEFINE_NAME_LIST,
                                  0, NULL, 
                                  reinterpret_cast<BYTE*>(szFileLocation),
                                  &dwDataSize);

        if (lResult != ERROR_SUCCESS)
        {
/*            char szErrorMsg[1000];
            FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL,
                          lResult, 0, szErrorMsg,
                          1000, NULL);
            MessageBox(NULL, szErrorMsg, "error", MB_OK);
*/
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
//            throw CPersistentDataLoadFailureException();
            return;
        }

        CDefineNameList* pdeflist;
        pdeflist = reinterpret_cast<CDefineNameList*>(pvData);

        if (!pdeflist->Load(szFileLocation))
        {
            RegCloseKey(hkeyVersion2Point0);
            RegCloseKey(hkeyAGIBLG);
            RegCloseKey(hkeyVvTGrouchSoftware);
            RegCloseKey(hkeySoftware);
//            throw CPersistentDataLoadFailureException();
            return;
        }

        break;

    case BLG_STATE_OPTIONS:
        // retrieve the data that goes into an instance of the COptions
        // class
        assert(pvData != NULL);
        COptions* poptions;
        poptions = reinterpret_cast<COptions*>(pvData);

        // there are a lot of options, so let a function take care of
        // loading them
        LoadOptions(poptions, hkeyVersion2Point0);
        break;

    case BLG_STATE_LOOK_WORD_GROUP:
        // get the WORDS.TOK group number for the word "look" so that
        // users who do not have the default value for the template
        // game do not have to keep entering their own group number
        // every time they generate a file
        assert(pvData != NULL);

        int* pnLookWordGroup;
        pnLookWordGroup = reinterpret_cast<int*>(pvData);

        dwDataSize = sizeof(int);

        lResult = RegQueryValueEx(hkeyVersion2Point0,
                                  REG_ENTRY_LOOK_WORD_GROUP,
                                  0, NULL, 
                                  reinterpret_cast<BYTE*>(pnLookWordGroup),
                                  &dwDataSize);

        if (lResult != ERROR_SUCCESS)
        {
            // if no look group, assume it is the template game default
            *pnLookWordGroup = DEFAULT_LOOK_WORD_GROUP;
        }

        break;

    case BLG_STATE_LAST_GAME_PATH:
        // load the game path that was used last time the BLG was run
        assert(pvData != NULL);

        PSTR pszGamePath;
        pszGamePath = reinterpret_cast<PSTR>(pvData);

        dwDataSize = _MAX_PATH + 1;

        lResult = RegQueryValueEx(hkeyVersion2Point0,
                                  REG_ENTRY_LAST_GAME_PATH,
                                  0, NULL, 
                                  reinterpret_cast<BYTE*>(pszGamePath),
                                  &dwDataSize);

        if (lResult != ERROR_SUCCESS)
        {
            // if no game path, assume nothing...the user can reenter it
            // in this case
            strcpy(pszGamePath, "");
        }

        break;
    }

    // finished loading state info...close out the registry keys
    RegCloseKey(hkeyVersion2Point0);
    RegCloseKey(hkeyAGIBLG);
    RegCloseKey(hkeyVvTGrouchSoftware);
    RegCloseKey(hkeySoftware);
}

/////////////////////////////////////////////////////
//
// SaveOptions
//
/////////////////////////////////////////////////////
//
// Purpose:
//    saves the contents of a COptions object to the
//    registry under the key hkey
// Parameter poptions:
//    a pointer to a COptions object containing the
//    options to be saved to the registry; must not be NULL
// Parameter hkey:
//    an handle to an ALREADY-OPENED registry key to
//    which the options are to be written
//
/////////////////////////////////////////////////////

void SaveOptions(COptions* poptions, HKEY hkey)
{
    // not much point in commenting every detail of this function --
    // basically, it writes each member of poptions to the registry;
    // for information about the registry functions, consult a Windows
    // API reference
    assert(poptions != NULL);

    DWORD dwCurSaveValue;
    LONG lResult;

    dwCurSaveValue = poptions->m_bAllowEmptyCompiledCodeBlocks;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_ALLOW_EMPTY_CODE_BLOCKS,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateBasicComments;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_BASIC_COMMENTS,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateCompiledCode;

    lResult = RegSetValueEx(hkey, 
                            REG_ENTRY_GEN_COMPILED_CODE,
                            0, REG_DWORD,
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateHeader;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_HEADER,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateSource;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_SOURCE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateTODOComments;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_TODO_COMMENTS,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateVerboseComments;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_VERBOSE_COMMENTS,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bGenerateWarningComments;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_GEN_WARNING_COMMENTS,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bUseDefineNames;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_USE_DEFINE_NAMES,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bWriteToClipboard;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_WRITE_TO_CLIPBOARD,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_bWriteToFile;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_WRITE_TO_FILE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nCurlyBraceStyle;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_CURLY_BRACE_STYLE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nEscapedQuoteResponse;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_ESCAPED_QUOTE_RESPONSE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nHeaderCommentStyle;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_HEADER_COMMENT_STYLE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nIndenterCount;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_INDENTER_COUNT,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nOneLineCommentStyle;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_ONE_LINE_COMMENT_STYLE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nTabsOrSpaces;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_TABS_OR_SPACES,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));

    dwCurSaveValue = poptions->m_nUnescapedQuoteResponse;

    lResult = RegSetValueEx(hkey,
                            REG_ENTRY_UNESCAPED_QUOTE_RESPONSE,
                            0, REG_DWORD, 
                            reinterpret_cast<BYTE*>(&dwCurSaveValue),
                            sizeof(dwCurSaveValue));
}

/////////////////////////////////////////////////////
//
// LoadOptions
//
/////////////////////////////////////////////////////
//
// Purpose:
//    loads the contents of a COptions object from the
//    registry under the key hkey
// Parameter poptions:
//    a pointer to a COptions object into which the values
//    will be loaded; must not be NULL
// Parameter hkey:
//    an handle to an ALREADY-OPENED registry key from
//    which the options are to be loaded
//
/////////////////////////////////////////////////////

void LoadOptions(COptions* poptions, HKEY hkey)
{
    // not much point in commenting every detail of this function --
    // basically, it reads each member of poptions from the registry and
    // resets defaults if no entry for a particular value is found in
    // the registry; for information about the registry functions, consult
    // a Windows API reference

    assert(poptions != NULL);

    DWORD dwCurValue;
    DWORD dwDataSize;
    LONG lResult;

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_ALLOW_EMPTY_CODE_BLOCKS,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bAllowEmptyCompiledCodeBlocks =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_BASIC_COMMENTS,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateBasicComments =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey,
                              REG_ENTRY_GEN_COMPILED_CODE,
                              0, NULL,
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateCompiledCode =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_HEADER,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateHeader = ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_SOURCE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateSource = ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_TODO_COMMENTS,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateTODOComments =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_VERBOSE_COMMENTS,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateVerboseComments =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_GEN_WARNING_COMMENTS,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bGenerateWarningComments =
            ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_USE_DEFINE_NAMES,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bUseDefineNames = ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_WRITE_TO_CLIPBOARD,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bWriteToClipboard = ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_WRITE_TO_FILE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_bWriteToFile = ((dwCurValue != 0) ? TRUE : FALSE);
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_CURLY_BRACE_STYLE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nCurlyBraceStyle = dwCurValue;
    }
    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_ESCAPED_QUOTE_RESPONSE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nEscapedQuoteResponse = dwCurValue;
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_HEADER_COMMENT_STYLE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nHeaderCommentStyle = dwCurValue;
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_INDENTER_COUNT,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nIndenterCount = dwCurValue;
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_ONE_LINE_COMMENT_STYLE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nOneLineCommentStyle = dwCurValue;
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_TABS_OR_SPACES,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);

    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nTabsOrSpaces = dwCurValue;
    }

    dwDataSize = sizeof(dwCurValue);

    lResult = RegQueryValueEx(hkey, 
                              REG_ENTRY_UNESCAPED_QUOTE_RESPONSE,
                              0, NULL, 
                              reinterpret_cast<BYTE*>(&dwCurValue),
                              &dwDataSize);
    if (lResult == ERROR_SUCCESS)
    {
        poptions->m_nUnescapedQuoteResponse = dwCurValue;
    }
}

#endif // defined(WIN32)