/////////////////////////////////////////////////////
//
// SourceCodeGeneration.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    contains functions that generate an AGI logic
//    source code file and either write it to disk
//    or write it to the clipboard
//
// Portability information:
//    the code in this file should be easily portable
//    to other platforms
//
//    a note about line breaks in generated source code:
//    always terminate a line in the generated code by
//    concatenating the macro STR_END_OF_LINE to the
//    source code string; the end of a line is handled
//    differently between operating systems...Windows
//    uses a CRLF ("\r\n") pair, while Unix uses just an 
//    LF ("\n")
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include <cstdio>
#include "BLGDefs.h"
#include "BLGExceptions.h"
#include "EgoPositionControlArray.h"
#include "DefineNameList.h"
#include "Clipboard.h"
#include "SourceCodeGeneration.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in MainDlgEventHandlers.cpp
extern CMainLogicOptions g_mlopt;
extern CEntryLookInfo g_elinfo;
extern CFirstRoomCtrl g_frctrl;
extern CEgoPositionControlArray g_aepc;
extern CEdgeCodeInfo g_ecinfo;
extern CDefineNameList g_deflist;
extern COptions g_options;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in BLGDefs.cpp
extern int g_anAGIConstValues[RESERVED_CONST_COUNT];
/////////////////////////////////////////////////////

char g_szIndentString[MAX_INDENT_STRING_LEN];

/////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////

// functions used for source code generation
CDefineNameList* MakeNoDefinesDefList();
void Indent(PSTR pszTarget, int nNestLevel);
void MakeIndentString();
void AddHeaderComment(PSTR pszTarget, PSTR pszCommentText, int nNestLevel);
void AddOneLineComment(PSTR pszTarget, PSTR pszCommentText, int nNestLevel);
void GenerateEdgeCodeBlock(PSTR pszTarget, CDefineNameList* pdeflist,
                           ULONG ulEdgeConst, int nNestLevel);
void GenerateIfElseOpenBrace(PSTR pszTarget, int nNestLevel);

/////////////////////////////////////////////////////
//
// GenerateSourceFile
//
/////////////////////////////////////////////////////
//
// Purpose:
//    the main function that should be called upon to
//    generate an AGI logic source file -- takes care
//    of the details of where the logic should be written
//    based on the options that have already been set
// Parameter pszGameDir:
//    the directory where the file should be written, if
//    a file is to be written...in any case, must not be
//    NULL
//
/////////////////////////////////////////////////////


void GenerateSourceFile(PSTR pszGameDir)
{
    char szSourceFile[MAX_LOGIC_SRC_SIZE];
    char szBuffer[500];
    int nCurNestingLevel = 0;
    int nRemainingPosCommands;
    int nNumPosCommands;
    BOOL bHaveAbsolutePosItem;
    CDefineNameList* pdeflist;

    assert(pszGameDir != NULL);

    strcpy(szSourceFile, "");

    MakeIndentString();

    // here we go...start by generating the header comment, if there is one

    if (g_options.m_bGenerateHeader)
    {
        if (g_mlopt.m_szLogicTitle[0] == '\0')
        {
            sprintf(szBuffer, "Logic %d", g_mlopt.m_nLogicNumber);
        }
        else
        {
            sprintf(szBuffer, "Logic %d: %s", g_mlopt.m_nLogicNumber,
                    g_mlopt.m_szLogicTitle);
        }
        AddHeaderComment(szSourceFile, szBuffer, nCurNestingLevel);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    // add the #include "defines.txt" directive, if defines are being used
    if (g_options.m_bUseDefineNames)
    {
        strcat(szSourceFile, "#include \"defines.txt\"");
        strcat(szSourceFile, STR_END_OF_LINE);
        strcat(szSourceFile, STR_END_OF_LINE);

        // the define name list used for this source file is g_deflist,
        // so get a pointer to it (see the else branch of this if
        // statement for an explanation of why things are done this way)
        pdeflist = &g_deflist;
    }
    else
    {
        // make a dummy define name list where the names of all the vars
        // are v0, v1, v2, etc.; this simplifies handling the case of the
        // user choosing not to use defines.txt; the same logic generation
        // code can be used no matter how the user has set the options
        pdeflist = MakeNoDefinesDefList();
    }

    // generate the new room section
    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "if (");
    pdeflist->GetFlagName(AGI_FLAG_NEW_ROOM, szBuffer, 
                          MAX_DEFINE_NAME_LEN + 1);
    strcat(szSourceFile, szBuffer);
    strcat(szSourceFile, ")");

    GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

    nCurNestingLevel++;

    if (g_options.m_bGenerateBasicComments)
    {
        AddOneLineComment(szSourceFile, 
                          "this is the first cycle through this room",
                          nCurNestingLevel);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    if (!g_mlopt.m_bUseRoomNumForPic)
    {
        // if not using the variable room_no for the picture, then set up
        // the temporary variable that is to be used
        if (g_options.m_bGenerateVerboseComments)
        {
            AddOneLineComment(szSourceFile,
                              "since the pic number is not equal to the "
                              "room number,",
                              nCurNestingLevel);
            AddOneLineComment(szSourceFile,
                              "assign the pic number to a temporary variable",
                              nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "v255 = ");
        sprintf(szBuffer, "%d", g_mlopt.m_nPicNumber);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, ";");
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    if (g_mlopt.m_bUseRoomNumForPic)
    {
        pdeflist->GetVarName(AGI_VAR_ROOM_NO, szBuffer, 
                             MAX_DEFINE_NAME_LEN + 1);
        
    }
    else
    {
        strcpy(szBuffer, "v255");
    }

    if (g_options.m_bGenerateVerboseComments)
    {
        AddOneLineComment(szSourceFile, 
                          "load the picture resource for this room",
                          nCurNestingLevel);
    }

    // generate the picture loading, drawing, and discarding commands
    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "load.pic(");
    strcat(szSourceFile, szBuffer);
    strcat(szSourceFile, ");");
    strcat(szSourceFile, STR_END_OF_LINE);

    if (g_options.m_bGenerateVerboseComments)
    {
        AddOneLineComment(szSourceFile, 
                          "draw the picture resource for this room "
                          "to memory",
                          nCurNestingLevel);
    }

    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "draw.pic(");
    strcat(szSourceFile, szBuffer);
    strcat(szSourceFile, ");");
    strcat(szSourceFile, STR_END_OF_LINE);

    if (g_options.m_bGenerateVerboseComments)
    {
        AddOneLineComment(szSourceFile, 
                          "done with the picture resource for this room - "
                          "discard it", nCurNestingLevel);
    }

    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "discard.pic(");
    strcat(szSourceFile, szBuffer);
    strcat(szSourceFile, ");");
    strcat(szSourceFile, STR_END_OF_LINE);
    strcat(szSourceFile, STR_END_OF_LINE);

    // set the horizon, if appropriate
    if (g_mlopt.m_nHorizon != DEFAULT_HORIZON_VALUE)
    {
        if (g_options.m_bGenerateVerboseComments)
        {
            AddOneLineComment(szSourceFile,
                              "set the horizon value for this room",
                              nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "set.horizon(");
        sprintf(szBuffer, "%d", g_mlopt.m_nHorizon);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, ");");
        strcat(szSourceFile, STR_END_OF_LINE);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    // generate the ego positioning commands
    bHaveAbsolutePosItem = g_aepc.GetAbsolutePositioningItem() != NULL;

    // determine how many of these commands there are
    nNumPosCommands = g_aepc.GetElementCount() +
                      ((bHaveAbsolutePosItem) ? 1 : 0) +
                      ((g_mlopt.m_bFirstRoom) ? 1 : 0);

    // need to decrement the count and keep it, so need another var
    nRemainingPosCommands = nNumPosCommands;
    EGOPOSITIONCONTROL epc;

    // generate the first room positioning commands, if applicable
    if (g_mlopt.m_bFirstRoom)
    {
        nRemainingPosCommands--;

        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "if ((");
        pdeflist->GetVarName(AGI_VAR_PREV_ROOM_NO, szBuffer,
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, " == 0 || ");
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, " == 1))");

        GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

        nCurNestingLevel++;

        if (g_options.m_bGenerateBasicComments)
        {
            AddOneLineComment(szSourceFile,
                              "this is the first room in the game",
                              nCurNestingLevel);
            strcat(szSourceFile, STR_END_OF_LINE);
        }

        if (g_frctrl.m_nEgoX != NO_POS_X &&
            g_frctrl.m_nEgoY != NO_POS_Y)
        {
            if (g_options.m_bGenerateVerboseComments)
            {
                AddOneLineComment(szSourceFile,
                                  "set ego's position for the beginning of "
                                  "the game",
                                  nCurNestingLevel);
            }

            Indent(szSourceFile, nCurNestingLevel);
            strcat(szSourceFile, "position(");
            pdeflist->GetObjName(AGI_OBJ_EGO, szBuffer, 
                                 MAX_DEFINE_NAME_LEN + 1);
            strcat(szSourceFile, szBuffer);
            sprintf(szBuffer, ", %d, %d);", 
                    static_cast<BYTE>(g_frctrl.m_nEgoX),
                    static_cast<BYTE>(g_frctrl.m_nEgoY));
            strcat(szSourceFile, szBuffer);
            strcat(szSourceFile, STR_END_OF_LINE);
        }

        // rather than add more if (FIRST ROOM) type stuff later on, just
        // care of all of it right here...turn on the status bar if
        // applicable and same with accepting input
        if (g_frctrl.m_bStatusBarOn)
        {
            if (g_options.m_bGenerateVerboseComments)
            {
                AddOneLineComment(szSourceFile,
                                  "turn on the status bar",
                                  nCurNestingLevel);
            }

            Indent(szSourceFile, nCurNestingLevel);
            strcat(szSourceFile, "status.line.on();");
            strcat(szSourceFile, STR_END_OF_LINE);
        }

        if (g_frctrl.m_bStatusBarOn)
        {
            if (g_options.m_bGenerateVerboseComments)
            {
                AddOneLineComment(szSourceFile,
                                  "allow the player to enter input",
                                  nCurNestingLevel);
            }

            Indent(szSourceFile, nCurNestingLevel);
            strcat(szSourceFile, "accept.input();");
            strcat(szSourceFile, STR_END_OF_LINE);
        }

        if (g_options.m_bGenerateTODOComments)
        {
            AddOneLineComment(szSourceFile,
                              "TODO: add one-time game initialization here",
                              nCurNestingLevel);
            strcat(szSourceFile, STR_END_OF_LINE);
        }

        nCurNestingLevel--;

        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "}");
        strcat(szSourceFile, STR_END_OF_LINE);

        if (nRemainingPosCommands > 0)
        {
            Indent(szSourceFile, nCurNestingLevel);
            strcat(szSourceFile, "else");

            GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

            nCurNestingLevel++;
        }
        else
        {
            strcat(szSourceFile, STR_END_OF_LINE);
        }

    }

    // generate the rest of the positionign commands
    for (int i = 0; i < g_aepc.GetElementCount(); i++)
    {
        epc = g_aepc[i];

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "if (");
        pdeflist->GetVarName(AGI_VAR_PREV_ROOM_NO, szBuffer, 
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(szSourceFile, szBuffer);
        sprintf(szBuffer, " == %d)", static_cast<BYTE>(epc.nSrcRoom));
        strcat(szSourceFile, szBuffer);

        GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

        nCurNestingLevel++;

        if (g_options.m_bGenerateVerboseComments)
        {
            sprintf(szBuffer, "coming from room %d, position ego at "
                              "(%d, %d)", epc.nSrcRoom, epc.nPosX,
                              epc.nPosY);
            AddOneLineComment(szSourceFile, szBuffer, nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "position(");
        pdeflist->GetObjName(AGI_OBJ_EGO, szBuffer, 
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(szSourceFile, szBuffer);
        sprintf(szBuffer, ", %d, %d);", epc.nPosX, epc.nPosY);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, STR_END_OF_LINE);

        nCurNestingLevel--;

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "}");
        strcat(szSourceFile, STR_END_OF_LINE);

        nRemainingPosCommands--;

        if (nRemainingPosCommands > 0)
        {
            Indent(szSourceFile, nCurNestingLevel);
            strcat(szSourceFile, "else");

            GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

            nCurNestingLevel++;
        }
    }

    // generate the absolute positioning command
    if (g_aepc.GetAbsolutePositioningItem() != NULL)
    {
        epc = *(g_aepc.GetAbsolutePositioningItem());

        if (g_options.m_bGenerateVerboseComments)
        {
            if (nNumPosCommands == 1)
            {
                sprintf(szBuffer, 
                        "coming from any room, position ego "
                        "at (%d, %d)", epc.nPosX, epc.nPosY);
            }
            else
            {
                sprintf(szBuffer,
                        "coming from any other room, position ego "
                        "at (%d, %d)", epc.nPosX, epc.nPosY);
            }

            AddOneLineComment(szSourceFile, szBuffer, nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "position(");
        pdeflist->GetObjName(AGI_OBJ_EGO, szBuffer, 
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(szSourceFile, szBuffer);
        sprintf(szBuffer, ", %d, %d);", epc.nPosX, epc.nPosY);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    // now close off all the braces that had to be opened
    for (i = 0; i < (nNumPosCommands - 1); i++)
    {
        nCurNestingLevel--;
        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "}");
        strcat(szSourceFile, STR_END_OF_LINE);
        
        if (i == (nNumPosCommands - 2))
        {
            strcat(szSourceFile, STR_END_OF_LINE);
        }
    }

    if (g_options.m_bGenerateTODOComments)
    {
        AddOneLineComment(szSourceFile,
                          "TODO: add additional room initialization here",
                          nCurNestingLevel);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    // add draw(ego) code here, if necessary
    if (g_mlopt.m_bDrawEgoInitially)
    {
        if (g_options.m_bGenerateVerboseComments)
        {
            AddOneLineComment(szSourceFile,
                              "draw ego on the screen",
                              nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "draw(");
        pdeflist->GetObjName(AGI_OBJ_EGO, szBuffer,
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(szSourceFile, szBuffer);
        strcat(szSourceFile, ");");
        strcat(szSourceFile, STR_END_OF_LINE);
    }
    else
    {
        if (g_options.m_bGenerateWarningComments)
        {
            AddOneLineComment(szSourceFile,
                              "NOTE: you have chosen not to draw ego "
                              "during", nCurNestingLevel);
            AddOneLineComment(szSourceFile,
                              "      the first cycle through the room",
                              nCurNestingLevel);
            strcat(szSourceFile, STR_END_OF_LINE);
        }
    }

    // add the show.pic() command
    if (g_options.m_bGenerateVerboseComments)
    {
        AddOneLineComment(szSourceFile,
                          "draw the visual screen in memory to the "
                          "actual computer screen",
                          nCurNestingLevel);
    }

    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "show.pic();");
    strcat(szSourceFile, STR_END_OF_LINE);
    strcat(szSourceFile, STR_END_OF_LINE);

    // add the print(ENTRY MESSAGE) command here
    if (g_elinfo.m_szEntryMessage[0] != '\0')
    {
        if (g_options.m_bGenerateVerboseComments)
        {
            AddOneLineComment(szSourceFile,
                              "print entry message;",
                              nCurNestingLevel);
        }

        if (g_options.m_bGenerateWarningComments)
        {
            AddOneLineComment(szSourceFile,
                              "NOTE: entry message is printed every time "
                              "player enters the room",
                              nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "print(\"");
        strcat(szSourceFile, g_elinfo.m_szEntryMessage);
        strcat(szSourceFile, "\");");
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    nCurNestingLevel--;

    Indent(szSourceFile, nCurNestingLevel);

    strcat(szSourceFile, "}");
    strcat(szSourceFile, STR_END_OF_LINE);
    strcat(szSourceFile, STR_END_OF_LINE);

    // generate all the edge code stuff in a separate function, since it's
    // all very similar to each other
    GenerateEdgeCodeBlock(szSourceFile, pdeflist,
                          AGI_CONST_LEFT_EDGE, nCurNestingLevel);
    GenerateEdgeCodeBlock(szSourceFile, pdeflist,
                          AGI_CONST_RIGHT_EDGE, nCurNestingLevel);
    GenerateEdgeCodeBlock(szSourceFile, pdeflist,
                          AGI_CONST_BOTTOM_EDGE, nCurNestingLevel);
    GenerateEdgeCodeBlock(szSourceFile, pdeflist,
                          AGI_CONST_HORIZON_EDGE, nCurNestingLevel);

    // add the input parsing section
    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "if (");
    pdeflist->GetFlagName(AGI_FLAG_INPUT_RECEIVED, szBuffer,
                          MAX_DEFINE_NAME_LEN + 1);
    strcat(szSourceFile, szBuffer);
    pdeflist->GetFlagName(AGI_FLAG_INPUT_PARSED, szBuffer,
                          MAX_DEFINE_NAME_LEN + 1);
    strcat(szSourceFile, " && !");
    strcat(szSourceFile, szBuffer);
    pdeflist->GetVarName(AGI_VAR_UNKNOWN_WORD_NO, szBuffer,
                         MAX_DEFINE_NAME_LEN + 1);
    strcat(szSourceFile, " && ");
    strcat(szSourceFile, szBuffer);
    strcat(szSourceFile, " == 0)");

    GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

    nCurNestingLevel++;

    if (g_options.m_bGenerateBasicComments)
    {
        AddOneLineComment(szSourceFile,
                          "input parsing section",
                          nCurNestingLevel);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    // add the test for if (said("look")), if applicable
    if (g_elinfo.m_szLookMessage[0] != '\0')
    {
        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "if (said(\"look\"))");

        GenerateIfElseOpenBrace(szSourceFile, nCurNestingLevel);

        nCurNestingLevel++;

        if (g_options.m_bGenerateVerboseComments)
        {
            AddOneLineComment(szSourceFile,
                              "user entered \"look\", so print the "
                              "look message",
                              nCurNestingLevel);
        }

        Indent(szSourceFile, nCurNestingLevel);

        strcat(szSourceFile, "print(\"");
        strcat(szSourceFile, g_elinfo.m_szLookMessage);
        strcat(szSourceFile, "\");");
        strcat(szSourceFile, STR_END_OF_LINE);

        nCurNestingLevel--;

        Indent(szSourceFile, nCurNestingLevel);
        strcat(szSourceFile, "}");
        strcat(szSourceFile, STR_END_OF_LINE);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    if (g_options.m_bGenerateTODOComments)
    {
        AddOneLineComment(szSourceFile,
                          "TODO: add additional input parsing "
                          "statements here",
                          nCurNestingLevel);
        strcat(szSourceFile, STR_END_OF_LINE);
    }

    nCurNestingLevel--;

    Indent(szSourceFile, nCurNestingLevel);
    strcat(szSourceFile, "}");
    strcat(szSourceFile, STR_END_OF_LINE);
    strcat(szSourceFile, STR_END_OF_LINE);


    // add the return command
    if (g_options.m_bGenerateWarningComments)
    {
        AddOneLineComment(szSourceFile,
                          "NOTE: the return command is required by AGI",
                          nCurNestingLevel);
    }

    assert(nCurNestingLevel == 0);

    strcat(szSourceFile, "return();");
    strcat(szSourceFile, STR_END_OF_LINE);

    if (!g_options.m_bUseDefineNames)
    {
        delete pdeflist;
    }

    char szFileLocation[_MAX_PATH + 1];

    // write the file, if that's what the user wants
    if (g_options.m_bWriteToFile)
    {
        sprintf(szFileLocation, "%s\\src\\logic%d.txt", pszGameDir,
                g_mlopt.m_nLogicNumber);

        FILE* file;
        file = fopen(szFileLocation, "wb");
        if (file)
        {
            fwrite(szSourceFile, 1, strlen(szSourceFile), file);
        }
        else
        {
            MessageBox(NULL, "could not open file for logic source",
                       "error", MB_OK);
        }

        fclose(file);
    }

    // write the text to the clipboard, if that's what the user wants
    if (g_options.m_bWriteToClipboard)
    {
        WriteTextToClipboard(szSourceFile);
    }
}

/////////////////////////////////////////////////////
//
// MakeNoDefinesDefList
//
/////////////////////////////////////////////////////
//
// Purpose:
//    creates a define name list where the variables,
//    flags, etc. have their no-define-name names, 
//    such as v0, f255, etc.; this simplifies logic
//    generation since the same code can be used to
//    generate the source code whether the user wants
//    defines or not
// Return value:
//    a pointer to a new CDefineNameList object that
//    must be deleted by the caller
//
/////////////////////////////////////////////////////

CDefineNameList* MakeNoDefinesDefList()
{
    CDefineNameList* pdeflist = new CDefineNameList();
    char szBuffer[15];

    ULONG ul;

    if (!pdeflist)
    {
        throw CMallocFailedException();
    }

    for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
    {
        sprintf(szBuffer, "v%d", ul);
        pdeflist->SetVarName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
    {
        sprintf(szBuffer, "f%d", ul);
        pdeflist->SetFlagName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
    {
        sprintf(szBuffer, "s%d", ul);
        pdeflist->SetStrName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
    {
        sprintf(szBuffer, "%d", g_anAGIConstValues[ul]);
        pdeflist->SetConstName(ul, szBuffer);
    }

    for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
    {
        sprintf(szBuffer, "o%d", ul);
        pdeflist->SetObjName(ul, szBuffer);
    }
    return pdeflist;
}

/////////////////////////////////////////////////////
//
// Indent
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds appropriate indenting characters to a
//    string, based on the user's options and the
//    current nesting level
// Parameter pszTarget:
//    a pointer to a string buffer onto which the
//    indent string will be concatenated; must not be
//    NULL
// Parameter nNestLevel:
//    specifies the current nesting level of the
//    source code, which indicates how many instances
//    of the indent string should be appended to the
//    string buffer; must be >= 0
//
/////////////////////////////////////////////////////

void Indent(PSTR pszTarget, int nNestLevel)
{
    assert(pszTarget != NULL);
    // asserting that the nest level is >= 0 isn't necessary for
    // stopping bad things from happening (the loop condition takes
    // care of that), but it might help find bugs in the source
    // code generation logic
    assert(nNestLevel >= 0);

    for (int i = 0; i < nNestLevel; i++)
    {
        strcat(pszTarget, g_szIndentString);
    }
}

/////////////////////////////////////////////////////
//
// MakeIndentString
//
/////////////////////////////////////////////////////
//
// Purpose:
//    makes an appropriate indention string, based
//    on the user's options such as Tabs or Spaces, etc.
// Remarks:
//    only needs to be called once per file generation
//
/////////////////////////////////////////////////////

void MakeIndentString()
{
    assert(g_options.m_nIndenterCount < MAX_INDENT_STRING_LEN);

    char szIndenter[2];

    if (g_options.m_nTabsOrSpaces == OPT_TOS_TABS)
    {
        strcpy(szIndenter, "\t");
    }
    else
    {
        strcpy(szIndenter, " ");
    }

    strcpy(g_szIndentString, "");

    for (int i = 0; i < g_options.m_nIndenterCount; i++)
    {
        strcat(g_szIndentString, szIndenter);
    }
}

/////////////////////////////////////////////////////
//
// AddHeaderComment
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds a header comment to the logic file, using
//    the user's preferences for the comment style
// Parameter pszTarget:
//    a string buffer to which the header comment will
//    be added; must not be NULL
// Parameter pszCommentText:
//    the text that will appear inside the header comment;
//    may be empty but must not be NULL
// Parameter nNestLevel:
//    the current nesting level in the source code;
//    must be >= 0
//
/////////////////////////////////////////////////////


void AddHeaderComment(PSTR pszTarget, PSTR pszCommentText, int nNestLevel)
{
    assert(pszTarget != NULL);
    assert(pszCommentText != NULL);
    assert(nNestLevel >= 0);

    Indent(pszTarget, nNestLevel);

    switch(g_options.m_nHeaderCommentStyle)
    {
    case OPT_HCS_BOUND_BY_STRING_OF_SLASHES:
        strcat(pszTarget, 
               "/////////////////////////////////////////////////////");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "// ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget,
               "/////////////////////////////////////////////////////");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_HCS_BOUND_BY_MULTILINE:
        strcat(pszTarget, "/*");
        strcat(pszTarget, STR_END_OF_LINE);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "*/");
        strcat(pszTarget, STR_END_OF_LINE);
        break;
        
    case OPT_HCS_BOUND_BY_DOUBLE_SLASH_THEN_ASTERISKS:
        strcat(pszTarget,
               "// **************************************************");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "// ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget,
               "// **************************************************");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_HCS_START_WITH_DOUBLE_SLASH:
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "// ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "//");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_HCS_START_WITH_LEFT_BRACKET:
        strcat(pszTarget, "[");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "[");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "[ ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "[");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "[");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_HCS_ENCLOSE_WITH_MULTILINE:
        int nIdealLen;

        nIdealLen = 
            strlen("/*                                                 */") -
            (strlen("/* ") + strlen(pszCommentText) + strlen(" */"));

        strcat(pszTarget,
               "/*                                                 */");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget,
               "/*                                                 */");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "/* ");
        strcat(pszTarget, pszCommentText);
        if (nIdealLen < 0)
        {
            strcat(pszTarget, " */");
        }
        else
        {
            while(nIdealLen >= 0)
            {
                strcat(pszTarget, " ");
                nIdealLen--;
            }

            strcat(pszTarget, "*/");
        }
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget,
               "/*                                                 */");
        strcat(pszTarget, STR_END_OF_LINE);

        Indent(pszTarget, nNestLevel);
        strcat(pszTarget,
               "/*                                                 */");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_HCS_ONE_LINE_HEADER:
        AddOneLineComment(pszTarget, pszCommentText, nNestLevel);
        break;

    default:
        assert(FALSE);
        break;
    }

}

/////////////////////////////////////////////////////
//
// AddOneLineComment
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds a one-line comment to the logic file, using
//    the user's preferences for the comment style
// Parameter pszTarget:
//    a string buffer to which the one-line comment will
//    be added; must not be NULL
// Parameter pszCommentText:
//    the text that will appear inside the one-line comment;
//    may be empty but must not be NULL
// Parameter nNestLevel:
//    the current nesting level in the source code;
//    must be >= 0
//
/////////////////////////////////////////////////////

void AddOneLineComment(PSTR pszTarget, PSTR pszCommentText, int nNestLevel)
{
    assert(pszTarget != NULL);
    assert(pszCommentText != NULL);

    Indent(pszTarget, nNestLevel);

    switch(g_options.m_nOneLineCommentStyle)
    {
    case OPT_OLCS_DOUBLE_SLASH:
        strcat(pszTarget, "// ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_OLCS_LEFT_BRACKET:
        strcat(pszTarget, "[ ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_OLCS_MULTILINE:
        strcat(pszTarget, "/* ");
        strcat(pszTarget, pszCommentText);
        strcat(pszTarget, " */");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    default:
        assert(FALSE);
        break;
    }

}

/////////////////////////////////////////////////////
//
// GenerateEdgeCodeBlock
//
/////////////////////////////////////////////////////
//
// Purpose:
//    generates source code for an ego-touching-edge
//    handler
// Parameter pszTarget:
//    a string buffer to which the code will
//    be added; must not be NULL
// Parameter pdeflist:
//    a pointer to the define name list that is to be
//    used to generate the source code
// Parameter ulEdgeConst:
//    the edge for which this code is being generated
// Parameter nNestLevel:
//    the current nesting level in the source code;
//    must be >= 0
//
/////////////////////////////////////////////////////

void GenerateEdgeCodeBlock(PSTR pszTarget, CDefineNameList* pdeflist,
                           ULONG ulEdgeConst, int nNestLevel)
{
    assert(pszTarget != NULL);
    assert(pdeflist != NULL);

    int nGotoRoom;
    PSTR pszMessage;
    BOOL bEmptyCodeBlock;
    char szDirection[15];

    switch(ulEdgeConst)
    {
    case AGI_CONST_LEFT_EDGE:
        nGotoRoom = g_ecinfo.m_nLeftGotoRoom;
        pszMessage = g_ecinfo.m_szLeftMessage;
        bEmptyCodeBlock = g_ecinfo.m_bEmptyLeft;
        strcpy(szDirection, "left");
        break;
    case AGI_CONST_RIGHT_EDGE:
        nGotoRoom = g_ecinfo.m_nRightGotoRoom;
        pszMessage = g_ecinfo.m_szRightMessage;
        bEmptyCodeBlock = g_ecinfo.m_bEmptyRight;
        strcpy(szDirection, "right");
        break;
    case AGI_CONST_BOTTOM_EDGE:
        nGotoRoom = g_ecinfo.m_nBottomGotoRoom;
        pszMessage = g_ecinfo.m_szBottomMessage;
        bEmptyCodeBlock = g_ecinfo.m_bEmptyBottom;
        strcpy(szDirection, "bottom");
        break;
    case AGI_CONST_HORIZON_EDGE:
        nGotoRoom = g_ecinfo.m_nHorizonGotoRoom;
        pszMessage = g_ecinfo.m_szHorizonMessage;
        bEmptyCodeBlock = g_ecinfo.m_bEmptyHorizon;
        strcpy(szDirection, "horizon");
        break;
    default:
        assert(FALSE);
        break;
    }

    char szBuffer[MAX_DEFINE_NAME_LEN + 1];

    if (nGotoRoom != NO_GOTO_ROOM ||
        pszMessage[0] != '\0' ||
        bEmptyCodeBlock)
    {
        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "if (");
        pdeflist->GetVarName(AGI_VAR_EGO_EDGE_CODE, szBuffer,
                             MAX_DEFINE_NAME_LEN + 1);
        strcat(pszTarget, szBuffer);
        strcat(pszTarget, " == ");
        pdeflist->GetConstName(ulEdgeConst, szBuffer,
                               MAX_DEFINE_NAME_LEN + 1);
        strcat(pszTarget, szBuffer);
        strcat(pszTarget, ")");

        GenerateIfElseOpenBrace(pszTarget, nNestLevel);

        nNestLevel++;
        
        if (g_options.m_bGenerateBasicComments)
        {
            sprintf(szBuffer, "ego is touching the %s edge", szDirection);
            AddOneLineComment(pszTarget, szBuffer, nNestLevel);
            strcat(pszTarget, STR_END_OF_LINE);
        }

        if (g_options.m_bGenerateTODOComments)
        {
            sprintf(szBuffer, 
                    "TODO: add additional code for ego touching %s edge here",
                    szDirection);
            AddOneLineComment(pszTarget, szBuffer, nNestLevel);
            strcat(pszTarget, STR_END_OF_LINE);
        }

        if (pszMessage[0] != '\0')
        {
            if (g_options.m_bGenerateVerboseComments)
            {
                sprintf(szBuffer, "print message for %s edge", szDirection);
                AddOneLineComment(pszTarget, szBuffer, nNestLevel);
            }

            Indent(pszTarget, nNestLevel);
            strcat(pszTarget, "print(\"");
            strcat(pszTarget, pszMessage);
            strcat(pszTarget, "\");");
            strcat(pszTarget, STR_END_OF_LINE);

            if (nGotoRoom == NO_GOTO_ROOM)
            {
                if (g_options.m_bGenerateVerboseComments)
                {
                    AddOneLineComment(pszTarget,
                                      "stop ego motion so message doesn't "
                                      "keep printing forever", nNestLevel);
                }

                Indent(pszTarget, nNestLevel);
                pdeflist->GetVarName(AGI_VAR_EGO_DIR, szBuffer,
                                     MAX_DEFINE_NAME_LEN + 1);

                strcat(pszTarget, szBuffer);
                strcat(pszTarget, " = ");
                pdeflist->GetConstName(AGI_CONST_STOPPED, szBuffer,
                                       MAX_DEFINE_NAME_LEN + 1);
                strcat(pszTarget, szBuffer);
                strcat(pszTarget, ";");
                strcat(pszTarget, STR_END_OF_LINE);
            }
        }

        if (nGotoRoom != NO_GOTO_ROOM)
        {
            if (g_options.m_bGenerateVerboseComments)
            {
                sprintf(szBuffer, "when ego touches %s edge, go to "
                                  "room %d", szDirection, nGotoRoom);
                AddOneLineComment(pszTarget, szBuffer, nNestLevel);
            }

            Indent(pszTarget, nNestLevel);
            sprintf(szBuffer, "new.room(%d);", nGotoRoom);
            strcat(pszTarget, szBuffer);
            strcat(pszTarget, STR_END_OF_LINE);
        }

        nNestLevel--;
        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "}");
        strcat(pszTarget, STR_END_OF_LINE);
        strcat(pszTarget, STR_END_OF_LINE);
    }
}

/////////////////////////////////////////////////////
//
// GenerateIfElseOpenBrace
//
/////////////////////////////////////////////////////
//
// Purpose:
//    generates an open brace for a command block,
//    based on the user's preferences
// Parameter pszTarget:
//    a string buffer to which the code will
//    be added; must not be NULL
// Parameter nNestLevel:
//    the current nesting level in the source code;
//    must be >= 0
//
/////////////////////////////////////////////////////

void GenerateIfElseOpenBrace(PSTR pszTarget, int nNestLevel)
{
    assert(pszTarget != NULL);

    switch(g_options.m_nCurlyBraceStyle)
    {
    case OPT_CBS_NEXT_LINE:
        strcat(pszTarget, STR_END_OF_LINE);
        Indent(pszTarget, nNestLevel);
        strcat(pszTarget, "{");
        strcat(pszTarget, STR_END_OF_LINE);
        break;

    case OPT_CBS_END_OF_LINE:
        strcat(pszTarget, " {");
        strcat(pszTarget, STR_END_OF_LINE);
        break;
    }
}
