/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#include "StdAfx.h"
#include "soundresource.h"
#include "DataStream.h"
#include "Exception.h"

SoundResource::SoundResource(void)
{
	for (int i = 0; i < 16; ++i) {
		mChannels[i].initialVoices = 0;
		mChannels[i].playFlags = 0;
	}
}

SoundResource::~SoundResource(void)
{
}


/*
 * Load an extracted sound resource. The DataStream must start out pointing to
 * the two byte 84 00 magic number.
 *
 * throws EofException and LoadException
 */
void SoundResource::Load(DataStream &in)
{
	int i;

	// load into these temp variables so the sound state can stay the same if
	//   there is an error while loading
	SoundEventList events;
	sound_channel channels[16];


	// check for the magic number signifying a sound resourced
	if (in.ReadShort() != 0x0084) throw LoadException("Not an SCI0 sound resource");

	// make sure this is a normal sound resource, without a digital sample
	if (in.ReadChar() != 0) throw LoadException("Resources with digital samples are not supported");

	// read channel header information
	for (i = 0; i < 16; ++i) {
		channels[i].initialVoices = in.ReadChar();
		channels[i].playFlags = in.ReadChar();
	}

	// build the event list
	int soundTime = 0;
	unsigned char status = 0;
	while (status != 0xFC) {
		sound_event event;

		// read a delta time, handling extensions
		int deltaTime = 0;
		unsigned char deltaByte = in.ReadChar();
		while (deltaByte == 0xF8) {
			deltaTime += 240;
			deltaByte = in.ReadChar();
		}
		deltaTime += deltaByte;
		soundTime += deltaTime;

		// check for a new status
		if (in.PeekChar() & 0x80) status = in.ReadChar();
		if (status == 0) throw LoadException("Running status invoked without a previous status");

		// pre-fill the event struct with expected values
		event.absoluteTime = soundTime;
		event.status = status & 0xF0;
		event.channel = status & 0x0F;
		event.param1 = 0xFF;
		event.param2 = 0xFF;

		// handle the status appropriately
		switch (status & 0xF0) {
			// normal, 2-parameter messages
			case sound_event::NOTE_OFF:
			case sound_event::NOTE_ON:
			case sound_event::KEY_AFTERTOUCH:
			case sound_event::CONTROL_CHANGE:
			case sound_event::PITCH_WHEEL:
				event.param1 = in.ReadChar();
				event.param2 = in.ReadChar();
				break;

			// normal, 1-parameter messages
			case sound_event::PATCH_CHANGE:
			case sound_event::CHANNEL_AFTERTOUCH:
				event.param1 = in.ReadChar();
				break;

			// the only statuses left are system messages
			default:
				// stop sound
				if (status == sound_event::STOP) {
					event.status = status;
					event.channel = 0xFF;
				} else {
					throw LoadException("Unknown status");
				}
		}

		events.AppendEvent(event);
	}

	// no errors, so update our state
	mEvents = events;
	for (i = 0; i < 16; ++i) mChannels[i] = channels[i];
}


/*
 * Import a standard MIDI file. Only type 0 and type 1 files are supported.
 *
 * Conversion is a two-pass process. The first pass reads MIDI tracks one at
 * a time and merges them into a single, time-sorted list of events. The
 * second pass scans over the merged list changing from MIDI time to sound
 * time.
 *
 * throws EofException, DataException, and LoadException
 */
enum time_format {SMPTE, BPM};
void SoundResource::ImportMidi(DataStream &in)
{
	int i;

	// a list to hold the merged MIDI tracks
	SoundEventList midiEvents;

	// check for 'MThd'
	if (in.ReadBigLong() !=  0x4D546864) throw LoadException("Not a MIDI file");

	// check for header size of 6 bytes
	if (in.ReadBigLong() != 6) throw LoadException("Bad MIDI header size");

	// check format
	short format = in.ReadBigShort();
	if ((format != 0) && (format != 1)) throw LoadException("Only format 0 and 1 MIDI files are supported");

	// read number of tracks
	short numTracks = in.ReadBigShort();

	// read timing information
	time_format timeFormat;
	int ticksPerSec;
	int tempo = 500000;   // default of 120 bpm
	int resolution;
	unsigned short timeInfo = in.ReadBigShort();
	if (timeInfo & 0x8000) {
		timeFormat = SMPTE;
		int frames = (char)(timeInfo >> 8) * -1;
		int subFrames = timeInfo & 0xFF;
		ticksPerSec = frames * subFrames;
	} else {
		timeFormat = BPM;
		resolution = timeInfo;
	}

	// read and merge tracks
	int biggestTrackTime = 0;
	for (i = 0; i < numTracks; ++i) {
		// check for 'MTrk'
		if (in.ReadBigLong() != 0x4D54726B) throw LoadException("Missing 'MTrk'");

		// read track length
		int trackLength = in.ReadBigLong();
		int nextTrackOffset = in.Tellg() + trackLength;

		// read track events
		int trackTime = 0;
		unsigned char status = 0;
		while (in.Tellg() != nextTrackOffset) {
			sound_event event;

			// read delta time
			trackTime += in.ReadVariableLength();

			// check for a new status
			if (in.PeekChar() & 0x80) status = in.ReadChar();
			if (status == 0) throw LoadException("Running status invoked without a previous status");

			// pre-fill the event struct with expected values
			event.absoluteTime = trackTime;
			event.status = status & 0xF0;
			event.channel = status & 0x0F;
			event.param1 = 0xFF;
			event.param2 = 0xFF;

			// handle the status appropriately
			int metaEvent;
			int metaEventLength;
			unsigned long newTempo;
			switch (status & 0xF0) {
				// normal, 2-parameter messages
				case sound_event::NOTE_OFF:
				case sound_event::NOTE_ON:
				case sound_event::KEY_AFTERTOUCH:
				case sound_event::CONTROL_CHANGE:
				case sound_event::PITCH_WHEEL:
					event.param1 = in.ReadChar();
					event.param2 = in.ReadChar();
					midiEvents.InsertEvent(event);
					break;

				// normal, 1-parameter messages
				case sound_event::PATCH_CHANGE:
				case sound_event::CHANNEL_AFTERTOUCH:
					event.param1 = in.ReadChar();
					midiEvents.InsertEvent(event);
					break;

				default:
					switch (status & 0x0F) {
						// sysex
						case 0:
						case 7:
							in.Seekg(in.ReadVariableLength());   // skip the whole sysex message
							break;

						// meta-event
						case 15:
							metaEvent = in.ReadChar();
							metaEventLength = in.ReadVariableLength();

							if (metaEvent == 0x2F) {
								// end of track
								in.Seekg(nextTrackOffset, DataStream::BEG);
							} else if (metaEvent == 0x51) {
								// tempo change
								newTempo = in.ReadBigTri();

								// We need to remember this tempo change event so we can
								//   properly adjust from MIDI file time to sound resource
								//   time in the next pass. To remember it, we simply
								//   insert it into the events list. The new tempo (which
								//   is three bytes wide) gets sliced and stored in a
								//   sound event structure. We will not add it to the
								//   final event list during the second pass; it is not a
								//   proper sound event for sound resources. The constant
								//   TEMPO_CHANGE is just a placeholder and is not a real
								//   MIDI status.
								event.status = sound_event::TEMPO_CHANGE;
								event.channel = (unsigned char)((newTempo >> 16) & 0xFF);
								event.param1 = (unsigned char)((newTempo >> 8) & 0xFF);
								event.param2 = (unsigned char)(newTempo & 0xFF);
								midiEvents.InsertEvent(event);
							} else {
								// skip all other meta-events
								in.Seekg(metaEventLength);
							}
							break;

						// ignored
						case 2:
							in.Seekg(2);
							break;

						// ignored
						case 3:
							in.Seekg(1);
							break;

						// ignored
						case 6:
						case 8:
						case 10:
						case 11:
						case 12:
						case 14:
							break;

						// unknown / invalid
						// cases 1, 4, 5, 9, 13
						default:
							throw LoadException("Invliad status");
					}
			}
		}
		if (trackTime > biggestTrackTime) biggestTrackTime = trackTime;
	}

	// add a STOP event at the end of the merged track list
	sound_event event;
	event.absoluteTime = biggestTrackTime;
	event.status = sound_event::STOP;
	event.channel = 0xFF;
	event.param1 = 0xFF;
	event.param2 = 0xFF;
	midiEvents.AppendEvent(event);

	// convert from MIDI time to sound time
	mEvents.Clear();
	int lastMidiTime = 0;
	int soundTime = 0;
	int remainder = 0;
	int remainder2 = 0;
	for (i = 0; i < midiEvents.GetLength(); ++i) {
		sound_event event = midiEvents.GetEvent(i);

		// calculate the new sound time based on the MIDI delta time
		int deltaMidi = event.absoluteTime - lastMidiTime;
		lastMidiTime = event.absoluteTime;
		if (timeFormat == SMPTE) {
			int numerator = deltaMidi * 60 + remainder;
			int deltaSound = numerator / ticksPerSec;
			remainder = numerator % ticksPerSec;
			soundTime += deltaSound;
		} else {   // timeFormat == BPM
			int numerator = deltaMidi * tempo + remainder;
			int usecs = numerator / resolution + remainder2;
			remainder = numerator % resolution;
			int deltaSound = usecs / 16667;
			remainder2 = usecs % 16667;
			soundTime += deltaSound;
		}
		event.absoluteTime = soundTime;

		if (event.status == sound_event::TEMPO_CHANGE) {
			// for tempo changes, update the tempo setting
			tempo = (event.channel << 16) | (event.param1 << 8) | event.param2;
		} else {
			// store the adjusted event
			mEvents.AppendEvent(event);
		}
	}

	// set default channel header information
	for (i = 0; i < 16; ++i) {
		mChannels[i].playFlags = 0x80;
		mChannels[i].initialVoices = 0;
	}
	mChannels[9].initialVoices = 128;   // FIXME: This is common for the percussion channel in Sierra's sounds. Why?
	mChannels[15].playFlags = 0;   // discourage use of channel 15
}


/*
 * Save the sound resource to a stream.
 */
void SoundResource::Save(DataStream &out) const
{
	int i;

	// save the magic number
	out.WriteShort(0x84);

	// save the header
	out.WriteChar(0);
	for (i = 0; i < 16; ++i) {
		out.WriteChar(mChannels[i].initialVoices);
		out.WriteChar(mChannels[i].playFlags);
	}

	// save events
	int lastEventTime = 0;
	unsigned char lastStatus = 0;
	for (i = 0; i < mEvents.GetLength(); ++i) {
		sound_event event = mEvents.GetEvent(i);

		// store the delta time
		int deltaTime = event.absoluteTime - lastEventTime;
		while (deltaTime >= 240) {
			out.WriteChar(0xF8);
			deltaTime -= 240;
		}
		out.WriteChar(deltaTime);
		lastEventTime = event.absoluteTime;

		// store the status if it's different from the last
		unsigned char status = ((event.status & 0xF0) == 0xF0) ? event.status : (event.status | event.channel);
		if (status != lastStatus) {
			lastStatus = status;
			out.WriteChar(status);
		}

		// store parameters
		switch (event.status) {
			// 2-parameter events
			case sound_event::NOTE_OFF:
			case sound_event::NOTE_ON:
			case sound_event::KEY_AFTERTOUCH:
			case sound_event::CONTROL_CHANGE:
			case sound_event::PITCH_WHEEL:
				out.WriteChar(event.param1);
				out.WriteChar(event.param2);
				break;

			// 1-parameter events
			case sound_event::PATCH_CHANGE:
			case sound_event::CHANNEL_AFTERTOUCH:
				out.WriteChar(event.param1);
				break;

			// 0-parameter events
			case sound_event::STOP:
				break;

			default:
				// just in case
				ASSERT("Unrecognized status");
		}
	}
}


/*
 * Get the absolute time in ticks of the last sound event.
 */
int SoundResource::GetTotalTime() const
{
	if (mEvents.GetLength() == 0) return 0;
	return mEvents.GetEvent(mEvents.GetLength() - 1).absoluteTime;
}


/*
 * Get the number of events in the event list.
 */
int SoundResource::GetNumEvents() const
{
	return mEvents.GetLength();
}


/*
 * Get the play flags for channel.
 */
unsigned char SoundResource::GetChannelPlayFlag(int channel) const
{
	return mChannels[channel].playFlags;
}


/*
 * Set the play flags for channel.
 */
void SoundResource::SetChannelPlayFlag(int channel, unsigned char flag)
{
	mChannels[channel].playFlags = flag;
}


/*
 * Get the number of voices allocated to channel by the header.
 */
unsigned char SoundResource::GetChannelVoices(int channel) const
{
	return mChannels[channel].initialVoices;
}


/*
 * Set the number of voices allocated to channel by the header.
 */
void SoundResource::SetChannelVoices(int channel, unsigned char voices)
{
	mChannels[channel].initialVoices = voices;
}


/*
 * Get the sound event at index i in the event list.
 */
sound_event SoundResource::GetEvent(int i) const
{
	return mEvents.GetEvent(i);
}


/*
 * Insert an event into the event list. Returns the inserted event's index.
 */
int SoundResource::InsertEvent(const sound_event &event)
{
	return mEvents.InsertEvent(event);
}


/*
 * Remove the event at index i from the event list.
 */
void SoundResource::DeleteEvent(int i)
{
	mEvents.DeleteEvent(i);
}


//-----------------------------------------------------------------------------


SoundEventList::SoundEventList(int capacity)
: mCapacity(capacity), mCount(0)
{
	mEvents = new sound_event[mCapacity];
}


SoundEventList::~SoundEventList()
{
	delete[] mEvents;
}

/*
 * Get the number of events in the list.
 */
int SoundEventList::GetLength() const
{
	return mCount;
}

/*
 * Get the event at a given index into the list.
 */
sound_event SoundEventList::GetEvent(int i) const
{
	return mEvents[i];
}

/*
 * Insert an event into the list, returning the index it occupies after
 * insertion. Insertion maintians a list sorted in chronological order.
 *
 * Events get inserted at the end of their tick. That is, if an event in the
 * list occupies the same tick as an inserted event, the inserted event will
 * occur after the pre-existing event (but still in the same tick). This is
 * important for properly merging tracks from format 1 MIDI files.
 */
int SoundEventList::InsertEvent(const sound_event &event)
{
	if (mCount == mCapacity) GrowCapacity();

	// TODO: Use binary serach to find the insertion point
	int i = 0;
	while ((i < mCount) && (mEvents[i].absoluteTime <= event.absoluteTime)) ++i;

	// make a hole then fill it
	memmove(mEvents + i + 1, mEvents + i, (mCount - i) * sizeof(sound_event));
	mEvents[i] = event;

	++mCount;
	return i;
}

/*
 * Add an event to the end of the list. AppendEvent does not maintain
 * chronological ordering; the preferred method of adding sound events is
 * InsertEvent. AppendEvent is useful during loading because events are known
 * to be in chronological order already. Using InsertEvent while loading or
 * importing will add the unnecessary overhead of searching to the end of the
 * event list for each event.
 */
void SoundEventList::AppendEvent(const sound_event &event)
{
	if (mCount == mCapacity) GrowCapacity();
	mEvents[mCount++] = event;
}

/*
 * Remove the event at index i from the list.
 */
void SoundEventList::DeleteEvent(int i)
{
	if (mCount > (i + 1)) {
		memmove(mEvents + i, mEvents + i + 1, (mCount - i - 1) * sizeof(sound_event));
	}
	--mCount;
}

/*
 * Remove all events from the list.
 */
void SoundEventList::Clear()
{
	mCount = 0;
}

/*
 * Make space for more elements in the list. The actual number of events, and
 * the events themselves, remains unchanged.
 */
void SoundEventList::GrowCapacity()
{
	sound_event *oldList = mEvents;
	mCapacity *= 2;
	mEvents = new sound_event[mCapacity];
	memcpy(mEvents, oldList, mCount * sizeof(sound_event));
	delete[] oldList;
}

/*
 * Copy the contents from one list into another.
 */
void SoundEventList::operator=(const SoundEventList &other)
{
	delete[] mEvents;
	mCount = other.mCount;
	mCapacity = other.mCount;
	mEvents = new sound_event[mCapacity];
	memcpy(mEvents, other.mEvents, mCount * sizeof(sound_event));
}
