//	detect.cpp	v.090	07/28/96
//
//	0.81	added detection for Vision964/968
//	0.83	added specific _868/_968 class chipset(s), GD-5434 separate
//			if S3+ chipset (Trio/x68/TrioV) detect fails, default to _968
//	0.84	added detection for GD-5436, distinguish GD-5430/40
//	0.85	added several more Trident chipsets, thanks to VGADOC4B.ZIP
//			I changed order of auto-detection, so TRIDENT is now LAST
//			Restored w32p code, however it is NOT autodetected.
//             The w32p code is only accessible with the "/F" option
//	0.86	Fixed corrupted screen bug
//	0.88	Added S3 Virge and Virge/VX detection
//	0.89	Added Cirrus Logic GD-5446 detection
//	0.90	Altered Trio64V+ detection routine, better S3 REV reporting
//
//	detects() installed svga hardware, tries to anyway!
//	As of now, I hope it detects Cirrus, S3, and Trident video cards.
//	As stated earlier, the W32p code has returned, but autodetection
//	routine is inadequate, so to operate W32p-MCLK, you must invoke
//	MCLK with "/F" option

#include "vga.h"
#include "s3.h"
#include "cirrus.h"
#include "trident.h"
#include "w32p.h"
#include<string.h>
#include<dos.h>
#include<stdio.h>
#include<stdlib.h>


/* The following char-tables are used by the detection routine to list
	available chipsets in each family.  I update the tables as I add
	more chipsets.  The last entry is always "", which serves as an
	endpoint marker...
	DO NOT CHANGE THE ORDER OF THE ITEMS WITHIN EACH LIST!!!
*/
const char table_cirrus[][30]={
	"GD-5420/22",
	"GD-5424/6/8/9",
	"GD-5430/40/M30/M40",
	"GD-5434",
	"GD-5436",
	"GD-5446",
	""};


const char table_s3[][30]={
	"S3-801/805",
	"S3-805i",
	"S3-864",
	"S3-964",	// added v0.86
	"S3-Trio32/64",
	"S3-Trio64V+",
	"S3-868",
	"S3-968",
	"S3-Virge",	// added v0.88
	"S3-Virge/VX",	// added v0.88, not yet working
	""	};


const char table_trident[][30]={
	"Trident 9440/96xx",
	""
	};

const char table_tseng[][30]={
	"ET4000/W32p",
	"ET4000/W32p RevB+",
	""
	};


// Help menu displays list(s) of either chipsets or families of chipsets.
// if chipset == _AUTO, help-menu returns list of supported families
// otherwise, help-menu returns list of chipsets indicated by chipset
// has two parameters,

message
detect::_help( int family )
{
	int i;	//	loop dummy variable
	INITMSG( msg.text );

	if ( family == _AUTO )	{
		msgout << "(F)orce manual selection of video-chipsets\n\n\t"
			<< _argv[ 0 ] << " /F %1\n\t\t(list available devices "
			<< "under family%1)"
			<< "\n\n\t" << _argv[ 0 ] << " /F %1 %2\n\t\t"
			<< "(select device%2 under family%1)\n\n"
			<< "Supported families\n------------------\n"
			<< "0 Generic VGA\n"
			<< (int)_FCIRRUS << " Cirrus Logic GD-54xx\n"
			<< (int)_FS3 << " S3 accelerator\n" << (int)_FW32P
			<< " Tseng Labs ET4000/W32 (not auto-detected! use '/F')\n"
			<< (int)_FTRIDENT << " Trident Microsystems\n";
	}	else
		switch ( family )	{
			case 0 :	msgout << "Generic VGA\n" << "0 VGA";
				break;
			case _FW32P :	msgout << "Tseng Labs ET4000/W32\n";
			for ( i = 0; table_tseng[ i ][ 0 ]; ++i )
				msgout << i << ' ' << table_tseng[ i ] << '\n' ;
				break;
			case _FCIRRUS:	msgout << "Cirrus Logic GD-54xx\n";
			for ( i = 0; table_cirrus[ i ][ 0 ]; ++i )
				msgout << i << ' ' << table_cirrus[ i ] << '\n' ;
				break;
			case _FS3 :	msgout << "S3 accelerator\n";
			for ( i = 0; table_s3[ i ][ 0 ]; ++i )
				msgout << i << ' ' << table_s3[ i ] << '\n' ;
				break;
			case _FTRIDENT: msgout << "Trident Microsystems\n";
			for ( i = 0; table_trident[ i ][ 0 ]; ++i )
				msgout << i << ' ' << table_trident[ i ] << '\n' ;
				break;
			default:	msgout << family << " = unknown chipset ID";
		}
	msgout << ends;
	return msg;
}


// detect will determine present video-hardware.  And initialize a new
// object of type VGA.  Returns a pointer to that object.
vga *
detect::_find( int chipset, int family )
{
	if  ( hardware != NULL ) {
		cerr << "\nInitialization error!";
		exit (EXIT_FAILURE );
	}
	strcpy( id.make, "VGA" );

	switch ( family )	{
		case 0 :
			break;
		case _FW32P :	hardware = detect_w32p( chipset );	//	Force w32p
			break;
		case _FCIRRUS:	hardware = detect_cirrus( chipset ); //	force Cirrus
			break;
		case _FS3 :	hardware = detect_s3( chipset );	//	force S3
			break;
		case _FTRIDENT: hardware= detect_trident( chipset ); // force TVGA
			break;
		default:	//	Regular auto detection
			family = _AUTO;
	}

	if ( family == _AUTO )	{
		hardware = detect_trident();	//	Check for Cirrus chipsets first!
		if ( hardware == NULL && id.make[ 0 ] == '?' )
			hardware = detect_s3();	//	If not found, _s3()
		if ( hardware == NULL && id.make[ 0 ] == '?' )
			hardware = detect_cirrus();	//	If not found, _trident()
	}

	if ( !hardware )		// If hardware == NULL ... never set-up...
		hardware = new vga( id );	//	Setup generic VGA hardware

	return hardware;
}


vga *	//	For debugging purposes only!
detect::_debug( int mode )
{
	vga_info temp = { "S3", "Vision968", "??" };
	if ( !hardware )
		hardware = new _864( temp );
	else
		exit( EXIT_FAILURE );
	return hardware;
}


vga *
detect::detect_cirrus( int mode )
{
//	vga *hardware = NULL;
	union REGS reg;
	hardware->write_SR( 0x06, 0x12 );	// Unlock Cirrus SVGA registers

	uchar _CR28 = hardware->read_CR( 0x28 ) ;	// ClassID (5430/40)
	uchar _CR27 = hardware->read_CR( 0x27 ) ;	// Read CL-GD ID register
	_CR27 = ( _CR27 >> 2 ) & 0x3F;		//	1234 5678 -> 0012 3456

	reg.h.ah = 0x12;
	reg.h.bl = 0x80;	//	Inquire VGA type
	int86( 0x10, &reg, &reg );
	if ( reg.h.bl != 0x80 )
		sprintf( id.revision, "%02X", reg.h.bl );
	else
		strcpy( id.revision, "??" );

	strcpy( id.make, "Cirrus Logic" );	//	Let's assume it's a cirrus

	if ( mode == _AUTO )
		switch ( _CR27 )	{
			case 34:	strcpy( id.chipset, "GD-5420" );
				hardware = new _cirrus( id );			break;
			case 35:	strcpy( id.chipset, "GD-5422" );
				hardware = new _cirrus( id ); 		break;
			case 37:	strcpy( id.chipset, "GD-5424" );
				hardware = new _GD5424( id );			break;
			case 36:	strcpy( id.chipset, "GD-5426" );
				hardware = new _GD5424( id );			break;
			case 38:	strcpy( id.chipset, "GD-5428" );
				hardware = new _GD5424( id );			break;
			case 39:	strcpy( id.chipset, "GD-5429" );
				hardware = new _GD5424( id );			break;
			case 42:	strcpy( id.chipset, "GD-5434" );
				hardware = new _GD5434( id );			break;
			case 41:	strcpy( id.chipset, "GD-5432 ??? " );
				hardware = new _GD543x( id );			break;
			case 43:	strcpy( id.chipset, "GD-5436" );
				hardware = new _GD5436( id );			break;
			case 46:	strcpy( id.chipset, "GD-5446" );
				hardware = new _GD5436( id );			break;
			case 40:
				switch( _CR28 )	{	// Distinguish 5430/40
					case 0xFF:	strcpy( id.chipset, "GD-5430" );
						break;
					case 0x01:	strcpy( id.chipset, "GD-54M30" );
						break;
					case 0x03:	strcpy( id.chipset, "GD-5440" );
						break;
					case 0x07:	strcpy( id.chipset, "GD-54M40" );
						break;
					default:	strcpy( id.chipset, "?!?GD-5430/40" );
				}
				hardware = new _GD543x( id );			break;
			default:		//  Couldn't identify Cirrus chipset
				strcpy( id.make, "? (tried Cirrus )" );
				strcpy( id.revision, "?" );
				sprintf( id.chipset, "? _CR27 = 0x%02X",
					hardware->read_CR( 0x27 ) );
				if ( mode == _FCIRRUS )	//	Forced Cirrus detection
					hardware = new _cirrus( id );
		}	// Else, if user FORCES a video-chipset (manual selection)
		else if ( mode != _AUTO )	{
			strcpy( id.chipset, table_cirrus[ mode ] );
			switch ( mode )	{
				case 0:   hardware = new _cirrus( id );
					break;
				case 1:	hardware = new _GD5424( id );
					break;
				case 2:	hardware = new _GD543x( id );
					break;
				case 3:	hardware = new _GD5434( id );
					break;
				case 4:	hardware = new _GD5436( id );
					break;
				case 5:	hardware = new _GD5436( id );
					break;
				default:
					cerr << "Unknown Cirrus Logic chipset.";
					exit( EXIT_FAILURE );
			}
		}

	return hardware;	//	Return whatever we found, even if nothing
}


vga *
detect::detect_s3( int mode )
{
//	vga *hardware = NULL;
	uchar _CR30, _CR2E, _CR2F, nibble, _DAC;
		// _CR2E for all newer S3 chips...Trio on up
		// _CR2F only needed to detect Trio64V+

	hardware->write_CR( 0x38, 0x48 );	// Unlock S3 VGA registers
	hardware->write_CR( 0x39, 0xA0 );	// Unlock S3 registers

	_CR30= hardware->read_CR( 0x30 );	// Read _CR30 "S3 chip ID"
	_CR2E= hardware->read_CR( 0x2E );	// Read _CR2E "S3 secondary ID"
	_CR2F= hardware->read_CR( 0x2F );	// If 4X, then we've got Trio64V+

	nibble = _CR30 & 0xF0;	//	Mask out revision status (lower 4bits)

	strcpy( id.make, "S3" );	//	Let's assume it's an S3 chip
	sprintf( id.revision, "_CR30(3:0) = %02X", _CR30 & 0x0F );

if ( mode == _AUTO )
	switch ( nibble )	{
		case 0x80:	strcpy( id.chipset, "86C911/924 not supported" );
			break;	//	Not supported
		case 0x90:	strcpy ( id.chipset, "86C928 not supported" );
			break;	//	Not supported
		case 0xA0:  if ( _CR30 == 0xA8 )	{
					strcpy( id.chipset, "86C805i" );
					hardware = new _805i( id );	}
			else	{	strcpy( id.chipset, "86C801/86C805" );
					hardware = new _S3( id );	}
			break;
		case 0xB0:	strcpy ( id.chipset, "86C928PCI not supported" );
			break;	//	Not supported
		case 0xC0:	strcpy( id.chipset, "Vision864" );
			hardware = new _864( id );			break;
		case 0xD0:	strcpy( id.chipset, "Vision964" );
			hardware = new _964( id );			break;
		case 0xE0:	//	0xE0, Trio/x68 or , newer S3chips
			sprintf( id.revision,"_CR2F=%02X",hardware->read_CR( 0x2F ) );
				//	These newer chips have different REV register
			switch( _CR2E )	{
				case 0x10:	strcpy( id.chipset, "Trio32" );
					hardware = new _Trio( id );	break;
				case 0x11:	// 0.90 changed 64V+ detection code
					if ( ( _CR2F >= 0x40 ) && ( _CR2F <= 0x5F ) ) {
						strcpy( id.chipset, "Trio64V+" );
						hardware = new _Triov( id );
					} else	{
						strcpy( id.chipset, "Trio64" );
						hardware = new _Trio( id );
					}
					break;
				case 0x31:	strcpy( id.chipset, "Virge" );
					hardware = new _Virge( id );	break;
				case 0x3D:	strcpy( id.chipset, "Virge/VX" );
					hardware = new _VirgeVX( id );	break;
				case 0x80:	strcpy( id.chipset, "Vision866");
					hardware = new _868( id );	break;
				case 0x90:	strcpy( id.chipset, "Vision868");
					hardware = new _868( id );	break;
				case 0xF0:	strcpy( id.chipset, "Vision968");
					hardware = new _968( id );	break;
				default:	sprintf( id.chipset,	// UNKNOWN EXT
				"Unknown _CR30 = 0x%02X, _CR2E = 0x%02X",_CR30, _CR2E );
					hardware = new _968( id );
			}
			break;
		default:	strcpy( id.make, "? (tried S3) " );
			sprintf ( id.chipset, "? _CR30 = 0x%02X, _CR2E = 0x%02X",
				_CR30, _CR2E );
			strcpy ( id.revision, "?" );
			if ( mode == _FS3 )	//	Forced S3 detection
				hardware = new _S3( id );
	}	// else, user-selected S3 chipset
	else if ( mode != _AUTO )	{
		strcpy( id.chipset, table_s3[ mode ] );
		switch ( mode )	{
			case 0:	hardware = new _S3( id );
				break;
			case 1:	hardware = new _805i( id );
				break;
			case 2:	hardware = new _864( id );
				break;
			case 3:	hardware = new _964( id );	// added v0.86
				break;
			case 4:	hardware = new _Trio( id );
				break;
			case 5:	hardware = new _Triov( id );
				break;
			case 6:	hardware = new _868( id );
				break;
			case 7:	hardware = new _968( id );
				break;
			case 8:	hardware = new _Virge( id );
				break;
			case 9:	hardware = new _VirgeVX( id );
				break;
			default:
				cerr << "Unknown S3 chipset.";
				exit( EXIT_FAILURE );
		}
	}

	return hardware;	//	Return whatever we found, even if nothing
}


vga *	//	Trident support not implemented
detect::detect_trident( int mode )
{
//	vga *hardware = NULL ;
	uchar _SR0B = 0, _SR0E;	//	Trident ID register
	uchar old_mode;	// Preserve low/high bytes of mode-register
	uchar temp;
//	INITMSG( id.make );
//	msgout << "Trident" << ends ;	// Let's assume it's a Trident

	strcpy( id.make, "Trident" );	//	Let's assume it's a Trident
	strcpy( id.revision, "??" );

//	INITMSG( id.chipset );

	// Following code adapted from VGADOC4B.ZIP
	hardware->write_SR( 0x0B, 0 );	//Write $00 to force new-mode
	_SR0B = hardware->read_SR( 0x0B );	// this forces old mode
	_SR0E = hardware->read_SR( 0x0E );
	hardware->write_SR( 0x0E, _SR0E ^ 0x55 ); // xor $55
	hardware->write_SR( 0x0E, _SR0E ^ 0x02 ); // xor $02

if ( mode == _AUTO )
	switch ( _SR0B )	{
		case 1 : case 2 :	strcpy( id.chipset, "8800" );
			break;	//	Not supported
		case 3: case 4: case 0x13: case 0x12: case 0x33:
			strcpy( id.chipset, "8900 series" );
			break;
		case 0x23: case 0x43: strcpy( id.chipset, "9000 series" );
			break;	//	Not supported
		case 0x53:	strcpy( id.chipset, "9200CXr" );
			break;	//	Not supported
		case 0xC3: case 0x73:	strcpy( id.chipset, "GUI9420??" );
			break;	//	Not supported
		case 0x93:	strcpy( id.chipset, "GUI9400CXi" );
			break;	//	Not supported
		case 0xF3:	strcpy( id.chipset, "GUI9430??" );
			break;
		case 0xE3:	strcpy( id.chipset, "GUI9440" );
			hardware = new _TR9440( id );
			break;
		case 0xD3:	strcpy( id.chipset, "GUI96xx" );
			hardware = new _TR9440( id );
			break;
		case 0x63: case 0x83: case 0xA3:
			strcpy( id.chipset, "A Trident LCD chipset" );
			break;
		default:
			sprintf( id.chipset, "? Unknown _SR0B = 0x%02X", _SR0B );
				strcpy( id.make, "? (tried Trident)" );
			strcpy( id.revision, "?" );
			if ( mode == _FTRIDENT )	{
				cerr << "\nTRIDENT not supported yet.";
				exit( EXIT_FAILURE );
			}
	}
	else if ( mode != _AUTO )	{
		strcpy( id.chipset, table_trident[ mode ] );

		switch( mode )	{
			case 0 :  hardware = new _TR9440( id );
				break;
			default:
				cerr << "Unknown Trident chipset.";
				exit( EXIT_FAILURE );
		}
	}
//}
	return hardware;	//	Return whatever we found, even if nothing
}


//	v0.85B...restored w32p code
vga *
detect::detect_w32p( int mode )
{
//	vga *hardware = NULL ;

	outportb( 0x3BF, 0x03 );	//	Unlock W32P "key"
	outportb( 0x3D8, 0xA0 );	//	Unlock W32P "key"

	strcpy( id.make, "? (tried W32p)" );
	strcpy( id.chipset, "?" );
	strcpy( id.revision, "?" );

	if ( mode == _AUTO )	{
		strcpy( id.make, "Tseng Labs" );
		strcpy( id.chipset, "ET4000/W32p" );
		strcpy( id.revision, "?" );
		hardware = new _w32p( id );
	}
	else if ( mode != _AUTO )	{
		strcpy( id.chipset, table_tseng[ mode ] );
		switch( mode )	{
			case 0 :  hardware = new _w32p( id );
				break;
			case 1 :	hardware = new _w32pb( id );
				break;
			default:
				cerr << "Unknown Tseng Labs chipset.";
				exit( EXIT_FAILURE );
		}
	}

	return hardware;
}