unit Filer;

{
 ***** version 2.0 ***** 
 (c) Pavel Zampach (zampach@volny.cz), 2003, 2005
 GNU GENERAL PUBLIC LICENSE 
 Source code for Borland Delphi Compiler (originally ver. 7.0)
}

interface

uses
  SysUtils;

const
  PACKET_DATA_SIZE = $0800;
  FNAME_LENGTH     = 13;

type
  TData  = array [1..PACKET_DATA_SIZE] of byte;
  TPData = ^TData;

  TDirEntry = packed record
    fattr  : byte;
    ftime  : word;
    fdate  : word;
    fsize  : longword;
    fname  : array [0..FNAME_LENGTH-1] of char;
  end;

  TDOSFileTime = packed record
    ftime  : word;
    fdate  : word;
  end;

  EHPLXException = class (EInOutError);

var
  Connected, HPLXError, HPLXExceptions : boolean;

{---------- Basic procedures and functions ----------
Functions return TRUE if I/O operation was succesfull
}

function  filerConnect (const CommPort : string; const CommSpeed : integer) : boolean;
{Connects HPLX to desktop, CommPort is 'COM1'..'COM8', ComSpeed is 2400..115200}

function  filerDisconnect : boolean;
{Disconnects HPLX from desktop}

procedure filerReset;
{For recovery past communications error}

{---------- File-handling procedures and functions ----------

  Error handling - two ways:
1. After HPLX I/O procedure test the boolean variable HPLXError and write error
   recovery routine like: if HPLXError then begin ... end;
2. set HPLXExceptions := true; and done error recovery via exception handling. 
These procedures and functions generate exception class EHPLXException with
appropriate ErrorCode (in round brackets below)
}

procedure HPLXReset (const FileName : string);
{(11) Procedure is combination of AssignFile and Reset; it opens file for
reading}

procedure HPLXCreateFile (const FileName : string);
{(3) Procedure is combination of AssignFile and opening file for writing.
If file already exists, error is generated}

procedure HPLXRewrite (const FileName : string);
{(5) Procedure is combination of AssignFile and Rewrite; it opens file for
writing. If file already exists, it is overwritten}

procedure HPLXCloseFile;
{(1) Procedure closes previously opened file}

procedure HPLXBlockRead (var Buf : TData; const Count: word ; var Result : word);
{(12) Procedure works similar as standard BlockRead, Count <= PACKET_DATA_SIZE}

function  HPLXBlockWrite (var Buf : TData; const Count : word) : word;
{(21) Function works similar as standard BlockWrite, Count <= PACKET_DATA_SIZE,
returns number of really written bytes (often no needed)}

function  HPLXFilePos : longword;  
{(19) Function returns actual position of file pointer, file must be opened}

procedure HPLXSeek (const N : longword);  
{(15) Procedure sets position of file pointer, file must be opened}

function  HPLXGetFileTime : longword;
{(23) Function returns file date/time in DOS format, file must be opened}

procedure HPLXSetFileTime (const DateTime : longword);
{(24) Procedure sets file date/time in DOS format, file must be opened}

function  HPLXGetAttr (const FileName : string) : byte; 
{(25) Function returns file attributes, file cannot be opened}

procedure HPLXSetAttr (const FileName : string; const Attr : byte);
{(26) Procedure sets file attributes, file cannot be opened}

procedure HPLXDeleteFile (const FileName : string);
{(4) Procedure deletes file}

procedure HPLXRenameFile (const OldName, NewName : string);
{(13) Procedure renames file or moves it in range of the same disk} 

function  HPLXFileExists (const FileName : string) : boolean;
{(8) Function returns TRUE if file or directory exists (directory name
must by without trailing backslash}

procedure HPLXSearchDirName (const DirName : string);
{(18) Procedure sets the search pattern (f.e. 'C:\*.*') for next use of
HPLXSearchDir function}

function  HPLXSearchDir (var DirEntry : TDirEntry) : boolean;
{(9) Function places into DirEntry next directory item. Pattern for
searching is set by HPLXSearchDirName procedure. Function returns TRUE
if data are valid and FALSE on end of search.}

function  HPLXGetDir (const Disk : char) : string;
{(6) Function returns active directory on desired disk)}

procedure HPLXChDir (const DirName : string); 
{(16) Procedure sets active directory on actual disk}

procedure HPLXMkDir (const DirName : string);
{(10) Procedure creates new directory}

procedure HPLXRmDir (const DirName : string);
{(14) Procedure removes directory (must be empty)}

function  HPLXGetDisk : char;
{(7) Function returns active disk as char ('A', 'C'...)}

procedure HPLXSetDisk (const Disk : char);
{(17) Procedure sets active disk (as char)}

function  HPLXDiskFree : longword;
{(20) Function returns free space on actual disk}

function  HPLXGetDiskVector : string;
{(27) Function returns string with disks defined in HPLX, standard response
is 'ABCDE'}

{------------------------- Help functions ---------------------------}

function getShortFileName (const FileName : string) : string;
{Function returns DOS short file name (8.3)}

{ ****************************************************************** }

implementation

uses
  Windows;

const
  NUL         = 0;
  SOH         = 1;
  STX         = 2;
  STATUS_OK   = $81;
  MAX_PKT_LEN = PACKET_DATA_SIZE*2+100;
  SIG_LEN     = 5;
  EXCEPT_MSG  = 'HPLX communications error';
  
  CMD_CLOSE_FILE        = $01;              //   Filer communications commands
  CMD_CREATE_FILE       = $03;
  CMD_DEL_FILE          = $04;
  CMD_REWRITE_FILE      = $05;
  CMD_GET_ACTIVEDIR     = $06;
  CMD_GET_ACTIVEDISK    = $07;
  CMD_CHECK_FILE        = $08;
  CMD_GET_DIR           = $09;
  CMD_MAKE_DIR          = $0A;
  CMD_RESET_FILE        = $0B;
  CMD_GET_DATA          = $0C;
  CMD_REN_FILE          = $0D;
  CMD_DEL_DIR           = $0E;
  CMD_SET_FILEPOINTER   = $0F;
  CMD_SET_DIR           = $10;
  CMD_SET_DISK          = $11;
  CMD_ASK_DIR           = $12;
  CMD_GET_FILEPOINTER   = $13;
  CMD_GET_DISKINFO      = $14;
  CMD_SEND_DATA         = $15;
  CMD_INIT_GET          = $16;
  CMD_GET_FILETIME      = $17;
  CMD_SET_FILETIME      = $18;
  CMD_GET_FILEATTR      = $19;
  CMD_SET_FILEATTR      = $1A;
  CMD_GET_DISKVECTOR    = $1B;
  CMD_CONNECT_SERVER    = $40;
  CMD_DISCONNECT_SERVER = $41;
  
// packet signature
  Signature : array [1..SIG_LEN] of byte = ($16, $16, $16, $10, $02);

type
  TWorkBuf = array [1..MAX_PKT_LEN] of byte;

  TDiskInfo = packed record
    volume  : array [1..11] of char;
    free    : longword;
  end;

var
  hCommFile : THandle;
  PacketFunct, PacketCount, PacketStatus : byte;
  PacketSize, PacketSize2 : word;
  PacketData, PacketData2 : TPData;
  ExistFlag : boolean;
       
{ ------------------------------------------------------------------ }

function updateCRC16 (CRC : word; Data : byte) : word;

var
  i : byte;

begin
  Result := CRC xor Data;
  for i := 1 to 8 do
    if (Result and 1) = 1 then
      Result := (Result shr 1) xor $A001
    else
      Result := Result shr 1;
end;

{ ------------------------------------------------------------------ }

procedure sendPacket;

var
  Checksum, i, WorkSize : word;
  WorkBuf : TWorkBuf;
  BytesWritten : longword;

{ -------------- }

procedure sendByte (value : byte; Chsum : boolean);

begin
  inc (WorkSize);
  WorkBuf[WorkSize] := value;
  if (value = $10) and Chsum then begin     // '$10' CRC trick, got to send it twice
    inc (WorkSize);
    WorkBuf[WorkSize] := value;
  end;
  if Chsum then                             // update checksum
    Checksum := updateCRC16 (Checksum, value);
end;                                        // function sendByte

{ -------------- }

begin                                       // function sendPacket
  Checksum := 0;
  WorkSize := 0;
  
  if not Connected then                     // synchonization - 4 NUL bytes
    for i := 1 to 4 do
      sendByte (NUL, false);
      
  for i := 1 to SIG_LEN do                  // send signature, no CRC yet
    sendByte (Signature[i], false);
  sendByte (SOH, true);
  sendByte (PacketFunct, true);
  sendByte (PacketCount, true);
  sendByte (SOH, true);                     // send Data marker
  sendByte (STX, true);

  if PacketFunct = CMD_SEND_DATA then begin
    sendByte (0, true);
    sendByte (0, true);
    sendByte (Lo (PacketSize), true);
    sendByte (Hi (PacketSize), true);
    for i := 1 to PacketSize do
      sendByte (PacketData^[i], true);
  end;

  if (PacketFunct = CMD_MAKE_DIR) or
     (PacketFunct = CMD_DEL_DIR) or
     (PacketFunct = CMD_DEL_FILE) or
     (PacketFunct = CMD_CHECK_FILE) or
     (PacketFunct = CMD_CREATE_FILE) or
     (PacketFunct = CMD_REWRITE_FILE) or
     (PacketFunct = CMD_RESET_FILE) or
     (PacketFunct = CMD_REN_FILE) or
     (PacketFunct = CMD_SET_DIR) or
     (PacketFunct = CMD_SET_FILEATTR) or
     (PacketFunct = CMD_GET_FILEATTR) or
     (PacketFunct = CMD_ASK_DIR) then
  begin
    sendByte (Lo (PacketSize), true);       // send path/filename (FROM) size LO, HI
    sendByte (Hi (PacketSize), true);
    for i := 1 to PacketSize do             // send path/filename (FROM)
      sendByte (PacketData^[i], true);
    if  PacketFunct = CMD_SET_FILEATTR then
      sendByte (PacketData2^[1], true)      // send FileAttr
    else  
      sendByte (0, true);
    sendByte (0, true);
  end;

  if PacketFunct = CMD_REN_FILE then begin
    sendByte (Lo (PacketSize2), true);      // send filename TO size LO, HI
    sendByte (Hi (PacketSize2), true);
    for i := 1 to PacketSize2 do            // send filename TO
      sendByte (PacketData2^[i], true);
    sendByte (0, true);
    sendByte (0, true);
  end;

  if PacketFunct = CMD_GET_DATA then begin
    sendByte (0, true);
    sendByte (0, true);
    sendByte (Lo (PacketSize), true);
    sendByte (Hi (PacketSize), true);
  end;

  if PacketFunct = CMD_SET_FILETIME then begin
    sendByte (0, true);
    sendByte (0, true);
    for i := 1 to 4 do                      // send date and time
      sendByte (PacketData^[i], true);
  end;

  if (PacketFunct = CMD_CLOSE_FILE) or
     (PacketFunct = CMD_GET_ACTIVEDISK) or
     (PacketFunct = CMD_GET_DISKVECTOR) or
     (PacketFunct = CMD_GET_FILEPOINTER) or
     (PacketFunct = CMD_GET_FILETIME) then
  begin
    sendByte (0, true);
    sendByte (0, true);
  end;  

  if (PacketFunct = CMD_SET_DISK) or
     (PacketFunct = CMD_GET_ACTIVEDIR) then
  begin
    sendByte (PacketData^[1], true);        // send disk
    sendByte (0, true);
  end;

  if PacketFunct = CMD_SET_FILEPOINTER then begin
    sendByte (0, true);
    sendByte (0, true);
    sendByte (0, true);
    sendByte (0, true);
    for i := 1 to 4 do                      // send date and time
      sendByte (PacketData^[i], true);
  end;  

// ----- COMMON PART -----
  sendByte ($10, false);                    // no CRC on this one (tricky!)
  sendByte ($03, true);
  sendByte (Lo (Checksum), false);          // finally send Checksum LO, HI
  sendByte (Hi (Checksum), false);

  writeFile (hCommFile, WorkBuf, WorkSize, BytesWritten, nil);  // really send packet from buffer
end;                                        // function sendPacket

{ ------------------------------------------------------------------ }

function getPacket : boolean;

var
  c, Data, ByteHi, ByteLo : byte;
  i, Checksum, EstimatePktLen, WorkPtr  : word;
  BytesRead : longword;
  WorkBuf : TWorkBuf;

{ -------------- }

function getByte (Chsum : boolean) : byte;

begin
  if WorkPtr = BytesRead then abort;
  inc (WorkPtr);
  Result := WorkBuf[WorkPtr];
  if (Result = $10) and Chsum then begin    // '$10' CRC trick, receive it twice
    if WorkPtr = BytesRead then abort;
    inc (WorkPtr);
    Result := WorkBuf[WorkPtr];
  end;
  if Chsum then                             // update checksum
    Checksum := updateCRC16 (Checksum, Result);
end;                                        // function getByte

{ -------------- }

procedure testNUL (NoN : integer);

VAR i : integer;

begin
  for i := 1 to NoN do
    if getByte (true) <> NUL then abort;
end;

{ -------------- }

begin
  Checksum := 0;
  Result   := true;
  WorkPtr  := 0;
  if PacketFunct = CMD_GET_DATA then
    EstimatePktLen := MAX_PKT_LEN
  else
    EstimatePktLen := 100;

  try

  readFile (hCommFile, WorkBuf, EstimatePktLen, BytesRead, nil);
                                            // really read packet into buffer, end by timeout
  c := 0;                                   // search signature
  repeat
    inc (c);
    if getByte (false) <> Signature[c] then c := 0;
  until c = SIG_LEN;

  if getByte (true) <> SOH then abort;      // get SOH
  PacketFunct  := getByte (true);
  PacketCount  := getByte (true);
  PacketStatus := getByte (true);
  if getByte (true) <> STX then abort;      // get Data marker

  if (PacketFunct = CMD_CONNECT_SERVER) or
     (PacketFunct = CMD_DISCONNECT_SERVER) then
    testNUL (2);

  if  (PacketFunct = CMD_SEND_DATA) then
  begin
    testNUL (4);
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    PacketSize := (ByteHi * $100) + ByteLo;
  end;

  if (PacketFunct = CMD_CHECK_FILE) then begin
    testNUL (4);
    Data := getByte (true);                 // result marker    
    if Data > 2 then abort;
    ExistFlag := (Data <> NUL);
    testNUL (1);
  end;

  if (PacketFunct = CMD_GET_FILETIME) or
     (PacketFunct = CMD_GET_FILEPOINTER) then
  begin
    testNUL (4);
    for i := 1 to 4 do                      // get date and time
      PacketData^[i] := getByte (true);
  end;

  if (PacketFunct = CMD_MAKE_DIR) or
     (PacketFunct = CMD_DEL_DIR) or
     (PacketFunct = CMD_DEL_FILE) or
     (PacketFunct = CMD_REN_FILE) or
     (PacketFunct = CMD_CREATE_FILE) or
     (PacketFunct = CMD_REWRITE_FILE) or
     (PacketFunct = CMD_RESET_FILE) or
     (PacketFunct = CMD_CLOSE_FILE) or
     (PacketFunct = CMD_INIT_GET) or
     (PacketFunct = CMD_SET_FILETIME) or
     (PacketFunct = CMD_SET_FILEATTR) or
     (PacketFunct = CMD_SET_FILEPOINTER) or
     (PacketFunct = CMD_SET_DISK) or
     (PacketFunct = CMD_SET_DIR) or
     (PacketFunct = CMD_ASK_DIR) then
    testNUL (6);                            // get Data size & end marker (6 bytes)

  if PacketFunct = CMD_GET_DATA then begin
    testNUL (4);
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    PacketSize := (ByteHi * $100) + ByteLo;
    for i := 1 to PacketSize do
      PacketData^[i] := getByte (true);
  end;

  if (PacketFunct = CMD_GET_ACTIVEDIR) or
     (PacketFunct = CMD_GET_DISKVECTOR) then
  begin
    testNUL (4);
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    PacketSize := (ByteHi * $100) + ByteLo;
    for i := 1 to PacketSize do
      PacketData^[i] := getByte (true);
    testNUL (4);
  end;

  if PacketFunct = CMD_GET_DIR then begin
    testNUL (4);
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    PacketSize := (ByteHi * $100) + ByteLo;
    for i := 1 to PacketSize do             // get Data
      PacketData^[i] := getByte (true);
    Data := getByte (true);                 // special 'dir' Data end marker    
    if Data > 1 then abort;
    ExistFlag := (Data <> NUL);
    testNUL (3);
  end;

  if PacketFunct = CMD_GET_DISKINFO then begin
    testNUL (4);
    getByte (true);
    getByte (true);
    for i := 1 to 15 do                     // PacketSize = 11 + 4 (always)
      PacketData^[i] := getByte (true);
  end;

  if PacketFunct = CMD_GET_ACTIVEDISK then begin
    testNUL (4);
    PacketData^[1] := getByte (true);
    testNUL (1);
  end;

  if PacketFunct = CMD_GET_FILEATTR then begin
    testNUL (4);
    PacketData2^[1] := getByte (true);
    testNUL (1);
  end;

// ----- COMMON PART -----
  if getByte (false) <> $10 then abort;     // get CRC Marker (no CRC on the this byte!)
  if getByte (true)  <> $03 then abort;
  ByteLo := getByte (false);                // get received CRC
  ByteHi := getByte (false);
  if ((ByteHi * $100) + ByteLo) <> Checksum then abort; 
                                            // check if CRC is good
  except
    Result := false;
  end;                                      // except  
end;                                        // function getPacket

{ ------------------------------------------------------------------ }

function filerRequest (Funct : byte) : boolean;

const
  MAX_ATT = 3;

var
  Att, Count : byte;
  BgetPacket : boolean;

begin
  Result := false;
  if PacketSize > PACKET_DATA_SIZE then exit;
  if (Funct = CMD_CONNECT_SERVER) or
     (Funct = CMD_DISCONNECT_SERVER) then PacketCount := 0;
  Count := PacketCount;
  PacketFunct := Funct;

  Att := 0;                                 // packet handshake
  repeat
    purgeComm (hCommFile, PURGE_TXCLEAR or PURGE_RXCLEAR);
    sendPacket;
    inc (Att);
    BgetPacket := getPacket;
  until BgetPacket or (Att = MAX_ATT);

  if (not BgetPacket) or                    // check received values
     (PacketStatus <> STATUS_OK) or
     (PacketCount  <> Count) or
     (PacketFunct  <> Funct) then exit;

  if PacketCount = $FF then                 // cyclic update packet counter
    PacketCount := 0
  else
    inc (PacketCount);
    
  Result := true;
end;                                        // funct filerRequest

{ ------------------------------------------------------------------ }

procedure filerReset;

begin
  sleep (300);
  purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
  filerRequest (CMD_CONNECT_SERVER);
end;

{ ------------------------------------------------------------------ }

function filerConnect (const CommPort : string; const CommSpeed : integer) : boolean;

const
  RX_BUF = $C00;
  TX_BUF = $C00;

  fEmpty               = $00000000;
  fBinary              = $00000001;
  fParity              = $00000002;
  fOutxCtsFlow         = $00000004;
  fOutxDsrFlow         = $00000008;
  fDtrControlEnable    = $00000010;
  fDtrControlHandshake = $00000020;
  fDsrSensitivity      = $00000040;
  fTXContinueOnXoff    = $00000080;
  fOutX                = $00000100;
  fInX                 = $00000200;
  fErrorChar           = $00000400;
  fNull                = $00000800;
  fRtsControlEnable    = $00001000;
  fRtsControlHandshake = $00002000;
  fAbortOnError        = $00004000;
  fDummy2              = $FFFF8000;


var
  DcbPort          : TDCB;
  CommPortTimeouts : TCommTimeouts;
  SpeedCorr        : word;

begin
  Result := true;
  if Connected then exit;

  try

// open file for serial communications
  hCommFile := CreateFile (PChar(CommPort), GENERIC_READ or GENERIC_WRITE, 0, nil, OPEN_EXISTING, 0, 0);
  if hCommFile = INVALID_HANDLE_VALUE then abort;
  purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);

  if not getCommState (hCommFile, dcbPort) then abort; 
  with DcbPort do begin                     // set port parameters
    BaudRate := CommSpeed;
    ByteSize := 8;
    Parity   := NOPARITY;
    StopBits := ONESTOPBIT;
    Flags    := fEmpty;                     // no handshake!
  end;
  setCommState (hCommFile, DcbPort);

  SpeedCorr := 10000 div CommSpeed;         // set port timeouts
  if not getCommTimeouts (hCommFile, CommPortTimeouts) then abort;
  with CommPortTimeouts do begin
    ReadIntervalTimeout         := 50 + SpeedCorr;
    ReadTotalTimeoutMultiplier  := 1 + SpeedCorr;
    ReadTotalTimeoutConstant    := 1000;
    WriteTotalTimeoutMultiplier := 1 + SpeedCorr;
    WriteTotalTimeoutConstant   := 1000;
  end;
  setCommTimeouts (hCommFile, CommPortTimeouts);

  setupComm (hCommFile, RX_BUF, TX_BUF);    // set recieve and transmit buffer size

  if not filerRequest (CMD_CONNECT_SERVER) then abort;

  Connected := true;

  except
    sleep (300);
    purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
    closeHandle (hCommFile);
    Result := false;
  end;                                      // except  
end;                                        // funct FilerConnect

{ ------------------------------------------------------------------ }

function filerDisconnect : boolean;

begin
  if not Connected then
    Result := true
  else begin
    Result := filerRequest (CMD_DISCONNECT_SERVER);
    closeHandle (hCommFile);
    Connected := false;
  end;
end;

{ ------------------------------------------------------------------ }

procedure HPLXCmd (const Cmd : byte);

var
  HPLXIOError : EHPLXException;
  
begin
  HPLXError := not filerRequest (Cmd);
  if HPLXExceptions and HPLXError then begin
    HPLXIOError := EHPLXException.Create (EXCEPT_MSG);
    HPLXIOError.ErrorCode := Cmd;  
    raise HPLXIOError;
  end;
end;

{ ------------------------------------------------------------------ }

procedure HPLXCmdStr (const FileName : string; const Cmd : byte);

begin
  PacketData := TPData (PChar (FileName));
  PacketSize := length (FileName);
  HPLXCmd (Cmd);
end;

{ ------------------------------------------------------------------ }

procedure HPLXCloseFile;

begin
  HPLXCmd (CMD_CLOSE_FILE);
end;

{ ------------------------------------------------------------------ }

procedure HPLXCreateFile (const FileName : string);

begin
  HPLXCmdStr (FileName, CMD_CREATE_FILE);
end;

{ ------------------------------------------------------------------ }

procedure HPLXDeleteFile (const FileName : string);

begin
  HPLXCmdStr (FileName, CMD_DEL_FILE);
end;

{ ------------------------------------------------------------------ }

procedure HPLXRewrite (const FileName : string);

begin
  HPLXCmdStr (FileName, CMD_REWRITE_FILE);
end;

{ ------------------------------------------------------------------ }

function  HPLXGetDir (const Disk : char) : string;

var Tmp : array [0..255] of char;

begin
  Tmp[0] := Disk;
  PacketData := @Tmp;
  HPLXCmd (CMD_GET_ACTIVEDIR);
  Tmp[PacketSize] := #0;
  Result := Tmp;
end;

{ ------------------------------------------------------------------ }

function HPLXGetDisk : Char;

begin
  PacketData := @Result;
  HPLXCmd (CMD_GET_ACTIVEDISK);
end;
{ ------------------------------------------------------------------ }

function HPLXFileExists (const FileName : string) : boolean;

begin
  HPLXCmdStr (FileName, CMD_CHECK_FILE);
  Result := ExistFlag;
end;

{ ------------------------------------------------------------------ }

procedure HPLXSearchDirName (const DirName : string);

begin
  HPLXCmdStr (DirName, CMD_ASK_DIR);
end;

{ ------------------------------------------------------------------ }

procedure HPLXMkDir (const DirName : string);

begin
  HPLXCmdStr (DirName, CMD_MAKE_DIR);
end;

{ ------------------------------------------------------------------ }

procedure HPLXReset (const FileName : string);

begin
  HPLXCmdStr (FileName, CMD_RESET_FILE);
end;

{ ------------------------------------------------------------------ }

procedure HPLXBlockRead (var Buf : TData; const Count: word ; var Result : word);

begin
  PacketSize := Count;
  PacketData := @Buf;
  HPLXCmd (CMD_GET_DATA);
  Result := PacketSize;
end;

{ ------------------------------------------------------------------ }

procedure HPLXRenameFile (const OldName, NewName : string);

begin
  PacketData2 := TPData (PChar (NewName));
  PacketSize2 := length (NewName);
  HPLXCmdStr (OldName, CMD_REN_FILE);
end;

{ ------------------------------------------------------------------ }

procedure HPLXRmDir (const DirName : string);

begin
  HPLXCmdStr (DirName, CMD_DEL_DIR);
end;

{ ------------------------------------------------------------------ }

procedure HPLXSeek (const N : longword);  

begin
  PacketData := @N;
  HPLXCmd (CMD_SET_FILEPOINTER);
end;

{ ------------------------------------------------------------------ }

procedure HPLXChDir (const DirName : string); 

begin
  HPLXCmdStr (DirName, CMD_SET_DIR);
end;

{ ------------------------------------------------------------------ }

procedure HPLXSetDisk (const Disk : char);

begin
  PacketData := @Disk;
  HPLXCmd (CMD_SET_DISK);
end;

{ ------------------------------------------------------------------ }

function HPLXSearchDir (var DirEntry : TDirEntry) : boolean;

begin
  ExistFlag := false;
  PacketData := @DirEntry;
  HPLXCmd (CMD_GET_DIR);
  Result := ExistFlag;
end;

{ ------------------------------------------------------------------ }

function HPLXFilePos : longword;  

begin
  PacketData := @Result;
  HPLXCmd (CMD_GET_FILEPOINTER);
end;

{ ------------------------------------------------------------------ }

function HPLXDiskFree : longword;

var Tmp : TDiskInfo;

begin
  PacketData := @Tmp;
  HPLXCmd (CMD_GET_DISKINFO);
  Result := Tmp.free;
end;

{ ------------------------------------------------------------------ }

function HPLXBlockWrite (var Buf : TData; const Count : word) : word;

begin
  PacketSize := Count;
  PacketData := @Buf;
  HPLXCmd (CMD_SEND_DATA);
  Result := PacketSize;
end;

{ ------------------------------------------------------------------ }

function HPLXGetFileTime : longword;

begin
  PacketData := @Result;
  HPLXCmd (CMD_GET_FILETIME);
end;

{ ------------------------------------------------------------------ }

procedure HPLXSetFileTime (const DateTime : longword);

begin
  PacketData := @DateTime;
  HPLXCmd (CMD_SET_FILETIME);
end;

{ ------------------------------------------------------------------ }

function HPLXGetAttr (const FileName : string) : byte; 

begin
  PacketData2 := @Result;
  HPLXCmdStr (FileName, CMD_GET_FILEATTR);
end;

{ ------------------------------------------------------------------ }

procedure HPLXSetAttr (const FileName : string; const Attr : byte);

begin
  PacketData2 := @Attr;
  HPLXCmdStr (FileName, CMD_SET_FILEATTR);
end;

{ ------------------------------------------------------------------ }

function HPLXGetDiskVector : string;

var Tmp : array [0..255] of char;

begin
  PacketData := @Tmp;
  HPLXCmd (CMD_GET_DISKVECTOR);
  Result := Tmp;
end;

{ ------------------------------------------------------------------ }

function getShortFileName (const FileName : string) : string;

var
  Tmp : array [0..255] of char;
  
begin
  if getShortPathName (PChar (FileName), Tmp, sizeOf (Tmp) - 1) = 0 then
    Result := FileName
  else
    Result := Tmp;
end;

{ ------------------------------------------------------------------ }


initialization
  Connected := false;
  HPLXExceptions := false;
end.