/* --------------------------------------------------------------------
   Project: PAL General Database routines
   Module:  SHOWROW.C
   Author:  Jean-Jacques Moreau
   Started: 11. Aug. 99
   Subject: Show database viewpoint row
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                          standard includes
   -------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

/* --------------------------------------------------------------------
                            local includes
   -------------------------------------------------------------------- */

#include "pal.h"


#define KEYDEPTH   9
#define MAXROWS   12
#define MAX_X3   213       /* MAX_X / 3 */


/* -------------------------------------------------------------------------
                           Answer the column a row will be displayed in
   ------------------------------------------------------------------------- */

static int Page(int RowNo)
{
   if (RowNo < 0)
      return -1;

   return RowNo/MAXROWS;
}


/* -------------------------------------------------------------------------
                           Answer the line a row will be displayed on
   ------------------------------------------------------------------------- */

static int Row(int RowNo)
{
   return RowNo-Page(RowNo)*MAXROWS;
}


/* -------------------------------------------------------------------------
                           Answer the max number of pages visible on a screen
   ------------------------------------------------------------------------- */

BYTE MaxVisiblePages(DBDESC *pDD)
{
   return (pDD->ZoomMode == ZOOM_1) ? 0 : 3;
}


/* -------------------------------------------------------------------------
                           Answer whether a row is currently visible
   ------------------------------------------------------------------------- */

BYTE IsRowVisible(DBDESC *pDD, int RowNo)
{
   int PageNo=Page(RowNo);

   return (pDD->ZoomMode == ZOOM_1) ? (PageNo == 0) : ((PageNo >= 0) && (PageNo < 3));
}


/* -------------------------------------------------------------------------
                           Normalize a row so it becomes visible
   ------------------------------------------------------------------------- */

int NormalizeRow(DBDESC *pDD, int RowNo, BYTE Type)
{
   int PageNo=Page(RowNo);
   int NormalizedRow=RowNo;
   int Offset=((Type == 1) && (pDD->ZoomMode == ZOOM_3)) ? MAXROWS*3 : MAXROWS;

   if (PageNo < 0)
      NormalizedRow=RowNo+Offset;
   else if (PageNo >= MaxVisiblePages(pDD))
      NormalizedRow=RowNo-Offset;

   return NormalizedRow;
}


/* -------------------------------------------------------------------------
                           Answer the first visible row
   ------------------------------------------------------------------------- */

void FirstVisibleRow(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   *pTargetRec=(RecNo-RowNo < 0) ? 0 : RecNo-RowNo;
   *pTargetRow=RowNo-(RecNo-(*pTargetRec));
}


/* -------------------------------------------------------------------------
                           Answer the last visible row
   ------------------------------------------------------------------------- */

void LastVisibleRow(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   int Offset=MaxDisplayableRows(pDD)-1-RowNo;
   int MaxRecs=NumOfRecords(pDD);

   *pTargetRec=(RecNo+Offset > MaxRecs-1) ? MaxRecs-1 : RecNo+Offset;
   *pTargetRow=RowNo+((*pTargetRec)-RecNo);
}


/* -------------------------------------------------------------------------
                           Answer the row at the closest position on
                           the left column
   ------------------------------------------------------------------------- */

void SameRowOnPreviousPage(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   *pTargetRec=(RecNo-MAXROWS < 0) ? 0 : RecNo-MAXROWS;
   *pTargetRow=RowNo-(RecNo-(*pTargetRec));
}


/* -------------------------------------------------------------------------
                           Answer the row at the closest position on
                           the right column
   ------------------------------------------------------------------------- */

void SameRowOnNextPage(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   int MaxRecs=NumOfRecords(pDD);

   *pTargetRec=(RecNo+MAXROWS > MaxRecs-1) ? MaxRecs-1 : RecNo+MAXROWS;
   *pTargetRow=RowNo+((*pTargetRec)-RecNo);
}


/* -------------------------------------------------------------------------
                           Answer the row on the previous screen
   ------------------------------------------------------------------------- */

void RowOnPreviousScreen(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   int Offset=(pDD->ZoomMode == ZOOM_1) ? MAXROWS : MAXROWS*3;

   *pTargetRec=(RecNo-Offset < 0) ? 0 : RecNo-Offset;
   *pTargetRow=RowNo-(RecNo-(*pTargetRec));
}


/* -------------------------------------------------------------------------
                           Answer the row on the next screen
   ------------------------------------------------------------------------- */

void RowOnNextScreen(DBDESC *pDD, int RecNo, int RowNo, int *pTargetRec, int *pTargetRow)
{
   int Offset=(pDD->ZoomMode == ZOOM_1) ? MAXROWS : MAXROWS*3;
   int MaxRecs=NumOfRecords(pDD);

   *pTargetRec=(RecNo+Offset > MaxRecs-1) ? MaxRecs-1 : RecNo+Offset;
   *pTargetRow=RowNo+((*pTargetRec)-RecNo);
}


/* -------------------------------------------------------------------------
                           Answer how many rows can be displayed on one screen
   ------------------------------------------------------------------------- */

int MaxDisplayableRows(DBDESC *pDD)
{
   return (pDD->ZoomMode == ZOOM_1) ? MAXROWS : MAXROWS*3;
}


/* -------------------------------------------------------------------------
                           Clear database screen
   ------------------------------------------------------------------------- */

void ClearRows()
{
   SetColor(WHITE_COLOR);
   SetRule(FORCE_RULE);
   Rectangle(0,12,639,MAX_Y-KEYDEPTH-2,SOLID_FILL);
}


/* -------------------------------------------------------------------------
                           Clear database viewpoint row
   ------------------------------------------------------------------------- */

int ClearRowAtIndex(DBDESC *pDD, int RowNo)
{
   int yoffset=13*Row(RowNo);
   int xoffset=Page(RowNo)*MAX_X3;

   SetColor(WHITE_COLOR);
   SetRule(FORCE_RULE);
   if (pDD->ZoomMode == ZOOM_1)
      Rectangle(1,29+yoffset,MAX_X-1,40+yoffset,SOLID_FILL);
   else
      Rectangle(xoffset+1,29+yoffset,xoffset+MAX_X3-1,40+yoffset,SOLID_FILL);
}


/* -------------------------------------------------------------------------
                           Clear database viewpoint rows
   ------------------------------------------------------------------------- */

int ClearRowsAtIndex(DBDESC *pDD, int FirstRowNo, int RowsNo)
{
   int i;

   for(i = FirstRowNo; i < FirstRowNo+RowsNo; i++)
      ClearRowAtIndex(pDD, i);
}


/* -------------------------------------------------------------------------
                           Show at tick at beginning of current row
   ------------------------------------------------------------------------- */

int HideTickAtIndex(DBDESC *pDD, int RowNo, int IsCurrent)
{
   int yoffset=13*Row(RowNo);
   int xoffset=Page(RowNo)*MAX_X3;


   SetColor((IsCurrent == TRUE) ? BLACK_COLOR : WHITE_COLOR);
   SetRule(FORCE_RULE);
   Rectangle(xoffset+1,30+yoffset,12+xoffset,40+yoffset,SOLID_FILL);
}


/* -------------------------------------------------------------------------
                           Show at tick at beginning of current row
   ------------------------------------------------------------------------- */

int ShowTickAtIndex(DBDESC *pDD, int RowNo, int IsCurrent)
{
   unsigned char TickOn[] = {
   0x01, 0x00, 0x01, 0x00, 0x0C, 0x00, 0x0C, 0x00, 0x00, 0x01,
   0x00, 0x30, 0x00, 0x60, 0x00, 0x60, 0x00, 0xC0, 0x00, 0xC0,
   0x21, 0x80, 0x31, 0x80, 0x1B, 0x00, 0x0F, 0x00, 0x04, 0x00,
   0x00, 0x00,
   };

   unsigned char InvTickOn[] = {
   0x01, 0x00, 0x01, 0x00, 0x0C, 0x00, 0x0C, 0x00, 0xFF, 0xFE,
   0xFF, 0xCF, 0xFF, 0x9F, 0xFF, 0x9F, 0xFF, 0x3F, 0xFF, 0x3F,
   0xDE, 0x7F, 0xCE, 0x7F, 0xE4, 0xFF, 0xF0, 0xFF, 0xFB, 0xFF,
   0x00, 0x00,
   };

   int xoffset=Page(RowNo)*MAX_X3;
   int yoffset=13*Row(RowNo);

   IMGHDR *pImg;

   if (IsCurrent)
      pImg=(IMGHDR*)InvTickOn;
   else
      pImg=(IMGHDR*)TickOn;

   PutImg(xoffset+1, 30+yoffset, FORCE_RULE, pImg);
}


/* -------------------------------------------------------------------------
                           Show database viewpoint row
   ------------------------------------------------------------------------- */

int ShowRow(DBDESC *pDD, void *pRecData, int RowNo, int IsCurrent)
{
   int f, from, to, fieldno, dataoffset, size, mark, index, value;
   int datacolor, rectcolor, xoffset, yoffset, width;

   BYTE *pNote, *pRec, *pData;
   char buffer[128];
   IMGHDR *pImg;

   unsigned char TickOn[] = {
   0x01, 0x00, 0x01, 0x00, 0x0C, 0x00, 0x0C, 0x00, 0x00, 0x01,
   0x00, 0x30, 0x00, 0x60, 0x00, 0x60, 0x00, 0xC0, 0x00, 0xC0,
   0x21, 0x80, 0x31, 0x80, 0x1B, 0x00, 0x0F, 0x00, 0x04, 0x00,
   0x00, 0x00,
   };

   unsigned char InvTickOn[] = {
   0x01, 0x00, 0x01, 0x00, 0x0C, 0x00, 0x0C, 0x00, 0xFF, 0xFE,
   0xFF, 0xCF, 0xFF, 0x9F, 0xFF, 0x9F, 0xFF, 0x3F, 0xFF, 0x3F,
   0xDE, 0x7F, 0xCE, 0x7F, 0xE4, 0xFF, 0xF0, 0xFF, 0xFB, 0xFF,
   0x00, 0x00,
   };

   if(!pRecData) {
      pDD->ErrorMsg = NO_REC_DATA;
      return 0;   /* nothing to display */
   }

   pData = (BYTE*)pRecData;

   if(!pDD->NumFields) {
      pDD->ErrorMsg = MISSING_FIELDEFS;
      return 0;
   }

   from=0; to=20;
   xoffset=Page(RowNo)*MAX_X3;
   yoffset=13*Row(RowNo);

   if (IsCurrent) {
      datacolor=WHITE_COLOR;
      rectcolor=BLACK_COLOR;
   } else {
      datacolor=BLACK_COLOR;
      rectcolor=WHITE_COLOR;
   }
   /* clear row area */
   SetColor(rectcolor);
   SetRule(FORCE_RULE);
   if (pDD->ZoomMode == ZOOM_1) {
      Rectangle(0,29+yoffset,639,40+yoffset,SOLID_FILL);
   } else {
      Rectangle(xoffset+1,29+yoffset,xoffset+MAX_X3-1,40+yoffset,SOLID_FILL);
      SetClip(xoffset+1,0,xoffset+MAX_X3-1,MAX_Y);
   }
   SetColor(datacolor);

   SelectFont(MEDIUM_FONT);
   xoffset=xoffset+13;
   for(f=from;f<to;f++) {

      /* find out the field number and width (in chars) */
      if (pDD->pVDEF->ColumnInfo[f] == 0x00ff)   /* no more fields? */
         break;
      fieldno=pDD->pVDEF->ColumnInfo[f] & 0x00ff;
      width=(pDD->pVDEF->ColumnInfo[f] & 0xff00) >> 8;

      /* calculate relative/absolute data offset */
      dataoffset = pDD->pFDEF[fieldno]->DataOffset;
      if(pDD->pFDEF[fieldno]->Flags & 0x20)      /* relative offset? */
         dataoffset = pData[dataoffset+1]*256 + pData[dataoffset];

      switch(pDD->pFDEF[fieldno]->FieldType) {

         /* strings */
         case STRING_FIELD:
         case PHONE_FIELD:
         case NUMBER_FIELD:
         case CURRENCY_FIELD:
         case CATEGORY_FIELD:
         case MULTILINE_FIELD:
         case COMBO_FIELD:

            pRec = &pData[dataoffset];

            strncpy(buffer, pRec, width-1);
            buffer[width-1]=0;
            SelectFont(MEDIUM_FONT);
            WriteText(xoffset, 30+yoffset, buffer);

            break;

         case NOTE_FIELD:

            index = pData[dataoffset+1]*256 + pData[dataoffset];

            /* read note record - if any */
            if(!(pNote = ReadRecord(pDD, TYPE_NOTE, index))) break;
            size = GetRecSize(pDD, TYPE_NOTE, index);
            pNote[size-HEADER_SIZE] = 0;   /* place string terminator */

            strncpy(buffer, pNote, width-1);
            buffer[width-1]=0;
            SelectFont(MEDIUM_FONT);
            WriteText(xoffset, 30+yoffset, buffer);

            free(pNote);
            break;

         case GROUP_FIELD:  /* nothing to display */
            break;

         case TIME_FIELD:
            if((pData[dataoffset] == 0) && (pData[dataoffset+1] == 0x80))
               break;   /* no time */

            value = pData[dataoffset+1]*256 + pData[dataoffset];
            sprintf(buffer, "%02d:%02d",
                    value / 60, value % 60);
            buffer[width-1]=0;
            SelectFont(MEDIUM_FONT);

            WriteText(xoffset, 30+yoffset, buffer);
            break;

         case DATE_FIELD:
            if((pData[dataoffset] == 255) &&
               (pData[dataoffset+1] == 255) &&
               (pData[dataoffset+2] == 255)) break;   /* no date */

            sprintf(buffer, "%02d/%02d/%02d",
                    pData[dataoffset+2]+1, pData[dataoffset+1]+1, pData[dataoffset]);
            buffer[width-1]=0;
            SelectFont(MEDIUM_FONT);

            WriteText(xoffset, 30+yoffset, buffer);
            break;


         case RADIO_FIELD:
            if(pData[dataoffset] == pDD->pFDEF[fieldno]->Reserved) {
               if (IsCurrent)
                  pImg=(IMGHDR*)InvTickOn;
               else
                  pImg=(IMGHDR*)TickOn;
               PutImg(xoffset, 30+yoffset, FORCE_RULE, pImg);
            }

            break;

         /* bit mask */
         case BYTEBOOL_FIELD:
         case WORDBOOL_FIELD:
            mark = pData[dataoffset+1]*256 + pData[dataoffset];
            mark &= pDD->pFDEF[fieldno]->Reserved;

            if(mark) {
               if (IsCurrent)
                  pImg=(IMGHDR*)InvTickOn;
               else
                  pImg=(IMGHDR*)TickOn;
               PutImg(xoffset, 30+yoffset, FORCE_RULE, pImg);
            }

            break;

      }

      xoffset=xoffset + 6 + (width * FNTW(MEDIUM_FONT));
   }
   if (pDD->ZoomMode == ZOOM_3)
      SetClip(0,0,MAX_X,MAX_Y);

   pDD->ErrorMsg = OPERATION_OK;

   return f;
}


/* -------------------------------------------------------------------------
                           Show database viewpoint row at index
   ------------------------------------------------------------------------- */

int ShowRowAtIndex(DBDESC *pDD, int RecIndex, int RowNo, int IsCurrent)
{
   void *pData;

   if (RecIndex >= 0) {
      pData = ReadRecord(pDD, TYPE_DATA, RecIndex);
      if (pData) {
         ShowRow(pDD, pData, RowNo, IsCurrent);
         if(((pDD->XTickMode == NO_FILTER) && pDD->pXTICKS[RecIndex]) ||
            (pDD->XTickMode == TICKS_FILTER))
            ShowTickAtIndex(pDD, RowNo, IsCurrent);
         else
            HideTickAtIndex(pDD, RowNo, IsCurrent);

         free(pData);
      }
   }
}


/* -------------------------------------------------------------------------
                           Show database viewpoint rows at index
   ------------------------------------------------------------------------- */

int ShowRowsAtIndex(DBDESC *pDD, int FirstRecIndex, int FirstRowNo, int RowsNo, int IsCurrent)
{
   int i;

   for(i = 0; i < RowsNo; i++)
      ShowRowAtIndex(pDD, FirstRecIndex+i, FirstRowNo+i, IsCurrent);
}


/* -------------------------------------------------------------------------
                           Show the tab for this database
   ------------------------------------------------------------------------- */

int ShowRowTab(DBDESC *pDD, int RecNo)
{
   char Title[128];
   int NumRec = NumOfRecords(pDD);
   int CurrentRec = (NumRec == 0) ? 0 : RecNo+1;

   sprintf(Title, "%.13s (%d/%d)", pDD->pVDEF->Name, CurrentRec, NumRec);
   ShowTab(Title);
}


/* -------------------------------------------------------------------------
                           Show one page full of rows, given the current
                           row number and position
   ------------------------------------------------------------------------- */

int ShowRows(DBDESC *pDD, int RecNo, int RowNo)
{
   int TopRec, TopRow, BottomRec, BottomRow;

   FirstVisibleRow(pDD, RecNo, RowNo, &TopRec, &TopRow);
   LastVisibleRow(pDD, RecNo, RowNo, &BottomRec, &BottomRow);

   ClearRowsAtIndex(pDD, 0, TopRow);
   ShowRowsAtIndex(pDD, TopRec, TopRow, RowNo-TopRow, 0);
   if (NumOfRecords(pDD) > 0)
      ShowRowAtIndex(pDD, RecNo, RowNo, 1);
   ShowRowsAtIndex(pDD, RecNo+1, RowNo+1, BottomRow-RowNo, 0);
   ClearRowsAtIndex(pDD, BottomRow+1, MaxDisplayableRows(pDD)-BottomRow-1);

   ShowRowTab(pDD, RecNo);
}


/* -------------------------------------------------------------------------
                           Show detailed information for a given row
   ------------------------------------------------------------------------- */

int ShowRowDetail(DBDESC *pDD, int RecNo)
{
   void *pData;

   ClearRows();
   ShowCard(pDD, 1);

   pData = ReadRecord(pDD, TYPE_DATA, RecNo);
   if (pData) {
      ShowRecord(pDD, 1, pData);
      free(pData);
      ShowRowTab(pDD, RecNo);
   }
}
