/* This file is a part of SecureDevice 1.4
   Copyright (C) 1994 by Max Loewenthal and Arthur Helwig
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <io.h>
#include <conio.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <dos.h>
#include <dir.h>
#include "md5.h"
#include "usuals.h"
#include "globals.h"
#include "common.h"

#define MinorName "CHPASS"
#define MinorVer "1.01"
#define MAGIC 1
#define KEYLEN KeySize/2
#define MAXPWDLEN 100
#define SBUFSIZE 50

#define low16(x) (x & 0xFFFF)
#define hi16(x) (x >> 16)
#define odd(x) (x & 1)
#define exists(fname) (access(fname,0)==0)

#define NOF_SYSERRORS 7

#define errWriteFile 1
#define errReadFile 2
#define errOpenFile 3
#define errSeekFile 4
#define errHideFile 5
#define errUnhideFile 6
#define errEraseFile 7

#define errPassphraseMismatch 8
#define errWrongPass 9
#define errUserAbort 10
#define errNotEnoughMem 11

char *errmsg[]={"This is not an error! Should not occur",
                "Error writing data to file",
                "Error reading data from file",
                "Error opening file",
                "Error seeking info file",
                "Error hiding file",
                "Error unhiding file",
                "Error erasing file",
                "Error: passphrases don't match",
                "Error: wrong passphrase",
                "Error: Operation aborted by user",
                "Error: Not enough heap memory available"};

extern far void IdeaCFB(void far *iv,void far *key,void far *plain,
         void far *cipher,unsigned len);
extern far void IdeaCFBx(void far *iv,void far *key,void far *cipher,
         void far *plain,unsigned len);

struct headerstruct
  {  word32 success_upto;    /* Point upto where recryption was successful */
     word16 old_iv[4];
     word16 new_iv[4];
     word16 oldpwdcheck[4];
     word16 newpwdcheck[4];
     word32 nofsects;
     word32 datalen;         /* Number of data bytes that follow           */
     word32 datacheck;       /* checksum of data bytes                     */
  } header;

sectortype *sectbuffer;
word16 Error=0;

void err(int errnum)
{  if(errnum <= NOF_SYSERRORS)
     perror(errmsg[errnum]);
     else
     printf("%s\n",errmsg[errnum]);
   exit(1);
}

void calcIV(word16 myiv[], word16 iv[], word32 sectnum)
{ myiv[0]=iv[0] ^ low16(sectnum);
  myiv[1]=iv[1] ^ hi16(sectnum);
  myiv[2]=iv[2] ^ low16(sectnum);
  myiv[3]=iv[3] ^ hi16(sectnum);
}

word32 calc_check (void *buf, word16 n)
{  word16 i;
   word32 result;
   result=0;
   for(i=0;i<n;i++)
     result += ((byte *)buf)[i];
   return(result);
}

int read_data (FILE *F, word32 sectnum, word16 n)
{   Error=0;
    if(fseek(F,sectnum * SectorSize,SEEK_SET) != 0)
      Error=errSeekFile;
    if(Error==0)
      if(fread(sectbuffer[0],SectorSize,n,F)!=n)
        Error=errReadFile;
    return(Error);
}

int write_data (FILE *F, word32 sectnum, word16 n)
{   Error=0;
    if(fseek(F,sectnum * SectorSize,SEEK_SET) != 0)
      Error=errSeekFile;
    if(Error==0)
      if(fwrite(sectbuffer[0],SectorSize,n,F)!=n)
        Error=errWriteFile;
    return(Error);
}

int copy_to_temp (char *tmpname, word32 sectnum, word16 n)
{  int i;
   FILE *Tmp;
   Error=0;
   header.success_upto = sectnum;
   header.datalen = n * SectorSize;
   header.datacheck = calc_check(sectbuffer, header.datalen);

   if((Tmp=fopen(tmpname,"rb+"))==NULL)
     Error=errOpenFile;
   if(Error==0)
     {  if(fwrite(&header,sizeof(header),1,Tmp)!=1 ||
           fwrite(sectbuffer[0],sizeof(sectbuffer[0]),n,Tmp)!=n)
          Error=errWriteFile;
        fclose(Tmp);
     }
   return(Error);
}

int copy_from_temp (char *tmpname, word16 *n, char *crcok)
{  word32 mycrc;
   FILE *Tmp;
   Error=0;
   if((Tmp=fopen(tmpname,"rb+"))==NULL)
     Error=errOpenFile;
   if(Error==0)
     {  if(fread(&header,sizeof(header),1,Tmp)!=1)
          Error=errReadFile;
        if(Error==0)
          if(fread(sectbuffer[0],header.datalen,1,Tmp)!=1)
            Error=errReadFile;
        fclose(Tmp);
     }
   mycrc = calc_check(sectbuffer[0],header.datalen);
   *crcok = mycrc==header.datacheck;
   *n = header.datalen / SectorSize;
   return(Error);
}

int recrypt(FILE *F, char *tmpname, word16 oldkey[], word16 newkey[],
            word32 startsect, char readfromtemp)
{  word32 i;
   word16 j,sectinbuf;
   word16 myiv[4];
   struct BOOT *newboot;
   char *dummy;
   Error=0;
   printf("About to recrypt %ld sectors...\n",header.nofsects);
   for(i=startsect;i<header.nofsects && Error==0;)
     {  printf("Recrypting sector %ld\r",i);
        sectinbuf = SBUFSIZE;
        if(i+SBUFSIZE < header.nofsects)
          sectinbuf = SBUFSIZE;
          else
          sectinbuf = header.nofsects - i;
        if(readfromtemp)
          {  Error=copy_from_temp (tmpname,&sectinbuf,dummy);
             readfromtemp = 0;
          }
          else
          Error=read_data (F,i,sectinbuf);
        if(Error) break;
        Error=copy_to_temp (tmpname,i,sectinbuf);
        if(Error) break;
        for(j=0;j<sectinbuf;j++)
          {  if((i+j)==0)
               /* This is the boot sector - don't recrypt, but fill in new data */
               {  newboot = (struct BOOT *)sectbuffer[j];
                  newboot->serial = rand32;
                  memcpy(newboot->check,header.newpwdcheck,sizeof(header.newpwdcheck));
                  memcpy(newboot->iv,header.new_iv,sizeof(header.new_iv));
               }
             else
             /* This is a 'normal' sector */
               {  calcIV (myiv, header.old_iv,i+j);
                  IdeaCFB(myiv,oldkey,dummy,dummy,1);
                  IdeaCFBx (myiv, oldkey, sectbuffer[j],
                            sectbuffer[j],1+SectorSize/8);

                  calcIV (myiv, header.new_iv,i+j);
                  IdeaCFB(myiv,newkey,dummy,dummy,1);
                  IdeaCFB  (myiv, newkey, sectbuffer[j],
                            sectbuffer[j],1+SectorSize/8);
               }
          }
        Error=write_data (F,i,sectinbuf);
        if(Error) break;
        i += sectinbuf;
     }
  return(Error);
}

void getpasswd(char *p,word16 maxlen)
{ word16 i=0;
  char c;
  while(i<(maxlen-1))
    {  c=getch();
       switch (c)
         { case 0x0d:
             printf("\n");
             *p='\0';
             return;
           case '\b':
             if(i>0)
               { i--;
                 p--;
                 printf("\b \b");
               }
             break;
           default:
             printf("*");
             *p++=c;
             i++;
       }
    }
}

/*      Compute IDEA encryption subkeys Z */
void en_key_idea(word16 *userkey, word16 *Z)
{ unsigned i,j;
  word16 *Y=Z;
  /*
   * shifts
   */
  for (j=0; j<8; j++)
     Z[j] = *userkey++;

  for (i=0; j<KEYLEN; j++)
  { i++;
    Z[i+7] = Z[i & 7] << 9 | Z[i+1 & 7] >> 7;
    Z += i & 8;
    i &= 7;
  }
  for(i=0;i<KEYLEN;i++)
     Y[i]^=0x0dae;
}        /* en_key_idea */

int getkey(word16 key[])
{  char pass1[MAXPWDLEN];
   MD5_CTX md5buf;
   char hash[16];

   Error=0;
   getpasswd(pass1,MAXPWDLEN);

   MD5Init(&md5buf);
   MD5Update(&md5buf,pass1,strlen(pass1));
   MD5Final(hash,&md5buf);
   burn(pass1);
   en_key_idea ((word16 *)hash, key);
   burn(hash);
   return(Error);
}


int getnewkey(word16 key[])
{  char pass1[MAXPWDLEN];
   char pass2[MAXPWDLEN];
   MD5_CTX md5buf;
   char hash[16];

   Error=0;
   printf("Enter new passphrase: ");
   getpasswd(pass1,MAXPWDLEN);
   printf("\nRe-enter new passphrase: ");
   getpasswd(pass2,MAXPWDLEN);
   if(strcmp(pass1,pass2))
     Error=errPassphraseMismatch;
   MD5Init(&md5buf);
   MD5Update(&md5buf,pass1,strlen(pass1));
   MD5Final(hash,&md5buf);
   burn(pass1);
   burn(pass2);
   en_key_idea ((word16 *)hash, key);
   burn(hash);
   return(Error);
}

int getpasswords(FILE *F, word16 oldkey[], word16 newkey[], struct BOOT **oldboot)
{  struct BOOT *boot;
   word16 i,myiv[4],check[4];
   char *dummy;
   Error=0;
   if(fread(sectbuffer[0],SectorSize,1,F)!=1)
     Error=errReadFile;
   if(Error==0)
     {  boot = (struct BOOT *)sectbuffer[0];
        *oldboot = boot;
        printf("Enter old passphrase: ");
        getkey(oldkey);

        for(i=0;i<4;i++)
          {  myiv[i]=~boot->iv[i];
             check[i]=boot->check[i];
          }
        IdeaCFB(myiv,oldkey,dummy,dummy,1);
        IdeaCFBx(myiv,oldkey,check,check,1+1);

        if(check[0] != 0x1234)
          Error=errWrongPass;
     }
   if(Error==0)
     Error=getnewkey(newkey);
   return(Error);
}

int create_temp(char *tmpname, word16 newkey[],struct BOOT *boot)
{  int i;
   FILE *Tmp;
   word16 myiv[4];
   char *dummy;

   Error=0;
   header.success_upto = 0;
   for(i=0;i<4;i++)
     {  header.old_iv[i] = boot->iv[i];
        header.oldpwdcheck[i] = boot->check[i];
        header.new_iv[i] = rand16;
        myiv[i] = ~header.new_iv[i];
     }

   header.nofsects = boot->totsectsl;

   for(i=1;i<4;i++)
     header.newpwdcheck[i]=rand16;
   header.newpwdcheck[0]=0x1234;
   IdeaCFB(myiv,newkey,dummy,dummy,1);
   IdeaCFB(myiv,newkey,header.newpwdcheck,header.newpwdcheck,1+1);

   if((Tmp=fopen(tmpname,"wb+"))==NULL)
     Error=errOpenFile;
   if(Error==0)
     fclose(Tmp);

   if(Error==0)
     Error=copy_to_temp(tmpname,0,0);

   return(Error);
}

int re_getpasswords(word16 oldkey[],word16 newkey[])
{  int i;
   word16 check[4];
   word16 myiv[4];
   char *dummy;
   printf("Enter old passphrase: ");
   getkey(oldkey);

   for(i=0;i<4;i++)
     {  myiv[i] = ~header.old_iv[i];
        check[i] = header.oldpwdcheck[i];
     }
   IdeaCFB(myiv,oldkey,dummy,dummy,1);
   IdeaCFBx(myiv,oldkey,check,check,1+1);

   if(check[0] != 0x1234)
     Error=errWrongPass;

   if(Error==0)
     {  printf("Enter new passphrase: ");
        getkey(newkey);

        for(i=0;i<4;i++)
          {  myiv[i] = ~header.new_iv[i];
             check[i] = header.newpwdcheck[i];
          }
        IdeaCFB(myiv,newkey,dummy,dummy,1);
        IdeaCFBx(myiv,newkey,check,check,1+1);

        if(check[0] != 0x1234)
          Error=errWrongPass;
      }
   return(Error);
}

int unhidefile (char *fname)
{  if(_dos_setfileattr(fname,0))
     Error = errUnhideFile;
   return(Error);
}

int hidefile (char *fname)
{  if(_dos_setfileattr(fname,FA_HIDDEN | FA_SYSTEM | FA_RDONLY))
     Error = errHideFile;
   return(Error);
}

int erase_file (char *fname)
{  if(unlink(fname)!=0)
     Error = errEraseFile;
   return(Error);
}

void giveinfo (char *s1,char *s2)
{  printf("About to chance password on volume file %s\n",s1);
   printf("Temp file will be: %s\n",s2);
   printf("\nChanging passwords is an INHERENTLY DANGEROUS operation!\n");
   printf("To minimize risc of losing your datafile, make sure that:\n");
   printf("a) You're logged out of the volume (run LOGOUT ALL if you're\n");
   printf("   not sure)\n");
   printf("b) No 'dangerous' TSR's are running\n");
   printf("c) The Temp file is located in a safe place (i.e., NOT on a\n");
   printf("   RAM disk or on a write-cached drive).\n");
   printf("\nThe process should NOT be interrupted. If it is, CHPASS.COM\n");
   printf("will attempt to continue and minimize loss of data the next time\n");
   printf("that it is run (the Temp file is used for this purpose).\n");
}

void givehelp(void)
{  printf("Usage: CHPASS <Volumename> <Tempfile name>\n\n");
   printf("You will be prompted for an old passphrase, and a new one.\n");
   printf("The entire volume will be recrypted with the new passphrase.\n");
   printf("The TempFile is used to keep track of the recrypting process.\n");
   printf("If anything goes wrong during the process, start CHPASS again\n");
   printf("with exactly the same parameters. CHPASS will then attempt to\n");
   printf("continue where it left off.\n");
   printf("NOTE 1: The entire volume will be recrypted. This can be a lengthy\n");
   printf("        process.\n");
   printf("NOTE 2: Please make sure that you are NOT logged in to the volume\n");
   printf("        that you're about to change passwords on.\n");
   exit(0);
}

int ask_continue(void)
{  char c;
   printf("Ready to continue (Y/N) ?");
   do c = toupper(getch());
   while (c != 'Y' && c != 'N');
   if(c != 'Y')
     Error = errUserAbort;
   printf("%c\n",c);
   return(Error);
}

void main(int argc, char *argv[])
{  FILE *F;
   struct BOOT *oldboot;
   word16 oldkey[KEYLEN],newkey[KEYLEN];
   word16 n;
   char dataok;

   /* Check parameters */

   printf("%s %s's %s %s\n",MajorName,MajorVer,MinorName,MinorVer);
   printf("Written by %s\n\n",AuthorName);

   randomize();
   if(argc<3) givehelp();

   if((sectbuffer=malloc(SBUFSIZE*SectorSize))==NULL)
     { Error=errNotEnoughMem;
       err(Error);
     }


   if(!exists(argv[2]))
     {  giveinfo(argv[1],argv[2]);

        if(ask_continue())
          err(Error);

        if(unhidefile(argv[1]))
          err(Error);

        if((F=fopen(argv[1],"rb+"))==NULL)
          {  Error = errOpenFile;
             err(Error);
          }

        if(getpasswords(F,oldkey,newkey,&oldboot))
          err(Error);

        /* Write initial temp file */

        if(create_temp(argv[2],newkey,oldboot))
          err(Error);

        /* Recrypt volume */

        if(recrypt(F,argv[2],oldkey,newkey,0,0))
          err(Error);
     }
     else
     {  /* Abnormal operation */
        printf("RECOVERY: Attempting to continue where left off...\n");

        if((F=fopen(argv[1],"rb+"))==NULL)
          {  Error = errOpenFile;
             err(Error);
          }

        if(copy_from_temp(argv[2],&n, &dataok))
          err(Error);

        if(re_getpasswords(oldkey,newkey))
          err(Error);

        if(dataok)
          {  if(recrypt(F,argv[2],oldkey,newkey,header.success_upto,1))
               err(Error);
          }
          else
          {  if(recrypt(F,argv[2],oldkey,newkey,header.success_upto,0))
               err(Error);
          }
    }

   printf("\nComplete.\n");

   fclose(F);
   burn(oldkey);
   burn(newkey);
   printf("Erasing temp file...\n");
   if(erase_file (argv[2]))
     err(Error);
   printf("Re-hiding volume file...\n");
   if(hidefile(argv[1]))
     err(Error);

   free(sectbuffer);

   printf("\nDone!\n");
}
