// Datex
// S.Lombaard


#define _VERSION "1.02"
#define _DATE    "24/11/98"

#include <dos.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include "serial.h"
#include "datex.h"   // Define data structures & constants



//_____________________________________________________________________
//     Serial Class Member Functions & definitions
//_____________________________________________________________________

#define VERSION 0x0102

#define FALSE           0
#define TRUE           (!FALSE)

#define NOERROR         0       /* No error               */
#define BUFOVFL         1       /* Buffer overflowed      */

#define ESC             0x1B    /* ASCII Escape character */
#define ASCII           0x007F  /* Mask ASCII characters  */
#define SBUFSIZ         0x4000  /* Serial buffer size     */

#define MARGIN				3       // Display margin

//------------------------------------------------------------------

enum subrecord_result 	{LAST_SR = 0, CORRECT_SR, WRONG_MT, WRONG_SR};
enum data_status_type	{EMPTY = 0, CORRECT, CHK_SUM_ER, INCOMPLETE};
enum data_type          {TIME = 0,
								HR,
								N_SYS,  N_DIA,  N_MEAN, N_OLD,
								SAO2, RR,
								FIO2, ETO2,
								FICO2,ETCO2,
								FIN2O,ETN2O,
								AA,   FIAA,   ETAA, MAC,
								P1,   P1_SYS, P1_DIA, P1_MEAN,
								P2,   P2_SYS, P2_DIA, P2_MEAN,
								TEMP,
								LAST };
char far *label_ptr[LAST] =
								{"hh:mm:ss ",
								"HR ",
								"Syst","Dias" ,"Mean","<60s",
								"SaO2","RR ",
								"FiO2","ETO2",
								"FiCO2","ETCO2",
								"FiN2O","ETN2O",
								"AA ","FiAA ","ETAA ", "MAC  ",
								"P1 " ,"P1-S","P1-D","P1-M",
								"P2 " ,"P2-S","P2-D","P2-M",
								"Temp"};

struct field_type
	{
	byte length;
	byte displ;
	};


//-----------------------------------------------------------------

int            SError          = NOERROR;
int            portbase        = 0;
void           interrupt(*oldvects[2])(...);

static   unsigned char  ccbuf[SBUFSIZ];
unsigned int   startbuf        = 0;
unsigned int   endbuf          = 0;

unsigned int   char_avail      = FALSE;  // TRUE if char available in buffer

byte error = FALSE;    // Pickup errors - let program exit & report if TRUE

/* Handle communications interrupts and put them in ccbuf */
void interrupt com_int(...)
{
	 disable();
	 if ((inportb(portbase + IIR) & RX_MASK) == RX_ID)
	 {
		  if (((endbuf + 1) & SBUFSIZ - 1) == startbuf)
				SError = BUFOVFL;

		  ccbuf[endbuf++] = inportb(portbase + RXR);
		  endbuf &= SBUFSIZ - 1;
	 }

	 /* Signal end of hardware interrupt */
	 outportb(ICR, EOI);
	 enable();
}

/* Output a character to the serial port */
serial& serial::operator<<( unsigned char x )
{
	 long int           timeout = 0x0000FFFFL;

	 outportb(portbase + MCR,  MC_INT | DTR | RTS);

	 /* Wait for Clear To Send from modem */
	 while ((inportb(portbase + MSR) & CTS) == 0)
		  if (!(--timeout))
			return *this;

	 timeout = 0x0000FFFFL;

	 /* Wait for transmitter to clear */
	 while ((inportb(portbase + LSR) & XMTRDY) == 0)
		  if (!(--timeout))
			return *this;

	 disable();
	 outportb(portbase + TXR, x);
//cout << "Tx:" << (int) x << ' ';
	 enable();

	return *this;
}

/* Output a string to the serial port */
serial& serial::operator<<( char *string )
{
	while (*string)
	{
		(*this) << *string;
		string++;
	}
	return *this;
}

/* This routine returns the current value in the buffer */
serial &serial::operator>>( unsigned char &ch )
{
	if (endbuf == startbuf)
	{
		ch = 0;
		char_avail = FALSE;
		return *this;
	}

	ch = ccbuf[startbuf];
//cout << "Rx:" << (int) ch << ' ';
	char_avail = TRUE;
	startbuf++;
	 startbuf %= SBUFSIZ;
	return *this;
}

/* Install our functions to handle communications */
void setvects(void)
{
	 oldvects[0] = getvect(0x0B);
	 oldvects[1] = getvect(0x0C);
	 setvect(0x0B, com_int);
	 setvect(0x0C, com_int);
}

/* Uninstall our vectors before exiting the program */
void resvects(void)
{
	 setvect(0x0B, oldvects[0]);
	 setvect(0x0C, oldvects[1]);
}

/* Turn on communications interrupts */
void i_enable(int pnum)
{
	 int c;

	 disable();
	 c = inportb(portbase + MCR) | MC_INT;
	 outportb(portbase + MCR, c);
	 outportb(portbase + IER, RX_INT);
	 c = inportb(IMR) & (pnum == COM1 ? IRQ4 : IRQ3);
	 outportb(IMR, c);
	 enable();
}

/* Turn off communications interrupts */
void i_disable(void)
{
	 int c;

	 disable();
	 c = inportb(IMR) | ~IRQ3 | ~IRQ4;
	 outportb(IMR, c);
	 outportb(portbase + IER, 0);
	 c = inportb(portbase + MCR) & ~MC_INT;
	 outportb(portbase + MCR, c);
	 enable();
}

/* Tell modem that we're ready to go */
void comm_on(void)
{
	 int                c, pnum;

	 pnum = (portbase == COM1BASE ? COM1 : COM2);
	 i_enable(pnum);
	 c = inportb(portbase + MCR) | DTR | RTS;
	 outportb(portbase + MCR, c);
}

/* Go off-line */
void serial::comm_off(void)
{
	 i_disable();
	 outportb(portbase + MCR, 0);
}

void serial::init_serial(void)
{
	 endbuf = startbuf = 0;
	 setvects();
	 comm_on();
}

serial::~serial()
{
	 comm_off();
	 resvects();
}

/* Set the port number to use */
int serial::SetPort(int Port)
{
	 int                Offset, far *RS232_Addr;

	 switch (Port)
	 { /* Sort out the base address */
		case COM1 : Offset = 0x0000;
						break;
		case COM2 : Offset = 0x0002;
						break;
		default   : return (-1);
	 }

	RS232_Addr = (int far *)MK_FP(0x0040, Offset);  /* Find out where the port is. */
	 if (*RS232_Addr == NULL) return (-1);/* If NULL then port not used. */
	 portbase = *RS232_Addr;              /* Otherwise set portbase      */

	 return (0);
}

/* This routine sets the speed; will accept funny baud rates. */
/* Setting the speed requires that the DLAB be set on.        */
int serial::SetSpeed(int Speed)
{
	 unsigned char		c;
	 int		divisor;

	 if (Speed == 0)            /* Avoid divide by zero */
		  return (-1);
	 else
		  divisor = (int) (115200L/Speed);

	 if (portbase == 0)
		  return (-1);

	 disable();
	 c = inportb(portbase + LCR);
	 outportb(portbase + LCR, (c | 0x80)); /* Set DLAB */
	 outportb(portbase + DLL, (divisor & 0x00FF));
	 outportb(portbase + DLH, ((divisor >> 8) & 0x00FF));
	 outportb(portbase + LCR, c);          /* Reset DLAB */
	 enable();

	 return (0);
}

/* Set other communications parameters */
int serial::SetOthers(int Parity, int Bits, int StopBit)
{
	 int                setting;

	 if (portbase == 0)					return (-1);
	 if (Bits < 5 || Bits > 8)				return (-1);
	 if (StopBit != 1 && StopBit != 2)			return (-1);
	 if (Parity != NO_PARITY && Parity != ODD_PARITY && Parity != EVEN_PARITY)
							return (-1);

	 setting  = Bits-5;
	 setting |= ((StopBit == 1) ? 0x00 : 0x04);
	 setting |= Parity;

	 disable();
	 outportb(portbase + LCR, setting);
	 enable();

	 return (0);
}

/* Set up the port */
serial::serial(int Port, int Speed, int Parity, int Bits, int StopBit)
{
	flag = 0;
	if (SetPort(Port))
	  flag = -1;
	if (SetSpeed(Speed))
	  flag = -1;
	if (SetOthers(Parity, Bits, StopBit))
	  flag = -1;

	if (!flag)
		init_serial();
}

/*  Control-Break interrupt handler */
int c_break(void)
{
	 i_disable();
	 fprintf(stderr, "\nStill online.\n");

	 return(0);
}

//_________________________________________________________________________
// Datex Receive & Transmit Class                                                                        9
//_________________________________________________________________________


class datex : public serial
{
private:
	byte rx_on;							// Indicates start or end of data
	int  add_5th_bit;		 			// Control char received, add 5th bit to next byte
	int  rx_byte_valid;				// Char ready in buffer

		// Main Datex Record variables
	datex_record_union_type far *record_ptr;	// record pointer to data area
	int monitor_version;				// Software version
	int main_type;						// Maintype of record
											// DRI_MT_PHDB = 0 ~_WAV = 1

public:
		// Constructor & destructor
	datex(void);
	~datex(void);

		// Redefine << & >> to interpret frame & ctrl characters
	datex&	operator<<( byte b );

		// Receive byte: If 1st framechar then rx_on = TRUE
		// Ignore byte if rx_byte_valid = FALSE
	datex&	operator>>( byte &b );
		// Send start & stop frame characters
	void		tx_start(void) { serial::operator<<(FRAMECHAR); };
	void		tx_end(void)   { serial::operator<<(FRAMECHAR); };

		// Send buffer with application data
		// This include Datex record and checksum
	void	tx_buffer(int len);

		// Receive buffer of application data (record + checksum)
		// until maximum length exceeded or framechar reached
		// This will only continue if a framehar is received.
	int   rx_buffer(void);

		// Clear data area & Set up request for data
	void clear_data_area(void);
	void req_transmission(byte type = 1, short interval = -1);

		// Get subrcord pointer
	phdb_type far* phdb_ptr(int nr, int &status);

		// Return record pointer
	datex_record_union_type far* main_record_ptr(void) {return record_ptr; };

		// Check serial port for Frame Char
	int check(void);
};

datex::datex(void) : serial(COM1, 19200, EVEN_PARITY, 8, 1)
	{
	rx_on				= FALSE;
	rx_byte_valid  = FALSE;
	add_5th_bit 	= FALSE;
	record_ptr		= new datex_record_union_type;
	monitor_version = -1;
	main_type 		= -1;

	}

datex::~datex(void)
	{
	delete record_ptr;
	}

datex& datex::operator<<( byte b)   // Transmit byte
	{
	switch (b)
		{
		case FRAMECHAR :	serial::operator<<(CTRLCHAR);
								serial::operator<<(FRAMECHAR & BIT5COMPL);
								break;
		case CTRLCHAR  :  serial::operator<<(CTRLCHAR);
								serial::operator<<(CTRLCHAR & BIT5COMPL);
								break;
		default : serial::operator<<(b);
		}
	return(*this);
	}


datex& datex::operator>>( byte &b)
	{
	serial::operator>>( b);
	if (char_avail)
		{
		switch (b)
			{
			case CTRLCHAR:	rx_byte_valid = FALSE;
								add_5th_bit   = TRUE;
								b = 0;
								break;
			case FRAMECHAR:	rx_byte_valid = FALSE;
									if (rx_on)	rx_on = FALSE;
									 else			rx_on = TRUE;
									break;
			default :   if (add_5th_bit)
								{
								b |= BIT5;
								add_5th_bit = FALSE;
								}
							if (rx_on)				 // Only receive following FRAMECHAR
								rx_byte_valid = TRUE;
							else
								rx_byte_valid = FALSE;
							break;
			}
		}
	else
		rx_byte_valid = 0;

	return(*this);
	}

int datex::check(void)
	{
	byte b, status;
	(*this) >> b;
	if (char_avail)
		{
		if ((b == FRAMECHAR) && (rx_on))
			return TRUE;
		}
	return FALSE;
	}

void datex::tx_buffer(int len)
	{
	int i;
	byte check_sum,
		  far *p = (*record_ptr).b;

	tx_start(); 					// Start Transfer
	check_sum = 0;
	for(i = 0; i < len;i++, p++)
		{
		check_sum += *p;
		(*this) << *p;				// Send byte
		}
	(*this) << check_sum;		// Send Checksum
	tx_end();                  // End transfer

	}

	//Receive data: 1: CORRECT, 2: CHK_SUM_ER, 3: INCOMPLETE

int datex::rx_buffer(void)
	{
	int i			= 0,
		 max_len	= sizeof(datex_record_union_type),
		 escape	= FALSE,
		 status;

	byte	a  = 0, b = 0, sum = 0,
			far *p = (*record_ptr).b;


	while ((rx_on) && (!escape))
		{
		if (kbhit())
			escape = ((getch() == ESC) ? TRUE : FALSE);
		(*this) >> b;
		if (rx_byte_valid)
			{
			if (i < max_len)
				p[i++] = b, sum += a, a = b;

			}
		}

	monitor_version = record_ptr->record.hdr.r_dri_level;

	if (escape)
		{
		rx_on = FALSE;
		status = INCOMPLETE;
		}
	else
		{
		if (sum == a) status = CORRECT;
		else status = CHK_SUM_ER;
		}

	return status;         // Exclude checksum from count
	}


void datex::clear_data_area(void)
	{
	int idx;
	for(idx = 0; idx < sizeof(datex_record_union_type); idx++)
			(*record_ptr).b[idx] = 0;
	}

void datex::req_transmission(byte type, short interval)
	{
	phdb_req_type *phdb_req_ptr;

		// Set pointer to start of data area
	phdb_req_ptr = (phdb_req_type*) (*record_ptr).record.data;

		// Clear data area
	clear_data_area();

		// Set record header
	record_ptr->record.hdr.r_len 			= 	sizeof(datex_hdr_type)
													+ sizeof(phdb_req_type);
	record_ptr->record.hdr.r_dri_level	= 0;
	record_ptr->record.hdr.r_time			= 0;
	record_ptr->record.hdr.r_maintype		= DRI_MT_PHDB; // phys data request type = 0
	record_ptr->record.hdr.sr_desc[0].offset = 0;
	record_ptr->record.hdr.sr_desc[0].sr_type= 0;      // Physiological data request
	record_ptr->record.hdr.sr_desc[1].offset = 0;
	record_ptr->record.hdr.sr_desc[1].sr_type= 0xFF;   // Last subrecord

		// Request transmission subrecord
	phdb_req_ptr->phdb_rcrd_type	=	type;			// Current values
	phdb_req_ptr->tx_interval		=  interval;	// Request single record

	tx_buffer(sizeof(datex_hdr_type) + sizeof(phdb_req_type));
	}


	// Return phdb data pointer
	// 0	-> LAST_SR, 1	-> CORRECT, 2	-> WRONG_MT, 3	-> WRONG_SR

phdb_type far* datex::phdb_ptr(int nr, int &status)
	{
	int i, sr_type;
	phdb_type far *p;

	if (record_ptr->record.hdr.r_maintype != DRI_MT_PHDB)
		{
		status = WRONG_MT;
		return NULL;
		}

		// Check subrecord type
	sr_type = record_ptr->record.hdr.sr_desc[nr].sr_type;

	if (sr_type == 0xFF)
		{
		status = LAST_SR;
		return NULL;
		}

	if ((sr_type < 1) || (sr_type > 3))
		{
		status = WRONG_SR;
		return NULL;
		}

		// Set pointer to subrecord
	p = (phdb_type far*) ((*record_ptr).record.data + (*record_ptr).record.hdr.sr_desc[nr].offset);


	status = CORRECT_SR;
	return p;
	}


//_________________________________________________________________________
// Display data                                                                        9
//_________________________________________________________________________

class display
{

public:
	enum line_usage { _menu = 1,
							_info,
							_status,
							_line,
							_headings,
							_results };

	display(void);
	~display(void);
	void results(datex &comm, int status);
	void clr(void);
	void status(char *s);
	void info(datex_record_union_type far* r, int check);
	void menu(char *s);
	void headings(void);
	void write_headings(void);	// Write labels
	void show(phdb_type far *ptr);
	char* i_convert(int x, int d, char *s);
	char* f_convert(int x, int d, char *s);
	char* fixed_itoa(int val, char *s);
	void write_data(data_type t);
	void change(data_type d);
	void set_result_window(void);
	void reset_window(void);

		// File operations
	void f_open(void);
	void f_open(char *s);
	void f_open_new(void);
	void close(void);


		// Settings

	void set(data_type t);
	void unset(data_type tt);

	char separator;


private:

		// Display variables
	word x,y;					// Last text position
	word x_max, y_max;
	char	str[80],			// String areas  - str = output string
			tmp[80],       //               - tmp = temporary strings
			tmp2[20];

	field_type field[LAST];	// Info on Column headings

		// Time variables
	dword start_time;

		// Misc
	void fill(char f, int m);   // Fill rest off line up to position m

		// File variables
	ofstream record_file;	// output file class
	char f_name[80];		// Filename
	int save_on;			// To write or not to write
	int write_labels;		// Labels written / not

};

display::display(void)
	{
	int i;

		// initialise display variables
	clr();
	x = 1, y = _results;
	x_max = 80, y_max = 25;
	textmode(BW80);
	gotoxy(x,y);
	separator = ',';


	cout << "Waiting";

		// Initialise file variables
	save_on = FALSE;
	write_labels = FALSE;

		// Initialise fields
	start_time = 0;

	field[TIME].displ = 1;
	field[HR].displ	= 1;
	field[N_SYS].displ = 1, field[N_DIA].displ = 1, field[N_MEAN].displ = 0, field[N_OLD].displ= 1; ;
	field[SAO2].displ = 1, field[RR].displ = 1;
	field[FIO2].displ = 1, field[ETO2].displ = 0;
	field[FICO2].displ = 0, field[ETCO2].displ = 1;
	field[FIN2O].displ = 0, field[ETN2O].displ	= 0;
	field[AA].displ	= 1, field[FIAA].displ = 0, field[ETAA].displ = 1, field[MAC].displ	= 1,
	field[P1].displ	= 0, field[P1_SYS].displ = 0, field[P1_DIA].displ = 0, field[P1_MEAN].displ = 0;
	field[P2].displ	= 0, field[P2_SYS].displ = 0, field[P2_DIA].displ = 0, field[P2_MEAN].displ = 0;
	field[TEMP].displ = 0;

	for(i=0;i<LAST;i++) field[i].length = strlen(label_ptr[i]) + 1;
	}

display::~display()
	{
	}


void display::fill(char f, int m)
	{
	int l, s;
	if ((s = wherex()) >= m) return;
	for(l=s; l < m; l++) cout << f;
	}

void display::write_data(data_type t)
	{
	int i;
	if (wherex() < x_max - MARGIN)
		cout << str;
	if (save_on)
		record_file << str << separator;
	fill(' ', x += field[t].length);
	}

char* display::i_convert(int x, int d, char *s)
	{
	if DATA_OK(x) itoa(x / d, s, 10);
	else s[0] = '-', s[1] = 0;
	return s;
	}

char* display::f_convert(int x, int d, char *s)
	{

	if (!DATA_OK(x))
		{
		strcpy(s, "-");
		return s;
		}
	strcpy(s, itoa(x / d, tmp, 10));
	strcat(s, ".");
	strcat(s, itoa(x % d, tmp, 10));
	return s;
	}

char* display::fixed_itoa(int v, char *s)
	{
	tmp2[0] = '0' + (char) (v / 10), tmp2[1] = '0' + (char) (v % 10), tmp2[2] = 0;
	strcat(s, tmp2);
	return s;
	}

void display::show(phdb_type far *p)
	{
	byte b;
	float fx,fy,fz;

	if (field[TIME].displ)
		{
		str[0] = 0;
		fixed_itoa((p->time - start_time) / 3600, str);
		strcat(str, ":");
		fixed_itoa(((p->time - start_time) / 60) % 60, str);
		strcat(str, ":");
		fixed_itoa((p->time - start_time) % 60, str);
		write_data(TIME);
		}
	if (field[HR].displ)
		{
		i_convert(p->ecg.hr, 1, str);
		if ((p->ecg.hr < 60) && (p->ecg.hr > 100))
			{
			if ((p->SpO2.pr > 60) && (p->SpO2.pr < 100))
				i_convert(p->SpO2.pr, 1, str);
			}
		write_data(HR);
		}
	if (field[N_SYS].displ)
		{
		i_convert(p->nibp.sys, 100, str);
		write_data(N_SYS);
		}
	if (field[N_DIA].displ)
		{
		i_convert(p->nibp.dia, 100, str);
		write_data(N_DIA);
		}
	if (field[N_MEAN].displ)
		{
		i_convert(p->nibp.mean, 100, str);
		write_data(N_MEAN);
		}
	if (field[N_OLD].displ)
		{
		if (p->nibp.hdr.label_info & 0x100) str[0] = '-';
		else str[0] = '*';
		str[1] = 0;
		write_data(N_OLD);
		}
	if (field[SAO2].displ)
		{
		i_convert(p->SpO2.SpO2, 100, str);
		write_data(SAO2);
		}
	if (field[RR].displ)
		{
		i_convert(p->co2.rr, 1, str);
		write_data(RR);
		}
	if (field[FIO2].displ)
		{
		i_convert(p->o2.fi, 100, str);
		write_data(FIO2);
		}
	if (field[ETO2].displ)
		{
		i_convert(p->o2.fi, 100, str);
		write_data(ETO2);
		}
	if (field[FICO2].displ)
		{
		fx = (float) p->co2.amb_press , fy = (float) p->co2.fi;
		fz = fx * fy / 1000;
		i_convert((int) fz, 100, str);
		write_data(FICO2);
		}
	if (field[ETCO2].displ)
		{
		fx = (float) p->co2.amb_press , fy = (float) p->co2.et;
		fz = fx * fy / 1000;
		i_convert((int) fz, 100, str);
		write_data(ETCO2);
		}
	if (field[FIN2O].displ)
		{
		i_convert(p->n2o.fi, 100, str);
		write_data(FIN2O);
		}
	if (field[ETN2O].displ)
		{
		i_convert(p->n2o.et, 100, str);
		write_data(ETN2O);
		}
	if (field[AA].displ)
		{
		switch (p->aa.hdr.label_info & 0x3F)
			{
			case 0:	strcpy(str, "-");		break;
			case 1:	strcpy(str, "Nil");	break;
			case 2:	strcpy(str, "Hal");	break;
			case 3:	strcpy(str, "Enf");	break;
			case 4:	strcpy(str, "Iso");	break;
			case 5:	strcpy(str, "Des");	break;
			case 6:	strcpy(str, "Sev");	break;
			default: strcpy(str, "?");		break;
			}
		write_data(AA);
		}
	if (field[FIAA].displ)
		{
		f_convert(p->aa.fi, 100, str);
		write_data(FIAA);
		}
	if (field[ETAA].displ)
		{
		f_convert(p->aa.et, 100, str);
		write_data(ETAA);
		}
	if (field[MAC].displ)
		{
		f_convert(p->aa.mac_sum, 100, str);
		write_data(MAC);
		}
	if (field[P1].displ)
		{
		switch (p->p[0].hdr.label_info)
			{
			case 0:	strcpy(str, "-");   break;
			case 1:	strcpy(str, "Art"); break;
			case 2:	strcpy(str, "Cvp"); break;
			case 3:	strcpy(str, "PA");  break;
			case 4:	strcpy(str, "RAP"); break;
			case 5:	strcpy(str, "RVP"); break;
			case 6:	strcpy(str, "LAP"); break;
			default: strcpy(str, "P1");  break;
			}
		write_data(P1);
		}
	if (field[P1_SYS].displ)
		{
		i_convert(p->p[0].sys, 100, str);
		write_data(P1_SYS);
		}
	if (field[P1_DIA].displ)
		{
		i_convert(p->p[0].dia, 100, str);
		write_data(P1_DIA);
		}
	if (field[P1_MEAN].displ)
		{
		i_convert(p->p[0].mean, 100, str);
		write_data(P1_MEAN);
		}
	if (field[P2].displ)
		{
		switch (p->p[1].hdr.label_info)
			{
			case 0:	strcpy(str, "-");   break;
			case 1:	strcpy(str, "Art"); break;
			case 2:	strcpy(str, "Cvp"); break;
			case 3:	strcpy(str, "PA");  break;
			case 4:	strcpy(str, "RAP"); break;
			case 5:	strcpy(str, "RVP"); break;
			case 6:	strcpy(str, "LAP"); break;
			default: strcpy(str, "P2");  break;
			}
		write_data(P2);
		}
	if (field[P2_SYS].displ)
		{
		i_convert(p->p[1].sys, 100, str);
		write_data(P2_SYS);
		}
	if (field[P2_DIA].displ)
		{
		i_convert(p->p[1].dia, 100, str);
		write_data(P2_DIA);
		}
	if (field[P2_MEAN].displ)
		{
		i_convert(p->p[1].mean, 100, str);
		write_data(P2_MEAN);
		}

	if (field[TEMP].displ)
		{
		i_convert(p->t[0].temp, 100, str);
		write_data(TEMP);
		}


	}

void display::results(datex &comm, int status)
	{
	int 	i = 0,
			result = 0;
	phdb_type far *ptr;

	gotoxy(1, y = wherey());
	if (save_on)
		write_headings();			// Make sure headings have been written before new set is displayed
	do	{
		ptr = comm.phdb_ptr(i++, result);
		if (result == CORRECT_SR)
			{
			if (start_time == 0) start_time = ptr->time;

				// Display results for this subrecord
			x = 1;
			show(ptr);

			cout << endl;
			record_file << endl;

			}

		} while ((result != LAST_SR) && (i < 8));
	}

void display::clr(void)
	{
	clrscr();
	}

void display::menu(char *s)
	{
	x = wherex();
	y = wherey();
	gotoxy(1,_menu);
	cout << s;
	fill(' ',x_max);
	gotoxy(x,y);
	}

void display::info(datex_record_union_type far* r, int check)
	{
	x = wherex();
	y = wherey();
	gotoxy(1,_info);
	cout  << "Monitor version: " << (int) r->record.hdr.r_dri_level
			<< " Main data: " << (int) r->record.hdr.r_maintype
			<< " Length:" << (int) r->record.hdr.r_len << " Check: ";
	if (check)	cout << "correct";
	else        cout << "wrong";
	if (save_on) cout << " File:" << f_name;
	fill(' ',x_max);
	gotoxy(x,y);
	}

void display::status(char *s)
	{
	x = wherex();
	y = wherey();
	gotoxy(1,_status);
	cout << s;
	fill(' ',x_max);
	gotoxy(x,y);
	}

void display::headings(void)
	{
	int i, tx;
	x = wherex(), y = wherey();
	gotoxy(tx = 1,_headings);
	for(i = 0; i < LAST;i++)
		{
		if ((field[i].displ) && (tx < x_max - MARGIN))
			{
			cout << label_ptr[i];
			fill(' ', tx += field[i].length);
			}
		}
	fill(' ',x_max);
	gotoxy(1,_line);
	fill('=',x_max);
	gotoxy(x,y);
	}

void display::write_headings(void)
	{
	int i;
	if (!write_labels) return;
	for(i = 0; i < LAST;i++)
		if (field[i].displ)	record_file << label_ptr[i] << separator;
	record_file << endl;
	write_labels = FALSE;
	}

void display::change(data_type d)
	{
	if (d >= LAST) return;
	field[d].displ = ((field[d].displ) ? 0 : 1);
	write_labels = TRUE;				// Save Label headings to file
	}

void display::f_open(void)
	{
	record_file.open(f_name, ios::app);
	if (record_file.fail())
		{
		status(" * Unable to open file - Press any key to continue *    ");
		getch();
		save_on = FALSE;
		return;
		}
	save_on = TRUE;
	write_labels = TRUE;
	record_file << "Datex Data CSV File, Version " << _VERSION << endl;
//	write_headings();

	}



void display::f_open(char *s)
	{
	strcpy(f_name, s);
	f_open();
	}

void display::f_open_new(void)
	{
	char c;

	if (save_on)
		{
		status(" * File open already - Continue? (y/n) *");
		c = getch();
		status("");
		if (c != 'y') return;
		else record_file.close();
		}
	status("Enter filename:");
	x = wherex(), y = wherey();
	gotoxy(20,_status);
	cin.getline(f_name, sizeof(f_name));
	gotoxy(x,y);
	status("");
	f_open();
	}

void display::close(void)
	{
	record_file.close();
	save_on = FALSE;
	}

void display::set(data_type t)
	{
	field[t].displ = TRUE;
	}

void display::unset(data_type t)
	{
	field[t].displ = FALSE;
	}

//_________________________________________________________________________
// Menu                                                                        9
//_________________________________________________________________________

class menu
{
private:
	int nr;
	display *out;
	datex   *com;
	char *list[3];
	int refresh;

	int m_list0(byte c);
	int m_list1(byte c);
	int m_list2(byte c);

public:
	menu(display *dp, datex *cp);
	~menu(void) { };
	menu_select(int i);
	void m_list(int &r);
	void set_refresh(void) { refresh = TRUE; };

};

menu::menu(display *dp, datex *cp)
	{
	nr  = 0;
	out = dp, com = cp;
	list[0] = "Esc:Exit  1:Xmit once 2:5s 3:10s 4:20s 5:1m 6:2m 7:5m c:Cancel l:Field f:File";
	list[1] = "Esc:Main  a:T b:HR c-:NIBP g:SaO2 h:RR ij:O2 kl:CO2 mn:N2O o-:AA s-:P1 w-:P2";
	list[2] = "Esc:Main  s:Save        c:Close";
	out->menu(list[nr]);
	refresh = FALSE;
	}

void menu::m_list(int &r)
	{
	char c;

	if (refresh)
		{
		out->menu(list[nr]);
		refresh = FALSE;
		}
	if (kbhit())
		{
		out->status("Press any key");
		c = getch();
		out->status("");
		switch (nr)
			{
			case 0: r = m_list0(c); break;
			case 1: r = m_list1(c); break;
			case 2: r = m_list2(c); break;
			}
		}
	}

int menu::m_list0(byte c)
	{
	int done = FALSE;
	switch (c)
		{
		case ESC:	done = TRUE;  /* Exit program */
						break;
		case '1':   out->status("Sending Xmit once");
						com->req_transmission(1, -1);
						out->status("Done");
						break;
		case '2':   out->status("Sending request for 5s transmissions");
						com->req_transmission(DRI_PH_DISPL,		5);
						out->status("Done");
						break;
		case '3':   out->status("Sending request for 10s transmissions");
						com->req_transmission(DRI_PH_DISPL,		10);
						out->status("Done");
						break;
		case '4':   out->status("Sending request for 20s transmissions");
						com->req_transmission(DRI_PH_DISPL,		20);
						out->status("Done");
						break;
		case '5':   out->status("Sending request for 1 min transmissions");
						com->req_transmission(DRI_PH_DISPL,		60);
						out->status("Done");
						break;
		case '6':   out->status("Sending request for 2 min transmissions");
						com->req_transmission(DRI_PH_DISPL,     120);
						out->status("Sending");
						break;
		case '7':   out->status("Sending request for 5 min transmissions");
						com->req_transmission(DRI_PH_DISPL,     300);
						out->status("Sending");
						break;
		case 'c':   out->status("Sending cancellation");
						com->req_transmission(DRI_PH_60S_TREND, 0);
						com->req_transmission(DRI_PH_DISPL,     0);
						out->status("Cancelled");
						break;
		case 'l':	refresh = TRUE;
						nr = 1;
						break;
		case 'f':	refresh = TRUE;
						nr = 2;
						break;
		default :	(*com) << c;
		}
	return done;
	}

int menu::m_list1(byte c)
	{
	if (c == ESC)
		{
		nr = 0;  /* Goto main menu  */
		refresh = TRUE;
		out->write_headings();
		}
	else
		{
		if (c >= 'a')
			{
			out->change((data_type) (c - 'a'));
			out->headings();
			}
		}
	return 0;
	}

int menu::m_list2(byte c)
	{
	switch (c)
		{
		case ESC:	nr = 0;  /* Goto main menu  */
						refresh = TRUE;
						break;
		case 's':   out->f_open_new();
						break;
		case 'c':   out->close();
						break;
		}

	return 0;
	}

//_________________________________________________________________________




void main(int argc, char *argv[])
{
	 /* Communications parameters */
	int	done  = FALSE,
			data_status = 0,
			i;
	byte	c;

	datex comport;							// Initialize communications class
	display output;						// Initialise output class
	menu options(&output, &comport);	// initialise menu class

	if (argc > 1)
		for(i = 1; i < argc; i++)
			{
			if (argv[i][0] == '/')
				{
				switch (argv[i][1])
					{
					case 't' :	output.separator = 9;
									break;
					case 'c' :  output.set(P2_MEAN);
									break;
					case 'a' :	output.set(P1_SYS);
									output.set(P1_DIA);
									output.unset(N_SYS);
									output.unset(N_DIA);
									output.unset(N_OLD);
									break;
					case '?' :  clrscr();
									cout << "Use: datex [/t] [/c] [/a] filename\n";
									cout << " S Lombaard " << _DATE << " Version " << _VERSION << '\n';
									cout << " /t : tab separated values\n";
									cout << " /c : enable p2_mean\n";
									cout << " /a : enable p1_syst & p1_diast, disable nibp\n\n";
									exit(0);
									break;
					}
				}
			else
				output.f_open(argv[1]);	// Use command line filename if supplied
			}
	output.headings();
	ctrlbrk(c_break);

	if (comport.main_record_ptr())
		{

		do {
			options.m_list(done);

			data_status = (comport.check() ? comport.rx_buffer() : EMPTY);

			switch (data_status)
				{
				case CORRECT	:	output.results(comport, data_status);
										output.status("Data received : Checksum ok");
										output.info(comport.main_record_ptr(), TRUE);
										output.headings();
										options.set_refresh();
										break;
				case CHK_SUM_ER:	// output.results(comport, data_status);
										output.status("Data received : Checksum error => Requesting Xmit");
										output.info(comport.main_record_ptr(), FALSE);
										comport.req_transmission(1, -1);
										output.status("Xmit once sent");
										output.headings();
										options.set_refresh();
										break;
				case INCOMPLETE:	output.status("Receive cancelled");
										break;
				}

			} while (!done && !SError);
		}
	 else
		{
		cout << "Unable to allocate memory\n";
		getch();
		}


	 /* Check for errors */
	switch (SError)
		{
		case NOERROR:	fprintf(stderr, "\nBye.\n");
							return;

		case BUFOVFL:	fprintf(stderr, "\nBuffer Overflow.\n");
							return;

		default:		fprintf(stderr, "\nUnknown Error, SError = %d\n",
						SError);
						return;

		}



}


