;;************************************************************
;;       memory.inc (memory and port utilities)
;;************************************************************
;;
;;  Copyright (C) 1990 Vance Morrison
;;
;; Permission to view, compile, and modify for LOCAL (intra-organization)
;; USE ONLY is hereby granted, provided that this copyright and permission
;; notice appear on all copies.  Any other use by permission only.
;;
;; Vance Morrison makes no representations about the suitability
;; of this software for any purpose.  It is provided "as is" without expressed
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;;******************************************************************************
;;
;; In order to do fast 16 bit transfers between cards with shared memory
;; we need to tell each card that it can use all 16 bits.  Unforunately,
;; we can't leave this on all the time (because this bit is shared among
;; hardward that may want it unasserted).  Thus before each transfer, the
;; source and destination card need to be places in 16 bit mode.
;;
;; Because in PCroute determining the destination card is difficult, and
;; since it is unlikely that there will be configurations with a mix
;; of 8 and 16 bit shared memory cards, we take a simplified view.
;; Namely if any 8 bit shared memory boards exist, everyone does 8 bit
;; transfers all the time.  If all boards are 16 bit, then before any copy,
;; all cards are placed in 16 bit mode.
 
;; of networking cards is not transparent (at least not with the WD8013EBT).
;; because of this, we need to toggle 16bit mode when it is necessary
;; (it is also necessary to do this with interupts disabled, which also
;; requires special processing).  memory.inc provides functions that allow
;; this toggling to be done automatically.
;;
;;******************************************************************************
;;
;; functions provided by this module are
;;
;;   MEM_DECLARE_16BIT on_macro, off_macro
;;   MEM_DECLARE_8BIT
;;   MEM_DECLARE_NEED_INTS lantency
;;   MEM_COPY
;;   PORT_READ
;;   PORT_WRITE
;;
;;******************************************************************************
 
mem_num_16bit = 0
mem_num_8bit = 0
mem_min_latency = -1
 
BIT8	= 0
BIT16	= 1

;;******************************************************************************
;;  MEM_DECLARE_16BIT declares the fact that a 16 bit shared memory card
;;  exists in the system.  'on_macro' turns 16 bit mode on for that card`
;;  'off_macro' turns it off.  Interfaces that use main memory for buffering
;;  need not be declared.  Note that on and off macros CAN ONLY MODIFY
;;  THE AX AND DX REGISTERS.  'id' is a number passed to the on  and off
;;  macros when they are called.
 
MEM_DECLARE_16BIT MACRO on_macro, off_macro, id
 
    mem_num_16bit = mem_num_16bit + 1
 
    HELPER MACRO num
        mem_16bit_&num&_on  equ <on_macro>
        mem_16bit_&num&_off equ <off_macro>
        mem_16bit_&num&_id = id
    ENDM
 
    HELPER %mem_num_16bit
ENDM
 
;;******************************************************************************
;; MEM_DECLARE_NEED_INTS declares the fact that there is an interface
;; that uses interupts that need to be serviced quickly, so interupts
;; cannot remain disabled for long times.  'lantency' is the worst case
;; latency required measured in usec.  For example a SLIP line at 19.2K
;; baud requires that interupts be serviced in 500 us.
;;
 
MEM_DECLARE_NEED_INTS MACRO latency
    IF mem_min_latency gt latency       ;; take the minimum
        mem_min_latency = latency
    ENDIF
ENDM
 
 
;;******************************************************************************
;;  MEM_DECLARE_8BIT declares the fact that a 8 bit shared memory card
;;  exists in the system.   In this implementation, this means don't ever
;;  use 16 bit mode at all.
 
MEM_DECLARE_8BIT MACRO
    mem_num_8bit = 1
ENDM
 

;;******************************************************************************
;; MEM_COPY does the copy operation.  It requires
;; that SI and DI are EVEN.   This function is esentially a 'rep movsw'
;; with added functionality that enables 16bit bus transfers if possible.
;;
 
MEM_COPY MACRO
    inc CX
    shr CX, 1
    IF (mem_num_8bit eq 0) and (mem_num_16bit gt 0) ;; use 16bit
        push AX
        push DX
        IF mem_min_latency lt 0
            cli
            MEM_ALL_16BIT_ON
            rep movsw
 
            MEM_ALL_16BIT_OFF
            sti
        ELSE
                ;; really should copy in pieces, but I don't have
                ;; the time to code it.  for now we just give up
                ;; on 16 bit access
            rep movsw
 
        ENDIF
        pop DX
        pop AX
    ELSE
        rep movsw
    ENDIF
ENDM
 

;;****************************************************************************
;; These macros call the stored on_macro and off_macro respectively
 
MEM_ALL_16BIT_ON MACRO
    FOR idx,<1,2,3,4,5>
    IF idx le mem_num_16bit
        MEM_16BIT_ON idx
    ENDIF
    ENDM
ENDM
 
;;****************************************************************************
 
MEM_ALL_16BIT_OFF MACRO
    FOR idx,<1,2,3,4,5>
    IF idx le mem_num_16bit
        MEM_16BIT_OFF idx
    ENDIF
    ENDM
ENDM
 
;;****************************************************************************
MEM_16BIT_ON MACRO num
    HELPER MACRO name, myid
        name myid
    ENDM
 
    HELPER %mem_16bit_&num&_on, %mem_16bit_&num&_id
ENDM
 
;;****************************************************************************
MEM_16BIT_OFF MACRO num
    HELPER MACRO name, myid
        name myid
    ENDM
 
    HELPER %mem_16bit_&num&_off, %mem_16bit_&num&_id
ENDM
;;*********************************************************************
;; utility functions for port I/O
;;	IN:	--
;;	OUT:	AL
;;	CHNG:	AL, DX
PORT_READ MACRO port, if_io	;; read port, changes AL, DX
    mov DX, if_io+port
    in  AL, DX                  ;; AL contains data read from port
ENDM
;;*********************************************************************
 
;;*********************************************************************
;;	IN:	AL
;;	OUT:	--
;;	CHNG:	DX
PORT_WRITE MACRO port, if_io, data ;; write port, changes DX
    IFNB <data>
	mov al,data
    ENDIF
    mov DX, if_io+port
    out DX, AL                  ;; AL contains data read from port
ENDM
;;*********************************************************************
;; -------------------- end of file ------------------------ 
