/*
 * PCbridge managing daemon.
 * taken from bootpd sources
 * Options are:
 * -c command	send a command
 *		0 = dump data segment (256 bytes)
 *		1 = dump filtering table (256 bytes)
 *		2 = clush filtering table
 * -o offset	send offset (for cmd 0 and 1)
 * -e ethaddr	send the offset corresponding to ethaddr (for cmd 1)
 * -t		get traffic data
 * -l filename	log data to file
 * -s		use as standalone, otherwise use with inetd
 *		-s is useless at the moment
 * -d		increase debugging verbosity
 */
#define MYNET	0xffffffff /* insert here you net. broadcast address */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <net/if.h>
#ifdef SUNOS40
#   include <sys/sockio.h>
#   include <net/if_arp.h>
#endif
#include <netinet/in.h>
#include <signal.h>
#include <stdio.h>
#include <strings.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>
#if 1 /* #ifdef SYSLOG */
#   include <syslog.h>
#endif

/*
 * Externals, forward declarations, and global variables
 */
#define byte char
#define TRUE 1
#define FALSE 0
#define MAXHADDRLEN 6
#define MAXPORTS	5
#define NUMCOUNTERS	8
struct bdg_if_addr {
    unsigned char a[8];
};

struct counter {
    unsigned char ct[8];
};


struct filters {
    char mask[6];
};

/*** all data here are in little-endian format ***/
struct pcbridge_data {
    char magic[16];
    char if_found[8];
    struct bdg_if_addr if_addr[8];
    struct counter stat[3+MAXPORTS*NUMCOUNTERS];
    short video_seg;
    char loop_counters[8];
    unsigned char filt[16*6+2];
    short table_data_off;
    struct bdg_if_addr table[64];
};
#define MIN_SIZE (sizeof(struct pcbridge_data)-8-16*6-2-2-512)

struct bridges {
	unsigned char hwaddr[6];
	char *bdg_name;
	char *port_name;
} known_bridges[1024];
int n_known_bridges=0;

void usage();
void report();
void readtable();
void cleanup();

/*
 * IP port numbers for client and server obtained from /etc/services
 */
#define PCBRIDGE_SPORT	2299
#define PCBRIDGE_CPORT	2298
u_short pcbridge_port;
struct servent *servp;

/*
 * Internet socket and interface config structures
 */

unsigned thecommand=0xff;
unsigned short theoffset=0;

struct sockaddr_in sin;
struct sockaddr_in cmdsock;
struct sockaddr_in from;	/* Packet source */
struct ifreq ifreq[10];		/* Holds interface configuration */
struct ifconf ifconf;		/* Int. config ioctl block (pnts to ifreq) */
/*
 * General
 */

char *logfilename=NULL;
FILE *logfile;

int traffic=FALSE;
int standalone = TRUE; /* set to FALSE to use with inetd */;
int debug = 3;			    /* Debugging flag */
int s;				    /* Socket file descriptor */
byte buf[2048];			    /* Receive packet buffer */
/* struct timeval tp;		    /* Time (extra baggage) */

unsigned long thecounter(struct counter i)
{
	return ( (  ( ( (i.ct[5]<<8)+i.ct[4])<<8) + i.ct[3])<<8) + i.ct[2];
}

char *stat_string[]= {
    "Rx packets",
    "Fwd all   ",
    "Tx packets",
    "Dropped   ",
    "Local drop",
    "Rx bridge ",
    "Rx bytes  ",
    "Bad rx pkt",
    NULL};

int addrcmp(char *a, unsigned char  *ha)
{
    int i;
    char *b= (char *)ha;
    for (i=0;i<6;i++) if (*a++ != *b++) return 0 ;
    return 1;
}

int dump_packet(struct pcbridge_data *p, int n)
{
    int i,j,stat;
    char string[80];
    struct bdg_if_addr h;

    char *name=NULL, *port=NULL;
    char *port_names[MAXPORTS];
    unsigned char *q;

    if (traffic && logfile) {
	for (i=1;i<=MAXPORTS;i++) {
	    if (p->if_found[i]) {
		h= (p->if_addr[i]);
		fprintf(logfile,
	    "TIME %6lx PORT %d-%02x:%02x:%02x:%02x:%02x:%02x RXBYTES %12u\n",
		    thecounter(p->stat[2]),i,
		    h.a[0],h.a[1],h.a[2],h.a[3],h.a[4],h.a[5],
		    thecounter(p->stat[2+5*6+i]) );
		fflush(logfile);
	    }
	}
    }
    if (traffic) return;
    /* printf("Packet size %d\n",n); */
    j=p->if_found[0];
    h= (p->if_addr[j]);
    for (i=0;i<n_known_bridges; i++)
	if (addrcmp(h.a,known_bridges[i].hwaddr)) {
		name=known_bridges[i].bdg_name;
		port=known_bridges[i].port_name;
	}
    printf("\"%s\" %s, Port %s (%d, %02x:%02x:%02x:%02x:%02x:%02x)\n",
	p->magic,name, port, j,h.a[0],h.a[1],h.a[2],h.a[3],h.a[4],h.a[5]);
    printf("  Loops 0x%lx  -- Hash Table Collisions %ld -- Time %06lx\n",
	thecounter(p->stat[0]), thecounter(p->stat[1]),thecounter(p->stat[2]));
    for (i=1;i<=MAXPORTS;i++) {
	port_names[i-1]="???";
	if (p->if_found[i]) {
	    h= (p->if_addr[i]);
	    for (j=0;j<n_known_bridges; j++)
		if (addrcmp(h.a,known_bridges[j].hwaddr)) {
		    port_names[i-1]=known_bridges[j].port_name;
		}
	    stat=h.a[7];
	    string[0]='\0';
	    if (stat & 1) strcat(string,"DISABLED ");
	    stat=h.a[6];
	    if (stat == 1) strcat(string,"RX_OK");
	    else {
	       if (stat & 1) strcat(string,"TX_OK ");
	       if (stat & 4) strcat(string,"TX_colls ");
	       if (stat & 8) strcat(string,"TX_ABORT ");
	       if (stat & 16) strcat(string,"CRS_LOST ");
	       if (stat & 32) strcat(string,"FIFO_UNDERRUN ");
	       if (stat & 64) strcat(string,"CD_HEARTBEAT ");
	       if (stat & 128) strcat(string,"OUT_WIN_COLLS");
	    }
	    printf("  %cPort %14s (%d, %02x:%02x:%02x:%02x:%02x:%02x) : %s\n",
	    i==p->if_found[0] ? '*':' ',
	    port_names[i-1], i, h.a[0],h.a[1],h.a[2],h.a[3],h.a[4],h.a[5], string);
	}
    }
    for (q= &(p->filt[0]); *q; q += 6) {
	printf("[%04x..%04x[: %s outside, %s inside\n",
	    q[2]+q[3]*256, q[4]+q[5]*256,
	    (q[0]==1 ? "keep" : (q[0]==2 ? "discard" : "--")),
	    (q[1]==1 ? "keep" : (q[1]==2 ? "discard" : "--"))
	);
    }
    printf("\n%10s","");
    for (j=0;j<5; j++) {
	printf("%14s",p->if_found[j+1]?port_names[j]:"");
    }
    printf("\n%10s","");
    for (j=0;j<5; j++) {
	printf(p->if_found[j+1]?"  == Port %d ==":"",j+1);
    }
    printf("\n");
    for (i=0;i<8;i++) {
	printf(stat_string[i]);
	for (j=0;j<5; j++) {
	    if (p->if_found[j+1]) printf("%14u",thecounter(p->stat[3+5*i+j]));
	    else printf("%14s","");
	}
	printf("\n");
    }
    {
	unsigned short off=p->table_data_off;
	printf("Table offset %4x\n",off);
	for (i=0; i<64;i++) {
	    struct bdg_if_addr bdg;
	    bdg= p->if_addr[p->if_found[0]];
	    h= (p->table[i]);
	    if ( h.a[6] != 255 ) {
		printf("[Entry %4d] %02x:%02x:%02x:%02x:%02x:%02x port %d - %d\n",
		    i, h.a[0],h.a[1],h.a[2],h.a[3],h.a[4],h.a[5], h.a[6],h.a[7]);
		if (logfile) {
		    fprintf(logfile,
"IF %02x:%02x:%02x:%02x:%02x:%02x %4x PORT %d %2x SRC %02x:%02x:%02x:%02x:%02x:%02x TIME %6lx\n",
		    h.a[0],h.a[1],h.a[2],h.a[3],h.a[4],h.a[5],
		    off + i*8 - 0x200 /* patch for bad EPROM 2.64 */,
		    h.a[6],h.a[7],
		    bdg.a[0],bdg.a[1],bdg.a[2],bdg.a[3],bdg.a[4],bdg.a[5],
		    thecounter(p->stat[2])
		    );
		    fflush(logfile);
		}
	    }
	}
    }
    printf("\n");
}



/*
 * Initialization such as command-line processing is done and then the main
 * server loop is started.
 */


main(argc, argv)
    int argc;
    char **argv;
{
    struct timeval actualtimeout, *timeout;
    int n, tolen, fromlen;
    int nfound, readfds;

    timeout = &actualtimeout;

    /*
     * Read switches.
     */
    for (argc--, argv++; argc > 0; argc--, argv++) {
	if (argv[0][0] == '-') {
	    switch (argv[0][1]) {
		case 't': /* traffic */
		    traffic=TRUE;
		    break;
		case 'c': /* command */
		    argc--; argv++;
		    thecommand=atoi(argv[0]);
		    if (thecommand>2) thecommand=0;
		    break;
		case 'o': /* offset */
		    argc--; argv++;
		    sscanf(argv[0],"%i",&theoffset);
		    break;
		case 'e': /* offset */
		    {int b2,b3,b4,b5;
		    argc--; argv++;
		    sscanf(argv[0],"%*x:%*x:%x:%x:%x:%x",
			&b2, &b3, &b4, &b5);
		    theoffset= (((b2 ^ b4) + ((b3 ^ b5)<<8) )<<3) & 0xffff;
		    }
		    break;
		case 'l': /* logging */
		    argc--; argv++;
		    logfilename=argv[0];
		    logfile=fopen(logfilename,"w+");
		    if (logfile==NULL) {
			fprintf(stderr,"Can't open logfile %s\n",logfilename);
		    }
		    break;
		case 'd':
		    debug++;
		    break;
		case 's':
		    standalone = TRUE;
		    break;
		default:
		    fprintf(stderr,
			    "pcbridged: unknown switch: -%c\n",
			    argv[0][1]);
		    usage();
		    break;
	    }
	} else {
	    usage();
	}
    }
    if (standalone) {
	/*
	 * Go into background and disassociate from controlling terminal.
	 */
	if (debug < 3) {
	    if (fork())
		exit(0);
	    for (n = 0; n < 10; n++)
		(void) close(n);
	    (void) open("/", O_RDONLY);
	    (void) dup2(0, 1);
	    (void) dup2(0, 2);
	    n = open("/dev/tty", O_RDWR);
	    if (n >= 0) {
		ioctl(n, TIOCNOTTY, (char *) 0);
		(void) close(n);
	    }
	}
	/*
	 * Nuke any timeout value
	 */
	timeout = NULL;
    }


#ifdef SYSLOG
    /*
     * Initialize logging.
     */
#ifndef LOG_CONS
#define LOG_CONS	0	/* Don't bother if not defined... */
#endif
#ifndef LOG_DAEMON
#define LOG_DAEMON	0
#endif
    openlog("pcbridged", LOG_PID | LOG_CONS, LOG_DAEMON);
    syslog(LOG_INFO, "%s", Version);
#endif

    readtable();

    if (standalone) {
	/*
	 * Get us a socket.
	 */
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	    report(LOG_ERR, "socket: %s\n", "can't get socket" );
	    exit(1);
	}

	/*
	 * Get server's listening port number
	 */
	servp = getservbyname("pcbridge", "udp");
	if (servp) {
	    pcbridge_port = ntohs((u_short) servp->s_port);
	} else {
	    report(LOG_ERR,
		   "udp/pcbridge: unknown service -- assuming port %d\n",
		   PCBRIDGE_SPORT);
	    pcbridge_port = (u_short) PCBRIDGE_SPORT;
	}

	/*
	 * Bind socket to pcbridge port.
	 */
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(pcbridge_port);
	if (bind(s, &sin, sizeof(sin)) < 0) {
	    report(LOG_ERR, "bind: %s\n", "can't bind to pcbridge_port");
	    exit(1);
	}
    } else {
	/*
	 * Assume socket was passed to us from inetd
	 */
	s = 0;
	tolen = sizeof(sin);
	bzero((char *) &sin, tolen);
	if (getsockname(s, &sin, &tolen) < 0) {
	    report(LOG_ERR, "getsockname: %s\n","failed" );
	    exit(1);
	}
	pcbridge_port = ntohs(sin.sin_port);
    }
    { int c=1;
	setsockopt(s, SOL_SOCKET, SO_BROADCAST, &c, sizeof(c) );
    }

    /*
     * Determine network configuration.
     */
    ifconf.ifc_len = sizeof(ifreq);
    ifconf.ifc_req = ifreq;
    if ((ioctl(s, SIOCGIFCONF, (caddr_t) &ifconf) < 0) ||
	(ifconf.ifc_len <= 0)) {
	    report(LOG_ERR, "ioctl: %s\n", "failed");
	    exit(1);
    }

    /*
     * Set up signals to clean up on exit. -mah
     */
    signal(SIGSEGV, cleanup);
    signal(SIGBUS , cleanup);
    signal(SIGINT,  cleanup);
    signal(SIGILL,  cleanup);
    signal(SIGQUIT, cleanup);
    signal(SIGFPE,  cleanup);
    signal(SIGIOT,  cleanup);

    /*
     * Set up signals to read or dump the table.
     */
    if ((int) signal(SIGHUP, cleanup) < 0) {
	report(LOG_ERR, "signal: %s\n", "cleanup...");
	exit(1);
    }

    /*** if necessary, send out command ***/
    if (thecommand != 0xff) {
	sendcmd(thecommand,theoffset);
	thecommand = 0xff;
    }

    /*
     * Process incoming requests.
     */
    for (;;) {
	struct pcbridge_data *pcd=(struct pcbridge_data *)buf;
	readfds = 1 << s;
	nfound = select(s + 1, &readfds, NULL, NULL, timeout);
	if (nfound < 0) {
	    if (errno != EINTR) {
		report(LOG_ERR, "select: %s\n", "not eintr");
	    }
	    continue;
	}
	if (!(readfds & (1 << s))) {
	    report(LOG_INFO, "exiting after %ld minutes of inactivity\n",
		   actualtimeout.tv_sec / 60);
	    exit(0);
	}
	fromlen = sizeof(from);
	n = recvfrom(s, buf, sizeof(buf), 0, &from, &fromlen);
	if (n <= 0) {
	    continue;
	}

	if (n < MIN_SIZE) {
	    if (debug) {
		report(LOG_INFO, "received short packet\n");
	    }
	    continue;
	}
	dump_packet(pcd,n);
	/* process data */
    }
}

char screen[80*24];

/*
 * Print "usage" message and exit
 */

void usage()
{
    fprintf(stderr,
	   "usage:  pcbridged [-d] [-s]\n");
    fprintf(stderr, "\t -d\tincrease debug verbosity\n");
    fprintf(stderr, "\t -s\trun standalone (without inetd)\n");
    exit(1);
}



/*
 * Process BOOTREQUEST packet.
 *
 * (Note, this version of the bootpd.c server never forwards 
 * the request to another server.  In our environment the 
 * stand-alone gateways perform that function.)
 *
 * (Also this version does not interpret the hostname field of
 * the request packet;  it COULD do a name->address lookup and
 * forward the request there.)
 */

/*
 * This call checks read access to a file.  It returns 0 if the file given
 * by "path" exists and is publically readable.  A value of -1 is returned if
 * access is not permitted or an error occurs.  Successful calls also
 * return the file size in bytes using the long pointer "filesize".
 *
 * The read permission bit for "other" users is checked.  This bit must be
 * set for tftpd(8) to allow clients to read the file.
 */


/*
 * Send a reply packet to the client.  'forward' flag is set if we are
 * not the originator of this reply packet.
 */
struct bdgcmd {
    unsigned char magic[4]; /* 'L' 'R' .... */
    unsigned short cmd[2];
} thecmd;

sendcmd(unsigned char cmd, unsigned short off)
{
	struct bdgcmd *bp = &thecmd;
	struct in_addr dst;
	struct sockaddr_in to;

	thecmd.magic[0]='L'; /* magic string */
	thecmd.magic[1]='R';
	thecmd.cmd[0]=htons(cmd);
	thecmd.cmd[1]=htons(off);

	to = sin;

	to.sin_port = htons(PCBRIDGE_CPORT);
	/*
	 * If the client IP address is specified, use that
	 * else if gateway IP address is specified, use that
	 * else make a temporary arp cache entry for the client's NEW 
	 * IP/hardware address and use that.
	 */
	dst.s_addr=htonl(MYNET);

	to.sin_addr = dst; 
	if (sendto(s, bp, sizeof(struct bdgcmd), 0, &to, sizeof(to)) < 0) {
	    fprintf(stderr, "sendto: %s\n", "failed <0");
	} else {
	    fprintf(stderr, "sendto: succeeds\n");
	}
}


/*
 * Compare function to determine whether two hardware addresses are
 * equivalent.  Returns TRUE if "host1" and "host2" are equivalent, FALSE
 * otherwise.
 *
 * This function is used when retrieving elements from the hardware address
 * hash table.
 */

/*
 * Convert a hardware address to an ASCII string.
 */

char *haddrtoa(haddr, htype)
    register byte *haddr;
    byte htype;
{
    static char haddrbuf[2 * MAXHADDRLEN + 1];
    register char *bufptr;
    register unsigned count;

    bufptr = haddrbuf;
    for (count = MAXHADDRLEN; count > 0; count--) {
	sprintf(bufptr, "%02X",	(unsigned) (*haddr++ & 0xFF));
	bufptr += 2;
    }
    return (haddrbuf);
}




/*
 * This routine reports errors and such via stderr and syslog() if
 * appopriate.  It just helps avoid a lot of "#ifdef SYSLOG" constructs
 * from being scattered throughout the code.
 *
 * The syntax is identical to syslog(3), but %m is not considered special
 * for output to stderr (i.e. you'll see "%m" in the output. . .).  Also,
 * control strings should normally end with \n since newlines aren't
 * automatically generated for stderr output (whereas syslog strips out all
 * newlines and adds its own at the end).
 */

/*VARARGS2*/
void report(priority, fmt, p0, p1, p2, p3, p4)
    int priority;
    char *fmt;
{
    /*
     * Print the message
     */
    if (debug > 2) {
	fprintf(stderr, "*** prbridge: ");
	fprintf(stderr, fmt, p0, p1, p2, p3, p4);
	fprintf(stderr, "\n\n");
    }
#ifdef SYSLOG
    syslog(priority, fmt, p0, p1, p2, p3, p4);
#endif
}

/*
 * shutdown the socket on weird signals
 * otherwise it wont go away on pmaxes
 * and we get an 'address already in use' on next try
 * -mah
 */
void
cleanup(sig)
int sig;
{
	shutdown(s,2);
	report(LOG_ERR, "cleanup: got signal %d\n", sig);
	exit(sig);
}

int ishex(char c)
{
    if (c>='0' && c<='9') return (c-'0');
    else if (c>='a' && c<='f') return (c-'a'+10);
    else if (c>='A' && c<='F') return (c-'A'+10);
    else return(-1);
}

void readtable()
{
    char linebuf[256];
    char *p;
    FILE *f;
    int i;
    unsigned char ha[6];
    char *bdg_name, *bdg_port;
    f=fopen("pcbridgetab","r");
    if (f==NULL) f=fopen("/etc/pcbridgetab","r");
    if (f==NULL) return;
    while (p=fgets(linebuf,sizeof(linebuf),f)) {
	while (*p && *p!='#' && *p!='\n') p++;
	*p='\0'; /* skip comments */
	for (p=linebuf;*p && (*p==' ' || *p=='\t'); p++); /* skip blanks */
	if (*p) {
	    printf("Found %s\n",p);
	    for (i=0;i<6;i++) {
		int c1,c2;
		c1=ishex(*p++);
		if (c1<0) {
		    fprintf(stderr, "Invalid first digit (%d) %d\n",i,c1);
		    break;
		}
		c2=ishex(*p);
		if (c2== -1 ) {
		    if (*p!=':' && *p!=' ' && *p!='\t') {
			fprintf(stderr, "Invalid second digit\n");
			break;
		    } else {
			c2=c1; c1=0;
		    }
		} else p++;
		known_bridges[n_known_bridges].hwaddr[i]=c1*16+c2;
		if (i<5) while (*p && ishex(*p)<0) p++;
	    }
	    if (i==6) { /* successful end */
		char *p1;
		char *p2;
		while (*p && (*p ==' ' || *p=='\t')) p++;
		p1=p;
		while (*p && *p !=' ' && *p!='\t') p++;
		if (*p) {
		    *p++='\0';
		    while (*p && (*p ==' ' || *p=='\t')) p++;
		    p2= p;
		    while (*p && *p !=' ' && *p!='\t') p++;
		    *p='\0';
		} else p2=NULL;

		known_bridges[n_known_bridges].bdg_name=
			(char *)malloc(1+strlen(p1));
		strcpy(known_bridges[n_known_bridges].bdg_name,p1);
		if (p2) {
		    known_bridges[n_known_bridges].port_name=
			(char *)malloc(1+strlen(p2));
		    strcpy(known_bridges[n_known_bridges].port_name,p2);
		} else known_bridges[n_known_bridges].port_name=NULL;
		fprintf(stderr,"    bdg: %s port: %s\n",
			known_bridges[n_known_bridges].bdg_name,
			known_bridges[n_known_bridges].port_name);
		n_known_bridges++;
	    }
	}
    }
    fclose(f);
}
