/*-
 * Copyright (c) 1994, 1995 Matt Thomas (matt@lkg.dec.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: if_de.c,v 1.29.2.5 1995/10/26 07:47:14 davidg Exp $
 *
 */

/*
 * DEC DC21040 PCI Ethernet Controller
 *
 * Written by Matt Thomas
 * BPF support code stolen directly from if_ec.c
 *
 *   This driver supports the DEC DE435 or any other PCI
 *   board which support DC21140 (mostly).
 */

#include "bridge.h"
#include <machine/clock.h>

#include <net/if_types.h>
#include <net/if_dl.h>

#ifdef INET
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

/* #include <pci/pcivar.h> */
#include "dc21040.h"

/*
 * Intel CPUs should use I/O mapped access.
 *    (NetBSD doesn't support it yet)
 */
#define	TULIP_IOMAPPED

/*
 * This module supports
 *	the DEC DC21140 PCI Fast Ethernet Controller.
 */

typedef struct {
    tulip_desc_t *ri_first;
    tulip_desc_t *ri_last;
    tulip_desc_t *ri_nextin;
    tulip_desc_t *ri_nextout;
    int ri_max;
    int ri_free;
} tulip_ringinfo_t;

#ifdef TULIP_IOMAPPED
typedef tulip_uint16_t tulip_csrptr_t;

#define	TULIP_PCI_CSRSIZE	8
#define	TULIP_PCI_CSROFFSET	0

#define	TULIP_READ_CSR(sc, csr)			(inl((sc)->tulip_csrs.csr))
#define	TULIP_WRITE_CSR(sc, csr, val)   	outl((sc)->tulip_csrs.csr, val)

#define	TULIP_READ_CSRBYTE(sc, csr)		(inb((sc)->tulip_csrs.csr))
#define	TULIP_WRITE_CSRBYTE(sc, csr, val)	outb((sc)->tulip_csrs.csr, val)

#else /* TULIP_IOMAPPED */

typedef volatile tulip_uint32_t *tulip_csrptr_t;

/*
 * macros to read and write CSRs.  Note that the "0 +" in
 * READ_CSR is to prevent the macro from being an lvalue
 * and WRITE_CSR shouldn't be assigned from.
 */
#define	TULIP_READ_CSR(sc, csr)		(0 + *(sc)->tulip_csrs.csr)
#ifndef __alpha__
#define	TULIP_WRITE_CSR(sc, csr, val) \
	    ((void)(*(sc)->tulip_csrs.csr = (val)))
#else
#define	TULIP_WRITE_CSR(sc, csr, val) \
	    ((void)(*(sc)->tulip_csrs.csr = (val), MB()))
#endif

#endif /* TULIP_IOMAPPED */

typedef struct {
    tulip_csrptr_t csr_busmode;			/* CSR0 */
    tulip_csrptr_t csr_txpoll;			/* CSR1 */
    tulip_csrptr_t csr_rxpoll;			/* CSR2 */
    tulip_csrptr_t csr_rxlist;			/* CSR3 */
    tulip_csrptr_t csr_txlist;			/* CSR4 */
    tulip_csrptr_t csr_status;			/* CSR5 */
    tulip_csrptr_t csr_command;			/* CSR6 */
    tulip_csrptr_t csr_intr;			/* CSR7 */
    tulip_csrptr_t csr_missed_frame;		/* CSR8 */

    /* DC21140/DC21041 common registers */

    tulip_csrptr_t csr_srom_mii;		/* CSR9 */
    tulip_csrptr_t csr_gp_timer;		/* CSR11 */

    /* DC21140 specific registers */

    tulip_csrptr_t csr_gp;			/* CSR12 */
    tulip_csrptr_t csr_watchdog;		/* CSR15 */

} tulip_regfile_t;

/*
 * The DC21040 has a stupid restriction in that the receive
 * buffers must be longword aligned.  But since Ethernet
 * headers are not a multiple of longwords in size this forces
 * the data to non-longword aligned.  Since IP requires the
 * data to be longword aligned, we need to copy it after it has
 * been DMA'ed in our memory.
 *
 * Since we have to copy it anyways, we might as well as allocate
 * dedicated receive space for the input.  This allows to use a
 * small receive buffer size and more ring entries to be able to
 * better keep with a flood of tiny Ethernet packets.
 *
 * The receive space MUST ALWAYS be a multiple of the page size.
 * And the number of receive descriptors multiplied by the size
 * of the receive buffers must equal the recevive space.  This
 * is so that we can manipulate the page tables so that even if a
 * packet wraps around the end of the receive space, we can 
 * treat it as virtually contiguous.
 *
 * The above used to be true (the stupid restriction is still true)
 * but we gone to directly DMA'ing into MBUFs because with 100Mb
 * cards the copying is just too much of a hit.
 */

#define	TULIP_RXDESCS		16
#define	TULIP_TXDESCS		128
#define	TULIP_RXQ_TARGET	8

typedef enum {
    TULIP_DC21140_DEC_EB,
    TULIP_DC21140_DEC_DE500,
    TULIP_DC21140_COGENT_EM100,
    TULIP_DC21140_ZNYX_ZX34X,
} tulip_board_t;

typedef struct _tulip_softc_t tulip_softc_t;

typedef struct {
    tulip_board_t bd_type;
    const char *bd_description;
    int (*bd_media_probe)(tulip_softc_t *sc);
    void (*bd_media_select)(tulip_softc_t *sc);
} tulip_boardsw_t;

typedef enum {
    TULIP_DC21140,
    TULIP_DE425,
    TULIP_CHIPID_UNKNOWN
} tulip_chipid_t;

typedef enum {
    TULIP_PROBE_INACTIVE, TULIP_PROBE_10BASET, TULIP_PROBE_AUI,
    TULIP_PROBE_BNC
} tulip_probe_state_t;

typedef enum {
    TULIP_MEDIA_UNKNOWN, TULIP_MEDIA_10BASET,
    TULIP_MEDIA_BNC, TULIP_MEDIA_AUI,
    TULIP_MEDIA_BNCAUI, TULIP_MEDIA_100BASET
} tulip_media_t;

struct _tulip_softc_t {
    int if_name;
    tulip_regfile_t tulip_csrs;
    unsigned tulip_flags;
    unsigned want_tulip_flags;
#define	TULIP_WANTSETUP		0x0001
#define	TULIP_WANTHASH		0x0002
#define	TULIP_DOINGSETUP	0x0004
#define	TULIP_ALTPHYS		0x0008	/* use AUI */
#define	TULIP_TXPROBE_ACTIVE	0x0010
#define	TULIP_TXPROBE_OK	0x0020
#define	TULIP_INRESET		0x0040
#define	TULIP_WANTRXACT		0x0080
#define	TULIP_SLAVEDROM		0x0100
#define	TULIP_ROMOK		0x0200
    unsigned char tulip_rombuf[128];
    char tulip_hwaddr[6];
    tulip_uint32_t tulip_setupbuf[192/sizeof(tulip_uint32_t)];
    tulip_uint32_t tulip_setupdata[192/sizeof(tulip_uint32_t)];
    tulip_uint32_t tulip_intrmask;
    tulip_uint32_t tulip_cmdmode;
    tulip_uint32_t tulip_revinfo;
    tulip_uint32_t tulip_gpticks;
    /* tulip_uint32_t tulip_bus; XXX */
    tulip_media_t tulip_media;
    tulip_probe_state_t tulip_probe_state;
    tulip_chipid_t tulip_chipid;
    const tulip_boardsw_t *tulip_boardsw;
    tulip_softc_t *tulip_slaves;
    struct ifqueue tulip_txq;
    struct ifqueue tulip_rxq;
    tulip_ringinfo_t tulip_rxinfo;
    tulip_ringinfo_t tulip_txinfo;

    int tulip_unit;
    int tulip_name;
    int tulip_if;
    int tulip_ac;
};

#ifndef IFF_ALTPHYS
#define	IFF_ALTPHYS	1		/* In case it isn't defined */
#endif
static const char *tulip_chipdescs[] = { 
    "DC21140 [10-100Mb/s]",
};

#define	NDE	5
typedef void ifnet_ret_t;
typedef int ioctl_cmd_t;
tulip_softc_t tulips[NDE];
#define	TULIP_UNIT_TO_SOFTC(unit)	(&(tulips[unit]))

#ifndef TULIP_BURSTSIZE
#define	TULIP_BURSTSIZE(unit)		3
#endif

#define	TULIP_CRC32_POLY	0xEDB88320UL	/* CRC-32 Poly -- Little Endian */
#define	TULIP_CHECK_RXCRC	0
#define	TULIP_MAX_TXSEG		30

#define	TULIP_ADDREQUAL(a1, a2) \
	(((u_short *)a1)[0] == ((u_short *)a2)[0] \
	 && ((u_short *)a1)[1] == ((u_short *)a2)[1] \
	 && ((u_short *)a1)[2] == ((u_short *)a2)[2])
#define	TULIP_ADDRBRDCST(a1) \
	(((u_short *)a1)[0] == 0xFFFFU \
	 && ((u_short *)a1)[1] == 0xFFFFU \
	 && ((u_short *)a1)[2] == 0xFFFFU)

static ifnet_ret_t tulip_start(struct if_vars *ifp);
static void tulip_rx_intr(tulip_softc_t *sc);
static void tulip_addr_filter(tulip_softc_t *sc);

static void
pr_do_10_100(tulip_softc_t * const sc, int speed)
{
    if ((sc->tulip_flags & TULIP_ALTPHYS) == 0)
	printf("de%d: enabling %s port\n", sc->if_name,
		speed==10 ? "10baseT UTP":"100baseTX UTP");
}

static int
tulip_dc21140_evalboard_media_probe( tulip_softc_t * const sc)
{
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EB_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EB_INIT);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) | TULIP_CMD_PORTSELECT |
	TULIP_CMD_PCSFUNCTION | TULIP_CMD_SCRAMBLER | TULIP_CMD_MUSTBEONE);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) & ~TULIP_CMD_TXTHRSHLDCTL);
    DELAY(1000000);
    return (TULIP_READ_CSR(sc, csr_gp) & TULIP_GP_EB_OK100) != 0;
}

static void
tulip_dc21140_evalboard_media_select( tulip_softc_t * const sc)
{
    sc->tulip_cmdmode |= TULIP_CMD_STOREFWD|TULIP_CMD_MUSTBEONE;
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EB_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EB_INIT);
    if (sc->want_tulip_flags & IFF_ALTPHYS) {
	pr_do_10_100(sc, 100);
	sc->tulip_cmdmode |= TULIP_CMD_PORTSELECT
	    |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER;
	sc->tulip_cmdmode &= ~TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags |= TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_100BASET;
    } else {
	pr_do_10_100(sc, 10);
	sc->tulip_cmdmode &= ~(TULIP_CMD_PORTSELECT
			       |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER);
	sc->tulip_cmdmode |= TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags &= ~TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_10BASET;
    }
}

static int
tulip_dc21140_cogent_em100_media_probe( tulip_softc_t * const sc)
{
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EM100_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EM100_INIT);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) | TULIP_CMD_PORTSELECT |
	TULIP_CMD_PCSFUNCTION | TULIP_CMD_SCRAMBLER | TULIP_CMD_MUSTBEONE);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) & ~TULIP_CMD_TXTHRSHLDCTL);
    return 1;
}

static void
tulip_dc21140_cogent_em100_media_select( tulip_softc_t * const sc)
{
    sc->tulip_cmdmode |= TULIP_CMD_STOREFWD|TULIP_CMD_MUSTBEONE;
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EM100_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_EM100_INIT);
    pr_do_10_100(sc, 100);
    sc->tulip_cmdmode |= TULIP_CMD_PORTSELECT
	|TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER;
    sc->tulip_cmdmode &= ~TULIP_CMD_TXTHRSHLDCTL;
    sc->tulip_flags |= TULIP_ALTPHYS;
    sc->tulip_media = TULIP_MEDIA_100BASET;
}

static int
tulip_dc21140_znyx_zx34x_media_probe( tulip_softc_t * const sc)
{
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_ZX34X_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_ZX34X_INIT);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) | TULIP_CMD_PORTSELECT |
	TULIP_CMD_PCSFUNCTION | TULIP_CMD_SCRAMBLER | TULIP_CMD_MUSTBEONE);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) & ~TULIP_CMD_TXTHRSHLDCTL);
    DELAY(1000000);

    return (TULIP_READ_CSR(sc, csr_gp) & TULIP_GP_ZX34X_OK10);
}

static void
tulip_dc21140_znyx_zx34x_media_select( tulip_softc_t * const sc)
{
    sc->tulip_cmdmode |= TULIP_CMD_STOREFWD|TULIP_CMD_MUSTBEONE;
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_ZX34X_PINS);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_ZX34X_INIT);
    if (sc->want_tulip_flags & IFF_ALTPHYS) {
	pr_do_10_100(sc, 100);
	sc->tulip_cmdmode |= TULIP_CMD_PORTSELECT
	    |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER;
	sc->tulip_cmdmode &= ~TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags |= TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_100BASET;
    } else {
	pr_do_10_100(sc, 10);
	sc->tulip_cmdmode &= ~(TULIP_CMD_PORTSELECT
			       |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER);
	sc->tulip_cmdmode |= TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags &= ~TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_10BASET;
    }
}

static int
tulip_dc21140_de500_media_probe( tulip_softc_t * const sc)
{
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_DE500_PINS);
    DELAY(1000);
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_DE500_HALFDUPLEX);
    if ((TULIP_READ_CSR(sc, csr_gp) &
	(TULIP_GP_DE500_NOTOK_100|TULIP_GP_DE500_NOTOK_10)) !=
	(TULIP_GP_DE500_NOTOK_100|TULIP_GP_DE500_NOTOK_10))
	return (TULIP_READ_CSR(sc, csr_gp) & TULIP_GP_DE500_NOTOK_100) != 0;
    TULIP_WRITE_CSR(sc, csr_gp,
	TULIP_GP_DE500_HALFDUPLEX|TULIP_GP_DE500_FORCE_100);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) | TULIP_CMD_PORTSELECT |
	TULIP_CMD_PCSFUNCTION | TULIP_CMD_SCRAMBLER | TULIP_CMD_MUSTBEONE);
    TULIP_WRITE_CSR(sc, csr_command,
	TULIP_READ_CSR(sc, csr_command) & ~TULIP_CMD_TXTHRSHLDCTL);
    DELAY(1000000);
    return (TULIP_READ_CSR(sc, csr_gp) & TULIP_GP_DE500_NOTOK_100) != 0;
}

static void
tulip_dc21140_de500_media_select( tulip_softc_t * const sc)
{
    sc->tulip_cmdmode |= TULIP_CMD_STOREFWD|TULIP_CMD_MUSTBEONE;
    TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_DE500_PINS);
    if (sc->want_tulip_flags & IFF_ALTPHYS) {
	pr_do_10_100(sc, 100);
	sc->tulip_cmdmode |= TULIP_CMD_PORTSELECT
	    |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER;
	sc->tulip_cmdmode &= ~TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags |= TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_100BASET;
	TULIP_WRITE_CSR(sc, csr_gp,
	    TULIP_GP_DE500_HALFDUPLEX|TULIP_GP_DE500_FORCE_100);
    } else {
	pr_do_10_100(sc, 10);
	sc->tulip_cmdmode &= ~(TULIP_CMD_PORTSELECT
			       |TULIP_CMD_PCSFUNCTION|TULIP_CMD_SCRAMBLER);
	sc->tulip_cmdmode |= TULIP_CMD_TXTHRSHLDCTL;
	sc->tulip_flags &= ~TULIP_ALTPHYS;
	sc->tulip_media = TULIP_MEDIA_10BASET;
	TULIP_WRITE_CSR(sc, csr_gp, TULIP_GP_DE500_HALFDUPLEX);
    }
}

static void
tulip_reset(tulip_softc_t * const sc)
{
    tulip_ringinfo_t *ri;
    tulip_desc_t *di;

    TULIP_WRITE_CSR(sc, csr_busmode, TULIP_BUSMODE_SWRESET);
    DELAY(10);	/* Wait 10 microsends (actually 50 PCI cycles but at 
		   33MHz that comes to two microseconds but wait a
		   bit longer anyways) */

    sc->tulip_flags |= TULIP_INRESET;
    sc->tulip_intrmask = 0;
    TULIP_WRITE_CSR(sc, csr_intr, sc->tulip_intrmask);

    TULIP_WRITE_CSR(sc, csr_txlist, vtophys(&sc->tulip_txinfo.ri_first[0]));
    TULIP_WRITE_CSR(sc, csr_rxlist, vtophys(&sc->tulip_rxinfo.ri_first[0]));
    TULIP_WRITE_CSR(sc, csr_busmode,
        (1 << (TULIP_BURSTSIZE(sc->tulip_unit) + 8))
	|TULIP_BUSMODE_CACHE_ALIGN8
	|(BYTE_ORDER != LITTLE_ENDIAN ? TULIP_BUSMODE_BIGENDIAN : 0));

    sc->tulip_txq.ifq_maxlen = TULIP_TXDESCS;
    /*
     * Free all the mbufs that were on the transmit ring.
     */
    for (;;) {
	struct mbuf *m;
	IF_DEQUEUE(&sc->tulip_txq, m);
	if (m == NULL)
	    break;
	m_freem(m);
    }

    ri = &sc->tulip_txinfo;
    ri->ri_nextin = ri->ri_nextout = ri->ri_first;
    ri->ri_free = ri->ri_max;
    for (di = ri->ri_first; di < ri->ri_last; di++)
	di->d_status = 0;

    /*
     * We need to collect all the mbufs were on the 
     * receive ring before we reinit it either to put
     * them back on or to know if we have to allocate
     * more.
     */
    ri = &sc->tulip_rxinfo;
    ri->ri_nextin = ri->ri_nextout = ri->ri_first;
    ri->ri_free = ri->ri_max;
    for (di = ri->ri_first; di < ri->ri_last; di++) {
	di->d_status = 0;
	di->d_length1 = 0; di->d_addr1 = 0;
	di->d_length2 = 0; di->d_addr2 = 0;
    }
    for (;;) {
	struct mbuf *m;
	IF_DEQUEUE(&sc->tulip_rxq, m);
	if (m == NULL)
	    break;
	m_freem(m);
    }

    (*sc->tulip_boardsw->bd_media_select)(sc);

    sc->tulip_intrmask |= TULIP_STS_NORMALINTR|TULIP_STS_RXINTR|TULIP_STS_TXINTR
	|TULIP_STS_ABNRMLINTR|TULIP_STS_SYSERROR|TULIP_STS_TXSTOPPED
	    |TULIP_STS_TXBABBLE|TULIP_STS_LINKFAIL|TULIP_STS_RXSTOPPED;
    sc->tulip_flags &= ~(TULIP_DOINGSETUP|TULIP_WANTSETUP|TULIP_INRESET);
    tulip_addr_filter(sc);
}

static void
tulip_init(tulip_softc_t * const sc)
{
    sc->tulip_cmdmode |= TULIP_CMD_PROMISCUOUS;
    sc->tulip_cmdmode |= TULIP_CMD_TXRUN;
    if ((sc->tulip_flags & TULIP_WANTSETUP) == 0) {
	tulip_rx_intr(sc);
	sc->tulip_cmdmode |= TULIP_CMD_RXRUN;
	sc->tulip_intrmask |= TULIP_STS_RXSTOPPED;
    } else {
	sc->tulip_intrmask &= ~TULIP_STS_RXSTOPPED;
	tulip_start(&sc->tulip_if);
    }
    TULIP_WRITE_CSR(sc, csr_intr, sc->tulip_intrmask);
    TULIP_WRITE_CSR(sc, csr_command, sc->tulip_cmdmode);
}

#if TULIP_CHECK_RXCRC
static unsigned
tulip_crc32( u_char *addr, int len)
{
    unsigned int crc = 0xFFFFFFFF;
    static unsigned int crctbl[256];
    int idx;
    static int done;
    /*
     * initialize the multicast address CRC table
     */
    for (idx = 0; !done && idx < 256; idx++) {
	unsigned int tmp = idx;
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	tmp = (tmp >> 1) ^ (tmp & 1 ? TULIP_CRC32_POLY : 0);	/* XOR */
	crctbl[idx] = tmp;
    }
    done = 1;

    while (len-- > 0)
	crc = (crc >> 8) ^ crctbl[*addr++] ^ crctbl[crc & 0xFF];

    return crc;
}
#endif

static void
tulip_rx_intr(tulip_softc_t * const sc)
{
    tulip_ringinfo_t * const ri = &sc->tulip_rxinfo;
    struct if_vars * const ifp = &sc->tulip_if;

    for (;;) {
	struct eth_packet eh;
	tulip_desc_t *eop = ri->ri_nextin;
	int total_len = 0;
	struct mbuf *m = NULL;
	int accept = 0;

	if (sc->tulip_rxq.ifq_len < TULIP_RXQ_TARGET)
	     goto queue_mbuf;

	if (((volatile tulip_desc_t *) eop)->d_status & TULIP_DSTS_OWNER)
	    break;
	
	total_len = ((eop->d_status >> 16) & 0x7FF) - 4;
	IF_DEQUEUE(&sc->tulip_rxq, m);
	if ((eop->d_status & TULIP_DSTS_ERRSUM) == 0) {

#if TULIP_CHECK_RXCRC
	    unsigned crc = tulip_crc32(mtod(m, unsigned char *), total_len);
	    if (~crc != *((unsigned *) &bufaddr[total_len])) {
		printf("%s%d: bad rx crc: %08x [rx] != %08x\n",
		       sc->tulip_name, sc->tulip_unit,
		       *((unsigned *) &bufaddr[total_len]), ~crc);
		goto next;
	    }
#endif
	    eh = *mtod(m, struct ether_header *);
	    accept = 1;
	    total_len -= sizeof(struct ether_header);
	}
      next:
	if (++ri->ri_nextin == ri->ri_last)
	    ri->ri_nextin = ri->ri_first;
      queue_mbuf:
	/*
	 * Either we are priming the TULIP with mbufs (m == NULL)
	 * or we are about to accept an mbuf for the upper layers
	 * so we need to allocate an mbuf to replace it.  If we
	 * can't replace, then count it as an input error and reuse
	 * the mbuf.
	 */
	if (accept || m == NULL) {
	    struct mbuf *m0;
	    MGETHDR(m0, M_DONTWAIT, MT_DATA);
	    if (m0 != NULL) {
#if defined(TULIP_COPY_RXDATA)
		if (!accept || total_len >= MHLEN) {
#endif
		    MCLGET(m0, M_DONTWAIT);
		    if ((m0->m_flags & M_EXT) == 0) {
			m_freem(m0);
			m0 = NULL;
		    }
#if defined(TULIP_COPY_RXDATA)
		}
#endif
	    }
	    if (accept) {
		if (m0 != NULL) {
#if !defined(TULIP_COPY_RXDATA)
		    m->m_data += sizeof(struct ether_header);
		    m->m_len = m->m_pkthdr.len = total_len;
		    m->m_pkthdr.rcvif = ifp;
		    ether_input(ifp, &eh, m);
		    m = m0;
#else
		    bcopy(mtod(m, caddr_t) + sizeof(struct ether_header),
			  mtod(m0, caddr_t), total_len);
		    m0->m_len = m0->m_pkthdr.len = total_len;
		    m0->m_pkthdr.rcvif = ifp;
		    ether_input(ifp, &eh, m0);
#endif
		} else {
		    ifp->if_ierrors++;
		}
	    } else {
		m = m0;
	    }
	}
	if (m == NULL)
	    break;
	/*
	 * Now give the buffer to the TULIP and save in our
	 * receive queue.
	 */
	ri->ri_nextout->d_length1 = MCLBYTES - 4;
	ri->ri_nextout->d_addr1 = vtophys(mtod(m, caddr_t));
	ri->ri_nextout->d_status = TULIP_DSTS_OWNER;
	if (++ri->ri_nextout == ri->ri_last)
	    ri->ri_nextout = ri->ri_first;
	IF_ENQUEUE(&sc->tulip_rxq, m);
    }
}

static int
tulip_tx_intr(tulip_softc_t * const sc)
{
    tulip_ringinfo_t * const ri = &sc->tulip_txinfo;
    struct mbuf *m;
    int xmits = 0;

    while (ri->ri_free < ri->ri_max) {
	if (((volatile tulip_desc_t *) ri->ri_nextin)->d_status & TULIP_DSTS_OWNER)
	    break;

	if (ri->ri_nextin->d_flag & TULIP_DFLAG_TxLASTSEG) {
	    if (ri->ri_nextin->d_flag & TULIP_DFLAG_TxSETUPPKT) {
		/*
		 * We've just finished processing a setup packet.
		 * Mark that we can finished it.  If there's not
		 * another pending, startup the TULIP receiver.
		 * Make sure we ack the RXSTOPPED so we won't get
		 * an abormal interrupt indication.
		 */
		sc->tulip_flags &= ~TULIP_DOINGSETUP;
		if ((sc->tulip_flags & TULIP_WANTSETUP) == 0) {
		    tulip_rx_intr(sc);
		    sc->tulip_cmdmode |= TULIP_CMD_RXRUN;
		    sc->tulip_intrmask |= TULIP_STS_RXSTOPPED;
		    TULIP_WRITE_CSR(sc, csr_status, TULIP_STS_RXSTOPPED);
		    TULIP_WRITE_CSR(sc, csr_command, sc->tulip_cmdmode);
		    TULIP_WRITE_CSR(sc, csr_intr, sc->tulip_intrmask);
		}
	   } else {
		IF_DEQUEUE(&sc->tulip_txq, m);
		m_freem(m);
		xmits++;
		if (sc->tulip_flags & TULIP_TXPROBE_ACTIVE) {
		    if ((ri->ri_nextin->d_status & (TULIP_DSTS_TxNOCARR|TULIP_DSTS_TxEXCCOLL)) == 0)
			sc->tulip_flags |= TULIP_TXPROBE_OK;
		    (*sc->tulip_boardsw->bd_media_select)(sc);
		} else {
		    sc->tulip_if.if_collisions +=
			(ri->ri_nextin->d_status & TULIP_DSTS_TxCOLLMASK)
			    >> TULIP_DSTS_V_TxCOLLCNT;
		    if (ri->ri_nextin->d_status & TULIP_DSTS_ERRSUM)
			sc->tulip_if.if_oerrors++;
		}
	    }
	}

	if (++ri->ri_nextin == ri->ri_last)
	    ri->ri_nextin = ri->ri_first;
	ri->ri_free++;
	sc->tulip_if.if_flags &= ~IFF_OACTIVE;
    }
    sc->tulip_if.if_opackets += xmits;
    return xmits;
}

static ifnet_ret_t
tulip_start(struct if_vars * const ifp)
{
    tulip_softc_t * const sc = TULIP_UNIT_TO_SOFTC(ifp->if_unit);
    struct ifqueue * const ifq = &ifp->if_snd;
    tulip_ringinfo_t * const ri = &sc->tulip_txinfo;
    struct mbuf *m, *m0, *next_m0;

    for (;;) {
	tulip_desc_t *eop, *nextout;
	int segcnt, free, recopy;
	tulip_uint32_t d_status;

	if (sc->tulip_flags & TULIP_WANTSETUP) {
	    if ((sc->tulip_flags & TULIP_DOINGSETUP) || ri->ri_free == 1) {
		ifp->if_flags |= IFF_OACTIVE;
		return;
	    }
	    bcopy(sc->tulip_setupdata, sc->tulip_setupbuf,
		   sizeof(sc->tulip_setupbuf));
	    sc->tulip_flags &= ~TULIP_WANTSETUP;
	    sc->tulip_flags |= TULIP_DOINGSETUP;
	    ri->ri_free--;
	    ri->ri_nextout->d_flag &= TULIP_DFLAG_ENDRING|TULIP_DFLAG_CHAIN;
	    ri->ri_nextout->d_flag |= TULIP_DFLAG_TxFIRSTSEG|TULIP_DFLAG_TxLASTSEG
		    |TULIP_DFLAG_TxSETUPPKT|TULIP_DFLAG_TxWANTINTR;
	    if (sc->tulip_flags & TULIP_WANTHASH)
		ri->ri_nextout->d_flag |= TULIP_DFLAG_TxHASHFILT;
	    ri->ri_nextout->d_length1 = sizeof(sc->tulip_setupbuf);
	    ri->ri_nextout->d_addr1 = vtophys(sc->tulip_setupbuf);
	    ri->ri_nextout->d_length2 = 0;
	    ri->ri_nextout->d_addr2 = 0;
	    ri->ri_nextout->d_status = TULIP_DSTS_OWNER;
	    TULIP_WRITE_CSR(sc, csr_txpoll, 1);
	    /*
	     * Advance the ring for the next transmit packet.
	     */
	    if (++ri->ri_nextout == ri->ri_last)
		ri->ri_nextout = ri->ri_first;
	    /*
	     * Make sure the next descriptor is owned by us since it
	     * may have been set up above if we ran out of room in the
	     * ring.
	     */
	    ri->ri_nextout->d_status = 0;
	}

	IF_DEQUEUE(ifq, m);
	if (m == NULL)
	    break;

	/*
	 * Now we try to fill in our transmit descriptors.  This is
	 * a bit reminiscent of going on the Ark two by two
	 * since each descriptor for the TULIP can describe
	 * two buffers.  So we advance through packet filling
	 * each of the two entries at a time to to fill each
	 * descriptor.  Clear the first and last segment bits
	 * in each descriptor (actually just clear everything
	 * but the end-of-ring or chain bits) to make sure
	 * we don't get messed up by previously sent packets.
	 *
	 * We may fail to put the entire packet on the ring if
	 * there is either not enough ring entries free or if the
	 * packet has more than MAX_TXSEG segments.  In the former
	 * case we will just wait for the ring to empty.  In the
	 * latter case we have to recopy.
	 */
	d_status = 0;
	recopy = 0;
	eop = nextout = ri->ri_nextout;
	m0 = m;
	segcnt = 0;
	free = ri->ri_free;
	do {
	    int len = m0->m_len;
	    caddr_t addr = mtod(m0, caddr_t);
	    unsigned clsize = CLBYTES - (((u_long) addr) & (CLBYTES-1));

	    next_m0 = m0->m_next;
	    while (len > 0) {
		unsigned slen = min(len, clsize);

		segcnt++;
		if (segcnt > TULIP_MAX_TXSEG) {
		    recopy = 1;
		    next_m0 = NULL; /* to break out of outside loop */
		    break;
		}
		if (segcnt & 1) {
		    if (--free == 0) {
			/*
			 * There's no more room but since nothing
			 * has been committed at this point, just
			 * show output is active, put back the
			 * mbuf and return.
			 */
			ifp->if_flags |= IFF_OACTIVE;
			IF_PREPEND(ifq, m);
			return;
		    }
		    eop = nextout;
		    if (++nextout == ri->ri_last)
			nextout = ri->ri_first;
		    eop->d_flag &= TULIP_DFLAG_ENDRING|TULIP_DFLAG_CHAIN;
		    eop->d_status = d_status;
		    eop->d_addr1 = vtophys(addr); eop->d_length1 = slen;
		} else {
		    /*
		     *  Fill in second half of descriptor
		     */
		    eop->d_addr2 = vtophys(addr); eop->d_length2 = slen;
		}
		d_status = TULIP_DSTS_OWNER;
		len -= slen;
		addr += slen;
		clsize = CLBYTES;
	    }
	} while ((m0 = next_m0) != NULL);

	/*
	 * The packet exceeds the number of transmit buffer
	 * entries that we can use for one packet, so we have
	 * recopy it into one mbuf and then try again.
	 */
	if (recopy) {
	    MGETHDR(m0, M_DONTWAIT, MT_DATA);
	    if (m0 != NULL) {
		if (m->m_pkthdr.len > MHLEN) {
		    MCLGET(m0, M_DONTWAIT);
		    if ((m0->m_flags & M_EXT) == 0) {
			m_freem(m);
			m_freem(m0);
			continue;
		    }
		}
		m_copydata(m, 0, m->m_pkthdr.len, mtod(m0, caddr_t));
		m0->m_pkthdr.len = m0->m_len = m->m_pkthdr.len;
		IF_PREPEND(ifq, m0);
	    }
	    m_freem(m);
	    continue;
	}

	/*
	 * The descriptors have been filled in.  Now get ready
	 * to transmit.
	 */
#if NBPFILTER > 0
	if (sc->tulip_bpf != NULL)
	    bpf_mtap(sc->tulip_bpf, m);
#endif
	IF_ENQUEUE(&sc->tulip_txq, m);

	/*
	 * Make sure the next descriptor after this packet is owned
	 * by us since it may have been set up above if we ran out
	 * of room in the ring.
	 */
	nextout->d_status = 0;

	/*
	 * If we only used the first segment of the last descriptor,
	 * make sure the second segment will not be used.
	 */
	if (segcnt & 1) {
	    eop->d_addr2 = 0;
	    eop->d_length2 = 0;
	}

	/*
	 * Mark the last and first segments, indicate we want a transmit
	 * complete interrupt, give the descriptors to the TULIP, and tell
	 * it to transmit!
	 */
	eop->d_flag |= TULIP_DFLAG_TxLASTSEG|TULIP_DFLAG_TxWANTINTR;

	/*
	 * Note that ri->ri_nextout is still the start of the packet
	 * and until we set the OWNER bit, we can still back out of
	 * everything we have done.
	 */
	ri->ri_nextout->d_flag |= TULIP_DFLAG_TxFIRSTSEG;
	ri->ri_nextout->d_status = TULIP_DSTS_OWNER;

	/*
	 * This advances the ring for us.
	 */
	ri->ri_nextout = nextout;
	ri->ri_free = free;

	TULIP_WRITE_CSR(sc, csr_txpoll, 1);
    }
    if (m != NULL) {
	ifp->if_flags |= IFF_OACTIVE;
	IF_PREPEND(ifq, m);
    }
}

static int
tulip_intr(
    void *arg)
{
    tulip_softc_t * sc = (tulip_softc_t *) arg;
    tulip_uint32_t csr;
    int progress = 0;

    do {
	while ((csr = TULIP_READ_CSR(sc, csr_status)) & (TULIP_STS_NORMALINTR|TULIP_STS_ABNRMLINTR)) {
	    progress = 1;
	    TULIP_WRITE_CSR(sc, csr_status, csr & sc->tulip_intrmask);

	    if (csr & TULIP_STS_SYSERROR) {
		if ((csr & TULIP_STS_ERRORMASK) == TULIP_STS_ERR_PARITY) {
		    tulip_reset(sc);
		    tulip_init(sc);
		    break;
		}
	    }
	    if (csr & TULIP_STS_ABNRMLINTR) {
		printf("%s%d: abnormal interrupt: 0x%05x [0x%05x]\n",
		       sc->tulip_name, sc->tulip_unit, csr, csr & sc->tulip_intrmask);
		TULIP_WRITE_CSR(sc, csr_command, sc->tulip_cmdmode);
	    }
	    if (csr & TULIP_STS_RXINTR)
		tulip_rx_intr(sc);
	    if (sc->tulip_txinfo.ri_free < sc->tulip_txinfo.ri_max) {
		tulip_tx_intr(sc);
		tulip_start(&sc->tulip_if);
	    }
	}
    } while ((sc = sc->tulip_slaves) != NULL);
    return progress;
}

/*
 *
 */

static void
tulip_delay_300ns(
    tulip_softc_t * const sc)
{
    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);
    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);

    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);
    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);

    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);
    TULIP_READ_CSR(sc, csr_busmode); TULIP_READ_CSR(sc, csr_busmode);
}

#define EMIT    do { TULIP_WRITE_CSR(sc, csr_srom_mii, csr); tulip_delay_300ns(sc); } while (0)

static void
tulip_idle_srom(
    tulip_softc_t * const sc)
{
    unsigned bit, csr;
    
    csr  = SROMSEL | SROMRD; EMIT;  
    csr ^= SROMCS; EMIT;
    csr ^= SROMCLKON; EMIT;

    /*
     * Write 25 cycles of 0 which will force the SROM to be idle.
     */
    for (bit = 3 + SROM_BITWIDTH + 16; bit > 0; bit--) {
        csr ^= SROMCLKOFF; EMIT;    /* clock low; data not valid */
        csr ^= SROMCLKON; EMIT;     /* clock high; data valid */
    }
    csr ^= SROMCLKOFF; EMIT;
    csr ^= SROMCS; EMIT; EMIT;
    csr  = 0; EMIT;
}

     
static void
tulip_read_srom(
    tulip_softc_t * const sc)
{   
    int idx; 
    const unsigned bitwidth = SROM_BITWIDTH;
    const unsigned cmdmask = (SROMCMD_RD << bitwidth);
    const unsigned msb = 1 << (bitwidth + 3 - 1);
    unsigned lastidx = (1 << bitwidth) - 1;

    tulip_idle_srom(sc);

    for (idx = 0; idx <= lastidx; idx++) {
        unsigned lastbit, data, bits, bit, csr;
        csr  = SROMSEL | SROMRD;        EMIT;
        csr ^= SROMCSON;                EMIT;
        csr ^=            SROMCLKON;    EMIT;
    
        lastbit = 0;
        for (bits = idx|cmdmask, bit = bitwidth + 3; bit > 0; bit--, bits <<= 1) {
            const unsigned thisbit = bits & msb;
            csr ^= SROMCLKOFF; EMIT;    /* clock low; data not valid */
            if (thisbit != lastbit) {
                csr ^= SROMDOUT; EMIT;  /* clock low; invert data */
            }
            csr ^= SROMCLKON; EMIT;     /* clock high; data valid */
            lastbit = thisbit;
        }
        csr ^= SROMCLKOFF; EMIT;

        for (data = 0, bits = 0; bits < 16; bits++) {
            data <<= 1;
            csr ^= SROMCLKON; EMIT;     /* clock high; data valid */ 
            data |= TULIP_READ_CSR(sc, csr_srom_mii) & SROMDIN ? 1 : 0;
            csr ^= SROMCLKOFF; EMIT;    /* clock low; data not valid */
        }
	sc->tulip_rombuf[idx*2] = data & 0xFF;
	sc->tulip_rombuf[idx*2+1] = data >> 8;
        csr  = SROMSEL | SROMRD; EMIT;
        csr  = 0; EMIT;
    }
}

#define	tulip_mchash(mca)	(tulip_crc32(mca, 6) & 0x1FF)
#define	tulip_srom_crcok(databuf)	( \
    ((tulip_crc32(databuf, 126) & 0xFFFF) ^ 0xFFFF)== \
     ((databuf)[126] | ((databuf)[127] << 8)))

static unsigned
tulip_crc32(
    const unsigned char *databuf,
    size_t datalen)
{
    u_int idx, bit, data, crc = 0xFFFFFFFFUL;

    for (idx = 0; idx < datalen; idx++)
        for (data = *databuf++, bit = 0; bit < 8; bit++, data >>= 1)
            crc = (crc >> 1) ^ (((crc ^ data) & 1) ? TULIP_CRC32_POLY : 0);
    return crc;
}


/*
 * This deals with the vagaries of the address roms and the
 * brain-deadness that various vendors commit in using them.
 */
static int
tulip_read_macaddr(
    tulip_softc_t *sc)
{
    int cksum, rom_cksum, idx;
    tulip_uint32_t csr;
    unsigned char tmpbuf[8];
    static const u_char testpat[] = { 0xFF, 0, 0x55, 0xAA, 0xFF, 0, 0x55, 0xAA };

    {
	int new_srom_fmt = 0;
	/*
	 * Assume all DC21140 board are compatible with the
	 * DEC 10/100 evaluation board.  Not really valid but
	 * it's the best we can do until every one switches to
	 * the new SROM format.
	 */
#if 0	/* XXX */
	if (sc->tulip_chipid == TULIP_DC21140)
	    sc->tulip_boardsw = &tulip_dc21140_eb_boardsw;
#endif
	tulip_read_srom(sc);
	if (tulip_srom_crcok(sc->tulip_rombuf)) {
	    /*
	     * SROM CRC is valid therefore it must be in the
	     * new format.
	     */
	    new_srom_fmt = 1;
	} else if (sc->tulip_rombuf[126] == 0xff && sc->tulip_rombuf[127] == 0xFF) {
	    /*
	     * No checksum is present.  See if the SROM id checks out;
	     * the first 18 bytes should be 0 followed by a 1 followed
	     * by the number of adapters (which we don't deal with yet).
	     */
	    for (idx = 0; idx < 18; idx++) {
		if (sc->tulip_rombuf[idx] != 0)
		    break;
	    }
	    if (idx == 18 && sc->tulip_rombuf[18] == 1 && sc->tulip_rombuf[19] != 0)
		new_srom_fmt = 2;
	}
	if (new_srom_fmt) {
	    /*
	     * New SROM format.  Copy out the Ethernet address.
	     * If it contains a DE500-XA string, then it must be
	     * a DE500-XA.
	     */
	    bcopy(sc->tulip_rombuf + 20, sc->tulip_hwaddr, 6);
	    if (bcmp(sc->tulip_rombuf + 29, "DE500-XA", 8) == 0)
		sc->tulip_boardsw = &tulip_dc21140_de500_boardsw;
	    if (sc->tulip_boardsw == NULL)
		return -6;
	    sc->tulip_flags |= TULIP_ROMOK;
	    return 0;
	}
    }


    if (bcmp(&sc->tulip_rombuf[0], &sc->tulip_rombuf[16], 8) != 0) {
	/*
	 * Some folks don't use the standard ethernet rom format
	 * but instead just put the address in the first 6 bytes
	 * of the rom and let the rest be all 0xffs.  (Can we say
	 * ZNYX???)
	 */
	for (idx = 6; idx < 32; idx++) {
	    if (sc->tulip_rombuf[idx] != 0xFF)
		return -4;
	}
	/*
	 * Make sure the address is not multicast or locally assigned
	 * that the OUI is not 00-00-00.
	 */
	if ((sc->tulip_rombuf[0] & 3) != 0)
	    return -4;
	if (sc->tulip_rombuf[0] == 0 && sc->tulip_rombuf[1] == 0
		&& sc->tulip_rombuf[2] == 0)
	    return -4;
	bcopy(sc->tulip_rombuf, sc->tulip_hwaddr, 6);
	sc->tulip_flags |= TULIP_ROMOK;
	return 0;
    }

    /*
     * This is the standard DEC address ROM test.
     */

    if (bcmp(&sc->tulip_rombuf[24], testpat, 8) != 0)
	return -3;

    tmpbuf[0] = sc->tulip_rombuf[15]; tmpbuf[1] = sc->tulip_rombuf[14];
    tmpbuf[2] = sc->tulip_rombuf[13]; tmpbuf[3] = sc->tulip_rombuf[12];
    tmpbuf[4] = sc->tulip_rombuf[11]; tmpbuf[5] = sc->tulip_rombuf[10];
    tmpbuf[6] = sc->tulip_rombuf[9];  tmpbuf[7] = sc->tulip_rombuf[8];
    if (bcmp(&sc->tulip_rombuf[0], tmpbuf, 8) != 0)
	return -2;

    bcopy(sc->tulip_rombuf, sc->tulip_hwaddr, 6);

    cksum = *(u_short *) &sc->tulip_hwaddr[0];
    cksum *= 2;
    if (cksum > 65535) cksum -= 65535;
    cksum += *(u_short *) &sc->tulip_hwaddr[2];
    if (cksum > 65535) cksum -= 65535;
    cksum *= 2;
    if (cksum > 65535) cksum -= 65535;
    cksum += *(u_short *) &sc->tulip_hwaddr[4];
    if (cksum >= 65535) cksum -= 65535;

    rom_cksum = *(u_short *) &sc->tulip_rombuf[6];
	
    if (cksum != rom_cksum)
	return -1;

    /*
     * Check for various boards based on OUI.  Did I say braindead?
     */

    if (sc->tulip_chipid == TULIP_DC21140) {
	if (sc->tulip_hwaddr[0] == TULIP_OUI_COGENT_0
		&& sc->tulip_hwaddr[1] == TULIP_OUI_COGENT_1
		&& sc->tulip_hwaddr[2] == TULIP_OUI_COGENT_2) {
	    if (sc->tulip_rombuf[32] == TULIP_COGENT_EM100_ID)
		sc->tulip_boardsw = &tulip_dc21140_cogent_em100_boardsw;
	}
	if (sc->tulip_hwaddr[0] == TULIP_OUI_ZNYX_0
		&& sc->tulip_hwaddr[1] == TULIP_OUI_ZNYX_1
		&& sc->tulip_hwaddr[2] == TULIP_OUI_ZNYX_2) {
	    /* this at least works for the zx342 from Znyx */
	    sc->tulip_boardsw = &tulip_dc21140_znyx_zx34x_boardsw;
	}
    }

    sc->tulip_flags |= TULIP_ROMOK;
    return 0;
}

static void
tulip_addr_filter(tulip_softc_t * const sc)
{
    tulip_uint32_t *sp = sc->tulip_setupdata;
    struct ether_multistep step;
    struct ether_multi *enm;
    int i;

    sc->tulip_flags &= ~TULIP_WANTHASH;
    sc->tulip_flags |= TULIP_WANTSETUP;
    sc->tulip_cmdmode &= ~TULIP_CMD_RXRUN;
    sc->tulip_intrmask &= ~TULIP_STS_RXSTOPPED;

    for (i=0; i< 16; i++)
	sp[i]= 0xffffffffUL;
	/*
	 * If we have more than 14 multicasts, we have
	 * go into hash perfect mode (512 bit multicast
	 * hash and one perfect hardware).
	 */

	bzero(sc->tulip_setupdata, sizeof(sc->tulip_setupdata));
	hash = tulip_mchash(etherbroadcastaddr);
	sp[hash >> 4] |= 1 << (hash & 0xF);
	ETHER_FIRST_MULTI(step, &sc->tulip_ac, enm);
	while (enm != NULL) {
	    hash = tulip_mchash(enm->enm_addrlo);
	    sp[hash >> 4] |= 1 << (hash & 0xF);
	    ETHER_NEXT_MULTI(step, enm);
	}
	sc->tulip_flags |= TULIP_WANTHASH;
	sp[39] = ((u_short *) sc->tulip_ac.ac_enaddr)[0]; 
	sp[40] = ((u_short *) sc->tulip_ac.ac_enaddr)[1]; 
	sp[41] = ((u_short *) sc->tulip_ac.ac_enaddr)[2];
}

static void
tulip_attach( tulip_softc_t * const sc)
{
    struct if_vars * const ifp = &sc->tulip_if;

#if 0
    ifp->if_flags = IFF_BROADCAST|IFF_SIMPLEX|IFF_NOTRAILERS|IFF_MULTICAST;
    ifp->if_ioctl = tulip_ioctl;
    ifp->if_output = ether_output;
    ifp->if_start = tulip_start;
#endif 
    printf("%s%d", sc->tulip_name, sc->tulip_unit);
    printf(": %s%s pass %d.%d Ethernet address %s\n", 
	   sc->tulip_boardsw->bd_description,
	   tulip_chipdescs[sc->tulip_chipid],
	   (sc->tulip_revinfo & 0xF0) >> 4,
	   sc->tulip_revinfo & 0x0F,
	   ether_sprintf(sc->tulip_hwaddr));

    if ((*sc->tulip_boardsw->bd_media_probe)(sc)) {
	ifp->if_flags |= IFF_ALTPHYS;
    } else {
	sc->tulip_flags |= TULIP_ALTPHYS;
    }
    tulip_reset(sc);
    if_attach(ifp);
}

static void
tulip_initcsrs(tulip_softc_t * const sc, tulip_csrptr_t csr_base,
    size_t csr_size)
{
    sc->tulip_csrs.csr_busmode		= csr_base +  0 * csr_size;
    sc->tulip_csrs.csr_txpoll		= csr_base +  1 * csr_size;
    sc->tulip_csrs.csr_rxpoll		= csr_base +  2 * csr_size;
    sc->tulip_csrs.csr_rxlist		= csr_base +  3 * csr_size;
    sc->tulip_csrs.csr_txlist		= csr_base +  4 * csr_size;
    sc->tulip_csrs.csr_status		= csr_base +  5 * csr_size;
    sc->tulip_csrs.csr_command		= csr_base +  6 * csr_size;
    sc->tulip_csrs.csr_intr		= csr_base +  7 * csr_size;
    sc->tulip_csrs.csr_missed_frame	= csr_base +  8 * csr_size;
    if (sc->tulip_chipid == TULIP_DC21140) {
	sc->tulip_csrs.csr_srom_mii		= csr_base +  9 * csr_size;
	sc->tulip_csrs.csr_gp_timer		= csr_base + 11 * csr_size;
	sc->tulip_csrs.csr_gp			= csr_base + 12 * csr_size;
	sc->tulip_csrs.csr_watchdog		= csr_base + 15 * csr_size;
    }
}

static void
tulip_initring( tulip_softc_t * const sc, tulip_ringinfo_t * const ri,
    tulip_desc_t *descs, int ndescs)
{
    ri->ri_max = ndescs;
    ri->ri_first = descs;
    ri->ri_last = ri->ri_first + ri->ri_max;
    bzero((caddr_t) ri->ri_first, sizeof(ri->ri_first[0]) * ri->ri_max);
    ri->ri_last[-1].d_flag = TULIP_DFLAG_ENDRING;
}

/*
 * This is the PCI configuration support.  Since the DC21040 is available
 * on both EISA and PCI boards, one must be careful in how defines the
 * DC21040 in the config file.
 */

#define	PCI_CFID	0x00	/* Configuration ID */
#define	PCI_CFCS	0x04	/* Configurtion Command/Status */
#define	PCI_CFRV	0x08	/* Configuration Revision */
#define	PCI_CFLT	0x0c	/* Configuration Latency Timer */
#define	PCI_CBIO	0x10	/* Configuration Base IO Address */
#define	PCI_CBMA	0x14	/* Configuration Base Memory Address */
#define	PCI_CFIT	0x3c	/* Configuration Interrupt */
#define	PCI_CFDA	0x40	/* Configuration Driver Area */

static int
tulip_pci_shutdown( struct kern_devconf * const kdc, int force)
{
    if (kdc->kdc_unit < NDE) {
	tulip_softc_t * const sc = TULIP_UNIT_TO_SOFTC(kdc->kdc_unit);
	TULIP_WRITE_CSR(sc, csr_busmode, TULIP_BUSMODE_SWRESET);
	DELAY(10);	/* Wait 10 microsends (actually 50 PCI cycles but at 
			   33MHz that comes to two microseconds but wait a
			   bit longer anyways) */
    }
    (void) dev_detach(kdc);
    return 0;
}

static char*
tulip_pci_probe( pcici_t config_id, pcidi_t device_id)
{
    if (PCI_VENDORID(device_id) != DEC_VENDORID)
	return NULL;
    if (PCI_CHIPID(device_id) == DC21140_CHIPID)
	return "Digital DC21140 Fast Ethernet";
    return NULL;
}

static void  tulip_pci_attach(pcici_t config_id, int unit);
static u_long tulip_pci_count;

static void
tulip_pci_attach( pcici_t config_id, int unit)
{
    tulip_softc_t *sc;
    int retval, idx, revinfo, id;
#if !defined(TULIP_IOMAPPED) && !defined(__bsdi__)
    vm_offset_t pa_csrs;
#endif
    unsigned csroffset = TULIP_PCI_CSROFFSET;
    unsigned csrsize = TULIP_PCI_CSRSIZE;
    tulip_csrptr_t csr_base;
    tulip_desc_t *rxdescs, *txdescs;
    tulip_chipid_t chipid = TULIP_CHIPID_UNKNOWN;

    if (unit >= NDE) {
	printf("de%d: not configured; kernel is built for only %d device%s.\n",
	       unit, NDE, NDE == 1 ? "" : "s");
	return;
    }

    revinfo = pci_conf_read(config_id, PCI_CFRV) & 0xFF;
    id = pci_conf_read(config_id, PCI_CFID);

    if (PCI_VENDORID(id) == DEC_VENDORID) {
	if (PCI_CHIPID(id) == DC21140_CHIPID) chipid = TULIP_DC21140;
    }
    if (chipid == TULIP_CHIPID_UNKNOWN)
	return;

    if (chipid == TULIP_DC21140 && revinfo < 0x11) {
	printf("de%d: not configured; DC21140 pass 1.1 required (%d.%d found)\n",
	       unit, revinfo >> 4, revinfo & 0x0f);
	return;
    }

    sc = (tulip_softc_t *) malloc(sizeof(*sc), M_DEVBUF, M_NOWAIT);
    if (sc == NULL)
	return;
    bzero(sc, sizeof(*sc));				/* Zero out the softc*/

    rxdescs = (tulip_desc_t *)
	malloc(sizeof(tulip_desc_t) * TULIP_RXDESCS, M_DEVBUF, M_NOWAIT);
    if (rxdescs == NULL) {
	free((caddr_t) sc, M_DEVBUF);
	return;
    }

    txdescs = &tulips(unit);

    sc->tulip_chipid = chipid;
    sc->tulip_unit = unit;
    sc->tulip_name = "de";
    sc->tulip_revinfo = revinfo;
#if defined(TULIP_IOMAPPED)
    retval = pci_map_port(config_id, PCI_CBIO, &csr_base);
#else
    retval = pci_map_mem(config_id, PCI_CBMA, (vm_offset_t *) &csr_base, &pa_csrs);
#endif
    if (!retval) {
	free((caddr_t) txdescs, M_DEVBUF);
	free((caddr_t) rxdescs, M_DEVBUF);
	free((caddr_t) sc, M_DEVBUF);
	return;
    }

    tulip_initcsrs(sc, csr_base + csroffset, csrsize);
    tulip_initring(sc, &sc->tulip_rxinfo, rxdescs, TULIP_RXDESCS);
    tulip_initring(sc, &sc->tulip_txinfo, txdescs, TULIP_TXDESCS);
    if ((retval = tulip_read_macaddr(sc)) < 0) {
	printf("%s%d", sc->tulip_name, sc->tulip_unit);
	printf(": can't read ENET ROM (why=%d) (", retval);
	for (idx = 0; idx < 32; idx++)
	    printf("%02x", sc->tulip_rombuf[idx]);
	printf("\n");
	printf("%s%d: %s%s pass %d.%d Ethernet address %s\n",
	       sc->tulip_name, sc->tulip_unit,
	       (sc->tulip_boardsw != NULL ? sc->tulip_boardsw->bd_description : ""),
	       tulip_chipdescs[sc->tulip_chipid],
	       (sc->tulip_revinfo & 0xF0) >> 4, sc->tulip_revinfo & 0x0F,
	       "unknown");
    } else {
	/*
	 * Make sure there won't be any interrupts or such...
	 */
	TULIP_WRITE_CSR(sc, csr_busmode, TULIP_BUSMODE_SWRESET);
	DELAY(10);	/* Wait 10 microsends (actually 50 PCI cycles but at 
			   33MHz that comes to two microseconds but wait a
			   bit longer anyways) */
	tulip_reset(sc);
	tulip_attach(sc);
    }
}
