/***
 *** prog84.c
 ***
 *** these routines use the parallel or serial port
 *** to program a PIC 16C84 in serial mode
 ***/

#include "prog84.h"

int
    p_power   = -1,
    p_mclr    = -1,
    p_clock   = -1,
    p_data    = -1,
    p_clock_f = -1,
    p_data_f  = -1;

struct lp_io lpb;

/* if you edit this array be sure to change the array lengths in the
   various calls that reference it */
static 
struct lp_name names[] = {
    { "base",    &(lpb.base) },

    { "power",   &p_power    },
    { "mclr",    &p_mclr     },
    { "clock",   &p_clock    },
    { "data",    &p_data     },
    { "clock_f", &p_clock_f  },
    { "data_f",  &p_data_f   },

    /* alternate names */
    { "rb6",     &p_clock    },
    { "rb7",     &p_data     },
    { "rb6_f",   &p_clock_f  },
    { "rb7_f",   &p_data_f   }
};

char *cfg_file = "lp_cfg";

int p84_verbose = 0;

/***
 *** poweron the pic and enter programming mode
 ***/
void power_and_reset()
{
    SET(mclr, 0);
    SET(power, 1);	/* the serial programmer needs this... */
    SET(clock, 0);
    SET(data, 0);
    FLUSH;
	if (p84_verbose) printf("Charging capacitor...\n");
    sleep(2);
    SET(mclr, 0);
    FLUSH;
    usleep(50000);
    SET(mclr, 1);
    FLUSH;
}

int testHW()
{
    int i,j;     /* utility variables */
    int ret[4];  /* array of test results */
    int m;       /* merged test results */

    power_and_reset();

    if (p_power == -1 || p_mclr == -1 || p_clock == -1 ||
	    p_data == -1 || p_data_f == -1) {
	fprintf(stderr, 
	    "Error: I need control over power, MCLR, clock, and data,\n"
	    "       and I need to be able to read data back.\n");
	return 1;
    }

    if (p84_verbose) {
	printf("[testing loopback");
	fflush(stdout);
    }

    for (i=0; i<4; i++) {
	SET(clock, i&1);
	SET(data, i&2);
	FLUSH;
	usleep(100); /* here I can be slow, no problem */
	j = GET(data_f) ? 2 : 0;
	if (p_clock_f != -1)
	    j |= GET(clock_f)? 1 : 0;
	else
	    j |= i & 1;  /* fake it if we don't have it */
	ret[i] = j;
    }

    SET(clock, 0);
    SET(data, 0);
    FLUSH;

    if (p84_verbose)
	printf("]\n");

    /***
     *** analyze the test results. We put our four 2-bit responses
     *** into an int, one per nybble so we can use handy hex notation.
     ***/

    m = (((((ret[3] << 4) | ret[2]) << 4) | ret[1]) << 4) | ret[0];

    if(m == 0x3210)	/*** this is what we expect ***/
	return 0;

    fprintf(stderr, "Bad test results!\n  [data, clock]->[data_f, clock_f]:");
    for(i = 0; i < 4; i ++)
	fprintf(stderr, " %d%d->%d%d", (i>>1)&1, i&1,
		(ret[i]>>1)&1, (ret[i])&1);
    fprintf(stderr,"\n");

    if (p_clock_f == -1)
	fprintf(stderr, "(clock_f simulated since I don't actually have it)\n");

    if (m == 0x3120)
	fprintf(stderr, "clock and data reversed?\n"
	    " (could be either the clock & data lines, or the sense lines)\n"
	);

    switch(m & 0x1111) {
    case 0x0000:
	fprintf(stderr, "clock_f stuck low?\n");
	break;
    case 0x0101:
	fprintf(stderr, "clock_f inverted?\n");
	break;
    case 0x1111:
	fprintf(stderr, "clock_f stuck high?\n");
	break;
    }

    switch(m & 0x2222) {
    case 0x0000:
	fprintf(stderr,"data_f stuck low?\n");
	break;
    case 0x0022:
	fprintf(stderr,"data_f inverted?\n");
	break;
    case 0x2222:
	fprintf(stderr,"data_f stuck high?\n");
	break;
    }

    /***
     *** here I evidently have a failure, and should return 1. However,
     *** I keep on trying in the hope it is a temporary thing.
     ***/
    return 0;
}

/*** 
 *** Pipe some bits through the 16C84. Returns the values read from
 *** the data pin at the appropriate point in the cycle for reading
 *** data mem.  In a write, will return 'bits'.  (Unless, of course,
 *** something's wrong.)
 ***/

unsigned int pipeBits(unsigned int bits, int count)
{
    unsigned int ret = 0;
    int bit;

    for (bit=0; bit<count; bit++) {
	SET(clock, 1);
	SET(data, bits&1);
	FLUSH;

	/* 100 ns. delay */
	SET(clock, 0);
	FLUSH;
	ret |= (GET(data_f)? 1 : 0 ) << bit;
	if (p84_verbose > 1)
	    printf(". bit %2d: send %d get %d\n",
		bit, bits&1, GET(data_f)?1:0);

	/* 100ns delay */
	bits >>= 1;
    }
    SET(data, 1); /* the HI-Z state */
    FLUSH;
    /* required guard time of at least 1 usec. */
    return ret;
}

void xPipeBits(unsigned int bits, int count)
{
    unsigned int ret;

    ret = pipeBits(bits, count);
    if (ret != bits) {
	fprintf(stderr,
	    "ERROR: sent %d bits: %d (0%o), got %d (0%o)\n"
	    "Resetting & powering down.\n",
	    count, bits, bits, ret, ret);
	progShutdown();
	exit(2);
    }
}

/* A generic command-plus-14-bits-to-write call */
void genericWrite(int cmd, unsigned int data)
     /* 6 bits of command, 14 bits of data */
{
    xPipeBits(cmd, 6);		  	/* Send the command */
    xPipeBits((data << 1) & 0x7FFE, 16);	/* send the data */
}

/* A generic command-plus-14-bits-to-read call */
unsigned int genericRead(int cmd)
{
    unsigned int ret;

    xPipeBits(cmd, 6);			/* send the command */
    ret = pipeBits(0x7FFE, 16);		/* read the result */
    return((ret & 0x7FFE) >> 1);	/* remove guard bits */
}


void progSetup()
{
    /* read the configuration file */
    if (lpb_cfg_read(cfg_file, 11, names)) {
	fprintf(stderr, "error reading cfg file %s\n", cfg_file);
	exit(1);
    }

    if (p84_verbose)
	lpb_dump_names(7, names);

    /* open /dev/port */
    open_io();

    /* check that the programmer's not totally dead */
    if (testHW()) {
	fprintf(stderr,"\nProgrammer is not responding, aborting.\n");
	exit(1);
    }
}

void progShutdown()
{
    SET(mclr, 0);
    FLUSH;
    usleep(1000);	/* shutdown... */
    SET(power, 0);
    SET(clock, 0);
    SET(data, 0);
    FLUSH;
  
    close_io();
}
