/*
 * sessions.c -- session handling for rmdpd
 *
 * This file is part of
 *
 * rmdp -- Reliable Multicast data Distribution Protocol
 * 
 * (C) 1996-1998 Luigi Rizzo and Lorenzo Vicisano
 *     (luigi@iet.unipi.it, vicisano@cs.ucl.ac.uk)
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Luigi Rizzo,
 *      Lorenzo Vicisano and other contributors.
 * 4. Neither the name of the Authors nor the names of other contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

#include "rmdp.h"
#include "sessions.h"
#include "event.h"
#include <sys/stat.h> /* for stat */

extern EVLIST *events;
extern ui32  source_id;
extern int data_pkt_len;
extern int k_glob;
extern int n_glob;
extern int ttl_glob;
extern ui32 dumm_addr;
extern ui16 dumm_port;
extern int datas;

void *fec_code;
SESSION *all_sessions=NULL;

void
prolong(n32 s_id)
{
    SESSION *se ;

    for (se = all_sessions ; se ; se = se -> next ) {
	if (se->id == s_id) {
	    DEB(fprintf(stderr,"session found\n");)
	    if (se->pkts_left == 0) {
		se_start(se, 1 /* set pkts_left */);
	    } else
		se->pkts_left += 3*se->B ;
	    return;
	}
    }
    fprintf(stderr, "--- Warning, session 0x%08x not found\n",
	s_id);
}

/*
 * check if a request for this object is already scheduled;
 * in case schedule it, or modify the schedule parameters.
 * return the session id and the actual bandwidth;
 */
void
schedule(char *path, ui32 c_id, ui32 r_id,
	ui32 *req_bw, ui32 *s_id, n32 *addr, n16 *port)
{
    SESSION *se;

    /* see if an already scheduled session for this pathname
     * can be extended and in case change its parameters...
     * Unicast sessions explicitly set m_addr, so there is no
     * already existing session.
     */
    for (se=all_sessions; se; se= se->next) {
	if (!strcmp(path, se->fname) && se->ch == datas) {
	    if ( (*addr == 0 ) /* multicast session */
		|| ( (se->r_id == r_id ) && (se->c_id == c_id) )
	       ) {
		DEB(fprintf(stderr, "already have an active session for %s\n",
		    path) );
		*s_id = se->id ;
		*addr = se->ip ;
		*port = se->port ;
		if (se->pkts_left == 0) {
		    if (*req_bw < 1000 || *req_bw > 1024*1024 )
			*req_bw = 64*1024 ; /* bit/s */
		    se->rate = (int)(8192 * 1.0e6/ *req_bw) ;
		    se_start(se, 1 /* set pkts_left */);
		} else {
		    *req_bw = (int)(8192*1.0e6/se->rate);
		    se->pkts_left = se->B * (3 + se->k); /* XXX fixme */
		}
		DDB(fprintf(stderr, "+++ running at %d Kbit/s\n",
		    (int)(8192*1.0e3/se->rate));)
		return ;
	    }
	}
    }
    {
	struct stat filest;
	int i=stat(path, &filest);
	if (i < 0 ) { /* not found ? */
	    *s_id = 0 ;
	    return ;				/* XXX what should we do? */
	}
    }

    if (NULL == (se = calloc(1, sizeof(SESSION)))) {
	perror("sender: calloc");
	exit(1); /* XXX */
    }
    se->next = all_sessions;
    all_sessions = se ;

    *s_id = compute_id();	/* XXX must see if unique and != 0 */
    if (*addr == 0) {
	se->ip = *addr = dumm_addr;			/* XXX */
	se->port = *port = dumm_port;			/* XXX */
    } else {
	se->ip = *addr ;
	se->port = *port ;
	se->ch = openssock(*addr, *port, 0 /* TTL XXX */ );
    }
    se->k = k_glob;
    se->n = n_glob;
    se->flags = 0;

    se->c_id = c_id;
    se->r_id = r_id;
    se->id = *s_id;
    se->ttl = ttl_glob;		/* XXX */
    if (*req_bw < 1000 || *req_bw > 1024*1024 )
	*req_bw = 64*1024 ; /* bit/s */
    se->rate = (int)(8192 * 1.0e6/ *req_bw) ;

    fprintf(stderr, "+++ running at %8.3f Kbit/s\n",
	(8000.0e3/se->rate)); /* 1K = 1024 bits! */
    se->idx = 0;

    if (NULL == (se->fname = calloc(1, strlen(path)+1))) {
	perror("sender: calloc");
	exit(1);  /* XXX */
    }
    strcpy(se->fname, path);
    se_start(se, 1 /* set pkts_left */);
}

int
se_start(SESSION *s, int set_pkts_left)
{
  ui32 fpkts;
  struct stat filest;
  int done, i, j;
  FILE *inf;

  if (s->alldata == NULL ) {
    fprintf(stderr, "--- reading file for %s\n", s->fname); 
    stat(s->fname, &filest);
    (s->flength) = filest.st_size;
    if ((s->flength) == 0) {
	DEB(fprintf(stderr, "XXX se_start: flength == 0\n"));
	/* XXX */
    }

    fpkts = ((s->flength)+data_pkt_len-1)/data_pkt_len;
    s->B = (fpkts+(s->k)-1)/(s->k);

    if (NULL == (inf=fopen(s->fname, "r"))) {
	DDB(fprintf(stderr, "XXX se_start: can't open %s\n", s->fname));
	/* XXX */
    }
    if (NULL == (s->alldata = (void *)calloc(s->B, sizeof(void *)))) {
	DDB(fprintf(stderr, "XXX se_start: can't allocate alldata\n"));
	/* XXX */
    }
    for (done=0, i=0; i < s->B ; i++ ) {
	s->alldata[i] = (void **)calloc((s->k), sizeof(void *)) ;
	if ( s->alldata[i] == NULL ) {
	    DDB(fprintf(stderr, "XXX se_start: can't allocate alldata\n"));
	    /* XXX */
	}
	for (j=0; j<(s->k); j++) {
	    if (NULL ==(s->alldata[i][j]  = (void *)calloc(1, data_pkt_len))) {
		DDB(fprintf(stderr, "XXX se_start: can't allocate alldata\n"));
		/* XXX */
	    }
	    if (!done && (fread((void *)(s->alldata[i][j]), 1,
		data_pkt_len, inf)) != data_pkt_len) {
		done = 1;
	    }
	}
    }
  }
    if (set_pkts_left) {
        s->pkts_left = s->B * (3 + s->k); /* XXX fixme */
        if (s->ch == 0) /* XXX */
	    s->ch = datas ;
	settimer(events, s->id, s->rate, (CALLBACK)se_send, (void *)s, 0);
    }
    return(0);
}

/*
 * se_flush flushes the data buffers for a file, but does not
 * completely clean the session. This is done so that memory can
 * be recovered but state is retained for a sufficient time.
 */
int
se_flush(SESSION *s)
{
    int i, j;

    if (s->alldata) {
	for (i=0; i< s->B ; i++){
	    for (j=0; j < s->k ; j++)
	       free(s->alldata[i][j]);
	    free(s->alldata[i]);
	}
	free(s->alldata);
	s->alldata = NULL ;
	DEB(fprintf(stderr, "--- freed data for \"%s\"\n", s->fname));
    }
    return 0 ;
}

/*
 * se_delete does delete a session, it aborts in case the session
 * has ben resumed (pkts_left > 0).
 *
 * note -- it might happen that a se_delete is received for a stale
 * session.
 */
int
se_delete(SESSION *s)
{
    int i;
    SESSION *se, *pr;
    struct timeval now;

    for (se=all_sessions; se ; pr = se, se = se->next)
	if (s == se)
	    break;
    if (se == NULL) {
	fprintf(stderr,"-- received a delete for a stale session, ignoring\n");
	return 0;
    }
    if (s->pkts_left > 0 ) {
	DDB(fprintf(stderr,"xxx session \"%s\" resumed!\n", s->fname);)
	s->busy = 0;
	return 0;
    }
    if (s->busy == 0) {
	s->busy = 1;
	settimer(events, s->id, 60*1000000, (CALLBACK)se_delete, s, 1);
    }
    timestamp_t (&now);
    fprintf(stderr,"--- now %ld.%06ld, flush %ld.%06ld, delete %ld.%06ld\n",
	now.tv_sec % 86400, now.tv_usec,
	s->flush_at.tv_sec % 86400 , s->flush_at.tv_usec,
	s->delete_at.tv_sec %86400 , s->delete_at.tv_usec);
    fprintf(stderr, "--- handling session 0x%08x\n", s);
    i= diff_t (now, s->flush_at);
    if (i < 0 ) {
	fprintf(stderr,"--- schedule a flush for 0x%08x in %d ms\n",
		s->id, -i/1000 );
	settimer(events, s->id, -i, (CALLBACK)se_delete, s, 0);
	return 0 ;
    }
    se_flush(s);
    i= diff_t (now, s->delete_at);
    if (i < 0 ) {
	fprintf(stderr,"--- schedule a delete for 0x%08x in %d ms\n",
		s->id, -i/1000 );
	settimer(events, s->id, -i, (CALLBACK)se_delete, s, 0);
	return 0 ;
    }
    /*
     * XXX should check unicast...
     */
    if (s->ch != datas) { /* XXX hack... */
	DDB(fprintf(stderr, "--- closing socket for \"%s\"\n",
	    s->fname));
	close(s->ch);
    }
    DDB(fprintf(stderr, "--- killing name for \"%s\"\n",
	s->fname));
    free(s->fname);
    s->fname = NULL ;

    if (se == all_sessions)
	all_sessions = all_sessions->next;
    else
	pr->next = se->next;
    free(se);

    return 0 ;
}

int
se_send(SESSION *s)
{
    int pkt_ix, blk_ix;
    DATA_P packet;
    ui32 thislen;
    static int do_err = 1;

send_again:

    if (s->alldata == NULL)
	se_start(s, 0 ); /* allocate and load... */
    blk_ix = (s->idx)%(s->B);
    pkt_ix = ((s->idx)/(s->B))%(s->n);
    fec_encode(fec_code, s->alldata[blk_ix], &packet.data, pkt_ix, data_pkt_len);

    DEB(fprintf(stderr, "sending data (%3d %3d) on session %x\n",
	pkt_ix, blk_ix, s->id))

    if (s->pkts_left < max(10, s->B) )
	s->flags |= RMDP_F_LAST_ROUND ;
    else
	s->flags &= ~RMDP_F_LAST_ROUND ;

    FILL_D_H(packet, htons((ui16)(s->idx)), htonl(source_id),
	s->id, htons(pkt_ix), htons(blk_ix),
	htonl(s->flength), htons(s->k), s->flags, htons(data_pkt_len));
#ifdef DO_TIMING
    packet.timestamp = htonl(timestamp_u());
#endif
    (s->idx)++;
    thislen = data_pkt_len;
    if (send(s->ch, (char *)&packet,
	PKT_LEN-MAX_DATA_PKT_LEN+thislen, 0) == -1) {
	if (do_err)
	    perror("se_send: send");
	do_err = 0;
	/* XXX */
    } else
	do_err = 1;

    if ( /* (s->flags & RMDP_F_LAST_ROUND) || */ (s->idx % s->B == 0)) {
	DDB(fprintf(stderr, "--- %4d packets left for session 0x%08x \"%s\"\r",
		s->pkts_left, s->id, s->fname) );
    }
    s->pkts_left -- ;
    if (s->pkts_left) {
	if ( ( (s->idx - 1) & 0xc) == 0xc )
		goto send_again;/* every 16 generates a burst of 5 */
	settimer(events, s->id, s->rate, (CALLBACK)se_send, s, 0);
    } else {
	timestamp_t( &(s->flush_at) );
	timestamp_t( &(s->delete_at) );
	s->flush_at.tv_sec += 10; /* XXX */
	s->delete_at.tv_sec += 120; /* XXX */
	DDB(fprintf(stderr, "--- Session %x terminates in 10/120 sec\n",
	    s->id));
	if (s->busy) {
	    DDB(fprintf(stderr, "--- pending delete for Session %x\n", s->id));
	} else {
	    s->busy = 1;
	    settimer(events, s->id, 10*1000000, (CALLBACK)se_delete, s, 1);
	}
    }
    return 0;
}
