/*
 * $Id: minixml.h 348 2008-06-01 17:15:30Z luigi $
 *
 * This library implements a small set of functions to parse, manipulate
 * and print, in functional or XML format, expressions involving
 * scalars, arrays, maps, and function calls.
 *
 * The main purpose is to issue and manipulate XMLRPC calls and results
 * from the command line or a shell.
 * The library uses the 'dynstring' functions to manipulate extensible
 * strings and byte arrays -- dynstrings are very simple to use,
 * think of them as a C string that grows as needed, and you use
 * ds_data(s) to access the actual string data.
 *
 * --- LIBRARY USAGE ---
 *
 * Declare an object with 	struct __obj *o;
 *
 * You can fill it from a string using
 *
 *	expr_parse(char *s, struct __obj **o, fmt)
 *
 * where fmt is FMT_FUNC or FMT_XML to parse text expressions or XMLRPC.
 * You can also create the object programmatically using
 * obj_calloc(...) -- see the .c file for more detail.
 *
 * Once the object is encoded, you can print it using
 *
 *	obj_print(dynstr *dst, const struct __obj *o, fmt);
 *
 * The result is stored into the dynamic string passed as first arg.
 * fmt = FMT_FUNC or FMT_XML selects the print format.
 *
 * An object is destroyed with	obj_free(o);
 *
 * To manipulate objects, e.g. iterate and extract fields, use:
 *
 *    obj_getlen(const struct __obj *) :
 *	    to know the number of elements in the object
 *
 *    obj_get(const struct __obj *, uint32_t i, const char **key) :
 *	    to extract the i-th component (and possibly the key);
 *
 *    obj_getfield(const struct __obj *, const char *key, int keylen) :
 *	    to extract the component with the given key;
 *
 * The auxiliary function
 *    obj_filter(dynstr *s, const struct __obj *o, const char *filter, fmt)
 *
 * supports a simple filtering language to extract, in textual form,
 * components from nested ojects, or list of keys, and so on.
 *
 * Filters are a sequence of character, applied one after the other
 * to the "current object", i.e. the result of previous filtering:
 *
 *	(empty)		prints the object in functional form
 *	.xml()		prints the object in XML
 *	.func()		prints the object in functional form
 *	.length()	returns the number of elements in the object
 *			(1 for a scalar, or number of arguments or
 *			elements for functions, arrays and maps)
 *	.keys()		returns the space-separated list of keys for a map
 *	[]		applies the following filter to all components
 *			of the current object
 *	[k1,k2,...]	applies the following filter to the selected keys
 *			of the current object. Number are considered indexes.
 *			If a key ends with '=', then the object is printed
 *			replacing first-level separators with spaces.
 *
 * Filter examples:
 *   assume the object is	{a=[1,[2,3],4],b=20,c=f(1,2,3)}
 *   ''		returns		{a=[1,[2,3],4],b=20,c=f(1,2,3)}
 *   .length()	returns		3
 *   .keys()	returns		a b c
 *   []		returns		[1,[2,3],4] 20 f(1,2,3)
 *   [0,2]	returns		[1,[2,3],4] f(1,2,3)
 *   [0][1]	returns		[2,3]
 *   [a=]	returns		1 [2,3] 4
 */

#include <sys/types.h>
#include "dynstring.h"

/* avoid using 0 and 1 to catch errors
 * FMT_FUNC is a functional format
 * FMT_FLAT is above but separators at the first level are
 * replaced by blanks
 */
enum expr_fmt {
	FMT_FUNC = 80, FMT_FLAT = 81, FMT_XML= 90
};

enum __xml_types {
	TY_INT=51, TY_DOUBLE, TY_STRING,
	TY_ARRAY, TY_MAP, TY_MAP_ENTRY, TY_FUNCTION,
};

struct __obj;

struct __map_entry {
	struct __obj *value;
	char *key; /* key_len + 1 */
	char key_value[0]; /* if allocated inline */
};

struct __func_data {
	uint32_t	len;	/* number of arguments */
	struct __obj	*arg[0];	// len entries */
	// char name[0];	/* dynamic */
};
struct __map_data {
	uint32_t	len;
	struct __map_entry d[0];
};
struct __array_data {
	uint32_t	len;
	struct __obj *d[0];
};

/*
 * The main object. The type here only serves to access
 * the field, allocation should include enough room for the
 * data elements.
 */
struct __obj {
	enum __xml_types type;
	union {
		int32_t	i[0];
		double	d[0];
		char 	s[0];	/* nul-terminated string */
		struct __func_data f[0];
		struct __map_entry me[0];
		struct __map_data m[0];
		struct __array_data a[0];
	} v;
};

/*
 * expr_parse() parses a functional or xml object
 * obj_print(dst, obj, fmt) prints the object in functional or xml form,
 * obj_calloc() creates a new object of specific type.
 * obj_free() releases the storage for the object
 */
const char *expr_parse(const char *s, struct __obj **, enum expr_fmt);

const char *obj_print(dynstr *dst, const struct __obj *o, enum expr_fmt);
struct __obj *obj_calloc(enum __xml_types, int len, const void *src);
struct __obj *obj_free(struct __obj *);

/*
 * Support functions for navigating in the object.
 * obj_get() returns the i-th element of an array or map,
 *    and also the key value in case of a map.
 * obj_getlen() returns the length of the array or map.
 * obj_getfield() returns the element with the given key for a map.
 */
const struct __obj *obj_get(const struct __obj *, uint32_t i, const char **key);
const struct __obj *obj_getfield(const struct __obj *, const char *key, int keylen);
int obj_getlen(const struct __obj *);
const char *obj_filter(dynstr *s, const struct __obj *o, const char *filter, enum expr_fmt);

