/*
 * SPCA5xx based usb camera driver (currently supports
 * yuv native stream spca501a, spca501c, spca505, spca508, spca506
 * jpeg native stream spca500, spca551, spca504a, spca504b, spca533a, spca536a, zc0301, zc0302, cx11646, sn9c102p
 * bayer native stream spca561a, sn9c101, sn9c102, tv8532 ).
 * Z-star Vimicro chips zc0301 zc0301P zc0302
 * Sunplus spca501a, spca501c, spca505, spca508, spca506, spca500, spca551, spca504a, spca504b, spca533a, spca536a
 * Sonix sn9c101, sn9c102, sn9c102p sn9c105 sn9c120
 * Conexant cx11646
 * Transvision tv_8532 
 * Etoms Et61x151 Et61x251
 * Pixat Pac207-BCA-32
 * SPCA5xx version by Michel Xhaard <mxhaard@users.sourceforge.net>
 * Based on :
 * SPCA50x version by Joel Crisp <cydergoth@users.sourceforge.net>
 * OmniVision OV511 Camera-to-USB Bridge Driver
 * Copyright (c) 1999-2000 Mark W. McClelland
 * Kernel 2.6.x port Michel Xhaard && Reza Jelveh (feb 2004)
 * Based on the Linux CPiA driver written by Peter Pregler,
 * Scott J. Bertin and Johannes Erdfelt.
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(__FreeBSD__)
/*
 * includes for FreeBSD kernel
 */
#define wait_event_interruptible(wq, condition)   1       /* always fail ... ok? */

#include "pwc.h"

#endif /* !__FreeBSD */
#include "spca5xx.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
#  undef CONFIG_VIDEO_PROC_FS
#	 undef CONFIG_PROC_FS
#endif

//#define RH9_REMAP 1

#include "spcadecoder.h"
#include "jpeg_qtables.h"

/* Video Size 640 x 480 x 4 bytes for RGB */
#define MAX_FRAME_SIZE (640 * 480 * 4)
#define MAX_DATA_SIZE (MAX_FRAME_SIZE + sizeof(struct timeval))

#if defined(__FreeBSD__)
static int compress = 0;
#else /* !__FreeBSD__ */
/* Hardware auto exposure / whiteness (PC-CAM 600) */
static int autoexpo = 1;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5)
/* Video device number (-1 is first available) */
static int video_nr = -1;
#endif				/* LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,5) */
#endif	/* !__FreeBSD__ */

/* 0=no debug messages
 * 1=init/detection/unload and other significant messages,
 * 2=some warning messages
 * 3=config/control function calls
 * 4=most function calls and data parsing messages
 * 5=highly repetitive mesgs
 * NOTE: This should be changed to 0, 1, or 2 for production kernels
 */
static int debug = 0;	/* XXX on freebsd, this is overridden by pwcdebug */

/* Force image to be read in RGB instead of BGR. This option allow
 * programs that expect RGB data (e.g. gqcam) to work with this driver. */
static int force_rgb = 0;
static int gamma = 3;
static int OffRed = 0;
static int OffBlue = 0;
static int OffGreen = 0;
static int GRed = 256;
static int GBlue = 256;
static int GGreen = 256;

static int usbgrabber = 0;

#ifdef SPCA50X_ENABLE_COMPRESSION
/* Enable compression. This is for experimentation only; compressed images
 * still cannot be decoded yet. */
static int compress = 0;
#endif				/* SPCA50X_ENABLE_COMPRESSION */

#ifdef SPCA5XX_ENABLE_REGISTERPLAY
static int RegAddress = 0;
static int RegValue = 0;
static int RegStrobe = 0;
#endif				/* SPCA5XX_ENABLE_REGISTERPLAY */

#if !defined(__FreeBSD__)
/* Initial brightness & contrast (for debug purposes) */
static int bright = 0x80;
static int contrast = 0x60;
#endif /* !__FreeBSD__ */

/* Parameter that enables you to set the minimal suitable bpp */
static int min_bpp = 0;

/* Parameter defines the average luminance that should be kept */
static int lum_level = 0x2d;


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9)

module_param(autoexpo, int, 0644);
module_param(debug, int, 0644);
module_param(force_rgb, int, 0644);
module_param(gamma, int, 0644);
module_param(OffRed, int, 0644);
module_param(OffBlue, int, 0644);
module_param(OffGreen, int, 0644);
module_param(GRed, int, 0644);
module_param(GBlue, int, 0644);
module_param(GGreen, int, 0644);

#ifdef SPCA50X_ENABLE_COMPRESSION
module_param(compress, int, 0644);
#endif				/* SPCA50X_ENABLE_COMPRESSION */
module_param(bright, int, 0444);
module_param(contrast, int, 0444);

module_param(min_bpp, int, 0444);
module_param(lum_level, int, 0444);
module_param(usbgrabber, int, 0444);
#ifdef SPCA5XX_ENABLE_REGISTERPLAY
module_param(RegAddress, int, 0644);
module_param(RegValue, int, 0644);
module_param(RegStrobe, int, 0644);
#endif				/* SPCA5XX_ENABLE_REGISTERPLAY */


#else				/* LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9) */


MODULE_PARM(autoexpo, "i");
MODULE_PARM(debug, "i");

MODULE_PARM(force_rgb, "i");
MODULE_PARM(gamma, "i");
MODULE_PARM(OffRed, "i");
MODULE_PARM(OffBlue, "i");
MODULE_PARM(OffGreen, "i");
MODULE_PARM(GRed, "i");
MODULE_PARM(GBlue, "i");
MODULE_PARM(GGreen, "i");

#ifdef SPCA50X_ENABLE_COMPRESSION
MODULE_PARM(compress, "i");
#endif				/* SPCA50X_ENABLE_COMPRESSION */
MODULE_PARM(bright, "i");
MODULE_PARM(contrast, "i");


MODULE_PARM(min_bpp, "i");
MODULE_PARM(lum_level, "0-255i");
MODULE_PARM(usbgrabber, "i");
#endif
/***************/


MODULE_PARM_DESC(autoexpo,
		 "Enable/Disable auto exposure (default=1: enabled) (PC-CAM 600/Zc03xx/spca561a/Etoms Only !!!)");
MODULE_PARM_DESC(debug,
		 "Debug level: 0=none, 1=init/detection, 2=warning, 3=config/control, 4=function call, 5=max");

MODULE_PARM_DESC(force_rgb, "Read RGB instead of BGR");
MODULE_PARM_DESC(gamma, "gamma setting range 0 to 7 3-> gamma=1");
MODULE_PARM_DESC(OffRed, "OffRed setting range -128 to 128");
MODULE_PARM_DESC(OffBlue, "OffBlue setting range -128 to 128");
MODULE_PARM_DESC(OffGreen, "OffGreen setting range -128 to 128");
MODULE_PARM_DESC(GRed, "Gain Red setting range 0 to 512 /256 ");
MODULE_PARM_DESC(GBlue, "Gain Blue setting range 0 to 512 /256 ");
MODULE_PARM_DESC(GGreen, "Gain Green setting range 0 to 512 /256 ");

#ifdef SPCA50X_ENABLE_COMPRESSION
MODULE_PARM_DESC(compress, "Turn on/off compression (not functional yet)");
#endif				/* SPCA50X_ENABLE_COMPRESSION */
MODULE_PARM_DESC(bright,
		 "Initial brightness factor (0-255) not know by all webcams !!");
MODULE_PARM_DESC(contrast,
		 "Initial contrast factor (0-255) not know by all webcams !!");
MODULE_PARM_DESC(min_bpp,
		 "The minimal color depth that may be set (default 0)");
MODULE_PARM_DESC(lum_level,
		 "Luminance level for brightness autoadjustment (default 32)");
MODULE_PARM_DESC(usbgrabber,
		 "Is a usb grabber 0x0733:0x0430 ? (default 1) ");

#ifdef SPCA5XX_ENABLE_REGISTERPLAY
MODULE_PARM_DESC(RegAddress, "Register Address of PAC207");
MODULE_PARM_DESC(RegValue, "Register Value for PAC207");
MODULE_PARM_DESC(RegStrobe,
		 "Strobe to read or write a register 1=write, 2=read");
#endif				/* SPCA5XX_ENABLE_REGISTERPLAY */

/****************/
MODULE_AUTHOR
    ("Michel Xhaard <mxhaard@users.sourceforge.net> based on spca50x driver by Joel Crisp <cydergoth@users.sourceforge.net>,ov511 driver by Mark McClelland <mwm@i.am>");
MODULE_DESCRIPTION("SPCA5XX USB Camera Driver");
MODULE_LICENSE("GPL");


/**********************************************************************
 * List of known SPCA50X-based cameras
 **********************************************************************/

/* Camera type jpeg yuvy yyuv yuyv grey gbrg*/
struct palette_list Plist[] = {
    {JPEG, "JPEG"},
    {JPGH, "JPEG"},
    {JPGC, "JPEG"},
    {JPGS, "JPEG"},
    {JPGM, "JPEG"},
    {YUVY, "YUVY"},
    {YYUV, "YYUV"},
    {YUYV, "YUYV"},
    {GREY, "GREY"},
    {GBRG, "GBRG"},
    {SN9C, "SN9C"},
    {GBGR, "GBGR"},
    {S561, "S561"},
    {PGBRG, "GBRG"},	/* XXX this is actually bayer offset by 1 */
    {-1, NULL}
};

/*
 * Let's include the initialization data for each camera type
 */
#include "spcausb.h"
#include "spca500_init.h"
#include "spca501_init.h"
#include "spca505_init.h"
#include "spca506.h"
#include "spca508_init.h"
#include "spca561.h"
#include "sp5xxfw2.h"
#include "sonix.h"
#include "zc3xx.h"
#include "cx11646.h"
#include "tv8532.h"
#include "et61xx51.h"
#include "mr97311.h"
#include "pac207.h"

/* Mode list for spca50x on external input */
/* Must be in descending order as the closest match which is equal or smaller than
 * the requested size is returned */
#if 0
/* Trashcan before goes out */
(bridge) == BRIDGE_SONIX ? sonix_ext_modes :
    (bridge) == BRIDGE_ZC3XX ? zc3xx_ext_modes :
    (bridge) == BRIDGE_ETOMS ? etoms_ext_modes :
    (bridge) == BRIDGE_SPCA561 ? spca561_ext_modes :
    (bridge) == BRIDGE_SN9CXXX ? sn9c102p_ext_modes :
    (bridge) == BRIDGE_SPCA504 ? spca504_ext_modes :
    (bridge) == BRIDGE_SPCA504B ? spca504_ext_modes :
    (bridge) == BRIDGE_SPCA504C ? spca504_pccam600_ext_modes :
    (bridge) == BRIDGE_SPCA533 ? spca533_ext_modes :
    (bridge) == BRIDGE_SPCA536 ? spca536_ext_modes :
    static __u16 pcam_ext_modes[][6] = {
    /* x , y , Code, Value (6), Value (7), pipe */
    {1280, 1024, 0x00, 0, 0, 1023},
    {640, 480, 0x01, 0, 0, 1023},
    {384, 288, 0x11, 16, 12, 1023},
    {352, 288, 0x11, 18, 12, 1023},
    {320, 240, 0x02, 0, 0, 896},
    {192, 144, 0x12, 8, 6, 896},
    {176, 144, 0x12, 9, 6, 896},
    {0, 0, 0, 0, 0}
};
static __u16 zc3xx_ext_modes[][6] = {
    /* x , y , Code, x multiplier, y multiplier, pipe */
    {640, 480, 0x00, 0, 0, 1023},	// altersetting 7 endpoint 0x01
    {352, 288, 0x10, 0, 0, 1023},	//0x00 SIF sensor
    {320, 240, 0x01, 0, 0, 1023},
    {176, 144, 0x11, 0, 0, 1023},	//0x01
    {0, 0, 0, 0, 0, 0}
};
static __u16 sonix_ext_modes[][6] = {
    /* x , y , Code, x multiplier, y multiplier, pipe */
    {640, 480, 0x00, 0, 0, 1023},
    {352, 288, 0x00, 0, 0, 1023},
    {320, 240, 0x01, 0, 0, 1023},
    {176, 144, 0x01, 0, 0, 1023},
    {160, 120, 0x02, 0, 0, 1023},
    {0, 0, 0, 0, 0, 0}
};
static __u16 etoms_ext_modes[][6] = {
    /* x , y , Code, x, y, pipe */
    {352, 288, 0x00, 0, 0, 1000},
    {320, 240, 0x01, 0, 0, 1000},
    {176, 144, 0x01, 0, 0, 1000},
    {0, 0, 0, 0, 0, 0}
};
static __u16 spca561_ext_modes[][6] = {
    {352, 288, 0x00, 0x27, 0x00, 1023},
    {320, 240, 0x01, 0x27, 0x00, 1023},
    {176, 144, 0x02, 0x23, 0x00, 1023},	// software mode hardware seem buggy slow shift in video
    {160, 120, 0x03, 0x23, 0x00, 1023},
    {0, 0, 0, 0, 0}
};
static __u16 sn9c102p_ext_modes[][6] = {
    /* x , y , Code, x, y, pipe */
    {640, 480, 0x00, 0, 0, 1023},
    {352, 288, 0x10, 0, 0, 1023},
    {320, 240, 0x01, 0, 0, 1023},
    {176, 144, 0x11, 0, 0, 1023},
    {0, 0, 0, 0, 0, 0}
    static __u16 spca533_ext_modes[][6] = {
	/* x , y , Code, Value (6), Value (7), pipe */
	//{ 640, 480, 0x41, 1, 0, 1023 },
	{464, 480, 0x01, 0, 0, 1023},	//PocketDVII unscaled resolution aspect ratio need to expand x axis
	{464, 352, 0x01, 0, 0, 1023},	//Gsmart LCD3 feature good aspect ratio 
	{384, 288, 0x11, 5, 4, 1023},
	{352, 288, 0x11, 7, 4, 1023},
	{320, 240, 0x02, 0, 0, 1023},
	{192, 144, 0x12, 8, 6, 1023},
	{176, 144, 0x12, 9, 6, 1023},
	//{ 160, 120, 0x22, 1, 1, 1023 },
	//{ 320, 128, 0x03, 0, 0, 1023 }, /* don't work with megapix V4*/
	{0, 0, 0, 0, 0}
    };

    static __u16 spca536_ext_modes[][6] = {
	/* x , y , Code, Value (6), Value (7), pipe */
	{640, 480, 0x01, 1, 0, 1023},
	//{ 464, 480, 0x01, 0, 0, 1023 },
	//{ 464, 352, 0x01, 0, 0, 1023 },
	{384, 288, 0x11, 5, 4, 1023},
	{352, 288, 0x11, 7, 4, 1023},
	{320, 240, 0x02, 0, 0, 896},
	{192, 144, 0x12, 8, 6, 896},
	{176, 144, 0x12, 9, 6, 896},
	{0, 0, 0, 0, 0}
    };

    static __u16 spca504_ext_modes[][6] = {
	/* x , y , Code, Value (6), Value (7), pipe */
	{640, 480, 0x01, 0, 0, 1023},
	{384, 288, 0x11, 16, 12, 1023},
	{352, 288, 0x11, 18, 12, 1023},
	{320, 240, 0x02, 0, 0, 896},
	{192, 144, 0x12, 8, 6, 896},
	{176, 144, 0x12, 9, 6, 896},
	//{ 160, 120, 0x22, 1, 1, 896 },
	{0, 0, 0, 0, 0}
    };


    static __u16 spca504_pccam600_ext_modes[][6] = {
	/* x , y , Code, Value (6), Value (7), pipe */
	{1024, 768, 0x00, 0, 0, 1023},
	{640, 480, 0x01, 1, 0, 1023},
	{352, 288, 0x02, 2, 0, 896},
	{320, 240, 0x03, 3, 0, 896},
	{176, 144, 0x04, 4, 0, 768},
/*	{ 160,  120, 0x05, 5, 0, 768 },  */
	{0, 0, 0, 0, 0}
    };

};

#endif
#define GET_EXT_MODES(bridge) (\
	(bridge) == BRIDGE_SPCA500 ? spca500_ext_modes : \
	(bridge) == BRIDGE_SPCA501 ? spca501_ext_modes : \
	(bridge) == BRIDGE_SPCA506 ? spca506_ext_modes : \
	(bridge) == BRIDGE_SPCA508 ? spca508_ext_modes : \
	(bridge) == BRIDGE_CX11646 ? cx11646_ext_modes : \
	(bridge) == BRIDGE_TV8532 ? tv8532_ext_modes : \
	spca50x_ext_modes)

/* code is used for harware low nibble and software hight nibble */



static __u16 cx11646_ext_modes[][6] = {
    /* x , y , Code, x multiplier, y multiplier, pipe */
    {640, 480, 0x00, 0, 0, 1023},	// altersetting 7 endpoint 0x01
    {352, 288, 0x01, 0, 0, 1023},
    {320, 240, 0x02, 0, 0, 1023},
    {176, 144, 0x03, 0, 0, 640},	// alt 4
    // {160, 120, 0x04, 0, 0, 512}, // alt 3 only works with external decoding
    {0, 0, 0, 0, 0, 0}
};

static __u16 spca500_ext_modes[][6] = {
    /* x , y , Code, x multiplier, y multiplier, pipe */
    {640, 480, 0x00, 40, 30, 1023},
    {352, 288, 0x00, 22, 18, 1023},
    {320, 240, 0x01, 40, 30, 1023},
    {176, 144, 0x01, 22, 18, 1023},
    /* 160x120 disable jpeg %16 */
    {0, 0, 0, 0, 0, 0}
};

static __u16 spca501_ext_modes[][6] = {
    /* x , y , Code, Value (6), Value (7), pipe */
    {640, 480, 0, 0x94, 0x004A, 1023},
    {352, 288, 0x10, 0x94, 0x004A, 1023},
    {320, 240, 0, 0x94, 0x104A, 896},
    {176, 144, 0x10, 0x94, 0x104A, 896},
    {160, 120, 0, 0x94, 0x204A, 384},
    {0, 0, 0, 0, 0}
};




/* default value used for spca505 and spca505b */
static __u16 spca50x_ext_modes[][6] = {
    /* x , y , Code, Value (6), Value (7), pipe */
    {640, 480, 0, 0x10, 0x10, 1023},	/* Tested spca505b+VGA sensor */
    {352, 288, 1, 0x1a, 0x1a, 1023},	/* Tested */
    {320, 240, 2, 0x1c, 0x1d, 896},	/* Tested 20:01:2004 */
    {176, 144, 4, 0x34, 0x34, 512},	/* Tested */
    {160, 120, 5, 0x40, 0x40, 384},	/* Tested */
    {0, 0, 0, 0, 0}
};
static __u16 spca506_ext_modes[][6] = {
    /* In this table, element 3 (clk) controls the
     * clock, and gets written to 0x8700.
     */
    /* x , y , Code, clk, n/a,  pipe */
    {640, 480, 0x00, 0x10, 0x10, 1023},
    {352, 288, 0x01, 0x1a, 0x1a, 1023},
    {320, 240, 0x02, 0x1c, 0x1c, 1023},	//896
    {176, 144, 0x04, 0x34, 0x34, 1023},
    {160, 120, 0x05, 0x40, 0x40, 1023},
    {0, 0, 0, 0, 0}
};

//  1a 1b 20 
static __u16 spca508_ext_modes[][6] = {
    /* In this table, element 3 (clk) controls the
     * clock, and gets written to 0x8700.
     */
    /* x , y , Code, clk, n/a,  pipe */
    {352, 288, 0x00, 0x28, 0x00, 1023},
    {320, 240, 0x01, 0x28, 0x00, 1023},
    {176, 144, 0x02, 0x23, 0x00, 1023},
    {160, 120, 0x03, 0x23, 0x00, 1023},
    {0, 0, 0, 0, 0}
};

static __u16 tv8532_ext_modes[][6] = {
    /* x , y , Code, clk, n/a,  pipe */
    {352, 288, 0x00, 0x28, 0x00, 1023},
    {320, 240, 0x10, 0x28, 0x00, 1023},
    {176, 144, 0x01, 0x23, 0x00, 1023},
    /*{160, 120, 0x03, 0x23, 0x00, 1023}, */
    {0, 0, 0, 0, 0}
};

#ifdef CONFIG_PROC_FS
/* Not sure what we should do with this. I think it is V4L level 2 stuff */
/* Currently only use RGB24 */
static struct palette_list plist[] = {
    {VIDEO_PALETTE_GREY, "GREY"},
    {VIDEO_PALETTE_HI240, "HI240"},
    {VIDEO_PALETTE_RGB565, "RGB565"},
    {VIDEO_PALETTE_RGB24, "RGB24"},
    {VIDEO_PALETTE_RGB32, "RGB32"},
    {VIDEO_PALETTE_RGB555, "RGB555"},
    {VIDEO_PALETTE_YUV422, "YUV422"},
    {VIDEO_PALETTE_YUYV, "YUYV"},
    {VIDEO_PALETTE_UYVY, "UYVY"},
    {VIDEO_PALETTE_YUV420, "YUV420"},
    {VIDEO_PALETTE_YUV411, "YUV411"},
    {VIDEO_PALETTE_RAW, "RAW"},
    {VIDEO_PALETTE_YUV422P, "YUV422P"},
    {VIDEO_PALETTE_YUV411P, "YUV411P"},
    {VIDEO_PALETTE_YUV420P, "YUV420P"},
    {VIDEO_PALETTE_YUV410P, "YUV410P"},
    {VIDEO_PALETTE_RAW_JPEG, "RJPG"},
    {VIDEO_PALETTE_JPEG, "JPEG"},
    {-1, NULL}
};
#endif				/* CONFIG_PROC_FS */

/* function for the tasklet */

void outpict_do_tasklet(unsigned long ptr);

int spca50x_set_packet_size(struct usb_spca50x *spca50x, int size)
{
    int alt;
    struct pwc_softc *sc = PWC_SC(spca50x);
    int ep = 0;

	/**********************************************************************/

    /* Select the alternate setting for the interface depending
     * on the desired transfer length
     */
    if (size == 0)			/* this is always like this */
	alt = SPCA50X_ALT_SIZE_0;
    else if (size == 128)
	alt = SPCA50X_ALT_SIZE_128;
    else if (size == 256)
	alt = SPCA50X_ALT_SIZE_256;
    else if (size == 384)
	alt = SPCA50X_ALT_SIZE_384;
    else if (size == 512)
	alt = SPCA50X_ALT_SIZE_512;
    else if (size == 640)
	alt = SPCA50X_ALT_SIZE_640;
    else if (size == 768)
	alt = SPCA50X_ALT_SIZE_768;
    else if (size == 896)
	alt = SPCA50X_ALT_SIZE_896;
    else if (size == 1000)
	alt = ETOMS_ALT_SIZE_1000;
    else if (size >= 1023) {	/* XXX was == */
	switch (PWC_SC(spca50x)->pwc_info.bridge) {
	case BRIDGE_SONIX:
	case BRIDGE_SN9CXXX:
	case BRIDGE_MR97311:
	case BRIDGE_PAC207:
	    alt = 8;
	    break;
	case BRIDGE_STV0602:
	    alt = 1;	/* other modes don't work */
	    break;
	case BRIDGE_EM2820:
	    alt = 5;	/* 5:... 7:3072 bytes */
	    break;
	default:
	    alt = SPCA50X_ALT_SIZE_1023;
	    break;
	}
    } else {
	/* if an unrecognised size, default to the minimum */
	printf( "Set packet size: invalid size (%d), defaulting to %d\n",
	       size, SPCA50X_ALT_SIZE_128);
	alt = SPCA50X_ALT_SIZE_128;
    }
    printf("%s: Set packet size: size (%d), alt becomes %d\n", __FUNCTION__, size, alt);


    /* check endpoint */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_EM2820:	/* has 1, 2 and 4 */
	ep = 2;
	break;
    case BRIDGE_PAC207:
	ep = PAC207_ENDPOINT_ADDRESS;
	break;
    default:
	ep = SPCA50X_ENDPOINT_ADDRESS;
	break;
    }

    spca50x->vendpoint = sc->vendpoint = ep;
    spca50x->alt = alt;

#if 0
    PDEBUG(5, "iface %d alt %d ep %d size %d", spca50x->iface, alt, ep, size);
    if (usbd_set_interface(sc->sc_iface, alt) != USBD_NORMAL_COMPLETION) {
	err("Set packet size: set interface error");
	return -EBUSY;
    }

    {
	int ret;
	int i;
	uint8_t nendpt;
	usb_endpoint_descriptor_t *found = NULL, *edesc = NULL;
	int mysize = 0;

        ret = usbd_endpoint_count(sc->sc_iface, &nendpt);
        if(ret != USBD_NORMAL_COMPLETION) {
                device_printf(sc->sc_dev, "Failed to get endpoint count (%d)\n", ret);
                return -ret;
        }
        for (i = 0; i < nendpt; i++) {
                edesc = usbd_interface2endpoint_descriptor(sc->sc_iface, i);
                if (edesc)
                        printf("desc %d endpoint %d size %d\n",
				i, UE_GET_ADDR(edesc->bEndpointAddress),
				UGETW(edesc->wMaxPacketSize));
                if(edesc != NULL && UE_GET_ADDR(edesc->bEndpointAddress) == sc->vendpoint) {
                        found = edesc;
		}
        }
        if(found == NULL) {
                device_printf(sc->sc_dev, "Failed to find videoendpoint %d\n", sc->vendpoint);
                return -EINVAL;
        }
        edesc = found;
	mysize = UGETW(edesc->wMaxPacketSize);
	if (mysize > 1023)
		mysize = 1023;
	PDEBUG(1, "set real packet size: %d, alt=%d", mysize, alt);
    }
#endif
    return 0;
}

/* Returns number of bits per pixel (regardless of where they are located; planar or
 * not), or zero for unsupported format.
 */
static int spca5xx_get_depth(struct usb_spca50x *spca50x, int palette)
{
    switch (palette) {
//      case VIDEO_PALETTE_GREY:     return 8;
    case VIDEO_PALETTE_RGB565:
	return 16;
    case VIDEO_PALETTE_RGB24:
	return 24;

//      case VIDEO_PALETTE_YUV422:   return 16;
 //   case VIDEO_PALETTE_YUYV:
//	return 16;
//      case VIDEO_PALETTE_YUV420:   return 24;
    case VIDEO_PALETTE_YUV420P:
	return 12;		/* strange need 12 this break the read method for this planar mode (6*8/4) */
//      case VIDEO_PALETTE_YUV422P:  return 24; /* Planar */

    case VIDEO_PALETTE_RGB32:
	return 32;
    case VIDEO_PALETTE_RAW_JPEG:
	return 24;		/* raw jpeg. what should we return ?? */
    case VIDEO_PALETTE_JPEG:
	if (spca50x->cameratype == JPEG ||
	    spca50x->cameratype == JPGH ||
	    spca50x->cameratype == JPGC ||
	    spca50x->cameratype == JPGS || 
	    spca50x->cameratype == JPGM) {
	    return 8;
	} else
	    return 0;
    default:
	printf("%s: invalid format %d\n", __FUNCTION__, palette);
	return 0;		/* Invalid format */
    }
}

/**********************************************************************
* spca50x_init_isoc
* Function starts the ISO USB transfer by enabling this process
* from USB side and enabling ISO machine from the chip side
***********************************************************************/
void spcaCameraStart(struct usb_spca50x *spca50x)
{
    int err_code;
    struct usb_device *dev = spca50x->dev;
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    default:
	printf("%s: unknown bridge %d (non fatal but should)\n", __FUNCTION__, PWC_SC(spca50x)->pwc_info.bridge);
	break;

    case BRIDGE_SN9CXXX:
	sn9cxxx_start(spca50x);
	break;

    case BRIDGE_ETOMS:
	Et_startCamera(spca50x);
	break;

    case BRIDGE_CX11646:
	cx11646_start(spca50x);
	break;

    case BRIDGE_ZC3XX:
	zc3xx_start(spca50x);
	break;

    case BRIDGE_SONIX:
	sonix_start(spca50x);
	break;

    case BRIDGE_SPCA500:
	err_code = spca500_initialise(spca50x);
	break;

    case BRIDGE_SPCA501:
	/* Enable ISO packet machine CTRL reg=2,
	 * index=1 bitmask=0x2 (bit ordinal 1)
	 */
	err_code = spca50x_reg_write(dev, SPCA501_REG_CTLRL, 0x1, 0x2);
	break;

    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504B:
	sp5xxfw2_start(spca50x);
	break;

    case BRIDGE_SPCA506:
	spca506_start(spca50x);
	break;

    case BRIDGE_SPCA505:
	//necessary because without it we can see stream only once after loading module
	//stopping usb registers Tomasz change
	spca50x_reg_write(dev, 0x2, 0x0, 0x0);

	/* Enable ISO packet machine - should we do this here or in ISOC init ? */
	err_code = spca50x_reg_write(dev, SPCA50X_REG_USB, SPCA50X_USB_CTRL, SPCA50X_CUSB_ENABLE);
//                      spca50x_reg_write(dev, 0x5, 0x0, 0x0);
//                      spca50x_reg_write(dev, 0x5, 0x0, 0x1);
//                      spca50x_reg_write(dev, 0x5, 0x11, 0x2);
	break;

    case BRIDGE_SPCA508:
	err_code = spca50x_reg_write(dev, 0, 0x8112, 0x10 | 0x20);
	break;

    case BRIDGE_SPCA561:
	/* Video ISO enable, Video Drop Packet enable: */
	spca561_start(spca50x);
	break;

    case BRIDGE_TV8532:
	tv8532_start(spca50x);
	break;

    case BRIDGE_MR97311:
	pcam_start(spca50x);
	break;

    case BRIDGE_PAC207:
	pac207_start(spca50x);
	break;
    }

}
static inline void spcaCameraStop2(struct usb_spca50x *spca50x)
{
    /* stop on alt 0 */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_CX11646:
	cx11646_stop(spca50x);
	break;

    case BRIDGE_ZC3XX:
	zc3xx_stop(spca50x);
	break;
    }
}

static inline void spcaCameraStop(struct usb_spca50x *spca50x)
{
    /* stop on alt x */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_SN9CXXX:
	sn9cxxx_stop(spca50x);
	break;

    case BRIDGE_ETOMS:
	Et_stopCamera(spca50x);
	break;

    case BRIDGE_CX11646:
	/* deferred after set_alt(0,0) 
	   cx11646_stop(spca50x); 
	 */
	break;

    case BRIDGE_ZC3XX:
	//zc3xx_stop(spca50x);
	break;

    case BRIDGE_SONIX:
	sonix_stop(spca50x);
	break;

    case BRIDGE_SPCA500:
	break;

    case BRIDGE_SPCA501:
	/* Disable ISO packet machine CTRL reg=2, index=1 bitmask=0x0 (bit ordinal 1) */
	spca50x_reg_write(spca50x->dev, SPCA501_REG_CTLRL, 0x1, 0x0);
	break;

    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504B:
	sp5xxfw2_stop(spca50x);
	break;

    case BRIDGE_SPCA505:
	spca50x_reg_write(spca50x->dev, 0x2, 0x0, 0x0);	//Disable ISO packet machine
	break;

    case BRIDGE_SPCA506:
	spca506_stop(spca50x);
	break;

    case BRIDGE_SPCA508:
	// Video ISO disable, Video Drop Packet enable:
	spca50x_reg_write(spca50x->dev, 0, 0x8112, 0x20);
	break;

    case BRIDGE_SPCA561:
	// Video ISO disable, Video Drop Packet enable:
	spca561_stop(spca50x);
	break;

    case BRIDGE_TV8532:
	tv8532_stop(spca50x);
	break;

    case BRIDGE_MR97311:
	pcam_stop(spca50x);
	break;

    case BRIDGE_PAC207:
	pac207_stop(spca50x);
	break;
    }
}

static int spca50x_init_isoc(struct usb_spca50x *spca50x)
{
#if !defined(__FreeBSD__)
    struct urb *urb;
    int fx, err, n;
    int intpipe;
    PDEBUG(3, "*** Initializing capture ***");
/* reset iso context */
    spca50x->compress = compress;
    spca50x->curframe = 0;
    spca50x->cursbuf = 0;
    spca50x->frame[0].seq = -1;
    spca50x->lastFrameRead = -1;

//    spca50x_set_packet_size(spca50x, spca50x->pipe_size);
#endif
    PDEBUG(2, "setpacketsize %d", spca50x->pipe_size);

#if defined(__FreeBSD__)
    spcaCameraStart(spca50x);
#else /* !__FreeBSD__ */
    for (n = 0; n < SPCA50X_NUMSBUF; n++) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0)
	urb = usb_alloc_urb(FRAMES_PER_DESC, GFP_KERNEL);
#else
	urb = usb_alloc_urb(FRAMES_PER_DESC);
#endif
	if (!urb) {
	    err("init isoc: usb_alloc_urb ret. NULL");
	    return -ENOMEM;
	}
	spca50x->sbuf[n].urb = urb;
	urb->dev = spca50x->dev;
	urb->context = spca50x;
	if (spca50x->bridge == BRIDGE_PAC207) {
	    urb->pipe =
		usb_rcvisocpipe(spca50x->dev, PAC207_ENDPOINT_ADDRESS);
	} else {
	    urb->pipe =
		usb_rcvisocpipe(spca50x->dev, SPCA50X_ENDPOINT_ADDRESS);
	}
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
	urb->transfer_flags = URB_ISO_ASAP;
	urb->interval = 1;
#else
	urb->transfer_flags = USB_ISO_ASAP;
#endif
	urb->transfer_buffer = spca50x->sbuf[n].data;
	urb->complete = spca50x_isoc_irq;
	urb->number_of_packets = FRAMES_PER_DESC;
	urb->transfer_buffer_length =
	    spca50x->packet_size * FRAMES_PER_DESC;
	for (fx = 0; fx < FRAMES_PER_DESC; fx++) {
	    urb->iso_frame_desc[fx].offset = spca50x->packet_size * fx;
	    urb->iso_frame_desc[fx].length = spca50x->packet_size;
	}
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,41)
    spca50x->sbuf[SPCA50X_NUMSBUF - 1].urb->next = spca50x->sbuf[0].urb;
    for (n = 0; n < SPCA50X_NUMSBUF - 1; n++)
	spca50x->sbuf[n].urb->next = spca50x->sbuf[n + 1].urb;
#endif
    spcaCameraStart(spca50x);
    if (spca50x->bridge == BRIDGE_SONIX) {
	intpipe = usb_rcvintpipe(spca50x->dev, 3);
	usb_clear_halt(spca50x->dev, intpipe);
    }
    PDEBUG(5, "init isoc int %d altsetting %d", spca50x->iface,
	   spca50x->alt);
    for (n = 0; n < SPCA50X_NUMSBUF; n++) {
	spca50x->sbuf[n].urb->dev = spca50x->dev;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,41)
	err = usb_submit_urb(spca50x->sbuf[n].urb, GFP_KERNEL);
#else
	err = usb_submit_urb(spca50x->sbuf[n].urb);
#endif
	if (err) {
	    err("init isoc: usb_submit_urb(%d) ret %d", n, err);
	    return err;
	}
    }

    for (n = 0; n < SPCA50X_NUMFRAMES; n++) {
	spca50x->frame[n].grabstate = FRAME_UNUSED;
	spca50x->frame[n].scanstate = STATE_SCANNING;
    }
#endif /* !__FreeBSD__ */

    spca50x->streaming = 1;

    return 0;

}


/**********************************************************************
* spca50x_stop_isoc
* Function stops the USB ISO pipe by stopping the chip ISO machine
* and stopping USB transfer
***********************************************************************/

void spca50x_stop_isoc(struct usb_spca50x *spca50x)
{

    if (!spca50x->streaming || !spca50x->dev)
	return;

    PDEBUG(3, "*** Stopping capture ***");
    spcaCameraStop(spca50x);
    // spca50x->streaming = 0;	/* XXX kill isoc */
    spca50x_set_packet_size(spca50x, 0);
    spcaCameraStop2(spca50x);

    PDEBUG(3, "*** Capture stopped ***");
}

/**********************************************************************
* spca50x_find_mode_index
* Function finds the mode index in the modes table
***********************************************************************/
static inline int
spca50x_find_mode_index(int width, int height, __u16 ext_modes[][6])
{
    int i = 0;
    int j = 0;
    /* search through the mode list until we hit one which has
     * a smaller or equal resolution, or we run out of modes.
     */
    for (i = 0; ext_modes[i][0] &&
	 (ext_modes[i][0] >= width && ext_modes[i][1] >= height); i++) {
	if (((ext_modes[i][2] & 0xF0) >> 4) == 0)
	    j = (i & 0x0F) << 4;
    }
    /* search to found the first hardware available mode */
    i--;			/* FIXME */
    PDEBUG(2, "find Hardware mode %d soft mode %d", j >> 4, i);
    /* check whether we found the requested mode, or ran out of
     * modes in the list.
     */
    if (!ext_modes[i][0] ||
	ext_modes[i][0] != width || ext_modes[i][1] != height) {
	PDEBUG(3, "Failed to find mode %d x %d", width, height);
	return -EINVAL;
    }
    i = (i & 0x0F) | j;
    PDEBUG(2, "Pass Hardware mode %d soft mode %d", (i >> 4), (i & 0x0F));
    /* return the index of the requested mode */
    return i;
}

/**********************************************************************
* spca50x_smallest_mode_index
* Function finds the mode index in the modes table of the smallest
* available mode.
***********************************************************************/
static int spca5xx_getDefaultMode(struct usb_spca50x *spca50x)
{
    int i;
    for (i = QCIF; i < TOTMODE; i++) {
	if (spca50x->mode_cam[i].method == 0 && spca50x->mode_cam[i].width) {
	    spca50x->width = spca50x->mode_cam[i].width;
	    spca50x->height = spca50x->mode_cam[i].height;
	    spca50x->method = 0;
	    spca50x->pipe_size = spca50x->mode_cam[i].pipe;
	    spca50x->mode = spca50x->mode_cam[i].mode;
	    return 0;
	}
    }
    return -EINVAL;
}

static inline int
spca50x_smallest_mode_index(__u16 ext_modes[][6],
			    int *cols, int *rows, int *pipe)
{
    int i;
    int width = INT_MAX;
    int height = INT_MAX;
    int intpipe = 1023;
    int index = -EINVAL;

    /* search through the mode list until we run out of modes */
    /* and take the smallest hardware mode and pipe size */
    for (i = 0; ext_modes[i][0]; i++) {
	if (ext_modes[i][0] < width || ext_modes[i][1] < height) {
	    if (!(ext_modes[i][2] & 0xF0)) {
		width = ext_modes[i][0];
		height = ext_modes[i][1];
		intpipe = ext_modes[i][5];
		index = i;
	    }
	}
    }

    /* return the index of the smallest mode */
    if (index != -EINVAL) {
	*pipe = intpipe;
	if (cols != NULL)
	    *cols = width;
	if (rows != NULL)
	    *rows = height;
    }
    return i;
}

/**********************************************************************
* spca50x_set_mode
* Function sets up the resolution directly. 
* Attention!!! index, the index in modes array is NOT checked. 
***********************************************************************/
static int spca5xx_getcapability(struct usb_spca50x *spca50x)
{

    int maxw, maxh, minw, minh;
    int i;
    minw = minh = 255 * 255;
    maxw = maxh = 0;
    for (i = QCIF; i < TOTMODE; i++) {
	if (spca50x->mode_cam[i].width) {
	    if (maxw < spca50x->mode_cam[i].width
		|| maxh < spca50x->mode_cam[i].height) {
		maxw = spca50x->mode_cam[i].width;
		maxh = spca50x->mode_cam[i].height;
	    }
	    if (minw > spca50x->mode_cam[i].width
		|| minh > spca50x->mode_cam[i].height) {
		minw = spca50x->mode_cam[i].width;
		minh = spca50x->mode_cam[i].height;
	    }
	}
    }
    spca50x->maxwidth = maxw;
    spca50x->maxheight = maxh;
    spca50x->minwidth = minw;
    spca50x->minheight = minh;
    PDEBUG(0, "maxw %d maxh %d minw %d minh %d", maxw, maxh, minw, minh);
    return 0;
}
static int wxh_to_size(int width, int height)
{
	switch (width) {
	case 640:
	if (height == 480)return VGA;	
	break;
	case 384:
	if (height == 288)return PAL;
	break;
	case 352:
	if (height == 288)return CIF;
	break;
	case 320:
	if (height == 240)return SIF;
	break;
	case 192:
	if (height == 144)return QPAL;
	break;
	case 176:
	if (height == 144)return QCIF;
	break;
	case 160:
	if (height == 120)return QSIF;
	break;
	}
return -EINVAL;
}
static int v4l_to_spca5xx(int format)
{
    switch (format) {
    case VIDEO_PALETTE_RGB565:
	return P_RGB16;
    case VIDEO_PALETTE_RGB24:
	return P_RGB24;
    case VIDEO_PALETTE_RGB32:
	return P_RGB32;
    case VIDEO_PALETTE_YUV422P:
	return P_YUV422;
    case VIDEO_PALETTE_YUV420P:
	return P_YUV420;
    case VIDEO_PALETTE_RAW_JPEG:
	return P_RAW;
    case VIDEO_PALETTE_JPEG:
	return P_JPEG;
    default:
	printf("%s: invalid format %d\n", __FUNCTION__, format);
	return -EINVAL;
    }
}
int spca5xx_setMode(struct usb_spca50x *spca50x, int width,
				  int height, int format)
{
    int i, j;
    int formatIn;
    int crop = 0, cropx1 = 0, cropx2 = 0, cropy1 = 0, cropy2 = 0, x =
	0, y = 0;
    /* Avoid changing to already selected mode */
    /* convert V4l format to our internal format */
    PDEBUG(2, "Set mode asked w %d h %d p %d", width, height, format);

    if ((formatIn = v4l_to_spca5xx(format)) < 0)
	return -EINVAL;
printf("Looking for %d x %d palette 0x%x\n", width, height, formatIn);
    for (i = QCIF; i < TOTMODE; i++) {
	if ((spca50x->mode_cam[i].width == width) &&
	    (spca50x->mode_cam[i].height == height) &&
	    (spca50x->mode_cam[i].t_palette & formatIn)) {
	    spca50x->width = spca50x->mode_cam[i].width;
	    spca50x->height = spca50x->mode_cam[i].height;
	    spca50x->pipe_size = spca50x->mode_cam[i].pipe;
	    spca50x->mode = spca50x->mode_cam[i].mode;
	    spca50x->method = spca50x->mode_cam[i].method;
	    spca50x->format = format;	//palette in use
	    if (spca50x->method) {
		if (spca50x->method == 1) {
		    for (j = i; j < TOTMODE; j++) {
			if (spca50x->mode_cam[j].method == 0
			    && spca50x->mode_cam[j].width) {
			    spca50x->hdrwidth = spca50x->mode_cam[j].width;
			    spca50x->hdrheight =
				spca50x->mode_cam[j].height;
			    spca50x->mode = spca50x->mode_cam[j].mode;	// overwrite by the hardware mode
			    break;
			}
		    }		// end match hardware mode
		    if (!spca50x->hdrwidth && !spca50x->hdrheight)
			return -EINVAL;
		}
	    }
	    /* match found */
	    break;
	}
    }				// end match mode
    /* initialize the hdrwidth and hdrheight for the first init_source */
    /* precompute the crop x y value for each frame */
    if (!spca50x->method) {
	/* nothing todo hardware found stream */
	cropx1 = cropx2 = cropy1 = cropy2 = x = y = 0;
	spca50x->hdrwidth = spca50x->width;
	spca50x->hdrheight = spca50x->height;
    }
    if (spca50x->method & 0x01) {
	/* cropping method */
	if (spca50x->hdrwidth > spca50x->width) {

	    crop = (spca50x->hdrwidth - spca50x->width);
	    if (spca50x->cameratype == JPEG || spca50x->cameratype == JPGH
		|| spca50x->cameratype == JPGS)
		crop = crop >> 4;
	    cropx1 = crop >> 1;
	    cropx2 = cropx1 + (crop % 2);
	} else {
	    cropx1 = cropx2 = 0;
	}
	if (spca50x->hdrheight > spca50x->height) {
	    crop = (spca50x->hdrheight - spca50x->height);
	    if (spca50x->cameratype == JPEG)
		crop = crop >> 4;
	    if (spca50x->cameratype == JPGH || spca50x->cameratype == JPGS)
		crop = crop >> 3;
	    cropy1 = crop >> 1;
	    cropy2 = cropy1 + (crop % 2);
	} else {
	    cropy1 = cropy2 = 0;
	}
    }
    if (spca50x->method & 0x02) {
	/* what can put here for div method */
    }
    if (spca50x->method & 0x04) {
	/* and here for mult */
    }
    PDEBUG(2, "Found code %d method %d", spca50x->mode, spca50x->method);
    PDEBUG(2, "Soft Win width height %d x %d", spca50x->width,
	   spca50x->height);
    PDEBUG(2, "Hard Win width height %d x %d", spca50x->hdrwidth,
	   spca50x->hdrheight);
#if !defined(__FreeBSD__)
    for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
	spca50x->frame[i].method = spca50x->method;
	spca50x->frame[i].cameratype = spca50x->cameratype;
	spca50x->frame[i].cropx1 = cropx1;
	spca50x->frame[i].cropx2 = cropx2;
	spca50x->frame[i].cropy1 = cropy1;
	spca50x->frame[i].cropy2 = cropy2;
	spca50x->frame[i].x = x;
	spca50x->frame[i].y = y;
	spca50x->frame[i].hdrwidth = spca50x->hdrwidth;
	spca50x->frame[i].hdrheight = spca50x->hdrheight;
	spca50x->frame[i].width = spca50x->width;
	spca50x->frame[i].height = spca50x->height;
	spca50x->frame[i].format = spca50x->format;
	spca50x->frame[i].scanlength = spca50x->width * spca50x->height * 3 / 2;
	// ?? assumes 4:2:0 data
    }
#endif /* !__FreeBSD__ */
    return 0;
}
static inline int
spca50x_set_mode(struct usb_spca50x *spca50x,
		 int index, __u16 ext_modes[][6])
{
    struct usb_device *dev = spca50x->dev;
    int i;
    int mode = 0;
    int method = 0;
    int crop = 0, cropx1 = 0, cropx2 = 0, cropy1 = 0, cropy2 = 0, x =
	0, y = 0;
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_ETOMS:
	spca50x->mode = (ext_modes[index][2] & 0x0f);
	break;
    case BRIDGE_TV8532:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca50x->mode = mode;
	break;
    case BRIDGE_CX11646:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca50x->mode = mode;
        break;
#if 0
    case BRIDGE_ZC3XX:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca50x->mode = mode;
	break;
    case BRIDGE_SONIX:
	mode = (ext_modes[index][2] & 0x0f);
	spca50x->mode = mode;
	break;
    case BRIDGE_SN9CXXX:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca50x->mode = mode;
	break;
#endif
    case BRIDGE_SPCA506:
	spca506_Setsize(spca50x, ext_modes[index][2], ext_modes[index][3],
			ext_modes[index][4]);
	break;
    case BRIDGE_SPCA505:
	spca50x_reg_write(dev, SPCA50X_REG_COMPRESS, 0x0, ext_modes[index][2]);
	spca50x_reg_write(dev, SPCA50X_REG_COMPRESS, 0x6, ext_modes[index][3]);
	spca50x_reg_write(dev, SPCA50X_REG_COMPRESS, 0x7, ext_modes[index][4]);
	break;
    case BRIDGE_SPCA501:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca50x_reg_write(dev, SPCA50X_REG_USB, 0x6, ext_modes[index][3]);
	spca50x_reg_write(dev, SPCA50X_REG_USB, 0x7, ext_modes[index][4]);
	break;
    case BRIDGE_SPCA508:
	spca50x_reg_write(dev, 0, 0x8500, ext_modes[index][2]);	// mode
	spca50x_reg_write(dev, 0, 0x8700, ext_modes[index][3]);	// clock
        break;
#if 0
    case BRIDGE_SPCA561:
	spca50x->mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	//spca50x_reg_write (dev, 0, 0x8500, mode);                     // mode
	//spca50x_reg_write (dev, 0, 0x8700, ext_modes[index][3]);      // clock
    	breal;
#endif
    case BRIDGE_SPCA500:
	mode = (ext_modes[index][2]) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	x = ext_modes[index][3] & 0x0F;
	y = ext_modes[index][4] & 0x0F;
	/* Experimental Wait until Marek Test 
	   // set x multiplier
	   spca50x_reg_write (dev, 0, 0x8001,ext_modes[index][3]);
	   // set y multiplier 
	   spca50x_reg_write (dev, 0, 0x8002,ext_modes[index][4]);
	   // use compressed mode, VGA, with mode specific subsample 
	   spca50x_reg_write (dev, 0, 0x8003,((ext_modes[index][2] & 0x0f) << 4));
	 */
    	break;
#if 0
    case BRIDGE_SPCA504:
	mode = (ext_modes[index][2] + 3) & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	x = ext_modes[index][3] & 0x0F;
	y = ext_modes[index][4] & 0x0F;
	if (spca50x->desc == AiptekMiniPenCam13) {
	    /* spca504a aiptek */
	    spca504A_acknowledged_command(spca50x, 0x8, mode, 0,
					  (0x80 | (mode & 0x0F)), 1);
	    spca504A_acknowledged_command(spca50x, 1, 3, 0, 0x9F, 0);
	} else {
	    spca504_acknowledged_command(spca50x, 0x8, mode, 0);
	}
	break;
    case BRIDGE_SPCA504C:
	spca50x_reg_write(spca50x->dev, 0xa0, (0x0500 | (ext_modes[index][2] & 0x0F)), 0x0);	// capture mode
	spca50x_reg_write(spca50x->dev, 0x20, 0x1, (0x0500 | (ext_modes[index][2] & 0x0F)));	// snapshot mode
	break;
    case BRIDGE_SPCA504B:
	mode = ext_modes[index][2] & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	x = ext_modes[index][3] & 0x0F;
	y = ext_modes[index][4] & 0x0F;
	spca504B_SetSizeType(spca50x, mode, 6);
	break;
    case BRIDGE_SPCA533:
	mode = ext_modes[index][2] & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	x = ext_modes[index][3] & 0x0F;
	y = ext_modes[index][4] & 0x0F;
	if ((spca50x->desc == MegapixV4)
	    || (spca50x->desc == LogitechClickSmart820)) {
	    // megapix specific stuff
	    spca533_Megapix(spca50x);
	} else {
	    spca504B_SetSizeType(spca50x, mode, 6);
	}
	break;
    case BRIDGE_SPCA536:
	mode = ext_modes[index][2] & 0x0F;
	method = (ext_modes[index][2] & 0xF0) >> 4;
	spca536_SetSizeType(spca50x, mode, 6);
	break;
#endif
    }
    spca50x->pipe_size = ext_modes[index][5];
    spca50x->method = method;
    /* initialize the hdrwidth and hdrheight for the first init_source */
    if (!(spca50x->hdrwidth) || !(spca50x->hdrheight)) {

	spca50x->hdrwidth = spca50x->width;
	spca50x->hdrheight = spca50x->height;
    }
    /* precompute the crop x y value for each frame */
    if (!method) {
	/* nothing todo hardware found stream */
	cropx1 = cropx2 = cropy1 = cropy2 = x = y = 0;
    }
    if (method & 0x01) {
	/* cropping method */
	if (spca50x->hdrwidth > spca50x->width) {

	    crop = (spca50x->hdrwidth - spca50x->width);
	    if (spca50x->cameratype == JPEG || spca50x->cameratype == JPGH
		|| spca50x->cameratype == JPGS)
		crop = crop >> 4;
	    cropx1 = crop >> 1;
	    cropx2 = cropx1 + (crop % 2);
	} else {
	    cropx1 = cropx2 = 0;
	}
	if (spca50x->hdrheight > spca50x->height) {
	    crop = (spca50x->hdrheight - spca50x->height);
	    if (spca50x->cameratype == JPEG)
		crop = crop >> 4;
	    if (spca50x->cameratype == JPGH || spca50x->cameratype == JPGS)
		crop = crop >> 3;
	    cropy1 = crop >> 1;
	    cropy2 = cropy1 + (crop % 2);
	} else {
	    cropy1 = cropy2 = 0;
	}
    }
    if (method & 0x02) {
	/* what can put here for div method */
    }
    if (method & 0x04) {
	/* and here for mult */
    }
    PDEBUG(2, "Found code %d method %d", mode, method);
    PDEBUG(2, "Soft Win width height %d x %d", spca50x->width,
	   spca50x->height);
    PDEBUG(2, "Hard Win width height %d x %d", spca50x->hdrwidth,
	   spca50x->hdrheight);

    i = 0; /* silence compiler */
#if !defined(__FreeBSD__)
    /* XXX copied from above */
    for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
	spca50x->frame[i].method = spca50x->method;
	spca50x->frame[i].cameratype = spca50x->cameratype;
	spca50x->frame[i].cropx1 = cropx1;
	spca50x->frame[i].cropx2 = cropx2;
	spca50x->frame[i].cropy1 = cropy1;
	spca50x->frame[i].cropy2 = cropy2;
	spca50x->frame[i].x = x;
	spca50x->frame[i].y = y;
	spca50x->frame[i].hdrwidth = spca50x->hdrwidth;
	spca50x->frame[i].hdrheight = spca50x->hdrheight;
	spca50x->frame[i].width = spca50x->width;
	spca50x->frame[i].height = spca50x->height;
	spca50x->frame[i].scanlength = spca50x->width * spca50x->height * 3 / 2;
	// ?? assumes 4:2:0 data
    }
#endif /* !__FreeBSD__ */
    return 0;
}

/**********************************************************************
* spca50x_mode_init_regs
* Function sets up the resolution with checking if it's necessary 
***********************************************************************/
static int
spca5xx_restartMode(struct usb_spca50x *spca50x, int width, int height,
		    int format)
{
    int was_streaming;
    int r;
    /* Avoid changing to already selected mode */
    if (spca50x->width == width && spca50x->height == height) {
	PDEBUG(1, "*** no need to change mode\n");
	// return 0;
    }
    PDEBUG(1, "Mode changing to %d,%d", width, height);

    was_streaming = spca50x->streaming;
    /* FIXME spca500 bridge is there a way to find an init like 
       Clicksmart310 ? */

    if (was_streaming) {
	if ((PWC_SC(spca50x)->pwc_info.bridge != BRIDGE_SPCA500)
	    || (spca50x->desc == LogitechClickSmart310))
	    spca50x_stop_isoc(spca50x);
    }

    PDEBUG(2, "Set mode asked w %d h %d p %d", width, height, format);
    r = spca5xx_setMode(spca50x, width, height, format);
    if (r < 0)
	goto out;
    if (was_streaming) {
	if ((PWC_SC(spca50x)->pwc_info.bridge != BRIDGE_SPCA500)
	    || (spca50x->desc == LogitechClickSmart310)) {
	    r = spca50x_init_isoc(spca50x);
	} else {
	    spcaCameraStart(spca50x);
	}
    }
  out:
    return r;
}
static int
spca50x_mode_init_regs(struct usb_spca50x *spca50x,
		       int width, int height, int mode,
		       __u16 ext_modes[][6])
{
    int i, j;
    int r;
    int was_streaming;

    /* Avoid changing to already selected mode */
    if (spca50x->width == width && spca50x->height == height)
	return 0;

    /* find a suitable mode */
    if ((i = spca50x_find_mode_index(width, height, ext_modes)) == -EINVAL)
	return -EINVAL;
    /* keep the hardware index from the hight nibble */
    j = (i & 0xF0) >> 4;
    /* the software too */
    i = i & 0x0F;
    /* make sure that the suitable mode isn't the current mode */
    if (spca50x->width == ext_modes[i][0]
	&& spca50x->height == ext_modes[i][1])
	return 0;

    PDEBUG(1, "Mode changing to %d,%d", width, height);
    was_streaming = spca50x->streaming;

    /* FIXME spca500 bridge is there a way to find an init like 
       Clicksmart310 ? */

    if (was_streaming) {
	if ((PWC_SC(spca50x)->pwc_info.bridge != BRIDGE_SPCA500)
	    || (spca50x->desc == LogitechClickSmart310))
	    spca50x_stop_isoc(spca50x);
    }
    spca50x->hdrwidth = ext_modes[j][0];
    spca50x->hdrheight = ext_modes[j][1];
    spca50x->width = ext_modes[i][0];
    spca50x->height = ext_modes[i][1];

    spca50x_set_mode(spca50x, i, ext_modes);

    r = 0;
    if (was_streaming) {
	if ((PWC_SC(spca50x)->pwc_info.bridge != BRIDGE_SPCA500)
	    || (spca50x->desc == LogitechClickSmart310)) {
	    r = spca50x_init_isoc(spca50x);
	} else {
	    spcaCameraStart(spca50x);
	}
    }
    return r;
}

/**********************************************************************
* spca50x_get_brightness
* Function reads the brightness from the camera
* Receives the pointer to the description structure
* returns the value of brightness
**********************************************************************/
__u16 spca50x_get_brightness(struct usb_spca50x *spca50x)
{
    __u16 brightness = 0;	// value of the brightness

    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_PAC207:
	brightness = pac207_getbrightness(spca50x);
	break;
    case BRIDGE_SN9CXXX:
	brightness = sn9cxxx_getbrightness(spca50x);
	break;
    case BRIDGE_ETOMS:
	brightness = Et_getbrightness(spca50x);
	break;
    case BRIDGE_ZC3XX:
	brightness = zc3xx_getbrightness(spca50x);
	break;
    case BRIDGE_SONIX:
	brightness = sonix_getbrightness(spca50x);
	break;
    case BRIDGE_TV8532:
	brightness = tv8532_getbrightness(spca50x);
	break;
    case BRIDGE_CX11646:
	brightness = cx_getbrightness(spca50x);
	break;
    case BRIDGE_SPCA500:
	{
	    /* brightness is stored by the bridge as
	     * 129 -> 128, with 0 being the midpoint.
	     * v4l always keeps these values unsigned
	     * 16 bit, that is 0 to 65535.
	     * scale accordingly.
	     */
	    brightness = spca50x_reg_read(spca50x->dev, 0x00, 0x8167, 1);
	    brightness = brightness << 8;
	    break;
	}

    case BRIDGE_SPCA505:	//here505b
	brightness =
	    65535 - ((spca50x_reg_read(spca50x->dev, 5, 0x01, 1) >> 2) +
		     (spca50x_reg_read(spca50x->dev, 5, 0x0, 1) << 6));

	break;
    case BRIDGE_SPCA506:
	brightness = spca50x_read_i2c(spca50x, SAA7113_I2C_BASE_READ, 0xa);
	break;
    case BRIDGE_SPCA501:
	{
	    brightness =
		spca50x_reg_read(spca50x->dev, 0x0, 0x00, 2) & 0xFF;
	    brightness -= 125;
	    brightness <<= 1;
	    break;
	}
    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA504C:
	brightness = sp5xxfw2_getbrightness(spca50x);
    case BRIDGE_SPCA561:
	break;
#ifdef SPCA50X_ENABLE_EXP_BRIGHTNESS
    case BRIDGE_SPCA508:
	brightness = spca50x_reg_read(spca50x->dev, 0, 0x8651, 1);
	brightness = brightness << 8;
	break;
#endif				/* SPCA50X_ENABLE_EXP_BRIGHTNESS */
    case BRIDGE_MR97311:
	brightness = 0x14;
	break;
    default:
	    brightness = 0;
	    break;
    }

    return brightness;
}

/**********************************************************************
* spca50x_set_brightness
* Function sets the brightness to the camera
* Receives the pointer to the description structure 
* and brightness value
**********************************************************************/
void
spca50x_set_brightness(struct usb_spca50x *spca50x, __u8 brightness)
{
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_PAC207:
	spca50x->brightness = brightness << 8;
	pac207_setbrightness(spca50x);
	break;
    case BRIDGE_SN9CXXX:
	spca50x->brightness = brightness << 8;
	sn9cxxx_setbrightness(spca50x);
	break;
    case BRIDGE_ETOMS:
	spca50x->brightness = brightness << 8;
	Et_setbrightness(spca50x);
	break;
    case BRIDGE_ZC3XX:
	spca50x->brightness = brightness << 8;
	zc3xx_setbrightness(spca50x);
	break;
    case BRIDGE_SONIX:
	spca50x->brightness = brightness << 8;
	sonix_setbrightness(spca50x);
	break;
    case BRIDGE_TV8532:
	spca50x->brightness = brightness << 8;
	tv8532_setbrightness(spca50x);
	break;
    case BRIDGE_CX11646:
	spca50x->brightness = brightness << 8;
	cx_setbrightness(spca50x);
	break;
    case BRIDGE_SPCA500:
	/* 0 - 255 with zero at 128 */
	spca50x_reg_write(spca50x->dev, 0x00, 0x8167, brightness);
	break;

    case BRIDGE_SPCA505:	//here505b
	// brightness
	spca50x_reg_write(spca50x->dev, 5, 0x0, (255 - brightness) >> 6);
	spca50x_reg_write(spca50x->dev, 5, 0x01, (255 - brightness) << 2);
	break;

    case BRIDGE_SPCA506:
	spca50x_write_i2c(spca50x, SAA7113_I2C_BASE_WRITE, 0xa, brightness);
	break;

    case BRIDGE_SPCA501:
	spca50x_reg_write(spca50x->dev, 0x0, 0x00, (brightness >> 1) + 125);
	break;

    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA504C:
	spca50x->brightness = brightness << 8;
	sp5xxfw2_setbrightness(spca50x);
	break;

    case BRIDGE_SPCA561:
	break;

#ifdef SPCA50X_ENABLE_EXP_BRIGHTNESS
    case BRIDGE_SPCA508:
	spca50x_reg_write(spca50x->dev, 0, 0x8651, brightness);
	spca50x_reg_write(spca50x->dev, 0, 0x8652, brightness);
	spca50x_reg_write(spca50x->dev, 0, 0x8653, brightness);
	spca50x_reg_write(spca50x->dev, 0, 0x8654, brightness);
	/* autoadjust is set to 0 to avoid doing repeated brightness
	   calculation on the same frame. autoadjust is set to 0
	   when a new frame is ready. */
	autoadjust = 0;
	break;
#endif				/* SPCA50X_ENABLE_EXP_BRIGHTNESS */

    case BRIDGE_MR97311:
	break;
    default:
	    break;
    }
}

#if !defined(__FreeBSD__)
/* XXX probably not fully necessary */
/**********************************************************************
 *
 * SPCA50X data transfer, IRQ handler
 *
 **********************************************************************/
static struct spca50x_frame *spca50x_next_frame(struct usb_spca50x
						*spca50x,
						unsigned char *cdata)
{
    int iFrameNext;
    struct spca50x_frame *frame = NULL;
    PDEBUG(2, "Frame %d State %d", spca50x->curframe,
	   spca50x->frame[spca50x->curframe].grabstate);
    if (spca50x->frame[spca50x->curframe].grabstate == FRAME_ERROR) {
	PDEBUG(2, "Frame %d errdrop", spca50x->curframe);
	frame = &spca50x->frame[spca50x->curframe];
	goto dropframe;
    }
    /* Cycle through the frame buffer looking for a free frame to overwrite */
    iFrameNext = (spca50x->curframe + 1) % SPCA50X_NUMFRAMES;
    while (frame == NULL && iFrameNext != (spca50x->curframe)) {
	if (spca50x->frame[iFrameNext].grabstate == FRAME_READY ||
	    spca50x->frame[iFrameNext].grabstate == FRAME_UNUSED ||
	    spca50x->frame[iFrameNext].grabstate == FRAME_ERROR) {
	    spca50x->curframe = iFrameNext;
	    frame = &spca50x->frame[iFrameNext];
	    break;
	} else {
	    iFrameNext = (iFrameNext + 1) % SPCA50X_NUMFRAMES;
	}
    }

    if (frame == NULL) {
	PDEBUG(3, "Can't find a free frame to grab into...using next. "
	       "This is caused by the application not reading fast enough.");
	spca50x->curframe = (spca50x->curframe + 1) % SPCA50X_NUMFRAMES;
	frame = &spca50x->frame[spca50x->curframe];
    }
  dropframe:
    frame->grabstate = FRAME_GRABBING;

    /* Record the frame sequence number the camera has told us */
    if (cdata) {
	switch (spca50x->bridge) {
	case BRIDGE_SPCA500:
	    frame->seq = cdata[SPCA500_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA501:
	    frame->seq = cdata[SPCA501_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA504:
	    frame->seq = cdata[SPCA50X_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA504C:
	    frame->seq = cdata[SPCA50X_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA504B:
	    frame->seq = cdata[SPCA50X_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA505:
	    frame->seq = cdata[SPCA50X_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA506:
	    frame->seq = cdata[SPCA50X_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA508:
	    frame->seq = cdata[SPCA508_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA533:
	    frame->seq = cdata[SPCA533_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA561:
	    frame->seq = cdata[SPCA561_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_SPCA536:
	    frame->seq = cdata[SPCA536_OFFSET_FRAMSEQ];
	    break;
	case BRIDGE_CX11646:
	case BRIDGE_SONIX:
	case BRIDGE_ZC3XX:
	case BRIDGE_TV8532:
	case BRIDGE_ETOMS:
	case BRIDGE_SN9CXXX:
	case BRIDGE_MR97311:
	case BRIDGE_PAC207:
	    frame->seq = 00;
	    break;
	}
    }
    if (spca50x->pictsetting.change) {
	memcpy(&frame->pictsetting, &spca50x->pictsetting,
	       sizeof(struct pictparam));
	/* reset flag change */
	spca50x->pictsetting.change = 0;
	PDEBUG(2, "Picture setting change Pass to decoding   ");
    }
    /* Reset some per-frame variables */
    frame->highwater = frame->data;
    frame->scanstate = STATE_LINES;
    frame->scanlength = 0;
    frame->last_packet = -1;
    frame->totlength = 0;
    spca50x->packet = 0;
    return frame;
}

/* Tasklet function to decode */

void outpict_do_tasklet(unsigned long ptr)
{
    int err;
    struct spca50x_frame *taskletframe = (struct spca50x_frame *) ptr;

    taskletframe->scanlength =
	taskletframe->highwater - taskletframe->data;
    PDEBUG(2,
	   "Tasklet ask spcadecoder hdrwidth %d hdrheight %d method %d ",
	   taskletframe->hdrwidth, taskletframe->hdrheight,
	   taskletframe->method);


    if ((err = spca50x_outpicture(taskletframe)) < 0) {
	PDEBUG(2, "frame decoder failed (%d)", err);
	taskletframe->grabstate = FRAME_ERROR;
    } else {
	taskletframe->grabstate = FRAME_DONE;
	PDEBUG(2, "Decode framestate return %d", taskletframe->grabstate);
    }

    if (waitqueue_active(&taskletframe->wq))
	wake_up_interruptible(&taskletframe->wq);
}
#endif /* !__FreeBSD__ */

/*********************************************************************
	time Helper function
**********************************************************************/
static inline unsigned long spca5xx_gettimes(void)
{
#if defined(__FreeBSD__)
    struct timeval tv;
    getmicrotime(&tv);
    return (tv.tv_sec * 1000 + tv.tv_usec / 1000);
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
    struct timeval tv;
    do_gettimeofday(&tv);
    return (tv.tv_sec * 1000 + tv.tv_usec / 1000);
#else
    u64 times_now;
    times_now = get_jiffies_64();
    return jiffies_to_msecs(times_now);
#endif
}

#if defined(__FreeBSD__)
#include "pwc.h"
/*
 * determines whether this is a good frame, etc
 */
int _unused_spca50x_move_data_helper(struct usb_spca50x *spca50x,
	unsigned char *cdata, int datalength)
{
    int iPix;			//Offset of pixel data in the ISO packet

    unsigned long ms_times_now;
    unsigned long ms_times_before;
    int totlen = 0;
    int tv8532 = 0;

    int j, p;
    int sequenceNumber;
    int seqframe = 0;
    int sof;

    unsigned char *pData;
    int __once = 1;
    struct pwc_frame_buf *frame = PWC_SC(spca50x)->fill_frame;

    frame->highwater = frame->data + frame->filled;
    while (__once-- > 0) {	/* to minimize diffs since we enter in a block */


#else
/* ******************************************************************
* spca50x_move_data
* Function serves for moving data from USB transfer buffers
* to internal driver frame buffers.
******************************************************************* */
static int spca50x_move_data(struct usb_spca50x *spca50x, struct urb *urb)
{
    unsigned char *cdata;	//Pointer to buffer where we do store next packet
    unsigned char *pData;	//Pointer to buffer where we do store next packet
    int i;
    struct spca50x_frame *frame;	//Pointer to frame data
    int iPix;			//Offset of pixel data in the ISO packet

    unsigned long ms_times_now;
    unsigned long ms_times_before;
    int totlen = 0;
    int tv8532 = 0;

    for (i = 0; i < urb->number_of_packets; i++) {
	int datalength = urb->iso_frame_desc[i].actual_length;
	int st = urb->iso_frame_desc[i].status;
	int sequenceNumber;
	int seqframe = 0;
	int sof;
	int j, p;
	/* PDEBUG(5,"Packet data [%d,%d,%d]", datalength, st,
	   urb->iso_frame_desc[i].offset); */

	urb->iso_frame_desc[i].actual_length = 0;
	urb->iso_frame_desc[i].status = 0;

	cdata = ((unsigned char *) urb->transfer_buffer) +
	    urb->iso_frame_desc[i].offset;
	/* Check for zero length block or no selected frame buffer */
	if (!datalength || spca50x->curframe == -1) {
	    tv8532 = 0;
	    continue;
	}

	PDEBUG(5, "Packet data [%d,%d,%d] Status: %d", datalength, st,
	       urb->iso_frame_desc[i].offset, st);

	if (st)
	    PDEBUG(2, "data error: [%d] len=%d, status=%d", i, datalength,
		   st);
	frame = &spca50x->frame[spca50x->curframe];
#endif /* !__FreeBSD__ */
	totlen = frame->totlength;

	/* read the sequence number */
        switch (PWC_SC(spca50x)->pwc_info.bridge) {
	case BRIDGE_SONIX:
	case BRIDGE_ZC3XX:
	case BRIDGE_CX11646:
	case BRIDGE_TV8532:
	case BRIDGE_ETOMS:
	case BRIDGE_SN9CXXX:
	case BRIDGE_MR97311:
	case BRIDGE_PAC207:
	    if (frame->last_packet == -1) {
		/*initialize a new frame */
		sequenceNumber = 0;
	    } else {
		sequenceNumber = frame->last_packet;
	    }
	    break;
	default:
	    sequenceNumber = cdata[SPCA50X_OFFSET_SEQUENCE];
	    PDEBUG(4, "Packet start  %x %x %x %x %x", cdata[0], cdata[1],
		   cdata[2], cdata[3], cdata[4]);
	}
	/* check frame start */
        switch (PWC_SC(spca50x)->pwc_info.bridge) {
	case BRIDGE_SN9CXXX:
	    iPix = 0;
	    sof = datalength - 64;
	    if (sof < 0)
		sequenceNumber++;
	    else if (cdata[sof] == 0xff && cdata[sof + 1] == 0xd9) {
		sequenceNumber = 0;	//start of frame
		// copy the end of data frame
		memcpy(frame->highwater, cdata, sof + 2);
		frame->highwater += (sof + 2);
		totlen += (sof + 2);
#if 0
		spin_lock(&spca50x->v4l_lock);
		spca50x->avg_lum =
		    (cdata[sof + 24] + cdata[sof + 26]) >> 1;
		spin_unlock(&spca50x->v4l_lock);
		PDEBUG(5, "mean luma %d", spca50x->avg_lum);
#endif
		PDEBUG(5,
		       "Sonix header packet found datalength %d totlength %d!!",
		       datalength, totlen);
		PDEBUG(5, "%03d %03d %03d %03d %03d %03d %03d %03d",
		       cdata[sof + 24], cdata[sof + 25], cdata[sof + 26],
		       cdata[sof + 27], cdata[sof + 28], cdata[sof + 29],
		       cdata[sof + 30], cdata[sof + 31]);
		// setting to skip the rest of the packet
		spca50x->header_len = datalength;
	    } else
		sequenceNumber++;
	    break;
	case BRIDGE_ETOMS:
		iPix = 8;
		seqframe = cdata[0] & 0x3f;
		datalength = (int) (((cdata[0] & 0xc0) << 2) | cdata[1]);
		if (seqframe == 0x3f) {
		    PDEBUG(5,
			   "Etoms header packet found datalength %d totlength %d!!",
			   datalength, totlen);
		    PDEBUG(5, "Etoms G %d R %d G %d B %d", cdata[2],
			   cdata[3], cdata[4], cdata[5]);
		    sequenceNumber = 0;	//start of frame
		    spca50x->header_len = 30;
		} else {
		    if (datalength) {
			sequenceNumber++;
		    } else {
			/* Drop Packet */
			continue;
		    }
		}
	    break;

	case BRIDGE_CX11646:
		iPix = 0;
		if (cdata[0] == 0xFF && cdata[1] == 0xD8) {
		    sequenceNumber = 0;
		    spca50x->header_len = 2;
		    PDEBUG(5,
			   "Cx11646 header packet found datalength %d totlength %d!!",
			   datalength, totlen);
		} else {
		    sequenceNumber++;
		}
	    break;

	case BRIDGE_ZC3XX:
		iPix = 0;
		if (cdata[0] == 0xFF && cdata[1] == 0xD8) {
		    sequenceNumber = 0;
		    spca50x->header_len = 2;	//18 remove 0xff 0xd8;
		    PDEBUG(5,
			   "Zc301 header packet found datalength %d totlength %d!!",
			   datalength, totlen);
		} else {
		    sequenceNumber++;
		}
	    break;

	case BRIDGE_SONIX:
		iPix = 0;
		sof = 0;
		j = 0;
		p = 0;
		if (datalength < 24) {
		    if (datalength > 6) {
			j = datalength - 6;
		    } else {
			j = 0;
		    }
		    if (j > 0) {
			for (p = 0; p < j; p++) {
			    if ((cdata[0 + p] == 0xFF)
				&& (cdata[1 + p] == 0xFF)
				&& (cdata[2 + p] == 0x00)
				&& (cdata[3 + p] == 0xC4)
				&& (cdata[4 + p] == 0xC4)
				&& (cdata[5 + p] == 0x96)) {
				sof = 1;
				break;
			    }
			}
		    }
		}
		if (sof) {
		    sequenceNumber = 0;
		    spca50x->header_len = p + 12;
		    PDEBUG(5,
			   "Sonix header packet found %d datalength %d totlength %d!!",
			   p, datalength, totlen);
		} else {
		    /* drop packet */
		    sequenceNumber++;
		}
	    break;

	case BRIDGE_SPCA500:
	case BRIDGE_SPCA533:
		iPix = 1;
		if (sequenceNumber == SPCA50X_SEQUENCE_DROP) {
		    if (cdata[1] == 0x01) {
			sequenceNumber = 0;
		    } else {
			/* drop packet */
			PDEBUG(5, "Dropped packet (expected seq 0x%02x)",
			       frame->last_packet + 1);
			continue;
		    }
		} else {
		    sequenceNumber++;
		}
	    break;

	case BRIDGE_SPCA536:
		iPix = 2;
		if (sequenceNumber == SPCA50X_SEQUENCE_DROP) {
		    sequenceNumber = 0;
		} else {
		    sequenceNumber++;
		}
	    break;

	case BRIDGE_SPCA504:
	case BRIDGE_SPCA504B:
	case BRIDGE_SPCA504C:
		iPix = 1;
		switch (sequenceNumber) {
		case 0xfe:
		    sequenceNumber = 0;
		    break;
		case SPCA50X_SEQUENCE_DROP:
		    /* drop packet */
		    PDEBUG(5, "Dropped packet (expected seq 0x%02x)",
			   frame->last_packet + 1);
		    continue;
		default:
		    sequenceNumber++;
		    break;
		}
	    break;

	case BRIDGE_TV8532:
		//iPix = 4;     
		iPix = 0;
		spca50x->header_len = 0;
		PDEBUG(1, "icm532, sequenceNumber: 0x%02x packet %d ",
		       sequenceNumber, spca50x->packet);
		/* here we detect 0x80 */
		if (cdata[0] == 0x80) {
		    /* counter is limited so we need few header for a frame :) */

		    /* header 0x80 0x80 0x80 0x80 0x80 */
		    /* packet  00   63  127  145  00   */
		    /* sof     0     1   1    0    0   */
		    /* update sequence */
		    if ((spca50x->packet == 63)
			|| (spca50x->packet == 127))
			tv8532 = 1;
		    /* is there a frame start ? */
		    if (spca50x->packet >= ((spca50x->hdrheight >> 1) - 1)) {
			PDEBUG(1, "SOF > %d seqnumber %d packet %d",
			       tv8532, sequenceNumber, spca50x->packet);
			if (!tv8532) {
			    sequenceNumber = 0;
			    spca50x->packet = 0;
			}
		    } else {

			if (!tv8532) {
			    // Drop packet frame corrupt
			    PDEBUG(1, "DROP SOF %d seqnumber %d packet %d",
				   tv8532, sequenceNumber,
				   spca50x->packet);
			    frame->last_packet = sequenceNumber = 0;
			    spca50x->packet = 0;
			    continue;
			}
			tv8532 = 1;
			sequenceNumber++;
			spca50x->packet++;
		    }

		} else {
		    sequenceNumber++;
		    spca50x->packet++;
		}
	    break;

	case BRIDGE_MR97311:
		iPix = 0;
		sof = 0;
		j = 0;
		p = 0;
		if (datalength < 6)
		    continue;
		else {
		    for (p = 0; p < datalength - 6; p++) {
			if ((cdata[0 + p] == 0xFF)
			    && (cdata[1 + p] == 0xFF)
			    && (cdata[2 + p] == 0x00)
			    && (cdata[3 + p] == 0xFF)
			    && (cdata[4 + p] == 0x96)
			    ) {
			    if ((cdata[5 + p] == 0x64)
				|| (cdata[5 + p] == 0x65)
				|| (cdata[5 + p] == 0x66)
				|| (cdata[5 + p] == 0x67)) {
				sof = 1;
				break;
			    }
			}
		    }

		    if (sof) {
			sequenceNumber = 0;
			spca50x->header_len = p + 16;
			PDEBUG(5,
			       "Pcam header packet found, %d datalength %d totlength %d!!",
			       p, datalength, totlen);
		    } else {
			/* drop packet */
			sequenceNumber++;
		    }

		}
	    break;

	case BRIDGE_PAC207:
		iPix = 0;
		sof = 0;
		j = 0;
		p = 0;
		if (datalength < 6)
		    continue;
		else {
		    for (p = 0; p < datalength - 6; p++) {
			if ((cdata[0 + p] == 0xFF)
			    && (cdata[1 + p] == 0xFF)
			    && (cdata[2 + p] == 0x00)
			    && (cdata[3 + p] == 0xFF)
			    && (cdata[4 + p] == 0x96)
			    ) {
			    sof = 1;
			    break;
			}
		    }

		    if (sof) {
			sequenceNumber = 0;
#if 1
		spin_lock(&spca50x->v4l_lock);
		spca50x->avg_lum = cdata[p+9] ;
		spin_unlock(&spca50x->v4l_lock);
		PDEBUG(5, "mean luma %d", spca50x->avg_lum);
#endif
			// copy the end of data to the current frame
			memcpy(frame->highwater, cdata, p);
			frame->highwater += p;
			totlen += p;
			spca50x->header_len = p;	//copy to the nextframe start at p
			PDEBUG(5,
			       "Pixartcam header packet found, %d datalength %d totlength %d!!",
			       p, datalength, totlen);
		    } else {
			/* drop packet */
			sequenceNumber++;
		    }

		}
	    break;

	case BRIDGE_SPCA561:
	    iPix = 1;
	    if (sequenceNumber == SPCA50X_SEQUENCE_DROP) {
		PDEBUG(3, "Dropped packet (expected seq 0x%02x)",
		       frame->last_packet + 1);
		continue;
	    }
#if 0
	    if (!sequenceNumber) {
		spin_lock(&spca50x->v4l_lock);
		spca50x->avg_lum =
		    (75 * (cdata[11] + cdata[14]) + 77 * cdata[12] +
		     29 * cdata[13]) >> 8;
		spin_unlock(&spca50x->v4l_lock);
		PDEBUG(4, "Frame %d,Win2 %02d ", cdata[4],
		       spca50x->avg_lum);
	    }
#endif
	    break;
	default:
		iPix = 1;
		/* check if this is a drop packet */
		if (sequenceNumber == SPCA50X_SEQUENCE_DROP) {
		    PDEBUG(3, "Dropped packet (expected seq 0x%02x)",
			   frame->last_packet + 1);
		    continue;
		}
	    break;
	}
if(0)
	PDEBUG(3, "spca50x: Packet seqnum = 0x%02x.  curframe=%2d",
	       sequenceNumber, spca50x->curframe);
	pData = cdata;

	/* Can we find a frame start */
	if (sequenceNumber == 0) {
	    totlen = 0;
	    iPix = spca50x->header_len;
	    PDEBUG(3, "spca50x: Found Frame Start!, framenum = %d",
		   spca50x->curframe);
	    // Start of frame is implicit end of previous frame
	    // Check for a previous frame and finish it off if one exists
	    if (frame->scanstate == STATE_LINES) {

		if (frame->format != VIDEO_PALETTE_RAW_JPEG) {
		    ms_times_now = spca5xx_gettimes();
		    /* overflow ? */
		    if (ms_times_now < spca50x->last_times)
			spca50x->last_times = 0;
		    spin_lock(&spca50x->v4l_lock);
		    ms_times_before =
			spca50x->last_times + spca50x->dtimes;
		    spin_unlock(&spca50x->v4l_lock);
		    if (ms_times_now >= ms_times_before) {
			PDEBUG(2, "Decode frame last %d",
			       (int) (ms_times_now - spca50x->last_times));
			spca50x->last_times = ms_times_now;
			/* Decode the frame */
#if defined(__FreeBSD__)
			// outpict_do_tasklet((unsigned long) frame);
#else /* !__FreeBSD__ */
			tasklet_init(&spca50x->spca5xx_tasklet,
				     outpict_do_tasklet,
				     (unsigned long) frame);
			tasklet_schedule(&spca50x->spca5xx_tasklet);
#endif /* !__FreeBSD__ */
		    } else
			frame->grabstate = FRAME_ERROR;
		} else {
		    /* RAW DATA stream */
		    frame->grabstate = FRAME_DONE;
		    if (waitqueue_active(&frame->wq))
			wake_up_interruptible(&frame->wq);
		}
		// If someone decided to wait for ANY frame - wake him up 
		if (waitqueue_active(&spca50x->wq))
		    wake_up_interruptible(&spca50x->wq);
#if !defined(__FreeBSD__)
		frame = spca50x_next_frame(spca50x, cdata);
#endif
	    } else
		frame->scanstate = STATE_LINES;
	}

	/* Are we in a frame? */
	if (frame == NULL || frame->scanstate != STATE_LINES)
	    continue;
	if (sequenceNumber != frame->last_packet + 1 &&
	    frame->last_packet != -1) {
	    /* Note, may get one of these for the first packet after opening */
	    PDEBUG(2, "Out of order packet, last = %d, this = %d",
		   frame->last_packet, sequenceNumber);

	}
	frame->last_packet = sequenceNumber;

	/* This is the real conversion of the raw camera data to BGR for V4L */
        switch (PWC_SC(spca50x)->pwc_info.bridge) {
	case BRIDGE_TV8532:
	case BRIDGE_ETOMS:
	    datalength -= iPix;	// correct length for packet header
	    break;
	}

	pData = cdata + iPix;	// Skip packet header (1 or 10 bytes)

	// Consume data
if (0)
	PDEBUG(5, "Processing packet seq  %d,length %d,totlength %d",
	       frame->last_packet, datalength, frame->totlength);
	/* this copy consume input data from the isoc stream */
	if ((datalength > 0) && (datalength <= 0x3ff)) {
	    memcpy(frame->highwater, pData, datalength);
	    frame->highwater += datalength;
	    totlen += datalength;
	}

	frame->totlength = totlen;
#ifdef SPCA50X_ENABLE_EXP_BRIGHTNESS
	/* autoadjust is set to 1 to so now autobrightness will be
	   calculated frome this frame. autoadjust will be set to 0 when
	   autobrightness has been corrected if needed. */
	autoadjust = 1;
#endif				/* SPCA50X_ENABLE_EXP_BRIGHTNESS */
#if defined(__FreeBSD__)
	frame->filled = frame->highwater - frame->data;
#endif


    }
    return totlen;
}

/**
 * Reset the camera and send the correct initialization sequence for the
 * currently selected source
 */
int spca50x_init_source(struct usb_spca50x *spca50x)
{
    int err_code;
    struct usb_device *dev = spca50x->dev;
    int index;			// index in the modes table

    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    default:
	err("spca50x_init_source: Unimplemented bridge type %d\n", PWC_SC(spca50x)->pwc_info.bridge);
	return -EINVAL;

    case BRIDGE_EM2820:
	err_code = em2820_init(spca50x);
	break;

    case BRIDGE_SN9CXXX:
	err_code = sn9cxxx_init(spca50x);
	PDEBUG(2, "Initializing Sonix finished %d", err_code);
	if (err_code < 0)
	    return err_code;
	break;

    case BRIDGE_ETOMS:
	err_code = Et_init(spca50x);
	break;

    case BRIDGE_CX11646:
	err_code = cx11646_init(spca50x);
	break;

    case BRIDGE_ZC3XX:
	err_code = zc3xx_init(spca50x);
	break;

    case BRIDGE_SONIX:
	err_code = sonix_init(spca50x);
	PDEBUG(2, "Initializing Sonix finished");
	break;

    case BRIDGE_SPCA500:
	    /* initialisation of spca500 based cameras is deferred */
	    PDEBUG(2, "Initializing SPCA500 started");
	    if (spca50x->desc == LogitechClickSmart310) {
		spca500_clksmart310_init(spca50x);
	    } else {
		spca500_initialise(spca50x);
	    }
	    PDEBUG(2, "Initializing SPCA500 finished");
	    break;

    case BRIDGE_SPCA501:
	    PDEBUG(2, "Initializing SPCA501 started");
#if defined(__FreeBSD__) /* XXX */
	    if (spca50x->vendor == 0x0506
		&& spca50x->product == 0x00df) {
#else
	    if (spca50x->dev->descriptor.idVendor == 0x0506
		&& spca50x->dev->descriptor.idProduct == 0x00df) {
#endif
		/* Special handling for 3com data */
		spca50x_write_vector(spca50x, spca501_3com_open_data);
	    } else if (spca50x->desc == Arowana300KCMOSCamera ||
		       spca50x->desc == SmileIntlCamera) {
		/* Arowana 300k CMOS Camera data */
		spca50x_write_vector(spca50x, spca501c_arowana_open_data);
	    } else if (spca50x->desc == MystFromOriUnknownCamera) {
		/* UnKnow  CMOS Camera data */
		spca50x_write_vector(spca50x,
				     spca501c_mysterious_init_data);
	    } else {
		/* Generic 501 open data */
		spca50x_write_vector(spca50x, spca501_open_data);
	    }
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
	    spca50x->a_blue = spca50x_reg_read(spca50x->dev,
					       SPCA501_REG_CCDSP,
					       SPCA501_A11, 2) & 0xFF;
	    spca50x->a_green =
		spca50x_reg_read(spca50x->dev, SPCA501_REG_CCDSP,
				 SPCA501_A21, 2) & 0xFF;
	    spca50x->a_red =
		spca50x_reg_read(spca50x->dev, SPCA501_REG_CCDSP,
				 SPCA501_A31, 2) & 0xFF;
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */
	    PDEBUG(2, "Initializing SPCA501 finished");
	    break;

    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504B:
	    PDEBUG(2, "Opening SPCA5xx FW2");
	    sp5xxfw2_init(spca50x);
	    break;

    case BRIDGE_SPCA505:
	    PDEBUG(2, "Initializing SPCA505");
	    if (spca50x->desc == Nxultra) {
		spca50x_write_vector(spca50x, spca505b_open_data_ccd);
	    } else {
		spca50x_write_vector(spca50x, spca505_open_data_ccd);
	    }
	    err_code = 0;
	    err_code = spca50x_reg_read(dev, 6, (__u16) 0x16, 2);

	    if (err_code < 0) {
		PDEBUG(1,
		       "register read failed for after vector read err = %d",
		       err_code);
		return -EIO;
	    } else {
		PDEBUG(3,
		       "After vector read returns : 0x%x should be 0x0101",
		       err_code & 0xFFFF);
	    }

	    err_code = spca50x_reg_write(dev, 6, 0x16, 0xa);
	    if (err_code < 0) {
		PDEBUG(1, "register write failed for (6,0xa,0x16) err=%d",
		       err_code);
		return -EIO;
	    }
	    spca50x_reg_write(spca50x->dev, 5, 0xc2, 18);
	    break;

    case BRIDGE_SPCA506:
	    spca506_init(spca50x);
	    break;

    case BRIDGE_SPCA561:
	    err_code = spca561_init(spca50x);
	    break;

    case BRIDGE_SPCA508:
	    spca50x_write_vector(spca50x, spca508_open_data);
	    break;

    case BRIDGE_TV8532:
	    /* To be sure that the init array is received by the cam, 
	       we send it 3 times. Yes, i'm tired of crashing, when trying to 
	       get a stream from a uninitalized cam... We probably should make
	       some fancy i2c stuff, but hey, if it ain't broken don't fix it ;)
	       BTW. it's broken! */
	    err_code = tv8532_init(spca50x);

	    break;

    case BRIDGE_PAC207:
	    err_code = pac207_init(spca50x);
	    break;

    case BRIDGE_MR97311:
	break;

    case BRIDGE_STV0602:
	err_code = qc_sensor_init(spca50x);
	break;

    }

    spca50x->norme = 0;
    spca50x->channel = 0;
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_ZC3XX:
    case BRIDGE_SONIX:
    case BRIDGE_SN9CXXX:
    case BRIDGE_ETOMS:
    case BRIDGE_SPCA561:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA536:
    case BRIDGE_STV0602:
    case BRIDGE_EM2820:
    case BRIDGE_PAC207:
    PDEBUG(2, "Set mode asked w %d h %d p %d", spca50x->width, spca50x->height, spca50x->format);
	err_code =
	    spca5xx_setMode(spca50x, spca50x->width, spca50x->height,
			    spca50x->format);
	PDEBUG(2, "set Mode return %d ", err_code);
	if (err_code < 0)
	    return -EINVAL;
	break;
    default:
	if ((index = spca50x_find_mode_index(spca50x->width, spca50x->height,
		     GET_EXT_MODES(PWC_SC(spca50x)->pwc_info.bridge))) == -EINVAL) {
	    err("Can't find proper mode?!!");
	} else {
	    err_code = spca50x_set_mode(spca50x, (index & 0x0F),
					GET_EXT_MODES(PWC_SC(spca50x)->pwc_info.bridge));
	    if (err_code) {
		err("Can't set proper camera mode");
		return -EINVAL;
	    }
	}
	break;
    }
    return 0;
}


/****************************************************************************
 *
 * V4L API
 *
 ***************************************************************************/
static inline void spca5xx_setFrameDecoder(struct usb_spca50x *spca50x)
{
    int i;
    /* configure the frame detector with default parameters */
    memset(&spca50x->pictsetting, 0, sizeof(struct pictparam));
    spca50x->pictsetting.change = 0x01;
    spca50x->pictsetting.force_rgb = force_rgb;
    spca50x->pictsetting.gamma = gamma;
    spca50x->pictsetting.OffRed = OffRed;
    spca50x->pictsetting.OffBlue = OffBlue;
    spca50x->pictsetting.OffGreen = OffGreen;
    spca50x->pictsetting.GRed = GRed;
    spca50x->pictsetting.GBlue = GBlue;
    spca50x->pictsetting.GGreen = GGreen;

    i = 0; /* silence compiler */
#if !defined(__FreeBSD__)
    /* Set default sizes in case IOCTL (VIDIOCMCAPTURE) is not used
     * (using read() instead). */
    for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
	spca50x->frame[i].width = spca50x->width;
	spca50x->frame[i].height = spca50x->height;
	spca50x->frame[i].cameratype = spca50x->cameratype;
	spca50x->frame[i].scanlength = spca50x->width * spca50x->height * 3 / 2;	// assumes 4:2:0 data
	spca50x->frame[i].depth = 24;

	/* Note: format reflects format of data as returned to
	 * a process, not as read from camera hardware.
	 * This might be a good idea for dumb programs always
	 * assuming the following settings.
	 */
	spca50x->frame[i].format = VIDEO_PALETTE_RGB24;
    }
#endif /* !__FreeBSD__ */
    // spca50x->format = VIDEO_PALETTE_RGB24;
    spca50x->format = VIDEO_PALETTE_YUV420P;	/* XXX default format */
}

static inline void spca5xx_initDecoder(struct usb_spca50x *spca50x)
{

    if (spca50x->cameratype == JPGH
	|| spca50x->cameratype == JPGC
	|| spca50x->cameratype == JPGS
	|| spca50x->cameratype == JPEG || spca50x->cameratype == JPGM)
	init_jpeg_decoder(spca50x);

    if (PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_SONIX)
	init_sonix_decoder(spca50x);
    if (PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_PAC207)
	init_pixart_decoder(spca50x);
}
static inline void spca5xx_chgAuto(struct usb_spca50x *spca50x,
				   __u8 autoval)
{
    spca50x->autoexpo = (int) autoval;
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_ZC3XX:
	zc3xx_setAutobright(spca50x);
	break;
    default:
	break;
    }
}
static inline void spca5xx_chgQtable(struct usb_spca50x *spca50x,
				     __u8 qtable)
{
/* one frame maybe corrupted  wait for the result */

    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_ZC3XX:
	spca50x->qindex = (int) qtable;
	zc3xx_setquality(spca50x);
	init_jpeg_decoder(spca50x);
	break;
    default:
	break;
    }

}
static inline void spca5xx_chgDtimes(struct usb_spca50x *spca50x,
				     __u16 dtimes)
{
    unsigned long flags;
    spin_lock_irqsave(&spca50x->v4l_lock, flags);
    spca50x->dtimes = (unsigned int) dtimes;
    spin_unlock_irqrestore(&spca50x->v4l_lock, flags);
}

#if !defined(__FreeBSD__)
static int spca5xx_open(struct inode *inode, struct file *file)
{
    ...
    /* initialize sensor and decoding */
    err = spca50x_init_source(spca50x);
    spca5xx_initDecoder(spca50x);
    spca5xx_setFrameDecoder(spca50x);
    err = spca50x_init_isoc(spca50x);
    /* Now, let's get brightness from the camera */
    spca50x->brightness = spca50x_get_brightness(spca50x) << 8;
    spca50x->whiteness = 0;
    return err;
}
#endif /* !__FreeBSD__ */

static void inline spcaCameraShutDown(struct usb_spca50x *spca50x)
{
    if (spca50x->dev) {
        switch (PWC_SC(spca50x)->pwc_info.bridge) {
	case BRIDGE_SPCA505:
	    spca50x_reg_write(spca50x->dev, 0x3, 0x3, 0x20);	// This maybe reset or power control
	    spca50x_reg_write(spca50x->dev, 0x3, 0x1, 0x0);
	    spca50x_reg_write(spca50x->dev, 0x3, 0x0, 0x1);
	    spca50x_reg_write(spca50x->dev, 0x5, 0x10, 0x1);
	    spca50x_reg_write(spca50x->dev, 0x5, 0x11, 0xF);
	    break;
	case BRIDGE_SPCA501:
	    // This maybe reset or power control
	    spca50x_reg_write(spca50x->dev, SPCA501_REG_CTLRL, 0x5, 0x0);
	    break;
	case BRIDGE_ZC3XX:
	    zc3xx_shutdown(spca50x);
	    break;
	case BRIDGE_SPCA561:
	    spca561_shutdown(spca50x);
	    break;
	case BRIDGE_PAC207:
	    pac207_shutdown(spca50x);
	    break;
	}
    }
}

#if !defined(__FreeBSD__)
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
static int spca5xx_close(struct inode *inode, struct file *file)
{

    struct video_device *vdev = file->private_data;
#else
static void spca5xx_close(struct video_device *vdev)
{
#endif
    struct usb_spca50x *spca50x = video_get_drvdata(vdev);
    int i;
    PDEBUG(2, "spca50x_close");

    down(&spca50x->lock);

    spca50x->user--;
    spca50x->curframe = -1;
    if (spca50x->present) {
	spca50x_stop_isoc(spca50x);
	spcaCameraShutDown(spca50x);


	for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
	    if (waitqueue_active(&spca50x->frame[i].wq))
		wake_up_interruptible(&spca50x->frame[i].wq);
	}
	if (waitqueue_active(&spca50x->wq))
	    wake_up_interruptible(&spca50x->wq);

    }
    /* times to dealloc ressource */
    up(&spca50x->lock);
    spca5xx_dealloc(spca50x);
    PDEBUG(2, "Release ressources done");

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
    MOD_DEC_USE_COUNT;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
    file->private_data = NULL;
    return 0;
#endif
}
#endif /* !__FreeBSD__ */

static int
spca5xx_testPalSize( struct usb_spca50x *spca50x, int pal, int w, int h)
{
int needpalette;
int needsize;
	    if ((needpalette = v4l_to_spca5xx(pal)) < 0)
	    	return -EINVAL;
	    if((needsize= wxh_to_size(w,h)) < 0)
	        return -EINVAL;
	    if(!(spca50x->mode_cam[needsize].t_palette & needpalette))
	    	return -EINVAL;
return 0;
}

#if defined(__FreeBSD__)
int
spca5xx_ioctl_helper(struct usb_spca50x *spca50x, unsigned int cmd, void *arg)
{
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
static int
spca5xx_do_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		 void *arg)
{
    struct video_device *vdev = file->private_data;
#else
static int
spca5xx_ioctl(struct video_device *vdev, unsigned int cmd, void *arg)
{
#endif
    struct usb_spca50x *spca50x = video_get_drvdata(vdev);

    PDEBUG(2, "do_IOCtl: 0x%X", cmd);

    if (!spca50x->dev)
	return -EIO;
#endif /* !__FreeBSD__ */
    switch (cmd) {
    case VIDIOCGCAP:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_capability *b = arg;
#else
	    struct video_capability j;
	    struct video_capability *b = &j;
#endif
	    PDEBUG(2, "VIDIOCGCAP %p :", b);

	    memset(b, 0, sizeof(struct video_capability));
	    snprintf(b->name, 32, "%s", camera_name(spca50x->desc));
	    b->type = VID_TYPE_CAPTURE;
	    b->channels = ((PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_SPCA506) ? 8 : 1);
	    b->audios = 0;
	    b->maxwidth = spca50x->maxwidth;
	    b->maxheight = spca50x->maxheight;

	    b->minwidth = spca50x->minwidth;
	    b->minheight = spca50x->minheight;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, b, sizeof(struct video_capability)))
		return -EFAULT;
#endif

	    return 0;
	}
    case VIDIOCGCHAN:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_channel *v = arg;
#else
	    struct video_channel k;
	    struct video_channel *v = &k;
	    if (copy_from_user(v, arg, sizeof(struct video_channel)))
		return -EFAULT;
#endif
	    switch (PWC_SC(spca50x)->pwc_info.bridge) {
	    case BRIDGE_SPCA500:
	    case BRIDGE_SPCA501:
	    case BRIDGE_SPCA504:
	    case BRIDGE_SPCA504B:
	    case BRIDGE_SPCA504C:
	    case BRIDGE_SPCA508:
	    case BRIDGE_SPCA533:
	    case BRIDGE_SPCA536:
	    case BRIDGE_SPCA561:
	    case BRIDGE_SONIX:
	    case BRIDGE_ZC3XX:
	    case BRIDGE_CX11646:
	    case BRIDGE_TV8532:
	    case BRIDGE_ETOMS:
	    case BRIDGE_SN9CXXX:
	    case BRIDGE_MR97311:
	    case BRIDGE_PAC207:
		    snprintf(v->name, 32, "%s",
			     bridge_name(PWC_SC(spca50x)->pwc_info.bridge));
		    break;

	    case BRIDGE_SPCA505:
		    strncpy(v->name,
			    ((v->channel == 0) ? "SPCA505" : "Video In"),
			    32);
		    break;

	    case BRIDGE_SPCA506:
		    spca506_GetNormeInput(spca50x,
					  (__u16 *) & (v->norm),
					  (__u16 *) & (v->channel));

		    if (v->channel < 4) {
			snprintf(v->name, 32, "SPCA506-CBVS-%d",
				 v->channel);
		    } else {
			snprintf(v->name, 32, "SPCA506-S-Video-%d",
				 v->channel);
		    }

		    break;
	    }

	    v->flags = 0;
	    v->tuners = 0;
	    v->type = VIDEO_TYPE_CAMERA;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, v, sizeof(struct video_channel)))
		return -EFAULT;
#endif

	    return 0;
	}
    case VIDIOCSCHAN:
	{
	    /* 
	       There seems to be some confusion as to whether this is a struct 
	       video_channel or an int. This should be safe as the first element 
	       of video_channel is an int channel number
	       and we just ignore the rest 
	     */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_channel *v = arg;
#else
	    struct video_channel k;
	    struct video_channel *v = &k;
	    if (copy_from_user(v, arg, sizeof(struct video_channel)))
		return -EFAULT;
#endif
	    /* exclude hardware channel reserved */
	    if ((v->channel < 0) || (v->channel > 9) || (v->channel == 4)
		|| (v->channel == 5))
		return -EINVAL;
	    if (PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_SPCA506) {
		spca506_SetNormeInput(spca50x, v->norm, v->channel);
	    }


	    return 0;
	}
    case VIDIOCGPICT:	/* XXX todo: move to the pwc.c code */
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_picture *p = arg;
#else
	    struct video_picture p1;
	    struct video_picture *p = &p1;
#endif

#if !defined(__FreeBSD__)
	    p->depth = spca50x->frame[0].depth;
#endif
	    p->palette = spca50x->format;
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
	    if (!autoadjust)
		p->brightness = spca50x_get_brightness(spca50x) << 8;
#else				/* SPCA50X_ENABLE_EXPERIMENTAL */
	    p->brightness = spca50x_get_brightness(spca50x) << 8;
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */

	    PDEBUG(4, "VIDIOCGPICT: depth=%d, palette=%d", p->depth,
		   p->palette);

	    switch (PWC_SC(spca50x)->pwc_info.bridge) {
	    case BRIDGE_SPCA506:
		spca506_GetBrightContrastHueColors(spca50x, &p->brightness,
						   &p->contrast, &p->hue,
						   &p->colour);
		p->whiteness = 0;
		break;
	    case BRIDGE_PAC207:
		spca50x->brightness = pac207_getbrightness(spca50x);
		spca50x->contrast = pac207_getcontrast(spca50x);
		p->brightness = spca50x->brightness;
		p->contrast = spca50x->contrast;
		p->colour = 0;
		break;
	    case BRIDGE_SN9CXXX:
		//spca50x->brightness = sn9cxxx_getbrightness(spca50x);
		sn9cxxx_getcontrast(spca50x);
		//spca50x->colour = sn9cxxx_getcolors(spca50x);
		p->brightness = spca50x->brightness;
		p->contrast = spca50x->contrast;
		p->colour = spca50x->colour;
		break;
	    case BRIDGE_ETOMS:
		spca50x->brightness = Et_getbrightness(spca50x);
		spca50x->contrast = Et_getcontrast(spca50x);
		spca50x->colour = Et_getcolors(spca50x);
		p->brightness = spca50x->brightness;
		p->contrast = spca50x->contrast;
		p->colour = spca50x->colour;
		break;
	    case BRIDGE_SPCA561:
		spca561_getbrightness(spca50x);
		spca561_getcontrast(spca50x);
		p->brightness = spca50x->brightness;
		p->contrast = spca50x->contrast;
		break;

	    case BRIDGE_SPCA500:
		{
		    int tmp;

		    p->colour = 0;	//0x8169
		    p->whiteness = 0;
		    p->contrast = spca50x_reg_read(spca50x->dev, 0x0, 0x8168, 1) << 8;	//0x8162
		    p->brightness = spca50x_get_brightness(spca50x);

		    /* get hue */
		    tmp =
			(spca50x_reg_read(spca50x->dev, 0x0, 0x816b, 1) &
			 12) << 6;
		    tmp += spca50x_reg_read(spca50x->dev, 0x0, 0x816a, 1);
		    p->hue = tmp << 6;
		    PDEBUG(2, "brightness = %d", p->brightness);
		    break;
		}
	    case BRIDGE_SPCA501:
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
		    p->whiteness = spca50x_get_whiteness(spca50x) << 8;
		    if (autoadjust)
			p->brightness = p->whiteness;
		    p->contrast = ((spca50x_reg_read(spca50x->dev,
						     SPCA501_REG_CCDSP,
						     SPCA501_A11,
						     2) & 0xFF) -
				   spca50x->a_blue) << 8;
		    printk("Contrast = %d\n", p->contrast >> 8);
#else				/* SPCA50X_ENABLE_EXPERIMENTAL */
		    p->whiteness = 0;
		    p->contrast = 0;
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */
		    p->colour = (spca50x_reg_read(spca50x->dev,
						  SPCA501_REG_CCDSP, 0x11,
						  2) & 0xFF) << 8;
		    p->hue =
			(spca50x_reg_read
			 (spca50x->dev, SPCA501_REG_CCDSP, 0x13,
			  2) & 0xFF) << 8;
		    break;
	    case BRIDGE_SPCA536:
	    case BRIDGE_SPCA533:
	    case BRIDGE_SPCA504:
	    case BRIDGE_SPCA504B:
	    case BRIDGE_SPCA504C:
		    p->colour = sp5xxfw2_getcolors(spca50x);
		    p->whiteness = 0;
		    p->contrast = sp5xxfw2_getcontrast(spca50x);
		    p->brightness = spca50x_get_brightness(spca50x);
		    p->hue = 0;

		    PDEBUG(4,
			   "color: 0x%02x contrast: 0x%02x brightness: 0x%02x hue: 0x%02x",
			   p->colour, p->contrast, p->brightness, p->hue);
		    break;

#ifdef SPCA50X_ENABLE_EXP_BRIGHTNESS
	    case BRIDGE_SPCA505:
		    p->whiteness = (spca50x_reg_read(spca50x->dev, 6, 0x59, 2) * 10) << 8;
		    p->contrast = spca50x_reg_read(spca50x->dev, 5, 0xc2, 2) << 8;
		    p->colour = spca50x_reg_read(spca50x->dev, 5, 0xc2, 2) << 8;
		    p->hue = spca50x_reg_read(spca50x->dev, 5, 0xc1, 2) << 8;
		    break;
#endif				/* SPCA50X_ENABLE_EXP_BRIGHTNESS */
	    default:
		    /* default to not allowing any settings to change */
		    p->colour = spca50x->colour;
		    p->whiteness = spca50x->whiteness;
		    p->contrast = spca50x->contrast;
		    p->brightness = spca50x->brightness;
		    p->hue = spca50x->hue;
		    break;
	    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, p, sizeof(struct video_picture)))
		return -EFAULT;
#endif

	    return 0;
	}
    case VIDIOCSPICT:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_picture *p = arg;
#else
	    struct video_picture p1;
	    struct video_picture *p = &p1;
	    if (copy_from_user(p, arg, sizeof(struct video_picture)))
		return -EFAULT;
#endif


	    PDEBUG(4, "VIDIOCSPICT");
	    if(spca5xx_testPalSize(spca50x,p->palette,spca50x->width,spca50x->height) < 0)
	    	return -EINVAL;

	    if (spca50x->format != p->palette) {
/* FIXME test if the palette is available */
		PDEBUG(4, "Setting depth=%d, palette=%d", p->depth,
		       p->palette);
#if !defined(__FreeBSD__)
		int i;
		/* change the output palette the input stream is the same */
		/* no need to stop the camera streaming and restart */
		for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
		    spca50x->frame[i].depth = p->depth;
		    spca50x->frame[i].format = p->palette;
		}
#endif /* !__FreeBSD__ */
		spca50x->format = p->palette;
	    }

	    switch (PWC_SC(spca50x)->pwc_info.bridge) {
	    case BRIDGE_SPCA506:
		spca506_SetBrightContrastHueColors(spca50x, p->brightness,
						   p->contrast, p->hue,
						   p->colour);

		break;
	    case BRIDGE_PAC207:
		spca50x->contrast = p->contrast;
		spca50x->brightness = p->brightness;
		pac207_setbrightness(spca50x);
		pac207_setcontrast(spca50x);
		break;
	    case BRIDGE_SN9CXXX:
		spca50x->contrast = p->contrast;
		spca50x->brightness = p->brightness;
		sn9cxxx_setbrightness(spca50x);
		sn9cxxx_setcontrast(spca50x);
		break;
	    case BRIDGE_ETOMS:
		spca50x->contrast = p->contrast;
		spca50x->brightness = p->brightness;
		spca50x->colour = p->colour;
		Et_setbrightness(spca50x);
		Et_setcontrast(spca50x);
		Et_setcolors(spca50x);
		break;
	    case BRIDGE_ZC3XX:
		spca50x->contrast = p->contrast;
		spca50x->brightness = p->brightness;
		zc3xx_setbrightness(spca50x);
		zc3xx_setcontrast(spca50x);
		break;
	    case BRIDGE_SONIX:
		spca50x_set_brightness(spca50x, p->brightness >> 8);
		spca50x->contrast = p->contrast;
		sonix_setcontrast(spca50x);
		break;
	    case BRIDGE_TV8532:
		spca50x_set_brightness(spca50x, p->brightness >> 8);
		spca50x->contrast = p->contrast;
		tv8532_setcontrast(spca50x);
		break;
	    case BRIDGE_CX11646:
		spca50x_set_brightness(spca50x, p->brightness >> 8);
		spca50x->contrast = p->contrast;
		cx_setcontrast(spca50x);
		spca50x->colour = p->colour;
		break;
	    case BRIDGE_SPCA561:
		spca50x->brightness = p->brightness;
		spca50x->contrast = p->contrast;
		spca561_setbrightness(spca50x);
		spca561_setcontrast(spca50x);
		break;
	    case BRIDGE_SPCA500:
		{
		    int tmp;

		    spca50x_set_brightness(spca50x, p->brightness >> 8);	//((p.brightness >> 8) + 128) % 255);
		    spca50x_reg_write(spca50x->dev, 0x00, 0x8168,
				      p->contrast >> 8);

		    /* set hue */
		    tmp = spca50x_reg_read(spca50x->dev, 0x00, 0x816b, 1);
		    tmp = tmp & 0xf3;
		    tmp += (p->hue & 0xc000) >> 12;
		    spca50x_reg_write(spca50x->dev, 0x00, 0x816b, tmp);
		    spca50x_reg_write(spca50x->dev, 0x00, 0x816a, (p->hue & 0x3fff) >> 8);
		}
		break;
	    case BRIDGE_SPCA501:
		{
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
		    __u16 kred, kgreen, kblue;	//coeffs of color correction

		    if (autoadjust)
			p->whiteness = p->brightness;
		    spca50x_set_whiteness(spca50x, p->whiteness >> 8);

		    /* Setting "contrast" (offset to color correction parameters) */
		    p->contrast >>= 8;
		    kred = spca50x->a_red + p->contrast;
		    kgreen = spca50x->a_green + p->contrast;
		    kblue = spca50x->a_blue + p->contrast;
		    PDEBUG(4, "Setting red = %d, green = %d, blue = %d",
			   (__u8) (kred & 0xFF), (__u8) (kgreen & 0xFF),
			   (__u8) (kblue & 0xFF));
		    if (kred < 256 && kgreen < 256 && kblue < 256) {
			spca50x_reg_write(spca50x->dev, SPCA501_REG_CCDSP, SPCA501_A11, (kblue & 0xFF));
			spca50x_reg_write(spca50x->dev, SPCA501_REG_CCDSP, SPCA501_A21, (kgreen & 0xFF));
			spca50x_reg_write(spca50x->dev, SPCA501_REG_CCDSP, SPCA501_A31, (kred & 0xFF));
		    }
		    spca50x_reg_write(spca50x->dev, SPCA501_REG_CCDSP, 0x11, ((p->colour >> 8) & 0xFF));
		    spca50x_reg_write(spca50x->dev, SPCA501_REG_CCDSP, 0x13, ((p->hue >> 8) & 0xFF));
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */
		    /* Setting brightness */
		    PDEBUG(4, "Setting brightness = %d", p->brightness);
//// This call must be moved to better place
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
		    if (!autoadjust)
			spca50x_set_brightness(spca50x,
					       p->brightness >> 8);
#else				/* SPCA50X_ENABLE_EXPERIMENTAL */
		    spca50x_set_brightness(spca50x, p->brightness >> 8);
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */
		    break;
		}
	    case BRIDGE_SPCA536:
	    case BRIDGE_SPCA533:
	    case BRIDGE_SPCA504:
	    case BRIDGE_SPCA504B:
	    case BRIDGE_SPCA504C:
		{
		    spca50x_set_brightness(spca50x,
					   ((p->brightness >> 8) +
					    128) % 255);
		    spca50x->contrast = p->contrast;
		    spca50x->colour = p->colour;
		    sp5xxfw2_setcolors(spca50x);
		    sp5xxfw2_setcontrast(spca50x);
		    p->whiteness = 0;

		    break;
		}
	    case BRIDGE_SPCA505:
		{
		    spca50x_set_brightness(spca50x, p->brightness >> 8);
#if 0				//#ifdef SPCA50X_ENABLE_EXP_BRIGHTNESS

		    spca50x_reg_write(spca50x->dev, 6, 0x1f, 0x16);
		    spca50x_reg_write(spca50x->dev, 6, 0x20, 0x0);

		    spca50x_reg_write(spca50x->dev, 5, 0xc1, p->hue >> 8);
		    spca50x_reg_write(spca50x->dev, 5, 0xc2, p->contrast >> 8);
		    spca50x_reg_write(spca50x->dev, 5, 0xca, 0x0);
////I have no this type of camera. Please, check this and move this
// call to better place
		    spca50x_set_brightness(spca50x, p->brightness >> 8);
////
		    spca50x_reg_write(spca50x->dev, 6, 0x59, max((p->whiteness >> 8) / 10, 20));
#endif				/* SPCA50X_ENABLE_EXP_BRIGHTNESS */
		    break;
		}
	    default:
		{
		    /* default to not allowing any settings to change ?? */
		    p->colour = spca50x->colour;
		    p->whiteness = spca50x->whiteness;
		    p->contrast = spca50x->contrast;
		    p->brightness = spca50x->brightness;
		    p->hue = spca50x->hue;
		    break;
		}
	    }

	    spca50x->contrast = p->contrast;
	    spca50x->brightness = p->brightness;
	    spca50x->colour = p->colour;
	    spca50x->hue = p->hue;
	    spca50x->whiteness = p->whiteness;
#ifdef SPCA50X_ENABLE_EXPERIMENTAL
	    spca50x->nstable = 0;
	    spca50x->nunstable = 0;
#endif				/* SPCA50X_ENABLE_EXPERIMENTAL */

	    return 0;
	}
    case VIDIOCGCAPTURE:
	{
	    int *vf = arg;

	    PDEBUG(4, "VIDIOCGCAPTURE");
	    *vf = 0;
	    // no subcapture
	    return -EINVAL;
	}
    case VIDIOCSCAPTURE:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_capture *vc = arg;
#else
	    struct video_capture vc1;
	    struct video_capture *vc = &vc1;
	    if (copy_from_user(vc, arg, sizeof(struct video_capture)))
		return -EFAULT;
#endif

	    if (vc->flags)
		return -EINVAL;
	    if (vc->decimation)
		return -EINVAL;

	    return -EINVAL;
	}
    case VIDIOCSWIN:
	{
	    int result;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_window *vw = arg;
#else
	    struct video_window vw1;
	    struct video_window *vw = &vw1;
	    if (copy_from_user(vw, arg, sizeof(struct video_window)))
		return -EFAULT;
#endif

	    PDEBUG(3, "VIDIOCSWIN: width=%d, height=%d, flags=0x%x x %d y %d",
		   vw->width, vw->height, vw->flags, vw->x, vw->y);
            	if(spca5xx_testPalSize(spca50x,spca50x->format,vw->width,vw->height) < 0)
	    	return -EINVAL;

	    if (vw->x)
		return -EINVAL;
	    if (vw->y)
		return -EINVAL;
	    if (vw->width > (unsigned int) spca50x->maxwidth)
		return -EINVAL;
	    if (vw->height > (unsigned int) spca50x->maxheight)
		return -EINVAL;
	    if (vw->width < (unsigned int) spca50x->minwidth)
		return -EINVAL;
	    if (vw->height < (unsigned int) spca50x->minheight)
		return -EINVAL;


	    switch (PWC_SC(spca50x)->pwc_info.bridge) {
	    case BRIDGE_ZC3XX:
	    case BRIDGE_SONIX:
	    case BRIDGE_SN9CXXX:
	    case BRIDGE_ETOMS:
	    case BRIDGE_SPCA561:
	    case BRIDGE_SPCA504:
	    case BRIDGE_SPCA504B:
	    case BRIDGE_SPCA504C:
	    case BRIDGE_SPCA533:
	    case BRIDGE_SPCA536:
	    case BRIDGE_PAC207:
	    result =
		spca5xx_restartMode(spca50x, vw->width, vw->height, PWC_SC(spca50x)->vpalette);
		break;
	    default:
		result =
		    spca50x_mode_init_regs(spca50x, vw->width, vw->height,
					   PWC_SC(spca50x)->vpalette,
					   GET_EXT_MODES(PWC_SC(spca50x)->pwc_info.bridge));
	    }
#if !defined(__FreeBSD__)
	    if (result == 0) {
		spca50x->frame[0].width = vw->width;
		spca50x->frame[0].height = vw->height;
	    }
#endif /* !__FreeBSD__ */

	    return result;
	}
    case VIDIOCGWIN:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_window *vw = arg;
#else
	    struct video_window vw1;
	    struct video_window *vw = &vw1;
#endif

	    memset(vw, 0, sizeof(struct video_window));
	    vw->x = 0;
	    vw->y = 0;
#if !defined(__FreeBSD__)
	    vw->width = spca50x->frame[0].width;
	    vw->height = spca50x->frame[0].height;
#endif /* !__FreeBSD__ */
	    vw->flags = 0;

	    PDEBUG(4, "VIDIOCGWIN: %dx%d", vw->width, vw->height);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, vw, sizeof(struct video_capture)))
		return -EFAULT;
#endif

	    return 0;
	}
    case VIDIOCGMBUF:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_mbuf *vm = arg;
#else
	    struct video_mbuf vm1;
	    struct video_mbuf *vm = &vm1;
#endif
	    int i;
	    PDEBUG(2, "VIDIOCGMBUF: %p ", vm);
	    memset(vm, 0, sizeof(struct video_mbuf));
	    vm->size = SPCA50X_NUMFRAMES * MAX_DATA_SIZE;
	    vm->frames = SPCA50X_NUMFRAMES;

	    for (i = 0; i < SPCA50X_NUMFRAMES; i++) {
		vm->offsets[i] = MAX_DATA_SIZE * i;
	    }
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, vm, sizeof(struct video_mbuf)))
		return -EFAULT;
#endif

	    return 0;
	}
    case VIDIOCMCAPTURE:
	{
	    int depth;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_mmap *vm = arg;
#else
	    struct video_mmap vm1;
	    struct video_mmap *vm = &vm1;
	    if (copy_from_user(vm, arg, sizeof(struct video_mmap)))
		return -EFAULT;
#endif
	
	    PDEBUG(4, "CMCAPTURE");
	    PDEBUG(4, "CM frame: %d, size: %dx%d, format: %d",
		   vm->frame, vm->width, vm->height, vm->format);
		   
	    depth = spca5xx_get_depth(spca50x, vm->format);
	    if (!depth || depth < spca50x->min_bpp) {
		err("VIDIOCMCAPTURE: invalid format (%d)", vm->format);
		return -EINVAL;
	    }

	    if ((vm->frame < 0) || (vm->frame > 3)) {
		err("VIDIOCMCAPTURE: invalid frame (%d)", vm->frame);
		return -EINVAL;
	    }

	    if (vm->width > spca50x->maxwidth
		|| vm->height > spca50x->maxheight) {
		err("VIDIOCMCAPTURE: requested dimensions too big");
		return -EINVAL;
	    }
	    if (vm->width < spca50x->minwidth
		|| vm->height < spca50x->minheight) {
		err("VIDIOCMCAPTURE: requested dimensions too small");
		return -EINVAL;
	    }
	    if(spca5xx_testPalSize(spca50x,vm->format,vm->width,vm->height) < 0)
	    	return -EINVAL;

	    /* 
	     * If we are grabbing the current frame, let it pass
	     */
	    /* why should i wait here CSYNCHRO do the job
	       if (spca50x->frame[vm->frame].grabstate == FRAME_GRABBING)
	       {
		int ret;
	       PDEBUG (4, "MCAPTURE: already grabbing");
	       //      ret = wait_event_interruptible (spca50x->wq,
	       ret = wait_event_interruptible (spca50x->frame[vm->frame].wq,
	       (spca50x->frame[vm->frame].
	       grabstate != FRAME_GRABBING));
	       if (ret)
	       return -EINTR;

	       }
	     */
#if !defined(__FreeBSD__)	/* XXX watch out this one */
	    if ((spca50x->frame[vm->frame].width != vm->width) ||
		(spca50x->frame[vm->frame].height != vm->height) ||
		(spca50x->frame[vm->frame].format != vm->format)) {
		int ret;
		if (spca50x->bridge == BRIDGE_ZC3XX ||
		    spca50x->bridge == BRIDGE_SONIX ||
		    spca50x->bridge == BRIDGE_SN9CXXX ||
		    spca50x->bridge == BRIDGE_ETOMS ||
		    spca50x->bridge == BRIDGE_SPCA561 ||
		    spca50x->bridge == BRIDGE_SPCA504 ||
		    spca50x->bridge == BRIDGE_SPCA504B ||
		    spca50x->bridge == BRIDGE_SPCA504C ||
		    spca50x->bridge == BRIDGE_SPCA533 ||
		    spca50x->bridge == BRIDGE_SPCA536 ||
		    spca50x->bridge == BRIDGE_PAC207) {
		    ret =
			spca5xx_restartMode(spca50x, vm->width, vm->height,
					    vm->format);
		} else {
		    ret =
			spca50x_mode_init_regs(spca50x, vm->width,
					       vm->height, vm->format,
					       GET_EXT_MODES(spca50x->
							     bridge));

		}
		if (ret < 0)
		    return ret;
		spca50x->frame[vm->frame].width = vm->width;
		spca50x->frame[vm->frame].height = vm->height;
		spca50x->frame[vm->frame].format = vm->format;
		spca50x->frame[vm->frame].depth = depth;

	    }
#endif /* !__FreeBSD__ */
	    if (spca50x->autoexpo) {
		/* set the autoexpo here */
		switch (PWC_SC(spca50x)->pwc_info.bridge) {
		case BRIDGE_SPCA561:
		    spca561_setAutobright(spca50x);
		    break;
		case BRIDGE_ETOMS:
		    PDEBUG(5, "Etoms AutoBrightness");
		    Et_setAutobright(spca50x);
		    break;
		case BRIDGE_PAC207:
		    PDEBUG(5, "Pac207 AutoBrightness");
		    pac207_setAutobright(spca50x);
		    break;
#if 0
		case BRIDGE_SN9CXXX:
		    sn9c102p_setAutobright(spca50x);
		    break;
#endif
		}
	    }
#ifdef SPCA5XX_ENABLE_REGISTERPLAY
	    __u8 Rval = 0;
	    if (RegStrobe != 0) {
		if (RegStrobe == 1) {
		    if (PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_PAC207) {
			pac207_RegWrite(spca50x);
		    } else {
			Rval = RegValue & 0xFF;
			spca5xxRegWrite(spca50x->dev, 0xa0, Rval,
					(__u16) (RegAddress & 0xFFFF),
					NULL, 0);
		    }
		} else {
		    if (PWC_SC(spca50x)->pwc_info.bridge == BRIDGE_PAC207) {
			pac207_RegRead(spca50x);
		    } else {
			usb_rd_vend_dev(spca50x->dev, 0xa1, 0x01, (__u16) (RegAddress & 0xFFFF), &Rval, 1);	// read Lowbyte
			RegValue = Rval;
		    }
		}
		RegStrobe = 0;
	    }
#endif				/* SPCA5XX_ENABLE_REGISTERPLAY */

#if !defined(__FreeBSD__)
	    /* Mark it as ready */
	    spca50x->frame[vm->frame].grabstate = FRAME_READY;
#endif /* !__FreeBSD__ */

	    return 0;
	}
    case VIDIOCSYNC:
	{
#if !defined(__FreeBSD__)
	    int ret;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    unsigned int frame = *((unsigned int *) arg);
#else
	    unsigned int frame;
	    if (copy_from_user((void *) &frame, arg, sizeof(int)))
		return -EFAULT;
#endif

	    PDEBUG(4, "syncing to frame %d, grabstate = %d", frame,
		   spca50x->frame[frame].grabstate);

	    switch (spca50x->frame[frame].grabstate) {
	    case FRAME_UNUSED:
		return -EINVAL;
	    case FRAME_ABORTING:
		return -ENODEV;
	    case FRAME_READY:
	    case FRAME_GRABBING:
	    case FRAME_ERROR:
	      redo:
		if (!spca50x->dev)
		    return -EIO;

		ret = wait_event_interruptible(spca50x->frame[frame].wq,
					       (spca50x->frame[frame].
						grabstate == FRAME_DONE));
		if (ret)
		    return -EINTR;

		PDEBUG(4, "Synch Ready on frame %d, grabstate = %d",
		       frame, spca50x->frame[frame].grabstate);

		if (spca50x->frame[frame].grabstate == FRAME_ERROR) {
		    goto redo;
		}
		/* Fallthrough.
		 * We have waited in state FRAME_GRABBING until it
		 * becomes FRAME_DONE, so now we can move along.
		 */
	    case FRAME_DONE:



		/* Release the current frame. This means that it
		 * will be reused as soon as all other frames are
		 * full, so the app better be done with it quickly.
		 * Can this be avoided somehow?
		 */
		spca50x->frame[frame].grabstate = FRAME_UNUSED;
		PDEBUG(4, "Release frame %d state %d\n", frame,
		       spca50x->frame[frame].grabstate);
		break;
	    }			/* end switch */
#endif /* !__FreeBSD__ */
	    return 0;
	}
    case VIDIOCGFBUF:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_buffer *vb = arg;
#else
	    struct video_buffer vb1;
	    struct video_buffer *vb = &vb1;
#endif
	    memset(vb, 0, sizeof(struct video_buffer));
	    vb->base = NULL;	/* frame buffer not supported, not used */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, vb, sizeof(struct video_buffer)))
		return -EFAULT;
#endif

	    return 0;
	}
    case SPCAGVIDIOPARAM:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_param *vp = arg;
#else
	    struct video_param vp1;
	    struct video_param *vp = &vp1;
#endif
	    vp->autobright = (__u8) spca50x->autoexpo;
	    vp->quality = (__u8) spca50x->qindex;
	    vp->time_interval = (__u16) spca50x->dtimes;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,22)
	    if (copy_to_user(arg, vp, sizeof(struct video_param)))
		return -EFAULT;
#endif
	    return 0;
	}

    case SPCASVIDIOPARAM:
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,22)
	    struct video_param *vp = arg;
#else
	    struct video_param vp1;
	    struct video_param *vp = &vp1;
	    if (copy_from_user(vp, arg, sizeof(struct video_param)))
		return -EFAULT;
#endif
	    switch (vp->chg_para) {
	    case CHGABRIGHT:
		spca5xx_chgAuto(spca50x, vp->autobright);
		break;
	    case CHGQUALITY:
		spca5xx_chgQtable(spca50x, vp->quality);
		break;
	    case CHGTINTER:
		spca5xx_chgDtimes(spca50x, vp->time_interval);
		break;
	    default:
		return -EINVAL;
		break;
	    }
	    return 0;
	}
/************************************/
    case VIDIOCKEY:
	return 0;
    case VIDIOCCAPTURE:
	return -EINVAL;
    case VIDIOCSFBUF:
	return -EINVAL;
    case VIDIOCGTUNER:
    case VIDIOCSTUNER:
	return -EINVAL;
    case VIDIOCGFREQ:
    case VIDIOCSFREQ:
	return -EINVAL;
    case VIDIOCGAUDIO:
    case VIDIOCSAUDIO:
	return -EINVAL;
    default:
	return -ENOIOCTLCMD;
    }				/* end switch */

    return 0;
}

#if !defined(__FreeBSD__)

static ssize_t
spca5xx_read(struct file *file, char *buf, size_t cnt, loff_t * ppos)
{
    ... wait for a frame t be ready
    ... trim and copy to userspace
    /* set the autoexpo here */
    if (spca50x->autoexpo) {
	/* set the autoexpo here */
	if (spca50x->bridge == BRIDGE_SPCA561)
	    spca561_setAutobright(spca50x);
	if (spca50x->bridge == BRIDGE_ETOMS) {
	    PDEBUG(5, "Etoms AutoBrightness");
	    Et_setAutobright(spca50x);
	}
	if (spca50x->bridge == BRIDGE_PAC207) {
	    PDEBUG(5, "Pac207 AutoBrightness");
            pac207_setAutobright(spca50x);
	}
    }
    return count;
}

#endif /* !__FreeBSD__ */

/****************************************************************************
 *
 * SPCA50X configuration
 *
 ***************************************************************************/

#define	_SENS_MINMAX(wmin, hmin, wmax, hmax)	do {	\
	    spca50x->maxwidth = wmax;			\
	    spca50x->maxheight = hmax;			\
	    spca50x->minwidth = wmin;			\
	    spca50x->minheight = hmin;			\
	} while (0)

static int spca50x_configure_sensor(struct usb_spca50x *spca50x)
{
    int err;
    PDEBUG(4, "starting configuration");
    /* Set sensor-specific vars */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_SPCA500:
	if (spca50x->desc == LogitechClickSmart310) {
	    _SENS_MINMAX(176, 144, 352, 288);
	} else {
	    _SENS_MINMAX(176, 144, 640, 480);
	}
	break;

    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_ETOMS:
    case BRIDGE_ZC3XX:
    case BRIDGE_SONIX:
    case BRIDGE_SN9CXXX:
    case BRIDGE_SPCA561:
    case BRIDGE_MR97311:
    case BRIDGE_PAC207:
    case BRIDGE_STV0602:
	err = spca5xx_getcapability(spca50x);
	break;

    case BRIDGE_CX11646:
	_SENS_MINMAX(176, 144, 640, 480);
	break;

    case BRIDGE_SPCA501:
    case BRIDGE_SPCA506:
	_SENS_MINMAX(160, 120, 640, 480);
	break;

    case BRIDGE_SPCA505:
	if (spca50x->desc == Nxultra) {
	    _SENS_MINMAX(160, 120, 640, 480);
	} else {
	    _SENS_MINMAX(160, 120, 352, 288);
	}
	break;

    case BRIDGE_SPCA508:
	_SENS_MINMAX(160, 120, 352, 288);
	break;

    case BRIDGE_TV8532:
	_SENS_MINMAX(176, 144, 352, 288);
	break;

    default:
	_SENS_MINMAX(160, 120, 640, 480);
	break;
    }
    return 0;
}


int spca50x_configure(struct usb_spca50x *spca50x)
{
    int i;
    int defaultcols = 0;
    int defaultrows = 0;
    int defaultpipe = 0;
    int error = 0;

/* DEPRECATED spca5xx_getDefaultMode() should be use instead */
    /* dynamically look up the smallest available frame size */
    /* That is not really true with Hardware size need smallest and hardware */
    /* Also give the pipe size for the hardware resolution */
    i = spca50x_smallest_mode_index(GET_EXT_MODES(PWC_SC(spca50x)->pwc_info.bridge),
				    &defaultcols, &defaultrows,
				    &defaultpipe);
    if (i < 0)
	return i;

    PDEBUG(2, "video_register_device succeeded");
    /* Initialise the camera bridge */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {

    case BRIDGE_SONIX:
    case BRIDGE_SN9CXXX:
	error = sonix_config(spca50x);
	break;

    case BRIDGE_ZC3XX:
	error = zc3xx_config(spca50x);
	break;

    case BRIDGE_ETOMS:
	error = Etxx_config(spca50x);
	break;

    case BRIDGE_SPCA561:
	error = spca561_config(spca50x);
	break;

    case BRIDGE_MR97311:
	error = mr97311_config(spca50x);
	break;

    case BRIDGE_SPCA536:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA504C:
	error = sp5xxfw2_config(spca50x);
	break;

    case BRIDGE_PAC207:
	error = pac207_config(spca50x);
	break;

    case BRIDGE_CX11646:
    case BRIDGE_SPCA500:
	spca50x->qindex = 5;
	break;

    case BRIDGE_TV8532:
	tv8532_configure(spca50x);
	break;

    case BRIDGE_SPCA501:
	    if (spca50x->desc == Arowana300KCMOSCamera ||
		spca50x->desc == SmileIntlCamera) {
		/* Arowana 300k CMOS Camera data */
		error = spca50x_write_vector (spca50x, spca501c_arowana_init_data);
	    } else if (spca50x->desc == MystFromOriUnknownCamera) {
		/* UnKnow Ori CMOS Camera data */
		error = spca50x_write_vector(spca50x, spca501c_mysterious_open_data);
	    } else {
		/* generic spca501 init data */
		error = spca50x_write_vector(spca50x, spca501_init_data);
	    }
	    break;

    case BRIDGE_SPCA505:
	    /* Determine the sensor type for cameras with an internal CCD or external input */
	    if (spca50x->desc == Nxultra) {
		error = spca50x_write_vector(spca50x, spca505b_init_data);
	    } else {
		error = spca50x_write_vector(spca50x, spca505_init_data);
	    }
	    break;

    case BRIDGE_SPCA506:
	    break;

    case BRIDGE_SPCA508:
	    error = config_spca508(spca50x);
	    break;

    case BRIDGE_STV0602:
	error = config_quickcam(spca50x);
	break;

    case BRIDGE_EM2820:
	error = em2820_config(spca50x);
	break;

    default:
	    err("Unknown bridge type in spca_configure\n");
	    error = -1;
    }
    if (error)
	goto error;

    spca50x_set_packet_size(spca50x, 0);
    /* Set an initial pipe size; this will be overridden by
     * spca50x_set_mode(), called indirectly by the open routine.
     */
    switch (PWC_SC(spca50x)->pwc_info.bridge) {
    case BRIDGE_ZC3XX:
    case BRIDGE_SONIX:
    case BRIDGE_SN9CXXX:
    case BRIDGE_ETOMS:
    case BRIDGE_SPCA561:
    case BRIDGE_SPCA504:
    case BRIDGE_SPCA504B:
    case BRIDGE_SPCA504C:
    case BRIDGE_SPCA533:
    case BRIDGE_SPCA536:
    case BRIDGE_STV0602:
    case BRIDGE_EM2820:
    case BRIDGE_PAC207:
	i = spca5xx_getDefaultMode(spca50x);
	if (i < 0)
	    return i;
	break;
    default:
	spca50x->pipe_size = defaultpipe;
	spca50x->width = defaultcols;
	spca50x->height = defaultrows;
    }
    spca50x->force_rgb = force_rgb;
    spca50x->min_bpp = min_bpp;
    spca50x->lum_level = lum_level;

    if (spca50x_configure_sensor(spca50x) < 0) {
	err("failed to configure");
	goto error;
    }
    /* configure the frame detector with default parameters */

    spca5xx_setFrameDecoder(spca50x);

    PDEBUG(2, "Spca5xx Configure done !!");
    return 0;

  error:

    return -EBUSY;
}

/**************************************************************************
All That stuff need to be rewrite to go in spca500.h
***************************************************************************/
static void spca500_ping310(struct usb_spca50x *spca50x)
{
    __u8 Data[2] = { 0, 0 };
    usb_rd_vend_dev(spca50x->dev, 0, 0, 0x0d04, Data, 2);
    PDEBUG(5, "ClickSmart310 ping 0x0d04 0x%02X  0x%02X ", Data[0],
	   Data[1]);
}

static int spca500_synch310(struct usb_spca50x *spca50x)
{
/* Synchro the Bridge with sensor */
/* Maybe that will work on all spca500 chip */
/* because i only own a clicksmart310 try for that chip */
/* using spca50x_set_packet_size() cause an Ooops here */
/* usb_set_interface from kernel 2.6.x clear all the urb stuff */
/* up-port the same feature as in 2.4.x kernel */

    __u8 Data;


    if (spca_set_interface(spca50x->dev, spca50x->iface, 0) < 0) {
	err("Set packet size: set interface error");
	goto error;
    }
    spca500_ping310(spca50x);

    usb_rd_vend_dev(spca50x->dev, 0, 0, 0x0d00, &Data, 1);

    /* need alt setting here */
    PDEBUG(5, "ClickSmart310 sync pipe %d altsetting %d ",
	   spca50x->pipe_size, spca50x->alt);
    /* Windoze use pipe with altsetting 6 why 7 here */
    if (spca_set_interface(spca50x->dev, spca50x->iface, spca50x->alt) < 0) {
	err("Set packet size: set interface error");

	goto error;

    }

    return 0;
  error:

    return -EBUSY;
}

static void spca500_clksmart310_init(struct usb_spca50x *spca50x)
{
    __u8 Data[2] = { 0, 0 };
    usb_rd_vend_dev(spca50x->dev, 0, 0, 0x0d05, Data, 2);
    PDEBUG(5, "ClickSmart310 init 0x0d05 0x%02X  0x%02X ", Data[0],
	   Data[1]);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8167, 0x5a);
    spca500_ping310(spca50x);

    spca50x_reg_write(spca50x->dev, 0x00, 0x8168, 0x22);
    spca50x_reg_write(spca50x->dev, 0x00, 0x816a, 0xc0);
    spca50x_reg_write(spca50x->dev, 0x00, 0x816b, 0x0b);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8169, 0x25);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8157, 0x5b);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8158, 0x5b);
    spca50x_reg_write(spca50x->dev, 0x00, 0x813f, 0x03);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8151, 0x4a);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8153, 0x78);
    spca50x_reg_write(spca50x->dev, 0x00, 0x0d01, 0x04);	//00 for adjust shutter
    spca50x_reg_write(spca50x->dev, 0x00, 0x0d02, 0x01);
    spca50x_reg_write(spca50x->dev, 0x00, 0x8169, 0x25);
    spca50x_reg_write(spca50x->dev, 0x00, 0x0d01, 0x02);
}

void spca500_reinit(struct usb_spca50x *spca50x)
{
    int err;
    __u8 Data;

    // some unknow command from Aiptek pocket dv and family300

    spca50x_reg_write(spca50x->dev, 0x00, 0x0d01, 0x01);
    spca50x_reg_write(spca50x->dev, 0x00, 0x0d03, 0x00);
    spca50x_reg_write(spca50x->dev, 0x00, 0x0d02, 0x01);

    /* enable drop packet */
    spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001);

    err =
	spca50x_setup_qtable(spca50x, 0x00, 0x8800, 0x8840,
			     qtable_pocketdv);
    if (err < 0) {
	PDEBUG(2, "spca50x_setup_qtable failed on init");
    }
    /* set qtable index */
    spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 2);
    /* family cam Quicksmart stuff */
    spca50x_reg_write(spca50x->dev, 0x00, 0x800a, 0x00);
    //Set agc transfer: synced inbetween frames
    spca50x_reg_write(spca50x->dev, 0x00, 0x820f, 0x01);
    //Init SDRAM - needed for SDRAM access
    spca50x_reg_write(spca50x->dev, 0x00, 0x870a, 0x04);
    /*Start init sequence or stream */

    spca50x_reg_write(spca50x->dev, 0, 0x8003, 0x00);
    /* switch to video camera mode */
    err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);
    wait_ms(2000);
    if (spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44) != 0)

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
    spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

}

static int spca500_initialise(struct usb_spca50x *spca50x)
{

    int index;
    int err;
    __u8 Data;
    int bridge = PWC_SC(spca50x)->pwc_info.bridge;

    /* read mode index */
    index = spca50x_find_mode_index(spca50x->width, spca50x->height,
				    GET_EXT_MODES(bridge));
    if (index == -EINVAL) {
	PDEBUG(2, "Can't find proper mode?!!");
	return -EINVAL;
    }
    /* EXPERIMENTAL Wait until test Marek */
    index = index & 0x0F;
    spca50x_set_mode(spca50x, index, GET_EXT_MODES(bridge));

    // spca50x_write_vector(spca50x, spca500_read_stats);

    /* is there a sensor here ? */
    usb_rd_vend_dev(spca50x->dev, 0, 0, 0x8a04, &Data, 1);
    PDEBUG(0, "Spca500 Sensor Address  0x%02X ", Data);

    /* setup qtable */
    switch (spca50x->desc) {
    case LogitechClickSmart310:
	//err = spca500_synch310(spca50x );

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x0d00, &Data, 1);

	/* set x multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8001, GET_EXT_MODES(bridge)[index][3]);

	/* set y multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8002, GET_EXT_MODES(bridge)[index][4]);

	/* use compressed mode, VGA, with mode specific subsample */
	spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));

	/* enable drop packet */
	if ((err = spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001)) < 0) {
	    PDEBUG(2, "failed to enable drop packet");
	    return err;
	}
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 3);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write failed");
	    return err;
	}
	err = spca50x_setup_qtable(spca50x,
				   0x00, 0x8800, 0x8840,
				   qtable_creative_pccam);
	if (err < 0) {
	    PDEBUG(2, "spca50x_setup_qtable failed");
	    return err;
	}
	//Init SDRAM - needed for SDRAM access
	spca50x_reg_write(spca50x->dev, 0x00, 0x870a, 0x04);



	/* switch to video camera mode */
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write camera mode failed");
	    return err;
	}
	wait_ms(500);
	if (spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44) != 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return -EIO;
	}

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

	err = spca500_synch310(spca50x);

	spca50x_write_vector(spca50x, spca500_visual_defaults);

	/* set x multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8001, GET_EXT_MODES(bridge)[index][3]);

	/* set y multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8002, GET_EXT_MODES(bridge)[index][4]);

	/* use compressed mode, VGA, with mode specific subsample */
	spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));

	/* enable drop packet */
	if ((err = spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001)) < 0) {
	    PDEBUG(2, "failed to enable drop packet");
	    return err;
	}
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 3);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write failed");
	    return err;
	}
	err = spca50x_setup_qtable(spca50x,
				   0x00, 0x8800, 0x8840,
				   qtable_creative_pccam);
	if (err < 0) {
	    PDEBUG(2, "spca50x_setup_qtable failed");
	    return err;
	}
	//Init SDRAM - needed for SDRAM access
	spca50x_reg_write(spca50x->dev, 0x00, 0x870a, 0x04);


	/* switch to video camera mode */
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write camera mode failed");
	    return err;
	}

	if (spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44) != 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return -EIO;
	}

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

	break;

    case CreativePCCam300:	/* Creative PC-CAM 300 */
    case IntelPocketPCCamera:	/* FIXME: Temporary fix for Intel Pocket PC Camera - NWG (Sat 29th March 2003) */

	/* do a full reset */
	if ((err = spca500_full_reset(spca50x)) < 0) {
	    PDEBUG(2, "spca500_full_reset failed");
	    return err;
	}

	if ((err = spca50x_reg_write(spca50x->dev, 0xe0, 0x0000, 0x0000)) < 0) {
	    PDEBUG(2, "spca50x_reg_write() failed");
	    return err;
	}
	if ((err = spca50x_reg_readwait(spca50x->dev, 0x06, 0, 0)) < 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return err;
	}

	/* enable drop packet */
	if ((err = spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001)) < 0) {
	    PDEBUG(2, "failed to enable drop packet");
	    return err;
	}
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 3);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write failed");
	    return err;
	}
	err = spca50x_setup_qtable(spca50x,
				   0x00, 0x8800, 0x8840,
				   qtable_creative_pccam);
	if (err < 0) {
	    PDEBUG(2, "spca50x_setup_qtable failed");
	    return err;
	}

	/* set x multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8001, GET_EXT_MODES(bridge)[index][3]);

	/* set y multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8002, GET_EXT_MODES(bridge)[index][4]);

	/* use compressed mode, VGA, with mode specific subsample */
	spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));

	spca50x_reg_write(spca50x->dev, 0x20, 0x0001, 0x0004);

	/* switch to video camera mode */
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write camera mode failed");
	    return err;
	}

	if (spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44) != 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return -EIO;
	}

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

	spca50x_write_vector(spca50x, spca500_visual_defaults);

	break;
    case KodakEZ200:		/* Kodak EZ200 */

	/* do a full reset */
	if ((err = spca500_full_reset(spca50x)) < 0) {
	    PDEBUG(2, "spca500_full_reset failed");
	    return err;
	}

	if ((err = spca50x_reg_write(spca50x->dev, 0xe0, 0x0000, 0x0000)) < 0) {
	    PDEBUG(2, "spca50x_reg_write() failed");
	    return err;
	}
	if ((err = spca50x_reg_readwait(spca50x->dev, 0x06, 0, 0)) < 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return err;
	}

	/* enable drop packet */
	if ((err = spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001)) < 0) {
	    PDEBUG(2, "failed to enable drop packet");
	    return err;
	}
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 0);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write failed");
	    return err;
	}
	err = spca50x_setup_qtable(spca50x,
				   0x00, 0x8800, 0x8840,
				   qtable_kodak_ez200);
	if (err < 0) {
	    PDEBUG(2, "spca50x_setup_qtable failed");
	    return err;
	}
	/* set x multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8001,
			  GET_EXT_MODES(bridge)[index][3]);

	/* set y multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8002,
			  GET_EXT_MODES(bridge)[index][4]);

	/* use compressed mode, VGA, with mode specific subsample */
	spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));

	spca50x_reg_write(spca50x->dev, 0x20, 0x0001, 0x0004);

	/* switch to video camera mode */
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);
	if (err < 0) {
	    PDEBUG(2, "spca50x_reg_write camera mode failed");
	    return err;
	}

	if (spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44) != 0) {
	    PDEBUG(2, "spca50x_reg_readwait() failed");
	    return -EIO;
	}

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

	//spca50x_write_vector(spca50x, spca500_visual_defaults);

	break;

    case BenqDC1016:
    case DLinkDSC350:		/* FamilyCam 300 */
    case AiptekPocketDV:	/* Aiptek PocketDV */
    case Gsmartmini:		/*Mustek Gsmart Mini */
    case MustekGsmart300:	// Mustek Gsmart 300
    case PalmPixDC85:
    case Optimedia:
    case ToptroIndus:
    case AgfaCl20:

	spca500_reinit(spca50x);
	spca50x_reg_write(spca50x->dev, 0x00, 0x0d01, 0x01);
	/* enable drop packet */
	spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001);

	err = spca50x_setup_qtable(spca50x, 0x00, 0x8800, 0x8840, qtable_pocketdv);
	if (err < 0) {
	    PDEBUG(2, "spca50x_setup_qtable failed");
	    return err;
	}
	spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 2);

	/* familycam Quicksmart pocketDV stuff */
	spca50x_reg_write(spca50x->dev, 0x00, 0x800a, 0x00);
	//Set agc transfer: synced inbetween frames
	spca50x_reg_write(spca50x->dev, 0x00, 0x820f, 0x01);
	//Init SDRAM - needed for SDRAM access
	spca50x_reg_write(spca50x->dev, 0x00, 0x870a, 0x04);


	/* set x multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8001, GET_EXT_MODES(bridge)[index][3]);

	/* set y multiplier */
	spca50x_reg_write(spca50x->dev, 0, 0x8002, GET_EXT_MODES(bridge)[index][4]);

	/* use compressed mode, VGA, with mode specific subsample */
	spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));

	/* switch to video camera mode */
	err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);


	spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44);

	usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);

	break;
    case LogitechTraveler:
    case LogitechClickSmart510:
	{

	    spca50x_reg_write(spca50x->dev, 0x02, 0x00, 0x00);
	    /* enable drop packet */
	    if ((err = spca50x_reg_write(spca50x->dev, 0x00, 0x850a, 0x0001)) < 0) {
		PDEBUG(2, "failed to enable drop packet");
		return err;
	    }

	    err = spca50x_setup_qtable(spca50x, 0x00, 0x8800, 0x8840, qtable_creative_pccam);
	    if (err < 0) {
		PDEBUG(2, "spca50x_setup_qtable failed");
		return err;
	    }
	    err = spca50x_reg_write(spca50x->dev, 0x00, 0x8880, 3);
	    if (err < 0) {
		PDEBUG(2, "spca50x_reg_write failed");
		return err;
	    }
	    spca50x_reg_write(spca50x->dev, 0x00, 0x800a, 0x00);
	    //Init SDRAM - needed for SDRAM access
	    spca50x_reg_write(spca50x->dev, 0x00, 0x870a, 0x04);
	    /* set x multiplier */
	    spca50x_reg_write(spca50x->dev, 0, 0x8001, GET_EXT_MODES(bridge)[index][3]);

	    /* set y multiplier */
	    spca50x_reg_write(spca50x->dev, 0, 0x8002, GET_EXT_MODES(bridge)[index][4]);

	    /* use compressed mode, VGA, with mode specific subsample */
	    spca50x_reg_write(spca50x->dev, 0, 0x8003, ((GET_EXT_MODES(bridge)[index][2] & 0x0f) << 4));


	    /* switch to video camera mode */
	    err = spca50x_reg_write(spca50x->dev, 0x00, 0x8000, 0x0004);

	    spca50x_reg_readwait(spca50x->dev, 0, 0x8000, 0x44);

	    usb_rd_vend_dev(spca50x->dev, 0, 0, 0x816b, &Data, 1);
	    spca50x_reg_write(spca50x->dev, 0x00, 0x816b, Data);
	    spca50x_write_vector(spca50x, Clicksmart510_defaults);
	}

	break;

    default:
	return -EINVAL;
    }

    /* all ok */
    return 0;
}

static int spca500_full_reset(struct usb_spca50x *spca50x)
{
    int err;

    /* send the reset command */
    err = spca50x_reg_write(spca50x->dev, 0xe0, 0x0001, 0x0000);
    if (err < 0) {
	return err;
    }

    /* wait for the reset to complete */
    err = spca50x_reg_readwait(spca50x->dev, 0x06, 0x0000, 0x0000);
    if (err < 0) {
	return err;
    }

    /* all ok */
    return 0;
}

/*
 *  camera detection - handler for lookup in probe/match/attach.
 */
int spcaDetectCamera(int vendor, int product, struct pwc_softc *sc)
{
    struct usb_spca50x *spca50x = &sc->spca50x;
    struct usb_device *dev = spca50x->dev;

    printf("%s called vend 0x%x prod 0x%x p %p\n",
	__FUNCTION__, vendor, product, spca50x);

    bzero(spca50x, sizeof(struct usb_spca50x));
    bzero(&sc->pwc_info, sizeof(struct pwc_info));
    bzero(&sc->camera_cb, sizeof(struct camera_callbacks));

    spca50x->desc = UnknownCamera;	/* this is the error case */

    /* Initialize pwc_info and callbacks as appropriate */
    sc->pwc_info.cb = &sc->camera_cb;

    sc->camera_cb = spca_callbacks;

    sc->pwc_info.devno.ud_vendor = vendor;
    sc->pwc_info.devno.ud_product = product;

    /* We switch on vendor and product, set spca50x->desc
     * if we find something.
     * Fields that are 0 can be left untouched.
     * The _CAM(desc, bridge, sensor, type) macro can help shorten the code.
     */
#define _CAM(_desc, _bridge, _sensor, _type)	do {	\
		spca50x->desc = (_desc);		\
		sc->pwc_info.bridge = (_bridge);		\
		sc->pwc_info.sensor = (_sensor);		\
		spca50x->cameratype = (_type);		\
	} while (0)

    switch (vendor) {
    case 0xeb1a:	/* silvercrest */
	switch (product) {
	case 0x2710:
	case 0x2820:
		_CAM(SilverCrest, BRIDGE_EM2820, SENSOR_PAS202, JPEG);
		break;
	}
	break;

    case 0x0733:		/* Rebadged ViewQuest (Intel) and ViewQuest cameras */
	switch (product) {
	case 0x430:
	    if (usbgrabber) {
		_CAM(UsbGrabberPV321c, BRIDGE_SPCA506, SENSOR_SAA7113, YYUV);
		spca50x->header_len = SPCA50X_OFFSET_DATA;
		spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
		info("USB SPCA5XX grabber found. Daemon PV321c(SPCA506+SAA7113)");
	    } else {
		_CAM(IntelPCCameraPro, BRIDGE_SPCA505, SENSOR_INTERNAL, YYUV);
		spca50x->header_len = SPCA50X_OFFSET_DATA;
		spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
		info("USB SPCA5XX camera found. Type Intel PC Camera Pro (SPCA505)");
	    }
	    break;

	case 0x1314:
	    _CAM(Mercury21, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mercury Digital Pro 2.1Mp ");
	    break;

	case 0x2211:
	    _CAM(Jenoptikjdc21lcd, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Jenoptik JDC 21 LCD");
	    break;

	case 0x2221:
	    _CAM(MercuryDigital, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mercury Digital Pro 3.1Mp ");
	    break;

	case 0x1311:
	    _CAM(Epsilon13, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Digital Dream Epsilon 1.3");
	    break;

	case 0x401:
	    _CAM(IntelCreateAndShare, BRIDGE_SPCA501, SENSOR_INTERNAL, YUYV);
	    // BRIDGE_SPCA501 is a guess. At least the chip looks closer to the 501 than the 505 */

	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Intel Create and Share (SPCA501 )");
	    break;
	case 0x402:
	    _CAM(ViewQuestM318B, BRIDGE_SPCA501, SENSOR_INTERNAL, YUYV);
	    // BRIDGE_SPCA501; is a guess. At least the chip looks closer to the 501 than the 505 */
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found.  ViewQuest M318B (SPCA501a)");
	    /*
	       Looks not perfectly but until we understand the difference
	       between spca501 and spca500 we'll treat them as one
	     */
	    break;

	case 0x110:
	    _CAM(ViewQuestVQ110, BRIDGE_SPCA508, SENSOR_INTERNAL, YUVY);
	    spca50x->header_len = SPCA508_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = SPCA508_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    info("USB SPCA5XX camera found. Type ViewQuest (SPCA508?)");
	    break;

	case 0x3261:
	    _CAM(Concord3045, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found.Concord 3045 Spca536 Mpeg4");
	    break;

	case 0x3281:
	    _CAM(CyberpixS550V, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found.Mercury Cyberpix Spca536 Mpeg4");
	    break;
	};
	break;

    case 0x0734:
	switch (product) {
	case 0x043b:
	    _CAM(DeMonUSBCapture, BRIDGE_SPCA506, SENSOR_SAA7113, YYUV);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("Detected DeMonUsbCapture  (SPCA506+SAA7113)");
	    break;
	};
	break;

    case 0x99FA:		/* GrandTec cameras */
	switch (product) {
	case 0x8988:
	    _CAM(GrandtecVcap, BRIDGE_SPCA506, SENSOR_SAA7113, YYUV);
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Grandtec V.cap (SPCA506+SAA7113?)");
	    break;
	};
	break;

    case 0x0AF9:		/* Hama cameras */
	switch (product) {
	case 0x0010:
	    _CAM(HamaUSBSightcam, BRIDGE_SPCA508, SENSOR_INTERNAL, YUVY);
	    spca50x->header_len = SPCA508_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Hama Sightcam 100 (SPCA508A+PAS106B)");
	    break;
	case 0x0011:
	    _CAM(HamaUSBSightcam2, BRIDGE_SPCA508, SENSOR_INTERNAL, YUVY);
	    spca50x->header_len = SPCA508_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Hama Sightcam 100 (2) (SPCA508A)");
	    break;
	};
	break;
    case 0x040A:		/* Kodak cameras */
	switch (product) {
	case 0x0002:
	    _CAM(KodakDVC325, BRIDGE_SPCA501, SENSOR_INTERNAL, YUYV);
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Kodak DVC-325 (SPCA501A )");
	    break;
	case 0x0300:
	    _CAM(KodakEZ200, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Kodak EZ200 (SPCA500+unknown CCD)");
	    break;
	};
	break;
    case 0x04a5:		/* Benq */
    case 0x08ca:		/* Aiptek */
    case 0x055f:		/* Mustek cameras */
    case 0x04fc:		/* SunPlus */
    case 0x052b:		/* ?? Megapix */
    case 0x04f1:		/* JVC */
	switch (product) {
	case 0xc520:
	    _CAM(MustekGsmartMini3, BRIDGE_SPCA504, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Type Mustek gSmart Mini 3(SPCA504A)");
	    break;
	case 0xc420:
	    _CAM(MustekGsmartMini2, BRIDGE_SPCA504, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Type Mustek gSmart Mini 2(SPCA504A)");
	    break;

	case 0xc360:
	    _CAM(MustekDV4000, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mustek DV4000 Spca536 Mpeg4");
	    break;

	case 0xc211:
	    _CAM(Bs888e, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Kowa Bs-888e Spca536 Mpeg4");
	    break;

	case 0xc005:		// zc302 chips 
	    _CAM(Wcam300A, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Mustek Wcam300a Zc0301 ");
	    break;

	case 0xd003:		// zc302 chips 
	    _CAM(MustekWcam300A, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Mustek PCCam300a Zc0301 ");
	    break;
	case 0xd004:		// zc302 chips 
	    _CAM(WCam300AN, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Mustek WCam300aN Zc0302 ");
	    break;
	case 0x504a: {
    	    __u8 fw = 0;
	    /*try to get the firmware as some cam answer 2.0.1.2.2 
	       and should be a spca504b then overwrite that setting */
	    usb_rd_vend_dev(dev, 0x20, 0, 0, &fw, 1);
	    if (fw == 1) {
		_CAM(AiptekMiniPenCam13, BRIDGE_SPCA504, SENSOR_INTERNAL, JPEG);
		spca50x->header_len = SPCA50X_OFFSET_DATA;
		info("USB SPCA5XX camera found. Type Aiptek mini PenCam 1.3(SPCA504A)");
	    } else if (fw == 2) {
		_CAM(Terratec2move13, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
		spca50x->header_len = SPCA50X_OFFSET_DATA;
		info("USB SPCA5XX camera found. Terratec 2 move1.3(SPCA504A FW2)");
	    } else
		return -ENODEV;
	}
	    break;

	case 0x2018:
	    _CAM(AiptekPenCamSD, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PenCam SD(SPCA504A FW2)");
	    break;
	case 0x1001:
	    _CAM(JvcGcA50, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. JVC GC-A50(SPCA504A FW2)");
	    break;
	case 0x2008:
	    _CAM(AiptekMiniPenCam2, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PenCam 2M(SPCA504A FW2)");
	    break;

	case 0x504b:
	    _CAM(MaxellMaxPocket, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Maxell MaxPocket 1.3 (SPCA504A FW2)");
	    break;

	case 0xffff:
	    _CAM(PureDigitalDakota, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Pure Digital Dakota (SPCA504A FW2)");
	    break;

	case 0x0103:
	    _CAM(AiptekPocketDV, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PocketDV");
	    break;

	case 0x0104:
	    _CAM(AiptekPocketDVII, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PocketDVII 1.3Mp");
	    break;

	case 0x0106:
	    _CAM(AiptekPocketDV3100, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PocketDV3100+");
	    break;

	case 0xc630:
	    _CAM(MustekMDC4000, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Musteck MDC4000");
	    break;

	case 0x5330:
	    _CAM(Digitrex2110, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. ApexDigital Digitrex 2110 spca533");
	    break;

	case 0x2020:
	    _CAM(AiptekSlim3000F, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    spca50x->cameratype = JPEG;
	    info("USB SPCA5XX camera found type: Aiptek Slim3000F spca533");
	    break;

	case 0x2022:
	    _CAM(AiptekSlim3200, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found type: Aiptek Slim 3200 spca533");
	    break;

	case 0x2028:
	    _CAM(AiptekPocketCam4M, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found type: Aiptek PocketCam 4M spca533");
	    break;

	case 0x5360:
	    _CAM(SunplusGeneric536, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek Generic spca536a");
	    break;

	case 0x2024:
	    _CAM(AiptekDV3500, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek DV3500 Mpeg4");
	    break;

	case 0x2042:
	    _CAM(AiptekPocketDV5100, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek DV5100 Mpeg4");
	    break;

	case 0x2060:
	    _CAM(AiptekPocketDV5300, BRIDGE_SPCA536, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA536_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek DV5300 Mpeg4");
	    break;

	case 0x3008:
	    _CAM(BenqDC1500, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Benq DC 1500 Spca533");
	    break;

	case 0x3003:
	    _CAM(BenqDC1300, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Benq DC 1300 Spca504b");
	    break;

	case 0x300a:
	    _CAM(BenqDC3410, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Benq DC 3410 Spca533");
	    break;

	case 0x300c:
	    _CAM(BenqDC1016, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Benq DC 1016 Spca500c ");
	    break;

	case 0x2010:
	    _CAM(AiptekPocketCam3M, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PocketCam 3M");
	    break;

	case 0x2016:
	    _CAM(AiptekPocketCam2M, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Aiptek PocketCam 2 Mega (SPCA504A FW2)");
	    break;

	case 0x0561:
	    _CAM(Flexcam100Camera, BRIDGE_SPCA561, SENSOR_INTERNAL, S561);
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    info("USB SPCA5XX camera found. Type Flexcam 100 (SPCA561A)");
	    break;

	case 0xc200:
	    _CAM(MustekGsmart300, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Mustek Gsmart 300");
	    break;

	case 0x7333:
	    _CAM(PalmPixDC85, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. PalmPix DC85");
	    break;

	case 0xc220:
	    _CAM(Gsmartmini, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mustek Gsmart Mini Spca500c ");
	    break;

        case 0xc230:
	    _CAM(Mustek330K, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mustek Digicam 330K ");
	    break;

	case 0xc530:
	    _CAM(MustekGsmartLCD3, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mustek Gsmart LCD 3");
	    break;

	case 0xc430:
	    _CAM(MustekGsmartLCD2, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Mustek Gsmart LCD 2");
	    break;

	case 0xc440:
	    _CAM(MustekDV3000, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. DV3000");
	    break;

	case 0xc540:
	    _CAM(GsmartD30, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found.Mustek Gsmart D30");
	    break;

	case 0xc650:
	    _CAM(MustekMDC5500Z, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found Mustek MDC5500Z");
	    break;

	case 0x1513:
	    _CAM(MegapixV4, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found Megapix V4");
	    break;
	};
	break;

    case 0x046d:		/* Logitech Labtec */
    case 0x041E:		/* Creative cameras */
	switch (product) {

	case 0x0870:	/* Dexxa webcam */
	    _CAM(DexxaWebcam, BRIDGE_STV0602, 0 /* quickcam_configure */, PGBRG);
	    if (qc_probe_sensor(sc) < 0) {
		printf("error configuring sensor\n");
		spca50x->desc = UnknownCamera;
		break;
	    }
	    spca50x->alt = 1;	/* interface 3 is biggest, use 1 though */
	    break;

	case 0x400A:
	    _CAM(CreativePCCam300, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative PC-CAM 300 (SPCA500+unknown CCD)");
	    break;

	case 0x4012:
	    _CAM(PcCam350, BRIDGE_SPCA504C, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA504_PCCAM600_OFFSET_DATA;;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative PC-CAM 350 (SPCA504c+unknown CCD)");
	    break;

	case 0x0890:
	    _CAM(LogitechTraveler, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QuickCam Traveler (SPCA500+unknown CCD)");
	    break;

	case 0x08a0:
	    _CAM(QCim, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC IM ");
	    break;

	case 0x08a1:
	    _CAM(QCimA1, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC IM ");
	    break;

	case 0x08a2:		// zc302 chips 
	    _CAM(LabtecPro, BRIDGE_ZC3XX, SENSOR_HDCS2020, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Labtec Webcam Pro Zc0302 + Hdcs2020");
	    break;

	case 0x08a3:
	    _CAM(QCchat, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC Chat ");
	    break;

	case 0x08a6:
	    _CAM(LogitechQCim, BRIDGE_ZC3XX, SENSOR_HV7131C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC IM ");
	    break;

	case 0x08a9:
	    _CAM(LogitechNotebookDeluxe, BRIDGE_ZC3XX, SENSOR_HDCS2020, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech Notebooks Deluxe Zc0302 + Hdcs2020");
	    break;
	case 0x08ae:
	    _CAM(QuickCamNB, BRIDGE_ZC3XX, SENSOR_HDCS2020, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC for Notebooks ");
	    break;

	case 0x08ad:
	    _CAM(LogitechQCCommunicateSTX, BRIDGE_ZC3XX, SENSOR_HV7131C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC Communicate STX ");
	    break;

	case 0x08aa:
	    _CAM(LabtecNotebook, BRIDGE_ZC3XX, SENSOR_HDCS2020, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Labtec for Notebooks ");
	    break;

	case 0x08b9:
	    _CAM(QCimB9, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech QC IM ??? ");
	    break;

	case 0x08da:
	    _CAM(QCmessenger, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
            spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
            spca50x->i2c_base = 0;
            spca50x->i2c_trigger_on_write = 0;
	    break;

	case 0x0900:
	    _CAM(LogitechClickSmart310, BRIDGE_SPCA500, SENSOR_HDCS1020, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech ClickSmart 310 (SPCA551+ Agilent HDCS1020)");
	    break;

	case 0x0901:
	    _CAM(LogitechClickSmart510, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech ClickSmart 510 (SPCA500+unknown CCD)");
	    break;

	case 0x0905:
	    _CAM(LogitechClickSmart820, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Logitech ClickSmart 820 (SPCA533+unknown CCD)");
	    break;

	case 0x400B:
	    _CAM(CreativePCCam600, BRIDGE_SPCA504C, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA504_PCCAM600_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative PC-CAM 600 (SPCA504+unknown CCD)");
	    break;

	case 0x4013:
	    _CAM(CreativePccam750, BRIDGE_SPCA504C, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA504_PCCAM600_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative PC-CAM 750 (SPCA504+unknown CCD)");
	    break;

	case 0x0960:
	    _CAM(LogitechClickSmart420, BRIDGE_SPCA504C, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA504_PCCAM600_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Logitech Clicksmart 420 (SPCA504+unknown CCD)");
	    break;

	case 0x4018:
	    _CAM(CreativeVista, BRIDGE_SPCA508, SENSOR_PB100_BA, YUVY);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Vista (SPCA508A+PB100)");
	    break;

	case 0x4028:
	    _CAM(CreativeVistaPlus, BRIDGE_PAC207, SENSOR_PAC207, PGBRG);
	    spca50x->header_len = 12;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Vista Plus (VF0090)");
	    break;

	case 0x401d:		//here505b
	    _CAM(Nxultra, BRIDGE_SPCA505, SENSOR_INTERNAL, YYUV);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Webcam NX Ultra (SPCA505b+unknown CCD)");
	    break;

	case 0x401c:		// zc301 chips 
	    _CAM(CreativeNX, BRIDGE_ZC3XX, SENSOR_PAS106, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative NX Zc301+ CCD PAS106B");
	    break;

	case 0x401e:		// zc301 chips 
	    _CAM(CreativeNxPro, BRIDGE_ZC3XX, SENSOR_HV7131B, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative NX Pro Zc301+hv7131b");
	    break;
	case 0x4034:		// zc301 chips 
	    _CAM(CreativeInstant1, BRIDGE_ZC3XX, SENSOR_PAS106, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Instant P0620");
	    break;
	case 0x4035:		// zc301 chips 
	    _CAM(CreativeInstant2, BRIDGE_ZC3XX, SENSOR_PAS106, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Instant P0620D");
	    break;
	case 0x403a:
	    _CAM(CreativeNxPro2, BRIDGE_ZC3XX, SENSOR_TAS5130C, JPGH);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type Creative Nx Pro FW2 Zc301+Tas5130c");
	    break;
	case 0x403b:
	    _CAM(CreativeVista3b, BRIDGE_SPCA561, SENSOR_INTERNAL, S561);
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    info("USB SPCA5XX camera found. Creative Vista VF0010 (SPCA561A)");
	    break;

#if 0 /* XXX fixme */
	case 0x4036:
	    spca50x->desc = CreativeLive;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Creative Live! Zc301+Tas5130c");
	    break;
	case 0x401f:		// zc301 chips 
	    spca50x->desc = CreativeNotebook;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Creative Webcam Notebook Zc301+Tas5130c");
	    break;
	case 0x4017:		// zc301 chips 
	    spca50x->desc = CreativeMobile;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_ICM105A;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Creative Webcam Mobile Zc301+Icm105a");
	    break;
	case 0x0920:
	    spca50x->desc = QCExpress;
	    spca50x->bridge = BRIDGE_TV8532;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = 4;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = GBGR;
	    info("USB SPCA5xx camera found. Type QC Express (unknown CCD)");
	    break;
	case 0x0921:
	    spca50x->desc = LabtecWebcam;
	    spca50x->bridge = BRIDGE_TV8532;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = 4;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = GBGR;
	    info("USB SPCA5xx camera found. Type Labtec Webcam (unknown CCD)");
	    break;
	case 0x0928:
	    spca50x->desc = QCExpressEtch2;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found.Logitech QuickCam Express II(SPCA561A)");
	    break;
	case 0x0929:
	    spca50x->desc = Labtec929;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found.Labtec WebCam Elch 2(SPCA561A)");
	    break;
	case 0x092a:
	    spca50x->desc = QCforNotebook;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found.Logitech QuickCam for Notebook (SPCA561A)");
	    break;
	case 0x092b:
	    spca50x->desc = LabtecWCPlus;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found.Labtec Webcam Plus (SPCA561A)");
	    break;
	case 0x092c:
	    spca50x->desc = LogitechQC92c;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found.Logitech QuickCam chat (SPCA561A)");
	    break;
#endif /* XXX fixme */
	};
	break;
#if 0 /* XXX fixme */
    case 0x0AC8:		/* Vimicro z-star */
	switch (product) {
	case 0x301b:		/* Wasam 350r */
	    spca50x->desc = Vimicro;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_PB0330;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Vimicro Zc301P 0x301b");

	    break;
	case 0x303b:		/* Wasam 350r */
	    spca50x->desc = Vimicro303b;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_PB0330;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Vimicro Zc301P 0x303b");

	    break;
	case 0x305b:		/* Generic */
	    spca50x->desc = Zc0305b;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Vimicro Zc305B 0x305b");
	    break;

	case 0x0302:		/* Generic */
	    spca50x->desc = Zc302;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_ICM105A;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Vimicro Zc302 ");

	    break;
	};
	break;
    case 0x084D:		/* D-Link / Minton */
	switch (product) {
	case 0x0003:		/* DSC-350 / S-Cam F5 */

	    spca50x->desc = DLinkDSC350;
	    spca50x->bridge = BRIDGE_SPCA500;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPEG;
	    info("USB SPCA5XX camera found. Type D-Link DSC-350 / Minton S-Cam F5 (SPCA500+unknown CCD)");
	    break;
	};
	break;
    case 0x0923:		/* ICM532 cams */
	switch (product) {
	case 0x010f:
	    spca50x->desc = ICM532cam;
	    spca50x->bridge = BRIDGE_TV8532;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = 4;
	    spca50x->cameratype = GBGR;
	    info("USB SPCA5xx camera found. Type ICM532 cam (unknown CCD)");
	    break;
	};
	break;
    case 0x0545:		/* tv8532 cams */
	switch (product) {
	case 0x808b:
	    spca50x->desc = VeoStingray2;
	    spca50x->bridge = BRIDGE_TV8532;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = 4;
	    spca50x->cameratype = GBGR;
	    info("USB SPCA5xx camera found. Type Veo Stingray (unknown CCD)");
	    break;
	case 0x8333:
	    spca50x->desc = VeoStingray1;
	    spca50x->bridge = BRIDGE_TV8532;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = 4;
	    spca50x->cameratype = GBGR;
	    info("USB SPCA5xx camera found. Type Veo Stingray (unknown CCD)");
	    break;
	};
	break;
    case 0x102c:		/* Etoms */
	switch (product) {
	case 0x6151:
	    spca50x->desc = Etoms61x151;
	    spca50x->bridge = BRIDGE_ETOMS;
	    spca50x->sensor = SENSOR_PAS106;
	    spca50x->cameratype = GBRG;
	    info("USB Etx61xx51 camera found.Qcam Sangha Et61x151+Pas 106 ");
	    break;
	case 0x6251:
	    spca50x->desc = Etoms61x251;
	    spca50x->bridge = BRIDGE_ETOMS;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->cameratype = GBRG;
	    info("USB Etx61xx51 camera found.Qcam xxxxxx Et61x251+Tas 5130c");
	    break;
	};
	break;
    case 0x1776:		/* Arowana */
	switch (product) {
	case 0x501c:		/* Arowana 300k CMOS Camera */
	    spca50x->desc = Arowana300KCMOSCamera;
	    spca50x->bridge = BRIDGE_SPCA501;
	    spca50x->sensor = SENSOR_HV7131B;
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = YUYV;
	    info("USB SPCA5XX camera found. Type Arowana 300k CMOS Camera (SPCA501C+HV7131B)");
	    break;
	};
	break;
    case 0x0000:		/* Unknow Camera */
	switch (product) {
	case 0x0000:		/* UnKnow from Ori CMOS Camera */
	    spca50x->desc = MystFromOriUnknownCamera;
	    spca50x->bridge = BRIDGE_SPCA501;
	    spca50x->sensor = SENSOR_HV7131B;
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = YUYV;
	    info("USB SPCA5XX camera found. UnKnow CMOS Camera (SPCA501C+HV7131B)");
	    break;
	};
	break;
    case 0x8086:		/* Intel */
	switch (product) {
	case 0x0110:
	    spca50x->desc = IntelEasyPCCamera;
	    spca50x->bridge = BRIDGE_SPCA508;
	    spca50x->sensor = SENSOR_PB100_BA;
	    spca50x->header_len = SPCA508_OFFSET_DATA;

	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA508_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = YUVY;
	    info("USB SPCA5XX camera found. Type Intel Easy PC Camera CS110 (SPCA508+PB100)");
	    break;
	case 0x0630:		/* Pocket PC Camera */

	    spca50x->desc = IntelPocketPCCamera;
	    spca50x->bridge = BRIDGE_SPCA500;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPEG;
	    info("USB SPCA5XX camera found. Type Intel Pocket PC Camera (SPCA500+unknown CCD)");
	    break;
	};
	break;
    case 0x0506:		/* 3COM cameras */
	switch (product) {
	case 0x00DF:
	    spca50x->desc = ThreeComHomeConnectLite;
	    spca50x->bridge = BRIDGE_SPCA501;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = YUYV;
	    info("USB SPCA5XX camera found. Type 3Com HomeConnect Lite (SPCA501A+?)");
	    break;
	};
	break;
    case 0x0458:		/* Genius KYE cameras */
	switch (product) {
	case 0x7004:
	    spca50x->desc = GeniusVideoCAMExpressV2;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found. Type Genius VideoCAM Express V2 (SPCA561A)");
	    break;
	case 0x7006:
	    spca50x->desc = GeniusDsc13;
	    spca50x->bridge = BRIDGE_SPCA504B;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPEG;
	    info("USB SPCA5XX camera found. Type Genius DSC 1.3 Smart Spca504B");
	    break;
	case 0x7007:		// zc301 chips 
	    spca50x->desc = GeniusVideoCamV2;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Genius VideoCam V2 Zc301+Tas5130c");
	    break;

	case 0x700c:		// zc301 chips 
	    spca50x->desc = GeniusVideoCamV3;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Genius VideoCam V3 Zc301+Tas5130c");
	    break;

	case 0x700f:		// zc301 chips 
	    spca50x->desc = GeniusVideoCamExpressV2b;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Genius VideoCam Express V2 Zc301+Tas5130c");
	    break;
	};
	break;
    case 0xabcd:		/* PetCam  */
	switch (product) {
	case 0xcdee:
	    spca50x->desc = PetCam;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found. Type Petcam (SPCA561A)");
	    break;
	};
	break;
    case 0x060b:		/* Maxell  */
	switch (product) {
	case 0xa001:
	    spca50x->desc = MaxellCompactPM3;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found. Type Maxell Compact PCPM3 (SPCA561A)");
	    break;
	};
	break;
    case 0x10fd:		/* FlyCam usb 100  */
	switch (product) {
	case 0x7e50:
	    spca50x->desc = Flycam100Camera;
	    spca50x->bridge = BRIDGE_SPCA561;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA561_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA561_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = S561;
	    info("USB SPCA5XX camera found. FlyCam Usb100 (SPCA561A)");
	    break;
	case 0x0128:
	case 0x8050:		// zc301 chips
	    spca50x->desc = TyphoonWebshotIIUSB300k;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Typhoon Webshot II Zc301p Tas5130c");
	    break;
	};
	break;
    case 0x0461:		/* MicroInnovation  */
	switch (product) {
	case 0x0815:
	    spca50x->desc = MicroInnovationIC200;
	    spca50x->bridge = BRIDGE_SPCA508;
	    spca50x->sensor = SENSOR_PB100_BA;
	    spca50x->header_len = SPCA508_OFFSET_DATA;

	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = SPCA508_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 1;
	    spca50x->cameratype = YUVY;
	    info("USB SPCA5XX camera found. Type MicroInnovation IC200 (SPCA508+PB100)");
	    break;
	case 0x0a00:		// zc301 chips 
	    spca50x->desc = WebCam320;
	    spca50x->bridge = BRIDGE_ZC3XX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGH;
	    info("USB SPCA5XX camera found. Type Micro Innovation PC Cam 300A Zc301");
	    break;
	};
	break;
    case 0x06e1:		/* ADS Technologies  */
	switch (product) {
	case 0xa190:
	    spca50x->desc = ADSInstantVCD;
	    spca50x->bridge = BRIDGE_SPCA506;
	    spca50x->sensor = SENSOR_SAA7113;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;	//SPCA508_INDEX_I2C_BASE;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = YYUV;
	    info("USB SPCA5XX camera found. Type ADS Instant VCD (SPCA506+SAA7113)");
	    break;
	};
	break;
    case 0x05da:		/* Digital Dream cameras */
	switch (product) {
	case 0x1018:
	    spca50x->desc = Enigma13;
	    spca50x->bridge = BRIDGE_SPCA504B;
	    spca50x->sensor = SENSOR_INTERNAL;
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = 0;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPEG;

	    info("USB SPCA5XX camera found. Digital Dream Enigma 1.3");
	    break;
	};
	break;
    case 0x0c45:		/* Sonix6025 TAS 5130d1b */
	switch (product) {
	case 0x6001:
	    _CAM(GeniusVideoCamNB, BRIDGE_SONIX, SENSOR_TAS5110, SN9C);
	    spca50x->customid = SN9C102;
	    spca50x->i2c_ctrl_reg = 0x20;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB Genius VideoCAM NB found. SONIX sn9c102 + Tas 5110 ");
	    break;
	case 0x6007:
	case 0x6005:
	    spca50x->desc = SweexTas5110;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_TAS5110;
	    spca50x->customid = SN9C101;
	    spca50x->i2c_ctrl_reg = 0x20;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c101 +Tas 5110 ");
	    break;
	case 0x6024:
	case 0x6025:
	    spca50x->desc = Sonix6025;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_TAS5130C;
	    spca50x->customid = SN9C102;
	    spca50x->i2c_ctrl_reg = 0x20;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c102 +Tas 5130d1b ");
	    break;
	case 0x6028:
	    spca50x->desc = BtcPc380;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_PAS202;
	    spca50x->customid = SN9C102;
	    spca50x->i2c_ctrl_reg = 0x80;
	    spca50x->i2c_base = 0x40;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c102 + Pas202");
	    break;
	case 0x6019:
	    spca50x->desc = Sonix6019;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_OV7630;
	    spca50x->customid = SN9C101;
	    spca50x->i2c_ctrl_reg = 0x80;
	    spca50x->i2c_base = 0x21;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c101 + Ov7630 ");
	    break;
	case 0x602c:
	case 0x602e:
	    spca50x->desc = GeniusVideoCamMessenger;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_OV7630;
	    spca50x->customid = SN9C102;
	    spca50x->i2c_ctrl_reg = 0x80;
	    spca50x->i2c_base = 0x21;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c102 + Ov7630 ");
	    break;
	case 0x602d:
	    spca50x->desc = Lic200;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_HV7131R;
	    spca50x->customid = SN9C102;
	    spca50x->i2c_ctrl_reg = 0x80;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;
	    info("USB SPCA5XX camera found. SONIX sn9c102 Hv7131R ");
	    break;
	case 0x6009:
	case 0x600d:
	case 0x6029:
	    spca50x->desc = Sonix6029;
	    spca50x->bridge = BRIDGE_SONIX;
	    spca50x->sensor = SENSOR_PAS106;
	    spca50x->customid = SN9C101;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x40;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = SN9C;	//GBRG ;
	    info("USB SPCA5XX camera found. SONIX sn9c102 + Pas106 ");
	    break;
	case 0x6040:
	    spca50x->desc = SpeedNVC350K;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_HV7131R;
	    spca50x->customid = SN9C102P;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. Speed NVC 350K sn9c102p + Hv7131R ");
	    break;
	case 0x607c:
	    spca50x->desc = SonixWC311P;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_HV7131R;
	    spca50x->customid = SN9C102P;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. SONIX sn9c102p + Hv7131R ");
	    break;
	case 0x613c:
	    spca50x->desc = Pccam168;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_HV7131R;
	    spca50x->customid = SN9C120;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. SONIX sn9c120 + Hv7131R ");
	    break;
	case 0x6130:
	    spca50x->desc = Pccam;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_MI0360;
	    spca50x->customid = SN9C120;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x5d;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. SONIX sn9c120 + MI0360 ");
	    break;
	case 0x60c0:
	    spca50x->desc = Sn535;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_MI0360;
	    spca50x->customid = SN9C105;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x5d;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. SONIX sn9c105 + MI0360 ");
	    break;
	case 0x60fc:
	    spca50x->desc = Lic300;
	    spca50x->bridge = BRIDGE_SN9CXXX;
	    spca50x->sensor = SENSOR_HV7131R;
	    spca50x->customid = SN9C105;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x11;
	    spca50x->i2c_trigger_on_write = 0;
	    spca50x->cameratype = JPGS;	// jpeg 4.2.2 whithout header ;
	    info("USB SPCA5XX camera found. SONIX sn9c105 + HV7131R ");
	    break;
#if 0
	case 0x624f:
	    /*
	     * XXX experimental entry for the Trust WB5500
	     * http://www.trust.nl/products/product.aspx?artnr=14830
	     * The manufacturer says 1.3mpix resolution.
		Windows .inf file:
		SBPCamDesc% =    SN.USBPCam,USB\VID_0c45&PID_624f   ; SN9C201 + OV9650
	       Also
		ASUS WebCam, 1.3M, USB2.0, FF
	     */
	    _CAM(TrustWB5500, BRIDGE_SN9CXXX, SENSOR_HV7131R, JPGS);
	    spca50x->customid = SN9C105;
	    /* XXX 0x81 0x11 don't work */
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x5d;
	    spca50x->i2c_trigger_on_write = 0;
	    break;
#endif
	};
	break;
#endif /* XXX fixme */

    case 0x0546:		/* Polaroid */
	switch (product) {
	case 0x3273:
	    _CAM(PolaroidPDC2030, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Polaroid PDC 2030");
	    break;

	case 0x3155:
	    _CAM(PolaroidPDC3070, BRIDGE_SPCA533, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA533_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Polaroid PDC 3070");
	    break;

	case 0x3191:
	    _CAM(PolaroidIon80, BRIDGE_SPCA504B, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    info("USB SPCA5XX camera found.Polaroid Ion80 (SPCA504A FW2)");
	    break;
	};
	break;
    case 0x0572:		/* Connexant */
	switch (product) {
	case 0x0041:
	    _CAM(CreativeNoteBook2, BRIDGE_CX11646, SENSOR_INTERNAL, JPGC);
	    info("USB SPCA5XX camera found. Creative NoteBook PD1170");
	    break;
	};
	break;
    case 0x06be:		/* Optimedia */
	switch (product) {
	case 0x0800:
	    _CAM(Optimedia, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Optimedia spca500a");
	    break;
	};
	break;
    case 0x2899:		/* ToptroIndustrial */
	switch (product) {
	case 0x012c:
	    _CAM(ToptroIndus, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Toptro Industrial spca500a");
	    break;
	};
	break;
    case 0x06bd:		/* Agfa Cl20 */
	switch (product) {
	case 0x0404:
	    _CAM(AgfaCl20, BRIDGE_SPCA500, SENSOR_INTERNAL, JPEG);
	    spca50x->header_len = SPCA500_OFFSET_DATA;
	    info("USB SPCA5XX camera found. Agfa ephoto CL20 spca500a");
	    break;
	};
	break;

    case 0x093a:		/* Mars-semi ~ Pixart */
	switch (product) {
	case 0x050f:
	    _CAM(Pcam, BRIDGE_MR97311, SENSOR_MI0360, JPGM);
	    spca50x->header_len = 16;
	    info("USB SPCA5XX camera found. Mars-Semi MR97311 MI0360 ");
	    break;

	case 0x2460:
	    _CAM(QtecWb100, BRIDGE_PAC207, SENSOR_PAC207, PGBRG);
	    spca50x->header_len = 12;
	    info("USB SPCA5XX camera found. Qtec Webcam 100 Pac207-BCA ");
	    break;

	case 0x2468:
	    _CAM(PAC207, BRIDGE_PAC207, SENSOR_PAC207, PGBRG);
	    spca50x->header_len = 12;	/* XXX unused ? */
	    info("USB SPCA5XX camera found. Pixart PAC207BCA");
	    break;

	case 0x2470:
	    _CAM(GeniusGF112, BRIDGE_PAC207, SENSOR_PAC207, PGBRG);
	    spca50x->header_len = 12;
	    info("USB SPCA5XX camera found.Genius GF112 (PAC207-BCA)");
	    break;

	case 0x2471:
	    _CAM(GeniusGe111, BRIDGE_PAC207, SENSOR_PAC207, PGBRG);
	    spca50x->header_len = 12;
	    info("USB SPCA5XX camera found. Genius VideoCam Ge111");
	    break;
	};
	break;

    case 0x0497:		/* Smile International */
	switch (product) {
	case 0xc001:
	    // Modal NO. VA30UC2 8/nq h0 106250
	    // Hone-Tec Inc. VA30UC2
	    _CAM(SmileIntlCamera, BRIDGE_SPCA501, SENSOR_INTERNAL, YUYV);
	    spca50x->header_len = SPCA501_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found Type Smile International ");
	    break;
	};
	break;

    case 0x0698:		/* Chuntex (CTX) */
	switch (product) {
	case 0x2003:		/* The Webcam built in the CTX M730V TFT-Display, behind an USB-HUB */
	    _CAM(CTXM730VCam, BRIDGE_ZC3XX, SENSOR_ICM105A, JPGH);
	    // SENSOR_ICM105A;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Type CTX M730V built in Webcam");
	    break;
	};
	break;

    case 0x0471:		/* Philips Product */
	switch (product) {
	case 0x0325:		/* Low cost Philips Webcam */
	    _CAM(PhilipsSPC200NC, BRIDGE_ZC3XX, SENSOR_PAS106, JPGH);
	    // SENSOR_PAS106 overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found.Type Philips SPC200NC Vimicro PAS106");
	    break;

	case 0x0326:		/* Low cost Philips Webcam */
	    _CAM(PhilipsSPC300NC, BRIDGE_ZC3XX, SENSOR_PAS106, JPGH);
	    // SENSOR_PAS106;	//overwrite by the sensor detect routine
	    spca50x->header_len = SPCA50X_OFFSET_DATA;
	    spca50x->i2c_ctrl_reg = SPCA50X_REG_I2C_CTRL;
	    spca50x->i2c_base = 0;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found.Type Philips SPC300NC Vimicro ????");
	    break;

	case 0x0328:
	    _CAM(PhilipsSPC700NC, BRIDGE_SN9CXXX, SENSOR_MI0360, JPGS);
	    // JPGS;	// jpeg 4.2.2 whithout header ;
	    spca50x->customid = SN9C105;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x5d;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Philips SPC700NC sn9c105 + MI0360 ");
	    break;

	case 0x0327:
	    _CAM(PhilipsSPC600NC, BRIDGE_SN9CXXX, SENSOR_MI0360, JPGS);
	    spca50x->customid = SN9C105;
	    spca50x->i2c_ctrl_reg = 0x81;
	    spca50x->i2c_base = 0x5d;
	    spca50x->i2c_trigger_on_write = 0;
	    info("USB SPCA5XX camera found. Philips SPC600NC sn9c105 + MI0360 ");
	    break;
	};
	break;
    }
    if (spca50x->desc == UnknownCamera) {
	printf("spcaCameraDetect failed\n");
    	return -ENODEV;
    }

    /* copy into the main descriptor */
    sc->pwc_info.name = camera_name(sc->spca50x.desc);
    sc->pwc_info.dataformat = sc->spca50x.cameratype;

    /* XXX not sure if this depends on the bridge or on the sensor */
    switch(sc->pwc_info.bridge) {
    case BRIDGE_ZC3XX:
	sc->camera_cb.cb_consume = zc3xx_consume;
	break;
    case BRIDGE_PAC207:
	sc->camera_cb.cb_consume = pac207_consume;
	break;
    case BRIDGE_SPCA561:
	sc->camera_cb.cb_consume = spca561_consume;
	break;

    case BRIDGE_STV0602:
	sc->camera_cb.cb_consume = qc_consume;
	break;
    }
    return 0;
}

//eof
