/* Go game AI for LCARS 24, adapted from Wally, by Bill Newman
*/
/* This program is free software. You may redistribute and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License or
 * (at your option) any later version.
 *
 * Refer to the file C:\LCARS24\DATA\COPYING.TXT for details.
 */


#include <assert.h>
#include "goai.h"

/*Return a string naming BLACKSIDE or WHITESIDE.*/
char whitename[6]="WHITE";
char blackname[6]="BLACK";
#define pname(c) (BLACKSIDE==(c)?blackname:WHITESIDE==(c)?whitename:\
	(GoAiPanic("illegal input to pname()"),""))


int flookup[]=	 /* sets of flags corresponding to codes */
{ F_BLACK, F_WHITE, F_EMPTY };

/*a table of moves which seem to make "good shape" for the computer, */
/* or which defend against a particularly common threat that the program */
/* can't understand without lookahead, or which act as primitive joseki*/
#define PATTERN 12345	/*This integer should appear at the start of each */
		/* pattern.  If it doesn't, someone has made a typographical */
		/* error.*/
#define PATTEND 7171	/*This integer should appear at the end of the table.*/
int patterns[]=
{  PATTERN,	/*a code for the beginning of a pattern*/
   4, 24,	/*Three points, basic urgency 24.  Urgency is highest */
		/* for lower numbers.  Original values: */
		/*   capturing an enemy group=16, */
		/*   defending one of ours= 20, */
		/*   atariing an enemy group= 32 */
  -1, 0, F_BLACK|F_OFF, /*Recognize the pattern by black stone at (x-1, y+0) */
		/* or that point off the board, */
   1, 0, F_BLACK,  /* a black stone at (x+1, y+0), and */
   0, -1, F_WHITE, /* a white stone at (x+0, y-1), and */
   0, 1, F_EMPTY|F_WHITE, /* a white stone or space at (x, y+1), */
	/* that is: */	/*   ~   */
			/* ~ $ # */
			/*   O   */

  PATTERN, 3, 22,	/* # $   */
  -1, 0, F_BLACK,	/*   O # */
   1, -1, F_BLACK,
   0, -1, F_WHITE,

  PATTERN, 5, 26,	/* # O # */
  -1, 1, F_BLACK|F_OFF,	/*   $ . */
   1, 1, F_BLACK,	/*       */
   0, 1, F_WHITE,	/*   ~   */
   1, 0, F_EMPTY,
   0, -3, F_EMPTY|F_WHITE,	/*This so that we don't get trapped in */
			/* an extremely common pattern against the edge */
			/* of the board, or in a shicho.*/

  PATTERN, 4, 26,	/* # O # */
  -1, 1, F_BLACK|F_OFF,	/*   $ # */
  1, 1, F_BLACK,
  1, 0, F_BLACK,
  0, 1, F_WHITE,

  PATTERN, 6, 24,	/*   ~     */
  -1, 0, F_BLACK,	/* # $ . # */
  0, -1, F_WHITE,	/*   O     */
  1, 0,  F_EMPTY,
  1, -1, F_EMPTY,
  2, 0,  F_BLACK,
  0, 1, F_EMPTY|F_WHITE,

  PATTERN, 5, 27,	/*   ~     */
  -1, 0, F_BLACK,	/* # $ . ~ */
  0, -1, F_WHITE,	/*   O     */
  1, 0,  F_EMPTY,
  2, 0,  F_OFF,
  0, 1, F_EMPTY|F_WHITE,

  PATTERN, 5, 24,	/* # . $ . # */
  -2, 0, F_BLACK,	/*     O     */
  -1, 0, F_EMPTY,
  0, -1, F_WHITE,
  1,  0, F_EMPTY,
  2,  0, F_BLACK,

  PATTERN, 5, 30,	/*  # . $ . ~  */
  -2, 0, F_BLACK,	/*      O      */
  -1, 0, F_EMPTY,
  0, -1, F_WHITE,
  1,  0, F_EMPTY,
  2,  0, F_OFF,

  PATTERN, 7, 26,		/*    ~ .    */
   -1, 0, F_BLACK,		/*  # $ # O  */
   1, -1, F_WHITE,		/*    ~ O    */
   1, 0, F_BLACK,
   2, 0, F_WHITE,
   1, 1, F_EMPTY,
   0, 1, F_EMPTY|F_WHITE,
   0, -1, F_EMPTY|F_WHITE,

  PATTERN, 8, 26,		/*  ~ # O  */
  -1, -1, F_BLACK|F_EMPTY,	/*  # . $  */
  -2, 0, F_BLACK,		/*  ~ ~ ~  */
  -1, 0, F_EMPTY,
  -1, 1, F_BLACK,
   0, 1, F_WHITE,
   0, -1, F_BLACK|F_EMPTY,
  -2, -1, F_BLACK|F_EMPTY,
  -2, 1, F_BLACK|F_EMPTY,

  PATTERN, 4, 26,	/*   .   */
  0, 2,  F_EMPTY,	/* #   # */
  -1, 1, F_BLACK,	/*   $ O */
  1, 1,  F_BLACK,
  1, 0,  F_WHITE,

  PATTERN, 4, 24,	/*   .   */
  0, 1, F_EMPTY,	/*   $   */
  -1, -1, F_WHITE,	/* O # O */
  0, -1, F_BLACK,
  1, -1, F_WHITE,

  PATTERN, 4, 24,	/*   .   */
  0, 1, F_EMPTY,	/*   $ O */
  -1, -1, F_WHITE,	/* O #   */
  0, -1, F_BLACK,
  1, 0, F_WHITE,

  PATTERN, 4, 26,	/*   O   */
  0, 1, F_EMPTY,	/* O . O */
  -1, 1, F_WHITE,	/*   $   */
  1, 1, F_WHITE,
  0, 2, F_WHITE,

  PATTERN, 8, 24,	  /*     O   */
  -2, 0, F_EMPTY|F_BLACK, /* ~ $ # O */
  0, -1, F_EMPTY,	  /*   . . . */
  1, -2, F_OFF,		  /*     ~   */
  1, -1, F_EMPTY,
  1, 0, F_BLACK,
  1, 1, F_WHITE,
  2, -1, F_EMPTY,
  1, 0, F_WHITE,

  PATTERN, 8, 28,	/* . # O   */
  -1, 0, F_EMPTY,	/* . $ # O */
  0, -1, F_EMPTY,	/*   . .   */
  0, 1, F_BLACK,
  1, 0, F_BLACK,
  1, 1, F_WHITE,
  2, 0, F_WHITE,
  -1, 1, F_EMPTY,
  1, -1, F_EMPTY,

  PATTERN, 8, 23,	/*   # #   */
  0, 2, F_BLACK,	/*   .   # */
  1, 2, F_BLACK,	/* ~ $ . # */
  2, 0, F_BLACK,	/*   ~     */
  2, 1, F_BLACK,
  1, 0, F_EMPTY,
  0, 1, F_EMPTY,
  -1, 0, F_EMPTY|F_BLACK|F_WHITE,
  0, -1, F_EMPTY|F_WHITE,

  PATTERN, 8, 24,	/*   O O   */
  0, 2, F_WHITE,	/*   .   O */
  1, 2, F_WHITE,	/* ~ $ . O */
  2, 0, F_WHITE,	/*   ~     */
  2, 1, F_WHITE,
  1, 0, F_EMPTY,
  0, 1, F_EMPTY,
  -1, 0, F_EMPTY|F_WHITE|F_BLACK,
  0, -1, F_EMPTY|F_BLACK,

  PATTERN, 8, 34,	/* ~ # # */
  -1, 1, F_BLACK,	/* # . # */
  -1, 0, F_BLACK,	/* # $   */
  0, 2, F_BLACK,	/*   ~   */
  0, 1, F_EMPTY,
  1, 1, F_BLACK,
  1, 2, F_BLACK,
  -1, 2, F_BLACK|F_EMPTY,
  0, -1, F_WHITE|F_EMPTY,

  PATTERN, 7, 34,		/* ~ # ~ */
  -1, 1, F_BLACK,		/* # . # */
  -1, 0, F_BLACK|F_EMPTY,	/* ~ $   */
  -1, 2, F_BLACK|F_EMPTY,
  0, 2, F_BLACK,
  0, 1, F_EMPTY,
  1, 1, F_BLACK,
  1, 2, F_BLACK|F_EMPTY,

  PATTERN, 10, 27,	/*   . . .   */	/*(This is Wally's fuseki for */
  -2, 0, F_BLACK,	/* # . $ . # */	/* a 9 x 9 board with 4 handicap */
  2, 0, F_BLACK,	/*   . . .   */	/* stones.)*/
  -1, 0, F_EMPTY,
  1, 0, F_EMPTY,
  -1, 1, F_EMPTY,
  0, 1, F_EMPTY,
  1, 1, F_EMPTY,
  -1, -1, F_EMPTY,
  0, -1, F_EMPTY,
  1, -1, F_EMPTY,

  PATTERN, 6, 25,	/*     # . #   */
  -1, 1, F_BLACK,	/*   ~ . $     */
  -1, -1, F_BLACK,	/*     #       */
  1, 1, F_BLACK,
  0, 1, F_EMPTY,
  -1, 0, F_EMPTY,
  -2, 0, F_EMPTY|F_BLACK|F_OFF,

  PATTERN, 5, 35,	/*     O #   */
  -1, 0, F_EMPTY,	/*   . $ ~   */
  0, 1, F_WHITE,	/*     ~     */
  1, 1, F_BLACK,
  0, -1, F_OFF,
  0, 1, F_BLACK|F_EMPTY,

  PATTERN,  5, 40,	/*   O #     */
  -1, 0, F_EMPTY,	/*   . $ ~   */
  -1, 1, F_WHITE,	/*     ~     */
  0, 1, F_BLACK,
  0, -1, F_OFF,
  1, 0, F_WHITE|F_EMPTY,

  PATTERN, 6, 38,	/*   O . #   */
  -1, 1, F_WHITE,	/*     $ .   */
  1, 1, F_BLACK,	/*     .     */
  0, 1, F_EMPTY,	/*     ~     */
  1, 0, F_EMPTY,
  0, -1, F_EMPTY,
  0, -2, F_OFF|F_BLACK,

  PATTERN, 6, 38,	/*     . #   */
  -1, 0, F_WHITE,	/*   O $ .   */
  1, 1, F_BLACK,	/*     .     */
  0, 1, F_EMPTY,	/*     ~     */
  1, 0, F_EMPTY,
  0, -1, F_EMPTY,
  0, -2, F_OFF|F_BLACK,

/*
some patterns for a 19 x 19 board, which is a little different from
the high handicap 9 x 9 game this program originally played
*/

  PATTERN, 11, 29,	  /*     O   */
  -2, 0, F_EMPTY|F_BLACK, /* ~ $ # O */
  0, -1, F_EMPTY,	  /*   . . . */
  1, -2, F_OFF,		  /*   ~ ~ ~ */
  1, -1, F_EMPTY,	  /*     ~   */
  1, 0, F_BLACK,
  1, 1, F_WHITE,
  2, -1, F_EMPTY,
  -1, -2, F_EMPTY|F_OFF,
  0, -2, F_EMPTY|F_OFF,
  1, -2, F_EMPTY|F_OFF,
  1, 0, F_WHITE,

  PATTERN, 12, 38,	/*     ~ #   */
  -1, 0, F_WHITE,	/*   O $ ~ ~ */
  1, 1, F_BLACK,	/*     ~ ~ ~ */
  0, 1, F_EMPTY|F_BLACK,/*     ~ ~ ~ */
  1, 0, F_EMPTY|F_WHITE,/*     ~     */
  0, -3, F_OFF,
  0, -1, F_EMPTY|F_BLACK,
  0, 2,  F_EMPTY|F_BLACK|F_OFF,
  1, -1, F_EMPTY|F_BLACK|F_OFF,
  2, -1, F_EMPTY|F_BLACK|F_OFF,
  0, -2, F_EMPTY|F_BLACK|F_OFF,
  1, -2, F_EMPTY|F_BLACK|F_OFF,
  2, -2, F_EMPTY|F_BLACK|F_OFF,

  PATTERN, 12, 38,	/*     # ~   */
  -1, 0, F_WHITE,	/*   O $ ~ ~ */
  0, 1, F_BLACK,	/*     ~ ~ ~ */
  1, 1, F_EMPTY|F_BLACK,/*     ~ ~ ~ */
  1, 0, F_EMPTY|F_WHITE,/*     ~     */
  0, -3, F_OFF,
  0, -1, F_EMPTY|F_BLACK,
  0, 2,  F_EMPTY|F_BLACK|F_OFF,
  1, -1, F_EMPTY|F_BLACK|F_OFF,
  2, -1, F_EMPTY|F_BLACK|F_OFF,
  0, -2, F_EMPTY|F_BLACK|F_OFF,
  1, -2, F_EMPTY|F_BLACK|F_OFF,
  2, -2, F_EMPTY|F_BLACK|F_OFF,

/*
Let's not just pass when confronted by
an empty 19 x 19 board.. hence the following patterns
*/

  PATTERN, 15, 25,	/*           ~     */
  0, 1, F_EMPTY,	/*           ~ ~   */
  0, 2, F_EMPTY,	/*     . . .       */
  1, 0, F_EMPTY,	/*   . . . . 	   */
  1, 1, F_EMPTY,	/*   . $ . . 	   */
  1, 2, F_EMPTY,	/*     . .	   */
  2, 0, F_EMPTY,
  2, 1, F_EMPTY,
  2, 2, F_EMPTY,
  4, 3, F_OFF,
  3, 4, F_OFF,
  -1, 0, F_EMPTY,
  -1, 1, F_EMPTY,
  0, -1, F_EMPTY,
  1, -1, F_EMPTY,
  3, 3, F_EMPTY|F_WHITE|F_BLACK,

  PATTERN, 10, 29,
  -1, 0, F_EMPTY,	/*         ~     */
  -1, 1, F_EMPTY,	/*         ~ ~   */
  0, -1, F_EMPTY,	/*   . . .  	 */
  1, -1, F_EMPTY,	/*   . $ .  	 */
  0, 1, F_EMPTY,	/*     . .	 */
  1, 0, F_EMPTY,
  1, 1, F_EMPTY,
  2, 2, F_EMPTY|F_WHITE|F_BLACK,
  3, 2, F_OFF,
  2, 3, F_OFF,

  PATTERN, 17, 29,	/*     . . .          */
  0, 1, F_EMPTY,	/*     . . .          */
  0, -1, F_EMPTY,	/*   ~ . $ .   ~ ~    */
  1, 0, F_EMPTY,	/*     . . .          */
  -1, 0, F_EMPTY,	/*     . . .          */
  1, 1, F_EMPTY,
  1, -1, F_EMPTY,
  -1, 1, F_EMPTY,
  -1, -1, F_EMPTY,
  -1, 2, F_EMPTY,
  0, 2, F_EMPTY,
  1, 2, F_EMPTY,
  -1, -2, F_EMPTY,
  0, -2, F_EMPTY,
  1, -2, F_EMPTY,
  3, 0, F_WHITE|F_BLACK|F_EMPTY,
  4, 0, F_OFF|F_BLACK,
  -2, 0, F_WHITE|F_EMPTY,
  

  PATTERN, 12, 31,	/*     . . .     */
  0, 1, F_EMPTY,	/*     . $ .     */
  0, -1, F_EMPTY,	/*   ~ . . . ~   */
  1, 0, F_EMPTY,	/*	 ~	 */
  -1, 0, F_EMPTY,	/*       ~       */
  1, 1, F_EMPTY,
  1, -1, F_EMPTY,
  -1, 1, F_EMPTY,
  -1, -1, F_EMPTY,
  -2, -1, F_EMPTY|F_BLACK,
  2, -1, F_EMPTY|F_BLACK,
  0, -2, F_EMPTY|F_BLACK|F_WHITE,
  0, -3, F_OFF,

  PATTERN, 22, 33,	/*   ~ ~ #       */
  0, 1, F_EMPTY,	/*   . . . . .   */
  0, -1, F_EMPTY,	/*   . . $ . .   */
  1, 0, F_EMPTY,	/*   . . . . .   */
  -1, 0, F_EMPTY,	/*   . . . . .   */
  1, 1, F_EMPTY,
  1, -1, F_EMPTY,
  -1, 1, F_EMPTY,
  -1, -1, F_EMPTY,
  2, -2, F_EMPTY,
  -2, -2, F_EMPTY,
  0, 2, F_BLACK,
  0, -2, F_EMPTY,
  2, 0, F_EMPTY,
  -2, 0, F_EMPTY,
  2, 1, F_EMPTY,
  2, -1, F_EMPTY,
  -2, 1, F_EMPTY,
  -2, -1, F_EMPTY,
  1, -2, F_EMPTY,
  -1, -2, F_EMPTY,
  -1, 2, F_EMPTY|F_WHITE,
  -2, 2, F_EMPTY|F_WHITE,

  PATTEND
};

int GoAiVerbose(char *message)
{
/* Was used for debugging, requires #include winfont.h */

/*	PutsArial10(screen,561,150+(MessageY*20), MANGO, message); */
	MessageY++;
}
void GoShowLimit() /* for checking ability to prevent freeze-up */
{
/* unplugged, necessary includes remmed out in goai.h */
/*
	char OutStr[16];

	itoa(GoAiLimit,OutStr,10);
	rectfill(screen, 600, 566, 700, 582, LAVENDER);
	PutsLuna(600,566, BLACK, LAVENDER, OutStr);
*/
}



/*Print an error message and the abort program.*/

int GoAiFatal(char *message)
{
	return(86);
}

/*Fatal error routine reserved for error conditions that ought not happen.*/

int GoAiPanic(char *message)
{
/*	fprintf(stderr, "\n?!panic--%s\n", message); */
	return(86);
}

/*  Return a (slightly) random number from 0 to n-1.	 */
/*  (This is a really crummy rng, it just keeps the 	  */
/*  program's moves from all lying in a trivial pattern.)  */

int GoAiRng(int n)
{
	static int seed= 0x1231;
	int r;

	seed= (seed*11+15) & 0x7FFF;
	r=(((long)seed)*n)/(0x7FFF+1);
	return(r);
}

void GoAiPrintGame()
{
}

void GoAiInitGame(int handicap)
{
	int x, y, j;
	int *h;

	edge=19;
	debugattack=0;
	evenmode=0;
	AtariWarning=0;

/*Clear the go board.*/
	for(y=0; y<edge; y++) {
		for(x=0; x<edge; x++) {
			wallyboard[x][y]=EMPTY;
		}
	}

/*Initialize miscellaneous other stuff.*/

	thegame.kox=thegame.koy=-1;
	thegame.tur=1;

	thegame.pla= WHITESIDE;
	thegame.qpa= 0;
	lex=ley= -1000;		/*nothing is near the enemy's last move*/

/*  simple handicap */

	if(handicap==2) {  /* Wally starts with 9 stones */
		for(y=3; y<16; y+=6) {
			for(x=3; x<16; x+=6) {
				wallyboard[x][y]=BLACKSIDE;
			}
		}
	}

	if(handicap==1) {  /* Wally starts with 4 stones */
		wallyboard[3][3]=BLACKSIDE;
		wallyboard[3][15]=BLACKSIDE;
		wallyboard[15][3]=BLACKSIDE;
		wallyboard[15][15]=BLACKSIDE;
	}

	if(handicap==0) {  /* Wally starts only the first move */
		wallyboard[9][9]=BLACKSIDE;
	}
}

/*
Recursively remove the stone at (x,y) and all stones in the same group
from the goboard.
*/

void GoAiRemoveStone(int x, int y)
{
	int color;

	assert(onboard(x,y));
	color=wallyboard[x][y];
	assert(color==BLACKSIDE || color==WHITESIDE);
	wallyboard[x][y]=EMPTY;

	if(onboard(x, y+1) && color==wallyboard[x][y+1]) GoAiRemoveStone(x, y+1);
	if(onboard(x, y-1) && color==wallyboard[x][y-1]) GoAiRemoveStone(x, y-1);
	if(onboard(x+1, y) && color==wallyboard[x+1][y]) GoAiRemoveStone(x+1, y);
	if(onboard(x-1, y) && color==wallyboard[x-1][y]) GoAiRemoveStone(x-1, y);
}

/*
Remove all of player p's dead stones (as implied by results of GoAiMakeGroups())
from the goboard.  Return # of stones removed.  If any stones are removed,
wallyboard[][] is left inconsistent with groups[].
*capx and *capy are set to the coordinates of a captured stone, if any,
so that if only one stone is captured, they may be used to set ko.
*/

int GoAiCapture(int p, int *capx, int *capy)
{
	int j, rv;
	group *g;

	MessageY=0;
/*	rectfill(screen,561,101,753,410,BLACK); */
/*	if(p!=WHITESIDE && p!=BLACKSIDE) */ p=Opponent;
	if(p==BLACKSIDE) GoAiVerbose("capture p=BLACKSIDE");
	if(p==WHITESIDE) GoAiVerbose("capture p=WHITESIDE");
/*	GoAiVerbose("Press the spacebar");
	getch();
*/

	rv=0;
	for(g=groups,j=0; j<ngroups; ++g,++j)
	if(p==g->color&&0==g->nliberties)
	{
		rv += g->nstones;
		*capx= g->x;
		*capy= g->y;
		GoAiRemoveStone(g->x, g->y);
	}
	return(rv);
}


/*
Do everything which must be done to complete a move after the stone is
placed.
*/

void GoAiMoveDone()
{

/*	thegame.pla=nextp(thegame.pla); */
	++thegame.tur;
}

/*
Recursively group together connected stones.  Three things must be done:
  (1) count liberties in thisgroup->nliberties,
  (2) count stones in thisgroup->nstones, and
We set scratch[][]= mark for each point and liberty of this group,
  so that we only count each only once.  The calling routine must
  see to it that scratch[][]!=mark for all points and liberties that
  it wants counted.
*/

void GoAiCount(int x, int y, group *thisgroup, goboard scratch, int mark)
{
	int *bxy, *sxy;
/* Common subexpressions to reduce array */
/* arithmetic. They point to wallyboard[x][y] and scratch[x][y].*/

endrecurse:
	assert(onboard(x,y));
	assert(thisgroup->color==wallyboard[x][y]);
	assert(thisgroup->nstones>=0);
	assert(thisgroup->nliberties>=0);
	assert(mark!=scratch[x][y]);

/*Set common subexpressions.*/
	bxy= &(wallyboard[x][y]);
	sxy= &(scratch[x][y]);

/*Count the point (x,y) and make it a member of the group.*/
	++(thisgroup->nstones);
	*sxy= mark;

/*"Process" (x, y-1): should it be in the group, or a liberty, or */
/* nothing? */

	y=y-1;
	bxy-=1;
	sxy-=1;
	assert(&wallyboard[x][y]==bxy);
	assert(&scratch[x][y]==sxy);
	if(y>=0) {
		if(thisgroup->color==*bxy) {
			if(*sxy!=mark) GoAiCount(x, y, thisgroup, scratch, mark);
		} else {
			if(EMPTY==*bxy) {
				if(*sxy!=mark) {
					*sxy=mark;
					++(thisgroup->nliberties);
				}
			}
		}
	}

/*Process (x, y+1) */

	y= y+2;
	bxy += 2;
	sxy += 2;
	assert(&wallyboard[x][y]==bxy);
	assert(&scratch[x][y]==sxy);
	if(y<edge) {
		if(thisgroup->color==*bxy) {
			if(*sxy!=mark) GoAiCount(x, y, thisgroup, scratch, mark);
		} else {
			if(EMPTY==*bxy) {
				if(*sxy!=mark) {
					*sxy=mark;
					++(thisgroup->nliberties);
				}
			}
		}
	}

/*Process (x-1, y) */

	y= y-1;
	x= x-1;
	bxy -= EDGE+1;
	sxy -= EDGE+1;
	assert(&wallyboard[x][y]==bxy);
	assert(&scratch[x][y]==sxy);
	if(x>=0) {
		if(thisgroup->color==*bxy) {
			if(*sxy!=mark) GoAiCount(x, y, thisgroup, scratch, mark);
		} else {
			if(EMPTY==*bxy) {
				if(*sxy!=mark) {
					*sxy=mark;
					++(thisgroup->nliberties);
				}
			}
		}
	}

/* Process (x+1, y) */

	x=x+2;
	bxy += 2*EDGE;
	sxy += 2*EDGE;
	assert(&wallyboard[x][y]==bxy);
	assert(&scratch[x][y]==sxy);
	if(x<edge) {
		if(thisgroup->color==*bxy) {
			if(*sxy!=mark) goto endrecurse;
		} else {
			if(EMPTY==*bxy) {
				if(*sxy!=mark) {
					*sxy=mark;
					++(thisgroup->nliberties);
				}
			}
		}
	}
}

int GoAiMakeGroups()
/*Analyzes groupings of stones on goboard; sets groups[].*/
{
  int x, y;		/*coords of point we are now assigning to a group*/
  goboard scratch;	/*scratch for GoAiCount() to use to say if a point is */
			/* already counted*/

	/*Clear old groups[] table and scratch[][].*/
  ngroups= 0;
  { int *s;
    for(s=scratch[0],x=0; x<EDGE*EDGE; ++x)
      *s++= 0;
  }

	/*Go through goboard[][] point by point.  Group any stone which has not */
	/* already been grouped (as indicated by negative scratch[][]).*/

/* modified, doesn't fix the problem */
   x=0; y=0;
   int *bxy;
    int *sxy;
    group *thisgroup=groups;
  bxy= &(wallyboard[x][y]);
  sxy= &(scratch[x][y]);


	GoAiVerbose("Ready to make groups");
/*	GoAiVerbose("Press the spacebar");
	getch();
*/


/* return(0); */

    for(x=0; x<edge; ++x,bxy+=EDGE-edge,sxy+=EDGE-edge) {
      for(y=0; y<edge; ++y,++bxy,++sxy) {
       if( bxy != &(wallyboard[x][y])) return(0);
	if( sxy != &(scratch[x][y])) return(0);
	if( thisgroup != &(groups[ngroups])) return(0);
	if((BLACKSIDE==*bxy||WHITESIDE==*bxy) && 0==*sxy) {
	 thisgroup->color= *bxy;
          thisgroup->x= x;
          thisgroup->y= y;
          thisgroup->nliberties= 0;
          thisgroup->nstones= 0;
	  GoAiCount(x, y, thisgroup, scratch, 1+thisgroup-groups);
          ++thisgroup;
          ++ngroups;
        }
      }
  }
	return(0);
}


int GoAiPlaceStone(int x, int y, int p)
/*
Put a stone for player p at (x,y) and remove all the captures which result.
Update ko information.
*/
{
	int ncap, capx, capy;
	int j;
	int *ip;
	group g;
	goboard scratch;
	int player;

  assert(onboard(x, y));
  assert(BLACKSIDE==p||WHITESIDE==p);
  assert(EMPTY==wallyboard[x][y]);

  wallyboard[x][y]= p;

	if(p==WHITESIDE) {
		GoAiVerbose("Setting a white stone");
/*
		GoAiVerbose("Press the spacebar");
		getch();
*/
	} else {
		GoAiVerbose("Setting a black stone");
	}

/* from original Wally */
/*

*/



  GoAiMakeGroups(); 				/*Group all stones and count libs.*/
/*  return(0); */

  /* modified for this test to make DJ happy */
/*  if(p==BLACKSIDE) p=WHITESIDE; else p=BLACKSIDE; */

	p=Opponent;


	if(p==BLACKSIDE) GoAiVerbose("p=BLACKSIDE");
	if(p==WHITESIDE) GoAiVerbose("p=WHITESIDE");
/*
	GoAiVerbose("Press the spacebar");
	getch();
*/
  ncap=GoAiCapture(player, &capx, &capy);/*Remove p's opponent's dead stones.*/
  thegame.kox= thegame.koy= (-1);	/*Cancel any old ko.*/
  if(1==ncap)	/*If exactly one stone was captured, check for new ko.*/
  {

    for(ip=(int*)scratch,j=EDGE*EDGE; j--; ) *ip++= 0;
    g.color= wallyboard[x][y];
    g.nstones= 0;
    g.nliberties= 0;
    g.x= x;
    g.y= y;
    GoAiCount(x, y, &g, scratch, 1); /*Count stones and libs for capturing group.*/
    if(1==g.nstones && 1==g.nliberties)	/*If c.g. is a single stone in atari */
    { thegame.kox= capx; thegame.koy= capy; } /* then ko, so mark it.*/
  }
  if(ncap)		/*If any enemy stones were removed, */
    GoAiMakeGroups();	/* recalculate the goboard.*/
  ncap=GoAiCapture(p, &capx, &capy); /*Remove any still-trapped stones (suicide).*/
  assert(WHITESIDE==p||0==ncap);	/*Wally knows not to make suicides.*/

	return(0);
}



void GoAiSortV(group **p, int n)
/*
Sort the table p[] or pointers to groups in order of vulnerability,
most vulnerable first.
Currently this is the slow simple bubble sort we all know.
*/
{
	int uns;	/*flag that an unsorted pair was found last pass*/
	group *t;	/*exchange temporary*/
	group **ip;	/*pointer through p[]*/
	int j;		/*loop index through p*/

  do
  { uns= 0;
    for(j=0,ip=p; j<n-1; ++j,++ip)
    { t= ip[1];
      if( t->nliberties< (*ip)->nliberties ||
	 (t->nliberties==(*ip)->nliberties && t->nstones>(*ip)->nstones) )
      { uns= 1;
        ip[1]= *ip;
	*ip= t;
      }
    }
  } while(uns);

}


int GoAiSubjLib(int x, int y)
/*
Return the subjunctive liberties of the group created by a BLACKSIDE move at x, y,
 that is, how many liberties would it have after the move was made?
This is done shoddily, without considering the liberties created by
 the disappearance of any groups captured by the move.
*/
{
	goboard scratch;	/*parameter array of flags for GoAiCount()*/
	group t;	/*parameter group for GoAiCount()*/
        int *s, j;

  assert(onboard(x,y));
  assert(EMPTY==wallyboard[x][y]); /*Make sure we can undo by restoring EMPTY.*/

  wallyboard[x][y]= BLACKSIDE;	/*Temporarily place the stone in question.*/

/*Clear scratch[][] so GoAiCount() works properly.*/
  {
    for(j=EDGE*EDGE,s=(int*)scratch; j--; )
      *s++= 0;
  }

/*Make a fake group for the new stone to be part of -- this is */
/* a necessary parameter for GoAiCount().*/
  t.color= BLACKSIDE;
  t.x= x;
  t.y= y;
  t.nliberties= 0;
  t.nstones= 0;

/*Show this little Potemkin village to the count().*/
  GoAiCount(x, y, &t, scratch, 1);

/*Undo the temporary move (x,y).*/
  wallyboard[x][y]= EMPTY;

/*Return the result from GoAiCount().*/
  return(t.nliberties);

	return(0);
}



/*
A routine called by GoAiPattern() to match possible moves to
tabulated patterns of specific orientation.
*u gives the urgency of the best move found so far; we
are to ignore moves which are inferior to this.

If this move has not already been found (compare to the
urgency value in movehere[]) then we set its urgency
value in *u and in movehere[], and put it into the table
goodmoves[].  If the move is better than any found before,
we put it at the beginning of the table; otherwise we
put it in at the pointer pgoodmoves.

masks[][] is an array which represents the pieces on the goboard,
translated into bit masks by lookup in flookup[].

A late-breaking modification is that only the EMPTY points which have
movehere[][] set are tested, so that GoAiPattern() can be called to look for
contact plays or possibly other restricted sets which make good patterns.
An even later modification permits the use of movehere[][] to
tell the urgency of the best move which has been made to that
point.
*/

int GoAiPattern1(int *u, goboard masks, goboard movehere)
{
	int *is, *iis;	/*pointers into patterns[]; or scratch*/
	int j;		/*& into a particular pattern, # points remaining*/
	int x, y;	/*current position we're trying to match pattern to*/
	int xs, ys;	/*position of a point from a pattern*/
	int ua;		/*urgency and adjustment for this move*/
	int thispat;	/*which pattern in the table are we currently */
			/* trying to match?*/


  for(is=patterns,thispat=0; PATTERN==*is; is+= 3+3*(is[1]),++thispat)
  { for(x=0; x<edge; ++x)
      for(y=0; y<edge; ++y)
      { if(F_EMPTY==masks[x][y] && 0==ko(x,y)
         &&(thegame.kox!=x||thegame.koy!=y) )
        { for(iis=is+1,j= *iis++,ua= *iis++; j; --j)
	  { xs= *iis++;
	    ys= *iis++;
	    if(onboard(x+xs,y+ys))
	    { if(0==(masks[x+xs][y+ys]&*iis++)) goto mismatch; }
	    else
	    { if(0==(F_OFF&*iis++)) goto mismatch; }
	  }
	  /*If we fall through here, then we matched a pattern.*/
	  /*Compute adjusted urgency.*/
	  if(abs(x-lex)<=1&&abs(y-ley)<=1)
	    ua-=4;	            /*Important to oppose enemy's last move.*/
	  for(xs=(-2); xs<=2; ++xs)
	    for(ys=(-2); ys<=2; ++ys)
	      if(onboard(x+xs,y+ys))/*Important to move to */
	        ++ua;		    /* uncrowded parts of the goboard.*/
	  if( ua<*u &&		    /*If this pattern is most urgent so far */
	      0==intoatari(x, y) )  /* and the move is not futile */
	  { *u= ua;	 	    /*Replace old */
	    movehere[x][y]= ua;     /* urgency values.*/
	    patnum= thispat;	    /*Record pattern # for debugging.*/
	    pgoodmoves= goodmoves;  /*Reinit goodmoves[].*/
	    goto intogoodmoves;
	  }
	  else if( ua==*u &&	    /*If there's no better pattern */
	           ua<movehere[x][y] /* and it's the best move here */
		   && !intoatari(x,y)) /*and not futile*/
	  { movehere[x][y]= ua;	/*Mark as best move here.*/
	  intogoodmoves: ;	/*Put it into goodmoves[].*/
	    *pgoodmoves++= x;  *pgoodmoves++= y;
	  }
        }
	mismatch: ;
      }
  }
/*
  if(PATTEND!=*is)
  { fprintf(stderr, "?error in pattern table, pattern # %d\n", thispat);
    GoAiPanic("programmer error in pattern table");
  }
*/
	return(0);
}



/*
Try to find a good black move by matching to a table of patterns.
Returns the urgency of the move; returns BIGU if no match found.

In order to match the patterns in all orientations, we
reflect the entire table eight times, checking for a match each time.
The reflections are
  (1)   y <-> edge-1-y	(across a mirror plane parallel to x-axis)
  (2)	x <-> edge-1-x	( " y-axis)
  (3)   y <-> edge-1-y	( " x-axis)
  (4)   x <-> y		( " the line y=x)
  (5)   y <-> edge-1-y	(across a mirror plane parallel to x-axis)
  (6)	x <-> edge-1-x	( " y-axis)
  (7)   y <-> edge-1-y	( " x-axis)
  (8)   x <-> -y	( " the line y=(-x))
Draw pictures if necessary to see how this works.
*/

int GoAiPattern(int *chosenx, int *choseny, int *urgency, goboard movehere)
{
	int x, y, t, j, *is;
	goboard scratch;

/*Translate the goboard to flags for easy comparison with pattern table.*/
  for(x=0; x<edge; ++x)
    for(y=0; y<edge; ++y)
      scratch[x][y]= flookup[wallyboard[x][y]];

  *urgency= BIGU;		/*No big move so far.*/
  pgoodmoves= goodmoves;	/*No moves so far.*/
  patnum= (-1);			/*For debugging: no pattern # so far.*/

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to untransformed table.*/

/*Invert y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { is += 1;
      *is= (-*is);
      is += 2;
    }
  }
  assert(PATTEND==*is);


  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Invert x coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { ;
      *is= (-*is);
      is += 3;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Invert y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { is += 1;
      *is= (-*is);
      is += 2;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Exchange x and y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { t= *is;
      *is= is[1];
      is[1]= t;
      is += 3;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Invert y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { is += 1;
      *is= (-*is);
      is += 2;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Invert x coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { ;
      *is= (-*is);
      is += 3;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Invert y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { is += 1;
      *is= (-*is);
      is += 2;
    }
  }
  assert(PATTEND==*is);

  GoAiPattern1(urgency, scratch, movehere);
				/*Find matches to transformed table.*/

/*Exchange x and -y coordinates in pattern table.*/
  for(is=patterns; PATTERN==*is; )
  { for(j= *++is,is+=2; j--; )
    { t= (-*is);
      *is= (-is[1]);
      is[1]= t;
      is += 3;
    }
  }
  assert(PATTEND==*is);

/*Try to pick a move from the table of equally good ones.*/
  { int nmoves= (pgoodmoves-goodmoves)/2;
    if(0==nmoves) return(0);
    else
    { int whichmove= GoAiRng(nmoves);
      *chosenx= goodmoves[2*whichmove];
      *choseny= goodmoves[2*whichmove+1];
    }
  }

/*We'd better not have decided on an illegal move.*/
#if ASSERT
  if( *urgency<BIGU )
    if( EMPTY!=wallyboard[*chosenx][*choseny] || ko(*chosenx,*choseny) )
      GoAiPanic("illegal move selected in GoAiPattern()");
#endif

	return(0);
}



/*
Set *rx and *ry to a move which attacks group *g and still has
at least ml liberties.

This routine tries to return the "best" contact play if several are possible.
This is currently implemented as the one as close to the
 (2.9)-th line as possible, that is, one on the third line is best,
 then one on the fourth, then one on the second, then one on the fifth,
 then etc.

Return TRUE on failure.
*/

int GoAiAttack(group *g, int *rx, int *ry, int ml)
{
	int bestx, besty;	/*best contact move found so far*/
	int edist;		/* 10--distance of this play from (2.9)-st line*/
	int x, y;
	goboard scratch;
	int j;
	int *is;
	int c, t;
	int toedge;

/*Clear scratch[][] so that GoAiCount() can work.*/
/*##### some strange brackets below? ###### */

  {
    for(j=EDGE*EDGE,is=(int*)scratch;  j--; )
      *is++= 0;
  }

/*Set scratch[][] again for the stones of the group, and recalculate the */
/* number of liberties of the group.  Also set scratch[][] for all the EMPTY */
/* points which are contact moves.*/
  g->nstones= 0;	/*Clear these before */
  g->nliberties= 0;	/* passing to GoAiCount, which increments them.*/
  GoAiCount(g->x, g->y, g, scratch, 1);

/*Print out scratch[][] as 9x9 table.*/

/*
  if(debugattack) {
    printf("$In attack after GoAiCount(), goboard and scratch are:\n");
    for(y=edge-1; y>=0; --y)
    { printf("$");
      for(x=0; x<edge; ++x)
      {
	t=wallyboard[x][y];
        if(EMPTY==t) c= '.';
        else if(BLACKSIDE==t) c= '#';
        else if(WHITESIDE==t) c= 'O';
        else c= '?';
        printf("%c%d ", c, scratch[x][y]);
      }
      printf("\n");
    }
  }
*/


/*Check that there are enough liberties to find a contact play.*/
  assert(g->nliberties>=0);

/*Look for the best adjacent point.*/
  edist= edge*10;	/*We haven't found a point near the (2.9)-th line yet.*/
  for(x=0; x<edge; ++x)  			/*For */
    for(y=0; y<edge; ++y)			/* all */
      if(EMPTY==wallyboard[x][y] && scratch[x][y]	/*  contact points*/
       && (thegame.kox!=x||thegame.koy!=y) )	/*  which don't violate ko */
	if(GoAiSubjLib(x,y)>=ml)			/* if not too suicidal*/
	{
	toedge= x;				/* distance to edge of goboard*/
	  if(toedge>edge-1-x) toedge= edge-1-x;
          if(toedge>y) toedge= y;
          if(toedge>edge-1-y) toedge= edge-1-y;
/*	  if(debugattack)
	    fprintf(stderr, "$Looking at %d,%d, toedge=%d, abs=%d.\n",
		x, y, toedge, abs(10*toedge-21));
*/
	  if(abs(10*toedge-19)<edist)	/*If closer to 2.9th line than before*/
          { edist= abs(10*toedge-21);	/* record */
/*	    if(debugattack) fprintf(stderr, "$Replacing edist.\n"); */
	    bestx= x;  besty= y;	/*  as the new best so far.*/
          }
/*
	  else
	    if(debugattack) fprintf(stderr, "$Not replacing edist=%d.\n", edist);
*/
	}

/*  if(debugattack) fprintf(stderr, "$Edist= %d.\n", edist); */

/*Return best result, if any.*/
  if(edist<edge*10)			/*If some good contact play found */
  { *rx= bestx;  *ry= besty;  		/* return it */
/*    if(debugattack) fprintf(stderr, "$Returning 0.\n"); */
    return(0);				/*  and success */
  }

  else
  { /* if(debugattack) fprintf(stderr, "$Returning 1.\n"); */
    return(1);
  }

}



/*
Set *rx and *ry to a play which lets group *g escape from atari.
*/

int GoAiEscape(group *g, int *rx, int *ry)
{
	int x, y;
	goboard scratch;
	int j;
	int *is;

/*Clear scratch[][] so that GoAiCount() can work.*/
  {
    for(j=EDGE*EDGE,is=(int*)scratch;  j--; )
      *is++= 0;
  }

/*Set scratch[][] again for the stones of the group, and recalculate the */
/* number of liberties of the group.  Also set scratch[][] for all the EMPTY */
/* points which are contact moves.*/
  g->nstones= 0;	/*Clear these before */
  g->nliberties= 0;	/* passing to GoAiCount, which increments them.*/
  GoAiCount(g->x, g->y, g, scratch, 1);

/*Check that there are enough liberties to find a contact play.*/
  assert(g->nliberties>=0);

/*Return any point which succeeds.*/
  for(x=0; x<edge; ++x)  			/*For */
    for(y=0; y<edge; ++y)			/* each */
      if(EMPTY==wallyboard[x][y] && scratch[x][y] /*  contact point */
       &&(thegame.kox!=x&&thegame.koy!=y) )	/*   which is not ko */
        if(0==intoatari(x,y))			/*    if not into atari */
	{ *rx= x;  *ry= y;  return(0); }	/*     then return it.*/

/*Otherwise return failure.*/
  return(1);
}


/*Calculate and execute and print out the computer's move.*/

int GoAiComputerMove()
{
	int j;
	group *g; 		/*index and pointer through groups[]*/
	int x, y;		/*coords of the move selected*/
	group *gt[EDGE*EDGE];	/*table of &groups in order of vulnerability*/
	group **fs, **es;	/*pointers into gt: start of friendly and */
				/* enemy tables*/
	group **igt;		/*scratch pointer through gt[]*/
	int nf, ne;		/*number of enemy and friendly groups*/
	int upattern, patternx, patterny;
	goboard scratch;
	int *ip;

/*Find the most urgent move to improve our shape.*/
/*First say that all moves are to be considered.*/

	for(ip=(int*)scratch,j=EDGE*EDGE; j--;) *ip++= BIGU;

	Opponent=WHITESIDE;
	GoAiLimit=0;

/*Then call pattern search.*/

	MessageY=0;
/*	rectfill(screen,561,101,753,410,BLACK); */
	GoAiVerbose("Pattern search");
	GoAiPattern(&patternx, &patterny, &upattern, scratch);

/*Make tables of friendly and enemy groups in gt[] at *fs and *es.*/


	GoAiVerbose("Group tables");
	for(j=0,g=groups,nf=0,fs=igt=gt; j<ngroups; ++j,++g) {
		if(BLACKSIDE==g->color) {
			++nf;
			*igt++= g;
			GoAiLimit++;
			GoShowLimit();
			if(GoAiLimit>100) goto random;
		}
	}

	for(j=0,g=groups,ne=0,es=igt; j<ngroups; ++j,++g) {
		if(WHITESIDE==g->color) {
			++ne;
			*igt++= g;
			GoAiLimit++;
			GoShowLimit();
			if(GoAiLimit>100) goto random;
		}
	}

/*Sort the tables in order of vulnerability to attack.*/

	GoAiVerbose("Sorting tables");
	GoAiSortV(fs, nf);
	GoAiSortV(es, ne);

/*Consider extremely urgent pattern moves.*/

	GoAiVerbose("Studying tables");

	if(upattern<16) {
movepattern:
	GoAiVerbose("at movepattern");
	GoAiLimit++;
	GoShowLimit();
	if(GoAiLimit>100) goto random;
		if(debugattack) {
			/* printf("$making pattern, pattern #=%d, urgency=%d.\n", patnum, upattern); */
			GoAiVerbose("Making pattern");
		}

		x=patternx;
		y=patterny;
		goto movexy;
	}

/*Capture the enemy!*/

  while(ne && 1==(*es)->nliberties)

  { if(0==GoAiAttack(*es, &x, &y, 0))
    { /* if(debugattack) printf("$Capturing the enemy.\n"); */
      goto movexy;
    }
    else /*(This case comes up if we can't capture because of ko.)*/
    { ++es; --ne; }		/*Go on to next most vulnerable group.*/
  }

	if(upattern<24) goto movepattern;

	GoAiVerbose("Plan B");


/*If we can't do that, then try to protect ourself.  But as per Dr. Millen, */
/* don't risk everything crawling along the edge without lookahead.*/
/*However, we have a more powerful computer than the good doctor, so we */
/* can afford to check another condition: if crawling connects to a healthy */
/* group immediately (as when the program makes a pattern match with */
/*	O O #	*/
/*	. . O .	*/
/*	~ ~ ~ ~ */
/* where '~' is off the board, then is attacked with */
/*	O O #   */
/*	. . O # */
/*	~ ~ ~ + */
/* it seems to make no sense to prohibit the program from defending with */
/*	O O #   */
/*	. O O # */
/*	~ ~ ~ ~ */
/* ..  so we *don't* prohibit it.*/

  while(nf && 1==(*fs)->nliberties)
  { if(GoAiEscape(*fs, &x, &y))	/*If we can't see how to escape */
    { ++fs; --nf; }		/* go on to next friendly group, */
    else			/*  otherwise */
    { if((0==x||edge-1==x||0==y||edge-1==y) && /* if crawling */
	 (2>=GoAiSubjLib(x,y)) )	/* and not connecting to strong group */
      { ++fs; --nf;  continue; } /* this is probably not a good move */
      else			/* but if not crawling or connecting to */
				/* a strong group, go for it.*/
/*
      if(debugattack)
        printf("$Protecting a friendly group from imminent capture.\n");
*/
      goto movexy;
    }
  }

	GoAiVerbose("Plan C");

	if(upattern<34) goto movepattern;

	GoAiVerbose("Plan D");

/*Try putting the enemy in atari.*/
  while(ne && 2==(*es)->nliberties)	/*For all vulnerable enemy groups */
  { if(GoAiAttack(*es, &x, &y, 2))		/* if we can't see how to attack */
    { ne--; ++es; }			/*  go on to the next enemy group, */
    else				/*   otherwise */
    { if(debugattack)
        	GoAiVerbose("Atari");
		AtariWarning=1;
	/* printf("$Putting an enemy group into atari.\n"); */
      goto movexy;			/*    attack the group.*/
    }
	GoAiLimit++;
	GoShowLimit();
	if(GoAiLimit>100) goto random;
  }

	GoAiVerbose("Plan E");

	if(upattern<BIGU) goto movepattern;

	GoAiVerbose("goto movepattern by passed");


/*Try to harass the most vulnerable enemy group.*/

	GoAiVerbose("Try attacking a weak group");


  while(ne)				/*For all remaining enemy groups */
  { if(GoAiAttack(*es, &x, &y, 3))	/* if we can't see how to attack safely */
    { --ne; ++es; }			/*  go on to next enemy group*/
    else				/*   otherwise*/
    { if(debugattack)
	/* printf("$Harassing the most vulnerable enemy group.\n"); */
      goto movexy;			/*    attack it*/
    }
	GoAiLimit++;
	GoShowLimit();
	if(GoAiLimit>100) goto random;
  }


	GoAiVerbose("Plan E");

random:

#if 0 /*This seems to lose points.  Let's see how we do without it.*/
/*Move randomly adjacent to no stones or edges.*/
  for(x=1; x<edge-1; ++x)  for(y=1; y<edge-1; ++y)
    if( EMPTY==wallyboard[x][y] &&
	EMPTY==wallyboard[x][y-1] && EMPTY==wallyboard[x][y+1] &&
        EMPTY==wallyboard[x-1][y] && EMPTY==wallyboard[x+1][y] )
    { if(debugattack)
	/* printf("$Moving randomly to fill space.\n");*/
      goto movexy;
    }
#endif

/*If nothing else comes to mind, pass.*/
/*  printf("%s passes.\n", progname); */


	GoAiVerbose("Um, the computer passes");

  thegame.kox= thegame.koy= (-1);
  GoAiMoveDone();
  if(thegame.qpa) return(BOTHPASS);
  else
  { thegame.qpa= 1;
    return(7);  /* understood as "Wally passes" by the LCARS interface */
  }

movexy:

/*
  printf("%s moves to %c%d.\n", progname, lettercol(x), y+1);
  fflush(stdout);
  if(ofile)
*/


/*	GoAiVerbose("Setting a black stone"); */


	GoAiPlaceStone(x, y, BLACKSIDE);
	ComputerMoveX=x;
	ComputerMoveY=y;
	thegame.qpa= 0;
	GoAiMoveDone();
	return(0);
}

/* execute human move */

int GoAiHumanMove(int x, int y, int pass)
{
	MessageY=0;
/*	rectfill(screen,561,101,753,410,BLACK); */
	Opponent=BLACKSIDE;

	thegame.pla=WHITESIDE;

	if(pass==2) return(4); 	/* If human resigns */
	if(pass==1) return(3);	/* If human passes */

/*	GoAiMoveDone(); */
	thegame.kox= thegame.koy= -1;
	ley= ley= -1000;	/* way off the board */

/*	if(thegame.qpa) return(5);
	else {
		thegame.qpa= 1;
		return(6);
	}
*/

	if(ko(x,y)) return(2); 	/* ko violation */

	GoAiPlaceStone(x, y, WHITESIDE);
	thegame.qpa= 0;
	GoAiMoveDone();
	return(0);
}

void GoAiFileScore(FILE *file,  int territory[2])
{
	int j;

  fprintf(file, "(the end of the game)\n");
  fprintf(file, "Final score is:\n");
  for(j=0; j<2; ++j)
    fprintf(file, "%s has %d territory.\n",
            BLACKSIDE==j?"Black":"White", territory[j]);
  j= territory[BLACKSIDE]-territory[WHITESIDE];
  if(0==j)
    fprintf(file, "(a tie)\n");
  else if(j>0)
    fprintf(file, "(Black wins.)\n");
  else
    fprintf(file, "(White wins.)\n");

}

/*This is the recursive part of GoAiIsTerritory().*/

int GoAiIsT(int x, int y, int n, goboard scratch)
{
	assert(onboard(x,y));
	assert(EMPTY==wallyboard[x][y]);
	assert(0==scratch[x][y]);
	scratch[x][y]=1;

	x+=1;
	if(onboard(x,y)) {
		if(EMPTY==wallyboard[x][y]) {
		if(0==scratch[x][y] && 0==GoAiIsT(x, y, n, scratch))  return(0);
	} else if(n==wallyboard[x][y]) return(0);
  }

  x= x-2;
  if(onboard(x,y))
  { if(EMPTY==wallyboard[x][y])
    { if(0==scratch[x][y] && 0==GoAiIsT(x, y, n, scratch))
        return(0);
    }
    else if(n==wallyboard[x][y])
      return(0);
  }

  x= x+1;
  y= y+1;
  if(onboard(x,y))
  { if(EMPTY==wallyboard[x][y])
    { if(0==scratch[x][y] && 0==GoAiIsT(x, y, n, scratch))
        return(0);
    }
    else if(n==wallyboard[x][y])
      return(0);
  }

  y= y-2;
  if(onboard(x,y))
  { if(EMPTY==wallyboard[x][y])
    { if(0==scratch[x][y] && 0==GoAiIsT(x, y, n, scratch))
        return(0);
    }
    else if(n==wallyboard[x][y])
      return(0);
  }

  return(1);

}


/*
Return TRUE if the EMPTY point x, y is to be counted as territory for
player p.  This is only called at the end of the game, and I see no
reason why it can't be slow.

I don't understand the (Ing?) fractional point rules, so I don't implement them.
 An empty point counts as territory only if only one player has stones adjacent
 to the block of empty spaces to which it belongs.
*/

int GoAiIsTerritory(int x, int y, int p)
{
	goboard scratch;
	int a,b;

/*	int x, y;  */ /* ## shadow parameters ## */

	assert(onboard(x,y));
	assert(EMPTY==wallyboard[x][y]);

/*Clear scratch[].*/

	for(a=0; a<edge; ++a)  for(b=0; b<edge; ++b) scratch[a][b]= 0;

/*Decide recursively.*/

	return(GoAiIsT(x, y, nextp(p), scratch));
}


void GoAiScore()
/*Compute and output the final game score.  Use Chinese rules, of course.*/
{
	int x, y, p;
	int territory[2];	/*each player's territory*/

/*Count territories.*/
	for(p=0; p<2; ++p) {
		territory[p]= 0;
		for(x=0; x<edge; ++x) {
			for(y=0; y<edge; ++y) {
				if(p==wallyboard[x][y]||EMPTY==wallyboard[x][y]&&GoAiIsTerritory(x,y,p)) ++territory[p];
			}
		}
	}

/*Output results.*/
/*  GoAiFileScore(stdout, territory); */
	BlackScore=territory[0];
	WhiteScore=territory[1];
}


