#include "printf.h"
#include "emit.h"


#define VERIFY_SPACE(num)		do { if ((uint32_t*)dest->bufEnd - (uint32_t*)dest->buf < (int32_t)(num)) return EmitErrNoSpace; }while(0)
#define VALIDATE_SHIFT()		do { if (!emitPrvValidateShift(shiftType, shiftAmt)) return EmitErrNotEncodeable; }while(0)
#define EMIT_WORD(val)			do { *((uint32_t*)dest->buf) = val; dest->buf = ((char*)dest->buf) + sizeof(uint32_t); } while(0)


#define ARM_INSTR_AND			0x00000000
#define ARM_INSTR_EOR			0x00200000
#define ARM_INSTR_SUB			0x00400000
#define ARM_INSTR_RSB			0x00600000
#define ARM_INSTR_ADD			0x00800000
#define ARM_INSTR_ADC			0x00a00000
#define ARM_INSTR_SBC			0x00c00000
#define ARM_INSTR_RSC			0x00e00000
#define ARM_INSTR_TST			0x01000000	//must have S set, Rd must be zero
#define ARM_INSTR_TEQ			0x01200000	//must have S set, Rd must be zero
#define ARM_INSTR_CMP			0x01400000	//must have S set, Rd must be zero
#define ARM_INSTR_CMN			0x01600000	//must have S set, Rd must be zero
#define ARM_INSTR_ORR			0x01800000
#define ARM_INSTR_MOV			0x01a00000	//Rn must be zero
#define ARM_INSTR_BIC			0x01c00000
#define ARM_INSTR_MVN			0x01e00000	//Rn must be zero



static bool emitPrvValidateShift(enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	switch (shiftType) {
		case EmitShiftLsl:
		case EmitShiftLsr:
		case EmitShiftAsr:
		case EmitShiftRor:
			return shiftAmt < 32;
		
		default:
			__builtin_unreachable();
			return false;
	}
}

static bool emitPrvFlagsMustSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags;
}

static bool emitPrvFlagsCanSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags || flagPrefs == EmitFlagsDoNotCare;
}

static bool emitPrvFlagsMustLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags;
}

static bool emitPrvFlagsCanLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags || flagPrefs == EmitFlagsDoNotCare;
}

void emitBufferInit(struct EmitBuf *dest, void* buf, uint32_t len)
{
	dest->curCc = EmitCcAl;
	dest->buf = dest->bufStart = buf;
	dest->bufEnd = (char*)dest->bufStart + (len &~ (sizeof(uint32_t) - 1));
}

enum EmitStatus emitSetCond(struct EmitBuf *dest, enum EmitCc cc)
{
	dest->curCc = cc;
	
	return EmitErrNone; 
}

enum EmitStatus emitSaveSpace(struct EmitBuf *dest, struct EmitBuf *spaceOutP, uint32_t numHalfwords)
{
	uint32_t numWords = numHalfwords / 2;
	
	if (numHalfwords % 2)
		return EmitErrInvalidInput;
	
	VERIFY_SPACE(numWords);
	
	emitBufferInit(spaceOutP, dest->buf, numWords * sizeof(uint32_t));
	dest->buf = ((char*)dest->buf) + sizeof(uint32_t) * numWords;
	
	return EmitErrNone; 
}

uintptr_t emitGetPtrToJumpHere(const struct EmitBuf *dest)
{
	return (uintptr_t)dest->buf;
}

void* emitBufferBackup(const struct EmitBuf *dest)
{
	return (void*)dest->buf;
}

void emitBufferRestore(struct EmitBuf *dest, void* position)
{
	dest->buf = (uint16_t*)position;
}

enum EmitStatus emitLLrawWord(struct EmitBuf *dest, uint32_t word)
{
	VERIFY_SPACE(1);
	EMIT_WORD(word);
	return EmitErrNone;
}

enum EmitStatus emitLLnop(struct EmitBuf *dest, bool w)
{
	//nop is mov r0, r0
	return emitLLmov(dest, 0, 0, EmitShiftLsl, 0, EmitLeaveFlags, false);
}

static enum EmitStatus emitPrvBxBlx(struct EmitBuf *dest, uint32_t regNo, bool withLink)
{
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t instr1		: 1;
			uint32_t link		: 1;
			uint32_t instr22	: 22;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.word = 0x012fff10, };
	
	instr.cc = dest->curCc;
	instr.rmNo = regNo;
	instr.link = withLink;

	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLbx(struct EmitBuf *dest, uint32_t regNo)
{
	return emitPrvBxBlx(dest, regNo, false);
}

#ifdef HAVE_ARM_v5

	enum EmitStatus emitLLblx(struct EmitBuf *dest, uint32_t regNo)
	{
		return emitPrvBxBlx(dest, regNo, true);
	}

#endif

enum EmitStatus emitLLudf(struct EmitBuf *dest, uint32_t imm, bool w)
{
	union {
		struct {
			uint32_t imm20		: 20;
			uint32_t instr8		: 8;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.word = 0x03000000, };
	
	if (imm >> 20)
		return EmitErrNotEncodeable;
	
	instr.cc = dest->curCc;
	instr.imm20 = imm;
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

static int32_t emitPrvCalcBranchOffset(struct EmitBuf *from, uintptr_t toAddr)
{
	uint32_t fromAddr = ((uintptr_t)from->buf) + 8;
	int32_t ofst = toAddr - fromAddr;
	
	return ofst / 4;
}

static enum EmitStatus emitPrvBranch(struct EmitBuf *dest, enum EmitCc cc, uintptr_t dstAddr, bool withLink)
{
	union {
		struct {
			uint32_t ofst24		: 24;
			uint32_t link		: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.word = 0x0a000000, };
	int32_t ofst = emitPrvCalcBranchOffset(dest, dstAddr);
	
	if ((ofst >> 23) != 0 && (ofst >> 23) != -1)
		return EmitErrNotEncodeable;
	
	instr.ofst24 = ofst;
	instr.link = withLink;
	instr.cc = cc;
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLbl(struct EmitBuf *dest, uintptr_t dstAddr)
{
	return emitPrvBranch(dest, dest->curCc, dstAddr, true);
}

enum EmitStatus emitLLbranch(struct EmitBuf *dest, uintptr_t dstAddr, enum EmitCc cc)
{
	uint32_t prevailingCond = dest->curCc;
	
	if (prevailingCond != EmitCcAl && cc != EmitCcAl && prevailingCond != cc)		//current "cond" state does not match this branch
		return EmitErrInternalErr;
	
	if (prevailingCond == EmitCcAl)
		prevailingCond = cc;
	
	return emitPrvBranch(dest, prevailingCond, dstAddr, false);
}

enum EmitStatus emitLLabsJump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	enum EmitStatus now;
	
	//LDR PC, [PC, #-4]	//load next word as address
	now = emitLLloadImm(dest, EMIT_REG_NO_PC, EMIT_REG_NO_PC, -4, EmitSzWord, false, EmitAdrModeIndex);
	if (now != EmitErrNone)
		return now;
	
	return emitLLrawWord(dest, dstAddr);
}

enum EmitStatus emitHLjump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	void *positionBackup;
	enum EmitStatus now;
	
	positionBackup = emitBufferBackup(dest);
	now = emitLLbranch(dest, dstAddr, EmitCcAl);
	if (now != EmitErrNotEncodeable)
		return now;
	emitBufferRestore(dest, positionBackup);

	now = emitLLabsJump(dest, dstAddr | 1);
	if (now != EmitErrNone)
		return now;
	
	return EmitErrNone;
}

static enum EmitStatus emitPrvDataProcessing(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, bool s, uint32_t adrMode)
{
	union {
		struct {
			uint32_t instr12	: 12;
			uint32_t rdNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t s			: 1;
			uint32_t instr7		: 7;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.word = instrVal | adrMode, };
	
	instr.rdNo = rdNo;
	instr.rnNo = rnNo;
	instr.s = s;
	instr.cc = dest->curCc;
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

//assumes a entirely valid request and processed shift values (incl rrx as ror 0)
static enum EmitStatus emitPrvThreeRegDataProcessing32(struct EmitBuf *dest, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool s, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	//encode third reg as addressing mode 1
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t zero		: 1;
			uint32_t shiftTyp	: 2;
			uint32_t shiftAmt	: 5;
			uint32_t unused		: 20;
		};
		uint32_t word;
	} mode = {.rmNo = rmNo, .shiftTyp = shiftType, .shiftAmt = shiftAmt, };
	
	return emitPrvDataProcessing(dest, EmitCcAl, instrVal, rdNo, rnNo, s, mode.word);
}

enum EmitStatus emitLLshiftByReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t srcReg, uint32_t shiftByReg, enum EmitShiftType shiftType, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	//encode 4th reg as addressing mode 1
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t one		: 1;
			uint32_t shiftTyp	: 2;
			uint32_t zero		: 1;
			uint32_t rsNo		: 4;
			uint32_t unused		: 20;
		};
		uint32_t word;
	} mode = {.rmNo = srcReg, .one = 1, .shiftTyp = shiftType, .rsNo = shiftByReg, };
	
	return emitPrvDataProcessing(dest, EmitCcAl, ARM_INSTR_MOV, rdNo, 0, emitPrvFlagsMustSetFlags(flagPrefs), mode.word);
}

static int32_t emitPrvImmEncodeEx(uint32_t imm, uint32_t rorBy)
{
	union {
		struct {
			uint32_t imm8		: 8;
			uint32_t rotAmt		: 4;
			uint32_t unused17	: 13;
			uint32_t one		: 1;
			uint32_t unused2	: 2;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} mode = {.imm8 = imm, .rotAmt = rorBy / 2, .one = 1, };
	
	if ((imm >> 8) || (rorBy >= 32) || (rorBy & 1))
		return -1;
	
	return mode.word;
}

static int32_t emitPrvImmEncode(uint32_t imm)
{
	uint32_t i;
	
	for (i = 0; i < 32; i += 2) {
		
		uint32_t roled = i ? (imm << i) | (imm >> (32 - i)) : imm;
		
		if (!(roled >> 8))
			return emitPrvImmEncodeEx(roled, i);
	}
	
	return -1;
}

static enum EmitStatus emitPrvImmDataProcessingLogical(struct EmitBuf *dest, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, bool s, uint32_t val, uint32_t rorBy)
{
	int32_t mode = s ? emitPrvImmEncodeEx(val, rorBy) : emitPrvImmEncode(rorBy ? (val >> rorBy) | (val << (32 - rorBy)) : val);	//if no S, do it in easy mode
	
	if (mode < 0)
		return EmitErrNotEncodeable;
	
	return emitPrvDataProcessing(dest, EmitCcAl, instrVal, rdNo, rnNo, s, mode);
}

static enum EmitStatus emitPrvImmDataProcessingArithmetic(struct EmitBuf *dest, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, bool s, uint32_t imm)
{
	int32_t mode = emitPrvImmEncode(imm);
	
	if (mode < 0)
		return EmitErrNotEncodeable;
	
	return emitPrvDataProcessing(dest, EmitCcAl, instrVal, rdNo, rnNo, s, mode);
}


enum EmitStatus emitLLmov(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_MOV, rdNo, 0, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLmvnReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_MVN, rdNo, 0, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLcmpReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_CMP, 0, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLcmnReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_CMN, 0, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLtstReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_TST, 0, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLteqReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_TEQ, 0, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLaddReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_ADD, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLsubReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_SUB, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLsbcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_SBC, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLorrReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_ORR, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLbicReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_BIC, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLandReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_AND, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLeorReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_EOR, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLadcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_ADC, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLrsbReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	return emitPrvThreeRegDataProcessing32(dest, ARM_INSTR_RSB, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

static enum EmitStatus emitPrvMulOrMla(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool s, bool mla)	//set Ra to 0 to get MUL
{
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t instr4		: 4;
			uint32_t rsNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t rdNo		: 4;
			uint32_t s			: 1;
			uint32_t acc		: 1;
			uint32_t instr6		: 6;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.word = 0x00000090, };
	
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	
	instr.rmNo = rnNo;
	instr.rsNo = rmNo;
	instr.rnNo = raNo;
	instr.rdNo = rdNo;
	instr.s = s;
	instr.acc = mla;
	instr.cc = dest->curCc;
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLmulReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvMulOrMla(dest, rdNo, rnNo, rmNo, 0, emitPrvFlagsMustSetFlags(flagPrefs), false);
}

enum EmitStatus emitLLmlaReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo)
{
	return emitPrvMulOrMla(dest, rdNo, rnNo, rmNo, raNo, false, true);
}

enum EmitStatus emitLLaddImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_ADD, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), imm);
}

enum EmitStatus emitLLsubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_SUB, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), imm);
}

enum EmitStatus emitLLeorImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_EOR, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), val, rorBy);
}

enum EmitStatus emitLLandImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_AND, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), val, rorBy);
}

enum EmitStatus emitLLbicImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_BIC, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), val, rorBy);
}

enum EmitStatus emitLLorrImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_ORR, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), val, rorBy);
}

enum EmitStatus emitLLtstImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_TST, 0, rnNo, true, val, rorBy);
}

enum EmitStatus emitLLteqImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_TEQ, 0, rnNo, true, val, rorBy);
}

enum EmitStatus emitLLadcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_ADC, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), imm);
}

enum EmitStatus emitLLsbcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_SBC, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), imm);
}

enum EmitStatus emitLLrsbImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_RSB, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), imm);
}

enum EmitStatus emitLLcmpImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_CMP, 0, rnNo, true, imm);
}

enum EmitStatus emitLLcmnImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm)
{
	return emitPrvImmDataProcessingArithmetic(dest, ARM_INSTR_CMN, 0, rnNo, true, imm);
}

enum EmitStatus emitLLmovImm(struct EmitBuf *dest, uint32_t regNo, uint32_t valIn, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_MOV, regNo, 0, emitPrvFlagsMustSetFlags(flagPrefs), valIn, rorBy);
}

enum EmitStatus emitLLmvnImm(struct EmitBuf *dest, uint32_t regNo, uint32_t valIn, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvImmDataProcessingLogical(dest, ARM_INSTR_MVN, regNo, 0, emitPrvFlagsMustSetFlags(flagPrefs), valIn, rorBy);
}

enum EmitStatus emitHLloadImmToReg(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool canCorruptNZ, bool canCorruptC, bool isInIt)
{
	bool first = true, inverted = false;
	uint32_t i, potential;
	enum EmitStatus now;
	int32_t imm;
	

	if (__builtin_popcount(val) >= 16) {
		val = ~val;
		inverted = true;
	}
	
	do {
		
		uint32_t instr = first ? (inverted ? ARM_INSTR_MVN : ARM_INSTR_MOV) : (inverted ? ARM_INSTR_BIC : ARM_INSTR_ORR);
		uint32_t rnNo = first ? 0 : regNo;
		
		for (i = 0; i < 32; i++) {
			
			potential = (val >> i) << i;
			
			if (emitPrvImmEncode(potential) >= 0)
				break;
		}
		
		if (i == 32)		//impossible
			return EmitErrNotEncodeable;
		
		val &=~ potential;
		
		now = emitPrvImmDataProcessingArithmetic(dest, instr, regNo, rnNo, false, potential);
		if (now != EmitErrNone)
			return now;
		
		first = false;
	} while (val);
	
	return EmitErrNone;
}

static enum EmitStatus emitPrvLdmStmW(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak, bool load, bool ia)
{
	union {
		struct {
			uint32_t regsMask	: 16;
			uint32_t rnNo		: 4;
			uint32_t L			: 1;
			uint32_t W			: 1;
			uint32_t S			: 1;
			uint32_t U			: 1;
			uint32_t P			: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.regsMask = regsMask, .cc = dest->curCc, .instr3 = 4, .rnNo = rnNo, .L = load, .W = wbak, .P = ia ? 0 : 1, .U = ia ? 1 : 0, };
	
	if (!regsMask || (regsMask >> 16))
		return EmitErrNotEncodeable;
	
	if (load && (regsMask & (1 << rnNo)) && wbak)
		return EmitErrNotEncodeable;
	
	if (!load && (regsMask & (1 << rnNo)) && wbak && (regsMask & ((1 << rnNo) - 1)))
		return EmitErrNotEncodeable;
	
	if (rnNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	#ifndef HAVE_ARM_v5
		//it IS encodeable but will not do a proper return on ARMv4. prevent the disaster by refusing to emit it
		if (load && (regsMask & (1 << EMIT_REG_NO_PC)))
			return EmitErrNotEncodeable;
	#endif
	
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, true, true);
}

enum EmitStatus emitLLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, true, false);
}

enum EmitStatus emitLLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, false, true);
}

enum EmitStatus emitLLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, false ,false);
}

static enum EmitStatus emitPrvLoadStoreImmMode2(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, int32_t imm, bool byte, enum EmitAddrMode adrMode)
{
	union {
		struct {
			uint32_t ofst		: 12;
			uint32_t rtNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t L			: 1;
			uint32_t W			: 1;
			uint32_t B			: 1;
			uint32_t U			: 1;
			uint32_t P			: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.instr3 = 2, .ofst = (imm < 0 ? -imm : imm), .cc = dest->curCc, .rtNo = rtNo, .rnNo = rnNo, .L = load, .U = imm >= 0, .B = byte, };
	
	if ((imm < 0 ? -imm : imm) >> 12)
		return EmitErrNotEncodeable;
	
	if (rtNo == EMIT_REG_NO_PC && byte)
		return EmitErrNotEncodeable;
	
	#ifndef HAVE_ARM_v5
		//it IS encodeable but will not do a proper return/jump/etc on ARMv4. prevent the disaster by refusing to emit it
		if (load && rtNo == EMIT_REG_NO_PC)
			return EmitErrNotEncodeable;
	#endif
	
	switch (adrMode) {
		case EmitAdrModeIndexWbak:
			instr.W = 1;
			//fallthrough
		case EmitAdrModeIndex:
			instr.P = 1;
			break;
		case EmitAdrModePostindex:
			break;
		default:
			return EmitErrNotEncodeable;
	}

	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvLoadStoreRegRegMode2(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, bool add, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, bool byte, enum EmitAddrMode adrMode)
{
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t zero		: 1;
			uint32_t shiftTyp	: 2;
			uint32_t shiftAmt	: 5;
			uint32_t rtNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t L			: 1;
			uint32_t W			: 1;
			uint32_t B			: 1;
			uint32_t U			: 1;
			uint32_t P			: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.instr3 = 3, .rmNo = rmNo, .shiftTyp = shiftType, .shiftAmt = shiftAmt, .cc = dest->curCc, .rtNo = rtNo, .rnNo = rnNo, .L = load, .U = add, .B = byte, };
	
	VALIDATE_SHIFT();
	
	if (rmNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	if (rnNo == EMIT_REG_NO_PC && adrMode != EmitAdrModeIndex)
		return EmitErrNotEncodeable;
	
	if (rnNo == rtNo && adrMode != EmitAdrModeIndex)
		return EmitErrNotEncodeable;
	
	if (rtNo == EMIT_REG_NO_PC && byte)
		return EmitErrNotEncodeable;
	
	switch (adrMode) {
		case EmitAdrModeIndexWbak:
			instr.W = 1;
			//fallthrough
		case EmitAdrModeIndex:
			instr.P = 1;
			break;
		case EmitAdrModePostindex:
			break;
		default:
			return EmitErrNotEncodeable;
	}

	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvLoadStoreImmMode3(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, int32_t imm, bool s, bool h, enum EmitAddrMode adrMode)
{
	union {
		struct {
			uint32_t immL		: 4;
			uint32_t one1		: 1;
			uint32_t H			: 1;
			uint32_t S			: 1;
			uint32_t one2		: 1;
			uint32_t immH		: 4;
			uint32_t rtNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t L			: 1;
			uint32_t W			: 1;
			uint32_t I			: 1;
			uint32_t U			: 1;
			uint32_t P			: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.cc = dest->curCc, .I = 1, .one1 = 1, .one2 = 1, .rnNo = rnNo, .rtNo = rtNo, .S = s, .H = h, .immL = (imm < 0 ? -imm : imm), .immH = (imm < 0 ? -imm : imm) >> 4, .L = load, .U = imm >= 0, };
	
	if ((imm < 0 ? -imm : imm) >> 8)
		return EmitErrNotEncodeable;
	
	if (rtNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	switch (adrMode) {
		case EmitAdrModeIndexWbak:
			instr.W = 1;
			//fallthrough
		case EmitAdrModeIndex:
			instr.P = 1;
			break;
		case EmitAdrModePostindex:
			break;
		default:
			return EmitErrNotEncodeable;
	}

	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvLoadStoreRegRegMode3(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, bool add, uint32_t rmNo, bool s, bool h, enum EmitAddrMode adrMode)
{
	union {
		struct {
			uint32_t rmNo		: 4;
			uint32_t one1		: 1;
			uint32_t H			: 1;
			uint32_t S			: 1;
			uint32_t one2		: 1;
			uint32_t zero		: 4;
			uint32_t rtNo		: 4;
			uint32_t rnNo		: 4;
			uint32_t L			: 1;
			uint32_t W			: 1;
			uint32_t B			: 1;
			uint32_t U			: 1;
			uint32_t P			: 1;
			uint32_t instr3		: 3;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.cc = dest->curCc, .one1 = 1, .one2 = 1, .rmNo = rmNo, .rnNo = rnNo, .rtNo = rtNo, .S = s, .H = h, .U = add, .L = load, };
	
	if (rtNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	if (rmNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	if (rnNo == EMIT_REG_NO_PC && adrMode != EmitAdrModeIndex)
		return EmitErrNotEncodeable;
	
	if (rnNo == rtNo && adrMode != EmitAdrModeIndex)
		return EmitErrNotEncodeable;
	
	switch (adrMode) {
		case EmitAdrModeIndexWbak:
			instr.W = 1;
			//fallthrough
		case EmitAdrModeIndex:
			instr.P = 1;
			break;
		case EmitAdrModePostindex:
			break;
		default:
			return EmitErrNotEncodeable;
	}

	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvLoadStoreImm(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	switch (opSz) {
		case EmitSzByte:
			if (!sext)			//LDRB/STRB
				return emitPrvLoadStoreImmMode2(dest, load, rtNo, rnNo, imm, true, adrMode);
			else if (load)		//LDRSB
				return emitPrvLoadStoreImmMode3(dest, load, rtNo, rnNo, imm, sext, false, adrMode);
			else
				return EmitErrNotEncodeable;
		
		case EmitSzHalfword:
			if (sext && !load)
				return EmitErrNotEncodeable;
			return emitPrvLoadStoreImmMode3(dest, load, rtNo, rnNo, imm, sext, true, adrMode);
		
		case EmitSzWord:
			if (sext)
				return EmitErrNotEncodeable;
			return emitPrvLoadStoreImmMode2(dest, load, rtNo, rnNo, imm, false, adrMode);
		
		default:
			return EmitErrNotEncodeable;
	}
}

enum EmitStatus emitLLstoreImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, enum EmitAddrMode adrMode)
{
	return emitPrvLoadStoreImm(dest, false, rtNo, rnNo, imm, opSz, false, adrMode);
}

enum EmitStatus emitLLloadImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	return emitPrvLoadStoreImm(dest, true, rtNo, rnNo, imm, opSz, sext, adrMode);
}

static enum EmitStatus emitPrvLoadStoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, bool add, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, enum EmitAddrMode adrMode, bool load, bool sext)
{
	switch (opSz) {
		case EmitSzByte:
			if (!sext)			//LDRB/STRB
				return emitPrvLoadStoreRegRegMode2(dest, load, rtNo, rnNo, add, rmNo, EmitShiftLsl, lslAmt, true, adrMode);
			else if (load && !lslAmt) 	//LDRSB
				return emitPrvLoadStoreRegRegMode3(dest, load, rtNo, rnNo, add, rmNo, sext, false, adrMode);
			else
				return EmitErrNotEncodeable;
		
		case EmitSzHalfword:
			if (sext && !load)
				return EmitErrNotEncodeable;
			return emitPrvLoadStoreRegRegMode3(dest, load, rtNo, rnNo, add, rmNo, sext, true, adrMode);
		
		case EmitSzWord:
			if (sext)
				return EmitErrNotEncodeable;
			return emitPrvLoadStoreRegRegMode2(dest, load, rtNo, rnNo, add, rmNo, EmitShiftLsl, lslAmt, false, adrMode);
		
		default:
			return EmitErrNotEncodeable;
	}
}

enum EmitStatus emitLLstoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz)
{
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, true, rmNo, lslAmt, opSz, EmitAdrModeIndex, false, false);
}

enum EmitStatus emitLLloadRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, bool sext)
{
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, true, rmNo, lslAmt, opSz, EmitAdrModeIndex, true, sext);
}

enum EmitStatus emitHLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	return emitLLldmia(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	return emitLLstmia(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	return emitLLldmdb(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	return emitLLstmdb(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLpush(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLstmdb(dest, EMIT_REG_NO_SP, regsMask, true);
}

enum EmitStatus emitHLpop(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLldmia(dest, EMIT_REG_NO_SP, regsMask, true);
}

#ifdef HAVE_ARM_v5
	enum EmitStatus emitLLclz(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
	{
		union {
			struct {
				uint32_t rmNo		: 4;
				uint32_t instr8		: 8;
				uint32_t rdNo		: 4;
				uint32_t instr12	: 12;
				uint32_t cc			: 4;
			};
			uint32_t word;
		} instr = {.word = 0x016f0f10, };
		
		instr.rmNo = rmNo;
		instr.rdNo = rdNo;
		instr.cc = dest->curCc;
		
		VERIFY_SPACE(1);
		EMIT_WORD(instr.word);
		return EmitErrNone;
	}
#endif

#ifdef HAVE_ARM_v5_DSP_EXTS
	enum EmitStatus emitLLldrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode)
	{
		if ((rtNo & 1) || rtNo2 - rtNo != 1 || rtNo == EMIT_REG_NO_LR)
			return EmitErrNotEncodeable;
		
		return emitPrvLoadStoreImmMode3(dest, false, rtNo, rnNo, imm, true, false, adrMode);
	}
	
	enum EmitStatus emitLLstrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode)
	{
		if ((rtNo & 1) || rtNo2 - rtNo != 1 || rtNo == EMIT_REG_NO_LR)
			return EmitErrNotEncodeable;
		
		return emitPrvLoadStoreImmMode3(dest, false, rtNo, rnNo, imm, true, true, adrMode);
	}
#endif

enum EmitStatus emitLLlongMul(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool isUnsigned, bool accumulate)
{
	union {
		struct {
			uint32_t rnNo		: 4;
			uint32_t instr4		: 4;
			uint32_t rmNo		: 4;
			uint32_t rdLoNo		: 4;
			uint32_t rdHiNo		: 4;
			uint32_t S			: 1;
			uint32_t accum		: 1;
			uint32_t sign		: 1;
			uint32_t instr5		: 5;
			uint32_t cc			: 4;
		};
		uint32_t word;
	} instr = {.cc = dest->curCc, .rnNo = rnNo, .instr4 = 9, .rmNo = rmNo, .rdLoNo = rdLoNo, .rdHiNo = rdHiNo, .accum = accumulate, .sign = !isUnsigned, .instr5 = 1,};

	if (((1 << rnNo) | (1 << rmNo) | (1 << rdLoNo) | (1 << rdHiNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	
	if (rdHiNo == rdLoNo || rdHiNo == rnNo || rdLoNo == rnNo)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_WORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLumull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, true, false);
}

enum EmitStatus emitLLumlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, true, true);
}

enum EmitStatus emitLLsmull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, false, false);
}

enum EmitStatus emitLLsmlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, false, true);
}

//XXX: emitLLtableBranch can easily be implemented for ARM but we do not need it yet

#ifdef HAVE_ARM_v5_DSP_EXTS

	static enum EmitStatus emitPrvSatAddSub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool doubling, bool subtract)
	{
		union {
			struct {
				uint32_t rmNo		: 4;
				uint32_t instr8		: 8;
				uint32_t rdNo		: 4;
				uint32_t rnNo		: 4;
				uint32_t zero		: 1;
				uint32_t sub		: 1;
				uint32_t dbl		: 1;
				uint32_t instr5		: 5;
				uint32_t cc			: 4;
			};
			uint32_t word;
		} instr = {.cc = dest->curCc, .instr5 = 2, .instr8 = 5, .dbl = doubling, .sub = subtract, .rnNo = rnNo, .rdNo = rdNo, .rmNo = rmNo, };
		
		if (((1 << rnNo) | (1 << rmNo) | (1 << rdNo)) & (1 << EMIT_REG_NO_PC))
			return EmitErrNotEncodeable;
	
		VERIFY_SPACE(1);
		EMIT_WORD(instr.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLqadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, false, false);
	}
	
	enum EmitStatus emitLLqsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, false, true);
	}
	
	enum EmitStatus emitLLqdadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, true, false);
	}
	
	enum EmitStatus emitLLqdsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, true, true);
	}
	
	//XXX: DSP multiplies can be supported but we do not need them yet

#endif

void* emitBufPtrToFuncPtr(void *buf)
{
	return buf;
}