#include "printf.h"
#include "emit.h"


#define VERIFY_SPACE(num)		do { if ((uint16_t*)dest->bufEnd - (uint16_t*)dest->buf < (int32_t)(num)) return EmitErrNoSpace; }while(0)
#define VALIDATE_SHIFT()		do { if (!emitPrvValidateShift(shiftType, shiftAmt)) return EmitErrNotEncodeable; }while(0)
#define EMIT_HALFWORD(val)		do { *((uint16_t*)dest->buf) = val; dest->buf = ((char*)dest->buf) + sizeof(uint16_t); } while(0)

#define INSTR_REV				0xba00
#define INSTR_REV16				0xba40
#define INSTR_REVSH				0xba60


static uint32_t emitPrvRor(uint32_t val, uint32_t rorBy)
{
	if (rorBy)
		val = (val >> rorBy) | (val << (32 - rorBy));
	
	return val;
}

static bool emitPrvIsDestWordAligned(const struct EmitBuf *dest)
{
	return !(((uintptr_t)(dest->buf)) & 3);
}

static bool emitPrvIsNullShift(enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return shiftType == EmitShiftLsl && shiftAmt == 0;
}

static bool emitPrvValidateShift(enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	switch (shiftType) {
		case EmitShiftLsl:
		case EmitShiftLsr:
		case EmitShiftAsr:
		case EmitShiftRor:
			return shiftAmt < 32;
		
		default:
			__builtin_unreachable();
			return false;
	}
}
	
static bool emitPrvFlagsCanUseShortInstr(enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	switch (flagPrefs) {
		case EmitLeaveFlags:
			return isInIt;
		
		case EmitSetFlags:
			return !isInIt;
		
		case EmitFlagsDoNotCare:
			return true;
		
		default:
			__builtin_unreachable();
			return false;
	}
}

static bool emitPrvFlagsMustSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags;
}

static bool emitPrvFlagsCanSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags || flagPrefs == EmitFlagsDoNotCare;
}

static bool emitPrvFlagsMustLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags;
}

static bool emitPrvFlagsCanLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags || flagPrefs == EmitFlagsDoNotCare;
}

void emitBufferInit(struct EmitBuf *dest, void* buf, uint32_t len)
{
	dest->buf = dest->bufStart = buf;
	dest->bufEnd = (char*)dest->bufStart + (len &~ (sizeof(uint16_t) - 1));
}

enum EmitStatus emitSaveSpace(struct EmitBuf *dest, struct EmitBuf *spaceOutP, uint32_t numHalfwords)
{
	VERIFY_SPACE(numHalfwords);
	
	emitBufferInit(spaceOutP, dest->buf, numHalfwords * sizeof(uint16_t));
	dest->buf = ((char*)dest->buf) + sizeof(uint16_t) * numHalfwords;
	
	return EmitErrNone; 
}

uintptr_t emitGetPtrToJumpHere(const struct EmitBuf *dest)
{
	return (uintptr_t)dest->buf;
}

void* emitBufferBackup(const struct EmitBuf *dest)
{
	return dest->buf;
}

void emitBufferRestore(struct EmitBuf *dest, void* position)
{
	dest->buf = position;
}


enum EmitStatus emitLLrawHalfword(struct EmitBuf *dest, uint32_t halfword)
{
	VERIFY_SPACE(1);
	EMIT_HALFWORD(halfword);
	return EmitErrNone;
}

enum EmitStatus emitLLnop(struct EmitBuf *dest, bool w)
{
	if (w) {	//just emit two nops on v6m
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(0xbf00);
		EMIT_HALFWORD(0xbf00);
		return EmitErrNone;
	}
	else {
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(0xbf00);
		return EmitErrNone;
	}
}

static enum EmitStatus emitPrvBxBlx(struct EmitBuf *dest, uint32_t regNo, bool withLink)
{
	union {
		struct {
			uint16_t instr3	: 3;
			uint16_t rmNo	: 4;
			uint16_t link	: 1;
			uint16_t inst8	: 8;
		};
		uint16_t word;
	} instr = {.word = 0x4700, };
	
	instr.rmNo = regNo;
	instr.link = withLink;

	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLbx(struct EmitBuf *dest, uint32_t regNo)
{
	return emitPrvBxBlx(dest, regNo, false);
}

enum EmitStatus emitLLblx(struct EmitBuf *dest, uint32_t regNo)
{
	return emitPrvBxBlx(dest, regNo, true);
}

enum EmitStatus emitLLcps(struct EmitBuf *dest, bool i, bool f, bool enable)
{
	union {
		struct {
			uint16_t inst4	: 4;
			uint16_t mask	: 1;
			uint16_t inst8	: 11;
		};
		uint16_t word;
	} instr = {.word = 0xb662, };
	
	if (!i || f)	//i must be set, f must be clear
		return EmitErrNotEncodeable;
	
	instr.mask = !enable;

	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLudf(struct EmitBuf *dest, uint32_t imm, bool w)
{
	if (w) {
		
		union {
			struct {
				uint16_t imm4		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		
		} instr1 = {.word = 0xf7f0};
		union {
			struct {
				uint16_t imm12		: 12;
				uint16_t instr4		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xa000, };
		
		instr2.imm12 = imm;
		instr1.imm4 = imm >> 12;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	else {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t instr8	: 8;
			};
			uint16_t word;
		} instr = {.word = 0xde00, };
		
		instr.imm8 = imm;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
}

static enum EmitStatus emitPrvBN(struct EmitBuf *dest, int32_t offset)
{
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t instr5	: 5;
		};
		uint16_t word;
	} instr = {.word = 0xe000, };
	
	if (offset >= 0x400 || offset < -0x400)
		return EmitErrNotEncodeable;
	
	instr.imm11 = offset;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvBccN(struct EmitBuf *dest, int32_t offset, enum EmitCc cc)
{
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t cc		: 4;
			uint16_t instr4	: 4;
		};
		uint16_t word;
	} instr = {.word = 0xd000, };
	
	if (cc == EmitCcAl || cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	if (offset >= 0x80 || offset < -0x80)
		return EmitErrNotEncodeable;
	
	instr.imm8 = offset;
	instr.cc = cc;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}
static enum EmitStatus emitPrvBW(struct EmitBuf *dest, int32_t offset)
{
	union {
		struct {
			uint16_t imm10	: 10;
			uint16_t s		: 1;
			uint16_t instr	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf000};
	
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t j2		: 1;
			uint16_t instr1	: 1;
			uint16_t j1		: 1;
			uint16_t instr2	: 2;
		};
		uint16_t word;
	} instr2 = {.word = 0x9000, };
	
	if (offset >= 0x800000 || offset < -0x800000)
		return EmitErrNotEncodeable;
	
	if (offset >= 0)
		offset ^= 0x600000;
	
	instr2.imm11 = offset;
	instr1.imm10 = offset >> 11;
	instr2.j2 = offset >> 21;
	instr2.j1 = offset >> 22;
	instr1.s = offset >> 23;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static int32_t emitPrvCalcBranchOffset(struct EmitBuf *from, uintptr_t toAddr)
{
	uint32_t fromAddr = ((uintptr_t)from->buf) + 4;
	int32_t ofst = toAddr - fromAddr;
	
	return ofst >> 1;
}

enum EmitStatus emitLLbl(struct EmitBuf *dest, uintptr_t dstAddr)
{
	int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
	
	union {
		struct {
			uint16_t imm10	: 10;
			uint16_t s		: 1;
			uint16_t instr	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf000};
	
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t j2		: 1;
			uint16_t instr1	: 1;
			uint16_t j1		: 1;
			uint16_t instr2	: 2;
		};
		uint16_t word;
	} instr2 = {.word = 0xd000, };
	
	if (offset >= 0x800000 || offset < -0x800000)
		return EmitErrNotEncodeable;
	
	if (offset >= 0)
		offset ^= 0x600000;
	
	instr2.imm11 = offset;
	instr1.imm10 = offset >> 11;
	instr2.j2 = offset >> 21;
	instr2.j1 = offset >> 22;
	instr1.s = offset >> 23;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLabsJump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	//PUSH {r0, r1}
	EMIT(HLpush, 0x0003);
	
	if (!emitPrvIsDestWordAligned(dest)) {
		
		//LDR R0, [PC, #8]
		EMIT(LLloadImm, 0, EMIT_REG_NO_PC, 1 * sizeof(uint32_t), EmitSzWord, false, EmitAdrModeIndex);

	}
	else {
		
		//LDR R0, [PC, #4]
		EMIT(LLloadImm, 0, EMIT_REG_NO_PC, 1 * sizeof(uint32_t), EmitSzWord, false, EmitAdrModeIndex);
		
		//NOP.N		//for alignment
		EMIT(LLnop, false);
	}
	
	//STR r0, [sp, #4]
	EMIT(LLstoreImm, 0, EMIT_REG_NO_SP, sizeof(uint32_t), EmitSzWord, EmitAdrModeIndex);

	//POP {r0, pc}
	EMIT(HLpop, 0x8001);

	//.long dstAddr
	EMIT(LLrawHalfword, dstAddr);
	EMIT(LLrawHalfword, dstAddr >> 16);
	
	return EmitErrNone;
}

enum EmitStatus emitLLbranch(struct EmitBuf *dest, uintptr_t dstAddr, enum EmitCc cc)
{
	int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
	void *positionBackup;
	enum EmitStatus now;
	
	//sanity check
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	//try simple things first
	if (cc == EmitCcAl)
		return emitPrvBN(dest, offset);
	else
		return emitPrvBccN(dest, offset, cc);
}

enum EmitStatus emitHLjump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	void *positionBackup;
	enum EmitStatus now;
	
	positionBackup = emitBufferBackup(dest);
	now = emitLLbranch(dest, dstAddr, EmitCcAl);
	if (now != EmitErrNotEncodeable)
		return now;
	emitBufferRestore(dest, positionBackup);
	
	now = emitLLabsJump(dest, dstAddr | 1);
	if (now != EmitErrNone)
		return now;
	
	return EmitErrNone;
}

enum EmitStatus emitLLextend(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, uint32_t rotateBy, bool byte, bool unsign)
{
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && !rotateBy) {
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t isByte	: 1;
				uint16_t unsign	: 1;
				uint16_t i8		: 8;
			};
			uint16_t word;	
		} instr = {.word = 0xb200, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		instr.isByte = byte;
		instr.unsign = unsign;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvReverse(struct EmitBuf *dest, int32_t instr16, uint32_t rdNo, uint32_t rmNo)
{
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = instr16, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLrev(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REV, rdNo, rmNo);
}

enum EmitStatus emitLLrev16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REV16, rdNo, rmNo);
}

enum EmitStatus emitLLrevsh(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REVSH, rdNo, rmNo);
}

enum EmitStatus emitLLbfx(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t lsb, uint32_t width, bool unsign, bool mayCorruptFlags)
{
	uint32_t msb = lsb + width;
	
	if (msb > 32 || lsb >= 32 || width > 32 || !mayCorruptFlags)
		return EmitErrNotEncodeable;
	
	if (msb != 32) {
		
		EMIT(LLmov, rdNo, rnNo, EmitShiftLsl, 32 - msb, EmitFlagsDoNotCare, false);
		rnNo = rdNo;
		lsb += 32 - msb;
	}
	
	if (lsb) {
		
		EMIT(LLmov, rdNo, rnNo, unsign ? EmitShiftLsr : EmitShiftAsr, lsb, EmitFlagsDoNotCare, false);
	}

	return EmitErrNone;
}

enum EmitStatus emitLLshiftByReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	uint32_t shiftType16;

	//no PC or SP in here
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	switch (shiftType) {
		case EmitShiftLsl:	shiftType16 = 0b010;	break;
		case EmitShiftLsr:	shiftType16 = 0b011;	break;
		case EmitShiftAsr:	shiftType16 = 0b100;	break;
		case EmitShiftRor:	shiftType16 = 0b111;	break;
		default: __builtin_unreachable();
	}
	
	if (EMIT_IS_LOREG(rdNo) && rdNo == rnNo && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t shiftType	: 3;
				uint16_t val_0x20	: 7;
			};
			uint16_t word;
		} instr = {.rdnNo = rdNo, .rmNo = rmNo, .shiftType = shiftType16, .val_0x20 = 0x20, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLmov(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//MOV.N
	if (!emitPrvFlagsMustSetFlags(flagPrefs) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		union {
			struct {
				uint16_t rdNo_lo	: 3;
				uint16_t rmNo		: 4;
				uint16_t rdNo_hi	: 1;
				uint16_t instr8		: 8;
			};
			uint16_t word;
		} instr = {.word = 0x4600, };
		
		instr.rdNo_lo = rdNo;
		instr.rmNo = rmNo;
		instr.rdNo_hi = rdNo >> 3;
		
		if (rdNo != rmNo) {				//do not emit useless code
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
		}

		return EmitErrNone;
	}
	
	//LSL.N (reg) LSR.N (reg) ASR.N(reg)
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && shiftType != EmitShiftRor) {
		
		static const uint16_t instrs[] = {[EmitShiftLsl] = 0x0000, [EmitShiftLsr] = 0x0800, [EmitShiftAsr] = 0x1000, };
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t imm5	: 5;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.word = instrs[shiftType], };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		instr.imm5 = shiftAmt;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLmvnReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//MVN.N
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x43c0, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLcmpReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	//for short instrs, only PC is forbidden
	if (((1 << rnNo) | (1 << rmNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4280, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else if (emitPrvIsNullShift(shiftType, shiftAmt)) {
		union {
			struct {
				uint16_t rnNo_lo	: 3;
				uint16_t rmNo		: 4;
				uint16_t rnNo_hi	: 1;
				uint16_t instr8		: 8;
			};
			uint16_t word;
		} instr = {.word = 0x4500, };
		
		instr.rnNo_lo = rnNo;
		instr.rmNo = rmNo;
		instr.rnNo_hi = rnNo >> 3;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLcmnReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x42c0, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLtstReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4200, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLaddReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt)) {
		//short 3-op add?
		if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
			union {
				struct {
					uint16_t rdNo	: 3;
					uint16_t rnNo	: 3;
					uint16_t rmNo	: 3;
					uint16_t instr7	: 7;
				};
				uint16_t word;
			} instr = {.word = 0x1800, };
			
			instr.rdNo = rdNo;
			instr.rnNo = rnNo;
			instr.rmNo = rmNo;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
		
		//short 2-op add?
		if ((rdNo == rmNo || rdNo == rnNo) && !emitPrvFlagsMustSetFlags(flagPrefs)) {
			
			union {
				struct {
					uint16_t rdnNo_lo	: 3;
					uint16_t rmNo		: 4;
					uint16_t rdnNo_hi	: 1;
					uint16_t instr8		: 8;
				};
				uint16_t word;
			} instr = {.word = 0x4400, };
			
			instr.rdnNo_lo = rdNo;
			instr.rmNo = rmNo + rnNo - rdNo;	//calculates the value of whichever is NOT same as Rd
			instr.rdnNo_hi = rdNo >> 3;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLsubReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//short 3-op add?
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rnNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t instr7	: 7;
			};
			uint16_t word;
		} instr = {.word = 0x1a00, };
		
		instr.rdNo = rdNo;
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLsbcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4180, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLorrReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4300, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLbicReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4380, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLmulReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	//no SP no PC
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rmNo || rdNo == rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdmNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4340, };
		
		instr.rdmNo = rdNo;
		instr.rnNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLandReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4000, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLeorReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4040, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo + rnNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLadcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4140, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvAddSubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt, bool isSub)
{
	enum EmitStatus now;
	int32_t encodedImm;

	//if imm is zero and flags aren't needed, use a mov (more encoding options)
	if (imm == 0 && flagPrefs != EmitSetFlags) {
		
		return emitLLmov(dest, rdNo, rnNo, EmitShiftLsl, 0, flagPrefs, isInIt);
	}

	//ADD.N/SUB.N (two reg variant)
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && imm < 0x08) {
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rnNo	: 3;
				uint16_t imm3	: 3;
				uint16_t isSub	: 1;
				uint16_t instr6	: 6;
			};
			uint16_t word;
		} instr = {.rdNo = rdNo, .rnNo = rnNo, .imm3 = imm, .isSub = isSub, .instr6 = 0b000111};
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N/SUB.N (one reg variant)
	if (rdNo == rnNo && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && imm < 0x100) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rdnNo	: 3;
				uint16_t isSub	: 1;
				uint16_t instr4	: 4;
			};
			uint16_t word;
		} instr = {.imm8 = imm, .rdnNo = rdNo, .isSub = isSub, .instr4 = 0b0011, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N/SUB.N SP, SP, #imm
	if (rdNo == rnNo && rnNo == EMIT_REG_NO_SP && !emitPrvFlagsMustSetFlags(flagPrefs) && !(imm & 3) && imm < 0x200) {
		
		union {
			struct {
				uint16_t imm7	: 7;
				uint16_t isSub	: 1;
				uint16_t instr8	: 8;
			};
			uint16_t word;
		} instr = {.imm7 = imm >> 2, .isSub = isSub, .instr8 = 0b10110000, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N Rx, SP, #imm
	if (!isSub && EMIT_IS_LOREG(rdNo) && rnNo == EMIT_REG_NO_SP && !emitPrvFlagsMustSetFlags(flagPrefs) && !(imm & 3) && imm < 0x400) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rdNo	: 3;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.imm8 = imm >> 2, .rdNo = rdNo, .instr5 = 0b10101, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLaddImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvAddSubImm(dest, rdNo, rnNo, imm, flagPrefs, isInIt, false);
}

enum EmitStatus emitLLsubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvAddSubImm(dest, rdNo, rnNo, imm, flagPrefs, isInIt, true);
}

enum EmitStatus emitLLrsbImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	//NEG.N
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rdNo) && !imm && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.rdNo = rdNo, .rnNo = rnNo, .instr10 =0b0100001001, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLcmpImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm)
{
	int32_t encodedImm;
	
	//CMP.N
	if (EMIT_IS_LOREG(rnNo) && imm < 0x100) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rnNo	: 3;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.imm8 = imm, .rnNo = rnNo, .instr5 = 0b00101, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvMovImm_16bit(struct EmitBuf *dest, uint32_t regNo, uint32_t val)	//keeping track of flag allowances is your business
{
	if (EMIT_IS_LOREG(regNo) && !(val >> 24)) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t reg	: 3;
				uint16_t in5	: 5;
			};
			uint16_t word;
		} instr = {.word = 0x2000};
		
		instr.imm8 = val;
		instr.reg = regNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

#ifdef HAVE_v8M_BASE
	
	static enum EmitStatus emitPrvMovwOrMovt(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool movt)
	{
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t imm3	: 3;
				uint16_t imm1	: 1;
				uint16_t imm4	: 4;
			};
			uint16_t word;
		} from = {.word = val,};
		
		union {
			struct {
				uint16_t imm4	: 4;
				uint16_t in3a	: 3;
				uint16_t movt	: 1;
				uint16_t ins2	: 2;
				uint16_t imm1	: 1;
				uint16_t ins5	: 5;
			};
			uint16_t word;
		} instr1 = {.word = 0xf240};
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t regN	: 4;
				uint16_t imm3	: 3;
				uint16_t ins1	: 1;
			};
			uint16_t word;
		} instr2 = {.imm8 = from.imm8, .imm3 = from.imm3, .regN = regNo, };
		
		
		instr1.imm1 = from.imm1;
		instr1.imm4 = from.imm4;
		instr1.movt = movt;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}

	static enum EmitStatus emitPrvCbzCbnz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr, bool isCbnz)
	{
		int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t imm5		: 5;
				uint16_t instr1a	: 1;
				uint16_t i			: 1;
				uint16_t instr1b	: 1;
				uint16_t isCbnz		: 1;
				uint16_t instr4		: 4;
			};
			uint16_t word;
		} instr = {.word = 0xb100, };
		
		if (offset < 0 || offset >= 128)
			return EmitErrNotEncodeable;
		
		instr.rnNo = rnNo;
		instr.imm5 = offset;
		instr.i = offset >> 5;
		instr.isCbnz = isCbnz;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLcbz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr)
	{
		return emitPrvCbzCbnz(dest, rnNo, dstAddr, false);
	}
	
	enum EmitStatus emitLLcbnz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr)
	{
		return emitPrvCbzCbnz(dest, rnNo, dstAddr, true);
	}

	static enum EmitStatus emitPrvDiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool unsign)
	{
		//No SP or PC allowed
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
			return EmitErrNotEncodeable;
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr1		: 1;
				uint16_t unsign		: 1;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr1 = {.word = 0xfb90, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t instr4a	: 4;
				uint16_t rdNo		: 4;
				uint16_t instr4b	: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xf0f0, };
		
		instr1.rnNo = rnNo;
		instr1.unsign = 1;
		instr2.rmNo = rmNo;
		instr2.rdNo = rdNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLsdiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDiv(dest, rdNo, rnNo, rmNo, false);
	}
	
	enum EmitStatus emitLLudiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDiv(dest, rdNo, rnNo, rmNo, true);
	}
	
	static enum EmitStatus emitPrvLdrexStrex(struct EmitBuf *dest, uint32_t rdNo, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool load)
	{
		if (opSz == EmitSzWord) {
			
			union {
				struct {
					uint16_t rnNo		: 4;
					uint16_t load		: 1;
					uint16_t instr11	: 11;
				};
				uint16_t word;
			} instr1 = {.rnNo = rnNo, .load = load, .instr11 = 0b11101000010, };
			
			union {
				struct {
					uint16_t imm8		: 8;
					uint16_t rdNo		: 4;	//0x0f for LDREX
					uint16_t rtNo		: 4;
				};
				uint16_t word;
			} instr2 = {.imm8 = imm >> 2, .rdNo = rdNo, .rtNo = rtNo, };
			
			if ((imm & 3) || (imm > 0x400))
				return EmitErrNotEncodeable;
			
			VERIFY_SPACE(2);
			EMIT_HALFWORD(instr1.word);
			EMIT_HALFWORD(instr2.word);
			return EmitErrNone;
		}
		else {
			
			union {
				struct {
					uint16_t rnNo		: 4;
					uint16_t load		: 1;
					uint16_t instr11	: 11;
				};
				uint16_t word;
			} instr1 = {.rnNo = rnNo, .load = load, .instr11 = 0b11101000110, };
			
			union {
				struct {
					uint16_t rdNo		: 4;	//0x0f for LDREXx
					uint16_t halfword	: 1;
					uint16_t instr7		: 7;
					uint16_t rtNo		: 4;
				};
				uint16_t word;
			} instr2 = {.rdNo = rdNo, .halfword = opSz == EmitSzHalfword, .instr7 = 0b1111010, .rtNo = rtNo, };
			
			if (imm)
				return EmitErrNotEncodeable;
			
			VERIFY_SPACE(2);
			EMIT_HALFWORD(instr1.word);
			EMIT_HALFWORD(instr2.word);
			return EmitErrNone;
		}
	}
	
	enum EmitStatus emitLLstrex(struct EmitBuf *dest, uint32_t rdNo, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz)
	{
		return emitPrvLdrexStrex(dest, rdNo, rtNo, rnNo, imm, opSz, false);
	}
	
	enum EmitStatus emitLLldrex(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz)
	{
		return emitPrvLdrexStrex(dest, EMIT_REG_NO_PC, rtNo, rnNo, imm, opSz, true);
	}

#endif

enum EmitStatus emitLLmovImm(struct EmitBuf *dest, uint32_t regNo, uint32_t valIn, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	uint32_t val = emitPrvRor(valIn, rorBy);
	enum EmitStatus now;
	
	if (regNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	//maybe a MOV.N will do? we cannot use emitPrvFlagsCanUseShortInstr due to how flags are set with rotation
	if (EMIT_IS_LOREG(regNo) && !(val >> 8) && ((flagPrefs == EmitSetFlags && !rorBy && !isInIt) || flagPrefs == EmitFlagsDoNotCare || (flagPrefs == EmitLeaveFlags && isInIt)))
		return emitPrvMovImm_16bit(dest, regNo, val);
	
#ifdef HAVE_v8M_BASE

	if (!(val >> 16) && flagPrefs != EmitSetFlags)
		return emitPrvMovwOrMovt(dest, regNo, val, false);

#endif
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitHLloadImmToReg(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool canCorruptNZ, bool canCorruptC, bool isInIt)
{
	//this instr is guaranteed to never corrupt the V flag, and some code relies on that!

	enum EmitStatus now;
	int32_t encodedImm;
	
	if (regNo == EMIT_REG_NO_PC || regNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	//mov.n does not corrupt C flag, but does corrupt NZ outside of IT
	if ((isInIt || canCorruptNZ) && !(val >> 8) && EMIT_IS_LOREG(regNo))
		return emitPrvMovImm_16bit(dest, regNo, val);
	
	
	#ifdef HAVE_v8M_BASE	//movw/movt
	
		now = emitPrvMovwOrMovt(dest, regNo, val & 0xffff, false);
		if (now != EmitErrNone)
			return now;
		
		if (EMIT_IS_LOREG(regNo) && (val >> 15) == 0x1ffff) {		//sxth?
			
			EMIT(LLextend, regNo, regNo, 0, false, false);
		}
		else if (val >> 16) {
			now = emitPrvMovwOrMovt(dest, regNo, val >> 16, true);
			if (now != EmitErrNone)
				return now;
		}
		
		return EmitErrNone;
		
	#endif
	
	if (canCorruptC && canCorruptNZ && EMIT_IS_LOREG(regNo)) {	//iterative slow approach
		
		bool first = true;
		
		while(val) {
			
			uint32_t pos = __builtin_clz(val);
			int32_t shift = pos > 24 ? 0 : 24 - pos;
			uint32_t imm = val >> shift;
			uint32_t left = val - (imm << shift);
			uint32_t posNext = __builtin_clz(left);
			int32_t shiftNext = posNext > 24 ? 0 : 24 - posNext;
			uint32_t lshNow = left ? shift - shiftNext : shift;
			
			if (first)
				EMIT(LLmovImm, regNo, imm, 0, EmitFlagsDoNotCare, false);
			else
				EMIT(LLaddImm, regNo, regNo, imm, EmitFlagsDoNotCare, false);
			
			EMIT(LLmov, regNo, regNo, EmitShiftLsl, lshNow, EmitFlagsDoNotCare, false);

			val = left;
			first = false;
		}
		
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvLdmStmW(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak, bool load, bool ia)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, spMask = 1 << EMIT_REG_NO_SP, loRegsMask = 0x00ff;
	bool isSingleReg = !(regsMask & (regsMask - 1));
	
	if (rnNo == EMIT_REG_NO_PC || isSingleReg || (regsMask & spMask))
		return EmitErrNotEncodeable;
	else {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t load		: 1;
				uint16_t w			: 1;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr1 = {.word = ia ? 0xe880 : 0xe900, };
		
		instr1.rnNo = rnNo;
		instr1.load = load;
		instr1.w = wbak;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(regsMask);
		return EmitErrNone;
	}
}

enum EmitStatus emitLLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16) || (wbak && (regsMask & rnMask)))
		return EmitErrNotEncodeable;
	
	//POP.N?
	if (rnNo == EMIT_REG_NO_SP && wbak && !(regsMask & ~(pcMask | loRegsMask))) {
		
		union {
			struct {
				uint16_t regsListLo	: 8;
				uint16_t pcToo		: 1;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.word = 0xbc00, };
		
		instr.regsListLo = regsMask;
		instr.pcToo = regsMask >> EMIT_REG_NO_PC;
	
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//LDMIA.N?
	if (EMIT_IS_LOREG(rnNo) && !(regsMask & ~loRegsMask) && !wbak != !(regsMask & rnMask)) {
		
		union {
			struct {
				uint16_t regsList	: 8;
				uint16_t rnNo		: 3;
				uint16_t instr7		: 5;
			};
			uint16_t word;
		} instr = {.word = 0xc800, };
		
		instr.regsList = regsMask;
		instr.rnNo = rnNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16))
		return EmitErrNotEncodeable;
	
	//STMIA.N?
	if (EMIT_IS_LOREG(rnNo) && wbak && !(regsMask & ~loRegsMask)) {
		
		union {
			struct {
				uint16_t regsList	: 8;
				uint16_t rnNo		: 3;
				uint16_t instr7		: 5;
			};
			uint16_t word;
		} instr = {.word = 0xc000, };
		
		//if Rn is in the list, it must be the lowest numbered reg
		if ((regsMask & rnMask) && (regsMask & (rnMask - 1)))
			return EmitErrNotEncodeable;
		
		instr.regsList = regsMask;
		instr.rnNo = rnNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, spMask = 1 << EMIT_REG_NO_SP, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16) || (regsMask & (pcMask | spMask)) || (wbak && (regsMask & rnMask)))
		return EmitErrNotEncodeable;
	
	//PUSH.N?
	if (rnNo == EMIT_REG_NO_SP && wbak && !(regsMask & ~(lrMask | loRegsMask))) {
		
		union {
			struct {
				uint16_t regsListLo	: 8;
				uint16_t lrToo		: 1;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.word = 0xb400, };
		
		instr.regsListLo = regsMask;
		instr.lrToo = regsMask >> EMIT_REG_NO_LR;
	
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitPrvLoadStoreImm(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	static const uint16_t instrs[] = {[EmitSzByte] = 0xf800, [EmitSzHalfword] = 0xf820, [EmitSzWord] = 0xf840, };
	bool emitted = false, add = (imm >= 0);
	uint32_t immAbs = add ? imm : -imm;
		
	//writeback with Rn == Rt is not allowed in v6M
	if (adrMode != EmitAdrModeIndex && rtNo == rnNo)
		return EmitErrNotEncodeable;
	
	//loading SP has special restrictions
	if (load && rtNo == EMIT_REG_NO_SP) {
		
		//Erratum 752419: loading with writeback can cause issues in v7M if an exception happens at the same time - avoid emitting that
		if (adrMode != EmitAdrModeIndex) {
			logw("refusing to load SP with writeback!\n");
			return EmitErrNotEncodeable;
		}
		
		//non-word loads to SP are forbidden
		if (opSz != EmitSzWord)
			return EmitErrNotEncodeable;
	}
	
	//try short ones
	if (EMIT_IS_LOREG(rtNo) && adrMode == EmitAdrModeIndex) {
		
		//LDR/STR Rx, [SP, #...]
		if (rnNo == EMIT_REG_NO_SP && opSz == EmitSzWord && add && !(imm & 3) && imm < 0x400 && !sext) {
			
			union {
				struct {
					uint16_t imm8	: 8;
					uint16_t rtNo	: 3;
					uint16_t load	: 1;
					uint16_t instr4	: 4;
				};
				uint16_t word;
			} instr = {.word = 0x9000, };
			
			instr.imm8 = imm >> 2;
			instr.rtNo = rtNo;
			instr.load = load;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
		
		if (EMIT_IS_LOREG(rnNo) && add && !sext) {
			
			union {
				struct {
					uint16_t rtNo	: 3;
					uint16_t rnNo	: 3;
					uint16_t imm5	: 5;
					uint16_t load	: 1;
					uint16_t instr4	: 4;
					
				};
				uint16_t word;
			} instr = {.rtNo = rtNo, .rnNo = rnNo, .load = load, };
			
			switch (opSz) {
				case EmitSzByte:
					if (imm < 0x20) {
						instr.imm5 = imm;
						instr.instr4 = 0x07;
						emitted = true;
					}
					break;
				case EmitSzHalfword:
					if (!(imm & 1) && imm < 0x40) {
						instr.imm5 = imm >> 1;
						instr.instr4 = 0x08;
						emitted = true;
					}
					break;
				case EmitSzWord:
					if (!(imm & 1) && imm < 0x80) {
						instr.imm5 = imm >> 2;
						instr.instr4 = 0x06;
						emitted = true;
					}
					break;
				default:
					__builtin_unreachable();
					break;
			}
			if (emitted) {
				VERIFY_SPACE(1);
				EMIT_HALFWORD(instr.word);
				return EmitErrNone;
			}
		}
		
		//LDR Rx, [PC, #...]
		if (rnNo == EMIT_REG_NO_PC && opSz == EmitSzWord && add && !(imm & 3) && imm < 0x400 && !sext && load) {
			
			union {
				struct {
					uint16_t imm8	: 8;
					uint16_t rtNo	: 3;
					uint16_t instr5	: 5;
				};
				uint16_t word;
			} instr = {.word = 0x4800, };
			
			instr.imm8 = imm >> 2;
			instr.rtNo = rtNo;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstoreImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, enum EmitAddrMode adrMode)
{
	//all the special things we can do are for words
	if (opSz == EmitSzWord) {
		
		//STMIA.N ?
		if (imm == 4 && adrMode == EmitAdrModePostindex && EMIT_IS_LOREG(rtNo) && EMIT_IS_LOREG(rnNo))
			return emitLLstmia(dest, rnNo, 1 << rtNo, true);
		
		//PUSH.N ?
		if (imm == -4 && adrMode == EmitAdrModeIndexWbak && EMIT_IS_LOREG(rtNo)&& rnNo == EMIT_REG_NO_SP)
			return emitLLstmdb(dest, rnNo, 1 << rtNo, true);
	}
	
	return emitPrvLoadStoreImm(dest, false, rtNo, rnNo, imm, opSz, false, adrMode);
}

enum EmitStatus emitLLloadImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	//cannot sign extend a loaded word
	if (sext && opSz == EmitSzWord)
		return EmitErrNotEncodeable;
	
	//only word sized loads allowed into PC or SP
	if ((rtNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_SP)&& opSz != EmitSzWord)
		return EmitErrNotEncodeable;
	
	//all the special things we can do are for words in postindex mode
	if (opSz == EmitSzWord && imm == 4 && adrMode == EmitAdrModePostindex) {
		
		//POP.N or LDMIA.N?
		if ((rnNo == EMIT_REG_NO_SP && (EMIT_IS_LOREG(rtNo) || rtNo == EMIT_REG_NO_PC)) || (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rtNo)))
			return emitLLldmia(dest, rnNo, 1 << rtNo, true);
	}
	
	return emitPrvLoadStoreImm(dest, true, rtNo, rnNo, imm, opSz, sext, adrMode);
}

static enum EmitStatus emitPrvLoadStoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, uint32_t instr16)
{
	if (opSz != EmitSzWord && (rtNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_SP))
		return EmitErrNotEncodeable;
	
	if (rmNo == EMIT_REG_NO_PC || rmNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	if (lslAmt > 3)
		return EmitErrNotEncodeable;
	
	if (EMIT_IS_LOREG(rtNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && !lslAmt) {
		
		union {
			struct {
				uint16_t rtNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.rtNo = rtNo, .rnNo = rnNo, .rmNo = rmNo, .instr7 = instr16, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz)
{
	uint32_t instr16;
	
	switch (opSz) {
		case EmitSzByte:		instr16 = 0b0101010;	break;
		case EmitSzHalfword:	instr16 = 0b0101001;	break;
		case EmitSzWord:		instr16 = 0b0101000;	break;
		default:	__builtin_unreachable();
	}
	
	if (rnNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, rmNo, lslAmt, opSz, instr16);
}

enum EmitStatus emitLLloadRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, bool sext)
{
	uint32_t instr16;
	
	switch (opSz) {
		case EmitSzByte:
			instr16 = sext ? 0b0101011 : 0b0101110;
			break;
		
		case EmitSzHalfword:
			instr16 = sext ? 0b0101111 : 0b0101101;
			break;
		
		case EmitSzWord:
			if (sext)
				return EmitErrNotEncodeable;
			instr16 = 0b0101100;
			break;
		
		default:
			__builtin_unreachable();
	}
	
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, rmNo, lslAmt, opSz, instr16);
}

enum EmitStatus emitHLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	return emitLLldmia(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	return emitLLstmia(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	return emitLLstmdb(dest, rnNo, regsMask, wbak);
}

enum EmitStatus emitHLpush(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLstmdb(dest, EMIT_REG_NO_SP, regsMask, true);
}

enum EmitStatus emitHLpop(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLldmia(dest, EMIT_REG_NO_SP, regsMask, true);
}

static enum EmitStatus emitPrvMsrMrs(struct EmitBuf *dest, uint32_t rdNo, uint32_t sysm, uint32_t mask, uint32_t rnNo, bool isMrs)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = isMrs ? 0xf3ef : 0xf380, };
	union {
		struct {	//msr version
			uint16_t sysmA		: 8;
			uint16_t instr2		: 2;
			uint16_t mask		: 2;
			uint16_t instr4a	: 4;
		};
		struct {	//mrs version
			uint16_t sysmB		: 8;
			uint16_t rdNo		: 4;
			uint16_t instr4b	: 4;
		};
		uint16_t word;
	} instr2 = {.word = 0x8000, };
	
	if (isMrs) {
		instr2.sysmB = sysm;
		instr2.rdNo = rdNo;
	}
	else {
		instr2.sysmA = sysm;
		instr1.rnNo = rnNo;
		instr2.mask = mask;
	}
	
	if(rdNo == EMIT_REG_NO_SP || rnNo == EMIT_REG_NO_SP || rdNo == EMIT_REG_NO_PC || rnNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLmrs(struct EmitBuf *dest, uint32_t rdNo, uint32_t sysm)
{
	return emitPrvMsrMrs(dest, rdNo, sysm, 0, 0, true);
}

enum EmitStatus emitLLmsr(struct EmitBuf *dest, uint32_t sysm, uint32_t mask, uint32_t rnNo)
{
	return emitPrvMsrMrs(dest, 0, sysm, mask, rnNo, false);
}

enum EmitStatus emitLLsvc(struct EmitBuf *dest, uint32_t imm)
{
	if (imm >> 8)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(0xdf00 + imm);
	return EmitErrNone;
}

void* emitBufPtrToFuncPtr(void *buf)
{
	return (void*)(((uintptr_t)buf) + 1);
}