#include <string.h>
#include <stdint.h>
#include <PalmTypes.h>
#include <ErrorBase.h>
#include "halVibAndLed.h"
#include "vibAndLed.h"
#include "printf.h"
#include "kal.h"
#include "dal.h"

static struct HwVibAndLedConfig mLedCfg = {
	.pattern = 0xffff0000,
	.csecPerPattern = 100,
	.csecBetween = 0,
	.numTimes = 5,
};

static struct HwVibAndLedConfig mVibCfg = {
	.pattern = 0xfc03f000,
	.csecPerPattern = 100,
	.csecBetween = 0,
	.numTimes = 10,
};

#ifdef VIB_AND_LED_USE_SIMPLE_DRIVER

	struct PeriodicHardware {
		struct HwVibAndLedConfig cfg;
		uint16_t itersLeft;
		uint32_t timer;
		void (*setF)(bool on);
		uint16_t msecPerBit;
		uint8_t bit;	//32 when delaying
	};

	static struct PeriodicHardware mLedHw = {.cfg = {.pattern = 0xff000000, .csecPerPattern = 100, .csecBetween = 0, .numTimes = 5,}, .setF = vibAndLedSimpleSetLedState,};
	static struct PeriodicHardware mVibHw = {.cfg = {.pattern = 0xf0f00000, .csecPerPattern = 100, .csecBetween = 0, .numTimes = 3,}, .setF = vibAndLedSimpleSetVibState,};
	static uint32_t mMutex = 0;
	
	static struct PeriodicHardware* vibAndLedHwGet(bool led)
	{
		if (!mMutex) {
			if (KALMutexCreate(&mMutex, '_pHW'))
				fatal("Cannot create mutex for periodic HW\n");
		}
		KALMutexReserve(mMutex, -1);
		
		if (led && VIB_AND_LED_SIMPLE_LED_SUPPORTED)
			return &mLedHw;
		else if (!led && VIB_AND_LED_SIMPLE_VIB_SUPPORTED)
			return &mVibHw;
		return NULL;
	}

	static void vibAndLedHwRelease(void)
	{
		KALMutexRelease(mMutex);
	}
	
	static bool vibAndLedHwIsSupported(bool led)
	{
		bool ret = !!vibAndLedHwGet(led);
		vibAndLedHwRelease();
		
		return ret;
	}
	
	static void vibAndLedHwStopLocked(struct PeriodicHardware* hw)
	{
		if (hw && hw->timer) {
			KALTimerDelete(hw->timer);
			hw->timer = 0;
			hw->setF(false);
		}
	}
	
	static void vibAndLedHwStop(bool led)
	{
		vibAndLedHwStopLocked(vibAndLedHwGet(led));
		vibAndLedHwRelease();
	}
	
	static void vibAndLetTimerProc(void* userData)
	{
		struct PeriodicHardware* hw = (struct PeriodicHardware*)userData;	//warning: mutex is not locked when this func runs
		uint32_t delayAmt = 1;
		
		if (++hw->bit == 33)
			hw->bit = 0;
		
		if (hw->bit == 32) {
			hw->setF(false);
			if (!--hw->itersLeft)		//we leave timer created. so what?
				return;
			delayAmt = hw->cfg.csecBetween * 10;
			if (!delayAmt)
				hw->bit = 0;
		}
		
		if (hw->bit != 32) {
			hw->setF(!!((hw->cfg.pattern >> (31 - hw->bit)) & 1));
			delayAmt = hw->msecPerBit;
		}
		
		if (hw->timer)		//can go away from under us...
			KALTimerSet(hw->timer, delayAmt);
	}
	
	static void vibAndLedHwStart(bool led, const struct HwVibAndLedConfig* cfg)
	{
		struct PeriodicHardware* hw = vibAndLedHwGet(led);
		if (hw) {
			
			hw->cfg = *cfg;
			vibAndLedHwStopLocked(hw);
			hw->itersLeft = hw->cfg.numTimes;
			if (hw->itersLeft) {
				hw->msecPerBit = (hw->cfg.csecPerPattern * 10 + 16) / 32;
				if (!hw->msecPerBit)
					hw->msecPerBit++;
				hw->bit = 32;				//pretend we're in a delay
				
				KALTimerCreate(&hw->timer, '_pHW', vibAndLetTimerProc, hw);
				KALTimerSet(hw->timer, 1);	//start right away
			}
		}
		vibAndLedHwRelease();
	}
	
	static bool vibAndLedHwIsActive(bool led)
	{
		struct PeriodicHardware* hw = vibAndLedHwGet(led);
		bool ret = hw && hw->timer;
		vibAndLedHwRelease();
		return ret;
	}
	
#endif


static Err halVibAndLedAttrGetParamsCheck(uint32_t attr, bool led)
{
	if (!vibAndLedHwIsSupported(led))
		return LED_VIB_ERR_NOT_AVAIL;
	
	return errNone;
}

static Err halVibAndLedAttrGet(uint32_t attr, void *dataP, bool led)
{
	struct HwVibAndLedConfig *cfg = led ? &mLedCfg : &mVibCfg;
	Err ret;
	
	logt("halVibAndLedAttr%cet(%s, 0x%08x, ...)\n", 'G', led ? "LED" : "vib", attr);
	ret = halVibAndLedAttrGetParamsCheck(attr, led);
	if (ret != errNone)
		return ret;
	
	switch (attr) {
		
		case LED_VIB_ATTR_TYPE:
			*(uint32_t*)dataP = led ? LED_VIB_TYPE_LED : LED_VIB_TYPE_VIBRATOR;
			break;
			
		case LED_VIB_ATTR_ACTIVE:
			*(uint8_t*)dataP = vibAndLedHwIsActive(led);
			break;
		
		case LED_VIB_ATTR_RATE:
			*(uint16_t*)dataP = cfg->csecPerPattern;
			break;
		
		case LED_VIB_ATTR_PATTERN:
			*(uint32_t*)dataP = cfg->pattern;
			break;
		
		case LED_VIB_ATTR_DELAY:
			*(uint16_t*)dataP = cfg->csecBetween;
			break;
		
		case LED_VIB_ATTR_REPEAT_COUNT:
			*(uint16_t*)dataP = cfg->numTimes;
			break;
		
		default:
			loge("%cetting unknown attribute 0x%04x for %s\n", 'G', attr, led ? "LED" : "vibrator");
			return LED_VIB_ERR_NOT_AVAIL;
	}
	
	return errNone;
}

static Err halVibAndLedAttrSet(uint32_t attr, const void *dataP, bool led)
{
	struct HwVibAndLedConfig *cfg = led ? &mLedCfg : &mVibCfg;
	Err ret;
	
	logt("halVibAndLedAttr%cet(%s, 0x%08x, ...)\n", 'S', led ? "LED" : "vib", attr);
	ret = halVibAndLedAttrGetParamsCheck(attr, led);
	if (ret != errNone)
		return ret;
	
	switch (attr) {
		
		case LED_VIB_ATTR_ACTIVE:
			vibAndLedHwStop(led);
			if (*(uint8_t*)dataP)
				vibAndLedHwStart(led, cfg);
			break;
		
		case LED_VIB_ATTR_RATE:
			cfg->csecPerPattern = *(uint16_t*)dataP;
			break;
		
		case LED_VIB_ATTR_PATTERN:
			cfg->pattern = *(uint32_t*)dataP;
			break;
		
		case LED_VIB_ATTR_DELAY:
			cfg->csecBetween = *(uint16_t*)dataP;
			break;
		
		case LED_VIB_ATTR_REPEAT_COUNT:
			cfg->numTimes = *(uint16_t*)dataP;
			break;
		
		case LED_VIB_ATTR_INTERRUPT_OFF:
			//vibAndLedHwStop(led);
			break;
		
		default:
			loge("%cetting unknown attribute 0x%04x for %s\n", 'S', attr, led ? "LED" : "vibrator");
			return LED_VIB_ERR_NOT_AVAIL;
	}
	
	return errNone;
}

Err DALEXPORT impl_HALAttnGetLEDAttributes(uint32_t attr, void *dataP)
{
	return halVibAndLedAttrGet(attr, dataP, true);
}

Err DALEXPORT impl_HALAttnSetLEDAttributes(uint32_t attr, const void *dataP)
{
	return halVibAndLedAttrSet(attr, dataP, true);
}

Err DALEXPORT impl_HALAttnGetVibrateAttributes(uint32_t attr, void *dataP)
{
	return halVibAndLedAttrGet(attr, dataP, false);
}

Err DALEXPORT impl_HALAttnSetVibrateAttributes(uint32_t attr, const void *dataP)
{
	return halVibAndLedAttrSet(attr, dataP, false);
}