#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "CortexEmuCpu.h"
#include "soundIface.h"
#include "audiohw.h"
#include "printf.h"
#include "irqs.h"
#include "heap.h"
#include "ral.h"
#include "cpu.h"


#define SND_IN_CTRL			(*(volatile uint32_t*)(SOUND_UNIT_BASE + SOUND_OFST_IN_CTRL))
#define SND_IN_STA			(*(volatile uint32_t*)(SOUND_UNIT_BASE + SOUND_OFST_IN_STA))
#define SND_IN_BUF			((volatile uint32_t*)(SOUND_UNIT_BASE + SOUND_OFST_IN_BUF_0))
#define SND_IN_CUR_BUF		(*(volatile uint32_t*)(SOUND_UNIT_BASE + SOUND_OFST_IN_CUR_BUF))
#define SND_IN_RATE			(*(volatile uint32_t*)(SOUND_UNIT_BASE + SOUND_OFST_IN_RATE))

static AudioInHwSamplesReadyF mSampleWriteF = NULL;
static uint32_t mNumSamplesPerBuffer;
static int16_t* mHwBuffer;



void AudioIn_IRQHandler(void)
{
	if (mSampleWriteF) {
		
		uint32_t r9state = ralSetSafeR9();
		mSampleWriteF((uint16_t*)(SND_IN_BUF[SND_IN_CUR_BUF]), 0);
		ralRestoreR9(r9state);
	}
	
	NVIC_ClearPendingIRQ(AudioIn_IRQn);
}

bool audioInHwInit(AudioInHwSamplesReadyF acceptSamplesF, uint32_t *numSamplesPerBufP)
{
	uint32_t nSampPerBuf;
	
	if (!(SND_IN_CTRL & SOUND_CTRL_BIT_AVAIL)) {
		loge("Audio input not available on this HW\n");
		return false;
	}
	
	mNumSamplesPerBuffer = nSampPerBuf = SND_IN_STA;
	if (numSamplesPerBufP)
		*numSamplesPerBufP = nSampPerBuf;
	
	mHwBuffer = (int16_t*)kheapAlloc(nSampPerBuf * sizeof(int16_t) * 2);	//returned pointer is 4 bytes aligned as needed
	if (!mHwBuffer)
		fatal("Cannot alloc audio in buffer\n");
	
	SND_IN_BUF[0] = (uintptr_t)(mHwBuffer + 0 * nSampPerBuf);
	SND_IN_BUF[1] = (uintptr_t)(mHwBuffer + 1 * nSampPerBuf);
	
	NVIC_EnableIRQ(AudioIn_IRQn);
	
	mSampleWriteF = acceptSamplesF;
	
	return true;
}

bool audioInHwSetState(bool on, enum AudioSampleRate rate)
{
	if (on) {
		uint32_t rateVal;
		
		switch (rate) {
			case AudioRate8000:
				rateVal = 8000;
				break;

			case AudioRate11025:
				rateVal = 11025;
				break;
			
			case AudioRate16000:
				rateVal = 16000;
				break;
			
			case AudioRate22050:
				rateVal = 22050;
				break;
			
			case AudioRate24000:
				rateVal = 24000;
				break;
			
			case AudioRate32000:
				rateVal = 32000;
				break;
			
			case AudioRate44100:
				rateVal = 44100;
				break;
			
			case AudioRate48000:
				rateVal = 48000;
				break;
			
			default:
				fatal("microphone does not support this rate: %d\n", rate);
				return false;
		}
		
		SND_IN_RATE = rateVal;
		SND_IN_CUR_BUF = 0;
		memset(mHwBuffer, 0, sizeof(int16_t) * 2 * mNumSamplesPerBuffer);
		SND_IN_CTRL |= SOUND_CTRL_BIT_ON;
	}
	else {
		
		SND_IN_CTRL &=~ SOUND_CTRL_BIT_ON;
	}
	
	return true;
}

int32_t adcGetValue(enum AdcValueIdx which)
{
	return DAL_ADC_VAL_INVALID;
}


