#include "msioMemoryBacking.h"
#include "printf.h"
#include <string.h>
#include "app.inc"
#include "msio.h"


//we are a write-locked 4MB memory stick: 512 flash blocks of 16 pages each, boot blocks are 0 and 1
//we have no bad blocks, and never even any error (correctable or not)

#define NUM_BLOCKS				512

#define PART_SIZE				(((NUM_BLOCKS) - 16 /* bad block allowance */ - 2 /* boot block and copy */) * (BACKING_PAGES_PER_BLOCK) - 1 /* MBR */)

#define APP_FIST_CLUSTER		7




#if DRIVER_APP_SIZE > 16384
	#error "app is over 2 clusters. not yet supported"
#elif DRIVER_APP_SIZE > 8192
	#define HAVE_SECOND_APP_CLUSTER
#endif


struct MsBootBlkNfo {
    uint32_t start;                    //0x00 @ 0x170 0x17C 0x188 0x194
    uint32_t len;                      //0x04 @ 0x174 0x180 0x18C 0x198
    uint8_t type;                      //0x08 @ 0x178 0x184 0x190 0x19C
    uint8_t rfu[3];                    //0x09 @ 0x179 0x185 0x191 0x19D
} __attribute__((packed));             //0x0C

#define MS_BBI_TYPE_BAD_BLOCK_TAB   0x01
#define MS_BBI_TYPE_DEVICE_INFO     0x0A

struct MsBootBlock {              //BE
    uint16_t blkid;                    //0x000 - 0x01 for boot block
    uint16_t ftlVer;                   //0x002 - major.minor. must be >= 1.0, < 2.0, 
    uint8_t rfu0[0xB8];                //0x004
    uint8_t numBbis;                   //0x0BC - how many entries in "bbi[]" are used
    uint8_t rfu1[0xB3];                //0x0BD
    struct MsBootBlkNfo bbi[4];   //0x170, only first two seem to ever be used, real number used is in .numBbis
                                  //[0].type must be MS_BBI_TYPE_BAD_BLOCK_TAB
    
    uint8_t msClass;                   //0x1A0 - memory sticks have 0x01 here
    uint8_t msSubclass;                //0x1A1 - memory sticks have 0x02 here
    uint16_t kbPerBlock;               //0x1A2 - only 8 and 16 are acceptable values
    uint16_t numBlocks;                //0x1A4 - powers of 2 from 512 to 8192 acceptable
    uint16_t usableBlocks;             //0x1A6 - multiples of 0x1f0 << n acceptable, for n = 0..4. this is the number of "sectors" we show to the user more or less
    uint16_t pageSize;                 //0x1A8 - must be 512 for memorysticks
    uint8_t oobSize;                   //0x1AA - must be 16 for memorysticks, but 7 are used for ECC so only 9 are avail to us
    uint8_t unk_1AB;                   //0x1AB
    uint8_t manufTimeTzOfst;           //0x1AC
    uint8_t manufTimeYearHi;           //0x1AD
    uint8_t manufTimeYearLo;           //0x1AE
    uint8_t manufTimeMonth;            //0x1AF
    uint8_t manufDateDay;              //0x1B0
    uint8_t manufTimeHour;             //0x1B1
    uint8_t manufTimeMinute;           //0x1B2
    uint8_t manufTimeSecond;           //0x1B3
    uint32_t serialNum;                //0x1B4
    uint8_t assemblyMakerCode;         //0x1B8
    uint8_t assemblyModelCode[3];      //0x1B9
    uint16_t memManufCode;             //0x1BC
    uint16_t memModelCode;             //0x1BE
    uint32_t unk_1C0;                  //0x1C0
    uint8_t vpp;                       //0x1C4 (in decivolts)
    uint8_t vcc;                       //0x1C5 (in decivolts)
    uint16_t controllerNum;            //0x1C6
    uint16_t controllerFunc;           //0x1C8 - set to 0x1001 for magic gate sticks
    uint8_t rfu2[0x09];                //0x1CA
    uint8_t interfaceType;             //0x1D3
    uint16_t controllerCode;           //0x1D4
    uint8_t formatType;                //0x1D6 - must be 0x01
    uint8_t unk_1D7;                   //0x1D7
    uint8_t deviceType;                //0x1D8 - 0 is flash, nonzero = rom
    uint8_t rfu3[0x27];                //0x1D9
} __attribute__((packed));

struct FatDirEntry {
	char name[8], ext[3];
	uint8_t attr, rfu1, createdMsec;
	uint16_t createdTime, createdDate, accessDate, rfu2, modifiedTime, modifiedDate, cluster;
	uint32_t fileSize;
} __attribute__((packed));

static const struct MsBootBlock mBootBlock = {
	.blkid = __builtin_bswap16(0x0001),
	.ftlVer = __builtin_bswap16(0x0100),
	.numBbis = 1,
	.bbi = {
		[0] = {
			.start = __builtin_bswap32(0),
			.len = __builtin_bswap32(0),
			.type = MS_BBI_TYPE_BAD_BLOCK_TAB,
		},
	},
	.msClass = 0x01,
	.msSubclass = 0x02,
	.kbPerBlock = __builtin_bswap16(512 * BACKING_PAGES_PER_BLOCK / 1024),
	.numBlocks = __builtin_bswap16(NUM_BLOCKS),
	.usableBlocks = __builtin_bswap16(NUM_BLOCKS * 0x1f0ull / 0x200),
	.pageSize = __builtin_bswap16(512),
	.oobSize = 16,
	.vpp = 33,
	.vcc = 33,
	.controllerNum = __builtin_bswap16(0x0006),
	.formatType = 0x01,
};

static const uint8_t mMbr[] = {
	
	[0x1be] = 0x80,	//active
	[0x1c2] = 0x01,	//fat12
	[0x1c6] = 0x01,	//partition starts at sector 0x01
	[0x1ca] = PART_SIZE & 0xff,	//partition length
	[0x1cb] = PART_SIZE >> 8,
	
	[0x1fe] = 0x55,
	[0x1ff] = 0xaa,
};

static const uint8_t mPbr[] = {
	[0x000] = 0xe9,
	[0x003] = 'D',
	[0x004] = 'M',
	[0x005] = 'I',
	[0x006] = 'T',
	[0x007] = 'R',
	[0x008] = 'Y',
	[0x009] = 'G',
	[0x00a] = 'R',
	[0x00c] = 2,	//512 byte sectors
	[0x00d] = 16,	//one block (16 pages) per cluster (convenient, eh). each fat will then be 1.5 sectors (4 total)
	[0x00e] = 10,	//FATs start on sector 10 such that they end at sector 13, rootfs is 14, clusters start at 15 (#16 on device)
	[0x010] = 2,	//2 FATs (partition Sectors 10..11 and 12..13)
	[0x011] = 16,	//16 root dir entries (one sector of them)
	[0x013] = PART_SIZE & 0xff,
	[0x014] = PART_SIZE >> 8,
	[0x015] = 0xf0,
	[0x016] = 2,	//2 sectors per fat
	[0x018]	= 16,	//sectors per track (sony FAT driver checks for nonzero)
	[0x01a]	= 16,	//numbr of heads (sony FAT driver checks for nonzero)
	[0x026] = 0x29,	//extended PPB
	[0x027] = 0x78,	//snum
	[0x028] = 0x56,
	[0x029] = 0x34,
	[0x02a] = 0x12,
	[0x02b] = 'r',	//label
	[0x02c] = 'e',
	[0x02d] = 'P',
	[0x02e] = 'a',
	[0x02f] = 'l',
	[0x030] = 'm',
	[0x031] = '-',
	[0x032] = 'M',
	[0x033] = 'S',
	[0x034] = 'I',
	[0x035] = 'O',
	[0x036] = 'F',	//FS
	[0x037] = 'A',
	[0x038] = 'T',
	[0x039] = '1',
	[0x03a] = '2',
	[0x03b] = ' ',
	[0x03c] = ' ',
	[0x03d] = ' ',
	
	[0x1fe] = 0x55,
	[0x1ff] = 0xaa,
};

//sets two pointer vlues at once. _idx must be even
#define FAT_SET_CHAIN_VALS(_idx, _valEven, _valOdd)									\
	[(_idx) / 2  * 3 + 0] = (uint8_t)((_valEven) & 0xff),							\
	[(_idx) / 2  * 3 + 1] = (uint8_t)(((_valEven) >> 8) + ((_valOdd) << 4)),		\
	[(_idx) / 2  * 3 + 2] = (uint8_t)((_valOdd) >> 4)								\
	


static const uint8_t mFatSec0[] = {
	FAT_SET_CHAIN_VALS(0, 0xff0, 0xfff),
	FAT_SET_CHAIN_VALS(2, 0xfff, 0xfff),
	FAT_SET_CHAIN_VALS(4, 0xfff, 0xfff),
	
#ifdef HAVE_SECOND_APP_CLUSTER
	
	FAT_SET_CHAIN_VALS(APP_FIST_CLUSTER / 2 * 2, (APP_FIST_CLUSTER & 1) ? 0xfff : (APP_FIST_CLUSTER + 1), (APP_FIST_CLUSTER & 1) ? (APP_FIST_CLUSTER + 1) : 0xfff),
	
#endif

};

static const char mMsAutorunConfigFile[] = 
		"[VolumeMounted]\r\n"
		"launch \\PALM\\START.PRC\r\n"
		"[VolumeUnmounted]\r\n"
		"uninstall -a\r\n";

static const struct FatDirEntry mRootDir[] = {
	[0] = {
		.name = "PALM    ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 2,
	},
};

static const struct FatDirEntry mPalmDir[] = {
	[0] = {
		.name = ".       ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 2,
	},
	[1] = {
		.name = "..      ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 0,
	},
	[2] = {
		.name = "LAUNCHER",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 3,
	},
	[3] = {
		.name = "PROGRAMS",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 4,
	},
	[4] = {
		.name = "DEFAULT ",
		.ext = "ARN",
		.cluster = 6,
		.fileSize = sizeof(mMsAutorunConfigFile),
	},
	[5] = {		//same file in three directories? hell yeah!
		.name = "START   ",
		.ext = "PRC",
		.cluster = APP_FIST_CLUSTER,
		.fileSize = DRIVER_APP_SIZE,
	},
};

static const struct FatDirEntry mLauncherDir[] = {
	[0] = {
		.name = ".       ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 3,
	},
	[1] = {
		.name = "..      ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 2,
	},
	[2] = {		//same file in three directories? hell yeah!
		.name = "DRIVER  ",
		.ext = "PRC",
		.cluster = APP_FIST_CLUSTER,
		.fileSize = DRIVER_APP_SIZE,
	},
};

static const struct FatDirEntry mProgramsDir[] = {
	[0] = {
		.name = ".       ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 4,
	},
	[1] = {
		.name = "..      ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 2,
	},
	[2] = {
		.name = "MSFILES ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 5,
	},
};

static const struct FatDirEntry mMsfilesDir[] = {
	[0] = {
		.name = ".       ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 5,
	},
	[1] = {
		.name = "..      ",
		.ext = "   ",
		.attr = 0x10,
		.cluster = 4,
	},
	[2] = {		//same file in three directories? hell yeah!
		.name = "DRIVER  ",
		.ext = "PRC",
		.cluster = APP_FIST_CLUSTER,
		.fileSize = DRIVER_APP_SIZE,
	},
};

struct FlashPageData {
	const void *data;
	uint16_t dataLen;
	uint8_t dataFiller;
	uint8_t oob[4];
};

struct FlashPageRef {
	uint16_t block;
	uint8_t page;
	const struct FlashPageData *data;
};

static const struct FlashPageData mBootBlockPageZero = {
	.data = &mBootBlock,
	.dataLen = sizeof(mBootBlock),
	.oob = {0xc0, 0xfb, 0xff, 0xff},
};

static const struct FlashPageData mBootBlockPageOne = {
	.dataFiller = 0xff,
	.oob = {0xc0, 0xfb, 0xff, 0xff},
};

static const struct FlashPageData mMbrPage = {
	.data = &mMbr,
	.dataLen = sizeof(mMbr),
	.oob = {0xf8, 0xff, 0x00, 0x00},
};

static const struct FlashPageData mPbrPage = {
	.data = &mPbr,
	.dataLen = sizeof(mPbr),
	.oob = {0xf8, 0xff, 0x00, 0x00},
};

static const struct FlashPageData mFat0Page = {
	.data = &mFatSec0,
	.dataLen = sizeof(mFatSec0),
	.dataFiller = 0xff,
	.oob = {0xf8, 0xff, 0x00, 0x00},
};

static const struct FlashPageData mFat1Page = {
	.dataFiller = 0xff,
	.oob = {0xf8, 0xff, 0x00, 0x00},
};

static const struct FlashPageData mRootDirPage = {
	.data = &mRootDir,
	.dataLen = sizeof(mRootDir),
	.oob = {0xf8, 0xff, 0x00, 0x00},
};

static const struct FlashPageData mPalmDirPage = {
	.data = &mPalmDir,
	.dataLen = sizeof(mPalmDir),
	.oob = {0xf8, 0xff, 0x00, 0x01},
};

static const struct FlashPageData mLauncherDirPage = {
	.data = &mLauncherDir,
	.dataLen = sizeof(mLauncherDir),
	.oob = {0xf8, 0xff, 0x00, 0x02},
};

static const struct FlashPageData mProgramsDirPage = {
	.data = &mProgramsDir,
	.dataLen = sizeof(mProgramsDir),
	.oob = {0xf8, 0xff, 0x00, 0x03},
};

static const struct FlashPageData mMsfilesDirPage = {
	.data = &mMsfilesDir,
	.dataLen = sizeof(mMsfilesDir),
	.oob = {0xf8, 0xff, 0x00, 0x04},
};

static const struct FlashPageData mmMsAutorunConfigFilePage = {
	.data = &mMsAutorunConfigFile,
	.dataLen = sizeof(mMsAutorunConfigFile),
	.oob = {0xf8, 0xff, 0x00, 0x05},
};

/*
	This is a fun one. Acer s10 MS slot driver does not clear out its mapping table properly on mount
	and for some readon when reading any page, it checks if the next block is mapped and if not, returns
	expErrCardBadSector. This is even if next block is not going to be read. To resolve the issue we
	need to proactively map the next block after the app to somewhere. We do.
*/

static const struct FlashPageData mPostAppDataPage = {
	.dataFiller = 0xff,
#ifdef HAVE_SECOND_APP_CLUSTER
	.oob = {0xf8, 0xff, 0x00, APP_FIST_CLUSTER + 1},
#else
	.oob = {0xf8, 0xff, 0x00, APP_FIST_CLUSTER},
#endif
};

#define OFST_OF(_n, _inAppClusNo)					(512 * ((_n) + ((_inAppClusNo) * BACKING_PAGES_PER_BLOCK)))
#define FLASH_DATA_PAGE_NAME(_inAppClusNo, _n)		mAppPage ## _ ## _inAppClusNo ## _ ## _n
#define FLASH_DATA_PAGE_DRIVER_APP(_inAppClusNo, _n)																			\
	static const struct FlashPageData FLASH_DATA_PAGE_NAME(_inAppClusNo, _n) = {												\
		.data = mDriverApp + OFST_OF(_n, _inAppClusNo),																			\
		.dataLen = (DRIVER_APP_SIZE < OFST_OF(_n, _inAppClusNo)) ? 																\
					0 : 																										\
					(																											\
						(DRIVER_APP_SIZE - OFST_OF(_n, _inAppClusNo) > 512) ?													\
							512 :																								\
							(DRIVER_APP_SIZE - OFST_OF(_n, _inAppClusNo))														\
					),																											\
		.oob = {0xf8, 0xff, 0x00, APP_FIST_CLUSTER - 1 + _inAppClusNo},															\
	}

FLASH_DATA_PAGE_DRIVER_APP(0, 0);
FLASH_DATA_PAGE_DRIVER_APP(0, 1);
FLASH_DATA_PAGE_DRIVER_APP(0, 2);
FLASH_DATA_PAGE_DRIVER_APP(0, 3);
FLASH_DATA_PAGE_DRIVER_APP(0, 4);
FLASH_DATA_PAGE_DRIVER_APP(0, 5);
FLASH_DATA_PAGE_DRIVER_APP(0, 6);
FLASH_DATA_PAGE_DRIVER_APP(0, 7);
FLASH_DATA_PAGE_DRIVER_APP(0, 8);
FLASH_DATA_PAGE_DRIVER_APP(0, 9);
FLASH_DATA_PAGE_DRIVER_APP(0, 10);
FLASH_DATA_PAGE_DRIVER_APP(0, 11);
FLASH_DATA_PAGE_DRIVER_APP(0, 12);
FLASH_DATA_PAGE_DRIVER_APP(0, 13);
FLASH_DATA_PAGE_DRIVER_APP(0, 14);
FLASH_DATA_PAGE_DRIVER_APP(0, 15);

#ifdef HAVE_SECOND_APP_CLUSTER
	FLASH_DATA_PAGE_DRIVER_APP(1, 0);
	FLASH_DATA_PAGE_DRIVER_APP(1, 1);
	FLASH_DATA_PAGE_DRIVER_APP(1, 2);
	FLASH_DATA_PAGE_DRIVER_APP(1, 3);
	FLASH_DATA_PAGE_DRIVER_APP(1, 4);
	FLASH_DATA_PAGE_DRIVER_APP(1, 5);
	FLASH_DATA_PAGE_DRIVER_APP(1, 6);
	FLASH_DATA_PAGE_DRIVER_APP(1, 7);
	FLASH_DATA_PAGE_DRIVER_APP(1, 8);
	FLASH_DATA_PAGE_DRIVER_APP(1, 9);
	FLASH_DATA_PAGE_DRIVER_APP(1, 10);
	FLASH_DATA_PAGE_DRIVER_APP(1, 11);
	FLASH_DATA_PAGE_DRIVER_APP(1, 12);
	FLASH_DATA_PAGE_DRIVER_APP(1, 13);
	FLASH_DATA_PAGE_DRIVER_APP(1, 14);
	FLASH_DATA_PAGE_DRIVER_APP(1, 15);
#endif

static const struct FlashPageData mCatchAllPage = {
	.dataFiller = 0xff,
	.oob = {0xf8, 0xff, 0xff, 0xff},
};

static const struct FlashPageRef mPages[] = {
	
	{0, 0, &mBootBlockPageZero},
	{0, 1, &mBootBlockPageOne},
	
	{1, 0, &mBootBlockPageZero},
	{1, 1, &mBootBlockPageOne},
	
	{2, 0, &mMbrPage},
	{2, 1, &mPbrPage},
	{2, 11, &mFat0Page},
	{2, 12, &mFat1Page},
	{2, 13, &mFat0Page},
	{2, 14, &mFat1Page},
	{2, 15, &mRootDirPage},
	
	{3, 0, &mPalmDirPage},				//this cluster (2) has the "/PALM" directory
	
	{4, 0, &mLauncherDirPage},			//this cluster (3) has the "/PALM/LAUNCHER" directory
	
	{5, 0, &mProgramsDirPage},			//this cluster (4) has the "/PALM/PROGRAMS" directory
	
	{6, 0, &mMsfilesDirPage},			//this cluster (5) has the "/PALM/PROGRAMS/MSFILES" directory
	
	{7, 0, &mmMsAutorunConfigFilePage},	//this cluster (6) has the "/PALM/DEFAULT.ARN" file that enables auto-run on earlier CLIEs
	
	{APP_FIST_CLUSTER + 1, 0, &FLASH_DATA_PAGE_NAME(0, 0)},					//this cluster (APP_FIST_CLUSTER) has the driver PRC
	{APP_FIST_CLUSTER + 1, 1, &FLASH_DATA_PAGE_NAME(0, 1)},
	{APP_FIST_CLUSTER + 1, 2, &FLASH_DATA_PAGE_NAME(0, 2)},
	{APP_FIST_CLUSTER + 1, 3, &FLASH_DATA_PAGE_NAME(0, 3)},
	{APP_FIST_CLUSTER + 1, 4, &FLASH_DATA_PAGE_NAME(0, 4)},
	{APP_FIST_CLUSTER + 1, 5, &FLASH_DATA_PAGE_NAME(0, 5)},
	{APP_FIST_CLUSTER + 1, 6, &FLASH_DATA_PAGE_NAME(0, 6)},
	{APP_FIST_CLUSTER + 1, 7, &FLASH_DATA_PAGE_NAME(0, 7)},
	{APP_FIST_CLUSTER + 1, 8, &FLASH_DATA_PAGE_NAME(0, 8)},
	{APP_FIST_CLUSTER + 1, 9, &FLASH_DATA_PAGE_NAME(0, 9)},
	{APP_FIST_CLUSTER + 1, 10, &FLASH_DATA_PAGE_NAME(0, 10)},
	{APP_FIST_CLUSTER + 1, 11, &FLASH_DATA_PAGE_NAME(0, 11)},
	{APP_FIST_CLUSTER + 1, 12, &FLASH_DATA_PAGE_NAME(0, 12)},
	{APP_FIST_CLUSTER + 1, 13, &FLASH_DATA_PAGE_NAME(0, 13)},
	{APP_FIST_CLUSTER + 1, 14, &FLASH_DATA_PAGE_NAME(0, 14)},
	{APP_FIST_CLUSTER + 1, 15, &FLASH_DATA_PAGE_NAME(0, 15)},

#ifndef HAVE_SECOND_APP_CLUSTER
	
	{APP_FIST_CLUSTER + 2, 0, &mPostAppDataPage},
	
#else

	{APP_FIST_CLUSTER + 2, 0, &FLASH_DATA_PAGE_NAME(1, 0)},					//this cluster (APP_FIST_CLUSTER+1) has the driver PRC's second cluster (if any)
	{APP_FIST_CLUSTER + 2, 1, &FLASH_DATA_PAGE_NAME(1, 1)},
	{APP_FIST_CLUSTER + 2, 2, &FLASH_DATA_PAGE_NAME(1, 2)},
	{APP_FIST_CLUSTER + 2, 3, &FLASH_DATA_PAGE_NAME(1, 3)},
	{APP_FIST_CLUSTER + 2, 4, &FLASH_DATA_PAGE_NAME(1, 4)},
	{APP_FIST_CLUSTER + 2, 5, &FLASH_DATA_PAGE_NAME(1, 5)},
	{APP_FIST_CLUSTER + 2, 6, &FLASH_DATA_PAGE_NAME(1, 6)},
	{APP_FIST_CLUSTER + 2, 7, &FLASH_DATA_PAGE_NAME(1, 7)},
	{APP_FIST_CLUSTER + 2, 8, &FLASH_DATA_PAGE_NAME(1, 8)},
	{APP_FIST_CLUSTER + 2, 9, &FLASH_DATA_PAGE_NAME(1, 9)},
	{APP_FIST_CLUSTER + 2, 10, &FLASH_DATA_PAGE_NAME(1, 10)},
	{APP_FIST_CLUSTER + 2, 11, &FLASH_DATA_PAGE_NAME(1, 11)},
	{APP_FIST_CLUSTER + 2, 12, &FLASH_DATA_PAGE_NAME(1, 12)},
	{APP_FIST_CLUSTER + 2, 13, &FLASH_DATA_PAGE_NAME(1, 13)},
	{APP_FIST_CLUSTER + 2, 14, &FLASH_DATA_PAGE_NAME(1, 14)},
	{APP_FIST_CLUSTER + 2, 15, &FLASH_DATA_PAGE_NAME(1, 15)},
	
	{APP_FIST_CLUSTER + 3, 0, &mPostAppDataPage},
#endif
	
//last one is the catch-all entry
	{0, 0, &mCatchAllPage},
};



bool msMemoryBackingRead(uint_fast16_t block, uint_fast8_t page, void **dataOutPP, uint8_t *oobOutP)
{
	const struct FlashPageData *defaultPgData = mPages[sizeof(mPages) / sizeof(*mPages) - 1].data, *pgData = defaultPgData;
	uint8_t *pageReadBuf = msioGetOutBuffer(0);
	const struct FlashPageRef *pgRef;
	
	if (block >= NUM_BLOCKS || page >= BACKING_PAGES_PER_BLOCK)
		return false;
	
	//i could binary search, but i am lazy and number of entries is O(50).
	// I do still rely on it being sorted, though :P
	pgData = mPages[sizeof(mPages) / sizeof(*mPages) - 1].data;
	for (pgRef = mPages; pgRef < &mPages[sizeof(mPages) / sizeof(*mPages) - 1]; pgRef++) {
		
		if (pgRef->block < block)
			continue;
		if (pgRef->block > block)
			break;
		if (pgRef->page < page)
			continue;
		if (pgRef->page > page)
			break;
		pgData = pgRef->data;
		break;
	}
	
	dtx(0x99);
	dtx(block >> 8);
	dtx(block);
	dtx(page);
//	printf("%x%x%c\n", block, page, pgData == defaultPgData ? 'N' : 'Y');
		
	if (dataOutPP) {
		
		if (pgData->dataLen > BACKING_PAGE_SIZE) {
			
			fatal("failure\n");
			while(1);
		}
		
		if (pgData->data)
			memcpy(pageReadBuf, pgData->data, pgData->dataLen);
		if (pgData->dataLen < BACKING_PAGE_SIZE)
			memset(pageReadBuf + pgData->dataLen, pgData->dataFiller, BACKING_PAGE_SIZE - pgData->dataLen);
		
		*dataOutPP = pageReadBuf;
	}
	memcpy(oobOutP, pgData->oob, sizeof(pgData->oob));
	memset(oobOutP + sizeof(pgData->oob), 0xff, 9 - sizeof(pgData->oob));
	
	return true;
}
