#include "xscaleScreen.h"
#include "platform.h"
#include "printf.h"
#include "memmap.h"
#include "xscale.h"


struct PxaLcdDescr {
	uint32_t FDADR, FSADR, FIDR, LDCMD;
};

static uint32_t xscaleScreenPrvGetFbOffset(void)
{
	return 0;
}

static uint32_t xscaleScreenPrvGetDataDmaDescrOffset(void)
{
	return CPU_HARDWIRED_VRAM_SIZE - 2 * sizeof(struct PxaLcdDescr) - 256 * sizeof(uint16_t);
}

static uint32_t xscaleScreenPrvGetClutDmaDescrOffset(void)
{
	return CPU_HARDWIRED_VRAM_SIZE - sizeof(struct PxaLcdDescr) - 256 * sizeof(uint16_t);
}

static uint32_t xscaleScreenPrvGetClutOffset(void)
{
	return CPU_HARDWIRED_VRAM_SIZE - 256 * sizeof(uint16_t);
}

uint16_t *xscaleScreenGetClutPtr(void)
{
	return (uint16_t*)(CPU_HARDWIRED_VRAM_ADDR + xscaleScreenPrvGetClutOffset());
}

void *xscaleScreenGetFbPtr(void)
{
	return (void*)(CPU_HARDWIRED_VRAM_ADDR + xscaleScreenPrvGetFbOffset());
}

static uint32_t xscaleScreenPrvCmdForClut(uint32_t depth)
{
	if (depth == 1)	//we always load at least 4 entries
		depth = 2;
	
	return 0x04000000 + (2 << depth);
}

void xscaleScreenResendClut(uint32_t curDepth)
{
	volatile struct PxaLcdDescr *descClut = (struct PxaLcdDescr*)(CPU_HARDWIRED_VRAM_ADDR + xscaleScreenPrvGetClutDmaDescrOffset());
	uint32_t clutDescrPa = CPU_HARDWIRED_VRAM_PA + xscaleScreenPrvGetClutDmaDescrOffset();
	struct PxaLcd *lcd = platPeriphP2V(PXA_BASE_LCD_CTRL);
	
	//wait till we are no longer already sending the clut
	while (lcd->dma[0].FDADR == clutDescrPa);
	
	//edit descriptor
	descClut->LDCMD = xscaleScreenPrvCmdForClut(curDepth); 
	
	//send it
	lcd->FBR[0] = clutDescrPa | 1;
}

bool xscaleScreenEnable(uint32_t lccr0, uint32_t lccr1, uint32_t lccr2, uint32_t lccr3)
{
	uint32_t w = (lccr1 & 0x3ff) + 1, h = (lccr2 & 0x3ff) + 1, d = 1 << ((lccr3 >> 24) & 0x07);
	struct PxaLcd *lcd = platPeriphP2V(PXA_BASE_LCD_CTRL);
	struct PxaLcdDescr *descClut, *descFb;
	
	xscaleScreenDisable();	//if it is enabled, disable it
	
	//verify space (we assume support for 16bpp) and calc pointers
	if (w * h * sizeof(uint16_t) + 2 * sizeof(struct PxaLcdDescr) + sizeof(uint16_t) * 256 > CPU_HARDWIRED_VRAM_SIZE) {
		
		loge("VRAM too small\n");
		return false;
	}
	
	//verify sanity
	if ((CPU_HARDWIRED_VRAM_SIZE | CPU_HARDWIRED_VRAM_ADDR) & 7) {
		
		loge("VRAM start/end misaligned\n");
		return false;
	}
	
	//set up descriptor pointers
	descFb = (struct PxaLcdDescr*)(CPU_HARDWIRED_VRAM_ADDR + xscaleScreenPrvGetDataDmaDescrOffset());
	descClut = (struct PxaLcdDescr*)(CPU_HARDWIRED_VRAM_ADDR + xscaleScreenPrvGetClutDmaDescrOffset());
	
	//prepare descriptor for data
	descFb->FDADR = CPU_HARDWIRED_VRAM_PA + xscaleScreenPrvGetDataDmaDescrOffset();	//points to itself
	descFb->FSADR = CPU_HARDWIRED_VRAM_PA + xscaleScreenPrvGetFbOffset();
	descFb->FIDR = 0;
	descFb->LDCMD = w * h * d / 8;
	
	//prepare descriptor for CLUT
	descClut->FDADR = CPU_HARDWIRED_VRAM_PA + xscaleScreenPrvGetDataDmaDescrOffset();	//points to data 
	descClut->FSADR = CPU_HARDWIRED_VRAM_PA + xscaleScreenPrvGetClutOffset();
	descClut->FIDR = 0;
	descClut->LDCMD = xscaleScreenPrvCmdForClut(d);
	 
	lcd->LCCR[1] = lccr1;
	lcd->LCCR[2] = lccr2;
	lcd->LCCR[3] = lccr3;
	
	//go
	lcd->dma[0].FDADR = CPU_HARDWIRED_VRAM_PA + (d == 16 ? xscaleScreenPrvGetDataDmaDescrOffset() : xscaleScreenPrvGetClutDmaDescrOffset());	//clut if not 16bpp, else data
	
	lcd->LCCR[0] = lccr0;
	
	return true;
}

void xscaleScreenDisable(void)
{
	struct PxaLcd *lcd = platPeriphP2V(PXA_BASE_LCD_CTRL);
	
	if (lcd->LCCR[0] & 1) {				//do not bother if it is not on 
	
		lcd->LCSR = 0x01;				//clear LCSR.LDD
		lcd->LCCR[0] |= 0x400;			//set LCCR0.DIS to start normal disable
		while (!(lcd->LCSR & 0x01));	//wait for disable to be done
		while (lcd->LCCR[0] & 1);
	}
}