#include "halPenAndKeys.h"
#include "drivers/input.h"
#include "drivers/rtc.h"
#include "kernel.h"
#include "printf.h"
#include "slab.h"
#include "kal.h"
#include "ral.h"


#define DRIVER_EVT_TYPE_RTC_HZ_TICK		0
#define DRIVER_EVT_TYPE_RTC_ALARM		1
#define DRIVER_EVT_TYPE_PEN				2
#define DRIVER_EVT_TYPE_KEY				3
#define DRIVER_EVT_TYPE_HARDBTN_DN		4
#define DRIVER_EVT_TYPE_HARDBTN_UP		5


struct DriverEvent {
	uint32_t type;
	union DriverEventData {
		struct {
			int16_t x, y;
		} pen;
		struct {
			uint16_t code;
			uint8_t mod;
		} key;
		uint32_t mask;
	} data;
};

static struct Slab *mDriverEventsSlab;
static mbx_t mDriverEvtsMbx;
#define MAX_DRIVER_EVENTS			128



void hwrEventListen(void)
{
	while (!mDriverEvtsMbx) {	//this is a poor way to do this but ok for now
		KTaskDelay(100);
		asm("":::"memory");
	}
	
	//TODO: hw
	while(1) {
		
		struct DriverEvent* evt;
		struct HalEvtPen penEvt = {};
		struct HalEvtKey keyEvt = {};
		Err e;
		
		e = KALMailboxWait(mDriverEvtsMbx, (uint32_t*)&evt, -1);
		if (errNone != e) {
			loge("wait = 0x%04x\n", e);
			continue;
		}
		
		logst("got evt type %u\n", evt->type);
		switch (evt->type) {
			case DRIVER_EVT_TYPE_RTC_ALARM:
				halCallEvtCallback(HAL_EVENT_TYPE_CLOCK, NULL);
				break;
			case DRIVER_EVT_TYPE_PEN:
				penEvt.x = evt->data.pen.x;
				penEvt.y = evt->data.pen.y;
				halCallEvtCallback(HAL_EVENT_TYPE_PEN, &penEvt);
				break;
			case DRIVER_EVT_TYPE_KEY:
				keyEvt.ascii = evt->data.key.code;
				keyEvt.modifiers = evt->data.key.mod;
				halCallEvtCallback(HAL_EVENT_TYPE_KEY, &keyEvt);
				break;
			case DRIVER_EVT_TYPE_HARDBTN_DN:
			case DRIVER_EVT_TYPE_HARDBTN_UP:
				halHardKeyChanged(evt->data.mask, evt->type == DRIVER_EVT_TYPE_HARDBTN_DN);
				break;
		}
		
		slabFree(mDriverEventsSlab, evt);
	}
}

static bool kernelDriverEvtEnqueue(uint32_t type, union DriverEventData *data)
{
	struct DriverEvent *ev;
	
	ev = (struct DriverEvent*)slabAlloc(mDriverEventsSlab);
	if (!ev) {
		logt("DriverEvent dropped: slab\n");
		return false;
	}
	
	ev->type = type;
	if (data)
		ev->data = *data;
	
	if (errNone != KALMailboxSend(mDriverEvtsMbx, (uint32_t)ev)) {
		logt("DriverEvent dropped: mbx\n");
		slabFree(mDriverEventsSlab, ev);
		return false;
	}
	
	return true;
}

static void kernelRtcAlarmF(void)
{
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	
	logt(" ** RTC ALM\n");
	kernelDriverEvtEnqueue(DRIVER_EVT_TYPE_RTC_ALARM, NULL);
	
	ralRestoreR9(r9);
}

static void kernelInputKeyF(uint32_t modifier, uint32_t code)
{
	logt("** key 0x%04x with mod 0x%01x\n", code, modifier);

	union DriverEventData d = {.key = {.code = code, .mod = modifier, }, };
	
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	
	kernelDriverEvtEnqueue(DRIVER_EVT_TYPE_KEY, &d);
	ralRestoreR9(r9);
}

static void kernelInputBtnF(uint32_t key, bool pressed)
{
	logt("** hard key 0x%08x %sed\n", key, pressed ? "press" : "releas");

	union DriverEventData d = {.mask = key, };
	
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	
	kernelDriverEvtEnqueue(pressed ? DRIVER_EVT_TYPE_HARDBTN_DN : DRIVER_EVT_TYPE_HARDBTN_UP, &d);
	ralRestoreR9(r9);
}

static void kernelInputPenF(int16_t x, int16_t y)
{
	union DriverEventData d = {.pen = {.x = x, .y = y, }, };
	static int16_t prevX = -0x7000, prevY = -0x7000;
	
	if (prevX == x && prevY == y)
		return;
	
	prevX = x;
	prevY = y;
	
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	kernelDriverEvtEnqueue(DRIVER_EVT_TYPE_PEN, &d);
	ralRestoreR9(r9);
}

kstatus_t kernelDriversInit(void)
{
	kstatus_t sta;
	mbx_t mbx;

	mDriverEventsSlab = slabCreate(MAX_DRIVER_EVENTS, sizeof(struct DriverEvent));
	if (!mDriverEventsSlab)
		return KERN_STATUS_MEM_ERR;
	
	if (errNone != KALMailboxCreate(&mbx, CREATE_4CC('_','d','E','v'), MAX_DRIVER_EVENTS)) {
		slabDestroy(mDriverEventsSlab);
		return KERN_STATUS_INTERNAL_ERR;
	}
	asm volatile("":::"memory");
	mDriverEvtsMbx = mbx;
	asm volatile("":::"memory");
	
	sta = drvRtcInit(NULL, kernelRtcAlarmF);
	if (sta != KERN_STATUS_OK) {
		logt("RTC init failed\n");
		return sta;
	}
	
	sta = drvInputInit(kernelInputKeyF, kernelInputBtnF, kernelInputPenF);
	if (sta != KERN_STATUS_OK) {
		logt("INPUT init failed\n");
		return sta;
	}
	
	//more init here
	
	return KERN_STATUS_OK;
}