#include "kernel_int.h"
#include "printf.h"
#include "mpu.h"

static bool mNeedResched = false;


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

bool __attribute__((naked,noinline,used)) userAccessFaultReturnPath(void)
{
	asm volatile(
		"	movs r0, #0				\n\t"
		"	bx   lr					\n\t"
	:::"memory","r0","r1","r2","r3","r12","lr");
}

bool __attribute__((naked,noinline,used)) copyFromUser(void* dst, const void* userSrc, uint32_t len)
{
	bool dummy;		//using this generates useless instructions, but they never run and it nicely silenced gcc's warnings so why not
	
	asm volatile(
		"1:							\n\t"
		"	subs  r2, #1			\n\t"
		"	blt   2f				\n\t"
		".globl user_access_3		\n\t"
		"user_access_3:				\n\t"
		"	ldrbt r3, [r1]			\n\t"
		"	strb  r3, [r0], #1		\n\t"
		"	adds  r1, #1			\n\t"
		"	b     1b				\n\t"
		"2:							\n\t"
		"	movs r0, #1				\n\t"
		"	bx   lr					\n\t"
	:"=r"(dummy)::"memory","r0","r1","r2","r3","r12","lr");
	
	return dummy;
}

bool __attribute__((naked,noinline,used)) copyToUser(void* userDst, const void* src, uint32_t len)
{
	bool dummy;		//using this generates useless instructions, but they never run and it ncely silenced gcc's warnings so why not
	
	asm volatile(
		"1:							\n\t"
		"	subs  r2, #1			\n\t"
		"	blt   2f				\n\t"
		"	ldrb  r3, [r1], #1		\n\t"
		".globl user_access_2		\n\t"
		"user_access_2:				\n\t"
		"	strbt r3, [r0]			\n\t"
		"	adds  r0, #1			\n\t"
		"	b     1b				\n\t"
		"2:							\n\t"
		"	movs r0, #1				\n\t"
		"	bx   lr					\n\t"
	:"=r"(dummy)::"memory","r0","r1","r2","r3","r12","lr");
	
	return dummy;
}

bool __attribute__((naked,noinline,used)) readUserWord(uint32_t* dst, const uint32_t *userPtr)
{
	bool dummy;		//using this generates useless instructions, but they never run and it nicely silenced gcc's warnings so why not
	
	asm volatile(
		"	tst  r1, #3				\n\t"
		"	bne  1f					\n\t"
		".globl user_access_1		\n\t"
		"user_access_1:				\n\t"
		"	ldrt r2, [r1]			\n\t"
		"	str  r2, [r0]			\n\t"
		"	movs r0, #1				\n\t"
		"	bx   lr					\n\t"
		"1:							\n\t"
		"	movs r0, #0				\n\t"
		"	bx   lr					\n\t"
	:"=r"(dummy)::"memory","r0","r1","r2","r3","r12","lr");
	
	return dummy;
}

bool __attribute__((naked,noinline,used)) writeUserWord(uint32_t* userDst, uint32_t src)
{
	bool dummy;		//using this generates useless instructions, but they never run and it ncely silenced gcc's warnings so why not
	
	asm volatile(
		"	tst  r0, #3				\n\t"
		"	bne  1f					\n\t"
		".globl user_access_0		\n\t"
		"user_access_0:				\n\t"
		"	strt r1, [r0]			\n\t"
		"	movs r0, #1				\n\t"
		"	bx   lr					\n\t"
		"1:							\n\t"
		"	movs r0, #0				\n\t"
		"	bx   lr					\n\t"
	:"=r"(dummy)::"memory","r0","r1","r2","r3","r12","lr");
	
	return dummy;
}


//called on tail end of syscalls and irqs. expects LR set such that return is to LR directly
void __attribute__((naked, used)) kernelMaybeContextSwitch(void)
{
	asm volatile(
		"	stmfd   sp!, {lr}					\n\t"
		".globl kernelMaybeContextSwitch_pushed	\n\t"
		"kernelMaybeContextSwitch_pushed:		\n\t"
		"	ldr     lr, =%0						\n\t"
		"	ldrb    lr, [lr]					\n\t"	//if no resched request, do nothing
		"	cmp     lr, #0						\n\t"
		"	ldmeqfd sp!, {pc}^					\n\t"
		"	mrs     lr, spsr					\n\t"
		"	ands    lr, #0x0f					\n\t"	//if we did not come from user/system mode, do not context switch
		"	cmpne   lr, #0x0f					\n\t"
		"	ldmnefd sp!, {pc}^					\n\t"
		"	sub     sp, #4 * 2					\n\t"
		"	stmia   sp, {r13, r14}^				\n\t"
		"	nop									\n\t"
		"	nop									\n\t"
		"	mrs     lr, spsr					\n\t"
		"	stmfd	sp!, {r0-r12, lr}			\n\t"
		"	mov     r0, sp						\n\t"
		"	bl      kernelPrvContextSwitch		\n\t"
		"	ldmfd   sp!, {r0-r12, lr}			\n\t"
		"	msr     spsr, lr					\n\t"
		"	ldmia   sp, {r13, r14}^				\n\t"
		"	nop									\n\t"
		"	nop									\n\t"
		"	add     sp, #4 * 2					\n\t"
		"	ldmfd   sp!, {pc}^					\n\t"
		:
		:"i"(&mNeedResched)
		:"memory", "cc"
	);
}

//creates a struct CortexExcFrame, for convenience of directly calling syscallHandle
void  __attribute__((naked)) kernelPrvHandleSwi(void)
{
	asm volatile(
		"	stmdb  sp, {r13}^				\n\t"	//yeah, not quite abi compliant, but we'll live
		"	nop								\n\t"
		"	sub    sp, #0x0C				\n\t"	//lr (userspace pc), sr, userspace_sp
		"	str    lr, [sp]					\n\t"	//pc is space for sr
		"	mrs    lr, spsr					\n\t"
		"	tst    lr, #0x20				\n\t"
		"	str    lr, [sp, #4]				\n\t"
		"	sub    sp, #0x18				\n\t"
		"	stmia  sp, {r0-r3, r12, lr}^	\n\t"
		"	nop								\n\t"
		"	mov    r1, sp					\n\t"
		"	ldr    r0, [sp, #0x18]			\n\t"
		"	ldreq  r0, [r0, #-4]			\n\t"
		"	ldrneh r0, [r0, #-2]			\n\t"
		"	biceq  r0, r0, #0xff000000		\n\t"
		"	bicne  r0, r0, #0xff00			\n\t"
		"	ldr    r3, [r1, #0x20]			\n\t"	//SP_usr
		"	bl     syscallHandle			\n\t"
		"	ldmia  sp, {r0-r3, r12, lr}^	\n\t"
		"	nop								\n\t"
		"	ldr    lr, [sp, #0x1c]			\n\t"
		"	msr    spsr, lr					\n\t"
		"	ldr    lr, [sp, #0x18]			\n\t"
		"	add    sp, #0x24				\n\t"
		"	ldmdb  sp, {r13}^				\n\t"
		"	nop								\n\t"
		"	b      kernelMaybeContextSwitch	\n\t"
		:
		:
		:"memory", "cc"
	);
}

bool __attribute__((noinline)) schedAmInSyscall(void)	//noinline will stop this form being inlined into thumb and them failing to link
{
	uint32_t sr;
	
	asm("mrs %0, cpsr":"=r"(sr));
	
	return (sr & 0x1f) == 0x13;
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif

void schedAssertCalledInSyscallMode(void)
{
	if (!schedAmInSyscall() && !irqsAreAllOff())
		fatal("Cannot call this func from non-syscall context or with irqs off\n");
}

void schedSwitchToScheduledRegimeByCallingYield(void)
{
	asm volatile(
		"	mov   r0, %0				\n\t"
		"	svc   %1					\n\t"
		:
		:"I"(SYSC_TASK_YIELD), "I"(KERNEL_SWI_NUM)
	);
	
	__builtin_unreachable();
	while(1);	//make it clear to compiler
}

void schedRequestContextSwitch(void)	//called only in SWI mode so thus safe to do this...
{
	mNeedResched = true;
}

void __attribute__((used)) kernelPrvContextSwitch(struct SchedRegs *regs)		//runs in SWI/irq/etc context and thus with interrupts off (FIQs on)
{
	struct Task *next = schedGetNextTask(), *volatile*curP = schedGetCurTaskP(), *cur = *curP;
	
	if (schedGetLockCount()) {
		if (!*schedGetCurTaskP())
			fatal("Scheduler off and no current task! not ok!\n");
		return;
	}
	
	mNeedResched = false;
	
	if (cur) {
		cur->ctx.regs = *regs;
		cur->ctx.tls = TLS;
		
		if (cur->ctx.regs.sr & 0xc0)
			fatal("Swapping out a process with IRQs/FIQs masked: sr=0x%08x\n", 	cur->ctx.regs.sr);
	}
	mpuSetStackGuard((uintptr_t)next->stackLimitAddr);
	*regs = next->ctx.regs;
	TLS = next->ctx.tls;
	*curP = cur = next;
}
