#include <stdio.h>
#include "bitfield.h"
#include "heap.h"
#include "slab.h"


struct Slab {
	struct Bitfield* bf;
	uint32_t itemSz;
	uint8_t storage[];
};

struct Slab* slabCreate(uint32_t nitems, uint32_t itemSz)
{
	struct Slab *s;
	
	itemSz = (itemSz + 3) &~ 3;
	if (!itemSz)
		return NULL;
	
	s = (struct Slab*)kheapAlloc(sizeof(struct Slab) + itemSz * nitems);
	if (!s)
		return NULL;
	
	s->bf = bitfieldCreate(nitems);
	if (!s->bf) {
		kheapFree(s);
		return NULL;
	}
	
	s->itemSz = itemSz;
	
	return s;
}

void slabDestroy(struct Slab* sla)
{
	bitfieldDestroy(sla->bf);
	kheapFree(sla);
}

void* slabAlloc(struct Slab* sla)
{
	int32_t idx = bitfieldAtomicFindZeroAndSet(sla->bf);
	if (idx < 0)
		return NULL;
	
	return sla->storage + idx * sla->itemSz;
}

void slabFree(struct Slab* sla, void* ptr)
{
	bitfieldAtomicClearBit(sla->bf, (((uint8_t*)ptr) - sla->storage) / sla->itemSz);
}

