#ifndef _KERNEL_COMMON_H_
#define _KERNEL_COMMON_H_

#include <stdint.h>
#include "boot.h"


typedef uint8_t kstatus_t;


//must be uint32_t basically
typedef uint32_t tid_t;
typedef uint32_t mutex_t;
typedef uint32_t sema_t;
typedef uint32_t mbx_t;
typedef uint32_t evtgrp_t;
typedef uint32_t tmr_t;


#define KERN_STATUS_OK				0x00
#define KERN_STATUS_TIMEOUT			0x01
#define KERN_STATUS_MEM_ERR			0x02
#define KERN_STATUS_NOT_FOUND		0x03
#define KERN_STATUS_INVAL_STATE		0x04
#define KERN_STATUS_INVAL_REQ		0x05
#define KERN_STATUS_FULL			0x06	//eg: a mailbox may be full
#define KERN_STATUS_NULL_PARAM		0x07	//eg: NULL param where not allowed
#define KERN_STATUS_USER_FAULT		0x08	//fault on an access to a user-provided memory location
#define KERN_STATUS_INTERNAL_ERR	0x0A


struct KernTaskInfo {
	void* pc;
	void* sp;
	void* stackLimit;
	uint32_t tag;
	void* exinf;
	uint8_t prio;
};


typedef void (*KernTimerCbk)(void* cbkData, tmr_t tmr); //order important


void kernelInit(void (*entryFunc)(void), void* hyperFuncIfAny);
kstatus_t kernelInitLate(void);			//call this form first task early on
kstatus_t kernelDriversInit(void);



//KAL errors from real device
#define ERR_KAL_NULL_PTR			0xF002			//NULL pointer passed in place where it is not allowed (usually handle outputs of create funcs)
#define ERR_KAL_OUT_OF_RESOURCES	0xF003			//returned when out of some resrouce (mutxes, semaphores, timers, etc)
#define ERR_KAL_OUT_OF_MEM			0xF004			//returned when out of memory (we do not use this, we return memErrNotEnoughSpace)
#define ERR_KAL_GENERIC				0xF009			//generic internal error
#define ERR_KAL_MAILBOX_FULL		0xF00A
#define ERR_KAL_ITEM_NOT_FOUND		0xF00C			//item id (thread, mutex, etc) is invalid

//our status to PalmOS error
static inline Err kernelStatusToPalmOsErr(kstatus_t sta)
{
	switch(sta) {
		case KERN_STATUS_OK:			return errNone;
		case KERN_STATUS_TIMEOUT:		return sysErrTimeout;
		case KERN_STATUS_MEM_ERR:		return memErrNotEnoughSpace;
		case KERN_STATUS_NOT_FOUND:		return ERR_KAL_ITEM_NOT_FOUND;
		case KERN_STATUS_FULL:			return ERR_KAL_MAILBOX_FULL;
		case KERN_STATUS_NULL_PARAM:	return ERR_KAL_NULL_PTR;
		default:						return ERR_KAL_GENERIC;
	}	
}

#endif

