#define NON_PORTABLE
#define RAL_EXPORT_SOME_RAW_RAL_FUNCS
#include "boot.h"
#include <string.h>
#include <stdint.h>
#include <SysEvent.h>
#include <SerialMgr.h>
#include <FatalAlert.h>
#include <FeatureMgr.h>
#include <HwrMiscFlags.h>
#include <palmOneNavigator.h>
#include <palmOneFeatures.h>
#include "dockAndBattery.h"
#include "zodiac/zodiac.h"
#include "halPenAndKeys.h"
#include "machServices.h"
#include "halDisplay.h"
#include "ral_export.h"
#include "audiohw.h"
#include "kernel.h"
#include "printf.h"
#include "entry.h"
#include "misc.h"
#include "heap.h"
#include "sony.h"
#include "irqs.h"
#include "emu.h"
#include "cpu.h"
#include "ral.h"
#include "dal.h"
#include "kal.h"


//todo: reset palette on depth change (is this ok)
//todo: set all the features palmos devices need (see tx-dal and palmOnefeatures.h) at stage 4


typedef Err (*HalEvtCbk)(const void* evtData);

static uint8_t mWakeFlags;
static uint8_t mInitStage;
static uint64_t mRndState[2];
static uint32_t mDalModRefNo;
static uint16_t mIntsEnabled = 1;
static uint16_t mAutoOffSeconds = 120;
static HalEvtCbk mHalEvtCbks[] = {[HAL_EVENT_TYPE_PEN] = NULL, [HAL_EVENT_TYPE_KEY] = NULL, [HAL_EVENT_TYPE_CLOCK] = NULL,};
static bool mDbgLockout = false;
static bool mIsGarnet = false;
static bool mIs50 = false;
static uint32_t mAutoOffEvtTime = 0, mAlarmSeconds;
static uint16_t mDockStatus = 0, mDoubleTapDelay = 10;


extern void vecs(void);

struct CardHeader {
	uint32_t					jumpToCode;				// 0x00 jump instr to code ('boot' 19000) res in DAL
	uint32_t					zeroNum0;				// 0x04 always 0
	uint32_t					signature;				// 0x08 0xFEEDBEEF
	uint16_t					hdrVersion;				// 0x0C 5 for small rom, 6 for big
	uint16_t					flags;					// 0x0E 0x2080 for big rom, 0x20A0 for small
	char						name[32];				// 0x10 card name
	char						manuf[32];				// 0x30 card manufacturer's name
	uint16_t					version;				// 0x50 always 1
	uint16_t					numRAMBlocks;			// 0x52 always 1
	uint32_t					creationDate;			// 0x54 card creation date in palm format
	uint32_t					blockListPtr;			// 0x58 RAM block list
	uint32_t					zeroNum1;				// 0x5C always 0
	uint32_t					zeroNum2;				// 0x60 always 0
	uint32_t					romTokenStorePtr;		// 0x64 pointer to rom token storage (RO)
	uint32_t					bigRomPtr;				// 0x68 VA of big rom (in both small and big rom)
	uint32_t					totalRomSz;				// 0x6C offset form this header to end of this ROM (small or big)
	uint32_t					zeroNum3;				// 0x70 always 0
	uint32_t					zeroNum4;				// 0x74 always 0
	uint32_t					dalAmddRsrcOfst;		// 0x78 offset from this header to ('amdd' 19000) resource in DAL
	uint32_t 					unknown;				// 0x7C probably checksum
	uint32_t					companyID;				// 0x80 4cc
	uint32_t					halID;					// 0x84 4cc
	uint32_t					romVersion;				// 0x88 seems to be zero always
	char						romVersionString[32];	// 0x8C rom name string
	uint32_t					dalAmdiRsrcOfst;		// 0xAC offset from this header to ('amdi' 19000) resource in DAL
};	//0xB0 bytes

struct StoreHeader {
	uint32_t					signature;				// 0x00 0xFEEDFACE
	uint16_t					version;				// 0x04 always 2
	uint16_t					flags;					// 0x06 always 0
	char						name[32];				// 0x08 name of store (usually unused)
	uint32_t					zeroNum0;				// 0x28 always 0
	uint32_t					zeroNum1;				// 0x2C always 0
	uint32_t					heapListPtr;			// 0x30 pointer to struct HeapList
	uint32_t					bootRsrsBoot10001;		// 0x34 pointer to Boot.prc ('boot' 10001) resource
	uint32_t					bootRsrsBoot10002;		// 0x38 pointer to Boot.prc ('boot' 10002) resource
	uint32_t					databaseListPtr;		// 0x3C database directory pointer
	uint32_t					zeros1[6];				// 0x40 zeroes
	uint32_t					bootScreen0ptr;			// 0x58 pointer to "Palm OS Data" ('absb' 19000) resource (soem sort of a boot screen - a full real palmos bitmap 320x320)
	uint32_t					bootScreen1ptr;			// 0x5C pointer to "Palm OS Data_enUS" ('absb' 19001) resource (soem sort of a boot screen - a full real palmos bitmap 320x320)
	uint32_t					unknown0x170000;		// 0x60 always 0x170000
	uint32_t					zeroNum2;				// 0x64 always 0 
	uint32_t					zeroNum3;				// 0x68 always 0 
	uint32_t					bootScreen2ptr;			// 0x6C pointer to "Palm OS Data_enUS" ('absb' 19002) resource (soem sort of a boot screen - a full real palmos bitmap 320x320)
	uint32_t					zeros2[34];				// 0x70 zeroes
	uint32_t					bootBootRsrcPtr;		// 0xF8 pointer to Boot.prc ('boot' 10003) resource
	uint32_t					zeroNum4;				// 0xFC always 0 
};


struct HeapChunkHeader {	//assumes compiler allocates from bottm to top bits with no holes

	//WORD @ 0x00
	uint32_t size				: 24;					// 0x00FFFFFF - size of the allocated chunk including this header
	uint32_t sizeExtra			:  4;					// 0x0F000000 - how many bytes were allocated over what was requested (excluding header)
	uint32_t unused				:  2;					// 0x30000000 - not used it seems
	uint32_t irrelevant			:  1;					// 0x40000000 - weird one (docs say this is called "moved")
	uint32_t free				:  1;					// 0x80000000 - set if chunk is currently free
	//WORD @ 0x04
	uint32_t handleOfst			: 24;					// 0x00FFFFFF - if free chunk, offset to next free chunk divided by 2; if not free, 0 if nonmovable, else negative offset to memhandle divided by 2 (memhandle is chunk header minus 2 times this)
	uint32_t owner				:  4;					// 0x0F000000 - ownerID of the owner
	uint32_t lockCt				:  4;					// 0xF0000000 - lock count (15 for seemingly all things in rom)
};


void prPutchar(char ch)
{
	cpuPutchar(ch);
}

static void* findMagicAtAddr(uint32_t addrStart, uint32_t maxLen, uint32_t magic, int32_t wordOffsetToAdd)
{
	uint32_t *ptr = (uint32_t*)(uintptr_t)addrStart;
	uint32_t i;
	
	for (i = 0; i < maxLen; i += 4, ptr++) {
		if (*ptr == magic)
			return ptr + wordOffsetToAdd;
	}
	
	return NULL;
}

static const struct StoreHeader *findStore(void)
{
	return (struct StoreHeader*)findMagicAtAddr(((uint32_t)(uintptr_t)&vecs) & 0xFFF00000, 1024 * 1024, 0xFEEDFACE, 0);
}

static const struct CardHeader *findCard(void)
{
	return (struct CardHeader*)findMagicAtAddr(((uint32_t)(uintptr_t)&vecs) & 0xFFF00000, 1024 * 1024, 0xFEEDBEEF, -2);
}

static uint32_t getMemChunkSize(const void* mem)	//assumes a lot of things
{
	const struct HeapChunkHeader *ch = ((const struct HeapChunkHeader*)mem) - 1;
	
	return ch->size - sizeof(struct HeapChunkHeader) - ch->sizeExtra;
}

static void callMainThreadProc(void* param)
{
	extern void PalmOSMain(void*);
	
	emuCoreInit();
	PalmOSMain(param);
}

static void callMain(void)
{
	void* exinf = kheapAlloc(8);	//do not ask why, this is necessary
	uint32_t tid;
	Err e;
	
	memset(exinf, 0, 8);
	struct KalTaskCreateInfo ti = {
		.exinf = exinf,
		.codeToRun = callMainThreadProc,
		.stackSz = 8192,
		.prio = 100,
		.tag = CREATE_4CC('f','r','s','t'),
	};
	
	e = KALTaskCreate(&tid, &ti);
	if (e)
		fatal("Failed to create first task: 0x%04x\n", e);
	
	e = KALTaskStart(tid, NULL);
	if (e)
		fatal("Failed to start first task: 0x%04x\n", e);
	
	logd("PalmOS main Task started\n");
}

//this is just convenient, normal would be ok too
static void __attribute__((used)) pErrDisplayFileLineMsgReport(const char *file, uint32_t line, const char *errMsg, uint32_t *regs)
{
	char *msg = halErrorGetBuffer(), *part2;
	uint32_t i;
	
	strcpy(msg, "FATAL ALERT");
	part2 = msg + strlen(msg) + 1;
	part2[0] = 0;
	spr(part2 + strlen(part2), "PALMOS ERROR@ %s:%u - '%s'\n", file, line, errMsg);
	spr(part2 + strlen(part2), "R3 =%08xh R4=%08xh\n", regs[0], regs[1]);
	spr(part2 + strlen(part2), "R5 =%08xh R6=%08xh\n", regs[2], regs[3]);
	spr(part2 + strlen(part2), "R7 =%08xh R8=%08xh\n", regs[4], regs[5]);
	spr(part2 + strlen(part2), "R9 =%08xh R10=%08xh\n", regs[6], regs[7]);
	spr(part2 + strlen(part2), "R11=%08xh SP=%08xh\n", regs[8], regs + 10);
	spr(part2 + strlen(part2), "PC=%08xh SP words:\n", regs[9]);
	for (i = 0; i < 33; i += 3)
		spr(part2 + strlen(part2), "%08x %08x %08x\n", regs[10 + i + 0], regs[10 + i + 1], regs[10 + i + 2]);
	
	loge("%s", part2);
	
	impl_HALErrDisplay(msg, false, NULL, false);
	fatal("%s\n", __func__);
}


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static void __attribute__((naked)) pErrDisplayFileLineMsg(const char *file, uint32_t line, const char *msg)
{
	asm volatile(
		#ifdef BUILD_FOR_THUMB_1
			"	push  {r0-r3, lr}					\n\t"	//push lr and make space on stack for 4 more regs
			"	push  {r3-r7}						\n\t"
			"	add   r3, sp, #20					\n\t"	//point to space where we saved space for r8..r11
			"	mov   r4, r8						\n\t"
			"	mov   r5, r9						\n\t"
			"	mov   r6, r10						\n\t"
			"	mov   r7, r11						\n\t"
			"	stmia r3!, {r4-r7}					\n\t"
		#else
			"	push  {r3-r11, lr}					\n\t"
		#endif
		
		"	mov  r3, sp								\n\t"
		"	bl   pErrDisplayFileLineMsgReport		\n\t"
		#ifdef BUILD_FOR_THUMB_1
			"	add   r3, sp, #20					\n\t"
			"	ldmia r3!, {r4-r7}					\n\t"
			"	mov   r8, r4						\n\t"
			"	mov   r9, r5						\n\t"
			"	mov   r10, r6						\n\t"
			"	mov   r11, r7						\n\t"
			"	pop   {r3-r7}						\n\t"
			"	pop   {r0-r3,pc}					\n\t"	//clobber r0-r3
		#else
			"	pop   {r3-r11, pc}					\n\t"
		#endif
	);
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif

static void (*mOldOmGetSystemLocale)(uint16_t* locInfo);
static void pOmGetSystemLocale(uint16_t* locInfo)
{
	logt("%s\n", __func__);
	mOldOmGetSystemLocale(locInfo);
	if(!locInfo[0] && !locInfo[1]) {			//NVFS needs this
		locInfo[1] = 23;
		logw("HACK: XXX: fixing NVFS locale issue\n");
	}
	logt("%s: %04x %04x\n", __func__, locInfo[0], locInfo[1]);
}

static Err (*mOldSerialMgrInstall)(void);
static Err pSerialMgrInstall(void)
{
	Err ret = mOldSerialMgrInstall();	//we need to run it ot init structures
	
	logt("SerialMgrInstall() returned 0x%04x\n", ret);
	
	if (ret == serErrNoDevicesAvail) {
		logw("Faking serial manager install success\n");
		ret = errNone;
	}
	
	return ret;						//but it will return an error since ti finds no drivers. Fake a success
}

/*
	struct SysLinkerStubRequrest {
		uint32_t dbTyp, dbCrid, rev, nEntries;
		uint32_t dataResTyp, infoResTyp, codeResTyp;
		uint16_t dataResID, infoResId, codeResId;
		uint16_t rfu;
	};
	
	static void (*mOldSysLinkerStub)(const struct SysLinkerStubRequrest *req, uint32_t clientRef, void **dispatchTblP);
	static void pSysLinkerStub(const struct SysLinkerStubRequrest *req, uint32_t clientRef, void **dispatchTblP)
	{
		logi("SysLinkerStub(&{"FMT_4CC", "FMT_4CC", ... %u, %u}, ...) for client %u from PC 0x%08x\n",
			CNV_4CC(req->dbTyp), CNV_4CC(req->dbCrid), req->rev, req->nEntries, clientRef, __builtin_return_address(0));
		
		return mOldSysLinkerStub(req, clientRef, dispatchTblP);
	}
*/

static bool loadModuleLowLevel(const void *code, const void *data, const void *info, uint32_t *refNoP, uint32_t *nEntriesP)
{
	struct RalModuleDescr descr = {};
	struct RalModuleInfo nfo;
	uint32_t ref;
	void *mem;
	Err err;
	
	descr.codePtr = code;
	descr.infoPtr = info;
	descr.initedDataPtr = data;
	descr.initedDataSz = data ? getMemChunkSize(data) : 0;
	err = impl_RALGetModuleInfo(&descr, &nfo);
	if (err) {
		fatal("Module info failed to be gotten: 0x%04x\n", err);
		return false;
	}
	if (!nfo.neededMem) {
		fatal("Unlikely: module needs no memory\n");
		return false;
	}
	mem = kheapAlloc(nfo.neededMem);
	if (!mem) {
		fatal("Failed to allocate memory for module data\n");
		return false;
	}
	
	err = impl_RALLoadModule(&descr, mem, NULL, &ref);
	if (err) {
		fatal("Module failed to load: 0x%04x\n", err);
		return false;
	}
	
	logt("RefNo: %u\n", ref);
	
	if (refNoP)
		*refNoP = ref;
	
	if (nEntriesP)
		*nEntriesP = nfo.numEntrypts;
	
	return true;
}

static bool patchModuleLowLevel(uint32_t refNo, uint32_t entryNum, void* replacementFunc, void **oldFuncStorageP)
{
	Err err;
	
	logt("Patching entry 0x%03x in lib with ref %u\n", entryNum, refNo);
	
	err = impl_RALPatchEntry(refNo, entryNum, replacementFunc, oldFuncStorageP);
	if (err) {
		fatal("Failed to patch: 0x%04x\n", err);
		return false;
	}
	
	return true;
}

void halCallEvtCallback(uint32_t type, const void* data)
{
	HalEvtCbk cbk = NULL;
	
	if (!zodFilterHalEvents(type, data))
		return;
	
	KALTaskSwitching(false);
	cbk = mHalEvtCbks[type];
	if (cbk)
		cbk(data);
	KALTaskSwitching(true);
}

static Err pMemMove(void *dstP, const void *sP, uint32_t numBytes)
{
	memmove(dstP, sP, numBytes);
	return errNone;
}

static Err pMemSet(void *dstP, uint32_t numBytes, uint8_t value)
{
	memset(dstP, value, numBytes);
	return errNone;
}

static uint64_t halHyperForwarder(uint32_t call, void* data)
{
	#ifdef MACH_TYPE_VIRTUAL
		return hyperCall(call, data);
	#else
		//TODO: hw
		fatal("hyper forwarder called on real hw!\n");
		
		return 0;
	#endif
}

static int32_t exportedAdcGetValue(enum AdcValueIdx which)
{
	return adcGetValue(which);
}

bool exportedPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return hwPwrCtl(selector, newValP, oldValP);
}

static void dalMain(void)
{
	uint32_t bootModRefNo, codeStart, numBootEntries;
	const struct StoreHeader *stor;
	const struct CardHeader *card;
	extern char __DAL_START[];
	
	logi("in dal main\n");
	kernelInitLate();
	
	mRndState[0] = 0;
	mRndState[1] = 0;
	(void)KGetUptimeMsec((uint32_t*)&mRndState[0]);
	
	machInit(STAGE_INIT_IN_THREAD, NULL);

	card = findCard();
	if (!card)
		fatal("Cannot find CARD\n");
	
	stor = findStore();
	if (!stor)
		fatal("Cannot find STORE\n");
	
	if ((char*)stor <= (char*)card)
		fatal("STORE must be after CARD\n");
	
	logi("Found store at 0x%08X and card at 0x%08x\n", (unsigned)stor, (unsigned)card);
	
	logi("initing module manager...\n");
	if (!ralInit())
		fatal("Module manager failed\n");
	
	//export hyper forwarder right away
	#ifdef MACH_TYPE_VIRTUAL
		if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_HYPER_FORWARDER, &halHyperForwarder))
			fatal("Hyper forwarder export fail\n");
	#endif
	
	//export funcs
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_GET_ADC_VALUE, &exportedAdcGetValue))
		fatal("ADC getter export failed\n");
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_PWR_CTL, &exportedPwrCtl))
		fatal("PwrCtl accessor export failed\n");
	
	logi("Loading DAL\n");
	codeStart = (uint32_t)__DAL_START;
	logd("DAL code expected at 0x%08x\n", codeStart);
	if (!loadModuleLowLevel((void*)codeStart, card->dalAmddRsrcOfst ? (void*)(uintptr_t)((uint32_t)card + card->dalAmddRsrcOfst) : NULL, card->dalAmdiRsrcOfst ? (void*)(uintptr_t)((uint32_t)card + card->dalAmdiRsrcOfst) : NULL, &mDalModRefNo, NULL))
		return;
	
	logi("Loading Boot (@0x%08x)\n", stor->bootBootRsrcPtr);
	if (!loadModuleLowLevel((void*)stor->bootBootRsrcPtr, (void*)stor->bootRsrsBoot10001, (void*)stor->bootRsrsBoot10002, &bootModRefNo, &numBootEntries))
		return;
	
	if (numBootEntries > 909) {
		
		#ifndef SUPPORT_OS_54
			
			fatal("Garnet support not enabled\n");
			__builtin_unreachable();
		#endif
		
		logt("Garnet Boot device detected\n");
		mIsGarnet = true;
	}
	
	if (numBootEntries < 840) {
		
		#ifndef SUPPORT_OS_50
			
			fatal("OS 5.0 support not enabled\n");
			__builtin_unreachable();
		#endif

		logt("5.0 Boot device detected\n");
		mIs50 = true;
	}
	
	logi("patching some funcs for convenience\n");
	if (!patchModuleLowLevel(bootModRefNo, 0x9F, (void*)pErrDisplayFileLineMsg, NULL))
		return;
	if (!patchModuleLowLevel(bootModRefNo, 0x19B, (void*)pSerialMgrInstall, (void**)&mOldSerialMgrInstall))
		return;
	
	//logi("patching some funcs for error reporting\n");
	//if (!patchModuleLowLevel(bootModRefNo, 0x21F, (void*)pSysLinkerStub, (void**)&mOldSysLinkerStub))
	//	return;


	logi("patching some funcs for speed\n");
	if (!patchModuleLowLevel(bootModRefNo, 0x156, (void*)pMemMove, NULL))
		return;
	if (!patchModuleLowLevel(bootModRefNo, 0x16C, (void*)pMemSet, NULL))
		return;
	if (!patchModuleLowLevel(bootModRefNo, 0x130, (void*)memcmp, NULL))	//MemCmp
		return;
	if (!patchModuleLowLevel(bootModRefNo, 0x1E1, (void*)strcmp, NULL))	//StrCompareAscii
		return;
	
	#ifdef SUPPORT_OS_54
	
		if (isGarnet()) {
			if (!patchModuleLowLevel(bootModRefNo, 0x5D8 / 4, (void*)pOmGetSystemLocale, (void**)&mOldOmGetSystemLocale))
				return;
		}
		
	#endif

	logi("initing some things\n");
	halDisplayInit();

	logi("creating thread to call main\n");
	callMain();

	logi("all is going well. going into hwr event listening loop\n");
	hwrEventListen();
}

uint32_t _entry(uint32_t cmd, void* cmdPBP, uint16_t flags)
{
	if (cmd == RAL_CMD_DAL_FIRST_BOOT) {
	
		kheapInit();
		kernelInit(dalMain, cmdPBP);
		//does not return
		
		//but make GCC happy
		__builtin_unreachable();
	}
	else if (cmd == RAL_CMD_GET_PACE_ENTRY) {
		
		*(void**)cmdPBP = dalGetPaceHandlerPtr();
	}
	
	return 0;
}

static void addFakeBtseFeatureToAppeaseVoicePad(void)
{
	static struct {
		uint32_t addr1;
		uint16_t instrs[2];	
	} mFtr;

	/*
		apparently T|T (and later) bt library exposes a large trap table via a 68k-callable feateure
		it has a lot of data (and is its own first param), first word in it points to a 68k func,
		callable via jsr, that returns in d0. It tkaes a selector in d2, and more than 50 exist
		it also takes a single param (pointer to more data, type TBD by selector)
		Known selectors are:
			* 0x35 to get bluetooth state. param is a bool* that gets state
			* 0x36 to set bt state. param is a large struct
		
		Launcher tests for this feature and silently carries on if not found - good, but VoicePad assumes
		it exists, and calls the pointer. This causes a crash if there is no feature (random shit is
		dereferenced and called). We here install this feature and make it callable. Luckily it will do
		no harm and we hope we find no more apps that use it. To make launcher happy, we return nonzero
		to indicate error. VoicePad will not care but now it can run!
	*/



	mFtr.addr1 = __builtin_bswap32((uintptr_t)&mFtr.instrs);
	mFtr.instrs[0] = __builtin_bswap16(0x7010);	//0x7010 = moveq #0xffffffff, d0		//to signal an error
	mFtr.instrs[1] = __builtin_bswap16(0x4e75);	//0x4e75 = rts
	
	(void)FtrSet(CREATE_4CC('b','t','s','e'), 1, (uintptr_t)&mFtr);
}

void DALEXPORT impl_HALSetInitStage(uint8_t stage)
{
	logi("HALSetInitStage(%u)\n", stage);
	mInitStage = stage;

	if (stage == DAL_INIT_STAGE_MEM_AND_DM) {
		kstatus_t sta = kernelDriversInit();
		if (sta != KERN_STATUS_OK)
			fatal("Cannot init drivers\n");
		halErrorHeapAvail();
		ralMemMgrAvail();
		machInit(STAGE_INIT_POST_RAL, NULL);
	}
	
	//handle sony-specific things
	sonyHalSetInitStage(stage);
	
	//zodiac too
	zodSetInitStage(stage);
	
	//patch ui entries for OS5 (sadly Palm and Sony did this differently)
	if (stage == DAL_INIT_STAGE_MEM_AND_DM && is50()) {
		
		//this is a bit messy. crt.c has more details as to why
		
		static const uint16_t palmIdxs50[] = {0x2EC, 0x2F0, 0x2F4, 0x2F8, 0x2FC, 0x300, 0x304};	// destinations for funcs 0x2F8 and on, in OS 5.2 order
		static const uint16_t sonyIdxs50[] = {0x2F0, 0x2F8, 0x2FC, 0x2F4, 0x300, 0x308, 0x304};
		const uint16_t *substitutionOrders = isSony() ? sonyIdxs50 : palmIdxs50;
		const void* drawingFuncs[7];
		uint32_t i;
		Err e;
		
		logt("Patching DAL drawing funcs for OS 5.0\n");
		
		e = RALGetEntryAddresses(mDalModRefNo, 0x2F8 / 4, 0x310 / 4, drawingFuncs);
		if (e)
			fatal("Failed to get addrs for 5.0 patching\n");
		
		for (i = 0; i < 7; i++) {
			e = RALPatchEntry(mDalModRefNo, substitutionOrders[i] / 4, drawingFuncs[i], NULL);
			if (e)
				fatal("Failed to set addr for 5.0 patching\n");
		}
	}
	
	if (stage == DAL_INIT_STAGE_NO_UI) {
		
		//do it here so real BtStack (if any) can override, but without this VoicePad from T|T will crash
		addFakeBtseFeatureToAppeaseVoicePad();
	}
	
	if (stage == DAL_INIT_STAGE_ALL_UP) {			//set features
		
		uint32_t val;
		//five way nav supported
		FtrSet(navFtrCreator, navFtrVersion, navVersion);
		
		//not sure
		FtrSet('PALM', 0, 16);
		
		//density (for conduits)
		val = 0;
		if (errNone != HALDisplayGetAttributes(isGarnet() ? hwrDispDensity54 : hwrDispDensity, &val))
			val = PALM_DENSITY_STANDARD;
		FtrSet(densityFtrCreator, densityFtrVersion, val);
		
		//real DPI
		if (errNone != HALDisplayGetAttributes(hwrDispRealDpi, &val))
			val = 60;
		FtrSet(screenFtrCreator, dpiFtrValue, val);
	}
}

uint32_t dalGetInitStage(void)
{
	return mInitStage;
}

static uint64_t halRand(void)//xorshift128plus
{
	uint64_t ret, x, y;
	
	//seed cannot be zero - make it one
	if (!mRndState[0] && !mRndState[1])
		mRndState[0] = 1;
	
	x = mRndState[0];
	y = mRndState[1];
	
	mRndState[0] = y;
	x ^= x << 23; // a
	mRndState[1] = x ^ y ^ (x >> 17) ^ (y >> 26); // b, c
	ret = mRndState[1] + y;

	
	//ret is the 64-bit random number generated
	
	return ret;
}

Err DALEXPORT impl_HALRandomGetBits(uint32_t numBits, void *bitsP)
{
	uint32_t i, numBytes = (numBits + 7) >> 3;
	uint8_t *dst = (uint8_t*)bitsP;
	uint64_t t;
	
	logt("%s\n", __func__);
	while(numBytes) {
		t = halRand();
		for (i = 0; i < 8 && numBytes; i++, numBytes--, t >>= 8)
			*dst++ = t;
	}
	
	return errNone;
}

Err DALEXPORT impl_HALRandomInitializeSeed(uint32_t *randomSeedP)
{
	uint64_t t;
	
	logt("%s\n", __func__);
	t = halRand();
	*randomSeedP = (t >> 32) ^ t;
	
	return errNone;
}

uint32_t DALEXPORT impl_HALProcessorID(void)
{
	logt("%s\n", __func__);
	
	return 0x160000;
}

uint32_t DALEXPORT impl_HALTimeGetSeconds(void)
{
	logst("%s\n", __func__);
	
	uint32_t ret;
		
	if (KERN_STATUS_OK != KRtcGet(&ret))
		fatal("syscall for RTC get\n");
	
	return ret;
}

uint32_t DALEXPORT impl_HALTimeGetSystemTime(void)
{
	uint32_t uptime;
		
	logst("%s\n", __func__);
	
	if (KERN_STATUS_OK != KGetUptimeMsec(&uptime))
		fatal("syscall for uptime failed\n");
	
	logst("%s -> 0x%08x\n", __func__, uptime);
	return uptime;
}

Err DALEXPORT impl_HALTimeSetSeconds(uint32_t seconds)
{
	logt("%s(%lu)\n", __func__, (unsigned long)seconds);
	
	return KERN_STATUS_OK == KRtcSet(seconds) ? errNone : 0xFF00;
}

Err DALEXPORT impl_HALTimeGetAlarm(uint32_t *secondsP)
{
	logt("%s\n", __func__);
	
	if (secondsP)
		*secondsP = mAlarmSeconds;
	
	return errNone;
}

Err DALEXPORT impl_HALTimeSetAlarm(uint32_t seconds)
{
	uint32_t now;
	
	logt("%s(%u)\n", __func__, seconds);
	now = impl_HALTimeGetSeconds();
	mAlarmSeconds = seconds;
	
	if (seconds && seconds <= now) {
		halCallEvtCallback(HAL_EVENT_TYPE_CLOCK, NULL);
		seconds = 0;
	}
	
	return KERN_STATUS_OK == KRtcSetAlarm(seconds) ? errNone : 0xFF00;
}

Err DALEXPORT impl_HALInvalidateICache(void *addr, int32_t sz)
{
	logt("%s\n", __func__);
	
	#ifdef ICACHE_CLEAR_PROC
	extern void ICACHE_CLEAR_PROC(uintptr_t addr, int32_t sz);
	ICACHE_CLEAR_PROC((uintptr_t)addr, sz);
	#endif
	
	return errNone;
}

Err DALEXPORT impl_HALBatteryGetInfo(uint16_t *warnThreshPercP, uint16_t *critThreshPercP, uint16_t *shtdnThreshPercP, uint32_t *warnTicksP, SysBatteryKind *battKindP, bool *haveMainsP, SysBatteryState *battStateP, uint8_t *curPrecP)
{
	logt("%s\n", __func__);
	
	if (warnThreshPercP)
		*warnThreshPercP = 10;
	
	if (critThreshPercP)
		*critThreshPercP = 5;
		
	if (shtdnThreshPercP)
		*shtdnThreshPercP = 2;
	
	if (warnTicksP)
		*warnTicksP = 60000;
	
	if (battStateP)
		*battStateP = sysBatteryStateNormal;
	
	batteryInfo(battKindP, haveMainsP, curPrecP);
	
	return errNone;
}

uint16_t DALEXPORT impl_HALInterruptAllSetStatus(uint16_t desiredVal)
{
	uint16_t ret = mIntsEnabled;
	mIntsEnabled = desiredVal;
	
	if (desiredVal)
		irqsAllOn();
	else
		irqsAllOff();
	
	return ret;
}

uint16_t DALEXPORT impl_HALInterruptAllGetStatus(void)
{
	return mIntsEnabled;
}

uint16_t DALEXPORT impl_HALPowerGetAutoOffSeconds(void)
{
	logt("%s\n", __func__);
	
	return mAutoOffSeconds;
}

void DALEXPORT impl_HALPowerSetAutoOffSeconds(uint16_t seconds)
{
	logt("%s\n", __func__);
	
	mAutoOffSeconds = seconds;
}

uint32_t DALEXPORT impl_HALPowerGetAutoOffEvtTime(void)
{
	logst("%s\n", __func__);
	
	return mAutoOffEvtTime;
}

void DALEXPORT impl_HALPowerSetAutoOffEvtTime(uint32_t evtTime)
{
	logt("%s\n", __func__);
	
	mAutoOffEvtTime = evtTime;
}

uint16_t DALEXPORT impl_HALSilkscreenGetID(void)	//open dal and return id of the silkscreen resource (19000 for us)
{
	logt("%s\n", __func__);
	
	if (isGarnet())
		return 10002;
	else if (isSony())
		return 10000;
	else {
		DmOpenDatabaseByTypeCreator(CREATE_4CC('r','s','r','c'), CREATE_4CC('p','d','a','l'), dmModeLeaveOpen);
		return 19000;
	}
}

Err DALEXPORT impl_HALEventRegisterCallBack(uint32_t evt, HalEvtCbk cbk, HalEvtCbk* oldCbkP)
{
	logt("%s\n", __func__);
	
	if (evt != HAL_EVENT_TYPE_PEN && evt != HAL_EVENT_TYPE_KEY && evt != HAL_EVENT_TYPE_CLOCK)
		return 0xFF00;
	
	if (oldCbkP)
		*oldCbkP = mHalEvtCbks[evt];
	
	mHalEvtCbks[evt] = cbk;
	
	return errNone;
}

uint16_t DALEXPORT impl_HALGetHwrMiscFlags(void)
{
	uint16_t flags;
	
	logt("%s\n", __func__);
	
	hwGetMiscFlags(&flags, NULL);
	
	return flags;
}

uint32_t DALEXPORT impl_HALGetHwrMiscFlagsExt(void)
{
	uint16_t extFlags;
	
	logt("%s\n", __func__);
	
	hwGetMiscFlags(NULL, &extFlags);
	
	return extFlags;
}

uint32_t DALEXPORT impl_HALOEMGetCompanyID(void)
{
	logt("%s\n", __func__);
	
	return findCard()->companyID;
}

uint32_t DALEXPORT impl_HALOEMGetHALID(void)
{
	logt("%s\n", __func__);
	
	return findCard()->halID;
}

uint32_t DALEXPORT impl_HALOEMGetDeviceID(void)
{
	logt("%s\n", __func__);
	
	return findCard()->halID;			//we pretend our device id is same as device id
}

Err DALEXPORT impl_HALEventPost(uint32_t evt, const void *evtData)
{
	logt("%s(%u, 0x%08x)\n", __func__, evt, evtData);
	
	if (evt != HAL_EVENT_TYPE_PEN && evt != HAL_EVENT_TYPE_KEY && evt != HAL_EVENT_TYPE_CLOCK)
		return 0xFFFF;
	
	//for some reason existing hal only alows posting of key data. for now we do the same
	if (evt != HAL_EVENT_TYPE_KEY || !evtData)
		return 0xFFFF;
	
	halCallEvtCallback(evt, evtData);
	
	return errNone;
}

Err DALEXPORT impl_HALHandleReset(uint32_t refNum, uint32_t resetType)
{
	logt("%s(0x%08lx, 0x%08lx) from 0x%08x\n", __func__, refNum, resetType, __builtin_return_address(0));
	
	return errNone;	//not sure if we need to return anything
}

Err DALEXPORT impl_HALGetROMToken(uint32_t tokenName, const void **dataP, uint16_t *szP)
{
	const uint32_t *tokenStore;
	
	logt("%s('%c%c%c%c')\n", __func__, ((tokenName >> 24) & 0xff), ((tokenName >> 16) & 0xff), ((tokenName >> 8) & 0xff), ((tokenName >> 0) & 0xff));
	
	if (hwMaybeGetRomToken(tokenName, dataP, szP))
		return errNone;
	
	tokenStore = (const uint32_t*)findCard()->romTokenStorePtr;
	
	while (tokenStore[0] != 0xFFFFFFFF) {
		
		if (tokenStore[0] == tokenName) {
			if (dataP)
				*dataP = tokenStore + 2;
			if (szP)
				*szP = tokenStore[1];
			
			return errNone;
		}
		
		tokenStore += 2 + tokenStore[1] / 4;
	}
	
	return 0xFFFF;
}

bool DALEXPORT impl_DALSetDbgLockout(bool setLockout)	//returns prev state
{
	bool ret = mDbgLockout;
	
	logt("%s\n", __func__);
	mDbgLockout = setLockout;
	
	return ret;
}

uint32_t DALEXPORT impl_HALGetHwrWakeUp(void)	//returns a bitfield of bits that indicate which hwr components are NOT awake (bitmask 0x10 is for display)
{
	logt("%s -> 0x%02x\n", __func__, mWakeFlags);
	
	return mWakeFlags;
}

void dalModifyWakeFlags(uint32_t setBits, uint32_t clearBits)
{
	mWakeFlags = (mWakeFlags | setBits) &~ clearBits;
}

void halDockStatusChanged(uint16_t bits)		//	HAL_DOCK_STATUS_*
{
	uint16_t prevStatus = mDockStatus;
	
	mDockStatus = bits;
	
	if (prevStatus != bits) {
		
		uint32_t diff = prevStatus ^ bits;
		uint32_t nowOff = diff & ~bits;
		uint32_t nowOn = diff & bits;
		
		if (dalGetInitStage() >= DAL_INIT_STAGE_ALL_UP) {		//notif mgr is only up at this stage
			
			if (nowOff)
				SysNotifyBroadcastFromInterrupt(sysExternalConnectorDetachEvent, sysFileCExternalConnector, NULL);
		
			if (nowOn)
				SysNotifyBroadcastFromInterrupt(sysExternalConnectorAttachEvent, sysFileCExternalConnector, (void*)(uintptr_t)bits);
		}
	}
}

Err DALEXPORT impl_HALDockStatus(uint16_t *statusBitfieldP)
{
	dockHwrInit();		//init the dock hwr to make sure "mDockStatus" is valid 
	
	*statusBitfieldP = mDockStatus;
	
	return errNone;
}

//In OS5.0 (and not anywhere after) HddExtension will read [[DalGlobals, 4], 0x424] and leave us alone if that's zero.
//[DalGlobals, 4] points to ScrGlobalsStruct, and [ScrGlobals, 0x424] is tranferFunc() pointer, if any.
//If we have one, more things will be attempted. If we do not, HddExtension will leave us alone, so we set
//up fake globals to appear thusly. We also make sure that the call if from inside HddExtension, and that
//the OS is 5.0, of course.
void* DALEXPORT impl_DALGetGlobalsPtr(void)
{
	uint32_t codeStart = 0, codeSz, retTo = ((uint32_t)__builtin_return_address(0)) &~ 1;
	static uint32_t fakeGlobals[2] = {0, (uint32_t)(fakeGlobals - 0x109)};	//very compact.
	MemHandle res;
	DmOpenRef db;
	
	if (!is50())
		fatal("%s only supported for the hackish way OS 5.0 drags itself up\n", __func__);

#ifdef SUPPORT_OS_50
	
	db = DmOpenDatabaseByTypeCreator(CREATE_4CC('a','e','x','o'), CREATE_4CC('h','i','d','d'), dmModeReadOnly);
	if (!db)
		fatal("HddExtension cannot be opened\n");
	
	res = DmGet1Resource(CREATE_4CC('a','m','d','c'), 0);
	if (!res)
		fatal("HddExtension's ('amdc', 0) cannot be opened\n");
	
	codeStart = (uint32_t)MemHandleLock(res);
	if (!codeStart)
		fatal("HddExtension's ('amdc', 0) cannot be locked\n");
	
	logt("HddExtension's ('amdc', 0)'s start is 0x%08lx\n", codeStart);
	
	codeSz = MemPtrSize((void*)codeStart);
	if (!codeSz || (codeSz & 3))
		fatal("%s: cannot find the size of HddExtension's ('amdc', 0)\n", __func__);
	
	if (errNone != MemHandleUnlock(res))
		fatal("HddExtension's ('amdc', 0) cannot be unlocked\n");
	if (errNone != DmCloseDatabase(db))
		fatal("HddExtension cannot be closed\n");
	
	logt("HddExtension's ('amdc', 0)'s size is 0x%08lx\n", codeSz);
	
	if (retTo < codeStart || retTo >= codeStart + codeSz)
		fatal("%s only supported when called form HddExtension\n", __func__);
	
	logt("returning\n");
#endif

	return fakeGlobals;
}

Err DALEXPORT impl_HALPowerSleepReady(void)
{
	const struct HalEvtKey mWakeKey = {.modifiers = commandKeyMask, .ascii = vchrResetAutoOff, };
	logt("%s\n", __func__);
	
	logi("Going to sleep\n");
	
	//clear old flags
	mWakeFlags &=~ (DAL_WAKE_FLAG_GENERAL | DAL_WAKE_FLAG_PWM | DAL_WAKE_FLAG_FROM_KEY | DAL_WAKE_FLAG_FROM_RTC);
	
	if (!(mWakeFlags & DAL_WAKE_FLAG_LCD_ASLEEP))
		HALDisplaySleep(false, false);
	machSleep();
	
	halCallEvtCallback(HAL_EVENT_TYPE_KEY, &mWakeKey);

	HALDisplayWake();
	
	return errNone;
}

Err DALEXPORT impl_dprintf(const char * str, ...)
{
	logw("DPRINTF: '%s'\n", str);
	
	return errNone;
}

bool isGarnet(void)
{
#ifdef SUPPORT_OS_54
	return mIsGarnet;
#else
	return false;
#endif
}

bool is50(void)
{
#ifdef SUPPORT_OS_50
	return mIs50;
#else
	return false;
#endif
}

Err DALEXPORT impl_HALInterruptGetHandler(uint32_t irqNo, HalIrqHandlerF *curHandlerP, void** curUserDataP)
{
	uint32_t realIrqNo = REPALM_IRQ_NO_MANGLE(irqNo);
	
	if (realIrqNo >= CPU_NUM_IRQS)
		return 0xFF00;
	
	return cpuIrqGetHandler(realIrqNo, curHandlerP, curUserDataP) ? errNone : 0xFF00;
}

Err DALEXPORT impl_HALInterruptSetHandler(uint32_t irqNo, HalIrqHandlerF newHandler, void* newUserData)
{
	uint32_t realIrqNo = REPALM_IRQ_NO_MANGLE(irqNo);
	
	if (realIrqNo >= CPU_NUM_IRQS)
		return 0xFF00;
	
	return cpuIrqSetHandler(realIrqNo, newHandler, newUserData) ? errNone : 0xff00;
}

bool DALEXPORT impl_HALInterruptSetState(uint32_t irqNo, bool on)	//returns old state
{
	uint32_t realIrqNo = REPALM_IRQ_NO_MANGLE(irqNo);
	
	if (realIrqNo >= CPU_NUM_IRQS)
		return 0xFF00;
	
	return cpuIrqSetState(realIrqNo, on);
}

Err DALEXPORT impl_HALKeyGetDoubleTapDelay(uint16_t *delayP)
{
	if (delayP)
		*delayP = mDoubleTapDelay;
	
	return errNone;
}

Err DALEXPORT impl_HALKeySetDoubleTapDelay(uint16_t delay)
{
	mDoubleTapDelay = delay;
	
	return errNone;
}

void DALEXPORT impl_HALReset(bool doHardReset)
{
	deviceReset(doHardReset);
}
