#define _GNU_SOURCE
#include <stdbool.h>
#include <getopt.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <errno.h>
#include <stdio.h>
#include <ctype.h>

#include "parsePrcPdb.h"
#include "osStructs.h"
#include "util.h"

//mkrom can only produce OS5 images. It is not meant to produce OS4 images

//block list is a list of RAM sizes in a given card
// first entrty is always 0
// terminator entry is 0xFFFFFFFF
// our ROM has no RAM, so we just need those 2 words in order there


#define ROM_SIZE_MAX					(32 * 1024 * 1024)		//this had better be a multiple of 8, or all hell will break loose
#define ROM_SIZE_DEFAULT				(4 * 1024 * 1024)		//this had better be a multiple of 8, or all hell will break loose
#define ROM_BASE_ADDR_DEFAULT			0xD0400000				//this had better be 8-byte aligned, or all hell will break loose

//#define MY_COMPANY_ID			MAKE_4CC('g','r','m','n')
//#define MY_COMPANY_ID			MAKE_4CC('s','o','n','y')
//#define MY_COMPANY_ID			MAKE_4CC('P','a','l','m')
#define ROM_MANUF_ID_DEFAULT	MAKE_4CC('D','m','G','R')
#define HAL_ID_DEFAULT			MAKE_4CC('r','e','P','a')

#define BLOCK_LIST_OFST			0x00000200						//POSE *REALLY* wants this to be at 0x200 offset from start
#define BLOCK_LIST_SIZE			0x00000008

#define ROM_TOKEN_STORE_OFST		0x000000C0
#define ROM_TOKEN_STORE_SIZE		0x00000030

#define ROM_STORE_OFST			0x00000100

#define HEAP_LIST_OFST			0x00000208
#define HEAP_LIST_SIZE			0x00000008

#define HEAP_HDR_OFST			0x00000210						//minimum. more can be added via "cfg->extraHeapOffset"


#define LANG_DEFAULT			0x0000	//english
#define COUNTRY_DEFAULT			0x0017


struct Token {
	struct Token *next;
	uint32_t name;
	uint16_t len;
	uint8_t data[];
};

struct MkromConfig {
	uint32_t baseAddr;
	uint32_t romSz;
	uint32_t fakeEntryAddr;
	uint32_t tokensAddr;
	uint32_t manuf;
	uint32_t halId;
	uint16_t lang;
	uint16_t country;
	uint16_t cardHdrVersion;

	uint32_t cardClaimedOsVerVal;
	char cardName[32];
	char cardManuf[32];
	char cardVersionString[32];
	uint32_t initialSp68K;
	bool fixup68kBootOfst;
	
	struct Token *tokens;
	
	bool haveFakeEntryAddr, unsafe, verbose, isOS5, hasRamBlocks;
	
	uint32_t extraHeapOffset;	//to move entry point sometimes
	
	char *self;
	char **files;
	int numFiles;
};



static uint8_t* addSysTokenOS5(uint8_t *dst, uint32_t name, uint32_t len, const void *data)	//returns pointer to next free byte in rom token store
{
	writeSE32((uint32_t*)(dst + 0), name);
	writeSE32((uint32_t*)(dst + 4), len);
	memcpy(dst + 8, data, len);
	
	return dst + ((len + sizeof(uint32_t) - 1) &~ (sizeof(uint32_t) - 1)) + 2 * sizeof(uint32_t);
}

static uint8_t* addSysTokenOS4(uint8_t *dst, uint32_t name, uint32_t len, const void *data)	//returns pointer to next free byte in rom token store
{
	writeSE32((uint32_t*)(dst + 0), name);
	writeSE16((uint16_t*)(dst + 4), len);
	memcpy(dst + 6, data, len);
	
	return dst + ((len + sizeof(uint16_t) - 1) &~ (sizeof(uint16_t) - 1)) + sizeof(uint32_t) + sizeof(uint16_t);
}

static struct HeapChunkHeader* getNextChunk(struct HeapChunkHeader* curChunk)
{
	struct HeapChunkHeader *next = (struct HeapChunkHeader*)(((uintptr_t)curChunk) + ((readSE32(&curChunk->w0) & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT));
	
	return readSE32(&next->w0) ? next : NULL;
}

static void heapShowFree(void *heapHdr, bool isOS5)
{
	struct HeapHeaderV4_ARM *heapV4_arm = (struct HeapHeaderV4_ARM*)heapHdr;
	struct HeapHeaderV4_68k *heapV4_68k = (struct HeapHeaderV4_68k*)heapHdr;
	struct HeapHeaderV3 *heapV3 = (struct HeapHeaderV3*)heapHdr;
	struct HeapChunkHeader *curChunk;
	bool heapIsV3;
	
	heapIsV3 = readSE16(&heapV4_68k->flags) & HEAP_FLAGS_IS_V3;
	
	//calc the first chunk pointer
	if (heapIsV3)
		curChunk = (struct HeapChunkHeader*)(heapV3 + 1);
	else if (isOS5)
		curChunk = (struct HeapChunkHeader*)(heapV4_arm + 1);
	else
		curChunk = (struct HeapChunkHeader*)(heapV4_68k + 1);
	
	fprintf(stderr, "FREE HEAP CHUNKS:\n");
	while (curChunk != NULL) {
		
		uint32_t w0, w1, freeSpace;
		
		w0 = readSE32(&curChunk->w0);
		w1 = readSE32(&curChunk->w1);
		
		if (w0 & CHUNK_HDR_W0_FREE_MASK) {
			
			freeSpace = (w0 & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT;
			
			fprintf(stderr, "free chunk at offset 0x%08lx, size %u\n", (unsigned long)(((uintptr_t)curChunk) - ((uintptr_t)heapHdr)), freeSpace);
		}
		
		curChunk = getNextChunk(curChunk);
	}
}

static void* cardStoreAndHeapFormat(uint8_t *rom, const struct MkromConfig *cfg, uint32_t romSz, struct CardHeader **cardHdrP, struct StoreHeader** storHdrP)	//return heap header
{
	struct StoreHeader* stor = (struct StoreHeader*)(rom + ROM_STORE_OFST);
	struct HeapListOS4 *heapListOS4 = (struct HeapListOS4*)(rom + HEAP_LIST_OFST);
	struct HeapListOS5 *heapListOS5 = (struct HeapListOS5*)(rom + HEAP_LIST_OFST);
	struct HeapHeaderV4_ARM *heapV4_arm = (struct HeapHeaderV4_ARM*)(rom + HEAP_HDR_OFST + cfg->extraHeapOffset);
	struct HeapHeaderV4_68k *heapV4_68k = (struct HeapHeaderV4_68k*)(rom + HEAP_HDR_OFST + cfg->extraHeapOffset);
	struct HeapHeaderV3 *heapV3= (struct HeapHeaderV3*)(rom + HEAP_HDR_OFST + cfg->extraHeapOffset);
	uint32_t *blockList = (uint32_t*)(rom + BLOCK_LIST_OFST), freeChunkSize;
	struct CardHeader *hdr = (struct CardHeader*)rom;
	const uint8_t heapVersion = 4;//cfg->isOS5 ? 4 : 3;
	uint32_t targetVa = cfg->baseAddr;
	struct HeapChunkHeader *chnk;
	bool tokenStoreInOurImage;
	uint8_t *romTokens = NULL;
	
	
	
	//card header and things it needs
	memset(hdr, 0, sizeof(*hdr));
	writeSE32(&hdr->signature, CARD_HEADER_SIG);
	writeSE16(&hdr->hdrVersion, cfg->cardHdrVersion);
	writeSE16(&hdr->flags, cfg->isOS5 ? (CARD_FLAGS_ARM | CARD_FLAGS_WORD_ALIGN) : CARD_FLAGS_68K_EZ);
	
	strncpy(hdr->name, cfg->cardName, sizeof(hdr->name));
	strncpy(hdr->manuf, cfg->cardManuf, sizeof(hdr->manuf));
	
	if (cfg->isOS5) {
		writeSE16(&hdr->diff1.os5.version, 1);
		writeSE16(&hdr->diff1.os5.numRAMBlocks, cfg->hasRamBlocks ? 1 : 0);
		writeSE32(&hdr->diff1.os5.creationDate, 0xF0000000);	//close enough
	}
	else {
		writeSE16(&hdr->diff1.os4.version, 1);
		writeSE16(&hdr->diff1.os4.numRAMBlocks, cfg->hasRamBlocks ? 1 : 0);
		writeSE32(&hdr->diff1.os4.creationDate, 0xF0000000);	//close enough
	}
	writeSE32(&hdr->blockListPtr, targetVa + BLOCK_LIST_OFST);
	writeSE32(&hdr->romTokenStorePtr, cfg->tokensAddr);
	writeSE32(&hdr->bigRomPtr, targetVa);	//can we get away with just a big rom?
	writeSE32(&hdr->totalRomSz, romSz);
	
	if (cfg->cardHdrVersion >= 5) {
	
		if (cfg->isOS5) {
			writeSE32(&hdr->diff2.os5.companyID, cfg->manuf);
			writeSE32(&hdr->diff2.os5.halID, cfg->halId);
			writeSE32(&hdr->diff2.os5.romVersion, cfg->cardClaimedOsVerVal);
			strncpy(hdr->diff2.os5.romVersionString, cfg->cardVersionString, sizeof(hdr->diff2.os5.romVersionString));
		}
		else {
			writeSE32(&hdr->diff2.os4.companyID, cfg->manuf);
			writeSE32(&hdr->diff2.os4.halID, cfg->halId);
			writeSE32(&hdr->diff2.os4.romVersion, cfg->cardClaimedOsVerVal);
			strncpy(hdr->diff2.os4.romVersionString, cfg->cardVersionString, sizeof(hdr->diff2.os4.romVersionString));
		}
	}
	
	//init rom token store if we're creating one
	tokenStoreInOurImage = (cfg->tokensAddr >= targetVa && cfg->tokensAddr - targetVa < romSz);
	if (!tokenStoreInOurImage && cfg->tokens) {
		fprintf(stderr, "Token store not part of our image address space, but tokens were requested\n");
		return NULL;
	}
	if (tokenStoreInOurImage) {
		
		bool seenSnum = false, seenKvnp = false, first = true;
		struct Token *t;
		
		romTokens = rom + cfg->tokensAddr - targetVa;
		
		memset(romTokens, 0xFF, ROM_TOKEN_STORE_SIZE);
		
		for (t = cfg->tokens; t; t = t->next, first = false) {
			
			if (t->name == MAKE_4CC('s','n','u','m')) {
				
				seenSnum = true;
				if (!first)
					fprintf(stderr, "Warning: 'snum' token should be first!\n");
			}
			
			if (t->name == MAKE_4CC('k','n','v','p'))
				seenKvnp = true;
			
			if (cfg->isOS5)
				romTokens = addSysTokenOS5(romTokens, t->name, t->len, t->data);
			else
				romTokens = addSysTokenOS4(romTokens, t->name, t->len, t->data);
		}
		
		if (!seenSnum)
			fprintf(stderr, "Warning: 'snum' token missing - this is probably not ok for full-OS ROMs!\n");
		if (!seenKvnp)
			fprintf(stderr, "Warning: 'knvp' token missing - this is probably going to mess up the keyboard for full-OS ROMs!\n");
	}
	
	//init block list
	writeSE32(&blockList[0], 0x00000000);
	writeSE32(&blockList[1], cfg->hasRamBlocks ? 0xFFFFFFFF : 0);
	
	//store header
	memset(stor, 0, sizeof(*stor));
	writeSE32(&stor->signature, STORE_HEADER_SIG);
	writeSE16(&stor->version, cfg->isOS5 ? 2 : 1);
	strncpy(stor->name, "ROM Store", sizeof(stor->name));
	writeSE32(&stor->heapListPtr, targetVa + HEAP_LIST_OFST);
	
	//heap list
	if (cfg->isOS5) {
		writeSE16(&stor->os5.romLanguage, cfg->lang);
		writeSE16(&stor->os5.romCountry, cfg->country);
		writeSE16(&heapListOS5->numItems, 1);
		writeSE16(&heapListOS5->unused, 0);
		writeSE32(&heapListOS5->heapHeaderPtrs[0], targetVa + HEAP_HDR_OFST + cfg->extraHeapOffset);
	}
	else {
		writeSE16(&stor->os4.romLanguage, cfg->lang);
		writeSE16(&stor->os4.romCountry, cfg->country);
		writeSE16(&heapListOS4->numItems, 1);
		writeSE32(&heapListOS4->heapHeaderPtrs[0], targetVa + HEAP_HDR_OFST + cfg->extraHeapOffset);
	}
	
	//heap
	switch (heapVersion) {
		case 3:
			writeSE16(&heapV3->flags, HEAP_FLAGS_RO | HEAP_FLAGS_IS_V3);	//read only
			writeSE32(&heapV3->size, romSz - HEAP_HDR_OFST - cfg->extraHeapOffset);
			writeSE32(&heapV3->firstFreeChunkOffset, 0);	//at the end the heap will have no free chunks
			writeSE16(&heapV3->mstrPtrTbl.numEntries, 0);
			writeSE32(&heapV3->mstrPtrTbl.nextTblOffset, 0);	//we do not have one
			chnk = (struct HeapChunkHeader*)(heapV3 + 1);
			freeChunkSize = romSz - HEAP_HDR_OFST - cfg->extraHeapOffset - sizeof(*heapV3) - sizeof(uint32_t) /* terminator*/;
			break;
		
		case 4:
			if (cfg->isOS5) {
				
				writeSE16(&heapV4_arm->flags, HEAP_FLAGS_RO | HEAP_FLAGS_IS_V4);	//read only
				writeSE16(&heapV4_arm->ffff, 0xffff);	//do not ask...
				writeSE32(&heapV4_arm->size, romSz - HEAP_HDR_OFST);
				writeSE32(&heapV4_arm->firstFreeChunkOffset, 0);	//at the end the heap will have no free chunks
				writeSE16(&heapV4_arm->mstrPtrTbl.numEntries, 0);
				writeSE16(&heapV4_arm->mstrPtrTbl.ffff, 0xffff);		//do not ask...
				writeSE32(&heapV4_arm->mstrPtrTbl.nextTblOffset, 0);	//we do not have one
				chnk = (struct HeapChunkHeader*)(heapV4_arm + 1);
				freeChunkSize = romSz - HEAP_HDR_OFST - cfg->extraHeapOffset - sizeof(*heapV4_arm) - sizeof(uint32_t) /* terminator*/;
			}
			else {
				
				writeSE16(&heapV4_68k->flags, HEAP_FLAGS_RO | HEAP_FLAGS_IS_V4);	//read only
				writeSE32(&heapV4_68k->size, romSz - HEAP_HDR_OFST);
				writeSE32(&heapV4_68k->firstFreeChunkOffset,  0);	//at the end the heap will have no free chunks
				writeSE16(&heapV4_68k->mstrPtrTbl.numEntries, 0);
				writeSE32(&heapV4_68k->mstrPtrTbl.nextTblOffset, 0);	//we do not have one
				chnk = (struct HeapChunkHeader*)(heapV4_68k + 1);
				freeChunkSize = romSz - HEAP_HDR_OFST - cfg->extraHeapOffset - sizeof(*heapV4_68k) - sizeof(uint32_t) /* terminator*/;
			}
			break;
		
		default:
			fprintf(stderr, "This heap version not supported in this tool\n");
			return NULL;
	}
	
	//our heap might be too big for one chunk, split into a few
	do {
		
		uint32_t now = (freeChunkSize > 0x00ff0000) ? 0x00800000 : freeChunkSize;
		
		writeSE32(&chnk->w0, ((now << CHUNK_HDR_W0_SIZE_SHIFT) & CHUNK_HDR_W0_SIZE_MASK) | CHUNK_HDR_W0_FREE_MASK);
		writeSE32(&chnk->w1, ((CHUNK_OWNER_NOBODY) << CHUNK_HDR_W1_OWNER_SHIFT) & CHUNK_HDR_W1_OWNER_MASK);
		
		chnk = (struct HeapChunkHeader*)(((uintptr_t)chnk) + now);
		freeChunkSize -= now;
		
	} while (freeChunkSize);
	
	heapShowFree(heapV3, cfg->isOS5);
	
	//terminator
	writeSE32((uint32_t*)(rom + romSz - 4), 0);
	
	*cardHdrP = hdr;
	*storHdrP = stor;
	
	return (void*)(rom + HEAP_HDR_OFST + cfg->extraHeapOffset);
}

//we do not want to leave free chunks in the rom heap (to start with, due to idiotic bugs, POSE chokes on them)
//so find our free chunk and make not not free :D
static void heapFinalize(void *heapHdr, bool isOS5)
{
	struct HeapHeaderV4_ARM *heapV4_arm = (struct HeapHeaderV4_ARM*)heapHdr;
	struct HeapHeaderV4_68k *heapV4_68k = (struct HeapHeaderV4_68k*)heapHdr;
	struct HeapHeaderV3 *heapV3 = (struct HeapHeaderV3*)heapHdr;
	uint32_t w0, w1, freeSpace = 0, freeChunks = 0;
	struct HeapChunkHeader *curChunk;
	bool heapIsV3;
	
	heapIsV3 = readSE16(&heapV4_68k->flags) & HEAP_FLAGS_IS_V3;
	
	//calc the first chunk pointer
	if (heapIsV3)
		curChunk = (struct HeapChunkHeader*)(heapV3 + 1);
	else if (isOS5)
		curChunk = (struct HeapChunkHeader*)(heapV4_arm + 1);
	else
		curChunk = (struct HeapChunkHeader*)(heapV4_68k + 1);
	
	while (curChunk != NULL) {
		
		w0 = readSE32(&curChunk->w0);
		w1 = readSE32(&curChunk->w1);
		
		if (w0 & CHUNK_HDR_W0_FREE_MASK) {
			
			freeSpace += (w0 & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT;
			freeChunks++;
			
			w0 &=~ CHUNK_HDR_W0_FREE_MASK;
			w1 &=~ CHUNK_HDR_W1_OWNER_MASK;
			w1 |= (CHUNK_OWNER_DM_REC_AND_RES << CHUNK_HDR_W1_OWNER_SHIFT) & CHUNK_HDR_W1_OWNER_MASK;
			
			writeSE32(&curChunk->w0, w0);
			writeSE32(&curChunk->w1, w1);
		}
		
		curChunk = getNextChunk(curChunk);
	}
	
	fprintf(stderr, "Free space left over in the rom : %u bytes in %u chunks\n", freeSpace, freeChunks);
}

//we always have just one free chunk
static void* heapAlloc(void *heapHdr, uint32_t sz, uint32_t owner, bool atStart, bool isOS5)	//allocates a chunk in the heap
{
	struct HeapHeaderV4_ARM *heapV4_arm = (struct HeapHeaderV4_ARM*)heapHdr;
	struct HeapHeaderV4_68k *heapV4_68k = (struct HeapHeaderV4_68k*)heapHdr;
	struct HeapHeaderV3 *heapV3 = (struct HeapHeaderV3*)heapHdr;
	struct HeapChunkHeader *curChunk, *bestChunk = NULL, *myChunk;
	uint32_t fullSz, extraSz, freeChunkSize;
	bool heapIsV3;
	
	
	heapIsV3 = readSE16(&heapV4_68k->flags) & HEAP_FLAGS_IS_V3;
	
	//calc sizes
	extraSz = (sz + 7) / 8 * 8 - sz;
	fullSz = extraSz + sz + sizeof(struct HeapChunkHeader);
	
	//calc the first chunk pointer
	if (heapIsV3)
		curChunk = (struct HeapChunkHeader*)(heapV3 + 1);
	else if (isOS5)
		curChunk = (struct HeapChunkHeader*)(heapV4_arm + 1);
	else
		curChunk = (struct HeapChunkHeader*)(heapV4_68k + 1);
	
	while (curChunk != NULL) {
		
		uint32_t chunkSize = (readSE32(&curChunk->w0) & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT;
		uint32_t prevBestSize;
		
		if (readSE32(&curChunk->w0) & CHUNK_HDR_W0_FREE_MASK) {
		
			//a free chunk found
			if (atStart) {
				
				if (chunkSize < fullSz) {
					fprintf(stderr, "not enough heap space up front to allocate %u+%u+%u bytes (free chunk is %u bytes)\n",
								(unsigned)sz, (unsigned)extraSz, (unsigned)sizeof(struct HeapChunkHeader), freeChunkSize);
					exit(-3);
				}
				bestChunk = curChunk;
				break;
			}
			
			if (chunkSize >= fullSz) {
			
				if (bestChunk)
					prevBestSize = (readSE32(&bestChunk->w0) & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT;
				
				if (!bestChunk || prevBestSize > chunkSize)
					bestChunk = curChunk;
			}
		}
		
		curChunk = getNextChunk(curChunk);
	}
	
	if (!bestChunk) {
		fprintf(stderr, "not enough heap space to allocate %u+%u+%u bytes\n",
						(unsigned)sz, (unsigned)extraSz, (unsigned)sizeof(struct HeapChunkHeader));
		heapShowFree(heapHdr, isOS5);
		exit(-3);
	}
	
	freeChunkSize = (readSE32(&bestChunk->w0) & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT;
	
	if (atStart) {	//split it and leave free memory at the end
		
		//grab current free chunk as my new chunk
		myChunk = bestChunk;
		
		//calc location of new free chunk
		bestChunk = (struct HeapChunkHeader*)(((uint8_t*)myChunk) + fullSz);
		
		//init new free chunk header
		writeSE32(&bestChunk->w0, (((freeChunkSize - fullSz) << CHUNK_HDR_W0_SIZE_SHIFT) & CHUNK_HDR_W0_SIZE_MASK) | CHUNK_HDR_W0_FREE_MASK);
		writeSE32(&bestChunk->w1, ((CHUNK_OWNER_NOBODY) << CHUNK_HDR_W1_OWNER_SHIFT) & CHUNK_HDR_W1_OWNER_MASK);
	}
	else {			//split it and leave free memory at the front
		
		//split it and get pointer to next chunk. since we cut from end, heap->firstFreeChunkOffset never needs updating
		//as size was a multiple of 8 and we are subtracing another multiple of 8, all is good size-wise
		writeSE32(&bestChunk->w0, (readSE32(&bestChunk->w0) &~ CHUNK_HDR_W0_SIZE_MASK) | (((freeChunkSize - fullSz) << CHUNK_HDR_W0_SIZE_SHIFT) & CHUNK_HDR_W0_SIZE_MASK));
	
		myChunk = (struct HeapChunkHeader*)(((uint8_t*)bestChunk) + freeChunkSize - fullSz);
	}
	
	//fill in our chunk's info
	writeSE32(&myChunk->w0, ((fullSz << CHUNK_HDR_W0_SIZE_SHIFT) & CHUNK_HDR_W0_SIZE_MASK) | ((extraSz << CHUNK_HDR_W0_SIZE_XTRA_SHIFT) & CHUNK_HDR_W0_SIZE_XTRA_MASK));
	writeSE32(&myChunk->w1, ((owner << CHUNK_HDR_W1_OWNER_SHIFT) & CHUNK_HDR_W1_OWNER_MASK) | ((CHUNK_LOCK_CT_NONMOVABLE << CHUNK_HDR_W1_LOCK_CT_SHIFT) & CHUNK_HDR_W1_LOCK_CT_MASK));
	
	return myChunk + 1;
}

static uint32_t calcPtr(const void *dst, const uint8_t *rom, uint32_t targetVa)
{
	return (uintptr_t)dst - (uintptr_t)rom + targetVa;
}

static void *uncalcPtr(uint32_t ptr, const uint8_t *rom, uint32_t targetVa)	//undoes what calcPtr did
{
	return (void*)(ptr + (uintptr_t)rom - targetVa);
}

static bool addResOrRecChunk(uint32_t* dstPtrP, struct Buffer* buf, void *heap, uint8_t *rom, uint32_t targetVa, bool atStartOfHeap, bool isOS5)
{
	void *data;
	
	if (!buf->data) {	//nonexistent buffer (distinct from zero-sized existent one)
		writeSE32(dstPtrP, 0);
		return true;
	}
	
	data = heapAlloc(heap, buf->sz, CHUNK_OWNER_DM_REC_AND_RES, atStartOfHeap, isOS5);
	if (!data)
		return false;
	
	memcpy(data, buf->data, buf->sz);
	
	writeSE32(dstPtrP, calcPtr(data, rom, targetVa));
	return true;
}

static bool setSpecialPtrGuts(uint32_t *dst, uint32_t val, const char *nameForErr, const char *locale /* if set, we'll allow re-setting if resetting to enUS */)
{
	if (readSE32(dst)) {
		if (!locale) {
			fprintf(stderr, "Refusing to set non-zero special value %s 0x%08x -> 0x%08x\n", nameForErr, readSE32(dst), val);
			return false;
		}
		else if (strcmp(locale, "enUS")) {
			fprintf(stderr, "Ignoring request to set non-zero locale-specific special value %s 0x%08x -> 0x%08x with locale '%s'\n", nameForErr, readSE32(dst), val, locale);
			return true;
		}
		else
			fprintf(stderr, "Allowing setting non-zero locale-specific special value %s 0x%08x -> 0x%08x with locale '%s'\n", nameForErr, readSE32(dst), val, locale);
	}
	if (!val) {
		fprintf(stderr, "Setting special value %s to zero\n", nameForErr);
		return false;
	}
	
	fprintf(stderr, "Setting special value %s to 0x%08x\n", nameForErr, val);
	writeSE32(dst, val);
	return true;
}

static bool verifyPtrGuts(uint32_t *valP, const char *nameForErr)
{
	if (readSE32(valP))
		return true;
	
	fprintf(stderr, "Mandatory pointer/offset %s not set!\n", nameForErr);
	return false;
}

#define setSpecialPtr(var, val, locale)		setSpecialPtrGuts(&var, val, #var, locale)
#define verifyPtr(ptr)						verifyPtrGuts(&ptr, #ptr)

static bool checkForSpecialResOS5(const struct MkromConfig *cfg, const char *dbName, uint32_t dbType, uint32_t dbCreator, uint32_t resType, uint16_t resId, uint32_t resAddr, const void* resData, uint32_t targetVa, struct CardHeader *cardHdr, struct StoreHeader* storHdr)
{
	//fprintf(stderr, "DB "F4CC","F4CC" res ("F4CC",%5u)\n", C4CC(dbType), C4CC(dbCreator), C4CC(resType), resId);
	
	if (dbType == MAKE_4CC('r','s','r','c') && dbCreator == MAKE_4CC('p','d','a','l') && resType == MAKE_4CC('b','o','o','t') && resId == 19000) {
		
		uint32_t instr, dalJumpInstr = *(uint32_t*)resData;
		
		if ((dalJumpInstr & 0xFF000000) == 0xEA000000) {
			fprintf(stderr, "DAL is in ARM mode, generating ARM jump instr\n");
		
			instr = 0xEA000000;	//guranteed to be a jump forward, that makes it easy
			instr += (resAddr - targetVa - 8) / 4;
	
			return setSpecialPtr(cardHdr->entry.os5.jumpToCode, instr, NULL);
		}
		else if ((dalJumpInstr & 0xD000F800) == 0x9000F000) {
			fprintf(stderr, "DAL is in Thumb2 mode, generating Thumb addr ptr\n");
		
			instr = resAddr + 1;
			
			return setSpecialPtr(cardHdr->entry.rePalm.initialPc, resAddr + 1, NULL) && setSpecialPtr(cardHdr->entry.rePalm.initialSp, 0xff000000, NULL);	// magic value we DO use to know we just entered the reset handler
		}
		else if ((dalJumpInstr & 0xFFFFF800) == 0x46C0E000) {
			fprintf(stderr, "DAL is in Thumb1 mode, generating Thumb addr ptr\n");
		
			instr = resAddr + 1;
			
			return setSpecialPtr(cardHdr->entry.rePalm.initialPc, resAddr + 1, NULL) && setSpecialPtr(cardHdr->entry.rePalm.initialSp, 0xff000000, NULL);	// magic value we DO use to know we just entered the reset handler
		}
		
		else {
			fprintf (stderr, "DAL format unknown\n");
			return false;
		}
	}
	
	if (dbType == MAKE_4CC('r','s','r','c') && dbCreator == MAKE_4CC('p','d','a','l') && resType == MAKE_4CC('a','m','d','d') && resId == 19000)
		return setSpecialPtr(cardHdr->diff2.os5.dalAmddRsrcOfst, resAddr - targetVa, NULL);
	
	if (dbType == MAKE_4CC('r','s','r','c') && dbCreator == MAKE_4CC('p','d','a','l') && resType == MAKE_4CC('a','m','d','i') && resId == 19000)
		return setSpecialPtr(cardHdr->diff2.os5.dalAmdiRsrcOfst, resAddr - targetVa, NULL);
	
	//Boot.prc
	if (dbType == MAKE_4CC('r','s','r','c') && dbCreator == MAKE_4CC('p','s','y','s') && resType == MAKE_4CC('b','o','o','t')) {
	
		if (resId == 10001)
			return setSpecialPtr(storHdr->bootRsrsBoot10001, resAddr, NULL);
	
		if (resId == 10002)
			return setSpecialPtr(storHdr->bootRsrsBoot10002, resAddr, NULL);
	
		if (cfg->isOS5 && resId == 10003)
			return setSpecialPtr(storHdr->bootBootRsrcPtr, resAddr, NULL);
	}
	
	//boot screens
	if ((dbType == MAKE_4CC('s','p','l','s') || dbType == MAKE_4CC('o','v','l','y')) && dbCreator == MAKE_4CC('p','s','y','s') && resType == MAKE_4CC('a','b','s','b')){
	
		const char *locale = NULL;
	
		if (dbType == MAKE_4CC('o','v','l','y')) {
			if (strlen(dbName) <= 5) {
				fprintf(stderr, "overlay db name too short '%s'\n", dbName);
				return false;
			}
			
			if (dbName[strlen(dbName) - 5] != '_') {
				fprintf(stderr, "overlay db name malformed '%s'\n", dbName);
				return false;
			}
			
			locale = dbName + strlen(dbName) - 4;
		}
		
		if (resId == 19000)
			return setSpecialPtr(storHdr->os5.bootScreen0ptr, resAddr, locale);
	
		if (resId == 19001)
			return setSpecialPtr(storHdr->os5.bootScreen1ptr, resAddr, locale);
	
		if (resId == 19002)
			return setSpecialPtr(storHdr->bootScreen2ptr, resAddr, locale);
	}
	
	return true;
}

static bool checkForSpecialResOS4(const struct MkromConfig *cfg, const char *dbName, uint32_t dbType, uint32_t dbCreator, uint32_t resType, uint16_t resId, uint32_t resAddr, const struct Buffer* resData, uint32_t targetVa, struct CardHeader *cardHdr, struct StoreHeader* storHdr)
{
	//system
	if (dbType == MAKE_4CC('r','s','r','c') && dbCreator == MAKE_4CC('p','s','y','s') && resType == MAKE_4CC('b','o','o','t')) {
		
		switch (resId) {
			
			case 10000:
				if (cfg->fixup68kBootOfst) {
					
					const uint8_t marker[] = {'E', 'N', 'T', 'R', 'Y', 'P', 'O', 'I'};
					const char *start = resData->data;
					const char *at = memmem(start, resData->sz, marker, sizeof(marker));
					uint32_t ofst = resAddr - cfg->baseAddr;		//to start of res. we'll also add offset in res later
					
					
					if (!at || resData->sz - (at - start) < 4) {
						
						fprintf(stderr, "Unable to find fixup point for requested m68k boot fixup\n");
						return false;
					}
					
					//peform fixup
					ofst += at + sizeof(marker) - start;
					writeSE32((uint32_t*)((char*)cardHdr + ofst), ofst);
					fprintf(stderr, "m68k fixup performed. Wrote 0x%08x\n", ofst);
				}
				return setSpecialPtr(cardHdr->entry.os4.initialPc, resAddr, NULL) && setSpecialPtr(cardHdr->entry.os4.initialSp, cfg->initialSp68K, NULL);
			
			case 10001:
				return setSpecialPtr(storHdr->bootRsrsBoot10001, resAddr, NULL);
			
			case 10002:
				return setSpecialPtr(storHdr->bootRsrsBoot10002, resAddr, NULL);

			case 10003:
				return setSpecialPtr(storHdr->bootBootRsrcPtr, resAddr, NULL);
		}
	}
	
	//HAL
	if (dbType == MAKE_4CC('b','h','a','l') && dbCreator == MAKE_4CC('p','s','y','s') && resType == MAKE_4CC('b','o','o','t')) {
	
		switch (resId) {
			
			case 19000:
				return setSpecialPtr(cardHdr->diff2.os4.halCodeOffset, resAddr - targetVa, NULL);
			
			case 19001:
				return setSpecialPtr(cardHdr->diff2.os4.halCode2offset, resAddr - targetVa, NULL);
		}
	}
	
	//boot screens
	if ((dbType == MAKE_4CC('s','p','l','s') || dbType == MAKE_4CC('o','v','l','y')) && dbCreator == MAKE_4CC('p','s','y','s') && resType == MAKE_4CC('T','b','s','b')){
	
		const char *locale = NULL;
	
		if (dbType == MAKE_4CC('o','v','l','y')) {
			if (strlen(dbName) <= 5) {
				fprintf(stderr, "overlay db name too short '%s'\n", dbName);
				return false;
			}
			
			if (dbName[strlen(dbName) - 5] != '_') {
				fprintf(stderr, "overlay db name malformed '%s'\n", dbName);
				return false;
			}
			
			locale = dbName + strlen(dbName) - 4;
		}
		
		if (resId == 19000)
			return setSpecialPtr(storHdr->os4.bootScreen0ptr, resAddr, locale);
	
		if (resId == 19001)
			return setSpecialPtr(storHdr->os4.bootScreen1ptr, resAddr, locale);
	}

	return true;
}

static bool addFile(const struct MkromConfig *cfg, uint32_t *dbHdrAddrP, const char *path, void *heap, uint8_t *rom, uint32_t targetVa, struct CardHeader *cardHdr, struct StoreHeader* storHdr, bool isOS5)
{
	struct DatabaseHeaderOS5 *hdrOS5;
	struct DatabaseHeaderOS4 *hdrOS4;
	struct DatabaseHeaderCommon *hdr;
	struct PalmDb *db;
	uint32_t i;
	
	db = parsePrcPdb(path);
	if (!db) {
		fprintf(stderr, "loading failed\n");
		return false;
	}
	
	hdr = heapAlloc(heap,
		(isOS5 ? sizeof(struct DatabaseHeaderOS5) : sizeof(struct DatabaseHeaderOS4)) +
		db->numChildren * (db->isPrc ? (isOS5 ? sizeof(struct ResourceListEntryOS5) : sizeof(struct ResourceListEntryOS4)) : sizeof(struct RecordListEntry)), CHUNK_OWNER_DM_MGMNT_DATA, false, isOS5);
	if (!hdr)
		return false;
	
	hdrOS5 = (struct DatabaseHeaderOS5*)hdr;
	hdrOS4 = (struct DatabaseHeaderOS4*)hdr;
	
	strncpy(hdr->name, db->name, sizeof(hdr->name));
	writeSE16(&hdr->attributes, db->attributes);
	writeSE16(&hdr->version, db->version);
	writeSE32(&hdr->creationDate, db->creationDate);
	writeSE32(&hdr->modificationDate, db->modificationDate);
	writeSE32(&hdr->lastBackupDate, db->lastBackupDate);
	writeSE32(&hdr->modificationNumber, db->modificationNumber);
	if (!addResOrRecChunk(&hdr->appInfoPtr, &db->appInfo, heap, rom, targetVa, false, isOS5)) {
		fprintf(stderr, "Failed to add AppInfo chunk\n");
		return false;
	}
	if (!addResOrRecChunk(&hdr->sortInfoPtr, &db->sortInfo, heap, rom, targetVa, false, isOS5)) {
		fprintf(stderr, "Failed to add SortInfo chunk\n");
		return false;
	}
	writeSE32(&hdr->type, db->type);
	writeSE32(&hdr->creator, db->creator);
	writeSE32(&hdr->uniqueIDSeed, db->uniqueIDSeed);
	writeSE32(&hdr->nextRecordListPtr, 0);
	writeSE16(&hdr->numChildren, db->numChildren);
	if (isOS5)
		writeSE16(&hdrOS5->val0xD2CE, 0xD2CE);
	
	if (!db->isPrc) {
	
		struct RecordListEntry *rec = (struct RecordListEntry*)(isOS5 ? (void*)(hdrOS5 + 1) : (void*)(hdrOS4 + 1));
		for (i = 0; i < db->numChildren; i++) {
			rec[i].attr = db->rec[i].attrs;
			writeSE24(rec[i].uniqId, db->rec[i].uniq);
			if (!addResOrRecChunk(&rec[i].dataPtr, &db->rec[i].buf, heap, rom, targetVa, false, isOS5)) {
				fprintf(stderr, "Failed to add record chunk\n");
				return false;
			} 
		}
	}
	else {
		
		struct ResourceListEntryOS5 *resOS5 = (struct ResourceListEntryOS5*)(hdrOS5 + 1);
		struct ResourceListEntryOS4 *resOS4 = (struct ResourceListEntryOS4*)(hdrOS4 + 1);
		for (i = 0; i < db->numChildren; i++) {
			bool isMainEntryCode;	//dal for os5, system for os4
			
			if (isOS5) {
			
				writeSE32(&resOS5[i].type, db->res[i].type);
				writeSE16(&resOS5[i].id, db->res[i].id);
				writeSE16(&resOS5[i].zero, 0);
			}
			else {
				
				writeSE32(&resOS4[i].type, db->res[i].type);
				writeSE16(&resOS4[i].id, db->res[i].id);
			}
			
			if (isOS5)
				isMainEntryCode = db->type == MAKE_4CC('r','s','r','c') && db->creator == MAKE_4CC('p','d','a','l') && db->res[i].type == MAKE_4CC('b','o','o','t') && db->res[i].id == 19000;
			else
				isMainEntryCode = db->type == MAKE_4CC('r','s','r','c') && db->creator == MAKE_4CC('p','s','y','s') && db->res[i].type == MAKE_4CC('b','o','o','t') && db->res[i].id == 10000;
			
			//DAL's main code resource gets put at start to give it a predictable address
			if (!addResOrRecChunk(isOS5 ? &resOS5[i].dataPtr : &resOS4[i].dataPtr, &db->res[i].buf, heap, rom, targetVa, isMainEntryCode, isOS5)) {
				fprintf(stderr, "Failed to add resource chunk\n");
				return false;
			} 
			
			if (isMainEntryCode)
				fprintf(stderr, "First entry code base address is 0x%08x\n", readSE32(isOS5 ? &resOS5[i].dataPtr : &resOS4[i].dataPtr));
			
			if (isOS5) {
				
				if (!checkForSpecialResOS5(cfg, db->name, db->type, db->creator, db->res[i].type, db->res[i].id, readSE32(&resOS5[i].dataPtr), db->res[i].buf.data, targetVa, cardHdr, storHdr)) {
					fprintf(stderr, "Failed to check for speciallness of resource\n");
					return false;
				} 
			 }
			 else { 
			 	
			 	if (!checkForSpecialResOS4(cfg, db->name, db->type, db->creator, db->res[i].type, db->res[i].id, readSE32(&resOS4[i].dataPtr), &db->res[i].buf, targetVa, cardHdr, storHdr)) {
					fprintf(stderr, "Failed to check for speciallness of resource\n");
					return false;
				}  
			 }
		}
	}
	
	freePrcPdb(db);
	writeSE32(dbHdrAddrP, calcPtr(hdr, rom, targetVa));
	return true;
}

static void sortDbList(const struct MkromConfig *cfg, uint32_t *dbHdrAddrs, uint32_t numDbs, uint8_t *rom)	//PalmOS assumes databases sorted in ascending order by type (in case of tie then createor (in case of tie then version))
{
	//selection sort
	uint32_t i, j, k, t, targetVa = cfg->baseAddr;
	
	for (i = 0; i < numDbs; i++) {
		k = i;
		for (j = i + 1; j < numDbs; j++) {
			struct DatabaseHeaderCommon *hdrK = uncalcPtr(readSE32(&dbHdrAddrs[k]), rom, targetVa);
			struct DatabaseHeaderCommon *hdrJ = uncalcPtr(readSE32(&dbHdrAddrs[j]), rom, targetVa);
			
			if (readSE32(&hdrJ->type) < readSE32(&hdrK->type)) {
				k = j;
				continue;
			}
			if (readSE32(&hdrJ->type) > readSE32(&hdrK->type))
				continue;
			
			if (readSE32(&hdrJ->creator) < readSE32(&hdrK->creator)) {
				k = j;
				continue;
			}
			if (readSE32(&hdrJ->creator) > readSE32(&hdrK->creator))
				continue;
			
			if (readSE16(&hdrJ->version) < readSE16(&hdrK->version)) {
				k = j;
				continue;
			}
			if (readSE16(&hdrJ->version) > readSE16(&hdrK->version))
				continue;
		}
		t = readSE32(&dbHdrAddrs[k]);
		writeSE32(&dbHdrAddrs[k], readSE32(&dbHdrAddrs[i]));
		writeSE32(&dbHdrAddrs[i], t);
	}		
}


static bool parseLongVal(const char *str, uint32_t *valP, const char *name)
{
	unsigned long long ret;
	
	errno = 0;
	ret = strtoull(str, NULL, 0);
	if (!errno && ret >=0 && ret <= 0xffffffffull) {
		*valP = ret;
		return true;
	}
	
	fprintf(stderr, "ERROR: '%s' could not be interpreted as a valid %s\n", str, name);
	return false;
}

static bool parseU32(const char *str, uint32_t *valP)
{
	return !!parseLongVal(str, valP, "32-bit value");
}

static bool parseAddress(const char *str, uint32_t *valP)
{
	return !!parseLongVal(str, valP, "address");
}

static bool parseU16(const char *str, uint16_t *valP)
{
	unsigned long long ret;
	
	errno = 0;
	ret = strtoull(str, NULL, 0);
	if (!errno && ret >=0 && ret <= 0xffffull) {
		*valP = ret;
		return true;
	}
	
	fprintf(stderr, "ERROR: '%s' could not be interpreted as a valid 16-bit value\n", str);
	return false;
}

static bool parseFourCC(const char *str, uint32_t *valP)
{
	unsigned long long ret;
	int i;
	
	if (strlen(str) == 4 && str[0] >= 0x20 && str[0] < 0x80 && str[1] >= 0x20 && str[1] < 0x80 && str[2] >= 0x20 && str[2] < 0x80 && str[3] >= 0x20 && str[3] < 0x80) {
		
		for (ret = 0, i = 0; i < 4; i++)
			ret = (ret << 8) + (uint32_t)(uint8_t)str[i];
		
		*valP = ret;
		return true;
	}
	
	errno = 0;
	ret = strtoull(str, NULL, 0);
	if (!errno && ret >=0 && ret <= 0xffffull) {
		*valP = ret;
		return true;
	}
	
	fprintf(stderr, "ERROR: '%s' could not be interpreted as a valid 32-bit value\n", str);
	return false;
}

static bool parseToken(const char *str, struct MkromConfig *cfg)
{
	struct Token *t;
	int i, j, L;
	
	L = strlen(str);
	for (i = 0; i < 4 && i < L; i++) {
		if (str[i] < 0x20 || str[i] >= 0x80)
			break;
	}
	
	for (j = i; j < L; j++) {
		if (!isxdigit(str[j]))
			break;
	}
	
	if (i != 4 || j != L || (L %2)) {
		fprintf(stderr, "ERROR: '%s' could not be interpreted as a valid token descriptor\n", str);
		return false;
	}
	L -= i;	//calc how big the token is
	L /= 2;
	
	t = malloc(sizeof(struct Token) + L);
	t->name = (((uint32_t)(uint8_t)str[0]) << 24) + (((uint32_t)(uint8_t)str[1]) << 16) + (((uint32_t)(uint8_t)str[2]) << 8) + ((uint32_t)(uint8_t)str[3]);
	t->len = L;
	
	for (j = 0; j < L; j++) {
		
		char chH = str[i + j * 2 + 0];
		char chL = str[i + j * 2 + 1];
		
		if (chH >= 'a')
			chH += 10 - 'a';
		else if (chH >= 'A')
			chH += 10 - 'A';
		else
			chH -= '0';
		if (chL >= 'a')
			chL += 10 - 'a';
		else if (chL >= 'A')
			chL += 10 - 'A';
		else
			chL -= '0';
		
		t->data[j] = (chH << 4) + chL;
	}
	
	t->next = cfg->tokens;
	cfg->tokens = t;
	
	return true;
}

static bool parseOpts(struct MkromConfig *cfg, int argc, char **argv)
{
	static const struct option opts[] = {
		{"base", required_argument, NULL, 'b', },
		{"size", required_argument, NULL, 's', },
		{"fake-entry", required_argument, NULL, 'f', },
		{"tokens-addr", required_argument, NULL, 't', },
		{"manufacturer", required_argument, NULL, 'm', },
		{"hal-id", required_argument, NULL, 'h', },
		{"lang", required_argument, NULL, 'L', },
		{"country", required_argument, NULL, 'C', },
		{"card-ver", required_argument, NULL, 'V', },
		{"token", required_argument, NULL, '_', },
		{"OS4", no_argument, NULL, '4', },
		{"unsafe", no_argument, NULL, 'u', },
		{"verbose", no_argument , NULL, 'v', },
		{"no-ram-blocks", no_argument, NULL, 'z', },
		
		{"card-name", required_argument, NULL, 1, },
		{"card-manuf", required_argument, NULL, 2, },
		{"m68k-init-sp", required_argument, NULL, 3, },
		{"m68k-entrypt-fixup", no_argument, NULL, 4, },
		{"card-version-str", required_argument, NULL, 5, },
		{"card-os-claimed-ver", required_argument, NULL, 6, },
		{"extra-heap-ofst", required_argument, NULL, 7, },
		
		{0,},
	};
	
	bool haveTokensAddr = false, haveClaimedCardOsVerVal = false;
	struct Token *t, *n;
	int choice;
	
	cfg->baseAddr = ROM_BASE_ADDR_DEFAULT;
	cfg->romSz = ROM_SIZE_DEFAULT;
	cfg->manuf = ROM_MANUF_ID_DEFAULT;
	cfg->halId = HAL_ID_DEFAULT;
	cfg->lang = LANG_DEFAULT;
	cfg->country = COUNTRY_DEFAULT;
	cfg->haveFakeEntryAddr = false;
	cfg->unsafe = false;
	cfg->verbose = false;
	cfg->hasRamBlocks = true;
	cfg->cardHdrVersion = 6;
	cfg->isOS5 = true;
	strcpy(cfg->cardName, "rePalm");
	strcpy(cfg->cardManuf, "Dmitry.GR");
	strcpy(cfg->cardVersionString, "Pre-Alpha 0.0.0");
	cfg->initialSp68K = 0x3000;
	cfg->fixup68kBootOfst = false;
	cfg->extraHeapOffset = 0;
	
	while ((choice = getopt_long(argc, argv, "b:s:f:t:m:h:L:C:V:uv4", opts, NULL)) != -1) switch(choice) {
		
		case 1:
		case 2:
		case 5:
			if (!strlen(optarg) || strlen(optarg) > 31)
				return false;
			switch (choice) {
				case 1:	strcpy(cfg->cardName, optarg);	break;
				case 2:	strcpy(cfg->cardManuf, optarg);	break;
				case 5:	strcpy(cfg->cardVersionString, optarg);	break;
			}
		break;
		
		case 3:
			if (!parseAddress(optarg, &cfg->initialSp68K))
				return false;
			break;
		
		case 4:
			cfg->fixup68kBootOfst = true;
			break;
		
		case 6:
			if (!parseU32(optarg, &cfg->cardClaimedOsVerVal))
				return false;
			haveClaimedCardOsVerVal = true;
			break;
		
		case 7:
			if (!parseU32(optarg, &cfg->extraHeapOffset))
				return false;
			break;
		
		case 'b':
			if (!parseAddress(optarg, &cfg->baseAddr))
				return false;
			break;
		
		case 's':
			if (!parseAddress(optarg, &cfg->romSz))
				return false;
			break;
		
		case 'f':
			if (!parseAddress(optarg, &cfg->fakeEntryAddr))
				return false;
			cfg->haveFakeEntryAddr = true;
			break;
		
		case 't':
			if (!parseAddress(optarg, &cfg->tokensAddr))
				return false;
			haveTokensAddr = true;
			break;
		
		case 'm':
			if (!parseFourCC(optarg, &cfg->manuf))
				return false;
			break;
		
		case 'h':
			if (!parseFourCC(optarg, &cfg->halId))
				return false;
			break;
		
		case 'L':
			if (!parseU16(optarg, &cfg->lang))
				return false;
			break;
		
		case 'C':
			if (!parseU16(optarg, &cfg->country))
				return false;
			break;
		
		case 'V':
			if (!parseU16(optarg, &cfg->cardHdrVersion))
				return false;
			break;
		
		case '_':	//token
			if (!parseToken(optarg, cfg))
				return false;
			break;
		
		case '4':
			cfg->isOS5 = false;
			if (cfg->cardHdrVersion == 6)
				cfg->cardHdrVersion = 5;
			break;
		
		case 'u':
			cfg->unsafe = true;
			break;
		
		case 'z':
			cfg->hasRamBlocks = false;
			break;
		
		case 'v':
			cfg->verbose = true;
			break;
		
		default:
			fprintf(stderr, "not sure how to cope with getopt returning %d\n", choice);
			return false;
	}
	cfg->self = argv[0];
	
	if (optind >= argc) {
		fprintf(stderr, "no files given to place into the rom\n");
		if (!cfg->unsafe)
			return false;
	}
	
	cfg->files = argv + optind;
	cfg->numFiles = argc - optind;
	
	//tokens list is currently in reverse order of given order. we shodul honor the request the user made and reverse it back to proper order
	t = cfg->tokens;
	n = NULL;
	while (t) {
		
		struct Token *q = t->next;
		
		t->next = n;
		n = t;
		t = q;
	}
	cfg->tokens = n;
	
	//if we have no tokens addr, calc it
	if(!haveTokensAddr)
		cfg->tokensAddr = cfg->baseAddr + ROM_TOKEN_STORE_OFST;
	
	//card version
	if (cfg->cardHdrVersion > 6) {
		fprintf(stderr, "card headers over 6 are unsupported\n");
		if (!cfg->unsafe)
			return false;
	}
	else if (cfg->isOS5 && cfg->cardHdrVersion != 6) {
		fprintf(stderr, "card header versions other than v6 not supported for OS5\n");
		if (!cfg->unsafe)
			return false;
	}
	else if (!cfg->isOS5 && cfg->cardHdrVersion == 6) {
		fprintf(stderr, "card header version v6 not supported for 68k OSs\n");
		if (!cfg->unsafe)
			return false;
	}
	
	if (!haveClaimedCardOsVerVal)
		cfg->cardClaimedOsVerVal = cfg->isOS5 ? 0x05030000 : 0x04010000;
	
	return true;
}

static uint16_t crc16(const void *srcP, uint32_t len, uint16_t crc)
{
	static const uint16_t crc16tab[] = {
		0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7,
		0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF,
		0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6,
		0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE,
		0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485,
		0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D,
		0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4,
		0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC,
		0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823,
		0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B,
		0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12,
		0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A,
		0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41,
		0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49,
		0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70,
		0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78,
		0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F,
		0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067,
		0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E,
		0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256,
		0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D,
		0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
		0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C,
		0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634,
		0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB,
		0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3,
		0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A,
		0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92,
		0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9,
		0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1,
		0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8,
		0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0,	
	};
	const uint8_t *src = (const uint8_t*)srcP;
	
	while(len--)
		crc = (crc << 8) ^ crc16tab[(crc >> 8) ^ *src++];
	
	return crc;
}

int main(int argc, char **argv)
{
	static uint8_t mRom[ROM_SIZE_MAX] = {0,};
	struct DatabaseListOS5 *dbList5;
	struct DatabaseListOS4 *dbList4;
	struct HeapHeaderV4 *heapOS5;
	struct HeapHeaderV3 *heapOS4;
	struct StoreHeader* storHdr;
	struct MkromConfig cfg = {};
	struct CardHeader *cardHdr;
	bool checksFailed = false;
	uint32_t *dbHdrs;
	void *dbList;
	uint32_t i;
	void* heap;
	
	

	if (!parseOpts(&cfg, argc, argv)) {
		fprintf(stderr,
			"USAGE: %s [opts] DAL.prc <other files> > rom.bin\n"
			"\tOTIONS:\n"
			"\t  --base          -b 0x????????    set base address, defaults to 0x%08lx\n"
			"\t  --size          -s 0x????????    set rom size, defaults to 0x%08lx\n"
			"\t  --fake-entry    -f 0x????????    instead of jumping to DAL, jump here\n"
			"\t  --tokens-addr   -t 0x????????    point tokens pointer at this address\n"
			"\t  --manufacturer  -m ABCD          set manufacturer ID, defaults to "F4CC"\n"
			"\t  --hal-id        -h ABCD          set HAL ID, defaults to "F4CC"\n"
			"\t  --lang          -L 123           set Language (see SDK), defaults to %u\n"
			"\t  --country       -C 123           set Country (see SDK), defaults to %u\n"
			"\t  --card-ver      -V 123           set card header version, defaults to 6 for ARM, 5 for 68k\n"
			"\t  --OS4           -4               make an OS4 ROM (defaults to no)\n"
			"\t  --token=NaMe00223344aacc         insert a given token. can be given more than once\n"
			"\t  --no-ram-blocks                  mark heap as having no ram\n"
			"\t  --card-name=PalmCard             set card name, defaults to \"rePalm\"\n"
			"\t  --card-manuf=SonyCorp            set card manufacturer, defaults to \"Dmitry.GR\"\n"
			"\t  --card-version-str=alpha5        set card version string, defaults to \"Pre-Alpha 0.0.0\"\n"
			"\t  --card-os-claimed-ver=0x05020800 set card claimed OS version value, defaults is 5.3.0.0 or 4.1.0.0\n"
			"\t  --extra-heap-ofst=0x????         offset heap from card start by this many more bytes for code alignment\n"
			"\t  --m68k-init-sp=0x????????        set initial SP for 68k images\n"
			"\t  --m68k-entrypt-fixup             perform the fixup in the binary needed for m68k images to actually boot\n"
			"\t  --unsafe        -u               make safety error merly warnings\n"
			"\t  --verbose       -v               verbosely print what we do\n",
			argv[0], (unsigned long)ROM_BASE_ADDR_DEFAULT, (unsigned long)ROM_SIZE_DEFAULT, C4CC(ROM_MANUF_ID_DEFAULT),
			C4CC(HAL_ID_DEFAULT), (unsigned)LANG_DEFAULT, (unsigned)COUNTRY_DEFAULT);
		
		return -1;
	}
	
	seSetLittleEndian(cfg.isOS5);
	
	if (cfg.isOS5) {
		heap = heapOS5 = cardStoreAndHeapFormat(mRom, &cfg, cfg.romSz, &cardHdr, &storHdr);
		if (!heapOS5) {
			fprintf(stderr, "Failed to format store and heap!\n");
			return -1;
		}
		
		dbList = dbList5 = (struct DatabaseListOS5*)heapAlloc(heapOS5, sizeof(struct DatabaseListOS5) + sizeof(uint32_t) * cfg.numFiles, CHUNK_OWNER_DM_MGMNT_DATA, false, cfg.isOS5);
		if (!dbList5) {
			fprintf(stderr, "Failed to alloc db list\n");
			return -1;
		}
		writeSE32(&dbList5->nextDatabaseDir, 0);
		writeSE32(&dbList5->numDatabases, cfg.numFiles);
		dbHdrs = dbList5->databaseHdr;
	}
	else {
		heap = heapOS4 = cardStoreAndHeapFormat(mRom, &cfg, cfg.romSz, &cardHdr, &storHdr);
		dbList = dbList4 = (struct DatabaseListOS4*)heapAlloc(heapOS4, sizeof(struct DatabaseListOS4) + sizeof(uint32_t) * cfg.numFiles, CHUNK_OWNER_DM_MGMNT_DATA, false, cfg.isOS5);
		if (!dbList4) {
			fprintf(stderr, "Failed to alloc db list\n");
			return -1;
		}
		writeSE32(&dbList4->nextDatabaseDir, 0);
		writeSE16(&dbList4->numDatabases, cfg.numFiles);
		dbHdrs = dbList4->databaseHdr;
	}
	
	writeSE32(&storHdr->databaseListPtr, calcPtr(dbList, mRom, cfg.baseAddr));
	
	for (i = 0; i < cfg.numFiles; i++) {
		
		fprintf(stderr, "Adding file '%s'...\n", cfg.files[i]);
		if (!addFile(&cfg, &dbHdrs[i], cfg.files[i], cfg.isOS5 ? (void*)heapOS5 : (void*)heapOS4, mRom, cfg.baseAddr, cardHdr, storHdr, cfg.isOS5))
			return -1;
	}
	
	//now sort database list
	sortDbList(&cfg, dbHdrs, cfg.numFiles, mRom);
	
	//verify all pointers are set
	switch (cfg.cardHdrVersion) {
		default:
			fprintf(stderr, "card version %u not known\n", cfg.cardHdrVersion);
			return -1;
				
		case 6:
		case 5:
			if (cfg.isOS5) {
				
				if (!verifyPtr(cardHdr->diff2.os5.dalAmdiRsrcOfst))
					checksFailed = true;
			}
			else {
				
				if (!verifyPtr(cardHdr->diff2.os4.halCode2offset))
					checksFailed = true;
			}
			//fallthrough
		
		case 4:
			if (cfg.isOS5) {
				
				if (!verifyPtr(cardHdr->diff2.os5.dalAmddRsrcOfst))
					checksFailed = true;
			}
			else {
				
				if (!verifyPtr(cardHdr->diff2.os4.halCodeOffset))
					checksFailed = true;
			}
			//fallthrough
		
		case 3:
		case 2:
			if (!verifyPtr(cardHdr->romTokenStorePtr))
				checksFailed = true;
			if (!verifyPtr(cardHdr->bigRomPtr))
				checksFailed = true;
			//fallthrough
		
		case 1:
			if (cfg.isOS5) {
			
				if (!verifyPtr(cardHdr->entry.os5.jumpToCode))
					checksFailed = true;
				if (!verifyPtr(storHdr->os5.bootScreen0ptr))
					checksFailed = true;
				if (!verifyPtr(storHdr->os5.bootScreen1ptr))
					checksFailed = true;
			}
			else {
				
				if (!verifyPtr(cardHdr->entry.os4.initialPc))
					checksFailed = true;
				if (!verifyPtr(cardHdr->entry.os4.initialSp))
					checksFailed = true;
				if (!verifyPtr(storHdr->os4.bootScreen0ptr))
					checksFailed = true;
				if (!verifyPtr(storHdr->os4.bootScreen1ptr))
					checksFailed = true;
			}
			if (!verifyPtr(cardHdr->blockListPtr))
				checksFailed = true;
			if (!verifyPtr(storHdr->bootRsrsBoot10001))
				checksFailed = true;
			if (!verifyPtr(storHdr->bootRsrsBoot10002))
				checksFailed = true;
			if (!verifyPtr(storHdr->databaseListPtr))
				checksFailed = true;
			if (!verifyPtr(storHdr->databaseListPtr))
				checksFailed = true;
			if (!verifyPtr(storHdr->databaseListPtr))
				checksFailed = true;
			if (cfg.isOS5) {
				if (!verifyPtr(storHdr->bootScreen2ptr))
					checksFailed = true;
				if (!verifyPtr(storHdr->bootBootRsrcPtr))
					checksFailed = true;
			}
			break;
	}
	if (checksFailed && !cfg.unsafe)
		return -2;
	
	if (cfg.haveFakeEntryAddr)
		writeSE32(&cardHdr->entry.rePalm.initialPc, cfg.fakeEntryAddr);
	
	//finalize the heap
	heapFinalize(heap, cfg.isOS5);
	
	//for os4, set rom checksum
	if (!cfg.isOS5) {
		
		uint16_t crc = 0;
		
		crc = crc16(cardHdr, offsetof(struct CardHeader, diff2.os4.romChecksum), crc);
		crc = crc16((&cardHdr->diff2.os4.romChecksum) + 1, readSE32(&cardHdr->totalRomSz) - offsetof(struct CardHeader, diff2.os4.romChecksum) - sizeof(cardHdr->diff2.os4.romChecksum), crc);
		writeSE16(&cardHdr->diff2.os4.romChecksum, crc);
	}
	
	fprintf(stderr, "ROM complete. writing\n");
	fprintf(stderr, "Wrote %u bytes\n", (unsigned)fwrite(mRom, 1, cfg.romSz, stdout));
	return 0;
}
