
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Prog. VOCInfo  : Zeigt detailierte Informationen ber VOC-Dateien (Datei- }
{                  version, Sample-Rate u.a.) und deren Datenblcke auf dem }
{                  Bildschirm an.                                           }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : VOCINFO.PAS                                              }
{ entwickelt am  : 09.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.05                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

PROGRAM VOCInfo;

{$D-}                                        { keine Debugger-Informationen }
{$F-}                                        { FAR-Aufrufe sind nicht ntig }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

USES CRT, DOS;                                { CRT- und DOS-Unit einbinden }

TYPE

  IDString = ARRAY[0..$13] OF Char;                { Identifikations-String }

  {=========================================================================}
  { TVOCRec: Struktur des VOC-Header                                        }
  {=========================================================================}

  TVOCRec = RECORD
    IDStr                : IDString;                        { Datei-Kennung }
    DataOffs, Ver, Check : Word;   { Datenblock-Offset, Version, Check-Code }
  END;

VAR
  VOCName : String;                                    { Name der VOC-Datei }
  VOCFile : File;                                       { VOC-Dateivariable }

{===========================================================================}
{ Prozedur ShowHelp: Hilfe zum Programm auf dem Bildschirm anzeigen.        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ShowHelp;

BEGIN
  WriteLn('Zeigt Informationen ber die angegebene VOC-Datei an.');
  WriteLn;
  WriteLn('Syntax: VOCINFO [Dateiname].VOC');
END;

{===========================================================================}
{ Funktion UpperString: Wandelt einen String beliebiger Lnge in Gro-      }
{                       schreibung um und liefert ihn an den Aufrufer der   }
{                       Funktion zurck.                                    }
{===========================================================================}
{ Eingabe: S = String, der in Groschreibung umgewandelt werden soll        }
{ Ausgabe: String in Groschreibung                                         }
{---------------------------------------------------------------------------}

FUNCTION UpperString(S : String) : String;

VAR
  Count : Word;                                                { ein Zhler }
  Upper : String;                  { in Groschreibung umgewandelter String }

BEGIN
  UpperString := '';
  IF S <> '' THEN                               { Ist S kein leerer String? }
    BEGIN                                          { nein, S ist nicht leer }
      Upper := '';
      FOR Count := 1 TO Length(S) DO                     { String umwandeln }
        Upper := Upper+UpCase(S[Count]);
      UpperString := Upper;                      { neuen String zurckgeben }
    END;
END;

{===========================================================================}
{ Prozedur Error: Zeigt einen Fehler an und beendet das Programm mit einem  }
{                 Halt-Befehl.                                              }
{===========================================================================}
{ Eingabe: Text = Fehlermeldung                                             }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Error(Text : String);

BEGIN
  WriteLn;
  WriteLn(Text);                                   { Fehlermeldung ausgeben }
  Halt;                                                  { Programm beenden }
END;

{===========================================================================}
{ Prozedur ShowInfo: Zeigt Informationen ber eine VOC-Datei auf dem Bild-  }
{                    schirm an.                                             }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ShowInfo;

VAR
  Result     : Integer;                             { Fehlerstatus-Variable }
  Header     : TVOCRec;                                        { VOC-Header }
  IDStr      : String[20];                         { Identifikations-String }
  Count      : Word;                                           { ein Zhler }
  BlockType  : Byte;                                             { Blocktyp }
  BlockSize  : LongInt;                           { Gre eines Datenblocks }
  BlockNum   : Word;                             { Nummer eines Datenblocks }
  Key        : Char;                           { speichert einen Tastencode }
  OldPos     : Word;              { speichert die Position in der VOC-Datei }

  {=========================================================================}
  { Prozedur ReadError: Wird beim Auftreten eines Lesefehlers aufgerufen.   }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadError;

  BEGIN
    Close(VOCFile);                                       { Datei schlieen }
    Error('Fehler: Lesefehler');                   { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur SeekError: Wird beim Auftreten eines Positionierungsfehlers    }
  {                     aufgerufen.                                         }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE SeekError;

  BEGIN
    Close(VOCFile);                                       { Datei schlieen }
    Error('Fehler: Positionierungsfehler');        { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur ReadBlockData: Liest die Daten eines Blocks ein. Die Variablen }
  {                         BlockType und BlockSize werden entsprechend ge- }
  {                         setzt.                                          }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadBlockData;

  VAR
    W : Word;
    B : Byte;

  BEGIN
    BlockRead(VOCFile, BlockType, SizeOf(Byte));           { Blocktyp lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;

    { Blockgre lesen }

    BlockRead(VOCFile, W, SizeOf(Word));
    BlockRead(VOCFile, B, SizeOf(Byte));

    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;

    { Blockgre aus B und W ermitteln (24 Bit lang) }

    BlockSize := B;
    BlockSize := BlockSize*65536+W;
  END;

  {=========================================================================}
  { Funktion ReadSampleRate1: Liest die Sample-Rate aus einem Datenblock    }
  {                           und liefert diese in Hz umgerechnet zurck.   }
  {                           Die Funktion wird nur bei Blocktypen "Neue    }
  {                           Sample-Daten" und "Stille" aufgerufen.        }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Sample-Rate in Hz                                              }
  {-------------------------------------------------------------------------}

  FUNCTION ReadSampleRate1 : Word;

  VAR
    TC : Byte;        { Sample-Rate in komprimierter Form - "time constant" }

  BEGIN
    ReadSampleRate1 := 0;
    BlockRead(VOCFile, TC, SizeOf(Byte));            { Sample-Rate auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;

    { Berechnung der eigentlichen Sample-Rate }

    IF TC <> 0 THEN
      ReadSampleRate1 := 1000000 DIV (256-TC);
  END;

  {=========================================================================}
  { Funktion ReadSampleRate2: Liest die Sample-Rate aus einem Datenblock    }
  {                           und liefert diese in Hz zurck. Bei Stereo    }
  {                           verdoppelt sich die Sample-Rate. Die Funktion }
  {                           wird nur bei dem Blocktyp "Zustzliche Daten" }
  {                           aufgerufen.                                   }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Sample-Rate in Hz                                              }
  {-------------------------------------------------------------------------}

  FUNCTION ReadSampleRate2 : Word;

  VAR
    TC : Word;        { Sample-Rate in komprimierter Form - "time constant" }

  BEGIN
    ReadSampleRate2 := 0;
    BlockRead(VOCFile, TC, SizeOf(Word));            { Sample-Rate auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;

    { Berechnung der eigentlichen Sample-Rate }

    IF TC <> 0 THEN
      ReadSampleRate2 := 256000000 DIV (65536-TC);
  END;

  {=========================================================================}
  { Funktion ReadSampleRate3: Liest die Sample-Rate aus einem Datenblock    }
  {                           und gibt diese zurck. Die Funktion wird nur  }
  {                           bei dem Blocktyp "Neue Sample-Daten (neues    }
  {                           VOC-Format)" aufgerufen.                      }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Sample-Rate in Hz                                              }
  {-------------------------------------------------------------------------}

  FUNCTION ReadSampleRate3 : Word;

  VAR
    TC : Word;        { Sample-Rate in komprimierter Form - "time constant" }

  BEGIN
    BlockRead(VOCFile, TC, SizeOf(Word));            { Sample-Rate auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadSampleRate3 := TC;
  END;

  {=========================================================================}
  { Funktion ReadPackMethod: Liest das Komprimierungsverfahren aus einem    }
  {                          Datenblock und liefert es zurck.              }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Komprimierungsverfahren                                        }
  {          0: 8-Bits komprimierte Samples                                 }
  {          1: 4-Bits-Komprimierung                                        }
  {          2: 2,6-Bits-Komprimierung                                      }
  {          3: 2-Bits-Komprimierung                                        }
  {-------------------------------------------------------------------------}

  FUNCTION ReadPackMethod : Byte;

  VAR
    PackMethod : Byte;                { speichert die Art der Komprimierung }

  BEGIN
    BlockRead(VOCFile, PackMethod, SizeOf(Byte));  { Komprimierung auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadPackMethod := PackMethod;
  END;

  {=========================================================================}
  { Funktion ReadMode: Liest den Modus, in dem ein Sample aufgenommen wur-  }
  {                    de, aus einem Datenblock und liefert diesen zurck   }
  {                    (gemeint ist Mono bzw. Stereo).                      }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Modus                                                          }
  {          0: Mono (bei 16-Bits-Samples der Wert 1)                       }
  {          1: Stereo (bei 16-Bits-Samples der Wert 2)                     }
  {-------------------------------------------------------------------------}

  FUNCTION ReadMode : Byte;

  VAR
    Mode : Byte;                                      { speichert den Modus }

  BEGIN
    BlockRead(VOCFile, Mode, SizeOf(Byte));                   { Modus lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadMode := Mode;
  END;

  {=========================================================================}
  { Funktion ReadBits: Liest die Anzahl der Bits pro Sample und liefert     }
  {                    diesen Wert zurck.                                  }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Anzahl der Bits                                                }
  {-------------------------------------------------------------------------}

  FUNCTION ReadBits : Byte;

  VAR
    Bits : Byte;                  { speichert die Zahl der Bits (8 oder 16) }

  BEGIN
    BlockRead(VOCFile, Bits, SizeOf(Byte));         { Anzahl der Bits lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadBits := Bits;
  END;

  {=========================================================================}
  { Funktion ReadPeriod: Liefert die Lnge der Stille in einem Block vom    }
  {                      Typ "Stille" als Sample-Rate-Einheit zurck.       }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Lnge der Ruhe-Periode                                         }
  {-------------------------------------------------------------------------}

  FUNCTION ReadPeriod : Word;

  VAR
    Period : Word;                   { speichert die Lnge der Ruhe-Periode }

  BEGIN
    BlockRead(VOCFile, Period, SizeOf(Word));   { Lnge der Stille auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadPeriod := Period+1;
  END;

  {=========================================================================}
  { Funktion ReadMarker: Liest den Wert einer Markierung aus einem Daten-   }
  {                      block vom Typ "Marker" und liefert diesen Wert zu- }
  {                      rck.                                              }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Wert einer Markierung                                          }
  {-------------------------------------------------------------------------}

  FUNCTION ReadMarker : Word;

  VAR
    Marker : Word;                      { speichert den Wert der Markierung }

  BEGIN
    BlockRead(VOCFile, Marker, SizeOf(Word));               { Wert auslesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadMarker := Marker;
  END;

  {=========================================================================}
  { Funktion ReadText: Liest einen Text aus einem Datenblock aus und lie-   }
  {                    fert diesen zurck.                                  }
  {=========================================================================}
  { Eingabe: Len = Lnge des zu lesenden Textes einschlielich einer Null   }
  { Ausgabe: Text                                                           }
  {-------------------------------------------------------------------------}

  FUNCTION ReadText(Len : Byte) : String;

  VAR
    S : String;                                                  { der Text }

  BEGIN
    S[0] := Chr(Len-1);
    BlockRead(VOCFile, S[1], Len-1);                           { Text lesen }
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadText := S;
  END;

  {=========================================================================}
  { Funktion ReadRepeats: Liest die Anzahl der Wiederholungen aus einem Da- }
  {                       tenblock vom Typ "Start der Wiederholungen" und   }
  {                       liefert diesen Wert zurck.                       }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: Anzahl der Wiederholungen                                      }
  {-------------------------------------------------------------------------}

  FUNCTION ReadRepeats : Word;

  VAR
    Repeats : Word;               { speichert die Anzahl der Wiederholungen }

  BEGIN
    BlockRead(VOCFile, Repeats, SizeOf(Word));
    Result := IOResult;
    IF Result <> 0 THEN
      ReadError;
    ReadRepeats := Repeats;
  END;

{ ShowInfo }

VAR
  Rate       : Word;                                          { Sample-Rate }
  PackMethod : Byte;                                { Komprimierungsmethode }
  Mode       : Byte;                             { Modus (Mono oder Stereo) }
  Period     : Word;                               { Lnge der Ruhe-Periode }
  Repeats    : Word;        { Anzahl der Wiederholungen eines Sample-Blocks }

BEGIN
  WriteLn('Datei-Info ber '+VOCName);
  BlockRead(VOCFile, Header, SizeOf(TVOCRec));               { Header lesen }
  Result := IOResult;                              { Fehlerstatus speichern }
  IF Result <> 0 THEN                                 { Fehler aufgetreten? }
    ReadError;                                             { ja, Lesefehler }
  IDStr := '';
  FOR Count := 0 TO $13 DO
    IDStr := IDstr+UpCase(Header.IDStr[Count]);
  IF (IDStr <> 'CREATIVE VOICE FILE'+#$1A)                  { Datei-Kennung }
  OR ($1233-Header.Ver <> Header.Check) THEN      { oder Check-Code falsch? }
    BEGIN                                                              { ja }
      Close(VOCFile);                                     { Datei schlieen }
      Error('Fehler: Datei nicht im VOC-Format');           { Fehlermeldung }
    END;
  WriteLn;
  WriteLn('Datenblock-Offset : ', Header.DataOffs);
  WriteLn('Dateiversion      : ', Hi(Header.Ver), '.', Lo(Header.Ver));
  WriteLn('Check-Code        : ', Header.Check);
  Seek(VOCFile, Header.DataOffs);            { Sprung zum ersten Datenblock }
  Result := IOResult;
  IF Result <> 0 THEN
    SeekError;

  { Daten aller Blcke einlesen und anzeigen }

  BlockNum := 1;
  REPEAT
    ReadBlockData;
    OldPos := FilePos(VOCFile);
    WriteLn;
    WriteLn('Info ber den Datenblock Nr. ', BlockNum);
    WriteLn(''+
      '');
    WriteLn;
    Write('Blocktyp   : ');
    CASE BlockType OF
      0 : WriteLn('Blockende');
      1 : WriteLn('Neue Sample-Daten');
      2 : WriteLn('Sample-Daten');
      3 : WriteLn('Stille');
      4 : WriteLn('Marker');
      5 : WriteLn('ASCII-Text');
      6 : WriteLn('Start der Wiederholungen');
      7 : WriteLn('Ende der Wiederholung');
      8 : WriteLn('Zustzliche Daten');
      9 : WriteLn('Neue Sample-Daten (neues VOC-Format)');
    ELSE
      WriteLn('nicht erkannt');
    END;
    WriteLn('Blockgre : ', BlockSize);
    WriteLn;
    CASE BlockType OF
      1 :
        BEGIN
          WriteLn('Sample-Rate             : ', ReadSampleRate1, ' Hz');
          PackMethod := ReadPackMethod;     { Komprimierungsverfahren lesen }
          Write('Komprimierungsverfahren : ');
          CASE PackMethod OF
            0 : WriteLn('8-Bits komprimierte Samples');
            1 : WriteLn('4-Bits-Komprimierung');
            2 : WriteLn('2,6-Bits-Komprimierung');
            3 : WriteLn('2-Bits-Komprimierung');
          ELSE
            WriteLn('nicht erkannt');
          END;
        END;
      3 :
        BEGIN
          Period := ReadPeriod;
          Rate := ReadSampleRate1;
          WriteLn('Lnge der Ruhe-Periode : ', Period/Rate:0:2, ' Sek.');
          WriteLn('Sample-Rate            : ', Rate, ' Hz');
        END;
      4 :
        WriteLn('Wert : ', ReadMarker);
      5 :
        WriteLn('Text : ', ReadText(BlockSize));
      6 :
        BEGIN
          Repeats := ReadRepeats;
          IF Repeats = $FFFF THEN         { Unendlich viele Wiederholungen? }
            WriteLn('Anzahl der Wiederholungen : unendlich viele')     { ja }
          ELSE                                                       { nein }
            WriteLn('Anzahl der Wiederholungen : ', Repeats+1);
        END;
      8 :
        BEGIN
          Rate := ReadSampleRate2;                    { Sampling-Rate lesen }
          PackMethod := ReadPackMethod;     { Komprimierungsverfahren lesen }
          Mode := ReadMode;                { Modus (Mono oder Stereo) lesen }
          IF Mode = 0 THEN
            WriteLn('Sample-Rate             : ', Rate, ' Hz')
          ELSE
            WriteLn('Sample-Rate             : ', Rate DIV 2, ' Hz');
          Write('Komprimierungsverfahren : ');
          CASE PackMethod OF
            0 : WriteLn('8-Bits komprimierte Samples');
            1 : WriteLn('4-Bits-Komprimierung');
            2 : WriteLn('2,6-Bits-Komprimierung');
            3 : WriteLn('2-Bits-Komprimierung');
          ELSE
            WriteLn('nicht erkannt');
          END;
          CASE Mode OF
            0 : WriteLn('Modus                   : Mono');
            1 : WriteLn('Modus                   : Stereo');
          END;
        END;
      9 :
        BEGIN
          WriteLn('Sample-Rate             : ', ReadSampleRate3, ' Hz');
          PackMethod := ReadPackMethod;
          Write('Komprimierungsverfahren : ');
          CASE PackMethod OF
            0 : WriteLn('8-Bits komprimierte Samples');
            1 : WriteLn('4-Bits-Komprimierung');
            2 : WriteLn('2,6-Bits-Komprimierung');
            3 : WriteLn('2-Bits-Komprimierung');
          ELSE
            WriteLn('nicht erkannt');
          END;
          Seek(VOCFile, FilePos(VOCFile)+1);         { eine Position weiter }
          Result := IOResult;
          IF Result <> 0 THEN
            SeekError;
          WriteLn('Anzahl Bits             : ', ReadBits);
          Mode := ReadMode;
          Dec(Mode);
          CASE Mode OF
            0 : WriteLn('Modus                   : Mono');
            1 : WriteLn('Modus                   : Stereo');
          END;
        END;
    END;
    IF (BlockType IN [1..9])
    AND (FilePos(VOCFile) < FileSize(VOCFile)-1) THEN
      BEGIN
        Seek(VOCFile, OldPos+BlockSize);        { Sprung zum nchsten Block }
        Result := IOResult;
        IF Result <> 0 THEN
          SeekError;
        Inc(BlockNum);                     { Nummer des Datenblocks erhhen }
        WriteLn;
        WriteLn('Drcken Sie bitte irgendeine Taste ...');
        Key := ReadKey;                               { Tastencode auslesen }
      END;
  UNTIL (BlockType = 0) OR (FilePos(VOCFile) >= FileSize(VOCFile)-1);
END;

{---------------------------------------------------------------------------}
{ Hauptprogramm                                                             }
{---------------------------------------------------------------------------}

VAR
  Result : Integer;                                 { Fehlerstatus-Variable }
  Dir    : DirStr;                              { Verzeichnis der VOC-Datei }
  Name   : NameStr;                   { Name der VOC-Datei ohne Erweiterung }
  Ext    : ExtStr;                                       { Dateierweiterung }

BEGIN
  TextColor(LightGray);                                { Textfarbe hellgrau }
  WriteLn;
  WriteLn('VOCINFO  *  Version 1.05  *  (c) 1993 by Arthur Burda');
  WriteLn(''+
    '');
  IF ParamCount = 0 THEN               { Kommandozeilenparameter angegeben? }
    Error('Fehler: Keine VOC-Datei angegeben.')                      { nein }
  ELSE
    IF ParamStr(1) = '/?' THEN                         { Hilfe angefordert? }
      ShowHelp                    { ja, Hilfetext auf dem Bildschirm zeigen }
    ELSE          { keine Hilfe, wahrscheinlich ein VOC-Dateiname angegeben }
      BEGIN
        VOCName := UpperString(ParamStr(1));      { VOC-Dateinamen auslesen }
        FSplit(VOCName, Dir, Name, Ext);          { VOC-Dateinamen zerlegen }
        IF Ext = '' THEN
          VOCName := VOCName+'.VOC';
        Assign(VOCFile, VOCName);              { Datei mit Namen verknpfen }
        Reset(VOCFile, 1);                                   { Datei ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result = 0 THEN                            { Fehler aufgetreten? }
          BEGIN                                                      { nein }
            ShowInfo;               { Informationen ber die Datei anzeigen }
            Close(VOCFile);                           { VOC-Datei schlieen }
          END
        ELSE                      { leider ein Fehler beim ffnen der Datei }
          Error('Fehler: Datei '+VOCName+' konnte nicht geffnet '+
            'werden.');
      END;
END.
