#define TRACE_ON 0
/*****************************************************************************/
/*                     ACPA Device Specific Non-MIDI Functions               */
/*                                                                           */
/*    DevInit()            Initialize Device                                 */
/*    DevOpen()            Open device, hook ints, etc.                      */
/*    DevClose()           Close device, free ints, etc.                     */
/*    DevWrite(data)       Write data                                        */
/*    DevIOCTLinit(dptr,mindex)  Process IOCTL init functions                */
/*    DevIOCTLstatus(dptr) Return status                                     */
/*    DevIOCTLload()       Load DSP module                                   */
/*    DevStart(trk)                                                          */
/*    DevPause(trk)                                                          */
/*    DevResume(trk)                                                         */
/*    DevStop(trk)                                                           */
/*    DevIOCTLread8()      diagnostic read byte from ACPA port               */
/*    DevIOCTLwrite8()     diagnostic write byte to ACPA port                */
/*    DevIOCTLread16()     diagnostic read word from ACPA port               */
/*    DevIOCTLwrite16()    diagnostic write word to ACPA port                */
/*    DevIOCTLwait()                                                         */
/*                                                                           */
/* 5/13/91 Changes for OS/2 MME flagged with "MME" comment                   */
/* 6691  misc changes - changed prv_buffer_count to host_buffer_count        */
/* 6791  Added moredata to DevInt.                                           */
/* 6/3/91  Changes for SpeechViewer/2 support - Bridget Ritthaler            */
/* 6/17/91 Changes for XA CD-ROM              - Bridget Ritthaler            */
/*         Output host_buffer_count at offset 6 in track control blocks      */
/* 6/25/91 BRR6 - Add multiple and variable length buffer support            */
/* 61491 Added IS_K12 changes for wave-only version (no MIDI)                */
/* 61991 Fixed DOS box problem -                                             */
/* 7191  Changed position updating to reflect # blocks played, not written   */
/* 7291  Added 'cardnum' support for up to 4 ACPA cards & device drivers     */
/* 7391a #define ...[trk] vars using NUM_TRACKS definition                   */
/* 71091 Removed eoi() from DevInt if not our interrupt.                     */
/* 71291 Added global_rearm                                                  */
/* 7/16/91 BRR10 - Setting up global vars (operation, etc..) if BESTFIT      */
/* 7/17/91 BRR12 - Stop DSP code in init                                     */
/* 7/18/91 RJL - Deleted all previously commented-out lines                  */
/* 7/19/91 BRR13 - Modify two track compatibility code for ADPCM/MIDI        */
/* 7/19/91 BRR14 - Reset AT_MAIL after underrun for ADPCM                    */
/* 7/22/91 BRR15 - Add callback support in DevInt                            */
/* 7/25/91 BRR16 - Fix position to reflect milliseconds not # buffers        */
/* 7/25/91 BRR18 - Don't report underrun on first buffer                     */
/* 7/30/91 BRR19 - Keep updating count if stopped but still playing          */
/* 7/30/91 BRR20 - Turn All Notes off on Pause or Stop in MIDI mode          */
/* 7/31/91 BRR22 - Set Master Volume to 100 for ADPCMXA mode                 */
/* 8/01/91 BRR23 - Let recieve data fit exactly into rec buffer              */
/* 8/01/91 BRR24 - Index into SPV2 compute arrays were corrected             */
/* 8/02/91 BRR28 - Display error message if initialization fails             */
/* 8/02/91 BRR29 - Always use first (and only) buffer when read/writing      */
/*                 data for SPV2 compute operations.                         */
/* 8/05/91 BRR30 - Make srate and bps don't cares for SPV2NONE so we don't   */
/*                 get divide by 0 errors for processed value.               */
/* 8/06/91 BRR31 - New AUDIO_CHANGE structure modifications                  */
/* 8/06/91 BRR32 - Send 0 command during record if START flag not set so     */
/*                 no last overrun reported and DSP stops when application   */
/*                 says stop - thus turning off monitor of music             */
/* 8/06/91 BRR33 - Set up Volume/Balance/etc... before writing command to    */
/*                 TMS to start the operation                                */
/* 8/07/91 BRR35 - In DevInt check to see if IOBUF_LOCK is set before up-    */
/*                 dating ->count                                            */
/* 8/08/91 BRR36 - Only clear the play buffer becuase the # blocks includes  */
/*                 the recieve buffers also for SPV2.                        */
/* 8/09/91 BRR37 - Add support for device sharing - multiple instances       */
/* 8/08/91 BRR39 - Don't go into underrun code if COMPUTE operation          */
/* 8/08/91 BRR40 - Clear all control blocks when doing DSP_RUN               */
/* 8/13/91 BRR42 - Fix setting of revtrk for AVC_VOICE and AVC_MUSIC         */
/* 8/15/91 BRR44 - Modifications for OS/2 2.0 driver - Real Mode mostly      */
/* 8/14/91 RJL83 - Moved int clr to right next to EOI (lose int's if not)    */
/* 8/23/91 BRR   - Merged in MME.07 changes                                  */
/* 8/23/91 BRR45 - Only update mailbox for ADPCM when underrun when          */
/*                 completely done filling up buffers. Aleviates high freq.  */
/*                 noise at startup.                                         */
/* 8/23/91 BRR46 - Fix silence for unsigned data 16-bit and signed/2's com.  */
/*                 data at 8-bit in dsp_run.                                 */
/* 8/27/91 BRR48 - Add audio_hpi flags callback support and REALMODE supp.   */
/* 8/27/91 BRR49 - More changes for SPV2 calculation clearing buffers.       */
/* 8/28/91 BRR50 - More changes for callback support.                        */
/* 9/04/91 BRR55 - Move processed calculation into audio_init from DevInt    */
/* 9/05/91 BRR57 - Don't set count=0 on underrun - fix for Nixon stuttering  */
/* 9/05/91 BRR59 - Check for Overrun for CHAINED_BUFFERS - don't assume      */
/*                 get_next_rbuf gave us another buffer.                     */
/* 9/9/91  BRR60 - Put correct position information for ADPCM                */
/* 9/10/91 BRR61 - Put in fix to support ALL_LINES for change.input          */
/* 9/11/91 BRR62 - Increment host_buffer_count to count silent blocks        */
/*                 played at the beginning of playback.                      */
/* 9/11/91 BRR63 - Call aud_control after iobuf ptrs set up.                 */
/* 9/12/91 BRR64 - Check devnum before devtype in case x is set to 8 to      */
/*                 exit                                                      */
/* 9/12/91 BRR65 - Add DevIOCTLwait function                                 */
/* 9/12/91 MME.10-                                                           */
/* 9/12/91 BRR66 - Add support for DEFAULT_INPUT and DEFAULT_OUTPUT          */
/* 9/13/91 BRR67 - Force AUDIO_LOAD for MIGRATE function.                    */
/* 9/13/91 BRR68 - Pad partial block at end with silence in AUDIO_WAIT       */
/* 9/17/91 BRR70 - Condense driver - IOCTLinit/dsp_run,etc...                */
/* 9/18/91 MME.12-                                                           */
/* 9/20/91 BRR76 - Don't let playback start before all blocks of silence     */
/*                 are played - counts/pointers get messed up if you do.     */
/* 9/20/91 BRR83 - Don't call callback for blocks of silence.                */
/* 9/24/91 BRR84 - Fix up the messages that are displayed at INIT time.      */
/* 9/25/91 BRR90 - Return error for MIDI mode if operation != PLAY.          */
/* 9/25/91 BRR95 - Return error for MIDI mode if operation != PLAY.          */
/* 9/30/91 BRR96 - Changes for DevChange to not pass pointers.               */
/* 9/30/91 BRR99 - Don't allow Source_Mix if other track is stereo and set   */
/*                 up the DSP blocks to tell about source_mix.               */
/* 9/30/91 BRR101- MME PTR#753 - ADPCM HQ position value is only 50msec/blk  */
/*10/03/91 BRR102- Fixes for MIDI with AVC                                   */
/*10/03/91 BRR105- Modifiactions for mode_info                               */
/*10/08/91 BRR114- Don't use moredata in determination of UNDERRUN           */
/*10/08/91 BRR117- Don't let buffer counts go less than numblocks when       */
/*                 they wrap around int range                                */
/*10/08/91 BRR119- Not checking CB_STATUS for DOS drivers                    */
/*10/09/91 BRR120- Clr trackcb and mailbox whether reset ACPA or not         */
/*10/16/91 BRR137- Clr trackcb and mailbox [trk^revtrk] not [trk]            */
/*10/17/91 BRR138- Return NO_RECORD instead of -1 if they try MIDI record    */
/*10/18/91 BRR142- Add support for cb_data_segment setup for callback        */
/*10/22/91 BRR145- Don't shift delays by 16 bits                             */
/*10/22/91 BRR108- Combine Windows in with driver.                           */
/*10/22/91 BRR131- Use queue_data instead of adding 0xF8 directly for MIDI   */
/*10/23/91 BRR148- Change spv2_dev_info to spv2_mode_info                    */
/*10/23/91 BRR149- Allow for irqnum > 9 in init messages                     */
/*10/25/91 BRR152- Fix calculation for MIDI position.                        */
/*10/28/91 BRR153- Let check for mode_info be in K12 code for SPV2 data      */
/*10/31/91 BRR156- DevIOCTLwait changes so its not same as AUDIO_WAIT        */
/*11/04/91 BRR172- Return -1 for all errors in DevIOCTLinit                  */
/*11/08/91 BRR181- Don't display install messages for OS/2 2.0               */
/*11/11/91 BRR182- Don't process track 0 int's if AVC_mode                   */
/*11/10/91 BRR183- Add DevDeInit function for AUDIO_INIT for IDLE mode.      */
/*11/12/91 BRR186- Let Dos have max_num_opens also.                          */
/*11/14/91 BRR189- Reset TRK_ARRAY to 0 if open fails                        */
/*11/14/91 BRR190- Free up dsp_use and interrupt in DevDeInit                */
/*11/19/91 BRR142- Push flags before calling CALLBACK for IRET return.       */
/*11/19/91 BRR195- Do EOI and rearm even if not our interrupt.               */
/*11/20/91 BRR118- Stop DSP in DevCLose and just call DevClose in DevDeInit  */
/*11/22/91 BRR44 - Add OS/2 2.0 V86 support - ask Wes                        */
/*12/04/91 BRR203- Make sure buffer >= blocksize, return error if not        */
/*12/09/91 BRR207- Don't stop the DSP if avc_mode in DevClose                */
/*12/11/91 BRR211- Add PADIT support.                                        */
/* 1/06/92 BRR221- Add CBUNDERRUN support.                                   */
/* 1/06/92 BRR216- Add DevPause routine.                                     */
/* 1/06/92 BRR217- Add DevResume routine.                                    */
/* 1/07/92 BRR222- Add VDD support.                                          */
/* 1/14/92 BRR227- Don't assume flags from bps in dsp_run                    */
/* 1/16/92 BRR228- Not setting flags - should |= not just | (Ken knows)      */
/* 1/21/92 BRR224- Change for V86 support - AddrtoGDT                        */
/* 1/23/92 BRR236- Add duration to DevMstVolume                              */
/* 1/23/92 BRR237- Add dither support to DevChange                           */
/* 1/23/92 BRR238- Correct master_volume support in DevChange                */
/* 1/24/92 BRR233- Initialize position_resolution on AUDIO_INIT call         */
/* 1/27/92 BRR240- Don't turn off the DSP if !rec_running                    */
/* 1/27/92 BRR244- Set underrun bit if tlen=0 no matter what                 */
/* 1/28/92 BRR247- Add support for new naming convention of device driver    */
/* 1/30/92 BRR249- Use CUMULATIVE_TIME and rollover_time for MIDI            */
/* 1/30/92 BRR250- Only initial control for current track in dsp_run.        */
/* 1/30/92 BRR252- PPROCESSED not inited yet at AUDIO_INIT for position_res  */
/*                 olution - so use default values.                          */
/* 2/4/92  RJL     Convert to General MIDI - search on "General MIDI"        */
/* 2/10/92 WES262  Add V86 callback support                                  */
/* 2/12/92 BRR265  Make version track dependent                              */
/* 2/12/92 BRR266  Tweek position calculations for PCM to avoid roundoffs    */
/* 2/19/92 BRR186  Add Dos 2 track support                                   */
/* 2/24/92 BRR282  Set irqnum = 9 when configured for IRQ 9 - not IRQ 2      */
/* 2/26/92 BRR283  Decrement count before get_next_buf is called             */
/* 2/27/92 BRR284  Add call to check_if_stopped to interrupt handler         */
/* 2/27/92 BRR285  Don't set cb until end of DevIOCTLinit in case it fails   */
/* 2/28/92 BRR286  Fix master_volume value being output in DevChange         */
/* 2/28/92 BRR287  Put dither out to pcm+3 not pcm in DevChange              */
/* 3/02/92 BRR288  MIDI using 100% CPU.                                      */
/* 3/02/92 BRR289  If count = 0, do a DevAllNotesOff() for MIDI              */
/* 3/03/92 BRR290  Multiply by 100 before dividing for master volume.        */
/* 3/03/92 BRR291  Add support for timing_prescaler for MIDI in there.       */
/* 3/04/92 BRR292  Call get_next_rbuf to report overrun if DSP overran.      */
/* 3/04/92 BRR293  Don't go through record loop again if get_next_rbuf       */
/*                 didn't give us a new buffer.                              */
/* 3/06/92 BRR301  Fix broken source mix                                     */
/* 3/10/92 RJL     Fix SetParm not working for drum kit voices               */
/* 3/11/92 BRR304  Return immediatley from DevIOCTLinit if error found       */
/* 3/18/92 BRRBOCA3Reverse balance if MIDI mode                              */
/* 3/27/92 BRR331  Put #if IS_OS2 around get_next_xbuf compilation.          */
/* 3/27/92 BRR337  Return error for mulaw and alaw 44K stereo                */
/* 3/27/92 BRR332  Add MPC support.                                          */
/* 3/27/92 BRR336  Add IBMPCME.DSP support.                                  */
/* 3/28/92 BRR342  Check dsp_loaded instead of mode for PCM compatibility.   */
/* 3/30/92 BRR347  Add buffer sizes and starting addr and numblocks for      */
/*                 M-motion modes.                                           */
/* 4/01/92 BRR338  Make sure revtrk = 0 for CDXA dsp code and SPV2 dsp code  */
/* 4/02/92 BRR341  Add MPC support.                                          */
/* 4/06/92 BRR357  Return init.bsize.                                        */
/* 4/06/92 BRR358  Add IBMPCMR8.DSP support.                                 */
/* 4/06/92 BRR359  Allow two inits to come in before load to get right dsp   */
/* 4/09/92 BRR369  Add passthru support.                                     */
/* 4/09/92 BRR370  Fix MIDI position when timing_prescaler used.             */
/* 4/10/92 BRR372  Don't switch on long in DevChange                         */
/* 4/11/92 RJL3621 Fix excessive MIDI interrupt handler duration             */
/* 4/12/92 RJL369  Remove passthru from K12, enable for all others           */
/* 4/15/92 BRR388  Fix MIDI performance problems.                            */
/* 4/16/92 BRR396  Don't fail claim_hdwr if other track is empty             */
/* 4/17/92 BRR397  Don't change Dither except for PCM/MU_LAW/A_LAW           */
/* 4/17/92 BRR399  Don't call DevAllNotesOff for windows when count=0        */
/* 4/23/92 RJL341  Move MIDI voice data location from 0x1400 to 0x0800       */
/* 4/24/92 BRR420  Stop writing data if !xmit_running                        */
/* 4/30/92 BRR427  Allow mu_law and A-law to play with mu_law and a_law      */
/* 4/30/92 BRR434  Allow app that VDM did claim hdwr for do a claim hdwr     */
/*                 itself if it wants to.                                    */
/* 4/30/92 BRR436  Don't allow MIDI - PCM 8k 8 together                      */
/* 5/01/92 BRR432  Init mastercb in devioctlload                             */
/* 5/04/92 BRR???  Creation of this file. Split out of ACPA.C.               */
/* 5/15/92 BRR456  OS/2 K12 fixes.                                           */
/* 5/18/92 BRR460  Init SRATE_G for default mode of AUDIO_INIT request       */
/* 5/18/92 BRR???  Add DevNotify support.                                    */
/* 5/20/92 BRR463  Allow INPUT to be changed even if operation != RECORD     */
/* 6/05/92 BRR???  When monitor = 2, set DSP monitor to 3.                   */
/* 6/10/92 BRR483  Restart passthru in Devclose whenever dsp_use_count=0     */
/*****************************************************************************/

/*****************************************************************************/
/* Include DEFINEs                                                           */
/*****************************************************************************/

#if IS_OS2
#define  INCL_DOS
#define  INCL_DOSINFOSEG
#include <os2.h>
#endif

/*****************************************************************************/
/* Include Files                                                             */
/*****************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>

#include "acpadef.h"
#include "audiodd.h"
#include "auddef.h"
#include "audproto.h"
#include "audiodd2.h"
#include "acpamode.h"
#include "audmsg.h"

#if IS_OS2
   #include "mme.h"
#endif
/*****************************************************************************/
/* DEFINES                                                                   */
/*****************************************************************************/

#define ACPA_ID           0x6e6c
#define DSP_HEADER_L          81  /* Length of DSP file header               */

/* Voice Data Locations for DSP modules                                      */
#define AUDS_PSPDATALOC   0x1400  /* Voice data location for IBMAUDS.DSP     */
#define MPC_PSPDATALOC    0x0800  /* Voice data location for IBMMPC.DSP      */

/* IBM PCM DSP Control Block Defines                                         */
#define MAILBOX1          0x0e18
#define MAILBOX2          0x0e20
#define MASTERCB          0x0e10
#define PCMCB1            0x0e30
#define PCMCB2            0x0e50
#define TRACK1CB          0x0e00
#define TRACK2CB          0x0e08

/* Speech Viewer II DSP Control Block Defines                                */
#define SPV2CB            0x1f70  /* SV/2 Specific Data Area                 */
#define SPV2MASTER        0x1f68  /* SPV/2 Master Control Block              */
#define SPV2MB1           0x1f60  /* Host Track 1 Command/Status Area SPV/2  */
#define SPV2MB2           0x1f60  /* not used - but make same as track 1     */
#define SPV2T1CB          0x1f78  /* Track 1 Control Block for SPV/2         */
#define SPV2T2CB          0x1f78  /* not used - but make same as track 1     */

/* CD XA DSP Control Block Defines                                           */
#define XACB1             0x0d30  /* XA Specific Data Area                   */
#define XACB2             0x0d50  /* XA Specific Data Area                   */
#define XAMASTER          0x0d10  /* Master Control Block                    */
#define XAMB1             0x0d18  /* Host Track 1 Command/Status Area for XA */
#define XAMB2             0x0d20  /* Host Track 2 Command/Status Area for XA */
#define XAT1CB            0x0d00  /* Track 1 Control Block for XA            */
#define XAT2CB            0x0d08  /* Track 2 Control Block for XA            */


/* Host Command Reg equates */
#define HINTENA                4  /* 1 = Enable ints to host from C25        */
#define TMSINT                 8  /* 0 = interrupt C25                       */
#define TMSRES                 1  /* 0 = Reset C25 (1 = C25 running)         */
#define SPKREN                16  /* 1 = Enable Speaker                      */

/* DSP Input Defines                                                         */
#define DSP_BOOSTED_MIC     0x00
#define DSP_LEFT_LINE       0x01
#define DSP_RIGHT_LINE      0x02
#define DSP_STEREO          0x03
#define DSP_MIC             0x04


/* OS/2 Specific Defines                                                     */
#if IS_OS2
   #define STDOUT              1
   #define MAXMSGLEN         256
#endif


/*****************************************************************************/
/* External Function declarations                                            */
/*****************************************************************************/
#if IS_OS2
  extern void ReportPTraceError(unsigned int);
#endif

#if NOT_K12
   #if IS_WIN
   #else
      extern passthru(int inpsel);
   #endif
#endif

/*****************************************************************************/
/* Internal Function declarations                                            */
/*****************************************************************************/

void          acpa_out            (unsigned int addr, unsigned int data);
unsigned int  acpa_in             (unsigned int addr);
int           acpa_status         (void);
void          acpa_command        (char newctrlval);
void          CheckRolloverCount  (int trk);
void          dsp_run             (int trk);
void          midi_cb_init        (void);
void          relink              (void);
unsigned long UpdatePosition      (int);

int           InitMIDI            (AIP,int);
int           InitADPCMVoice      (AIP,int);
int           InitADPCMMusic      (AIP,int);
int           InitADPCMStereo     (AIP,int);
int           InitADPCMHQ         (AIP,int);
int           InitPCM             (AIP,int);
int           InitMMotion         (AIP,int);
int           InitSPV2            (AIP,int);
int           InitADPCMXA         (AIP,int);
int           InitSourceMix       (AIP,int);
int           InitClaimHardware   (int);

/* Debug Internal Function Declarations                                      */
#if TRACE_ON
   void       trace        (unsigned char trace_point);
#endif

/* NOT Windows Internal Function Declarations                                */
#if IS_WIN
#else
   void       GetParms     (char far *parms);
#endif

/* Track Specific Internal Function Declarations                             */
#if NUM_TRACKS > 1
char          check_compatibility (int,char *);
void          restore_cbs(int);
void          save_cbs(int);
#endif

/* NOT K12 Specific Internal Function Declarations                           */
#if NOT_K12
unsigned long UpdateMIDIPosition(int);
#endif

/*****************************************************************************/
/* External Variable Declarations                                            */
/*****************************************************************************/

extern unsigned int  hostbase;
extern unsigned char hostrnge = 8;

/* OS2 Specific External Variable Declarations                               */
#if IS_OS2
  extern ACPAMME       AcpaMME;
  extern unsigned char os2_version;

  #if NUM_TRACKS > 1
     extern unsigned long trk_hvdm[NUM_TRACKS];
  #else
     extern unsigned long trk_hvdm;
  #endif

#endif

/* Non Operating Specific - Track Specific External Variable Declarations    */
#if NUM_TRACKS > 1
   extern unsigned long  app_version[];
   extern long           bits_per_sample[];
   extern long           bsize[];
   extern int            channels[];
   extern unsigned char  data_processed[];
   extern long           flags[];
   extern int            hpilock[];
   extern int            initflags[];
   extern int            mode[];
   extern long           operation[];
   extern unsigned long  pos_buffer_count[];
   extern int            ppqn[];
   extern unsigned long  rollover_time[];
   extern long           srate[];
   extern unsigned int   tempo[];
   extern struct vscb    xmitio[];
#else
   extern unsigned long  app_version;
   extern long           bits_per_sample;
   extern long           bsize;
   extern int            channels;
   extern unsigned char  data_processed;
   extern long           flags;
   extern int            hpilock;
   extern int            initflags;
   extern int            mode;
   extern long           operation;
   extern unsigned long  pos_buffer_count;
   extern int            ppqn;
   extern unsigned long  rollover_time;
   extern long           srate;
   extern unsigned int   tempo;
   extern struct vscb    xmitio;
#endif

/* NOT K12 Specific External Variable Declarations                           */
#if NOT_K12
   extern int            ProgChgEnable;
   #if NUM_TRACKS > 1
      extern long        delay[];
   #else
      extern long        delay;
   #endif
#endif

/*****************************************************************************/
/* Internal Variable Declarations                                            */
/*****************************************************************************/

char             acpalock          = 0;
int              cardnum           = 1;
unsigned char    cb_already_set;
char            *cptr;
char             ctrlval           = 0;
ATTACHAREA       ddaddr;
char             dsp_loaded        = -1; /* Set with dsp load module ID      */
char             dsp_to_load       = -1; /* Mode ID, 0x4000=Record,0x2000=HQ */
int              dsp_use_count     = 0;  /* bitmap of hook int users         */
int              irqnum            = 3;
int              mailbox[2];
int              mastercb;
int              oldcard           = 0;
unsigned int     oldposlocs[4]     = {0x0300,0x310,0x320,0x330};
int              opencounter       = 0;
int              pcmcb[2];
int              posirqs[8]        = {3,4,5,6,9,10,11,12};
unsigned int     poslocs[4]        = {0xfdc0,0xfdc8,0xfdd0,0xfdd8 };
int              revtrk            = 0;   /* 1 if AUDIO1$ playing on track 2 */
int              slot_num          = 0;
unsigned int     timing_prescaler;
int              trackcb[2];

/* Track Specific Internal Variable Declarations                             */
#if NUM_TRACKS > 1
   unsigned      clrchar[NUM_TRACKS];    /* Silence char for srate/bps       */
   int           host_buffer_count[2];   /* # of buffers written by host     */
   int           index[NUM_TRACKS];
   unsigned long max_buffer_count[NUM_TRACKS];
   int           modeid[2];              /* 0=Music, 1=Voice, 2=Stereo       */
   int           numblocks[2];           /* # of TMS data blocks             */
   unsigned int  pblocksize[2];          /* Size of ADPCM play data blocks   */
   int           pbufstart[2];           /* Beginning of play data buffers   */
   int           prv_buffer_count[2];    /* Previous # TMS buffers played    */
   unsigned int  rblocksize[2];          /* Size of ADPCM record data blocks */
   int           rbufstart[2];           /* Beginning of record data buffers */
   unsigned char run_both_trks;          /* flag to tell whether to call     */
                                         /* dsp_runfor both trks after a load*/
                                         /* 0 - means only run current trk   */
                                         /* 1 - means run only first trk     */
                                         /* 2 - means run both tracks        */
   struct cbs    saved_cbs;
   unsigned long time_per_block[NUM_TRACKS];
   int           trk_loading = 0;            /* bitmap of trks that can load dsp       */
#else
   unsigned      clrchar;
   int           host_buffer_count;      /* # of buffers written by host     */
   int           index;
   unsigned long max_buffer_count;
   int           modeid;                 /* 0=Music, 1=Voice, 2=Stereo       */
   int           numblocks;              /* # of TMS data blocks             */
   unsigned int  pblocksize;             /* Size of ADPCM play data blocks   */
   int           pbufstart;              /* Beginning of play data buffers   */
   int           prv_buffer_count;       /* Previous #TMS buffers played     */
   unsigned int  rblocksize;             /* Size of ADPCM record data blocks */
   int           rbufstart;              /* Beginning of record data buffers */
   unsigned long time_per_block;
#endif



int   blocksize_arr[NUM_MODES] = { 0,                                //MIDI
                                   288,564,1128,564,                 //ADPCM
                                   280,280,280,280,280,280,280,280,  //PCM
                                   280,280,280,280,                  //MU_LAW
                                   280,280,280,280,                  //A_LAW
                                   0,                                //SOURCE_MIX
                                   0,                                //CLAIM_HW
                                   1152,1152,                        //ADPCMXA
                                   119,213,85,                       //SPV2
                                   280,280,280,280,280,280,          //MMOTION PCM
                                   280,280,280,                      //MMOTION MU_LAW
                                   280,280,280,                      //MMOTION A_LAW
                                   0,0};                             //OTHER PCM

int   bufstart_arr[2][NUM_MODES] ={0,
                                   0xe48,0xe48,0xe48,0xe48,
                                   0xe70,0xe70,0xe70,0xe70,0xe70,0xe70,0xe70,0xe70,
                                   0xe70,0xe70,0xe70,0xe70,
                                   0xe70,0xe70,0xe70,0xe70,
                                   0,
                                   0,
                                   0x0d70,0x0d70,
                                   0x1d84,0x1c0c,0x1e0c,
                                   0xe70,0xe70,0xe70,0xe70,0xe70,0xe70,
                                   0xe70,0xe70,0xe70,
                                   0xe70,0xe70,0xe70,
                                   0,0,
                                   0,

                                   0x1b80,0x1718,0xe48,0xe48,
                                   0x1730,0x1730,0x1730,0x1730,0x1730,0x1730,0x1730,0x1730,
                                   0x1730,0x1730,0x1730,0x1730,
                                   0x1730,0x1730,0x1730,0x1730,
                                   0,
                                   0,
                                   0x1672,0x1672,
                                   0,0,0,
                                   0x1730,0x1730,0x1730,0x1730,0x1730,0x1730,
                                   0x1730,0x1730,0x1730,
                                   0x1730,0x1730,0x1730,
                                   0,0};

int   numblocks_arr[NUM_MODES] = { 0,
                                   4,4,4,8,
                                   8,8,8,8,8,8,8,8,
                                   8,8,8,8,
                                   8,8,8,8,
                                   0,
                                   0,
                                   4,4,
                                   4,4,4,
                                   8,8,8,8,8,8,
                                   8,8,8,
                                   8,8,8,
                                   0,0};

/* Only need mode id for ADPCM modes - modeid set to 0 for all other modes   */
int   modeid_arr[4] =             {1,0,2,0};


/* Speech Viewer II Specific Internal Variable Declarations                  */
                 /* matrix for buffer array size for SPV/2 COMPUTE operations*/
                 /* with a buffer size for both input and output buffers     */
int spv2cbufsize_arr[3][2] = { 213,213, 482,1, 128,69 };
                 /* matrix for addr of first buffer for each of the 3        */
                 /* calculations that can be done - one for input buffer     */
                 /* and one for output buffer                                */
int spv2cbufstart[3][2] = { 0x1db6,0x1e8b, 0x1d7d,0x1f5f, 0x1e9b,0x1f1b};
                 /* number of input (same number for output) blocks for the  */
                 /* three types of computations                              */
int spv2cnumblocks[3]   = { 2, 2, 2};

int spv2inpbufsize[3]   = {119,213,85};
int spv2inpbufstart[3]  = {0x1d84,0x1c0c,0x1e0c };
int spv2inpnumblocks    = 4;
int spv2playbufsize[3]  = {35,129,0};
int spv2playbufstart[3] = {0x1e8e,0x1c5a,0x0000 };
int spv2playnumblocks   = 6;

/* NOT K12 Specific Internal Variable Declarations                           */
#if NOT_K12

   unsigned short   pspdataloc;

   /* NOT Windows Specific Internal Variable Declarations                    */
   #if IS_WIN
   #else
      unsigned char do_passthru = 0;
   #endif
#endif

/* AVC 1.03 Specific Internal Variable Declarations                          */
#if IS_AVC103
   int              avc103_mode = 0;        /* AVC 1.03 compatibility mode   */
#endif

/* OS/2 Specific Internal Variable Declarations - all for MME for OS/2 2.0   */
#if IS_OS2
   unsigned long   IvTable;
   unsigned char   MsgId[]                 = "AUD";
   unsigned short  MsgLen                  = 0;
   unsigned char   nszMsgBuff [MAXMSGLEN];
   unsigned char   nszMsgFile []           = "AUDIODD.MSG\0";
   unsigned char   nszMsgNum[]             = "0\0";
#endif

/*****************************************************************************/
/* DevStart                                                                  */
/*****************************************************************************/
void DevStart(trk)
int trk;
{
   /* Nothing to do */
}

/*****************************************************************************/
/* DevPause                                                                  */
/*****************************************************************************/
void DevPause(trk)
int trk;
{
   /* Nothing to do */
}

/*****************************************************************************/
/* DevResume                                                                 */
/*****************************************************************************/
void DevResume(trk)
int trk;
{
   /* Nothing to do */
}

/*****************************************************************************/
/* DevStop                                                                   */
/*****************************************************************************/
void DevStop(trk)
int trk;
{
   /* nothing to do */
}


/*****************************************************************************/
/* DevNotify                                                                 */
/*****************************************************************************/
char DevNotify(pos,trk)
unsigned long pos;
int trk;
{
   /* Nothing to do */
   return(0);
}

/*****************************************************************************/
/* DevChange                                                                 */
/*****************************************************************************/
int DevChange(changes, dev_info, any_dev_info, mode_info, any_mode_info, trk)
struct audio_change far *changes;
int *dev_info;
char any_dev_info;
int *mode_info;
char any_mode_info;
int trk;
{
   int   trkcb,pcmcblock;
   int   rc=0;
   short x;
   unsigned int devtype;

   trkcb = trackcb[trk^revtrk];
   pcmcblock = pcmcb[trk^revtrk];

   if(changes->input >= 0) { /* No-Change if -1               */
      if (changes->input == INPUTS_LISTED) {
         changes->input = 0;
         for (x=0; x<8; x++) {
            if (changes->input_list[x].devnum > DEVICE_1) rc = INVALID_INPUT_LIST;

            if (changes->input_list[x].devtype == DEFAULT_INPUT) {
               devtype = STEREO_LINE_INPUT;
            } else {
               devtype = (unsigned int)changes->input_list[x].devtype;
            } /* endif */
            switch(devtype) {
               case NULL_INPUT:
                    x = 8;
                    break;
               case STEREO_LINE_INPUT:
                    if (changes->input == 0) {
                       changes->input |= DSP_STEREO;
                    } else {
                       rc = INVALID_INPUT_LIST;
                    } /* endif */
                    break;
               case LEFT_LINE_INPUT:
                    if (changes->input == 0) {
                       changes->input |= DSP_LEFT_LINE;
                    } else {
                       rc = INVALID_INPUT_LIST;
                    } /* endif */
                    break;
               case RIGHT_LINE_INPUT:
                    if (changes->input == 0) {
                       changes->input |= DSP_RIGHT_LINE;
                    } else {
                       rc = INVALID_INPUT_LIST;
                    } /* endif */
                    break;
               case MIC_INPUT:
                    if (changes->input == 0) {
                       changes->input |= DSP_MIC;
                    } else {
                       rc = INVALID_INPUT_LIST;
                    } /* endif */
                    break;
               case BOOSTED_MIC_INPUT:
                    if (changes->input == 0) {
                       changes->input |= DSP_BOOSTED_MIC;
                    } else {
                       rc = INVALID_INPUT_LIST;
                    } /* endif */
                    break;
               default:
                    rc = INVALID_INPUT_LIST;
            } /* end switch */
         } /* endfor */
      } else if (changes->input == ALL_LINES) {
         changes->input = LINES_1AND2;
      } /* endif */
      if (!rc) acpa_out(mastercb,(int)changes->input);
   } /* if record */
   if(changes->output >= 0){                          /* Output selection               */
      if (changes->output == OUTPUTS_LISTED) {
         ctrlval &= ~SPKREN;              /*    turn internal speaker off   */
         for (x=0; x<8; x++) {
            if (changes->output_list[x].devnum > DEVICE_1) rc = INVALID_OUTPUT_LIST;
            if ((changes->output_list[x].devtype == SPEAKER_OUTPUT) ||
                (changes->output_list[x].devtype == DEFAULT_OUTPUT)) {
               ctrlval |= SPKREN;            /*    turn internal speaker on   */
            } else if (changes->output_list[x].devtype == NULL_OUTPUT) {
               x = 8;
            } else if ((changes->output_list[x].devtype == PHONE_LINE_OUTPUT) ||
                       (changes->output_list[x].devtype == HANDSET_OUTPUT)) {
               rc = INVALID_OUTPUT_LIST;
            } /* endif */
         } /* end for */
      } else {
         if(changes->output & INTERNAL_SPEAKER) {        /*  Is internal speaker set?   */
            ctrlval |= SPKREN;               /*  turn internal speaker on   */
         }else{                              /*  Nope,                      */
            ctrlval &= ~SPKREN;              /*  turn internal speaker off  */
         }
      } /* endif */
   }
   if(changes->monitor >= 0 && OPERATION_G==RECORD){
      if (changes->monitor == 2) {
         acpa_out(mastercb+1,3);
      } else {
         acpa_out(mastercb+1,(int)changes->monitor);
      } /* endif */
   }
   if(changes->volume >= 0) {
      if(changes->volume_delay >= 0) {
         if (changes->volume_delay > 0x00007FFF)
            changes->volume_delay = 0x00007FFF;
      }
      DevTrkVolume((int)(changes->volume>>16),(int)(changes->volume_delay),trk);
   }
   if(changes->balance >= 0) {
      if(changes->balance_delay >= 0) {
         if (changes->balance_delay > 0x00007FFF)
            changes->balance_delay = 0x00007FFF;
      }
      DevTrkBalance((int)(changes->balance>>16),(int)(changes->balance_delay),trk);
   }
   if (any_dev_info) {
      if (dev_info[0] != AUDIO_IGNORE) {
         DevMstVolume(dev_info[0],0,trk);
      } /* endif */
      if ((MODE_G == PCM) || (MODE_G == A_LAW) || (MODE_G == MU_LAW)) {
         if (dev_info[1] != AUDIO_IGNORE) {
            acpa_out(pcmcblock+3,dev_info[1]/0x7FFF*100);
         } /* endif */
      } /* endif */
   } /* endif */

   if (any_mode_info) {
#if NOT_K12
      if (MODE_G==MIDI) {
         if (mode_info[0] >= 0) TEMPO = mode_info[0];
         if (mode_info[1] >= 0) PPQN = mode_info[1];
         if (mode_info[0] >=0 || mode_info[1] >= 0) SetClock(trk);
      } else
#endif
      if ((MODE_G==SPV2PCM) ||
          (MODE_G==SPV2BCPCM) ||
          (MODE_G==SPV2NONE)) {
         for (x=0; x<6; x++) {
            acpa_out(SPV2CB+x,mode_info[x]);
         } /* endfor */
         acpa_out(SPV2CB+7,1);               /* Set flag to say we have    */
                                             /* updated the data           */
      } /* endif */
   } /* endif */
#if IS_WIN
   acpa_command(SPKREN+TMSINT+HINTENA+TMSRES); /* Enable ACPA   */
#endif
   return(rc);
}

/*****************************************************************************/
/* DeInit                                                                    */
/*****************************************************************************/
void DevDeInit(trk)                             /* Process deinit requests   */
int trk;
{
   DevClose(trk);
}

/*****************************************************************************/
/* Init IOCTL                                                                */
/*****************************************************************************/
int DevIOCTLinit(dptr,mindex,trk)           /* Process ioctl init requests   */
AIP dptr;
int mindex,trk;
{
   extern int slot_num;
   int  x;
   int at_mail;

/* Note that the following logic is used for determining which DSP to load   */
/* or not to load (that is the question.....)                                */
/* dsp_to_load = -1   <----  means was asked to load and already did the load*/
/*                           (changed to -1 from >0 in DevIOCLTload)         */
/* dsp_to_load =  0   <----  no load required, correct dsp already loaded    */
/* dsp_to_load >  0   <----  dsp needs to be loaded and hasn't been done yet */
/*                                                                           */
/* Now if the other track has been inited already, then we want to check to  */
/* see if a different DSP module than the one loaded is required. If it is   */
/* then do the following:                                                    */
/*    if (data_processed)                                                    */
/*       to late to do anything                                              */
/*    else data not processed yet                                            */
/*       turn off the DSP code and tell them the new module to load          */
/*       set run_both_trks = 2 so when it is loaded, both tracks will be     */
/*                               initialized (dsp_run called for both tracks)*/
/*    }                                                                      */
/*                                                                           */
/* Note that there is a window in there where if the first track does the    */
/* INIT and is told to load the dsp code.... and then the second track comes */
/* down and does an INIT and is told to load a dsp module because the first  */
/* track hasn't processed any data yet...the second track loads his code.... */
/* second track starts processing and then the first track comes down and    */
/* starts loading the code it was told to load......booooooooom - second track*/
/* is dead!                                                                  */
/* This can be solved by knowing which track I think should be loading the   */
/* code - and don't let the other track ever load. When load is complete     */
/* set this track_loading flag to 0 and only let load complete if trk is     */
/* the same as track_loading flag. If both tracks want same DSP module, then */
/* want to set track_loading to 3 to say either track may load the code.     */
/* If the correct DSP code was already requested to be loaded, then tell this*/
/* AUDIO_INIT to load also - and let either track actually do the load - and */
/* just don't do it the second time if both tracks actually try to do the    */
/* load.                                                                     */


   INDEX = mindex;
   cb_already_set = 0;
   dptr->rc = 0;                          /*  set rc = 0                   */
   dptr->slot_number = slot_num;
   dptr->device_id = ACPA;                /* Set device ID number          */

   /* Initialize count defaults for non-PLAY operations */
   PRV_BUFFER_COUNT = 0;
   HOST_BUFFER_COUNT = 0;

#if NUM_TRACKS > 1
   run_both_trks = 0;
#endif
   cptr = NULL;
   dptr->bsize = mode_table[INDEX].bsize;
#if NOT_K12
   if (INDEX == MIDI_MODE) {

      if (InitMIDI(dptr,trk))                            /* MIDI             */
         return(-1);

   } else
#endif
   if (INDEX == AVC_VOICE) {

      if (InitADPCMVoice(dptr,trk))                      /* ADPCM Voice      */
         return(-1);

   } else if (INDEX == AVC_MUSIC) {

      if (InitADPCMMusic(dptr,trk))                      /* ADPCM Music      */
         return(-1);

   } else if (INDEX == AVC_STEREO) {

      if (InitADPCMStereo(dptr,trk))                     /* ADPCM Stereo     */
         return(-1);

   } else if (INDEX == AVC_HQ) {

      if (InitADPCMHQ(dptr,trk))                         /* ADPCM HQ         */
         return(-1);

   } else if ((INDEX >= PCM8k8b) && (INDEX <= ALAW44K)) {

      if (InitPCM(dptr,trk))                             /* PCM/A_LAW/Mu_Law */
         return(-1);

   } else if ((INDEX >= PCM315k8b) && (INDEX <= ALAW48K)) {

      if (InitMMotion(dptr,trk))                         /* M-Motion         */
         return(-1);

   } else if ((INDEX >= SPV2BCPCM) && (INDEX <= SPV2NONE)) {

      if (InitSPV2(dptr,trk))                            /* Speech Viewer II */
         return(-1);

   } else if ((INDEX >= ADPCMXAC) && (INDEX <= ADPCMXAB)) {

      if (InitADPCMXA(dptr,trk))                         /* CD ROM XA        */
         return(-1);

   } else if (INDEX == SOURCE_MIX_MODE) {

      if (InitSourceMix(dptr,trk))                       /* Source Mix       */
         return(-1);

   } else if (INDEX == CLAIM_HDWR_MODE) {

      if (InitClaimHardware(trk) == -1)  {               /* Claim Hardware   */
         return(-1);
      } else {
         return(INDEX);    /* Return - don't hook int or set loadpath field   */
      }

   } else {

      #if IS_K12
         INDEX=AVC_VOICE;
      #else
         INDEX=MIDI_MODE;
      #endif

      dptr->flags &= ~(mode_table[INDEX].flags_mask);
      dptr->flags |= mode_table[INDEX].flags | BESTFIT_PROVIDED;
      dptr->mode = mode_table[INDEX].mode;
      dptr->srate = mode_table[INDEX].srate_low;
      dptr->bits_per_sample = mode_table[INDEX].bps_low;
      dptr->channels = mode_table[INDEX].channels_low;
      SetGlobalParms(dptr,trk);

      #if IS_K12
         if (InitADPCMVoice(dptr,trk))
            return(-1);
      #else
         if (InitMIDI(dptr,trk))
            return(-1);
      #endif

   } /* endif INDEX == */

   dptr->flags = dptr->flags | VOLUME | VOLUME_DELAY
               | BALANCE | BALANCE_DELAY
               | INPUT | OUTPUT | MONITOR;


   MODEID = 0;
#if NOT_K12
   if (INDEX != MIDI_MODE) {
#endif
      if (INDEX != SOURCE_MIX_MODE) {
         if ((INDEX != SPV2BCPCM) && (INDEX != SPV2PCM) && (INDEX != SPV2NONE)) {
            PBLOCKSIZE = blocksize_arr[INDEX];
            RBLOCKSIZE = blocksize_arr[INDEX];
            if ((MODE_G == ADPCM) || (MODE_G == ADPCMXA)) {
               PBUFSTART = bufstart_arr[trk^revtrk][INDEX];
               RBUFSTART = bufstart_arr[trk^revtrk][INDEX];
               NUMBLOCKS = numblocks_arr[INDEX];
               if (MODE_G == ADPCM) {
                  MODEID = modeid_arr[INDEX-1];
                  if (INDEX == AVC_HQ) {
                     TIME_PER_BLOCK = 50 * SRATE_G;
                  } else {
                     TIME_PER_BLOCK = 100 * SRATE_G;
                  } /* endif */
               } /* endif */
            } else if (MODE_G != ADPCMXA) { /* PCM/Mu-Law/A-Law */
               TIME_PER_BLOCK = ((unsigned long)PBLOCKSIZE*2*1000);
               if (CHANNELS_G == 2) TIME_PER_BLOCK /= 2;
               if (BITS_PER_SAMPLE_G == 16) TIME_PER_BLOCK /= 2;
            } /* endif */
         } /* endif */
         if (dptr->operation == PLAY) {
            HOST_BUFFER_COUNT = NUMBLOCKS;
            PRV_BUFFER_COUNT = NUMBLOCKS;
         } /* endif */
      } else {
         TIME_PER_BLOCK = 21;
         HOST_BUFFER_COUNT = NUMBLOCKS;
         PRV_BUFFER_COUNT = NUMBLOCKS;
      } /* endif */

      /* Determine the maximum number of buffers before we rollover and */
      /* then subtract 20 for good measure */
      MAX_BUFFER_COUNT = 0xFFFFFFFF / TIME_PER_BLOCK - 20;

      dptr->position_resolution = TIME_PER_BLOCK/SRATE_G;

#if NOT_K12

   } else {

      MAX_BUFFER_COUNT = (0xFFFFFFFF / 600000) - 1;
      dptr->position_resolution = 600000 / (1200*96);

   } /* endif */

#endif


   if (!cb_already_set) {

      mastercb   = MASTERCB;          /* Init default addr of master ctrl blk  */
      trackcb[0] =  TRACK1CB;         /* Init default addr of track 1 ctrl blk */
      mailbox[0] =  MAILBOX1;         /* Init default addr of track 1 mailbox  */
      pcmcb[0]   =  PCMCB1;           /* Init default addr of track 1 pcm cb   */
#if NOT_K12
      trackcb[1] =  TRACK2CB;         /* Init default addr of track 2 ctrl blk */
      mailbox[1] =  MAILBOX2;         /* Init default addr of track 2 mailbox  */
      pcmcb[1]   =  PCMCB2;           /* Init default addr of track 2 pcm cb   */
#endif
   } /* endif */



   if(dsp_use_count == 0){                /* Nobody hooked int yet?        */
      acpa_command(0);                    /* Make sure ACPA is shut down   */
      if(GetDevInt(irqnum)){                    /* Hook Device Interrupt   */
         return(-1);
      }
   }
   dsp_use_count |= (1 << trk);                 /* Flag our use of dsp     */

   if (cptr) {                                  /* Do we need to load dsp? */
#if NUM_TRACKS > 1
      if (trk_loading & 0x8000) {               /* Check to see if in the     */
                                                /* middle of loading code     */
                                                /* right now                  */
         /* If loading code from other trk right now - bad timing - even      */
         /* though we haven't started playing the data yet - we don't want to */
         /* tell this guy he can load also in case he starts loading before   */
         /* the other guy has finished - thus who knows what we actually would*/
         /* end up with on the ACPA card.                                     */
         return(-1);
      } else { /* No one currently loading code                               */
         trk_loading |= (1 << trk);             /* Mark this trk as being able*/
                                                /* to load the code           */
#endif
         for(x=0; x<13; x++){                   /*  if so, set module name    */
            dptr->loadpath[x] = *cptr++;        /* Copy module name           */
         }
         dptr->flags |= LOAD_CODE;
#if NUM_TRACKS > 1
      } /* endif */
#endif
   }else{                                       /* Otherwise, init dsp        */
      dsp_to_load = 0;
      if ((ctrlval & TMSRES) && (MODE_G != MIDI)) {
         acpa_out(mailbox[trk^revtrk],0);       /* Stop operation   */
         while(at_mail = acpa_in(mailbox[trk^revtrk]+2));
      } /* endif */
      dsp_run(trk);                             /*  if so, go do it           */
   }

   return(INDEX);
}

#if NUM_TRACKS > 1
/*****************************************************************************/
/* Check Other Active Track for Compatibility with this Track                */
/*****************************************************************************/
char check_compatibility(trk,dsp_array)
int trk;
char *dsp_array;
{
   char dsp_needed;

   /* Check the mode compatibility table to see if the other tracks mode     */
   /* is compatible with this tracks                                         */
   dsp_needed = dsp_array[index[1-trk]];

   if (dsp_needed) {

      if ((dsp_needed != dsp_loaded) &&
          (dsp_needed != dsp_to_load)) {
         hpilock[1-trk]++;                 /* Don't process anything on      */
                                           /* other trk while following      */
                                           /* checks are being done...       */
         if (data_processed[1-trk]) {      /* Has the other trk              */
                                           /* processed any data yet?        */
            hpilock[1-trk]--;              /* continue processing            */
            return(-1);                    /* can't do it if they have       */
         } else {                          /* haven't started any processing */
            dsp_to_load = dsp_needed;      /* still have time to             */
            cptr = dsp_string[dsp_to_load];/* rectify the situation          */
            run_both_trks = 2;             /* Flag so 1st trk started        */
            save_cbs(1-trk);               /* Save other trks control blocks */
                                           /* so they can be restored after  */
                                           /* new DSP is loaded.             */
            acpa_command(0);               /* turn dsp code off so other     */
                                           /* trk doesn't do any processing. */
            hpilock[1-trk]--;              /* reset flag                     */
         } /* endif */
      } /* endif */                        /* else - we're OK - no code      */
                                           /* load required.                 */
   } else {
      return(-1);
   } /* endif */

   /* If RECORD operation or 2 Channels - then no modes are compatible       */
   if ((operation[1-trk] == RECORD) ||
       (operation[trk] == RECORD) ||
       (channels[1-trk] == 2) ||
       (channels[trk] == 2)) {
      return(-1);
   } /* endif */

   return(0);

} /* end check_compatibility */

/*****************************************************************************/
/* save_cbs(int trk)                                                         */
/* Save the track, master, and mailbox control blocks so they can be restored*/
/* after the new DSP code is loaded.                                         */
/*****************************************************************************/
void save_cbs(trk)
int trk;
{
   int x;

   saved_cbs.trk = trk;
   for (x=0; x<8; x++) {
      saved_cbs.trackcb[x] = acpa_in(trackcb[trk]+x);
      saved_cbs.mastercb[x] = acpa_in(mastercb+x);
      saved_cbs.mailbox[x] = acpa_in(mailbox[trk]+x);
   } /* endfor */

} /* end save_cbs */

/*****************************************************************************/
/* restore_cbs(int trk)                                                      */
/* Restore the track, master, and mailbox control blocks.                    */
/*****************************************************************************/
void restore_cbs(trk)
int trk;
{
   int x;

   for (x=0; x<8; x++) {
      acpa_out(trackcb[trk]+x,saved_cbs.trackcb[x]);
      acpa_out(mastercb+x,saved_cbs.mastercb[x]);
      acpa_out(mailbox[trk]+x,saved_cbs.mailbox[x]);
   } /* endfor */

} /* end restore_cbs */

#endif


#if NOT_K12
/*****************************************************************************/
/* Init MIDI Mode                                                            */
/*****************************************************************************/
int InitMIDI(dptr,trk)
AIP dptr;
int trk;
{

      if (dptr->operation != PLAY) return(NO_RECORD);
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         revtrk = 1-trk;                     /*  ok, MIDI must be 2nd trk  */
         if(dsp_loaded != IBMAUDS_DSP){      /* Is module already loaded?  */
            dsp_to_load = IBMAUDS_DSP;
            cptr = dsp_string[dsp_to_load];
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other trk isn't idle       */
                                             /* Other track compatible and*/
         if ((check_compatibility(trk,mode_compat_array[MIDI_INDEX])) ||
             (revtrk == trk)) {              /* are we on the right trk?  */
            return(-1);
         } /* endif */
      }
#endif

      /* moved pprocess calculation to setclock */
      midi_cb_init();
      ProgChgEnable = 1;                     /* Prog Chgs enabled by default  */
      timing_prescaler = 1;
      return(0);

} /* end InitMIDI */
#endif

/*****************************************************************************/
/* Init ADPCM Voice                                                          */
/*****************************************************************************/
int InitADPCMVoice(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
   if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
                                                /* out of conditional section */
      revtrk = trk;                          /*  ok, put ADPCM on 1st trk  */
      if (dptr->operation==RECORD) {
         if(dsp_loaded!=IBMAUDR_DSP){
            dsp_to_load = IBMAUDR_DSP;       /* Mode = ADPCM + Record bit*/
            cptr = dsp_string[dsp_to_load];
         }
      } else if (dptr->operation==PLAY) {
         if(dsp_loaded!=IBMAUDP_DSP &&
            dsp_loaded!=IBMAUDS_DSP){
#if IS_K12
            dsp_to_load = IBMAUDP_DSP;       /* Mode = ADPCM            */
            cptr = dsp_string[dsp_to_load];
#else
            dsp_to_load = IBMAUDS_DSP;       /* Mode = SYNTH+VOICE      */
            cptr = dsp_string[dsp_to_load];
#endif
         }
      }else if(dptr->operation==PLAY_AND_RECORD){
         dptr->rc = NO_PLAY_AND_RECORD;
         return(-1);
      }else{
         dptr->rc = INVALID_REQUEST;
         return(-1);
      }
#if NUM_TRACKS > 1

   }else{                                 /* Other track is active      */

                                             /* Other track compatible and*/
      if (check_compatibility(trk,mode_compat_array[VOICE_INDEX])) {
         return(-1);
      } /* endif */

   }
#endif
   return(0);

} /* end InitADPCMVoice */

/*****************************************************************************/
/* Init ADPCM Music                                                          */
/*****************************************************************************/
int InitADPCMMusic(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dptr->operation==RECORD){
            if(dsp_loaded!=IBMAUDR_DSP){
               dsp_to_load = IBMAUDR_DSP;      /* Mode = ADPCM + Record bit*/
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY){
            if(dsp_loaded!=IBMAUDP_DSP){
               dsp_to_load = IBMAUDP_DSP;       /* Mode = ADPCM Playback   */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
                                             /* Other track compatible and*/
         if (check_compatibility(trk,mode_compat_array[MUSIC_INDEX])) {
            return(-1);
         } /* endif */
      }
#endif
      return(0);
} /* end InitADPCMMusic */


/*****************************************************************************/
/* Init ADPCM Stereo Mode                                                    */
/*****************************************************************************/
int InitADPCMStereo(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dptr->operation==RECORD){
            if(dsp_loaded!=IBMAUDR_DSP){
               dsp_to_load = IBMAUDR_DSP;      /* Mode = ADPCM + Record bit*/
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY){
            if(dsp_loaded!=IBMAUDP_DSP){
               dsp_to_load = IBMAUDP_DSP;      /* Mode = ADPCM Playback   */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
         return(-1);                            /* No other modes compatible*/
      }
#endif
      revtrk = 0;
      return(0);

} /* end InitADPCMStereo */


/*****************************************************************************/
/* Init ADPCM Hiqh Quality Mode                                              */
/*****************************************************************************/
int InitADPCMHQ(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dptr->operation==RECORD){
            if(dsp_loaded!=IBMAUDG_DSP){
               dsp_to_load = IBMAUDG_DSP;    /* Mode = ADPCM + Record bit    */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY){
            if(dsp_loaded!=IBMAUDF_DSP){
               dsp_to_load = IBMAUDF_DSP;    /* Mode = ADPCM HQ Playback     */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
         return(-1);                            /* No other modes compatible*/
      }
#endif
      revtrk = 0;
      return(0);
} /* end InitADPCMHQ */


/*****************************************************************************/
/* Init PCM/Mu_Law/A_Law 8k, 11k, 22k, 44K Modes                             */
/*****************************************************************************/
int InitPCM(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dptr->operation==RECORD){
            if (SRATE_G == 8000) {
               if (dsp_loaded!=IBMPCMR8_DSP) {
                  dsp_to_load = IBMPCMR8_DSP;   /* Mode = PCM + 8K Record    */
                  cptr = dsp_string[dsp_to_load];
               }
            } else if (dsp_loaded!=IBMPCMR_DSP) {
               dsp_to_load = IBMPCMR_DSP;       /* Mode = PCM + Record bit   */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY){
            if(dsp_loaded!=IBMPCMP_DSP){
               dsp_to_load = IBMPCMP_DSP;       /* Mode = ADPCM Playback     */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                           /* Other track is active      */
                                             /* Other track compatible and*/
         if ((MODE_G == PCM) &&
             (BITS_PER_SAMPLE_G == 8) &&
             ((SRATE_G == 11025) || (SRATE_G == 22050))) {
            if (check_compatibility(trk,mode_compat_array[PCM11K22K_INDEX])) {
               return(-1);
            } /* endif */
         } else {
            if (check_compatibility(trk,mode_compat_array[PCM_INDEX])) {
               return(-1);
            } /* endif */
         } /* endif */
      }
#endif
      if(dptr->operation==RECORD          /* PCM Record always uses 16 blocks */
      || dptr->channels==2){
         RBUFSTART = 0xe70;
         PBUFSTART = 0xe70;
         NUMBLOCKS = 16;
      }else{
         RBUFSTART = bufstart_arr[trk^revtrk][INDEX];
         PBUFSTART = bufstart_arr[trk^revtrk][INDEX];
         NUMBLOCKS = numblocks_arr[INDEX];
      }
      return(0);

} /* end InitPCM */


/*****************************************************************************/
/* Init Speech Viewer II Mode                                                */
/*****************************************************************************/
int InitSPV2(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if ((dptr->operation==RECORD) ||
             (dptr->operation==PLAY) ||
             (dptr->operation==ANALYSIS) ||
             (dptr->operation==DISTANCE) ||
             (dptr->operation==MIGRATION)) {
            if(dsp_loaded!=IBMSPV2_DSP || dptr->operation==MIGRATION){
               dsp_to_load = IBMSPV2_DSP;       /* Mode = Speech Viewer/2    */
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
         return(-1);
      }
#endif
      revtrk = 0;
      if (dptr->operation==RECORD) {
         PBUFSTART = spv2inpbufstart[INDEX-25];
         RBUFSTART = spv2inpbufstart[INDEX-25];
         NUMBLOCKS = spv2inpnumblocks;
         PBLOCKSIZE = spv2inpbufsize[INDEX-25];
         RBLOCKSIZE = spv2inpbufsize[INDEX-25];
      } else if (dptr->operation==PLAY) {
         PBUFSTART = spv2playbufstart[INDEX-25];
         RBUFSTART = spv2playbufstart[INDEX-25];
         NUMBLOCKS = spv2playnumblocks;
         PBLOCKSIZE = spv2playbufsize[INDEX-25];
         RBLOCKSIZE = spv2playbufsize[INDEX-25];
      } else {
                                              /* get the starting addr of    */
                                              /* the input data to the DSP   */
         PBUFSTART = spv2cbufstart[dptr->operation-ANALYSIS][0];
         RBUFSTART = spv2cbufstart[dptr->operation-ANALYSIS][1];
         NUMBLOCKS = spv2cnumblocks[dptr->operation-ANALYSIS];
                                              /* get size of buffer for input*/
                                              /* data to the DSP             */
         PBLOCKSIZE = spv2cbufsize_arr[dptr->operation-ANALYSIS][0];
         RBLOCKSIZE = spv2cbufsize_arr[dptr->operation-ANALYSIS][1];
      }
      TIME_PER_BLOCK = 1;
      mastercb   =  SPV2MASTER;
      trackcb[0] =  SPV2T1CB;         /* Init SPV/2 addr of track 1 ctrl blk */
      mailbox[0] =  SPV2MB1;          /* Init SPV/2 addr of track 1 mailbox  */
      trackcb[1] =  SPV2T2CB;         /* Init SPV/2 addr of track 2 ctrl blk */
      mailbox[1] =  SPV2MB2;          /* Init SPV/2 addr of track 2 mailbox  */
      cb_already_set = 1;
      return(0);
} /* end InitSPV2 */


/*****************************************************************************/
/* Init M-Motion Modes (PCM/Mu_Law/A_Law 7875, 31500, 48000)                 */
/*****************************************************************************/
int InitMMotion(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dptr->operation==RECORD){
            dptr->rc = NO_RECORD;
            return(-1);
         }else if(dptr->operation==PLAY){
            if(dsp_loaded!=IBMPMM48_DSP){
               dsp_to_load = IBMPMM48_DSP;
               cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{
                                             /* Other track compatible and*/
         if (check_compatibility(trk,mode_compat_array[MMOTION_INDEX])) {
            return(-1);
         } /* endif */
      }
#endif
      revtrk = 0;
      if (dptr->channels==2){
         RBUFSTART = 0xe70;
         PBUFSTART = 0xe70;
         NUMBLOCKS = 16;
      }else{
         RBUFSTART = bufstart_arr[trk^revtrk][INDEX];
         PBUFSTART = bufstart_arr[trk^revtrk][INDEX];
         NUMBLOCKS = numblocks_arr[INDEX];
      }
      return(0);
} /* end InitMMotion */


/*****************************************************************************/
/* Init ADPCM XA Modes                                                       */
/*****************************************************************************/
int InitADPCMXA(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if (dptr->operation == PLAY) {
            revtrk = 0;
            if (dsp_loaded!=IBMXA_DSP) {
              dsp_to_load = IBMXA_DSP;          /* Mode = ADPCMXA Playback   */
              cptr = dsp_string[dsp_to_load];
            }
         }else if(dptr->operation==PLAY_AND_RECORD){
            dptr->rc = NO_PLAY_AND_RECORD;
            return(-1);
         }else if(dptr->operation==RECORD){
            dptr->rc = NO_RECORD;
            return(-1);
         }else{
            dptr->rc = INVALID_REQUEST;
            return(-1);
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
         return(-1);
      }
#endif

      mastercb   = XAMASTER;
      trackcb[0] =  XAT1CB;         /* Init XA addr of track 1 ctrl blk */
      mailbox[0] =  XAMB1;          /* Init XA addr of track 1 mailbox  */
      pcmcb[0]   =  XACB1;          /* Init XA addr of track 1 pcm cb   */
      trackcb[1] =  XAT2CB;         /* Init XA addr of track 2 ctrl blk */
      mailbox[1] =  XAMB2;          /* Init XA addr of track 2 mailbox  */
      pcmcb[1]   =  XACB2;          /* Init XA addr of track 2 pcm cb   */
      cb_already_set = 1;
      if (INDEX == ADPCMXAB) {
         if (CHANNELS_G == 1) {
            TIME_PER_BLOCK = 1066 * SRATE_G / 10;
         } else { /* stereo */
            TIME_PER_BLOCK = 533 * SRATE_G / 10;
         } /* endif */
      } else { /* ADPCMXAC */
         if (CHANNELS_G == 1) {
            TIME_PER_BLOCK = 2132 * SRATE_G / 10;
         } else { /* stereo */
            TIME_PER_BLOCK = 1066 * SRATE_G / 10;
         } /* endif */
      } /* endif */
      return(0);
} /* end InitADPCMXA */


/*****************************************************************************/
/* Init Source Mix Mode                                                      */
/*****************************************************************************/
int InitSourceMix(dptr,trk)
AIP dptr;
int trk;
{
#if NUM_TRACKS > 1
      if(!(initflags[1-trk]&INITED)){        /* Is the other track inited?   */
#endif
         if(dsp_loaded!=IBMPCMP_DSP){
            dsp_to_load = IBMPCMP_DSP;       /* Mode = ADPCM + Record bit*/
            cptr = dsp_string[dsp_to_load];
         }
#if NUM_TRACKS > 1
      }else{                                 /* Other track is active      */
                                             /* Other track compatible and*/
         if (check_compatibility(trk,mode_compat_array[SOURCE_MIX_INDEX])) {
            return(-1);
         } else {
            run_both_trks = 1;     /* defaults to 2 - but only want 1 for */
                                   /* Source Mix Mode                     */
         } /* endif */
      }
#endif
      return(0);
} /* end InitSourceMix */


/*****************************************************************************/
/* Init Claim Hardware Mode                                                  */
/*****************************************************************************/
int InitClaimHardware(trk)
int trk;
{
#if NUM_TRACKS > 1
      if(initflags[1-trk]&INITED){          /* Is the other track inited?   */
#if IS_OS2
         if ((trk_hvdm[1-trk] != trk_hvdm[trk]) ||
             (TRK_HVDM == 0)) {
#endif
            return(-1);
#if IS_OS2
         }
#endif
      }
#endif
      dsp_loaded = -1;
      dsp_to_load = -1;
      return(0);
} /* end InitClaimHardware */



/*****************************************************************************/
/* Status IOCTL                                                              */
/*****************************************************************************/
void DevIOCTLstatus(dptr,trk)
ASP dptr;
int trk;
{
   struct audio_change far *cptr;
   struct track_info far *tptr;
   struct spv2_mode_info_struct far *sptr;
#if NOT_K12
   struct midi_mode_info_struct far *mptr;
#endif
   int   trkcb;

   trkcb = trackcb[trk ^ revtrk];

   dptr->mode = MODE_G;                         /*    a SPV/2 mode         */
   dptr->srate = SRATE_G;                       /*    rate = tempo         */
   dptr->bits_per_sample = BITS_PER_SAMPLE_G;   /*    byte data            */
   dptr->bsize = BSIZE_G;                       /*    block size = 1       */
   dptr->channels = CHANNELS_G;                 /*    1 channel            */
   dptr->operation = OPERATION_G;               /*    operation            */
   cptr=&(dptr->change);                        /*    ->audio_change block */
   cptr->input = acpa_in(mastercb+0);           /*    input                */
   cptr->output = EXTERNAL_SPEAKER+OUTPUT_1;    /*    output               */
   if(ctrlval & SPKREN){                        /*    Is speaker enabled?  */
      cptr->output |= INTERNAL_SPEAKER;         /*     if so, add it       */
   }
   cptr->monitor = acpa_in(mastercb+1);         /*    monitor              */
   cptr->balance = acpa_in(trkcb+3);            /*    balance              */
   cptr->balance_delay = acpa_in(trkcb+4);      /*    balance rate         */
   cptr->volume = acpa_in(trkcb+1);             /*    volume               */
   cptr->volume_delay = acpa_in(trkcb+2);       /*    volume rate          */
   cptr->treble = 0;                            /*    treble               */
   cptr->bass = 0;                              /*    bass                 */
   cptr->pitch = 0;                             /*    pitch                */

   if((MODE_G == SPV2BCPCM) ||                  /* SPV/2                   */
      (MODE_G == SPV2PCM)   ||
      (MODE_G == SPV2NONE)) {
      dptr->flags = TWOS_COMPLEMENT             /*    flags                */
                  | VOLUME | VOLUME_DELAY       /*    volume supported     */
                  | BALANCE | BALANCE_DELAY     /*    balance supported    */
                  | INPUT | OUTPUT;             /*    selects supported    */
      if(sptr = cptr->mode_info) {              /*    mode info            */
         sptr->pitch = acpa_in(SPV2CB);         /*    pitch extraction req'*/
         sptr->num_LPC_coeff = acpa_in(SPV2CB+1);/*   # LPC coefficients   */
         sptr->rate_of_LPC = acpa_in(SPV2CB+2); /*    rate of LPC          */
         sptr->hamming_width = acpa_in(SPV2CB+3);/*   hamming window width */
         sptr->ret_signal = acpa_in(SPV2CB+4);  /*    type of return signal*/
         sptr->slow_down = acpa_in(SPV2CB+5);   /*    slow down factor     */
      }                                         /*                         */
   }else if (MODE_G != MIDI) {               /* WAVEFORM                */
      dptr->flags = FIXED                       /*    flags                */
                  | VOLUME | VOLUME_DELAY       /*    volume supported     */
                  | BALANCE | BALANCE_DELAY     /*    balance supported    */
                  | INPUT | OUTPUT | MONITOR;   /*    selects supported    */
      if(tptr = cptr->dev_info){                /*    dev_info             */
         tptr->master_volume = acpa_in(trkcb);  /*    master volume        */
      }
#if NOT_K12                                                          /*61491*/
   } else { /* MIDI MODE */
      dptr->srate = TEMPO;                      /*    rate = tempo         */
      dptr->flags = VOLUME | VOLUME_DELAY       /*    volume supported     */
                  | BALANCE | BALANCE_DELAY     /*    balance supported    */
                  | INPUT | OUTPUT | MONITOR;   /*    select's supported   */
      if(tptr = cptr->dev_info){                /*    dev_info             */
         tptr->master_volume = acpa_in(trkcb);  /*    master vol           */
      }
      if (APP_VERSION >= VERSION_01000005) {
         mptr=dptr->change.mode_info;
         mptr->tempo = TEMPO;
         mptr->cpqn = PPQN;
      } /* endif */
#endif
   }
}


/*****************************************************************************/
/* Load DSP IOCTL                                                            */
/*****************************************************************************/
void DevIOCTLload(buf,size,flags,trk)
unsigned char far *buf;
unsigned long size,flags;
int trk;
{
   static int  load_addr,odd_size;        /* These MUST be static!         */
   static unsigned char odd_byte;         /* This MUST be static!          */
   unsigned long x;
   unsigned int far *ptr;                 /* int ptr to buffer             */
   unsigned int w1;                       /* Build 1st word in this var    */
   unsigned int y;

   _asm cli;                              /* don't let int hndlr in        */

   acpalock++;                            /* Sieze control of ACPA         */

#if NUM_TRACKS > 1
   if(trk_loading && (1<<trk)){           /* Can this trk load code?       */
#endif
      if(flags & LOAD_START){             /* Is this the first block?      */
#if NUM_TRACKS > 1
                                          /*                               */
         trk_loading = (1<<trk)+0x8000;   /* Make sure that this is the    */
                                          /* only trk that is marked for   */
                                          /* loading once it has started   */
                                          /* and set high bit to show it is*/
                                          /* in the process of loading     */
#endif
         load_addr = 0;                   /* If so, reset address to start */
         acpa_command(0);                 /* Disable ACPA (TMSRES = 0)     */

         if(size < DSP_HEADER_L){         /* Ooops, can't do it!           */
            if(--acpalock==0) _asm sti;
            return;
         }

         /* Determine which load module is in */
         dsp_loaded = dsp_to_load;        /* Indicate that MIDI mod loaded */
         dsp_to_load = -1;                /* Flag that it has been loaded  */
                                          /* !!!! Makes a dangerous assumption*/

         ptr = (unsigned int far *)(buf + DSP_HEADER_L);  /* Skip over header */
         x = size-DSP_HEADER_L;           /* Set x = # bytes to write      */
         if(x & 1){                       /* Is this an odd number of bytes?*/
            odd_size = 1;                 /* If so, set odd flag           */
         }else{
            odd_size = 0;
         }
         x /= 2;                          /* Convert bytes to words        */
         x--;
         w1 = *ptr++;                     /* Get first word to write       */

      }else{
         if(odd_size){
            w1 = *buf++;
            w1 <<= 8;
            w1 += odd_byte;
            ptr = (unsigned int far *)buf;   /* No header to skip          */
            size--;
         }else{
            ptr = (unsigned int far *)buf;   /* No header to skip          */
            w1 = *ptr++;
            size -= 2;
         }
         x = size/2;
         if(size & 1) odd_size = 1;       /* Are we still odd?          */
         else odd_size = 0;
      }

      outpw(hostbase+4,load_addr);        /* Set address -> to zero        */
      load_addr += x+1;                   /*  and update it for next time  */
      outpw(hostbase,w1);                 /* Write first word to ACPA      */
      while(x--){                         /* Write all the rest to ACPA    */
         outpw(hostbase,*ptr++);
      }
      if(odd_size){
         buf = (char far *)ptr;
         odd_byte = *buf;                 /* Save last byte for next call  */
      }

      if(flags & LOAD_END){

#if NOT_K12
         if (dsp_loaded == IBMAUDS_DSP) {
            pspdataloc = AUDS_PSPDATALOC;
         } else {
            pspdataloc = MPC_PSPDATALOC;
         } /* endif */
#endif


#if NUM_TRACKS > 1
         trk_loading = 0;                 /* Reset trk_loading now that load */
                                          /* is finished.                    */
         if (run_both_trks) {
            dsp_run(0);
            if (run_both_trks == 2) {
               dsp_run(1);
            } /* endif */
         } else {
#endif
            for (y=0; y<8; y++) {
               acpa_out(mastercb+y,0x0000);
            } /* endfor */
            dsp_run(trk);
#if NUM_TRACKS > 1
         } /* endif */
#endif
      }
#if NUM_TRACKS > 1
   } /* endif */
#endif

   if(--acpalock == 0) _asm sti;          /* Ints back on (if as before)   */
}


void dsp_run(trk)
int trk;
{
   int tms_mail;
#if NUM_TRACKS > 1
   char set_volume;
#endif

   #if TRACE_ON
   trace(0x0B);
   #endif
   if (MODE_G == CLAIM_HDWR) {
      acpa_command(SPKREN+TMSINT+HINTENA+TMSRES); /* Enable ACPA            */
      tms_mail = 0;
   } else {
      if(!(ctrlval & TMSRES)){               /* Is DSP already running?      */
         acpa_out(mailbox[0],0x55aa);
         acpa_command(SPKREN+TMSINT+HINTENA+TMSRES);  /* Enable ACPA         */
         while(acpa_in(mailbox[0])==0x55aa);
      } /* endif */
#if NUM_TRACKS > 1
      if ((run_both_trks) &&
          (saved_cbs.trk == trk)) {
         restore_cbs(trk);
         set_volume = 0;
      } else {
         set_volume = 1;
#endif
         clr_control_blocks(trk);
#if NUM_TRACKS > 1
      } /* endif */
#endif
#if IS_K12
      if(MODE_G==ADPCM){
#else
      if(MODE_G == MIDI){                    /* MIDI?                        */
         SetClock(trk);
         acpa_command(SPKREN+TMSINT+HINTENA+TMSRES); /* Enable ACPA             */
         tms_mail = 0;
      }else if(MODE_G == ADPCM){             /* ADPCM Playback               */
#endif

         acpa_out(trackcb[trk^revtrk]+5,MODEID);   /* Set ADPCM mode          */
         acpa_out(mailbox[trk^revtrk]+4,0);     /* Buffer count = 0        */
         acpa_clr_block(PBUFSTART,PBLOCKSIZE*NUMBLOCKS,0);   /* Zero buffers   */
      }else if(MODE_G == PCM                 /* PCM                           */
         || MODE_G == ADPCMXA
         || MODE_G == MU_LAW
         || MODE_G == SOURCE_MIX
         || MODE_G == A_LAW){
         acpa_out(trackcb[trk^revtrk]+5,0);        /* ADPCM mode not used        */
         acpa_out(mailbox[trk^revtrk]+4,0);        /* Buffer count = 0       6691*/
         if(MODE_G==MU_LAW)
            acpa_out(pcmcb[trk^revtrk]+1,2);       /* PCM type 2 = Mu-Law        */
         else if(MODE_G==A_LAW)
            acpa_out(pcmcb[trk^revtrk]+1,3);       /* PCM type 3= A-Law          */
         else
            acpa_out(pcmcb[trk^revtrk]+1,0);       /* PCM type 0=normal          */
         acpa_out(pcmcb[trk^revtrk]+2,(unsigned)SRATE_G); /* Sample rate          */
         acpa_out(pcmcb[trk^revtrk]+3,33);         /* Dither rate                */
         acpa_out(pcmcb[trk^revtrk]+4,(unsigned)BITS_PER_SAMPLE_G); /* bps (8/16) */
         acpa_out(pcmcb[trk^revtrk]+5,0);          /* byte swapping off          */
         acpa_out(pcmcb[trk^revtrk]+8,CHANNELS_G);  /* # channels                 */
         acpa_out(pcmcb[trk^revtrk]+11,0);             /* Mix off                    */

         if(FLAGS_G & TWOS_COMPLEMENT) {
            acpa_out(pcmcb[trk^revtrk]+6,0);       /* 2's complement      */
         } else if(FLAGS_G & SIGNED) {
            acpa_out(pcmcb[trk^revtrk]+6,1);       /* signed data         */
         } else {
            acpa_out(pcmcb[trk^revtrk]+6,2);       /* unsigned data       */
         } /* endif */

         if(BITS_PER_SAMPLE_G==8){
            if (MODE_G==MU_LAW || MODE_G==A_LAW){  /* If ulaw or alaw,     */
               CLRCHAR = 0xffff;
            } else if ((FLAGS_G & SIGNED) ||    /* ZERO = 0x00 if signed/2's  */
                   (FLAGS_G & TWOS_COMPLEMENT)) {
               CLRCHAR = 0x0000;
            } else CLRCHAR = 0x8080;
         }else if(BITS_PER_SAMPLE_G==16){
            if ((FLAGS_G & SIGNED) ||           /* ZERO = 0x00 if signed/2's  */
               (FLAGS_G & TWOS_COMPLEMENT)) {
               CLRCHAR = 0;
            } else CLRCHAR = 0x8000;
         }else{
            CLRCHAR = 0;
         }
         acpa_clr_block(PBUFSTART,PBLOCKSIZE*NUMBLOCKS,CLRCHAR);
         if (MODE_G == SOURCE_MIX) {
            acpa_out(pcmcb[trk^revtrk]+11,1);        /* Mix On        */
         } /* endif */

      }else if((MODE_G == SPV2BCPCM)         /* Speech Viewer/2               */
            || (MODE_G == SPV2PCM)
            || (MODE_G == SPV2NONE)){
         acpa_out(trackcb[trk^revtrk]+5,0);     /* ADPCM mode not used        */
         if (PBUFSTART != RBUFSTART) {
            acpa_clr_block(PBUFSTART,PBLOCKSIZE*(NUMBLOCKS/2),0); /* Zero buffers   */
            acpa_clr_block(RBUFSTART,RBLOCKSIZE*(NUMBLOCKS/2),0); /* Zero buffers   */
         } else {
            acpa_clr_block(PBUFSTART,PBLOCKSIZE*NUMBLOCKS,0); /* Zero buffers   */
         } /* endif */
      } /* endif */

      if(OPERATION_G==RECORD){
         tms_mail = 1;               /* Kick off recording            */
      }else if(OPERATION_G==PLAY){
         tms_mail = 5;               /* Kick off playback             */
      } else {
         tms_mail = (int)OPERATION_G;
      } /* endif */
   } /* endif */

#if NUM_TRACKS > 1
   if (set_volume) {
#endif
      DevTrkVolume(0x7fff,0,trk);
      DevTrkBalance(0x4000,0,trk);
      DevMstVolume(0x7fff,0,trk);
#if NUM_TRACKS > 1
   } /* endif */
#endif

   if (tms_mail) acpa_out(mailbox[trk^revtrk],tms_mail); /* Start-up ACPA   */
   #if TRACE_ON
   trace(0xBB);
   #endif
}

void clr_control_blocks(int trk)
{
   int x;

   for (x=0; x<8; x++) {
     acpa_out(trackcb[trk^revtrk]+x,0);
     acpa_out(mailbox[trk^revtrk]+x,0);
   } /* endfor */
}
/*****************************************************************************/
/* Diagnostic IOCTLs                                                         */
/*****************************************************************************/
void DevIOCTLread8(dptr)
DR8 dptr;
{
   dptr->data=(unsigned char)inp(hostbase+dptr->offset); /* Read data from reg*/
}

void DevIOCTLwrite8(dptr)
DR8 dptr;
{
   outp(hostbase+dptr->offset,dptr->data);   /* Write data to port         */
}

void DevIOCTLread16(dptr)
DR16 dptr;
{
   dptr->data=inpw(hostbase+dptr->offset);   /* Read data from reg         */
}

void DevIOCTLwrite16(dptr)
DR16 dptr;
{
   outpw(hostbase+dptr->offset,dptr->data);  /* Write data to port         */
}

/*****************************************************************************/
/* Wait for Buffers to empty -                                               */
/*****************************************************************************/
void DevIOCTLwait(int trk)
{
   /* Want to wait until the DSP code has played all the data in it's buffers */
   /* So wait until the DSP underruns                                         */
   while (!(((IOB)XMITIO.Ptr)->runflags & IOB_UNDERRUN)) {
      /* Wait until xmit queue has underrun */
      _asm sti
      INITFLAGS |= WAITING;       /* Set waiting flag           */
#if IS_OS2
      Block((long)((char far *)&XMITIO),100l,0); /* Block this thread*/
#endif
      INITFLAGS &= ~WAITING;      /* Clear waiting flag         */
   }
}

/*****************************************************************************/
/* Track Volume                                                              */
/*****************************************************************************/
void DevTrkVolume(level,duration,trk)
int level,duration,trk;
{
   acpa_out(trackcb[trk^revtrk]+1,level);
   acpa_out(trackcb[trk^revtrk]+2,duration);
}

/*****************************************************************************/
/* Track Balance                                                             */
/*****************************************************************************/
void DevTrkBalance(balance,duration,trk)
int balance,duration,trk;
{
   /* Reverse left and right for MIDI mode until DSP is fixed      */
   if (MODE_G == MIDI) {
      balance = 0x7fff-balance;
   } /* endif */

   acpa_out(trackcb[trk^revtrk]+3,balance);
   acpa_out(trackcb[trk^revtrk]+4,duration);
}

/*****************************************************************************/
/* Master Volume                                                             */
/*****************************************************************************/
void DevMstVolume(level,duration,trk)
int duration,trk;
unsigned int level;
{

   unsigned long ullevel;

   if(MODE_G==PCM
   || MODE_G==MU_LAW
   || MODE_G==ADPCMXA
   || MODE_G==A_LAW){
      /* convert level to 0-100 scale */
      ullevel = (unsigned long)level;
      ullevel = (ullevel*100)/0x7fff;
      level = (unsigned int)ullevel;
      /* make sure doesn't go above 116 max */
      if (level > 116)
         level = 116;
   }else{
      if (level > 0x7fff)
         level = 0x7fff;
   }
   acpa_out(trackcb[trk^revtrk],level);
}


/*****************************************************************************/
/* Device Open                                                               */
/*****************************************************************************/
int DevOpen(trk)
int trk;
{
   acpalock = 0;
   opencounter++;
   return(0);
}

/*****************************************************************************/
/* Device Close                                                              */
/*****************************************************************************/
int DevClose(trk)
int trk;
{
   int at_mail;

#if NUM_TRACKS > 1
   data_processed[trk] = 0;
#endif
#if IS_OS2
   if ((ctrlval & TMSRES) && (MODE_G != MIDI) && (!avc103_mode)) {
#else
   if ((ctrlval & TMSRES) && (MODE_G != MIDI)) {
#endif
      acpa_out(mailbox[trk^revtrk],0);       /* Stop operation */
      while(at_mail = acpa_in(mailbox[trk^revtrk]+2));
   } /* endif */

   if( (dsp_use_count &= ~(1<<trk)) == 0){   /* are we the last user?      */
      acpa_command(0);                    /* Shut-down ACPA                */
      FreeDevInt();                       /* free interrupt vector         */
#if NOT_K12
#if IS_WIN
#else
      if (do_passthru) {
         dsp_loaded = PASSTHRU_DSP;
         passthru(0x108);
      } /* endif */
#endif
#endif
   }
   return(0);
}

DevWrite(data,trk)
int   data,trk;
{

   return(0);
}

/*****************************************************************************/
/* ACPA Out                                                                  */
/*****************************************************************************/
void acpa_out(addr,data)
unsigned int addr,data;
{
   _asm cli;
   acpalock++;
   _asm {
      mov   dx,hostbase
      add   dl,4
      mov   ax,addr
      out   dx,ax
      mov   ax,data
      sub   dl,4
      out   dx,ax
   }
   if(--acpalock == 0) _asm sti;
}

/*****************************************************************************/
/* ACPA In                                                                   */
/*****************************************************************************/
unsigned int acpa_in(addr)
unsigned int addr;
{
   int data;

   _asm cli;
   acpalock++;
   _asm {
      mov   dx,hostbase
      add   dl,4
      mov   ax,addr
      out   dx,ax
      sub   dl,4
      in    ax,dx
      mov   data,ax
   }
   if(--acpalock == 0) _asm sti;
   return(data);
}

/*****************************************************************************/
/* ACPA Write Block                                                          */
/*****************************************************************************/
void acpa_wblock(addr,data,len)
unsigned int addr,len;
char far *data;
{
   if(len==0) return;
   _asm cli;
   acpalock++;
   _asm {
      push  si
      cld
      mov   dx,hostbase
      add   dl,4
      mov   ax,addr                    /* Set address                      */
      out   dx,ax
      sub   dl,4
      mov   cx,len                     /* Set length (# words)             */
      push  ds
      lds   si,data                    /* Get data pointer                 */
#if IS_OS2
    rep   outsw                      /* Write it                         */
#else
   wloop:
      lodsw
      out   dx,ax
      loop  wloop
#endif
      pop   ds
      pop   si
   }
   if(--acpalock==0) _asm sti;
}

/*****************************************************************************/
/* ACPA Clear Block                                                          */
/*****************************************************************************/
void acpa_clr_block(addr,len,data)
unsigned int addr,len,data;
{
   _asm cli;
   acpalock++;
   _asm {
      mov   dx,hostbase
      add   dl,4
      mov   ax,addr                    /* Set address                      */
      out   dx,ax
      sub   dl,4
      mov   cx,len                     /* Set length (# words)             */
      mov   ax,data
   acb_loop:
      out   dx,ax                      /* Write it                         */
      loop  acb_loop
   }
   if(--acpalock==0) _asm sti;
}

/*****************************************************************************/
/* ACPA Read Block                                                           */
/*****************************************************************************/
void acpa_rblock(addr,data,len)
unsigned int addr,len;
char far *data;
{
   if(len==0) return;
   _asm cli;
   acpalock++;
   _asm {
      push  di
      cld
      mov   dx,hostbase
      add   dl,4
      mov   ax,addr                    /* Set address                      */
      out   dx,ax
      sub   dl,4
      mov   cx,len                     /* Set length (# words)             */
      push  es
      les   di,data                    /* Get data pointer                 */
#if IS_OS2
      rep   insw                       /* Write it                         */
#else
   rloop:
      in    ax,dx
      stosw
      loop  rloop
#endif
      pop   es
      pop   di
   }
   if(--acpalock==0) _asm sti;
}

/*****************************************************************************/
/* ACPA Status                                                               */
/*****************************************************************************/
int acpa_status()
{

   return(inp(hostbase+6));
}

/*****************************************************************************/
/* ACPA Command                                                              */
/*****************************************************************************/
void acpa_command(command)
char command;
{
   ctrlval = command;
   outp(hostbase+6,ctrlval);
}


/*****************************************************************************/
/* Device Initialization                                                     */
/*****************************************************************************/
#if IS_WIN
int DevInit()
{
#else
int DevInit(parms)
char far *parms;
{
#endif
   int x,found;
   unsigned char posdata;
   int error_flag = 0;
   extern char midinum;
#if NUM_TRACKS > 1
#if NOT_OS2
   extern char midinum2;
#endif
#endif
#if NOT_WIN
   GetParms(parms);
#endif

#if NOT_K12
   midi_cb_init();
#endif

#if NOT_OS2
   startmsg2[CARD_NUM_POS] = (char)('0'+cardnum);
#endif

   /* Determine base I/O address of card */
   for(x=8, found=0; x<16 && found<cardnum; x++){
      outp(POS_SEL,x);                                /* Select next slot  */
      if( ((inp(POS1)<<8) + inp(POS0)) == ACPA_ID){
         found++;
         slot_num = x-7;                              /* Remember slot #   */
      }
   }
   if(found==cardnum){
      posdata = (unsigned char)inp(POS2);
      irqnum = posirqs[(posdata>>3) & 7];
      if(oldcard){
#if NOT_OS2
         startmsg2[OLD_POS] = 'O';
         startmsg2[OLD_POS+1] = 'l';
         startmsg2[OLD_POS+2] = 'd';
#endif
         hostbase = oldposlocs[ (posdata >> 1) & 3];
      }else{
         hostbase = poslocs[ (posdata >> 1) & 3];
      }
   }
   outp(POS_SEL,0);                       /* Deselect POS selects */

   if(found!=cardnum){                    /* Try the family 1 locations   7291*/
      found = 0;
      for(hostbase=0x300; hostbase<0xfff8 && found!=cardnum; hostbase+=8){
         if(inp(hostbase+7) == 0x6c){
            found++;
            irqnum =  (~(acpa_status() >> 4)) & 7;
         }
      }
      hostbase-=8;
   }
#if IS_OS2
   nszMsgNum[0] = ACPA_OK_INSTALL_MSG;
#endif
   /* Make sure we found a card */
   if(found!=cardnum){
        error_flag = -1;        // Return failure
#if IS_OS2
        nszMsgNum[0] = ACPA_FAIL_INSTALL_MSG;
#endif

   }
#if NOT_OS2
   if (irqnum > 9) {
      startmsg2[IRQ_NUM_POS] = (char)('1');
      startmsg2[IRQ_NUM_POS+1] = (char)('0'+irqnum-10);
   } else {
      startmsg2[IRQ_NUM_POS] = (char)('0'+irqnum);
      startmsg2[IRQ_NUM_POS+1] = (char)('.');
      startmsg2[IRQ_NUM_POS+2] = (char)(' ');
   } /* endif */

#else
   /* Determine if this is OS/2 Version 2.0+ */
   if (os2_version < 20) {

      //*************************
      // Get message from AUDIODD.MSG
      //*************************
      DosGetMessage ((PCHAR FAR *)IvTable, 1,
                     nszMsgBuff, MAXMSGLEN,
                     *nszMsgNum, nszMsgFile, &MsgLen);

      //*******************************************
      // If the message file ever changes the below
      // code will need to be changed to reflect it
      // Setup: card #, device #, interrupt level
      //*******************************************
      if (!error_flag)     {
            nszMsgBuff[AUDIO_NUM_POS] = midinum;
         if (irqnum > 9) {
            nszMsgBuff[IRQ_NUM_POS] = (char)('1');
            nszMsgBuff[IRQ_NUM_POS+1] = (char)('0'+irqnum-10);
         } else {
            nszMsgBuff[IRQ_NUM_POS] = (char)('0'+irqnum);
            nszMsgBuff[IRQ_NUM_POS+1] = (char)('.');
            nszMsgBuff[IRQ_NUM_POS+2] = (char)(' ');
         } /* endif */
      }

      nszMsgBuff[CARD_NUM_POS] = (char)('0'+cardnum);

      DosPutMessage (STDOUT, MsgLen, nszMsgBuff);

   } /* endif */

#endif

#if NOT_K12
#if IS_WIN
#else
   if (!error_flag && do_passthru) {
      acpa_command(0);   /* Make sure ACPA is shut down    */
      dsp_loaded = PASSTHRU_DSP;
      passthru(0x108);
   } /* endif */
#endif
#endif

   return (error_flag);
}

#if IS_WIN
#else
/************************************************************************/
/* This routine will parse the parameters from the CONFIG.SYS.          */
/*                                                                      */
/* Expected parameters are:                                             */
/*                                                                      */
/*    DEVICE=d:\path\ACPADD.SYS [1-9] [o] [A-D]                         */
/*       where 1-9 specifies the AUDIOx$ device name number             */
/*             o   specifies to use the old ACPA card I/O addresses     */
/*             A-D specifies to use the 1st (A) through 4th (D) card    */
/*             P   specifies want to use passthru                       */
/*                                                                      */
/*      Written by Ron Lisle, Multi Media Products, Austin, TX.         */
/************************************************************************/
void GetParms(parms)
char far *parms;
{
#if IS_OS2
    extern  int trk;
#endif
    extern unsigned int max_num_opens;

   extern char midinum;
#if NUM_TRACKS > 1
#if NOT_OS2
   extern char midinum2;
#endif
#endif
    char    sDigit[2];
    unsigned short usMaxNumStreams = 0;
   char ch;
#if IS_OS2
   extern char audioname[8];
   unsigned short x;
   unsigned short registered = 0;
#endif
   //**************************
   // Set 1 as default midinum
   // Overrides default in the
   // device header (OS2DD.ASM)
   //************************** */
#if IS_OS2
   midinum = '0';
#else
   midinum = '1';
#endif

   while((ch = *parms++)!=10 && ch!=13){
      /* find DOS delimiter */
      if(ch==0 || ch==' ' || ch==',' || ch==';' || ch=='+' || ch=='=' || ch=='\t'){
         break;
      }
   }
// if(ch==10 || ch==13 || ch==0) return;
   if(ch!=10 && ch!=13 && ch!=0) {

      while((ch = *parms++)!=10 && ch!=13 && ch!=0){
         /* Look for AUDIO# */
#if NUM_TRACKS > 1
#if NOT_OS2
         if(ch>='1' && ch<='8'){
#else
         if(ch>='1' && ch<='9'){
#endif
#else
         if(ch>='1' && ch<='9'){
#endif
            midinum=ch;
#if NOT_OS2
#if NUM_TRACKS > 1
            midinum2=(char)((int)midinum+1);
            startmsg2[AUDIO_NUM_POS2] = (char)((int)ch+1);
#endif
            startmsg2[AUDIO_NUM_POS] = ch;
#endif
         }
         /* Now look for old flag 'o' */
         if(ch=='o' || ch=='O'){
            oldcard = 1;

            /********************************************************/
            /* Stream information will be a parameter on CONFIG.SYS */
            /* and will be in the format:     S:nn                  */
            /********************************************************/
         }  else if (ch=='s' || ch=='S')  {

              if (*parms == ':')                      // skip over colon
                      parms++;
              sDigit[0] = *(parms)++;                 // get first digit

              if (*parms >= '0' && *parms <= '9')
                    sDigit[1] = *parms;               // get second digit
              else
                    sDigit[1] = '\0';

              //********************************************
              // Convert ASCII to integer from right to left
              //********************************************
              if (sDigit[1] == '\0')    {            // single digit
                    usMaxNumStreams = sDigit[0] & 0x000F;
              } else {
                    usMaxNumStreams = sDigit[1] & 0x000F;
                    usMaxNumStreams += (sDigit[0] & 0x000F)*10;
              }
              max_num_opens = usMaxNumStreams;
              parms++;

#if NOT_K12
         }  else if (ch=='p' || ch=='P')  {
            do_passthru = 1;
#endif
         }
         /* Now look for multiple card letter A-Z */
         if(ch>='A' && ch<='D'){
            cardnum = 1+ch-'A';
         }
         if(ch>='a' && ch<='d'){
            cardnum = 1+ch-'a';
         }
      }

   } /* if ch != ... */

#if IS_OS2
   /* Check to see if audio number was specified or not          */
   if (midinum != '0') {
      if (AttachDD(audioname,(char *)&ddaddr))
         registered = 1;
   } /* endif */
   /* If not already registered, then try to find a number to use */
   if (!registered) {
      /* No number specified so loop trying to find one not used yet */
      for (x=1; x<=9; x++) {
         midinum = (char)('0'+x);
         if (AttachDD(audioname,(char *)&ddaddr)) {
            break;
         } /* endif */
      } /* endfor */
   } /* endif */

   //*****************************************************************
   // If stream info set as a parameter, setup AcpaMME usMaxNumStreams
   // field, otherwise leave the default value of usMaxNumStreams
   //***************************************************************** */
   if (usMaxNumStreams)    {
        AcpaMME.usMaxNumStreams = usMaxNumStreams;
   }
#endif
}
#endif

/*****************************************************************************/
/*****************************************************************************/
/** Position Calculations                                                   **/
/*****************************************************************************/
/*****************************************************************************/

#if NOT_K12
/*****************************************************************************/
/* UpdateMIDIPosition                                                        */
/*****************************************************************************/
unsigned long UpdateMIDIPosition(trk)
int trk;
{

   return(((POS_BUFFER_COUNT * 600000)/TIME_PER_BLOCK) + ROLLOVER_TIME);

} /* end UpdateMIDIPosition(trk) */
#endif


/*****************************************************************************/
/* UpdatePosition                                                            */
/*****************************************************************************/
unsigned long UpdatePosition(trk)
int trk;
{

   POS_BUFFER_COUNT++;      /* Update our position counter */
   return(ROLLOVER_TIME + (TIME_PER_BLOCK * POS_BUFFER_COUNT / SRATE_G));

} /* end UpdatePosition */


/*****************************************************************************/
/* PositionNotCountedYet                                                     */
/*****************************************************************************/
unsigned long PositionNotCountedYet(trk)
int trk;
{
   if (OPERATION_G == PLAY) {

#if NOT_K12
      if (MODE_G == MIDI)
         return((DELAY * 600000) / TIME_PER_BLOCK);
#endif

      return((HOST_BUFFER_COUNT - PRV_BUFFER_COUNT) * TIME_PER_BLOCK / SRATE_G);

   } else {

      return(0);

   } /* endif */

} /* end PositionNotCountedYet */



/*****************************************************************************/
/* CheckRolloverCount                                                        */
/*****************************************************************************/
void CheckRolloverCount(trk)
int trk;
{

   if (POS_BUFFER_COUNT > MAX_BUFFER_COUNT) {
      POS_BUFFER_COUNT -= MAX_BUFFER_COUNT;
      ROLLOVER_TIME += TIME_PER_BLOCK * MAX_BUFFER_COUNT / SRATE_G;
   }

} /* end CheckRolloverCount */


#if IS_OS2

#if TRACE_ON
//write number out to ACPA ID register - which is read only
void trace (unsigned char trace_point) {
   _asm {
      mov   dx,hostbase
      add   dl,7
      mov   al,byte ptr trace_point
      out   dx,ax
   }
}
#endif

#endif
