/***************************************************************************/
/*                   MPU-401 Device Specific Functions                     */
/*                                                                         */
/*    DevInit()            Initialize Device                               */
/*    DevOpen()            Open device, hook ints, etc.                    */
/*    DevClose()           Close device, free ints, etc.                   */
/*    DevWrite(data,trk)   Write data                                      */
/*    DevNoteOn()                                                          */
/*    DevNoteOff()                                                         */
/*    DevAllNotesOff()     Make sure all notes are off                     */
/*    DevAfterTouch()                                                      */
/*    DevControlChange                                                     */
/*    DevProgramChange                                                     */
/*    DevChannelPressure()                                                 */
/*    DevPitchBend                                                         */
/*    DevMTC()                                                             */
/*    DevSysEx()                                                           */
/*    DevIOCTLinit(dptr)   Process IOCTL init functions                    */
/*    DevIOCTLstatus(dptr) Return status                                   */
/*    DevIOCTLload()       Load DSP module                                 */
/*    DevIOCTLread8()      diagnostic read byte from ACPA port             */
/*    DevIOCTLwrite8()     diagnostic write byte to ACPA port              */
/*    DevIOCTLread16()     diagnostic read word from ACPA port             */
/*    DevIOCTLwrite16()    diagnostic write word to ACPA port              */
/*    DevIOCTLwait()       Wait routine                                    */
/*    SetClock()           Called when Tempo or PPQN changes               */
/*    DevInt()             (optional)                                      */
/*    TimerInt()           (optional)                                      */
/*  MPU-IMC Interrupt Handler and device specific routines                 */
/*  This routine is invoked by Receive and Timer Interrupts.               */
/*  It will update position, perform any pending synchronized output, and  */
/*  read any input data.                                                   */
/***************************************************************************/

#include <conio.h>

#include <audiodd.h>
#include <audiodd2.h>
#include <auddef.h>
#include <audproto.h>
#include <audmsg.h>

#define TRK 0

/* MPU port definitions */
#define  MPU_REC_RDY       0x80
#define  MPU_XMIT_RDY      0x40
#define  MPUBASE           0x330
#define  MPU_DATA_PORT     0x330
#define  MPU_COMMAND_PORT   0x331
#define  MPU_STATUS_PORT   0x331

#define  TIMER             0x40


/* Function declarations */
int DevRead(void);
void GetParms(char far *parms);

/* External data references */
extern unsigned int midiflags,timingflags;
extern int initflags;
extern int ppqncntr,ppqndiv;
extern struct vscb recio,xmitio;
extern unsigned char *prvtimemsg,*prvtimecounthi,*prvtimecountlo;
extern long operation;
extern long delay;


/* Device Specific Data */
int   irqnum = 2;

unsigned char qcaps_sysex[10] = { 0xf0,0,0,0x3a,5,1,0,0x10,0,0xf7 };
unsigned char qid_sysex[10] = { 0xf0,0,0,0x3a,5,4,'M','P','U',0xf7 };


#define  NUM_MODES   2
int   num_modes = NUM_MODES;
#define  CLAIM_HDWR_MODE   1
#define  UNKNOWN_MODE      NUM_MODES
struct mode_data mode_table[NUM_MODES] = {
   MIDI,0,0,8,8,0,0,0,0,1,DATAFLAGS+BPS+SRATE+CHANNELS+BSIZE,      /* MIDI_MODE   */
   CLAIM_HDWR,0,0,0,0,0,0,0,0,0,DATAFLAGS+BPS+SRATE+CHANNELS+BSIZE /* Get hardware*/
};


/****************************/
/* Device Interrupt Handler */
/****************************/
int DevInt()
{
   int data,count;

   if(inp(MPU_STATUS_PORT) & MPU_REC_RDY){   /* Is this our int?           */
      eoi(irqnum);                  /* Must have been cleared in DevWrite! */
      return(-1);                            /* If not, return -1          */
   }

   do{                                    /* Loop until all data read      */
      data = inp(MPU_DATA_PORT);          /*  Read data from MPU           */
      if(!rec_running(TRK)){              /* If receive isn't running      */
         /* don't do anything */          /* Then throw data away          */
      }else if(data==0xf8){               /* Is this a MIDI timing clock?  */
         if(TIMINGFLAGS & 0x4000){ /* Is external clocking enabled?  */
            if(TIMINGFLAGS & 0x400){ /* Is timing compression enabled? */
               QTimeSX(TRK);
            }else{
               *((IOB)RECIO.Ptr)->head = 0xf8;           /* Queue up a MIDI clock      */
               if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
                  ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
               }
               if(++((IOB)RECIO.Ptr)->count >= ((IOB)RECIO.Ptr)->buf[0].length){ /* Limit counter to 512    */
                  ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;
               }
            }
         }
      }else if(data!=0xfe){                  /* Not an active sense?          */
         if(MIDIFLAGS & 1){             /* Is time sysex last in queue?  */
            if(*PRVTIMEMSG==0xf0){           /* Verify it's still valid       */
               count = *PRVTIMECOUNTLO + (*PRVTIMECOUNTHI << 7);
               if(count < 3){             /* Is count < 3?              */
                  ((IOB)RECIO.Ptr)->count -= 8;      /* Remove sysex from queue    */
                  ((IOB)RECIO.Ptr)->head = PRVTIMEMSG;   /* Move head pointer back     */
                  while(count--){         /* Convert sysex to F8's      */
                     *((IOB)RECIO.Ptr)->head = 0xf8;     /* Convert to timing clocks   */
                     if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
                        ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;   /*  wrap pointer if needed    */
                     }
                     if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){ /* Increment rec counter      */
                        ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;   /* Limit it to 512            */
                     }
                  }
               }
            }
         }
         *((IOB)RECIO.Ptr)->head = (unsigned char)data;
         if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){
            ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
         }
         if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){
            ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;
         }
      }
   }while(!(inp(MPU_STATUS_PORT) & MPU_REC_RDY)); /* Loop until all data read*/

   MIDIFLAGS &= 0xfffe;                 /* Clear timing sysex queued flag*/
   eoi(irqnum);
   return(0);                                /* and return 0                  */
}

/***************************/
/* Timer Interrupt Handler */
/***************************/
int TimerInt()                   /* Invoked by the Timer 0 interrupt handler  */
{
   if(!is_running(TRK)){                     /* If neither rec nor xmit running*/
      return(0);                             /*  then nothing to do           */
   }
   if(--PPQNCNTR == 0){                 /* Prescale ppqn counter         */
      PPQNCNTR = PPQNDIV;
      ((IOB)XMITIO.Ptr)->position++;         /* increment system time         */
      ((IOB)RECIO.Ptr)->position++;          /*  in both buffers              */
      if( (((IOB)XMITIO.Ptr)->count)         /* Has xmit buffer been updated? */
       && !(MIDIFLAGS & OUTSYNC)){      /*  without us knowing?          */
         MIDIFLAGS |= OUTSYNC;          /* If so, then make sure that    */
         DELAY = 1;                     /*  we process it                   */
      }
      if(MIDIFLAGS & OUTSYNC){          /* Is output synch pending?      */
         if(--DELAY <= 0){              /* Time to do updates?           */
            DoOutput(TRK);
#if IS_OS2
            /* If thread is blocked on room in xmit buffer,                   */
            if(MIDIFLAGS & OUTPUT_WAITING){
               DevHlp_Run((long)((char far *)&MIDIFLAGS));/* unblock it  */
            }
            if(INITFLAGS & WAITING){ /* If thread is blocked on WAIT  */
               DevHlp_Run((long)((char far *)&XMITIO));  /* Then unblock it*/
            }
#endif
         }
      }
   }
   if(TIMINGFLAGS & 0x1000){         /* Are input timing clocks enabled? */
      if(TIMINGFLAGS & 0x400){       /* Is timing compression enabled?   */
         QTimeSX(TRK);                    /*  if so, go queue a timeing sysex */
      }else{
         *((IOB)RECIO.Ptr)->head = 0xf8;               /* Queue a MIDI timing clock       */
         if(++((IOB)RECIO.Ptr)->head >= ((IOB)RECIO.Ptr)->buf[0].Virt+((IOB)RECIO.Ptr)->buf[0].length){    /* Time to wrap?         */
            ((IOB)RECIO.Ptr)->head = ((IOB)RECIO.Ptr)->buf[0].Virt;
         }
         if(++((IOB)RECIO.Ptr)->count > ((IOB)RECIO.Ptr)->buf[0].length){
            ((IOB)RECIO.Ptr)->count = ((IOB)RECIO.Ptr)->buf[0].length;
         }
      }
   }
   if(TIMINGFLAGS & 0x2000){         /* Are output clocks enabled?      */
      DevWrite(0xf8,TRK);                 /*  if so, go write one            */
   }
    return(0);
}

/***********/
/* Note On */
/***********/
void DevNoteOn(chan,key,vel)
char  chan,key,vel;
{
   DevWrite(0x90+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(vel,TRK);
}

/************/
/* Note Off */
/************/
void DevNoteOff(chan,key,velocity)
char  chan,key,velocity;
{
   DevWrite(0x90+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(0,TRK);
}

/*****************/
/* All Notes Off */
/*****************/
void DevAllNotesOff()
{
   int chan;

   for(chan=0; chan<16; chan++){
      DevWrite(0xb0+chan,TRK);     /* Issue All Notes Off */
      DevWrite(123,TRK);
      DevWrite(0,TRK);
   }
}

/*******************/
/* ChannelPressure */
/*******************/
void  DevChannelPressure(chan,press)
char  chan,press;
{
   DevWrite(0xa0+chan,TRK);
   DevWrite(press,TRK);
}

/******************/
/* Control Change */
/******************/
void  DevControlChange(chan,cnum,cval)
char  chan,cnum,cval;
{
   DevWrite(0xb0+chan,TRK);
   DevWrite(cnum,TRK);
   DevWrite(cval,TRK);
}

/******************/
/* Program Change */
/******************/
void  DevProgramChange(chan,prog)
char  chan,prog;
{
   DevWrite(0xc0+chan,TRK);
   DevWrite(prog,TRK);
}

/**************/
/* AfterTouch */
/**************/
void  DevAfterTouch(chan,key,press)
char  chan,key,press;
{
   DevWrite(0xd0+chan,TRK);
   DevWrite(key,TRK);
   DevWrite(press,TRK);
}

/*************/
/* PitchBend */
/*************/
void  DevPitchBend(chan,lsb,msb)
char  chan,lsb,msb;
{
   DevWrite(0xe0+chan,TRK);
   DevWrite(lsb,TRK);
   DevWrite(msb,TRK);
}

/*************/
/* MTC       */
/*************/
void  DevMTC(byte1,byte2)
char  byte1,byte2;
{
   DevWrite(0xf2,TRK);
   DevWrite(byte1,TRK);
   DevWrite(byte2,TRK);
}

/**************************/
/* Device Dependent SysEx */
/**************************/
void DevDepSX(d1,d2,d3)
char d1,d2,d3;
{
   return;
}

/************************/
/* Generic Voice Select */
/************************/
void DevVoiceSel(inst,type,sel)
char inst,type,sel;
{
   return;
}

/**************************/
/* Query Timbre Parameter */
/**************************/
void DevQTimbreParm(parmid)
long parmid;
{
   return;
}

/************************/
/* Set Timbre Parameter */
/************************/
void DevSetTimbreParm(parmid,parmval)
long parmid;
int parmval;
{
   return;
}

/************************/
/* Request Timbre Block */
/************************/
void DevReqTimbreBlk(blknum)
int blknum;
{
   return;
}

/**********************/
/* Write Timbre Block */
/**********************/
void DevWriteTimbreBlk(blknum,len,blk)
int blknum,len;
unsigned char *blk;
{
   return;
}

/****************/
/* Track Volume */
/****************/
void DevTrkVolume(level,duration,trk)
int level,duration,trk;
{
   return;
}

/*****************/
/* Track Balance */
/*****************/
void DevTrkBalance(balance,duration,trk)
int balance,duration,trk;
{
   return;
}

/*****************/
/* Master Volume */
/*****************/
void DevMstVolume(level,duration,trk)
int duration,trk;
unsigned int level;
{
   return;
}

/********************/
/* Set Clock values */
/********************/
void SetClock(trk)
int trk;
{
   extern int tempo,ppqn;
   extern int INT1C_DIVISOR,INT1C_COUNTER;

   int   ttempo,tppqn,tdiv;

   ttempo = TEMPO;
   tppqn = PPQN;

   _asm{
      mov   dx,ttempo
      mov   ax,tppqn
      mul   dx          ; dx:ax = tempo*ppqn
      mov   bx,600
      div   bx          ; ax = (tempo*ppqn)/600
      mov   bx,ax
      mov   dx,18       ; dx:ax = 1193180
      mov   ax,13532
      div   bx          ; ax = Timer Divisor
      mov   bx,ax

      mov   AL,36H       ; Set timer 0 mode 3
      out   TIMER+3,AL
      jmp   $+2
      mov   AL,BL
      out   TIMER+0,AL   ;  Set divisor LSB
      jmp   $+2
      mov   AL,BH
      out   TIMER+0,AL    ;  and MSB

      mov   ax,10
      mul   bx            ; dx:ax = FO_RATE*10
      mov   cx,182
      div   cx            ; ax = fo_rate*10 / 182
      mov   tdiv,ax
   }
   INT1C_DIVISOR = tdiv;
   INT1C_COUNTER = tdiv;
}

/***************/
/* Device Open */
/***************/
int DevOpen(trk)
int trk;
{

   GetDevInt(irqnum);                        /* Go Hook Device Interrupt   */
   GetTimerInt();                            /*  and System Timer interrupt*/
   SetClock(trk);
   _asm sti;
   outp(MPU_COMMAND_PORT,0x3f);              /* Put MPU into UART mode     */
 while(DevRead() != -1);
   return(0);
}

/****************/
/* Device Close */
/****************/
int DevClose(trk)
int trk;
{
   int x;

   FreeTimerInt();
   outp(MPU_COMMAND_PORT,0xff);              /* Reset MPU                  */
   for(x=0; x<32000 && DevRead()== -1; x++); /* Wait for ack (doesn't happen)*/
   FreeDevInt();
   return(0);
}

/************/
/* DevStart */
/************/
void DevStart(trk)
int trk;
{
   /* nothing to do */
}

/************/
/* DevPause */
/************/
void DevPause(trk)
int trk;
{
   /* nothing to do */
}


/*************/
/* DevResume */
/*************/
void DevResume(trk)
int trk;
{
   /* nothing to do */
}



/*************/
/* DevDeInit */
/*************/
void DevDeInit(trk)
int trk;
{
   /* nothing to do */
}

/***********/
/* DevStop */
/***********/
void DevStop(trk)
int trk;
{
   /* nothing to do */
}


/*****************************************************************************/
/* DevNotify                                                                 */
/*****************************************************************************/
char DevNotify(pos,trk)
unsigned long pos;
int trk;
{
   /* Nothing to do */
   return(0);
}

/*************/
/* DevChange */
/*************/
int DevChange(changes, dev_info, any_dev_info, mode_info, any_mode_info, trk)
struct audio_change far *changes;
int *dev_info;
char any_dev_info;
int *mode_info;
char any_mode_info;
int trk;
{
   /* nothing to do */
   return(0);
}

/*********************/
/* Device init IOCTL */
/*********************/
int DevIOCTLinit(dptr,index,trk)          /* Process ioctl init requests   */
AIP   dptr;                               /* pointer to audio_init data    */
int   index;
int   trk;
{

   switch(index){
   default:
      index = MIDI_MODE;
      dptr->flags &= ~(mode_table[index].flags_mask);
      dptr->flags |= mode_table[index].flags | BESTFIT_PROVIDED;
      dptr->mode = mode_table[index].mode;
      dptr->srate = mode_table[index].srate_low;
      dptr->bits_per_sample = mode_table[index].bps_low;
      dptr->bsize = mode_table[index].bsize;
      dptr->channels = mode_table[index].channels_low;

   case MIDI_MODE:
      dptr->position_resolution = 0;
      dptr->flags |= OUTPUT;
      break;
   case CLAIM_HDWR:
      break;
   }/*switch*/

   dptr->rc = 0;                          /*  set rc = 0                   */
   dptr->slot_number = 0;
   dptr->device_id = MPU_401;             /* Set device ID number          */

   return(index);
}

/****************/
/* Status IOCTL */
/****************/
void DevIOCTLstatus(dptr,trk)
ASP dptr;
int trk;
{
   extern int tempo;

   struct audio_change far *cptr;
   struct track_info far *tptr;

   dptr->mode = MIDI;                        /*    MIDI mode               */
   dptr->srate = TEMPO;                 /*    rate = tempo            */
   dptr->bits_per_sample = 8;                /*    byte data               */
   dptr->bsize = 1;                          /*    block size = 1          */
   dptr->channels = 1;                       /*    1 channel               */
   dptr->flags = 0;                          /*    flags                   */
   dptr->operation = OPERATION_G;            /*    operation               */
   cptr=&(dptr->change);                     /* ->audio_change block       */
   cptr->input = 0;                          /*    input                   */
   cptr->output = OUTPUT_1;                  /* output                     */
   cptr->monitor = 0;                        /*    monitor                 */
   cptr->volume = 0;                         /*    volume                  */
   cptr->volume_delay = 0;                   /*    volume delay            */
   cptr->balance = 0;                        /*    balance                 */
   cptr->balance_delay = 0;                  /*    balance delay           */
   cptr->treble = 0;                         /*    treble                  */
   cptr->bass = 0;                           /*    bass                    */
   cptr->pitch = 0;                          /*    pitch                   */
   if(tptr = cptr->dev_info){                /*    dev_info                */
      /* not defined for MPU-401 */
   }
}


/*********************/
/* Device load IOCTL */
/*********************/
void DevIOCTLload(buf,size,flags,trk)
unsigned char far *buf;
unsigned long size,flags;
int trk;
{
   return;                                /* Nothing to do for MPU-401     */
}


/*********************/
/* Device wait IOCTL */
/*********************/
void DevIOCTLwait(void)
{
   return;                                /* Nothing to do for MPU-401     */
}

/*********************/
/* Diagnostic IOCTLs */
/*********************/
void DevIOCTLread8(dptr)
DR8 dptr;
{
   dptr->data=(unsigned char)inp(MPUBASE+dptr->offset);  /* Read data from reg*/
}

void DevIOCTLwrite8(dptr)
DR8 dptr;
{
   outp(MPUBASE+dptr->offset,dptr->data); /* Write data to port            */
}

void DevIOCTLread16(dptr)
DR16 dptr;
{
   dptr->data=inpw(MPUBASE+dptr->offset); /* Read data from reg            */
}

void DevIOCTLwrite16(dptr)
DR16 dptr;
{
   outpw(MPUBASE+dptr->offset,dptr->data);   /* Write data to port         */
}


/*************************/
/* Device Initialization */
/*************************/
DevInit(parms)
char far *parms;
{
   /* Parse command line arguments for MIDIn and IRQn numbers */
   GetParms(parms);

   return(0);
}

/*************************************************************/
/* Device Write - Wait for Xmit Rdy, then write word in ax   */
/* Leaves with WIE disabled when done, ax=0                  */
/*************************************************************/
int DevWrite(data,trk)
int data,trk;
{
   int x,ch;

   /* Wait for transmit ready */
   for(x=0; x<10000 && ((ch=inp(MPU_STATUS_PORT)) & MPU_XMIT_RDY); x++){
      if(!(ch & MPU_REC_RDY)){
         ch = DevRead();
         queue_data((unsigned char *)&ch,1,trk);
      }
   }
   if(x==10000){
      return(-1);            /* Not ready, return -1             */
   }
   outp(MPU_DATA_PORT,data);           /* Write the data                   */
   return(0);
}

/***************/
/* Device Read */
/***************/
int DevRead()
{
   int ch;

   if(inp(MPU_STATUS_PORT) & MPU_REC_RDY){
      ch = -1;
   }else{
      ch = inp(MPU_DATA_PORT);
   }
   return(ch);
}

/************************************************************************/
/* This routine will parse the parameters from the CONFIG.SYS.          */
/*                                                                      */
/* Expected parameters are:                                             */
/*                                                                      */
/*              DEVICE=d:\path\MPUDD.SYS midi# int#                     */
/*                                                                      */
/************************************************************************/
void GetParms(parms)
char far *parms;
{
   extern char midinum;
   char ch;

   while((ch = *parms++)!=10 && ch!=13){
      /* find DOS delimiter */
      if(ch==0 || ch==' ' || ch==',' || ch==';' || ch=='+' || ch=='=' || ch=='\t'){
         break;
      }
   }
   if(ch==10 || ch==13) return;

   while((ch = *parms++)!=10 && ch!=13){
      /* Look for MIDI# */
      if(ch>='1' && ch<='9'){
         midinum=ch;
         startmsg2[AUDIO_NUM_POS] = ch;

         /* Got it.  Now look for IRQ # */
         while((ch = *parms++)!=10 && ch!=13){
            if(ch>='1' && ch<='9'){
               startmsg2[IRQ_NUM_POS] = ch;
               irqnum = ch-'0';
            }
         }
      }
   }
}
