/* OVRMENU.C
*
* This file contains the code for a simple 
* overlay graphics menu library.
*
* Copyright (C) 1993 Matrox Electronic Systems, Ltd.
* All rights reserved
*
*/

/************************/
/* Inclusion statements */
/************************/
#include <grabtest.h>

/***********/
/* Globals */
/***********/
extern int    OVRLocation;
extern BUTTON button[];
extern LABEL  label[];
extern long  histogram_buffer[];
extern	int		mouse_present, mouse_type, mouse_x_pos, mouse_y_pos;
extern	int		button_state;
extern	unsigned long	last_pressed;
extern	unsigned char	last_overflow;

char	text_buff[10] = {0,};

/* OVR_Draw3DRect()
*
* OVR_Draw3DRect is used to draw a 3D rectangle to the overlay screen using the
* following format.
*
*  1111111111111111111   0- interior color 
*  1222222222222222224   1- outside_top_left_color
*  1222222222222222254   2- top_left_color
*  1223333333333333554   3- inside_top_left_color
*  1223000000000006554   4- outside_bottom_right
*  1223000000000006554   5- bottom_right
*  1223000000000006554   6- inside_bottom_right
*  1223000000000006554
*  1223000000000006554
*  1223666666666666554
*  1225555555555555554
*  1255555555555555554
*  1444444444444444444
*
*
* Designed by: Henry Hurdon      Date: May    7, 1991
* Reviewed by: ................. Date: ..............
* Modified by: Michel Pelletier  Date: Apr    6, 1993
*
*    Calls: nothing.
*
*     Uses: nothing.
* Modifies: nothing
*  Returns: nothing.
*
*/
void OVR_Draw3DRect(int left, int top, int width, int height,
                    int interior_color,
                    int outside_top_left_color,
                    int top_left_color,     int inside_top_left_color,
                    int outside_bottom_right_color,
                    int bottom_right_color, int inside_bottom_right_color)
{
HideTheMouse();
if( OVRLocation == IM_OVERLAY )
   {
   /******************************************************/
   /* Use Image-Shell graphics functions for drawing in  */
   /* the IM-1280 or IM-640 overlay frame buffer         */
   /******************************************************/

   /* Select overlay buffer */
   krhwslcsurf( I_OVL_SURF );

   /* draw rectangle background */
   krhwsfcol( interior_color );
   krhwsbcol( interior_color );
   scarrect( left, top, left + width - 1, top + height - 1);

   /* draw top left corner.. */
   krhwsfcol( outside_top_left_color );
   scolmvto(left +width -1,     top);
   scollnto(left,               top);
   scollnto(left,               top +height -1);

   krhwsfcol(top_left_color);
   scolmvto(left +width -1 -1,  top +1);
   scollnto(left +1,            top +1);
   scollnto(left +1,            top +height -1 -1);

   scolmvto(left +width -2 -1,  top +2);
   scollnto(left +2,            top +2);
   scollnto(left +2,            top +height -2 -1);

   krhwsfcol(inside_top_left_color);
   scolmvto(left +width -3 -1,  top +3);
   scollnto(left +3,            top +3);
   scollnto(left +3,            top +height -3 -1);

   /* draw bottom right corner.. */
   krhwsfcol(outside_bottom_right_color);
   scolmvto(left +1,            top +height -1);
   scollnto(left +width -1,     top +height -1);
   scollnto(left +width -1,     top +1);

   krhwsfcol(bottom_right_color);
   scolmvto(left +2,            top +height -1 -1);
   scollnto(left +width -1 -1,  top +height -1 -1);
   scollnto(left +width -1 -1,  top +2);

   scolmvto(left +3,            top +height -2 -1);
   scollnto(left +width -2 -1,  top +height -2 -1);
   scollnto(left +width -2 -1,  top +3);

   krhwsfcol(inside_bottom_right_color);
   scolmvto(left +4,            top +height -3 -1);
   scollnto(left +width -3 -1,  top +height -3 -1);
   scollnto(left +width -3 -1,  top +4);
   }
else
   {
   /*****************************************************/
   /* Use Microsoft C graphics functions for drawing in */
   /* the VGA's frame buffer.                           */
   /*****************************************************/
   /* draw rectangle background */
   _setcolor( interior_color );
   _setbkcolor( interior_color );
   _rectangle( _GFILLINTERIOR, left, top, left + width - 1, top + height - 1);

   /* draw top left corner.. */
   _setcolor( outside_top_left_color );
   _moveto(left +width -1,     top);
   _lineto(left,               top);
   _lineto(left,               top +height -1);

   _setcolor(top_left_color);
   _moveto(left +width -1 -1,  top +1);
   _lineto(left +1,            top +1);
   _lineto(left +1,            top +height -1 -1);

   _moveto(left +width -2 -1,  top +2);
   _lineto(left +2,            top +2);
   _lineto(left +2,            top +height -2 -1);

   _setcolor(inside_top_left_color);
   _moveto(left +width -3 -1,  top +3);
   _lineto(left +3,            top +3);
   _lineto(left +3,            top +height -3 -1);

   /* draw bottom right corner.. */
   _setcolor(outside_bottom_right_color);
   _moveto(left +1,            top +height -1);
   _lineto(left +width -1,     top +height -1);
   _lineto(left +width -1,     top +1);

   _setcolor(bottom_right_color);
   _moveto(left +2,            top +height -1 -1);
   _lineto(left +width -1 -1,  top +height -1 -1);
   _lineto(left +width -1 -1,  top +2);

   _moveto(left +3,            top +height -2 -1);
   _lineto(left +width -2 -1,  top +height -2 -1);
   _lineto(left +width -2 -1,  top +3);

   _setcolor(inside_bottom_right_color);
   _moveto(left +4,            top +height -3 -1);
   _lineto(left +width -3 -1,  top +height -3 -1);
   _lineto(left +width -3 -1,  top +4);
   }
ShowTheMouse();
}

/* OVR_PushButton()
*
* OVR_PushButton is used to draw a BUTTON to the overlay screen.
* Note: Please refer to OVRMENU.H for a description of the BUTTON structure.
*
* Designed by: Henry Hurdon      Date: May    7, 1991
* Reviewed by: ................. Date: ..............
* Modified by: Michel Pelletier  Date: Apr   26, 1993
*
*    Calls: OVR_Draw3DRect(), (*button[num].function)(num);
*
*     Uses: BUTTON button[].
* Modifies: nothing
*  Returns: nothing.
*
*/
void OVR_PushButton(int num)
{
   OVR_Draw3DRect(button[num].left, button[num].top,
                  button[num].width, button[num].height,
                  button[num].interior_color[label[num].state],
                  button[num].outside_top_left_color[label[num].state],
                  button[num].top_left_color[label[num].state],
                  button[num].inside_top_left_color[label[num].state],
                  button[num].outside_bottom_right_color[label[num].state],
                  button[num].bottom_right_color[label[num].state],
                  button[num].inside_bottom_right_color[label[num].state]);

   if (button[num].function != NULL) 
	{
      	(*button[num].function)(num);
   	}
}

/* OVR_DrawLabel()
*
* OVR_DrawLabel is used to draw a LABEL to the overlay screen.
* Note: Please refer to OVRMENU.H for a description of the LABEL structure.
*
* Designed by: Henry Hurdon      Date: May    7, 1991
* Reviewed by: ................. Date: ..............
* Modified by: Michel Pelletier  Date: Apr   26, 1993
*
*    Calls: (*label[num].function)(num);
*
*     Uses: LABEL label[].
* Modifies: nothing
*  Returns: nothing.
*
*/
void OVR_DrawLabel(int num)
{
HideTheMouse();
if( OVRLocation == IM_OVERLAY )
   {
   /******************************************************/
   /* Use Image-Shell graphics functions for drawing in  */
   /* the IM-1280 or IM-640 overlay frame buffer.        */
   /******************************************************/
   krhwslcsurf( I_OVL_SURF );
   krhwsfcol( label[num].color[label[num].state] );
   sctxbstring( label[num].left, label[num].top, label[num].text[label[num].log_state]);
   }
else
   {
   /*****************************************************/
   /* Use Microsoft C graphics functions for drawing in */
   /* the VGA's frame buffer.                           */
   /*****************************************************/
   _setcolor(label[num].color[label[num].state]);
   _moveto(label[num].left, label[num].top);
   _outgtext(label[num].text[label[num].log_state]);
   }
if (label[num].function != NULL) 
   {
   (*label[num].function)(num);
   }
ShowTheMouse();
}


/* OVR_Print()
*
* OVR_Print is used to print to the overlay screen.
*
* Designed by: Henry Hurdon      Date: May    7, 1991
* Reviewed by: ................. Date: ..............
* Modified by: Michel Pelletier  Date: Apr    6, 1993
*
*    Calls: nothing.
*
*     Uses: nothing.
* Modifies: nothing
*  Returns: nothing.
*
*/
#define BUFFER_SIZE  128
int OVR_Print(int  bkleft, int bktop, int bkwidth, int bkheight,
              int  left,   int top,
              int  text_color, int  text_bkcolor, 
              char *format, ...)
{
va_list  arg_ptr;
char     buffer[BUFFER_SIZE];
int      status;

HideTheMouse();
va_start(arg_ptr, format);
status = vsprintf(buffer, format, arg_ptr);
va_end(arg_ptr);

if( OVRLocation == IM_OVERLAY )
   {
   /******************************************************/
   /* Use Image-Shell graphics functions for drawing in  */
   /* the IM-1280 or IM-640 overlay frame buffer.        */
   /******************************************************/
   /* Select overlay buffer */
   krhwslcsurf( I_OVL_SURF );

   /* clear rectangle background */
   krhwsfcol( text_bkcolor );
   scarrect( bkleft, bktop, bkleft +bkwidth -1, bktop +bkheight -1 );

   krhwsfcol( text_color );
   krhwsbcol( text_bkcolor );
   sctxbstring( left, top, buffer );
   }
else
   {
   /*****************************************************/
   /* Use Microsoft C graphics functions for drawing in */
   /* the VGA's frame buffer.                           */
   /*****************************************************/
   /* clear rectangle background */
   _setcolor(text_bkcolor);
   _rectangle(_GFILLINTERIOR, bkleft, bktop, bkleft +bkwidth -1, bktop +bkheight -1);

   _setcolor(text_color);
   _setbkcolor(text_bkcolor);
   _moveto(left, top);
   _outgtext(buffer);
   }
ShowTheMouse();
return (status);
}

/* ggetch()
*
* ggetch is used to get slightly-processed keyboard input.
*
* Designed by: Henry Hurdon      Date: Mar.  15, 1990
* Reviewed by: ................. Date: ..............
* Modified by: Michel Pelletier  Date: Apr.   7, 1993
*
*    Calls: getch()
*
*     Uses: nothing.
* Modifies: nothing
*  Returns: int char or ERROR
*
*/

int ggetch( void )
{
int     		ch, i;
int			finished, match_found;
int			previous_button_state;
unsigned char far 	*data_ptr;
unsigned long far	*clk_ptr;
unsigned long		this_press;
unsigned char		this_overflow;

clk_ptr = (unsigned long far *)0x46cL;
data_ptr = (unsigned char far *)0x470L;

finished = NO;
match_found = NO;
while( finished == NO )
	{
	this_press = *clk_ptr;
	this_overflow = *data_ptr;
	previous_button_state = button_state;
	UpdateMouseInfo();
	if( kbhit() )
		{
		finished = YES;
		ch = getch();
		ch &= 0x00FF;
		if ((ch == 0x00)||(ch == 0xE0)) 
			{
			ch = getch() | 0x80;
			}
		}
	else if( ((button_state == DOWN) && (previous_button_state == UP)) ||
	         ((button_state == DOWN) && (previous_button_state == DOWN) &&
	          ((this_press - last_pressed) > TIME_DELAY)) ||
	         ((button_state == DOWN) && (previous_button_state == DOWN) &&
	          (this_overflow != last_overflow)) )
		{
		for( i = 0; i < GetNumButtons(); i++ )
			{
			if( (mouse_x_pos >= button[i].left) &&
		            (mouse_x_pos < (button[i].left + button[i].width)) &&
		            (mouse_y_pos >= button[i].top) &&
		            (mouse_y_pos < (button[i].top  + button[i].height)) )
				{
				ungetch( label[i].code );
				match_found = YES;
				}
			}
		if( match_found == NO )
			{
			ungetch( MOUSE_CLICK );
			}
		}
	}
return (ch);
}


/**********************************************************/
/* This function draws a Histogram in the overlay surface */
/**********************************************************/
void	DrawHistogram( void )
{
int 	i;
long	maxval;

HideTheMouse();

/* find the largest element for scaling purposes */
maxval = 0L;
for( i = 0; i < 256; i++ )
	{
	maxval = max( maxval, histogram_buffer[i] );
	}

if( OVRLocation == IM_OVERLAY )
   {
   /******************************************************/
   /* Use Image-Shell graphics functions for drawing in  */
   /* the IM-1280 or IM-640 overlay frame buffer.        */
   /******************************************************/
   /* Select overlay buffer */
   krhwslcsurf( I_OVL_SURF );

   /* clear rectangle background (interior of histogram only) */
   krhwsfcol( OVR_TRANSPARENT );
   scarrect( DX_HISTO + 10, DY_HISTO + 16,
	DX_HISTO + DW_HISTO - 11, DY_HISTO + DH_HISTO - 25 );

   /* Draw histogram outline */
   krhwsfcol( OVR_CYAN );
   krhwsbcol( OVR_TRANSPARENT );
   scolmvto( DX_HISTO + 9           , DY_HISTO );
   scollnto( DX_HISTO + 9           , DY_HISTO + DH_HISTO - 24);
   scolmvto( DX_HISTO               , DY_HISTO + DH_HISTO - 24 );
   scollnto( DX_HISTO + DW_HISTO - 4, DY_HISTO + DH_HISTO - 24 );

   /* Draw title */
   sctxbstring( DX_HISTO + 100, DY_HISTO, "Histogram" );

   /* Draw the histogram itself */
   for( i = 0; i < 256; i++ )   
	{
	if( !(i%32) )
		{
		krhwstdef( 1 );
		sctxbstring( DX_HISTO + 6 + i,
			DY_HISTO + DH_HISTO - 32, "|" );
		krhwstdef( 0 );
		sprintf( text_buff, "%.2xh", i );
		sctxbstring( DX_HISTO + 6 + i,
			DY_HISTO + DH_HISTO - 16, text_buff );
		}
	scolmvto( DX_HISTO + 10 + i, DY_HISTO + DH_HISTO - 24 );
	scollnto( DX_HISTO + 10 + i, DY_HISTO + DH_HISTO - 24 - 
		(short)(histogram_buffer[i] * ((long)DH_HISTO - 40) / maxval) );
	}
   }
else
   {
   /*****************************************************/
   /* Use Microsoft C graphics functions for drawing in */
   /* the VGA's frame buffer.                           */
   /*****************************************************/
   /* clear rectangle background (interior of histogram only) */
   _setcolor( OVR_TRANSPARENT );
   _rectangle(_GFILLINTERIOR, DX_HISTO + 10, DY_HISTO + 16,
	DX_HISTO + DW_HISTO - 11, DY_HISTO + DH_HISTO - 25 );

   /* Draw histogram outline */
   _setcolor( OVR_CYAN );
   _setbkcolor( OVR_TRANSPARENT );
   _moveto( DX_HISTO + 9           , DY_HISTO );
   _moveto( DX_HISTO + 9           , DY_HISTO + DH_HISTO - 24);
   _moveto( DX_HISTO               , DY_HISTO + DH_HISTO - 24 );
   _moveto( DX_HISTO + DW_HISTO - 4, DY_HISTO + DH_HISTO - 24 );

   /* Draw title */
   _moveto( DX_HISTO + 100, DY_HISTO );
   _outgtext( "Histogram" );

   /* Draw the histogram itself */
   for( i = 0; i < 256; i++ )   
	{
	if( !(i%32) )
		{
		_moveto( DX_HISTO + 6 + i, DY_HISTO + DH_HISTO - 32 );
		_outgtext( "|" );
		sprintf( text_buff, "%.2xh", i );
		_moveto( DX_HISTO + 6 + i, DY_HISTO + DH_HISTO - 16 );
		_outgtext( text_buff );
		}
	_moveto( DX_HISTO + 10 + i, DY_HISTO + DH_HISTO - 24 );
	_lineto( DX_HISTO + 10 + i, DY_HISTO + DH_HISTO - 24 - 
		(short)(histogram_buffer[i] * ((long)DH_HISTO - 40) / maxval) );
	}
   }
ShowTheMouse();
}

/****************************************************************/
/* This function removes the Histogram from the overlay surface */
/****************************************************************/
void	RemoveHistogram( void )
{
HideTheMouse();

if( OVRLocation == IM_OVERLAY )
   {
   /******************************************************/
   /* Use Image-Shell graphics functions for drawing in  */
   /* the IM-1280 or IM-640 overlay frame buffer.        */
   /******************************************************/
   /* Select overlay buffer */
   krhwslcsurf( I_OVL_SURF );

   /* clear rectangle background */
   krhwsfcol( OVR_TRANSPARENT );
   scarrect( DX_HISTO, DY_HISTO, 
	DX_HISTO + DW_HISTO - 1, DY_HISTO + DH_HISTO - 1 );
   }
else
   {
   /*****************************************************/
   /* Use Microsoft C graphics functions for drawing in */
   /* the VGA's frame buffer.                           */
   /*****************************************************/
   /* clear rectangle background */
   _setcolor( OVR_TRANSPARENT );
   _rectangle(_GFILLINTERIOR, DX_HISTO, DY_HISTO, 
	DX_HISTO + DW_HISTO - 1, DY_HISTO + DH_HISTO - 1);
   }
ShowTheMouse();
}
