
/*

    xati - manage the ATI Stereo F/X card.
    Copyright (C) 1994 Torsten Eichner

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    email: 	eichner@rhrk.uni-kl.de
    papermail:	Torsten Eichner, Kampshardtweg 1, D-57587 Birken-Honigsessen, Germany

*/

/*
    Start of file: xati.c
*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

Widget w_shell, w_main, w_command, w_sb, w_toggle;
XtAppContext context;
int volume;

#define ERROR(string) { char error[512]; sprintf(error, "[%s:%d]: %s\n", __FILE__, __LINE__, string); XtError(error); }

static void usage(char*);
static void parse(int, char *[]);
static void set_volume(int);
static void set_mix(int);
static void toggle_cb(Widget, XtPointer, XtPointer);
static void sb_cb(Widget, XtPointer, XtPointer);
static void quit_cb(Widget, XtPointer, XtPointer);

static void usage(char *errorarg) {
	fprintf(stderr, "xati: error in commandline (%s)\nUsage: xati [X-Options] [-v volume] [-m (on|off)]\n\n", errorarg);
	ERROR("command line");
	}	

static void parse(int argc, char *argv[]) {
	int i = 1;
	int newval;
	char *endptr;

	while (argc > 1) {
		if (*(argv[i]) == '-')
			switch(*(argv[i]+1)) {
				case 'v':
					i++;
					argc--;
					if (argc < 2)
						usage(argv[i-1]);
					newval = strtol(argv[i], &endptr, 0);
					if (*endptr != '\0')
						usage(argv[i]);
					if ((newval < 0) || (newval > 255))
						usage(argv[i]);
					set_volume(newval);
					break;	
				case 'm':
					i++;
					argc--;
					if (argc < 2)
						usage(argv[i-1]);
					if (strcmp(argv[i], "on") == 0)
						set_mix(1);
					else
					if (strcmp(argv[i], "off") == 0)
						set_mix(0);
					else
						usage(argv[i]);	
					break;
				default:
					usage(argv[i]);
				}
		else
			usage(argv[i]);

		i++; argc--;
		}

	if (i != 1)
		exit(0);
	}

static void set_volume(int volume) {
	int fd;
	unsigned char vol;

	if (volume < 0 || volume > 255)
		ERROR("volume out of range");
	vol = (volume >= 128 ? volume & 0x7f : (volume & 0x7f) | 128);

	if ((fd = open("/dev/port", O_WRONLY)) < 0)
		ERROR("cannot open /dev/port");
	if (lseek(fd, VOLUME, SEEK_SET) < 0)
		ERROR("cannot seek volume-port");

	if (write(fd, &vol, 1) != 1)
		ERROR("cannot write to port");

	close(fd);
	}

static void set_mix(int flag) {
	int fd;
	unsigned char mix_on = MIXON;
	unsigned char mix_off = MIXOFF;

	if ((fd = open("/dev/port", O_WRONLY)) < 0)
		ERROR("cannot open /dev/port");
	if (lseek(fd, MIX, SEEK_SET) < 0)
		ERROR("cannot seek mix-port");

	if (write(fd, (flag ? &mix_on : &mix_off), 1) != 1)
		ERROR("cannot write to port");

	close(fd);
	}

static void toggle_cb(Widget w, XtPointer cld, XtPointer cad) {
	static int flag = 1;

	flag = !flag;
	set_mix(flag);
	}

static void sb_cb(Widget w, XtPointer cld, XtPointer cad) {
	if ((int) cld == 1) {
		float percent;

		if ((int) cad < 0)
			volume = volume-10;
		else
			volume = volume+10;	

		if (volume < 0)
			volume = 0;
		if (volume > 255)
			volume = 255;

		percent = (float) volume / 255.0;

		XawScrollbarSetThumb(w_sb, percent, 0.01); /* is this ok? */

		set_volume(volume);
		}
	if ((int) cld == 2) {
		float percent = *((float*) cad);

		volume = (int) (percent	* 255.0);

		if (volume > 255)
			volume = 255;

		if (volume < 0)
			volume = 0;

		set_volume(volume);
		}
	}

static void quit_cb(Widget w, XtPointer cld, XtPointer cad) {
	exit(0);
	}

int main(int argc, char *argv[]) {
	if (strcmp(argv[0]+strlen(argv[0])-4, "xati") != 0) {
		printf("ati 1.0: set volume/mix on ATI Stereo F/X soundcards\n");
		parse(argc, argv);
		exit(0);
		}

	w_shell = XtAppInitialize(&context, "xati", NULL, 0, &argc, argv, NULL, NULL, 0);

	parse(argc, argv);

	w_main = XtVaCreateWidget(NULL, panedWidgetClass, w_shell, NULL);	

	w_sb = XtVaCreateWidget("Volume", scrollbarWidgetClass, w_main, XtNshowGrip, False, XtNlength, 100, XtNminimumThumb, 2, XtNthickness, 16, XtNorientation, XtorientHorizontal, NULL);
	w_toggle = XtVaCreateWidget("Mix", toggleWidgetClass, w_main, XtNstate, True, NULL);
	w_command = XtVaCreateWidget("Quit", commandWidgetClass, w_main, XtNshowGrip, False, NULL);

	XtAddCallback(w_toggle, XtNcallback, toggle_cb, NULL);
	XtAddCallback(w_sb, XtNscrollProc, sb_cb, (XtPointer)1);
	XtAddCallback(w_sb, XtNjumpProc, sb_cb, (XtPointer)2);
	XtAddCallback(w_command, XtNcallback, quit_cb, NULL);

	XtManageChild(w_command);
	XtManageChild(w_toggle);
	XtManageChild(w_sb);

	XtManageChild(w_main);

	set_mix(1);
	set_volume(0);
	volume = 0;

	XtRealizeWidget(w_shell);

	XtAppMainLoop(context);	
	}


/*
    End of file
*/
