/**
 ** GRX.H
 **
 **  Copyright (C) 1992, Csaba Biegl
 **    820 Stirrup Dr, Nashville, TN, 37221
 **    csaba@vuse.vanderbilt.edu
 **
 **  This file is distributed under the terms listed in the document
 **  "copying.cb", available from the author at the address above.
 **  A copy of "copying.cb" should accompany this file; if not, a copy
 **  should be available from where this file was obtained.  This file
 **  may not be distributed without a verbatim copy of "copying.cb".
 **  You should also have received a copy of the GNU General Public
 **  License along with this program (it is in the file "copying");
 **  if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 **  Cambridge, MA 02139, USA.
 **
 **  This program is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **/

#ifndef _GRX_H_
#define _GRX_H_

#ifdef __cplusplus
extern "C" {
#endif

#if defined(__GNUC__) && !defined(near)
# define near
# define far
# define huge
#endif


/* ================================================================== */
/*			     MODE SETTING			      */
/* ================================================================== */

typedef enum {
    GR_80_25_text,
    GR_default_text,
    GR_width_height_text,
    GR_biggest_text,
    GR_320_200_graphics,
    GR_default_graphics,
    GR_width_height_graphics,
    GR_biggest_noninterlaced_graphics,
    GR_biggest_graphics,
    GR_width_height_color_graphics
} GR_graphics_modes;

#ifdef __cplusplus
  void  GrSetMode(int which,int width=0,int height=0,int colors=0);
#elif !defined(_SETMODE_C_)
  void  GrSetMode(int which,...);
#else
  void  GrSetMode(int which,int width,int height,int colors);
#endif
void	GrSetBIOSMode(int BIOSmodeNo,int width,int height,int colors);
void	GrSetModeHook(void (*)(void));
int	GrCurrentMode(void);
int	GrAdapterType(void);

/*
 * return values from 'GrAdapterType()'
 */
#define GR_VGA		0		/* VGA adapter */
#define GR_EGA		1		/* EGA adapter */
#define GR_HERC		2		/* Hercules mono adapter */
#define GR_8514A	3		/* 8514A or compatible */
#define GR_S3		4		/* S3 graphics accelerator */


/* ================================================================== */
/*			CONTEXT AND WINDOW STUFF		      */
/* ================================================================== */

typedef struct _GR_context_ {
    char  far *gc_baseaddr;		/* base address of display memory */
    long  gc_frameaddr;			/* upper left corner coordinate */
    long  gc_planeoffset;		/* offset to next color plane */
    int	  gc_lineoffset;		/* offset to next scan line in bytes */
    char  gc_onscreen;			/* is it in video memory ? */
    char  gc_memflags;			/* memory allocation flags */
    int	  gc_xmax;			/* max X coord (width  - 1) */
    int	  gc_ymax;			/* max Y coord (height - 1) */
    int	  gc_xcliplo;			/* low X clipping limit */
    int	  gc_ycliplo;			/* low Y clipping limit */
    int	  gc_xcliphi;			/* high X clipping limit */
    int	  gc_ycliphi;			/* high Y clipping limit */
    int	  gc_usrxbase;			/* user window min X coordinate */
    int	  gc_usrybase;			/* user window min Y coordinate */
    int	  gc_usrwidth;			/* user window width */
    int	  gc_usrheight;			/* user window height */
    int	  gc_xoffset;			/* X offset from root's base */
    int	  gc_yoffset;			/* Y offset from root's base */
    struct _GR_context_ *gc_root;	/* context which owns frame buf */
} GrContext;

/*
 * This MUST BE the same as the head of the 'GrContext' structure!!!!
 * This structure contains all the slots describing the physical layout of
 * the display memory, but no clipping limits, etc...
 */
typedef struct {
    char  far *gc_baseaddr;		/* base address of display memory */
    long  gc_frameaddr;			/* upper left corner coordinate */
    long  gc_planeoffset;		/* offset to next color plane */
    int	  gc_lineoffset;		/* offset to next scan line in bytes */
    char  gc_onscreen;			/* is it in video memory ? */
    char  gc_memflags;			/* memory allocation flags */
} GrVidRAM;


int	GrNumPlanes(void);
int	GrLineOffset(int width);
long	GrPlaneSize(int w,int h);
long	GrContextSize(int w,int h);

GrContext *GrCreateContext(int w,int h,char far *memory,GrContext *where);
GrContext *GrCreateSubContext(int x1,int y1,int x2,int y2,GrContext *parent,GrContext *where);
GrContext *GrSaveContext(GrContext *where);

void	GrResizeSubContext(GrContext *context,int x1,int y1,int x2,int y2);
void	GrSetContext(GrContext *context);
void	GrDestroyContext(GrContext *context);
void	GrSetClipBox(int x1,int y1,int x2,int y2);
void	GrGetClipBox(int *x1p,int *y1p,int *x2p,int *y2p);
void	GrResetClipBox(void);

int	GrMaxX(void);
int	GrMaxY(void);
int	GrSizeX(void);
int	GrSizeY(void);
int	GrScreenX(void);
int	GrScreenY(void);


/* ================================================================== */
/*			      COLOR STUFF			      */
/* ================================================================== */

/*
 * color table (for primitives using several colors):
 *   it is an array of int's with the first element being the number of
 *   colors in the table
 */
typedef int *GrColorTableP;

#define GR_CTABLE_SIZE(table)		((table)[0])
#define GR_CTABLE_COLOR(table,index)	(		\
    ((unsigned)(index) < GR_CTABLE_SIZE(table)) ?	\
	(table)[(index) + 1] :				\
	(table)[GR_CTABLE_SIZE(table)]			\
)
#define GR_CTABLE_ALLOCSIZE(colors)	((colors) + 1)

/*
 * Flags to 'OR' to colors for various operations
 */
#ifdef __TURBOC__
# define GrXOR		0x100		/* to "XOR" any color to the screen */
# define GrOR		0x200		/* to "OR" to the screen */
# define GrAND		0x300		/* to "AND" to the screen */
#endif
#ifdef __GNUC__				/* changed for 16 (and then 24) bit colors */
# define GrXOR		0x1000000	/* to "XOR" any color to the screen */
# define GrOR		0x2000000	/* to "OR" to the screen */
# define GrAND		0x3000000	/* to "AND" to the screen */
#endif
#define GrWRITE		0		/* write color */
#define GrNOCOLOR	(GrXOR | 0)	/* GrNOCOLOR is used for "no" color */


int	GrNumColors(void);
int	GrNumFreeColors(void);

int	GrAllocColor(int r,int g,int b);	/* shared, read-only */
int	GrAllocCell(void);			/* unshared, read-write */
int	GrBlack(void);
int	GrWhite(void);

void	GrQueryColor(int c,int *r,int *g,int *b);
void	GrFreeColor(int c);
void	GrSetColor(int c,int r,int g,int b);
void	GrSetRGBcolorMode(void);
void	GrResetColors(void);
void	GrRefreshColors(void);


/* ================================================================== */
/*			 GRAPHICS PRIMITIVES			      */
/* ================================================================== */

/*
 * framed box colors
 */
typedef struct {
    int  fbx_intcolor;
    int  fbx_topcolor;
    int  fbx_rightcolor;
    int  fbx_bottomcolor;
    int  fbx_leftcolor;
} GrFBoxColors;

#ifdef __TURBOC__
#define GR_MAX_POLY_VERTICES	128
#endif
#ifdef __GNUC__
#define GR_MAX_POLY_VERTICES	512
#endif

void	GrClearScreen(int bg);
void	GrClearContext(int bg);
void	GrClearClipBox(int bg);
void	GrPlot(int x,int y,int c);
void	GrLine(int x1,int y1,int x2,int y2,int c);
void	GrHLine(int x1,int x2,int y,int c);
void	GrVLine(int x,int y1,int y2,int c);
void	GrBox(int x1,int y1,int x2,int y2,int c);
void	GrFilledBox(int x1,int y1,int x2,int y2,int c);
void	GrFramedBox(int x1,int y1,int x2,int y2,int wdt,GrFBoxColors *c);
void	GrCircle(int xc,int yc,int r,int c);
void	GrEllipse(int xc,int yc,int xa,int ya,int c);
void	GrCircleArc(int xc,int yc,int r,int start,int end,int c);
void	GrEllipseArc(int xc,int yc,int xa,int ya,int start,int end,int c);
void	GrFilledCircle(int xc,int yc,int r,int c);
void	GrFilledEllipse(int xc,int yc,int xa,int ya,int c);
void	GrFilledCircleArc(int xc,int yc,int r,int start,int end,int c);
void	GrFilledEllipseArc(int xc,int yc,int xa,int ya,int start,int end,int c);
void	GrGetLastArcCoords(int *xs,int *ys,int *xe,int *ye,int *xc,int *yc);
void	GrPolyLine(int numpts,int points[][2],int c);
void	GrPolygon(int numpts,int points[][2],int c);
void	GrFilledConvexPolygon(int numpts,int points[][2],int c);
void	GrFilledPolygon(int numpts,int points[][2],int c);
void	GrBitBlt(GrContext *dst,int x,int y,GrContext *src,int x1,int y1,int x2,int y2,int oper);
int	GrPixel(int x,int y);


/* ================================================================== */
/*		   NON CLIPPING DRAWING PRIMITIVES		      */
/* ================================================================== */

void	GrPlotNC(int x,int y,int c);
void	GrLineNC(int x1,int y1,int x2,int y2,int c);
void	GrHLineNC(int x1,int x2,int y,int c);
void	GrVLineNC(int x,int y1,int y2,int c);
void	GrBoxNC(int x1,int y1,int x2,int y2,int c);
void	GrFilledBoxNC(int x1,int y1,int x2,int y2,int c);
void	GrFramedBoxNC(int x1,int y1,int x2,int y2,int wdt,GrFBoxColors *c);
void	GrBitBltNC(GrContext *dst,int x,int y,GrContext *src,int x1,int y1,int x2,int y2,int oper);
int	GrPixelNC(int x,int y);


/* ================================================================== */
/*	      THICK AND DASHED LINE DRAWING PRIMITIVES		      */
/* ================================================================== */

/*
 * custom line option structure
 *   zero or one dash pattern length means the line is continuous
 *   the dash pattern always begins with a drawn section
 */
typedef struct {
    int  lno_color;			/* color used to draw line */
    int  lno_width;			/* width of the line */
    int  lno_pattlen;			/* length of the dash pattern */
    unsigned char *lno_dashpat;		/* draw/nodraw pattern */
} GrLineOption;

void	GrCustomLine(int x1,int y1,int x2,int y2,GrLineOption *o);
void	GrCustomBox(int x1,int y1,int x2,int y2,GrLineOption *o);
void	GrCustomCircle(int xc,int yc,int r,GrLineOption *o);
void	GrCustomEllipse(int xc,int yc,int xa,int ya,GrLineOption *o);
void	GrCustomCircleArc(int xc,int yc,int r,int start,int end,GrLineOption *o);
void	GrCustomEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrLineOption *o);
void	GrCustomPolyLine(int numpts,int points[][2],GrLineOption *o);
void	GrCustomPolygon(int numpts,int points[][2],GrLineOption *o);


/* ================================================================== */
/*	       PATTERNED DRAWING AND FILLING PRIMITIVES		      */
/* ================================================================== */

/*
 * BITMAP: a mode independent way to specify a fill pattern of two
 *   colors. It is always 8 pixels wide (1 byte per scan line), its
 *   height is user-defined. SET THE TYPE FLAG TO ZERO!!!
 */
typedef struct {
    int  bmp_ispixmap;			/* type flag for pattern union */
    int  bmp_height;			/* bitmap height */
    unsigned char *bmp_data;		/* pointer to the bit pattern */
    int  bmp_fgcolor;			/* foreground color for fill */
    int  bmp_bgcolor;			/* background color for fill */
    int  bmp_memflags;			/* set if dynamically allocated */
} GrBitmap;

/*
 * PIXMAP: a fill pattern stored in a layout identical to the video RAM
 *   for filling using 'bitblt'-s. It is mode dependent, typically one
 *   of the library functions is used to build it. KEEP THE TYPE FLAG
 *   NONZERO!!!
 */
typedef struct {
    int  pxp_ispixmap;			/* type flag for pattern union */
    int  pxp_width;			/* pixmap width (in pixels)  */
    int  pxp_height;			/* pixmap height (in pixels) */
    int  pxp_oper;			/* bitblt mode (SET, OR, XOR, AND) */
    GrVidRAM pxp_source;		/* source context for fill */
} GrPixmap;

/*
 * Fill pattern union -- can either be a bitmap or a pixmap
 */
typedef union {
    int	     gp_ispixmap;		/* nonzero for pixmaps */
    GrBitmap gp_bitmap;			/* fill bitmap */
    GrPixmap gp_pixmap;			/* fill pixmap */
} GrPattern;

#define gp_bmp_data			gp_bitmap.bmp_data
#define gp_bmp_height			gp_bitmap.bmp_height
#define gp_bmp_fgcolor			gp_bitmap.bmp_fgcolor
#define gp_bmp_bgcolor			gp_bitmap.bmp_bgcolor

#define gp_pxp_width			gp_pixmap.pxp_width
#define gp_pxp_height			gp_pixmap.pxp_height
#define gp_pxp_oper			gp_pixmap.pxp_oper
#define gp_pxp_source			gp_pixmap.pxp_source

/*
 * Draw pattern for line drawings -- specifies both the:
 *   (1) fill pattern, and the
 *   (2) custom line drawing option
 */
typedef struct {
    GrPattern	  *lnp_pattern;		/* fill pattern */
    GrLineOption  *lnp_option;		/* width + dash pattern */
} GrLinePattern;


GrPattern *GrBuildPixmap(char *pixels,int w,int h,GrColorTableP colors);
GrPattern *GrBuildPixmapFromBits(char *bits,int w,int h,int fgc,int bgc);
GrPattern *GrConvertToPixmap(GrContext *src);
GrPattern *GrLoadBitmap(char *filename,int fg,int bg);
GrPattern *GrLoadPixmap(char *filename,GrColorTableP colors);
GrPattern *GrLoadIcon(char *filename);

void	GrDestroyPattern(GrPattern *p);

void	GrPatternedLine(int x1,int y1,int x2,int y2,GrLinePattern *lp);
void	GrPatternedBox(int x1,int y1,int x2,int y2,GrLinePattern *lp);
void	GrPatternedCircle(int xc,int yc,int r,GrLinePattern *lp);
void	GrPatternedEllipse(int xc,int yc,int xa,int ya,GrLinePattern *lp);
void	GrPatternedCircleArc(int xc,int yc,int r,int start,int end,GrLinePattern *lp);
void	GrPatternedEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrLinePattern *lp);
void	GrPatternedPolyLine(int numpts,int points[][2],GrLinePattern *lp);
void	GrPatternedPolygon(int numpts,int points[][2],GrLinePattern *lp);

void	GrPatternFilledPlot(int x,int y,GrPattern *p);
void	GrPatternFilledLine(int x1,int y1,int x2,int y2,GrPattern *p);
void	GrPatternFilledBox(int x1,int y1,int x2,int y2,GrPattern *p);
void	GrPatternFilledCircle(int xc,int yc,int r,GrPattern *p);
void	GrPatternFilledEllipse(int xc,int yc,int xa,int ya,GrPattern *p);
void	GrPatternFilledCircleArc(int xc,int yc,int r,int start,int end,GrPattern *p);
void	GrPatternFilledEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrPattern *p);
void	GrPatternFilledConvexPolygon(int numpts,int points[][2],GrPattern *p);
void	GrPatternFilledPolygon(int numpts,int points[][2],GrPattern *p);


/* ================================================================== */
/*		     FONTS AND TEXT PRIMITIVES			      */
/* ================================================================== */

/*
 * font structure - the part visible to the user.
 * for the internal stuff see "grxfont.h" and "grxfile.h"
 * BE CAREFUL when hacking it! TCC and GCC have to produce the
 * same alignments!!!!
 */
#define GR_NAMEWIDTH		16

typedef struct {
    short   fnt_width;			/* width (average for proportional) */
    short   fnt_height;			/* font height */
    unsigned short fnt_minchar;		/* lowest character code in font */
    unsigned short fnt_maxchar;		/* highest character code in font */
    short   fnt_isfixed;		/* nonzero if fixed font */
    short   fnt_internal;		/* nonzero if BIOS font */
    short   fnt_baseline;		/* baseline from top of font */
    short   fnt_undwidth;		/* underline width (at bottom) */
    char    fnt_name[GR_NAMEWIDTH];	/* font file name (w/o path) */
    char    fnt_family[GR_NAMEWIDTH];	/* font family name */
} GrFont;

/*
 * text drawing directions
 */
#define GR_TEXT_RIGHT		0	/* normal */
#define GR_TEXT_DOWN		1	/* downward */
#define GR_TEXT_LEFT		2	/* upside down, right to left */
#define GR_TEXT_UP		3	/* upward */
#define GR_TEXT_DEFAULT		GR_TEXT_RIGHT

/*
 * text alignment options
 */
#define GR_ALIGN_LEFT		0	/* X only */
#define GR_ALIGN_TOP		0	/* Y only */
#define GR_ALIGN_CENTER		1	/* X, Y */
#define GR_ALIGN_RIGHT		2	/* X only */
#define GR_ALIGN_BOTTOM		2	/* Y only */
#define GR_ALIGN_BASELINE	3	/* Y only */
#define GR_ALIGN_DEFAULT	GR_ALIGN_LEFT

/*
 * character types in text strings
 */
#define GR_BYTE_TEXT		0	/* one byte per character */
#define GR_WORD_TEXT		1	/* two bytes per character */
#define GR_ATTR_TEXT		2	/* chr w/ PC style attribute byte */

/*
 * OR this to the foreground color value for underlined text when
 * using GR_BYTE_TEXT or GR_WORD_TEXT modes.
 */
#define GR_UNDERLINE_TEXT	(GrXOR << 6)

/*
 * text attribute macros for the GR_ATTR_TEXT type
 */
#define GR_BUILD_ATTR(fgcolor,bgcolor,underline)  \
    ((fgcolor) & 15) | (((bgcolor) & 7) << 4) | (((underline) & 1) << 7))
#define GR_ATTR_FGCOLOR(attr)	((attr) & 15)
#define GR_ATTR_BGCOLOR(attr)	(((attr) >> 4) & 7)
#define GR_ATTR_UNDERLINE(attr) (((attr) >> 7) & 1)

/*
 * text option structure - contains a font and the options specifying
 * how to draw it. The text drawing functions expect a pointer to this.
 */
typedef struct {
    GrFont *txo_font;			/* font to be used */
    int	    txo_xmag;			/* X magnification */
    int	    txo_ymag;			/* Y magnification */
    union {
	int v;				/* color when no attributes */
	GrColorTableP p;		/* ptr to color table otherwise */
    } txo_fgcolor,txo_bgcolor;		/* foreground, background */
    char    txo_direct;			/* direction (see above) */
    char    txo_xalign;			/* X alignment (see above) */
    char    txo_yalign;			/* Y alignment (see above) */
    char    txo_chrtype;		/* character type (see above) */
} GrTextOption;

/*
 * structure to define a rectangular text window (use fixed fonts only!!)
 */
typedef struct {
    GrFont *txr_font;			/* font to be used */
    char   *txr_buffer;			/* pointer to text buffer */
    char   *txr_backup;			/* optional backup buffer */
    int	    txr_xpos;			/* upper left corner X coordinate */
    int	    txr_ypos;			/* upper left corner Y coordinate */
    int	    txr_width;			/* width of area in chars */
    int	    txr_height;			/* height of area in chars */
    int	    txr_lineoffset;		/* offset in buffer(s) between lines */
    union {
	int v;				/* color when no attributes */
	GrColorTableP p;		/* ptr to color table otherwise */
    } txr_fgcolor,txr_bgcolor;		/* foreground, background */
    char    txr_chrtype;		/* character type (see above) */
} GrTextRegion;

GrTextOption *GrFindBestFont(int width,int height,int magnify,char *family,GrTextOption *where);
GrFont *GrLoadFont(char *name);
GrFont *GrLoadBIOSFont(char *name);
void	GrUnloadFont(GrFont *font);
void	GrSetFontPath(char *path);

void	GrDrawChar(int chr,int x,int y,GrTextOption *opt);
void	GrDrawString(char *text,int length,int x,int y,GrTextOption *opt);
void	GrTextXY(int x,int y,char *text,int fg,int bg);
void	GrDumpChar(int chr,int col,int row,GrTextRegion *r);
void	GrDumpText(int col,int row,int wdt,int hgt,GrTextRegion *r);
void	GrDumpTextRegion(GrTextRegion *r);

int	GrFontHeight(GrTextOption *opt);
int	GrFontWidth(GrTextOption *opt);
int	GrCharWidth(int chr,GrTextOption *opt);
int	GrCharHeight(int chr,GrTextOption *opt);
int	GrStringWidth(char *text,int length,GrTextOption *opt);
int	GrStringHeight(char *text,int length,GrTextOption *opt);


/* ================================================================== */
/*		 DRAWING IN USER WINDOW COORDINATES		      */
/* ================================================================== */

void	GrSetUserWindow(int x1,int y1,int x2,int y2);
void	GrGetUserWindow(int *x1,int *y1,int *x2,int *y2);
void	GrGetScreenCoord(int *x,int *y);
void	GrGetUserCoord(int *x,int *y);

void	GrUsrPlot(int x,int y,int c);
void	GrUsrLine(int x1,int y1,int x2,int y2,int c);
void	GrUsrHLine(int x1,int x2,int y,int c);
void	GrUsrVLine(int x,int y1,int y2,int c);
void	GrUsrBox(int x1,int y1,int x2,int y2,int c);
void	GrUsrFilledBox(int x1,int y1,int x2,int y2,int c);
void	GrUsrFramedBox(int x1,int y1,int x2,int y2,int wdt,GrFBoxColors *c);
void	GrUsrCircle(int xc,int yc,int r,int c);
void	GrUsrEllipse(int xc,int yc,int xa,int ya,int c);
void	GrUsrCircleArc(int xc,int yc,int r,int start,int end,int c);
void	GrUsrEllipseArc(int xc,int yc,int xa,int ya,int start,int end,int c);
void	GrUsrFilledCircle(int xc,int yc,int r,int c);
void	GrUsrFilledEllipse(int xc,int yc,int xa,int ya,int c);
void	GrUsrFilledCircleArc(int xc,int yc,int r,int start,int end,int c);
void	GrUsrFilledEllipseArc(int xc,int yc,int xa,int ya,int start,int end,int c);
void	GrUsrPolyLine(int numpts,int points[][2],int c);
void	GrUsrPolygon(int numpts,int points[][2],int c);
void	GrUsrFilledConvexPolygon(int numpts,int points[][2],int c);
void	GrUsrFilledPolygon(int numpts,int points[][2],int c);
int	GrUsrPixel(int x,int y);

void	GrUsrCustomLine(int x1,int y1,int x2,int y2,GrLineOption *o);
void	GrUsrCustomBox(int x1,int y1,int x2,int y2,GrLineOption *o);
void	GrUsrCustomCircle(int xc,int yc,int r,GrLineOption *o);
void	GrUsrCustomEllipse(int xc,int yc,int xa,int ya,GrLineOption *o);
void	GrUsrCustomCircleArc(int xc,int yc,int r,int start,int end,GrLineOption *o);
void	GrUsrCustomEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrLineOption *o);
void	GrUsrCustomPolyLine(int numpts,int points[][2],GrLineOption *o);
void	GrUsrCustomPolygon(int numpts,int points[][2],GrLineOption *o);

void	GrUsrPatternedLine(int x1,int y1,int x2,int y2,GrLinePattern *lp);
void	GrUsrPatternedBox(int x1,int y1,int x2,int y2,GrLinePattern *lp);
void	GrUsrPatternedCircle(int xc,int yc,int r,GrLinePattern *lp);
void	GrUsrPatternedEllipse(int xc,int yc,int xa,int ya,GrLinePattern *lp);
void	GrUsrPatternedCircleArc(int xc,int yc,int r,int start,int end,GrLinePattern *lp);
void	GrUsrPatternedEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrLinePattern *lp);
void	GrUsrPatternedPolyLine(int numpts,int points[][2],GrLinePattern *lp);
void	GrUsrPatternedPolygon(int numpts,int points[][2],GrLinePattern *lp);

void	GrUsrPatternFilledPlot(int x,int y,GrPattern *p);
void	GrUsrPatternFilledLine(int x1,int y1,int x2,int y2,GrPattern *p);
void	GrUsrPatternFilledBox(int x1,int y1,int x2,int y2,GrPattern *p);
void	GrUsrPatternFilledCircle(int xc,int yc,int r,GrPattern *p);
void	GrUsrPatternFilledEllipse(int xc,int yc,int xa,int ya,GrPattern *p);
void	GrUsrPatternFilledCircleArc(int xc,int yc,int r,int start,int end,GrPattern *p);
void	GrUsrPatternFilledEllipseArc(int xc,int yc,int xa,int ya,int start,int end,GrPattern *p);
void	GrUsrPatternFilledConvexPolygon(int numpts,int points[][2],GrPattern *p);
void	GrUsrPatternFilledPolygon(int numpts,int points[][2],GrPattern *p);

GrTextOption *GrUsrFindBestFont(int width,int height,int magnify,char *family,GrTextOption *where);
void	GrUsrDrawChar(int chr,int x,int y,GrTextOption *opt);
void	GrUsrDrawString(char *text,int length,int x,int y,GrTextOption *opt);
void	GrUsrTextXY(int x,int y,char *text,int fg,int bg);


#ifdef __cplusplus
}
#endif

#endif  /* whole file */

