/**************************************************************************
 *
 * FILENAME:    xprtscr.c
 *
 * DESCRIPTION:
 *  A sample print screen program.
 *
 * PUBLIC FUNCTIONS:
 *  None
 *
 * NOTES:
 *  Syntax: xprtscr [-printer <printer>] [-display <display>]
 *
 *  Copyright (c) Quarterdeck Office Systems, Inc. 1992
 *
 * PVCS INFO:
 *  $Header::xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx$
 *
 * CHANGES:
 *  $Log::xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx$
 *
 **************************************************************************

static char id_xprtscr_c[3][] =  {
                                "$Workfile::xxxxxxxxxxxxxxxxxxxxxxxxxx$\n",
                                "$Revision::xxxxxxxxxxxxxxxxxxxxxxxxxx$\n",
                                "    $Date::xxxxxxxxxxxxxxxxxxxxxxxxxx$\n"
                                };
        
/*----- compilation and control switches --------------------------------*/


/**************************************************************************
 *  INCLUDE FILES
 *************************************************************************/

/*----- system and platform files ---------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>

/*----- program files ---------------------------------------------------*/


/**************************************************************************
 *  EXTERNAL REFERENCES
 *************************************************************************/

/*----- data declarations -----------------------------------------------*/

/*----- function prototypes ---------------------------------------------*/


/**************************************************************************
 *  PUBLIC DECLARATIONS
 *************************************************************************/

/*----- context ---------------------------------------------------------*/

/*----- data declarations -----------------------------------------------*/

/*----- function prototypes ---------------------------------------------*/


/**************************************************************************
 *  PRIVATE DECLARATIONS
 *************************************************************************/

/*----- context ---------------------------------------------------------*/

#define LOCAL_DISPLAY       ":0"
#define LOCAL_PRINTER       ":7"
#define PRINT_SCREEN        92

typedef enum
{
    FALSE = 0, TRUE
}               BOOL;

/*----- data declarations -----------------------------------------------*/

/*----- Set as a result of command line options. ------------------------*/

char           *pPrinterName = (char *) NULL;   /* Display name for prtr */
char           *pDisplayName = (char *) NULL;   /* Display name for disp */

/*----- X data structures for displaying. -------------------------------*/

Display        *pDisplay;       /* X display for display. */
int             ScreenNumber;	/* X screen number for printer. */
Window          DisplayRoot;    /* X window for display. */
GC              DisplayContext; /* X graphics context for display. */

/*----- X data structures for printing. ---------------------------------*/

Display        *pPrinter;       /* X display for printer. */
Window          Page;           /* X window for printer. */
GC              PrintContext;   /* X graphics context for printer. */

/*----- Page size information. ------------------------------------------*/

unsigned int    PageWidthInPixels;  /* Width of printer page in pixels. */
unsigned int    PageHeightInPixels; /* Height of printer page in pixels. */

/*----- function prototypes ---------------------------------------------*/

int             main(int argc, char **argv);
void            ParseArguments(int argc, char **argv);
void            Usage(void);
void            GrabKeys(void);
void            EventLoop(void);
void            PrintScreen(void);
void            StartPrintJob(void);
void            FinishPrintJob(void);
void            ReleaseKeys(void);

/*----- macros ----------------------------------------------------------*/



/**************************************************************************
 *  PUBLIC FUNCTION DECLARATIONS
 *************************************************************************/


/**************************************************************************
 *  PRIVATE FUNCTION DECLARATIONS
 *************************************************************************/


/**************************************************************************
 * NAME:
 *  main(int argc, char **argv)
 *
 * DESCRIPTION:
 *  Program entry point.
 *
 * INPUTS:
 *  PARAMATERS:
 *      int    argc     Count of command line arguments
 *      char **argv     Array of pointers to command line argument strings
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      Exits to DOS on completion.
 *
 * NOTES:
 *  Proposed enhancements
 *      Allow for printing of multiple files.
 *
 * CHANGES:
 *  None
 */

int
main(int argc, char **argv)
{

    /*********************************************************************/


    /****** CODE *********************************************************/

    ParseArguments(argc, argv);

	/*----- Establish connection to X Print Server ----------------------*/

	if (pDisplayName == (char *) NULL)
		pDisplayName = LOCAL_DISPLAY;

	pDisplay     = XOpenDisplay(pDisplayName);
	ScreenNumber = DefaultScreen(pDisplay);
    DisplayRoot  = RootWindow(pDisplay, ScreenNumber);

	if (pDisplay == (Display *) NULL)
		{
		fprintf(stderr,
				"xprtscr: cannot connect to X display %s\n",
				pDisplayName);

		exit(-1);
		}


    GrabKeys();

    EventLoop();

    ReleaseKeys();


	/*----- Close Display -----------------------------------------------*/

	XCloseDisplay(pDisplay);

    exit(0);
}


/**************************************************************************
 * NAME:
 *  ParseArguments(int argc, char ** argv)
 *
 * DESCRIPTION:
 *  Parse command line arguments and set corresponding global variables.
 *
 * INPUTS:
 *  PARAMATERS:
 *      int    argc     Count of command line arguments
 *      char **argv     Array of pointers to command line argument strings
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  None
 *
 * CHANGES:
 *  None
 */

void
ParseArguments(int argc, char **argv)
{

    /*********************************************************************/

    int             ArgumentIndex;

    /****** CODE *********************************************************/

    for (ArgumentIndex = 1; ArgumentIndex < argc; ArgumentIndex++)
        {
        if (strcmp(argv[ArgumentIndex], "-printer") == 0)
            {
            if (++ArgumentIndex >= argc)
                Usage();

            pPrinterName = argv[ArgumentIndex];

            continue;
            }

        if (strcmp(argv[ArgumentIndex], "-display") == 0)
            {
            if (++ArgumentIndex >= argc)
                Usage();

            pDisplayName = argv[ArgumentIndex];

            continue;
            }

        Usage();
        }
}


/**************************************************************************
 * NAME:
 *  Usage(void)
 *
 * DESCRIPTION:
 *  Prints usage message and then exits to DOS.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  This should be updated when command line switches are added.
 *
 * CHANGES:
 *  None
 */

void
Usage(void)
{

    /*********************************************************************/


    /****** CODE *********************************************************/

    fprintf(stderr, "usage: xprtscr [-printer <printer>] [-display <display>]\n");

    exit(1);
}


/**************************************************************************
 * NAME:
 *  GrabKeys(void)
 *
 * DESCRIPTION:
 *  Sets a passive grap on the PrintScreen key.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Grab the keys that we want.
 *
 * CHANGES:
 *  None
 */

void
GrabKeys(void)
{

    /*********************************************************************/

    /****** CODE *********************************************************/

    XGrabKey(pDisplay,
             PRINT_SCREEN,
             0,
             DisplayRoot,
             False,
             GrabModeSync,
             GrabModeAsync);

}


/**************************************************************************
 * NAME:
 *  EventLoop(void)
 *
 * DESCRIPTION:
 *  This is the main event loop.  It sets up the events it is interested in
 *  and then loops waiting for these events.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Grab the keys that we want.
 *
 * CHANGES:
 *  None
 */

void
EventLoop(void)
{

    /*********************************************************************/

    XEvent  Event;

    /****** CODE *********************************************************/

    fprintf(stdout, "Waiting for the  PrintScrn key to be pressed...\n");
    XSelectInput(pDisplay, 
                 DisplayRoot,
                 StructureNotifyMask | KeyPressMask | KeyReleaseMask);

    while (1)
        {
        XNextEvent(pDisplay, &Event);

        switch (Event.type)
            {
            case KeyPress:
#ifdef DEBUG
                fprintf(stdout, "KeyPress: \n");
#endif
                fprintf(stdout, "Printing screen\n");
                PrintScreen();
                break;

            case KeyRelease:
#ifdef DEBUG
                fprintf(stdout, "KeyRelease: \n");
#endif
                break;

            case DestroyNotify:
                return;
                break;

            default:
                fprintf(stdout, "Unrecognized event: %d\n", Event.type);
            }
        }
}


/**************************************************************************
 * NAME:
 *  PrintScreen(void)
 *
 * DESCRIPTION:
 *  Get the screen image and send it to the X printer
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Release the keys that were grabbed.
 *
 * CHANGES:
 *  None
 */

void
PrintScreen(void)
{

    /*********************************************************************/

    XWindowAttributes   WindowAttributes;
    XImage             *pImage;
    Status              status;

    int                 ImageX;
    int                 ImageY;

    unsigned int        ImageWidth;
    unsigned int        ImageHeight;

    unsigned int        PixelsPerByte;
    unsigned int        BytesPerLine;

    /****** CODE *********************************************************/

    StartPrintJob();

    status = XGetWindowAttributes(pDisplay, DisplayRoot, &WindowAttributes);

    if(status == 0)
        {
        fprintf(stderr, "Can't get target window attributes.");
        exit(1);
        }

    /*
     * Server can't currently handle GetImage > 16K so we do multiple
     * GetImages.
     */

    ImageWidth    = WindowAttributes.width;
    PixelsPerByte = 8 / WindowAttributes.depth;
    BytesPerLine  = ImageWidth / PixelsPerByte;
    ImageHeight   = 0x2000 / BytesPerLine;

    ImageX = 0;

	for (ImageY = 0; ImageY < WindowAttributes.height - 1; ImageY += ImageHeight)
		{
		if (ImageY + ImageHeight > WindowAttributes.height)
			ImageHeight = WindowAttributes.height - ImageY;

#ifdef DEBUG
        fprintf(stdout, "%d, %d, %d, %d\n", ImageX, ImageY, ImageWidth, ImageHeight);
#endif

        pImage = XGetImage(pDisplay,
                           DisplayRoot,
                           ImageX,
                           ImageY,
                           ImageWidth,
                           ImageHeight,
                           AllPlanes,
                           ZPixmap);

		if (pImage == (XImage *) NULL)
            {
            fprintf(stderr, "Unable to get image\n");
            exit(1);
            }

        XPutImage(pPrinter,
                  Page,
                  PrintContext,
                  pImage,
                  0,
                  0,
                  ImageX,
                  ImageY,
                  ImageWidth,
                  ImageHeight);

		XDestroyImage(pImage);
		XFlush(pPrinter);
		}

    FinishPrintJob();
}


/**************************************************************************
 * NAME:
 *  StartPrintJob(void)
 *
 * DESCRIPTION:
 *  Connect to X Print Server and setup up printing environment.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      char  *pPrinterName     Pointer to display name for printer.
 *      double FontSize         Font size to use when printing (in points).
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      Display     *pPrinter;              X display for printer.
 *      Window       Page;                  X window for printer.
 *      GC           PrintContext;          X graphics context for printer.
 *      unsigned int PageWidthInPixels      Width of print page in pixels.
 *      unsigned int PageHeightInPixels     Height of print page in pixels.
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Basic steps for starting print job.
 *      Connect to X Print Server.
 *      Query for page size and resolution.
 *      Create window that will represent the printed page.
 *      Create X font string and load printer font.
 *      Create graphics context for printing.
 *
 * CHANGES:
 */

void
StartPrintJob(void)
{

	/*********************************************************************/

	int             ScreenNumber;	/* X screen number for printer. */

	/****** CODE *********************************************************/

	/*----- Establish connection to X Print Server ----------------------*/

	if (pPrinterName == (char *) NULL)
		pPrinterName = LOCAL_PRINTER;

	pPrinter = XOpenDisplay(pPrinterName);

	if (pPrinter == NULL)
		{
		fprintf(stderr,
				"xprint: cannot connect to X printer %s\n",
				pPrinterName);

		exit(-1);
		}


	/*----- Query information from XLib ---------------------------------*/

	ScreenNumber = DefaultScreen(pPrinter);

	PageWidthInPixels = DisplayWidth(pPrinter, ScreenNumber);

	PageHeightInPixels = DisplayHeight(pPrinter, ScreenNumber);

	/*----- Create window that will represent the printed page. ---------*/

	Page = XCreateSimpleWindow(pPrinter,
							   RootWindow(pPrinter, ScreenNumber),
							   0,
							   0,
							   PageWidthInPixels,
							   PageHeightInPixels,
							   0,
							   BlackPixel(pPrinter, ScreenNumber),
							   WhitePixel(pPrinter, ScreenNumber));

	PrintContext = DefaultGC(pPrinter, ScreenNumber);
}


/**************************************************************************
 * NAME:
 *  FinishPrintJob(void)
 *
 * DESCRIPTION:
 *  Closes connection to X Print Server.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      Display     *pPrinter               X display for printer
 *      Window       Page;                  X window for printer.
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  An X print job is finished when the connection to the X Print Server is
 *  closed.  A side effect of this is that the last page is automatically
 *  ejected.
 *
 * CHANGES:
 *  None
 */

void
FinishPrintJob(void)
{

	/*********************************************************************/


	/****** CODE *********************************************************/

	/*----- Free drawing window -----------------------------------------*/

	XDestroyWindow(pPrinter, Page);

	/*----- Close Display ending print jobs -----------------------------*/

	XCloseDisplay(pPrinter);
}


/**************************************************************************
 * NAME:
 *  ReleaseKeys(void)
 *
 * DESCRIPTION:
 *  Releases the passive grab on the PrintScreen key.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Release the keys that were grabbed.
 *
 * CHANGES:
 *  None
 */

void
ReleaseKeys(void)
{

    /*********************************************************************/

    /****** CODE *********************************************************/

    XUngrabKey(
             pDisplay,
             PRINT_SCREEN,
             0,
             DisplayRoot
            );
}
