/* Project SWORD
   V2.0

   SubSystem : Mathematical toolbox
   File      : LibSrc/ToolBox/Math/Poly.CC
   Author    : Eric NICOLAS
   Overview  : Using vectors as polynomial functions
   UpDate    : May 09, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <math.h>
#include "Common/Common.H"
#include "ToolBox/Math/MError.H"
#include "ToolBox/Math/Complex.H"
#include "ToolBox/Math/Vector.H"
#include "ToolBox/Math/Matrix.H"
#include "ToolBox/Math/Poly.H"

#define Epsilon 1E-5

TPoly::TPoly(int Degree) : TVector(Degree+1)
{ }

complex TPoly::Evaluate(complex X)
{ int     i;
  complex R=Tabl[Size-1];
  if (Size>0)
    for(i=Size-2;i>=0;i--) R=Tabl[i]+X*R;
  return R;
}

TPoly TPoly::Factor(complex X0)
{ if (Degree()==0) MathError=MError_Poly_NoFactorisation;
  if (MathError) return (*this);
  // Create the new polynomial function. NewDegree = Degree - 1
  TPoly P(Degree()-1);
  int   i;
  // Compute new coeficients
  P(Size-2)=Tabl[Size-1];
  for(i=Size-3;i>=0;i--)
    P(i)=Tabl[i+1]+X0*P(i+1);
  return P;
}

int TPoly::Degree(void)
{ return Size-1;
}

TPoly TPoly::Differentiate(void)
{ int NewDegree,i;
  if (Degree()>0) NewDegree=Degree()-1; else NewDegree=0;
  TPoly D(NewDegree);
  if (Degree()>0)
  { for(i=0;i<=NewDegree;i++)
      D(i)=Tabl[i+1]*(i+1);
  }
  else D(0)=0;
  return D;
}

float TPoly::Radius(void)
{ float Somme=0,r;
  int   i,j;
  // Seek for the dominant term in the polynomial function
  j=Size-1;
  while((abs(Tabl[j])==0)&&(j>0)) j--;
  if (j==0) return 0;
  //
  for(i=0;i<j;i++)
    Somme+=abs(Tabl[i]);
  r=Somme/abs(Tabl[j]);
  if (r<1) return 1;
      else return r;
}

complex TPoly::Solve(void)
{ TPoly   D=Differentiate();
  float   r=Radius();
  complex Step(r,r);
  complex Pos(0,0),OldPos;
  complex Grad;
  complex Pz0,PzD,Pz1,Z;
  float   mStep,mGrad,mm,ps;
  boolean Found=FALSE;

  while(!Found)
  { OldPos=Pos;
    // Apply movement
    Pos+=Step;
    mStep=abs(Step);
    // Compute Grad
    Pz0=Evaluate(Pos);
    PzD=D.Evaluate(Pos);
    Grad=2*complex( Pz0.re*PzD.re+Pz0.im*PzD.im,
		   -Pz0.re*PzD.im+Pz0.im*PzD.re);
    mGrad=abs(Grad);
    mm=mStep/mGrad;
    // Change Step
    ps=Grad.re*Step.re+Grad.im*Step.im;
    if (ps!=0) ps/=mStep*mGrad;
    if (ps>0) Step=-Grad*mm/(1+ps);
    // Test for the end of the search
    Z=Pos+Grad;
    Pz1=Evaluate(Z);
    Pz0=Pz1-Pz0;
    PzD=OldPos-Pos;
    if ((abs(PzD)<Epsilon) || (abs(Pz0)<Epsilon))
    { Pz0=Evaluate(Pos);
      if (abs(Pz0)<Epsilon) Found=TRUE;
		       else Step=complex(r,r);
    }
  }
  return Pos;
}
