/* $Id: dedupe.c,v 1.1 2002/07/21 18:24:44 richdawe Exp $ */

/*
 * dedupe.c - Check that package_dedupe() copes with different
 * dsm-file-versions.
 * Copyright (C) 2002 by Richard Dawe
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <libpakke/package.h>
#include <libpakke/packlist.h>
#include <libpakke/user.h>

/* -----------------
 * - alloc_package -
 * ----------------- */

static PACKAGE_INFO *
alloc_package (const char *user_string)
{
  PACKAGE_INFO *package;

  package = calloc(1, sizeof(*package));

  if ((package != NULL) && !user_parse_package_string(user_string,
						      package->name,
						      PACKAGE_MAXNAMELEN,
						      &package->version)) {
    free(package);
    package = NULL;
  }

  return(package);
}

/* --------
 * - main -
 * -------- */

int
main (int argc, char *argv[])
{
#define N_PACKAGES 10
  PACKAGE_INFO *packages_array[N_PACKAGES];
  PACKAGE_INFO *packages = NULL;
  PACKAGE_INFO *package  = NULL;
  int i;
  int quiet = 0;
  int ret = EXIT_SUCCESS; /* Succeed by default */

  /* Parse arguments. */
  for (i = 1; i < argc; i++) {
    /* Quiet operation, for automatic testing. */
    if (   (strcmp(argv[i], "--quiet") == 0)
	|| (strcmp(argv[i], "-q") == 0))
      quiet = 1;
  }

  for (i = 0; i < N_PACKAGES; i++) {
    packages_array[i] = alloc_package("foo 1.0 (binaries)");
    assert(packages_array[i] != NULL);

    /* Set up the dsm-file-version to be slightly different
     * for each package. */
    packages_array[i]->dsm_file_version.has_version
      = packages_array[i]->dsm_file_version.has_major
      = packages_array[i]->dsm_file_version.has_minor
      = 1;

    packages_array[i]->dsm_file_version.major = 1;
    packages_array[i]->dsm_file_version.minor = i + 1; /* 1...N_PACKAGES */
  }

  /* Link list */
  for (i = 0; i < N_PACKAGES; i++) {
    if (i)
      packages_array[i]->q_back = packages_array[i - 1];

    if (i < (N_PACKAGES - 1))
      packages_array[i]->q_forw = packages_array[i + 1];
  }

  packages = packages_array[0];

  /* Display packages in list. */
  if (!quiet) {
    printf("Packages before dedupe:\n");

    for (package = packages; package != NULL; package = package->q_forw) {
      printf("\t%s %s: ",
	     package->name,
	     package_version_string(&package->version));
      printf("dsm-file-version: %s\n",
	     package_version_string(&package->dsm_file_version));
    }
  }

  /* Dedupe */
  packages = packlist_dedupe(packages);
  /* NB: packages_array is no longer valid! */

  /* Display packages left in list. */
  if (!quiet) {
    printf("Packages after dedupe:\n");

    for (package = packages; package != NULL; package = package->q_forw) {
      printf("\t%s %s: ",
	     package->name,
	     package_version_string(&package->version));
      printf("dsm-file-version: %s\n",
	     package_version_string(&package->dsm_file_version));
    }
  }

  /* Check that's there only one package left. */
  if ((packages->q_back != NULL) && (packages->q_forw != NULL)) {
    if (!quiet)
      printf("package_dedupe() did not remove "
	     "the older duplicate package descriptions!\n");

    ret = EXIT_FAILURE;
  }

  /* Check that the highest dsm-file-version one is the one left. */
  if (packages->dsm_file_version.minor != N_PACKAGES) {
    if (!quiet)
      printf("package_dedupe() did not leave the expected package!\n");

    ret = EXIT_FAILURE;
  }

  /* Tidy up */
  packlist_free(packages);
  packages = NULL;
#undef N_PACKAGES

  return(ret);
}
