/* libpgm1.c - pgm utility library part 1
**
** Copyright (C) 1989 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/
#define _POSIX_SOURCE
    /* This makes sure fileno() is in stdio.h */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include "pgm.h"
#include "libpgm.h"
#include "pbm.h"
#include "libpbm.h"

void
pgm_init( argcP, argv )
    int* argcP;
    char* argv[];
    {
    pbm_init( argcP, argv );
    }

void
pgm_nextimage(FILE *file, int * const eofP) {
    pm_nextimage(file, eofP);
}


void
pgm_readpgminitrest( file, colsP, rowsP, maxvalP )
    FILE* file;
    int* colsP;
    int* rowsP;
    gray* maxvalP;
    {
    int maxval;

    /* Read size. */
    *colsP = pbm_getint( file );
    *rowsP = pbm_getint( file );

    /* Read maxval. */
    maxval = pbm_getint( file );
    if ( maxval > PGM_OVERALLMAXVAL )
	pm_error( "maxval is too large.  The largest we can handle is %d.", 
              PGM_OVERALLMAXVAL );
    *maxvalP = maxval;
    }

gray pgm_pbmmaxval = 1;

void
pgm_readpgminit( file, colsP, rowsP, maxvalP, formatP )
    FILE* file;
    int* colsP;
    int* rowsP;
    int* formatP;
    gray* maxvalP;
    {
    /* Check magic number. */
    *formatP = pbm_readmagicnumber( file );
    switch ( PGM_FORMAT_TYPE(*formatP) )
	{
        case PGM_TYPE:
	pgm_readpgminitrest( file, colsP, rowsP, maxvalP );
	break;

        case PBM_TYPE:
	pbm_readpbminitrest( file, colsP, rowsP );
	*maxvalP = pgm_pbmmaxval;
	break;

	default:
	pm_error( "bad magic number - not a pgm or pbm file" );
	}
    }

gray
pgm_getrawsample(FILE *file, const gray maxval) {

    if (maxval < 256) {
        /* The sample is just one byte.  Read it. */
        return(pbm_getrawbyte(file));
    } else {
        /* The sample is two bytes.  Read both. */
        unsigned char byte_pair[2];
        size_t pairs_read;

        pairs_read = fread(&byte_pair, 2, 1, file);
        if (pairs_read == 0) 
            pm_error("EOF /read error while reading a long sample");
        /* This could be a few instructions faster if exploited the internal
           format (i.e. endianness) of a pixval.  Then we might be able to
           skip the shifting and oring.
           */
        return((byte_pair[0]<<8) | byte_pair[1]);
    }
}



#if __STDC__
void
pgm_readpgmrow( FILE* file, gray* grayrow, int cols, gray maxval, int format )
#else /*__STDC__*/
void
pgm_readpgmrow( file, grayrow, cols, maxval, format )
    FILE* file;
    gray* grayrow;
    int cols;
    gray maxval;
    int format;
#endif /*__STDC__*/
    {
    register int col;
    register gray* gP;
    bit* bitrow;
    register bit* bP;

    switch ( format )
	{
	case PGM_FORMAT:
	for ( col = 0, gP = grayrow; col < cols; ++col, ++gP )
	    {
	    *gP = pbm_getint( file );
#ifdef DEBUG
	    if ( *gP > maxval )
		pm_error( "value out of bounds (%u > %u)", *gP, maxval );
#endif /*DEBUG*/
	    }
	break;
	
	case RPGM_FORMAT:
        for ( col = 0; col < cols; ++col) {
	    grayrow[col] = pgm_getrawsample( file, maxval );
#ifdef DEBUG
	    if ( grayrow[col] > maxval )
		pm_error( "value out of bounds (%u > %u)", grawrow[col], maxval );
#endif /*DEBUG*/
	    }
	break;
	
	case PBM_FORMAT:
	case RPBM_FORMAT:
	bitrow = pbm_allocrow( cols );
	pbm_readpbmrow( file, bitrow, cols, format );
	for ( col = 0, gP = grayrow, bP = bitrow; col < cols; ++col, ++gP, ++bP )
	    *gP = ( *bP == PBM_WHITE ) ? maxval : 0;
	pbm_freerow( bitrow );
	break;

	default:
	pm_error( "can't happen" );
	}
    }

gray**
pgm_readpgm( file, colsP, rowsP, maxvalP )
    FILE* file;
    int* colsP;
    int* rowsP;
    gray* maxvalP;
    {
    gray** grays;
    int row;
    int format;

    pgm_readpgminit( file, colsP, rowsP, maxvalP, &format );

    grays = pgm_allocarray( *colsP, *rowsP );

    for ( row = 0; row < *rowsP; ++row )
	pgm_readpgmrow( file, grays[row], *colsP, *maxvalP, format );

    return grays;
    }


void
pgm_check(FILE * file, const enum pm_check_type check_type, 
          const int format, const int cols, const int rows, const int maxval,
          enum pm_check_code * const retval_p) {

    if (check_type != PM_CHECK_BASIC) {
        if (retval_p) *retval_p = PM_CHECK_UNKNOWN_TYPE;
    } else if (PGM_FORMAT_TYPE(format) == PBM_TYPE) {
        pbm_check(file, check_type, format, cols, rows, retval_p);
    } else if (format != RPGM_FORMAT) {
        if (retval_p) *retval_p = PM_CHECK_UNCHECKABLE;
    } else {        
        const unsigned int bytes_per_row =
            cols * (maxval > 255 ? 2 : 1);
        const unsigned int need_raster_size = rows * bytes_per_row;
        
        pm_check(file, check_type, need_raster_size, retval_p);
        
    }
}
