#! /usr/bin/perl
##
## vi:ts=4
##
##---------------------------------------------------------------------------##
##
##  Author:
##      Markus F.X.J. Oberhumer         markus.oberhumer@jk.uni-linz.ac.at
##
##  Description:
##      Convert a disassmbly to a MASM/TASM/WASM assembler file.
##
##  Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
##
##---------------------------------------------------------------------------##

$VERSION = '0.01.1';
$PROG = $0;

sub usage {
	print <<"EOF";
usage: $PROG [-o file.asm] file.dis
  -c cc    select compiler/assembler
  -o file  print to 'file' (instead of stdout)
  -D       print debug infos
  -V       print version
EOF
	exit(1);
}


# /***********************************************************************
# // init
# ************************************************************************/

$opt_c = "asm";

# options
require 'getopts.pl';
&Getopts('c:o:DV') || &usage();
print STDERR "$PROG version $VERSION\n" if ($opt_V || $opt_D);
exit(0) if ($opt_V);

$opt_c =~ tr/A-Z/a-z/;


# open output file
if ($opt_o) {
	open(HANDLE,">$opt_o") || die "$PROG: cannot open '$opt_o': $!\n";
} else {
	*HANDLE = STDOUT;
}
binmode(HANDLE);


# /***********************************************************************
# //
# ************************************************************************/

&asm_header();

&parse_wdisasm if ($opt_c eq "asm");
&parse_objdump if ($opt_c eq "db");

&asm_footer();

exit(0);


# /***********************************************************************
# //
# ************************************************************************/

sub parse_wdisasm {
	local ($state) = 0;
	local ($x, $l, $i);

	while (<>) {
		chop;
		$x = $_;
		$l = substr($_,25);

		if ($state == 0) {
			next unless (/^ 0000/);
			$state++;
			# fall through
		}

		if ($state == 1) {
			next unless ($l =~ /^_*(lzo[\w]+)\:/);
			&asm_public($1);
			$state++;
			next;
		}

		if ($state == 2) {
			if (/^----/) {
				$state++;
				next;
			}
			$l =~ s/^(\w+) /$1\:/;
			$l =~ s/\s+$//;
			print HANDLE "$l\n" if ($l);
		}

	}
}


# /***********************************************************************
# //
# ************************************************************************/

sub parse_objdump {
	local ($state) = 0;
	local ($b, $i);

	while (<>) {
		chop;

		if ($state == 0) {
			next unless (/^\s*0+\s+\<_*(lzo[\w]+)\>/);
			&asm_public($1);
			$state++;
			# fall through
		}

		if (/^\s*\w+\s+\<[\w\-\+]+\>\s+/) {
			$b = $';
			$i = 0;
			print HANDLE "db";
			while ($b =~ s/^\s*([0-9a-fA-F]{2})\s+//) {
				print HANDLE "," if ($i++ > 0);
				printf HANDLE " %d", hex($1);
			}
			print HANDLE "\n";
			die if ($i == 0);
		}
	}
}


# /***********************************************************************
# //
# ************************************************************************/

$asm_public_s = "";

sub asm_public {
	local ($s) = @_;
	&asm_public_wc($s) if $opt_c eq "asm";
	&asm_public_wc($s) if $opt_c eq "db";
	$asm_public_s = $s;
}

sub asm_public_wc {
	local ($s) = @_;
	$s = "_" . $s;

	print HANDLE ".code\n";
##	print HANDLE "public \"C\",$s\n";
	print HANDLE "public $s\n";
	print HANDLE "\n";
##	print HANDLE "align 4\n";
	print HANDLE "$s:\n";
}


# /***********************************************************************
# //
# ************************************************************************/

sub asm_header {
	&asm_comment("/*** DO NOT EDIT - GENERATED AUTOMATICALLY ***/\n");
	&asm_comment("/*** Copyright (C) 1996, 1997 Markus F.X.J. Oberhumer ***/\n");
	print HANDLE "\n";
	&asm_header_wc if $opt_c eq "asm";
	&asm_header_wc if $opt_c eq "db";
}


sub asm_header_wc {
	print HANDLE ".386p\n";
	print HANDLE ".model flat\n";
	print HANDLE "\n";
}

sub asm_comment {
	local ($s) = @_;
	print HANDLE "; ", $s;
}


# /***********************************************************************
# //
# ************************************************************************/

sub asm_footer {
	print HANDLE <<"EOF";
EOF
	&asm_footer_wc if $opt_c eq "asm";
	&asm_footer_wc if $opt_c eq "db";
}


sub asm_footer_wc {
	print HANDLE <<"EOF";

end
EOF
}


