#include <ctype.h>
#include <keys.h>

#include "appl.h"
#include "desktop.h"
#include "dialog.h"
#include "dlgobj.h"
#include "error.h"
#include "event.h"
#include "menu.h"

/*

	event.c

		Handles the	events.

		C-Desktop
		Copyright (C)1998, Brett Porter.

*/

/*
   TYPE DEFINITIONS
*/

typedef struct	S_KeyBinding
{
	int	fKey;
	int	fCommand;

	struct S_KeyBinding	*fNext;
} T_KeyBinding;

typedef struct	S_EventStatus
{
	int	fID;
	bool	fActive;

	struct S_EventStatus	*fNext;
} T_EventStatus;

typedef struct S_Message
{
	E_MessageType	fMessage;

	struct S_Message	*fNext;
} T_Message;

/*
	DATA
*/

static	T_EventStatus	*sEvent_CommandStatusList;
static	T_KeyBinding	*sEvent_KeyBindingList;
static	T_Message		*sEvent_MessageQueue;

T_Group	*gEvent_ModalGroup;

/*
   FUNCTION PROTOTYPES
*/

static	bool	Event_CheckKeyBinding( int aKeypress );

static	T_EventStatus	*Event_FindCommand( int aID );

/*
	FUNCTION	DEFINITIONS
*/

void	Event_Initialise( void )
{
	gEvent_ModalGroup = NULL;
	sEvent_KeyBindingList = NULL;
	sEvent_CommandStatusList = NULL;
	sEvent_MessageQueue = NULL;
}

void	Event_Terminate( void )
{
	T_EventStatus	*lCurrent = sEvent_CommandStatusList, *lRec;
	T_KeyBinding	*lBinding = sEvent_KeyBindingList, *lPrev;
	T_Message		*lMessage = sEvent_MessageQueue, *lLast;

	while ( lCurrent != NULL )
	{
		lRec = lCurrent;
		lCurrent = lCurrent->fNext;
		free( lRec );
	}

	while ( lBinding != NULL )
	{
		lPrev = lBinding;
		lBinding = lBinding->fNext;
		free( lPrev );
	}

	while ( lMessage != NULL )
	{
		lLast = lMessage;
		lMessage = lMessage->fNext;
		free( lLast );
	}
}

void	Event_HandleLeftMousePress( int aColumn, int aRow )
{
	Group_HandleLeftMousePress( gEvent_ModalGroup, aColumn-gEvent_ModalGroup->fX1, aRow-gEvent_ModalGroup->fY1 );
}

void	Event_HandleRightMousePress( int aColumn, int aRow )
{
	Group_HandleRightMousePress( gEvent_ModalGroup, aColumn-gEvent_ModalGroup->fX1, aRow-gEvent_ModalGroup->fY1 );
}

void	Event_HandleCommand( int aID )
{
	T_EventStatus	*lCurrent = Event_FindCommand( aID );

	if ( !lCurrent->fActive )
	{
		return;
	}

	if ( !Application_HandleCommand( aID ))
	{
		switch ( aID )
		{
			case	CMD_EXIT:
				if ( Application_ConfirmDialog( "Are you sure you want to exit?", true ) == ENDMODAL_YES )
				{
					gEvent_ModalGroup->fCloseModalCommand = ENDMODAL_CLOSE;
					Event_SendMessage( MSG_CLOSEMODAL );
				}
				break;
			default:
				Error( "Unhandled command" );
				break;
		}
	}
}

void	Event_HandleKeypress( int aKeypress )
{
	if ( !Group_HandleKeypress( gEvent_ModalGroup, aKeypress ))
	{
		if ( gEvent_ModalGroup == gDesktop )	/// better way to handle this...
		{
			Event_CheckKeyBinding( aKeypress );
		}
	}
}

void	Event_EnableCommand( int aID, bool aEnabled )
{
	T_EventStatus	*lCurrent;
	T_EventRec		lEvent;

	lCurrent = Event_FindCommand( aID );

	lCurrent->fActive = aEnabled;

	lEvent.fMessage = EVENT_COMMANDENABLED;
	lEvent.fParameters.fCommandStatus.fID = aID;
	lEvent.fParameters.fCommandStatus.fEnabled = aEnabled;

	Group_Broadcast( gDesktop, &lEvent );
}

void	Event_AddKeyBinding( int aKey, int aCommand )
{
	T_KeyBinding	*lBinding;

	P_InitVar( lBinding, T_KeyBinding );

	lBinding->fKey = aKey;
	lBinding->fCommand = aCommand;
	lBinding->fNext = sEvent_KeyBindingList;

	sEvent_KeyBindingList = lBinding;
}

bool	Event_CheckKeyBinding( int aKeypress )
{
	T_KeyBinding	*lBinding = sEvent_KeyBindingList;

	while ( lBinding != NULL )
	{
		if ( lBinding->fKey == aKeypress )
		{
			T_EventStatus	*lCommand = Event_FindCommand( lBinding->fCommand );

			if ( lCommand->fActive )
			{
				Event_HandleCommand( lBinding->fCommand );
				return true;
			}
		}
		lBinding = lBinding->fNext;
	}
	return false;
}

T_EventStatus	*Event_FindCommand( int aID )
{
	T_EventStatus	*lCurrent =	sEvent_CommandStatusList;

	while ( lCurrent != NULL )
	{
		if ( lCurrent->fID == aID )
		{
			return lCurrent;
		}
		lCurrent = lCurrent->fNext;
	}
	FatalError( "Command not found" );
}

void	Event_RegisterCommand( int aID )
{
	T_EventStatus	*lCurrent;

	P_InitVar( lCurrent, T_EventStatus );

	lCurrent->fID = aID;
	lCurrent->fActive = false;
	lCurrent->fNext = sEvent_CommandStatusList;
	sEvent_CommandStatusList = lCurrent;
}

void	Event_SendMessage( E_MessageType aMessage )
{
	T_Message	*lNewMessage;

	P_InitVar( lNewMessage, T_Message );

	lNewMessage->fMessage = aMessage;
	lNewMessage->fNext = sEvent_MessageQueue;

	sEvent_MessageQueue = lNewMessage;
}

bool	Event_CheckMessage( E_MessageType aMessage )
{
	T_Message	*lMessage = sEvent_MessageQueue, *lLast = NULL;

	while ( lMessage != NULL )
	{
		if ( lMessage->fMessage == aMessage )
		{
			if ( lLast != NULL )
			{
				lLast->fNext = lMessage->fNext;
			}
			else
			{
				sEvent_MessageQueue = lMessage->fNext;
			}
			free( lMessage );
			return true;
		}
		lLast = lMessage;
		lMessage = lMessage->fNext;
	}
	return false;
}

bool	Event_CommandEnabled( int aID )
{
	return Event_FindCommand( aID )->fActive;
}


