/****************************************************************************/
/* TAPPLICA                                                                 */
/*--------------------------------------------------------------------------*/
/* TApplication class : application                                         */
/*--------------------------------------------------------------------------*/
/* Most of the methods of this class are also provided as macros            */
/* (see JPAPPLI.H)                                                          */
/****************************************************************************/

#ifndef _TAPPLICA_H_
#define _TAPPLICA_H_

#include <sys\timeb.h>
#include "Keyboard.h"
#include "Types.h"

/*ͻ*/
/*                                 TYPES                                  */
/*ͼ*/

// List of windows

typedef struct window_node
{
  PWindow                   window;
  struct window_node        *next;
  struct window_node        *last;
} TWindowNode,*PWindowNode,*PWindowList;

// List of open windows

typedef struct open_window_node
{
  int                       window_number;
  int                       order;
  PObject                   owner;
  struct open_window_node   *next;
  struct open_window_node   *last;
} TOpenWindowNode,*POpenWindowNode,*POpenWindowList;


/*ͻ*/
/*                             CLASS DEFINITION                           */
/*ͼ*/

class TApplication
{
  /*----------------*/
  /* Friend classes */
  /*----------------*/

  friend                TObject;
  friend                TWindow;

  /*----------------*/
  /* Private fields */
  /*----------------*/

  private:

    // Application currently running
    // Only one application can be running

    static
    PApplication        f_current_application;

    // Windows of the application

    int                 f_nb_windows;
    PWindowList         f_window_list,
			f_last_window;

    PWindow             f_active_window;

    // Number and list of open windows by order of superposition

    int                 f_nb_open_windows;
    POpenWindowList     f_open_window_list,
			f_last_open_window;

    // Application state

    enum                { STOPPED,INITIALIZED,RUNNING }
			f_state;

    // Number of dialogs currently running

    int                 f_nb_running_dialogs;

    // The application or the current dialog
    // is going to be stopped as soon as possible

    boolean             f_stop_wanted;

    // Callback processing is enabled

    boolean             f_callbacks_enabled;

    // Double-click state

    enum    { FIRST_PRESSION,
	      FIRST_RELEASE,
	      SECOND_PRESSION,
	      SECOND_RELEASE
	    }
			f_double_click_state;

    // Last mouse button click time

    struct timeb        f_last_button_click_time;

    // The screen need to be refreshes
    // What needs to be refreshed

    boolean             f_need_refresh;
    TIntZone            f_zone_to_refresh;
    int                 f_upper_changed_window_height;

    // Clipboard

    char                *f_clipboard;

    // Callback invoked when no event occurs

    void                (*f_no_event_action)(const char *);
    const char *f_no_event_argument;

    // Callback invoked when the contents of the clipboard change.

    void                (*f_clipboard_changed_action)(const char *);
    const char *f_clipboard_changed_argument;

  /*----------------*/
  /* Public methods */
  /*----------------*/

  public:

    // Constructor / Destructor

			TApplication(TLanguage language);
    virtual             ~TApplication();

    // Handler called in case a memory allocation fails :
    // stops the program

//	    static
//	    void        m_new_handler();

    // Handler called when the user use CTRL-BREAK

	    static
	    void        m_ctrl_break_handler(int);

    // Returns the running application

	    static
	    PApplication
			m_get_current_application() { return(f_current_application);};

    // Initializes and starts the application, runs a dialog,
    // processes events once, stops a dialog or the application

	    void        m_init();
	    void        m_run();
	    void        m_run_dialog();
	    void        m_process_events();
	    void        m_stop();


    // Refreshes :
    // - the part of the screen that has changed
    // - the whole screen

	    void        m_refresh();
	    void        m_refresh_all();

    // Redraws the contents of windows

	    void        m_redraw_windows();

    // Makes the next window (relatively to the currently-selected
    // window) active

	    boolean     m_make_next_window_active();

    // Returns a pointer to the active window (NULL if none)

	    PWindow     m_get_active_window();

    // Clipboard reading and writing

	    char        *m_get_clipboard()    { return(f_clipboard);};
	    void        m_set_clipboard(const char *text);

    // Enables/Disables callback processing

	    void        m_enable_callbacks()  { f_callbacks_enabled=TRUE;};
	    void        m_disable_callbacks() { f_callbacks_enabled=FALSE;};
	    boolean     m_callbacks_enabled() { return(f_callbacks_enabled);};


    // Defines the callbacks

	    void        m_set_no_event_callback(void (*no_event_action)(const char *),
						const char *no_event_argument);
	    void        m_set_clipboard_changed_callback(void (*clipboard_changed_action)(const char *),
							 const char *clipboard_changed_argument);

  /*-----------------*/
  /* Private methods */
  /*-----------------*/

  private:

    // The application main loop

	    void        m_main_loop();

    // Closes the application

	    void        m_close();

    // Display routines

	    void        m_refresh_window(int open_window_number,int x1,int y1,int x2,int y2);

    // To add, remove, open or close a window

	    int         m_add_window(PWindow window);
	    void        m_del_window(int window_number);

	    void        m_open_window(int window_number,PObject owner);
	    void        m_close_window(int window_number);


    // To add a zone to refresh

	    void        m_add_zone_to_refresh(int x1,int y1,int x2,int y2,int window_height);

    // Misc routines

	    PWindow     m_number_to_window(int window_number);
	    PWindowNode m_number_to_window_node(int window_number);

	    int         m_height_to_window_nb(int window_height);
	    int         m_window_nb_to_height(int window_nb);

	    boolean     m_set_active_window(int window_number, boolean even_if_current_modal=FALSE);

    // Events

    virtual boolean     m_left_button_pressed_event(int x,int y);
    virtual boolean     m_left_button_double_click_event(int x,int y);
    virtual boolean     m_key_pressed_event(TKey key);

};

#endif
