-- BOOK.ADA   Ver. 3.00   22-AUG-1994   Copyright 1988-1994 John J. Herro
-- Software Innovations Technology
-- 1083 Mandarin Drive NE, Palm Bay, FL  32905-4706   (407)951-0233
--
-- This program creates two printable "book" files from the tutorial text in
-- ADA_TUTR.DAT.  This is by no means required for the course, but is provided
-- because several users have asked for it.  Be prepared to print about 500
-- pages!  For the format of the input data file, please see the preliminary
-- comments in ADA_TUTR.ADA.
--
with Direct_IO, Text_IO; use Text_IO;
procedure Book is
   subtype Block_Subtype is String(1 .. 64);
   package Random_IO is new Direct_IO(Block_Subtype);
   IxSize     : constant := 35;               -- Number of blocks in the index.
   Data_File  : Random_IO.File_Type;   -- The file from which screens are read.
   Prnt       : File_Type;                   -- The output file, to be printed.
   Block      : Block_Subtype;               -- Block read from the input file.
   Vpos       : Integer;                        -- Number of the current block.
   Hpos       : Integer;              -- Current position within current block.
   Highest_SN : Integer;                -- Highest screen number in the course.
   Middle_SN  : Integer;          -- Screen number when we change output files.
   Indx       : String(1 .. 64*IxSize);            -- Index from the Data File.
   Answer     : String(1 .. 80);                 -- User response to questions.
   Len        : Integer;                                   -- Length of Answer.
   File_OK    : Boolean := False;    -- True when data file opens successfully.
   Legal_Note : constant String := " Copyright 1988-94 John J. Herro ";
                       -- Legal_Note isn't used by the program, but it causes
                       -- most compilers to place this string in the .EXE file.
   procedure Open_Input_File is separate;
   procedure Open_Output_File(S: in String) is separate;
   procedure Print_Instructions is separate;
   procedure Print_Title_Page is separate;
   procedure Print_Screen(SN : in Integer) is separate;
begin
   Open_Input_File;
   if File_OK then
      Print_Instructions;
      Open_Output_File("FIRST");
      Print_Title_Page;
      Middle_SN := (101 + Highest_SN)/2;
      for SN in 101 .. Highest_SN loop
         if SN = Middle_SN then
            Close(Prnt);
            Open_Output_File("SECOND");
         end if;
         Print_Screen(SN);
      end loop;
      Put_Line("Both book files are created and ready for printing.");
      Random_IO.Close(Data_File);
      Close(Prnt);
   end if;
end Book;

separate (Book)
procedure Open_Input_File is
   Data_File_Name : constant String := "ADA_TUTR.DAT";
begin
   Random_IO.Open(Data_File, Random_IO.In_File, Data_File_Name);
   for I in 1 .. IxSize loop             -- Read index from start of Data File.
      Random_IO.Read(Data_File, Item => Block, From => Random_IO.Count(I));
      Indx(64*I - 63 .. 64*I) := Block;
   end loop;
   Highest_SN := Integer'Value(Indx(6 .. 8));
   File_OK := True;
exception
   when Random_IO.Name_Error =>
      Put("I'm sorry.  The file " & Data_File_Name & " seems to be missing.");
   when others =>
      Put("I'm sorry.  The file " & Data_File_Name);
      Put_Line(" seems to have the wrong form.");
end Open_Input_File;



separate (Book)
procedure Open_Output_File(S: in String) is
   OK : Boolean := False;                 -- True when file opens successfully.
begin
   Put_Line("Please type the name of the output file for the " & S & " half");
   Put("of the tutorial:  ");
   Get_Line(Answer, Len);
   while not OK loop
      begin
         Create(File => Prnt, Mode => Out_File, Name => Answer(1 .. Len));
         OK := True;
      exception
         when others => null;
      end;
      if not OK then
         Put_Line("Unable to create file.  Please retype name:  ");
         Get_Line(Answer, Len);
      end if;
   end loop;
   New_Line(2);
end Open_Output_File;

separate (Book)
procedure Print_Instructions is
begin
   Put_Line("This program creates two printable ""book"" files from the");
   Put_Line("tutorial text in ADA_TUTR.DAT.  This is by no means required");
   Put_Line("for the course, but is provided because several users have");
   Put_Line("asked for it.  Be prepared to print about 500 pages!");
   New_Line(2);
end Print_Instructions;



separate (Book)
procedure Print_Title_Page is
begin
   New_Page(Prnt);
   New_Line(Prnt);
   Put_Line(Prnt, "   AAA   DDDD    AAA          TTTTT  U   U  TTTTT  RRRR");
   Put_Line(Prnt, "  A   A  D   D  A   A           T    U   U    T    R   R");
   Put_Line(Prnt, "  AAAAA  D   D  AAAAA   ===     T    U   U    T    RRRR");
   Put_Line(Prnt, "  A   A  D   D  A   A           T    U   U    T    R  R");
   Put_Line(Prnt, "  A   A  DDDD   A   A           T     UUU     T    R   R");
   New_Line(Prnt);
   Put_Line(Prnt, "This is a copy of the tutorial text from ADA-TUTR, The");
   Put_Line(Prnt, "Interactive Ada Tutor, ver. 3.00.  BEGIN WITH SCREEN 104.");
   New_Line(Prnt);
   Put_Line(Prnt, "            Copyright 1988-1994 John J. Herro");
   New_Line(Prnt);
   Put_Line(Prnt, "You may copy this book, in printed or machine-readable");
   Put_Line(Prnt, "form, if you observe the Shareware notice in Screen 104.");
   Put_Line(Prnt, "Please distribute complete copies of the ADA-TUTR program");
   Put_Line(Prnt, "along with this book.  If you don't have a copy of");
   Put_Line(Prnt, "ADA-TUTR, send $10 for a trial copy or $30 for a");
   Put_Line(Prnt, "registered copy for full use by one individual.  Send $5");
   Put_Line(Prnt, "more if you prefer a 3.5"" diskette.");
   New_Line(Prnt);
   Put_Line(Prnt, "             Software Innovations Technology");
   Put_Line(Prnt, "                  1083 Mandarin Drive NE");
   Put_Line(Prnt, "                 Palm Bay, FL  32905-4706");
   New_Line(Prnt);
   Put_Line(Prnt, "                      (407) 951-0233");
   New_Page(Prnt);
   New_Line(Prnt);
end Print_Title_Page;

separate (Book)
procedure Print_Screen(SN : in Integer) is
   Expanding  : Boolean := False;       -- True when expanding multiple spaces.
   Prompting  : Boolean := False;      -- True for first character in a prompt.
   Bold       : Boolean := False;        -- True when text is being emphasized.
   Out1, Out2 : String(1 ..120) := (others => ' ');    -- Lines of output text.
   Place      : Integer := 1;         -- Current position within Out1 and Out2.
   Limit      : Integer;           -- Position of last non-space char. in Out2.
   Line_Num   : Integer := 1;                  -- Current line being displayed.
   Space      : constant String(1 .. 69) := (others => ' ');
   procedure Show(C : in Character) is separate;
   procedure Screen_Char is separate;
   procedure End_Of_Screen is separate;
begin
   if SN = 103 then
      Put(Prnt, Space(1 .. 27) & "*** X TAKES YOU HERE. ***");
      Put_Line(Prnt, Space(1 .. 17) & "Screen 103");
   else
      Put_Line(Prnt, Space & "Screen" & Integer'Image(SN));
   end if;
   New_Line(Prnt, 2);
   Vpos := 95*(Character'Pos(Indx(SN*4 - 394)) - 32) +        -- Point to start
               Character'Pos(Indx(SN*4 - 393)) - 32;          -- of current
   Hpos := Character'Pos(Indx(SN*4 - 392)) - 32;              -- screen.
   Random_IO.Read(Data_File, Item => Block, From => Random_IO.Count(Vpos));
   while Block(Hpos) /= '[' or Expanding loop     -- [ starts the control info.
      Screen_Char;
   end loop;
   End_Of_Screen;
end Print_Screen;

separate (Book.Print_Screen)
procedure Show(C : in Character) is
begin
   Out1(Place) := C;
   if Bold then
      Out2(Place) := '-';
   end if;
   Place := Place + 1;
end Show;



separate (Book.Print_Screen)
procedure Screen_Char is
   procedure Process_Char is separate;
begin
   if Expanding then
      for I in Integer range 1 .. Character'Pos(Block(Hpos)) - 32 loop
         Show(' ');
      end loop;
      Expanding := False;
   elsif Prompting then
      Prompting := False;
      if Block(Hpos) = 'b' then
         Put(Prnt, "Please type a space to go on, or B to go back.");
      elsif Block(Hpos) = 'q' then
         PUT(Prnt, "Please type a space to go on, or B or Q to go ");
         PUT(Prnt, "back to the question.");
      else
         Process_Char;
      end if;
   else
      Process_Char;
   end if;
   Hpos := Hpos + 1;
   if Hpos > Block'Length then
      Vpos := Vpos + 1;
      Hpos := 1;
      Random_IO.Read(Data_File, Block, From => Random_IO.Count(Vpos));
   end if;
end Screen_Char;

separate (Book.Print_Screen.Screen_Char)
procedure Process_Char is
begin
   case Block(Hpos) is
      when '{'    => Put_Line(Prnt, Out1(1 .. Place - 1));
                     Limit := Out2'Last;
                     while Limit > 0 and then Out2(Limit) = ' ' loop
                        Limit := Limit - 1;
                     end loop;
                     Put_Line(Prnt, Out2(1 .. Limit));
                     Out2 := (others => ' ');
                     Line_Num := Line_Num + 1;
                     Out1 := (others => ' ');
                     Place := 1;
      when '@'    => Expanding := True;                  -- @ = several spaces.
      when '^'    => Show(' ');                          -- ^ = bright + space.
                     Bold := True;
      when '~'    => Bold := False;                      -- ~ = normal + space.
                     Show(' ');
      when '%'    => Bold := True;                       -- % = bright.
      when '`'    => Bold := False;                      -- ` = normal.
      when '}'    => for I in Line_Num .. 23 loop        -- } = go to line 24.
                        New_Line(Prnt, 2);
                     end loop;
                     Prompting := True;
      when '\'    => Show(' ');                          -- \ = rev. vid. + sp.
      when '$'    => if SN = 103 then                    -- $ = screen #.
                        Show(' '); Show('_'); Show('_'); Show('_');
                     else
                        Show('$');
                     end if;
      when '#'    => if SN = 103 then                    -- # = % completed.
                        Show(' '); Show('_'); Show('_');
                     else
                        Show('#');
                     end if;
      when others => Show(Block(Hpos));
   end case;
end Process_Char;

separate (Book.Print_Screen)
procedure End_Of_Screen is
   Ctrl_Info : Block_Subtype;          -- Control info. for the current screen.
   I         : Integer;                     -- Used to index through Ctrl_Info.
begin
   Put_Line(Prnt, Out1(1 .. Place - 1));
   Limit := Out2'Last;
   while Limit > 0 and then Out2(Limit) = ' ' loop
      Limit := Limit - 1;
   end loop;
   Put_Line(Prnt, Out2(1 .. Limit));
   Place := 1;
   while Block(Hpos) /= ']' loop    -- Read control information from Data File.
      Hpos := Hpos + 1;
      if Hpos > Block'Length then
         Vpos := Vpos + 1;
         Hpos := 1;
         Random_IO.Read(Data_File, Block, From => Random_IO.Count(Vpos));
      end if;
      Ctrl_Info(Place) := Block(Hpos);
      Place := Place + 1;
   end loop;
   if Ctrl_Info(1 .. Place - 1) = "]" then
      Put_Line(Prnt, "(Program ends after this screen.)");
   elsif Ctrl_Info(1 .. Place - 1) = "#]" then
      Put_Line(Prnt, "(User types the next screen number.)");
   else
      I := 1;
      while I + 4 < Place loop
         Put(Prnt, "  '" & Ctrl_Info(I) & "' " & Ctrl_Info(I+1..I+3));
         I := I + 4;
         if I = 33 then
            New_Line(Prnt);
         end if;
      end loop;
      New_Line(Prnt);
   end if;
   New_Page(Prnt);
   New_Line(Prnt);
end End_Of_Screen;
