/*++

Module Name:

    ufsdvfs.c

Abstract:

    This module implements VFS antry points for
    UFSD-based Linux filesystem driver.

Author:

    Ahdrey Shedel

Revision History:

    27/12/2002 - Andrey Shedel - Created

--*/
// The necessary header files
// Standard in kernel modules
#include <linux/kernel.h>   // We're doing kernel work
#include <linux/module.h>   // Specifically, a module

#include <linux/blkdev.h>
#include <linux/locks.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/vmalloc.h>
#include <linux/nls.h>
#include <asm/uaccess.h>
#include <linux/smp_lock.h>


// Deal with CONFIG_MODVERSIONS
//#if CONFIG_MODVERSIONS==1
// #define MODVERSIONS
// #include <linux/modversions.h>
//#endif

#include "ufsdapi.h"

#define Dbg  DEBUG_TRACE_VFS

time_t UFSD_CurrentTime(void) {
	return CURRENT_TIME;
}


//
// Memory allocation routines.
//


DEBUG_ONLY(static size_t TotalAllocs);

PVOID UFSD_HeapAlloc (IN ULONG Size) {
	unsigned int* p;
	int Use_kmalloc;

	Use_kmalloc = Size <= 16380;

	if (Use_kmalloc) {
		Size =  (Size + 3U) & ~3U; // algn to next 4 bytes and keep mark clean.
		p = (unsigned int*)kmalloc(Size + sizeof(unsigned int), GFP_KERNEL);
	}
	else {
		Size = PAGE_ALIGN(Size + sizeof(unsigned int)) - sizeof(unsigned int);
		p = (unsigned int*)vmalloc(Size + sizeof(unsigned int));
	}

	ASSERT(NULL != p);
	if (NULL == p) {
		DebugTrace(0, DEBUG_TRACE_ERROR, ("UFSD_HeapAlloc(%ld) failed\n", Size));
		return NULL;
	}
	ASSERT(0 == (Size & 1U));
	*p = Size;
	DEBUG_ONLY(TotalAllocs += Size);
	if (!Use_kmalloc) {
		DebugTrace(0, DEBUG_TRACE_DEBUG_HOOKS, ("UFSD_HeapAlloc(%ld) vmalloc used\n", Size));
		*p |= 1U;
	}

	return p + 1;
}

void UFSD_HeapFree (IN PVOID Pointer) {
	if (NULL != Pointer) {
		unsigned int* p = ((unsigned int*)Pointer) - 1;
		DEBUG_ONLY(TotalAllocs -= (*p & ~1U));

		if (*p & 1U) {
			vfree(p);
		}
		else {
			kfree(p);
		}
	}
}

PVOID UFSD_HeapRealloc (IN PVOID Pointer, IN ULONG Size) {

	PVOID new_one;
	ULONG OldSize;

	if (NULL == Pointer) {
		return UFSD_HeapAlloc (Size);
	}

	OldSize = *(((unsigned int*)Pointer) - 1) & ~1U;

	if (OldSize >= Size) {
		return Pointer;
	}

	new_one = UFSD_HeapAlloc(Size);
	ASSERT(NULL != new_one);
	if (NULL == new_one) {
		return NULL;
	}

	memcpy(new_one, Pointer, OldSize);
	UFSD_HeapFree(Pointer);
	return new_one;
}




//
// Device IO functions.
//

#ifndef WIN32
BOOL
UFSD_BdRead (
    IN struct super_block* sb,
    IN ULONGLONG StartBlock,
    IN ULONG NrBlocks,
    OUT PVOID Buffer
   )
{
	struct buffer_head* bh;
	int Chunk;
	int ChunksPerBlock;
	ULONG ChunkSize = sb->s_blocksize;
	if (ChunkSize > PAGE_SIZE) {
		ChunkSize = PAGE_SIZE;
	}

	ChunksPerBlock = sb->s_blocksize / ChunkSize;
	//ASSERT(0 == (sb->s_blocksize % (ChunkSize)));

	while (NrBlocks--) {

		for (Chunk = 0; Chunk < ChunksPerBlock; Chunk++) {
			bh = bread(sb->s_dev, StartBlock * ChunksPerBlock + Chunk, ChunkSize);
			ASSERT(NULL != bh);
			if (NULL == bh)
				return FALSE;
			memcpy(Buffer, bh->b_data, ChunkSize);
			brelse(bh);
			Buffer = (char*)Buffer + ChunkSize;
		}

		++StartBlock;
	}

	return TRUE;
}


#if !UFSD_READONLY
BOOL
UFSD_BdWrite (
    IN struct super_block* sb,
    IN ULONGLONG StartBlock,
    IN ULONG NrBlocks,
    IN PCVOID Buffer
   )
{
	struct buffer_head* bh;
	int Chunk;
	int ChunksPerBlock;
	ULONG ChunkSize = sb->s_blocksize;
	if (ChunkSize > PAGE_SIZE) {
		ChunkSize = PAGE_SIZE;
	}

	ChunksPerBlock = sb->s_blocksize / ChunkSize;
	//ASSERT(0 == (sb->s_blocksize % (ChunkSize)));

	while (NrBlocks--) {

		for (Chunk = 0; Chunk < ChunksPerBlock; Chunk++) {

			bh = bread(sb->s_dev, StartBlock * ChunksPerBlock + Chunk, ChunkSize);
			ASSERT(NULL != bh);
			if (NULL == bh)
				return FALSE;
			memcpy(bh->b_data, Buffer, ChunkSize);
			mark_buffer_dirty(bh);

			if (sb->s_flags & MS_SYNCHRONOUS) {
				ll_rw_block (WRITE, 1, &bh); // TODO: do SG IO.
				wait_on_buffer (bh);
			}

			brelse(bh);
			Buffer = (char*)Buffer + ChunkSize;
		}

		++StartBlock;
	}

	return TRUE;
}
#endif //!UFSD_READONLY
#endif //WIN32

BOOL
UFSD_BdSetBlockSize (
    IN struct super_block* sb,
    IN ULONG NewBlockSize
   )
{
	int bits;
	DebugTrace(0, Dbg, ("UFSD_BdSetBlockSize(%p, %ld)\n", sb, NewBlockSize));
	set_blocksize(sb->s_dev, (NewBlockSize > PAGE_SIZE) ? PAGE_SIZE : NewBlockSize);
	sb->s_blocksize = NewBlockSize;
	for (bits = 9, NewBlockSize >>= 9; NewBlockSize >>= 1; bits++)
		;
	sb->s_blocksize_bits = bits;
	DebugTrace(0, Dbg, ("UFSD_BdSetBlockSize -> TRUE, bits=%d\n", sb->s_blocksize_bits));
	return TRUE;
}


ULONG
UFSD_BdGetBlockSize (
    IN struct super_block* sb
   )
{
	return sb->s_blocksize;
}

BOOL
UFSD_BdIsReadonly (
    IN struct super_block* sb
   )
{
	return (sb->s_flags & MS_RDONLY) ? TRUE : FALSE;
}

const char*
UFSD_BdGetName (
    IN struct super_block* sb
   )
{
	return bdevname(sb->s_dev);
}






//
// Mutex.
// Implementation is quire simpe because
// assumption about kernel been locked has been made.
//

typedef struct _UFSD_MUTEX {
    int Count;
} UFSD_MUTEX;


PUFSD_MUTEX
UFSD_MutexCreate (void)
{
    PUFSD_MUTEX Mutex = UFSD_HeapAlloc(sizeof(UFSD_MUTEX));
    ASSERT(NULL != Mutex);
    if (NULL == Mutex) {
        return NULL;
    }

    Mutex->Count = 0;
    return Mutex;
}

EXTERN_C
void
UFSD_MutexDelete (
    IN PUFSD_MUTEX Mutex
   )
{
    if (NULL == Mutex) {
        return;
    }

    ASSERT(0 == Mutex->Count);

    UFSD_HeapFree(Mutex);
    return;
}

EXTERN_C
int
UFSD_MutexEnter (
    IN PUFSD_MUTEX Mutex,
    IN BOOL Wait
   )
{
    ASSERT(kernel_locked());

    if (Wait) {

        while (Mutex->Count > 0) {
		    unlock_kernel();
    		schedule();
	    	lock_kernel();
        };

        ASSERT(0 == Mutex->Count);
    }


    Mutex->Count += 1;
	return Mutex->Count;
}


EXTERN_C
int
UFSD_MutexLeave (
    IN PUFSD_MUTEX Mutex
   )
{
    ASSERT(Mutex->Count > 0);
    Mutex->Count -= 1;
    return Mutex->Count;
}



//
// Actual VFS routines.
//
// First are macroses used to reference
// my sb and inode fields in general sb and inode structures
//

#define UFSD_SB(sb)   ((struct ufsd_sb_info*)&(sb)->u.generic_sbp)
#define UFSD_INODE(i) ((struct ufsd_inode_info*)&(i)->u.generic_ip)

#define UFSD_FH(i) UFSD_INODE(i)->FileHandle
#define UFSD_VOLUME(sb) UFSD_SB(sb)->VolumeHandle

// First and last reserved inode nr.
#define UFSD_ROOT_INO 2

// Supported mode flags bitmask.
#define UFSD_VALID_MODE (S_IFREG | S_IFDIR | S_IRWXU | S_IRWXG | S_IRWXO)


//
// Mount options structure.
// It will be filled from user' parameter string.
//

struct ufsd_mount_options {
	uid_t fs_uid;
	gid_t fs_gid;
	unsigned short fs_umask;
	unsigned short codepage;  /* Codepage for shortname conversions */
	unsigned
		quiet:1,         /* set = fake successful chmods and chowns */
		showexec:1,      /* set = only set x bit for com/exe/bat */
		sys_immutable:1, /* set = system files are immutable */
		dotsOK:1,        /* set = hidden and system files are named '.filename' */
		nocase:1;	     /* Does this need case conversion? 0=need case conversion*/
};


//
// Private superblock structure.
// It will be treated as a member of union
// in superblock structure.
//

struct ufsd_sb_info {
	PUFSD_VOLUME VolumeHandle;
	struct ufsd_mount_options options;
};


//
// Private inode structure.
// Will be the member of inode union.
//

struct ufsd_inode_info {
	PUFSD_FILE FileHandle;
};



//
// Parameter structure for
// iget4 call to be passed as 'opaque'.
//

struct ufsd_iget4_param {
	PUFSD_FILE parent;
	const char* name;
	unsigned name_len;
	int create_new;
	unsigned mode_for_create;
};







static
int
ufsd_readdir (
    IN struct file *filp,
    IN void *dirent,
    IN filldir_t filldir
   )
/*++

Routine Description:

    This routine is a callback used to fill readdir() buffer.

Arguments:

    filp - Directory pointer.
        'f_pos' member contains position to start scan from.

    dirent, filldir - data to be passed to
        'filldir()' helper

Return:

    int - zero on succedd.

--*/
{
	unsigned pos = (unsigned)filp->f_pos;
	unsigned search_anchor;
	unsigned current_pos;
	struct inode *inode = filp->f_dentry->d_inode;
	ino_t ino;
	int is_dir;
	struct qstr qname;
	char* Name;
	size_t NameLen;
	PUFSD_SEARCH DirScan = NULL;

	DebugTrace(+1, Dbg, ("readdir(%p, %d)\n", inode, (int)pos));

	ASSERT(kernel_locked());

	//
	// I can simulate EOF
	// if offset exceeds my predefined value.
	//

	if (pos >= 0x0ffffff0UL) {
		DebugTrace(-1, Dbg, ("readdir -> no more\n"));
		return 0;
	}


	//
	// In bounds.
	// Let's try...
	//

	__try {

		//
		// If offset is less then '2' or '1' then
		// I will add fake entries for self and parent.
		// I should do it because UFSD implementations differs
		// in implementing dot-dirs.
		//

		if (0 == pos) {

			if (filldir(dirent, ".", 1, pos, inode->i_ino, DT_DIR) < 0) {
				__leave;
			}

			++pos;
		}

		if (1 == pos) {

			ASSERT(NULL != filp->f_dentry->d_parent);
			ASSERT(NULL != filp->f_dentry->d_parent->d_inode);
			ASSERT(0 != filp->f_dentry->d_parent->d_inode->i_ino);

			if (filldir(dirent, "..", 2, pos,
						filp->f_dentry->d_parent->d_inode->i_ino,
						DT_DIR) < 0)
				__leave;

			++pos;
		}


		//
		// The rest requires
		// real search to be performed.
		//


		ASSERT(pos >= 2);
		search_anchor = pos - 2;

		if (filp->f_version != inode->i_version) {
			search_anchor = 0;
		}

		current_pos = search_anchor;

		if (UFSD_FindOpen(UFSD_VOLUME(inode->i_sb),
						  UFSD_FH(inode),
						  search_anchor,
						  &DirScan)) {
			__leave;
		}

		//
		// Real index as it is passed from here
		// will not contain fake dirs index.
		// 'Find next' will adjust the scan
		// on every next loop.
		//

		while (!UFSD_FindGet(DirScan, &search_anchor, &Name, &NameLen, &is_dir)) {

			ASSERT((int)search_anchor > 0);

			//
			// Skip already emulated dirs
			// if implementation is capable of returning them.
			//

			if (is_dir) {
				if (((1 == NameLen) && ('.' == Name[0])) ||
					((2 == NameLen) && ('.' == Name[0]) && ('.' == Name[1]))) {
					continue;
				}
			}


			//
			// Try to get existing inode nr from dcache.
			// (qname will be hashed inside api)
			//

			qname.name = Name;
			qname.len = NameLen;
			ino = find_inode_number(filp->f_dentry, &qname);

			//
			// Set it to something unique in order to
			// mark as valid.
			//

			if (0 == ino) {
				ino = iunique(inode->i_sb, UFSD_ROOT_INO);
			}

            pos = current_pos + 2;
			if (filldir(dirent, Name, NameLen, pos, ino,
						(is_dir ? DT_DIR : DT_REG))  < 0) {
				__leave;
			}

            current_pos = search_anchor;
		}

		//
		// Done.
		// At this point scan will be finished.
		//

		pos = 0x0ffffff0UL;

	}
	__finally {
		DebugUnwind(ufsd_readdir);


		if (NULL != DirScan) {
			UFSD_FindClose(DirScan);
		}

	}


	//
	// Set current state and
	// return to our caller.
	//

	filp->f_pos = pos;
	filp->f_version = inode->i_version;
	UPDATE_ATIME(inode);

	DebugTrace(-1, Dbg, ("readdir -> 0 (next=%d)\n", (int)pos));
	return 0;
}




static struct file_operations ufsd_dir_operations = {
	read:		generic_read_dir,
	readdir:	ufsd_readdir,
	fsync:		file_fsync,
};





static int ufsd_compare(struct dentry *de, struct qstr *name1, struct qstr *name2)
{
	// Custom compare used to support case-insensitive scan.
	// Should return zero on name match.

	DebugTrace(0, Dbg,
			   ("ufsd_compare(%p, '%.*s', '%.*s')\n",
				de,
				(int)name1->len, name1->name,
				(int)name2->len, name2->name));
	ASSERT(NULL != de->d_inode);

	if (UFSD_NamesEqual(UFSD_VOLUME(de->d_inode->i_sb),
						name1->name, name1->len,
						name2->name, name2->len)) {
		return 0;
	}

	return 1;
}


static struct dentry_operations ufsd_dop = {
	d_compare: ufsd_compare,
};






static
int
ufsd_create_or_open (
    IN struct inode *dir,
    IN struct dentry *dentry,
    IN int create,
    IN int mode_for_create
   )
/*++

Routine Description:

    This routine is a callback used to load or create inode for a
    direntry when this direntry was not found in dcache or direct
    request for create or mkdir is being served.

Arguments:

    dir - container inode for this operation.

    dentry - On entry contains name of the entry to find.
         On exit should contain inode loaded.

    create - non-zero for create operation.

    mode_for_create - name speaks for itself.

Return:

    int - errno.

--*/
{
	struct inode * inode;
	struct ufsd_iget4_param param;
	ino_t ino;
	int err;

	DebugTrace(+1, Dbg,
			   ("ufsd_create_or_open(%p, %p(%.*s), %s, %07o)\n",
				dir, dentry,
				(int)dentry->d_name.len, dentry->d_name.name,
				create ? "<create>" : "<open>", mode_for_create));
    ASSERT(kernel_locked());

	//
	// Simply load it
	// using generated inode nr and
	// add it to the cache.
	//

	ASSERT((NULL == dentry->d_op) || (dentry->d_op == &ufsd_dop));
	dentry->d_op = &ufsd_dop;

	param.parent = UFSD_FH(dir);
	param.name = dentry->d_name.name;
	param.name_len = dentry->d_name.len;
	param.create_new = create;
	param.mode_for_create = mode_for_create;
	ino = iunique(dir->i_sb, UFSD_ROOT_INO);
	inode = iget4(dir->i_sb, ino, NULL, &param);

	if (NULL == inode) {
		err = -ENOENT;
	}
	else if (IS_ERR(inode)) {
		err = PTR_ERR(inode);
		inode = NULL;
	}
	else if (NULL == UFSD_FH(inode)) {

		err = -ENOENT;
		iput(inode);
		inode = NULL;
	}
	else {

		err = 0;

		if (create) {

			if (S_ISDIR (inode->i_mode)) {
				dir->i_nlink++;
			}

			dir->i_mtime = dir->i_ctime = CURRENT_TIME;
			// Mark dir as requiring resync.
			dir->i_version = ++event;

			mark_inode_dirty(dir);
		}
	}

	d_instantiate(dentry, inode);


	DebugTrace(-1, Dbg, ("ufsd_create_or_open -> %d, inode=%p\n", err, inode));
	return err;
}


#if !UFSD_READONLY

// create/open use the same helper.
static int ufsd_create(struct inode * dir, struct dentry * dentry, int mode)
{
	DebugTrace(0, Dbg, ("ufsd_create(%.*s, %07o)\n",
						(int)dentry->d_name.len, dentry->d_name.name,
						mode));
	ASSERT(kernel_locked());
	return ufsd_create_or_open (dir, dentry, 1, mode);
}

static int ufsd_mkdir(struct inode * dir, struct dentry * dentry, int mode)
{
	DebugTrace(0, Dbg, ("ufsd_mkdir(%.*s, %07o)\n",
						(int)dentry->d_name.len, dentry->d_name.name,
						mode));
	ASSERT(kernel_locked());
	return ufsd_create_or_open (dir, dentry, 1, mode | S_IFDIR);
}



// unlink/rmdir helper.
static int ufsd_unlink(struct inode * dir, struct dentry *dentry)
{
	int error;
	struct inode* inode = dentry->d_inode;


	DebugTrace(+1, Dbg,
			   ("ufsd_unlink(%p, %p(%.*s))\n",
				dir, dentry, (int)dentry->d_name.len, dentry->d_name.name));

	ASSERT(kernel_locked());
	ASSERT(NULL != UFSD_FH(inode));

	// Should not leave dirty data!!!
	if (inode->i_data.nrpages)
		truncate_inode_pages(&inode->i_data, 0);

	error = UFSDAPI_FileClose(UFSD_VOLUME(inode->i_sb),
							  &UFSD_FH(inode),
							  1);

	if (NULL == UFSD_FH(inode)) {
		inode->i_nlink = 0;
	}

	if (0 != error) {
		DebugTrace(-1, Dbg, ("ufsd_unlink -> EACCES\n"));
		return -EACCES;
	}

	// Mark dir as requiring resync.
	dir->i_version = ++event;


	if (S_ISDIR(inode->i_mode)) {
		ASSERT(dir->i_nlink > 0);
		dir->i_nlink--;
		mark_inode_dirty(dir);
	}

	//make_bad_inode(inode);

	DebugTrace(-1, Dbg, ("ufsd_unlink -> %d\n", 0));
	return 0;
}



// Change notify.
static int ufsd_notify_change(struct dentry * dentry, struct iattr * attr)
{
	struct super_block *sb = dentry->d_sb;
	struct inode *inode = dentry->d_inode;
	int error;

	DebugTrace(+1, Dbg, ("ufsd_notify_change(%p, %p)\n", dentry, attr));

	/*
	// Check if too late.
	if (!d_unhashed(dentry)) {
	    DebugTrace(-1, Dbg, ("ufsd_notify_change -> EBUSY\n"));
        return -EBUSY;
	}
	*/


    //
	// Caller uses next code if this callback has not been implemented:
	//	error = inode_change_ok(inode, attr);
	//	if (!error) {
	//		if ((ia_valid & ATTR_UID && attr->ia_uid != inode->i_uid) ||
	//		    (ia_valid & ATTR_GID && attr->ia_gid != inode->i_gid))
	//			error = DQUOT_TRANSFER(inode, attr) ? -EDQUOT : 0;
	//		if (!error)
	//			error = inode_setattr(inode, attr);
	//	}
	// I will do the same here but will block permission change,
	// will not allow unsupported mode change
	// and perform actual attribute ser on my file object.
	//

	error = inode_change_ok(inode, attr);
	if (error) {
	    DebugTrace(-1, Dbg, ("ufsd_notify_change -> %d (!inode_change_ok(%p))\n", error, inode));
		return UFSD_SB(sb)->options.quiet ? 0 : error;
	}

	ASSERT(kernel_locked());

	if (((attr->ia_valid & ATTR_UID) &&
		 (attr->ia_uid != UFSD_SB(sb)->options.fs_uid)) ||
		((attr->ia_valid & ATTR_GID) &&
		 (attr->ia_gid != UFSD_SB(sb)->options.fs_gid)) ||
		((attr->ia_valid & ATTR_MODE) &&
		 (attr->ia_mode & ~UFSD_VALID_MODE)))
		error = -EPERM;

	if (error) {
		DebugTrace(-1, Dbg, ("ufsd_notify_change -> %d (not changeable)\n", error));
		return UFSD_SB(sb)->options.quiet ? 0 : error;
	}

	// Call handler.
	if (UFSDAPI_FileSetAttributes(UFSD_VOLUME(inode->i_sb),
								  UFSD_FH(inode),
								  (attr->ia_valid & ATTR_ATIME), attr->ia_atime,
								  (attr->ia_valid & ATTR_MTIME), attr->ia_mtime,
								  (attr->ia_valid & ATTR_CTIME), attr->ia_ctime,
								  (attr->ia_valid & ATTR_SIZE),  attr->ia_size,
								  (attr->ia_valid & ATTR_MODE),  (attr->ia_mode & S_IWUGO) ? 0 : 1
								 )) {
		error = -EPERM;
	}

	if (error) {
		DebugTrace(-1, Dbg, ("ufsd_notify_change -> %d (ufsd failed)\n", error));
		return UFSD_SB(sb)->options.quiet ? 0 : error;
	}

	error = inode_setattr(inode, attr);
	if (error) {
		DebugTrace(-1, Dbg, ("ufsd_notify_change -> %d (vfs failed)\n", error));
		return error;
	}

	// Always allow directory traversal.
	if (S_ISDIR(inode->i_mode))
		inode->i_mode |= S_IXUGO;

	// Adjust the mode.
	// Probably something smart happening here,
	// don't ask me, it's the same like in FAT
	inode->i_mode = ((inode->i_mode & S_IFMT) | ((((inode->i_mode & S_IRWXU
													& ~UFSD_SB(sb)->options.fs_umask) | S_IRUSR) >> 6)*S_IXUGO)) &
		~UFSD_SB(sb)->options.fs_umask;

	DebugTrace(-1, Dbg, ("ufsd_notify_change -> 0\n"));
	return 0;
}



static
int
ufsd_rename(struct inode *old_dir, struct dentry *old_dentry,
			struct inode *new_dir, struct dentry *new_dentry)
{
	int error = 0;
	DebugTrace(+1, Dbg,
			   ("ufsd_rename(%p, %p(%.*s), %p, %p(%.*s))\n",
				old_dir, old_dentry,
				(int)old_dentry->d_name.len, old_dentry->d_name.name,
				new_dir, new_dentry,
				(int)new_dentry->d_name.len, new_dentry->d_name.name));

	ASSERT(kernel_locked());

    //
    // If the target already exists, delete it first.
    // I will not unwind it on move failure. Although it's a weak point
    // it's better to not have it implemented then trying to create
    // a complex workaround.
    //

	if (NULL != new_dentry->d_inode) {

        DebugTrace(0, Dbg, ("ufsd_rename: deleting existing target %p\n",
                            new_dentry->d_inode));

        dget(new_dentry);
        error = ufsd_unlink(new_dir, new_dentry);

        if (!error)
			d_drop(new_dentry);
		dput(new_dentry);
        if (error) {
			DebugTrace(-1, Dbg, ("ufsd_rename -> failed to unlink target, %d\n", error));
            return error;
        }
    }


	// Simply call rename helper.
	if (UFSDAPI_FileMove(UFSD_VOLUME(old_dir->i_sb),
						 UFSD_FH(old_dentry->d_inode),
						 UFSD_FH(new_dir),
						 new_dentry->d_name.name,
						 new_dentry->d_name.len)) {
		error = -EPERM;
		DebugTrace(-1, Dbg, ("ufsd_rename -> failed, %d\n", error));
		return error;
	}


	// Mark dir as requiring resync.
	old_dir->i_version = ++event;
	old_dir->i_ctime = old_dir->i_mtime = CURRENT_TIME;
	mark_inode_dirty(old_dir);

	if (new_dir != old_dir) {

        new_dir->i_version = ++event;

		if (S_ISDIR(old_dentry->d_inode->i_mode)) {
			ASSERT(old_dir->i_nlink > 0);
            old_dir->i_nlink--;
            new_dir->i_nlink++;
        }
	}


	DebugTrace(-1, Dbg, ("ufsd_rename -> %d\n", error));
	return error;
}


#endif //!UFSD_READONLY


static
struct dentry*
ufsd_lookup (
    IN struct inode *dir,
    IN struct dentry *dentry
   )
/*++

Routine Description:

    This routine is a callback used to load inode for a
    direntry when this direntry was not found in dcache.

Arguments:

    dir - container inode for this operation.

    dentry - On entry contains name of the entry to find.
         On exit should contain inode loaded.

Return:

    struct dentry* - direntry in case of one differs from one
       passed to me. I return NULL to indicate original direntry has
       been used.
       ERRP() can also be returned to indicate error condition.

--*/
{
	int error;
	DebugTrace(0, Dbg, ("ufsd_lookup(%.*s)\n",
						(int)dentry->d_name.len, dentry->d_name.name));
	ASSERT(kernel_locked());

	error = ufsd_create_or_open (dir, dentry, 0, 0);

	// Must put it back to hash even if lookup failed.
	d_rehash(dentry);

	// ENOENT is expected and will be handled by the caller.
	return (error && (-ENOENT != error)) ? ERR_PTR(error) : NULL;
}





static struct inode_operations ufsd_dir_inode_operations = {
	lookup:  ufsd_lookup, // REQUIRED FOR DIR!!!
#if !UFSD_READONLY
	create:     ufsd_create,
	unlink:     ufsd_unlink,
	mkdir:      ufsd_mkdir,
	rmdir:      ufsd_unlink,
	rename:     ufsd_rename,
	setattr:    ufsd_notify_change,
#endif //!UFSD_READONLY
};




static
struct page*
ufsd_file_mmap_nopage(struct vm_area_struct *area,
					  unsigned long address, int write_access)
{
	struct file *file = area->vm_file;
	struct dentry *dentry = file->f_dentry;
	struct inode *inode = dentry->d_inode;
	struct page* page;
	char *pg_addr;
	unsigned int already_read;
	unsigned int count;
	int bufsize;
	int pos;

	page = alloc_page(GFP_HIGHUSER);
	if (!page)
		return page;
	pg_addr = kmap(page);
	address &= PAGE_MASK;
	pos = address - area->vm_start + (area->vm_pgoff << PAGE_SHIFT);

	count = PAGE_SIZE;
	if (address + PAGE_SIZE > area->vm_end) {
		count = area->vm_end - address;
	}

	/* what we can read in one go */
	bufsize = count; // may be adjusted.

	already_read = 0;
	while (already_read < count) {
		unsigned long  read_this_time;
		int to_read;

		to_read = bufsize - (pos % bufsize);

		to_read = min_t(unsigned int, to_read, count - already_read);

		lock_kernel();
		if (UFSDAPI_FileRead(UFSD_VOLUME(inode->i_sb), UFSD_FH(inode),
							 pos, to_read,
							 pg_addr + already_read,
							 &read_this_time)) {
			read_this_time = 0;
		}
		unlock_kernel();
		pos += read_this_time;
		already_read += read_this_time;

		if (read_this_time < to_read) {
			break;
		}
	}

	if (already_read < PAGE_SIZE)
		memset(pg_addr + already_read, 0, PAGE_SIZE - already_read);
	flush_dcache_page(page);
	kunmap(page);
	return page;
}

static struct vm_operations_struct ufsd_file_mmap =
{
	nopage: ufsd_file_mmap_nopage,
};


/* This is used for a general mmap of a ncp file */
static
int
ufsd_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct inode *inode = file->f_dentry->d_inode;

	if (NULL == UFSD_FH(inode)) {
		return -EIO;
	}
	/* only PAGE_COW or read-only supported now */
	if (vma->vm_flags & VM_SHARED)
		return -EINVAL;
	if (!inode->i_sb || !S_ISREG(inode->i_mode))
		return -EACCES;
	/* we do not support files bigger than 4GB... We eventually
	 supports just 4GB... */
	if (((vma->vm_end - vma->vm_start) >> PAGE_SHIFT) + vma->vm_pgoff
		> (1U << (32 - PAGE_SHIFT)))
		return -EFBIG;
	UPDATE_ATIME(inode);

	vma->vm_ops = &ufsd_file_mmap;
	return 0;
}


static
ssize_t
ufsd_file_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	struct dentry *dentry = file->f_dentry;
	struct inode *inode = dentry->d_inode;
	size_t already_read = 0;
	off_t pos;
	size_t bufsize;
	int error;
	void* bouncebuffer;

	DebugTrace(+1, Dbg, ("ufsd_file_read(%p(%p), %p, %d, %p(%d))\n",
						 file, inode, buf, count, ppos, (int)*ppos));

	error = -EINVAL;
	if (!S_ISREG(inode->i_mode)) {

		DebugTrace(0, DEBUG_TRACE_ERROR,
				   ("ufsd_file_read: read from non-file, mode %07o\n",
					inode->i_mode));
		goto out;
	}

	pos = *ppos;
	error = 0;
	if (!count)	/* size_t is never < 0 */
		goto out;

	bufsize = PAGE_SIZE * 2;
	bouncebuffer = vmalloc(bufsize);
	if (!bouncebuffer) {
		error = -EIO;	/* -ENOMEM */
		goto out;
	}

	/* First read in as much as possible for each bufsize. */
	while (already_read < count) {
		unsigned long  read_this_time;
		size_t to_read = min_t(unsigned int,
							   bufsize - (pos % bufsize),
							   count - already_read);

        lock_kernel();
		error = UFSDAPI_FileRead(UFSD_VOLUME(inode->i_sb),
								 UFSD_FH(inode),
								 pos, to_read, bouncebuffer,
								 &read_this_time);
        unlock_kernel();
		if (error) {
			error = -EIO;	/* Linux errno */
			break;
		}

		if (copy_to_user(buf, bouncebuffer, read_this_time)) {
			error = -EFAULT;
			break;
		}

		pos += read_this_time;
		buf += read_this_time;
		already_read += read_this_time;

		if (read_this_time != to_read) {
			break;
		}
	}

	vfree(bouncebuffer);

	*ppos = pos;

	UPDATE_ATIME(inode);

out:

	DebugTrace(-1, Dbg, ("ufsd_file_read -> \n"));
	return already_read ? already_read : error;
}

#if !UFSD_READONLY
static
ssize_t
ufsd_file_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	struct dentry *dentry = file->f_dentry;
	struct inode *inode = dentry->d_inode;
	size_t already_written = 0;
	off_t pos;
	size_t bufsize;
	int error;
	void* bouncebuffer;

	DebugTrace(+1, Dbg, ("ufsd_file_write(%p(%p), %p, %d, %p(%d))\n",
						 file, inode, buf, count, ppos, (int)*ppos));

	error = -EINVAL;
	if (!S_ISREG(inode->i_mode)) {

		DebugTrace(0, DEBUG_TRACE_ERROR,
				   ("ufsd_file_read: read from non-file, mode %07o\n",
					inode->i_mode));
		goto out;
	}


	error = 0;
	if (!count)
		goto out;
	pos = *ppos;

	if (file->f_flags & O_APPEND) {
		pos = inode->i_size;
	}
	bufsize = PAGE_SIZE * 2;

	bouncebuffer = vmalloc(bufsize);
	if (!bouncebuffer) {
		error = -EIO;	/* -ENOMEM */
		goto out;
	}

	while (already_written < count) {
		unsigned long  written_this_time;
		size_t to_write = min_t(unsigned int,
								bufsize - (pos % bufsize),
								count - already_written);

		if (copy_from_user(bouncebuffer, buf, to_write)) {
			error = -EFAULT;
			break;
		}

        lock_kernel();
		error = UFSDAPI_FileWrite(UFSD_VOLUME(inode->i_sb),
								  UFSD_FH(inode),
								  pos, to_write, bouncebuffer,
								  &written_this_time);
        unlock_kernel();
		if (error) {
			error = -EIO;
			break;
		}

		pos += written_this_time;
		buf += written_this_time;
		already_written += written_this_time;

        if (written_this_time != to_write) {
            if (0 == written_this_time) {
				error = -EIO;
            }
			break;
		}
	}
	vfree(bouncebuffer);
	inode->i_mtime = inode->i_atime = CURRENT_TIME;

	*ppos = pos;

	if (pos > inode->i_size) {
		inode->i_size = pos;
	}
out:
	DebugTrace(-1, Dbg, ("ufsd_file_write -> \n"));
	return already_written ? already_written : error;
}
#endif


static struct file_operations ufsd_file_operations = {
	llseek:		generic_file_llseek,
	read:		ufsd_file_read,
	mmap:		ufsd_mmap,
	fsync:		file_fsync,
#if !UFSD_READONLY
	write:		ufsd_file_write,
#endif //!UFSD_READONLY
};




static struct inode_operations ufsd_file_inode_operations = {
#if !UFSD_READONLY
	setattr:	ufsd_notify_change,
#endif //!UFSD_READONLY
};






// returns non-zero if this is executable file extension.
static int is_exec(const char* file_name) {

	static const char *exe_extensions = "EXECOMBAT";
	const char *walk;
	const char *extension;

	extension = strrchr(file_name, '.');
	if (!extension)
		return 0;

	for (walk = exe_extensions; *walk; walk += 3)
		if (!strncmp(extension, walk, 3))
			return 1;
	return 0;
}

// read_inode2() callback
// 'opaque' is passed from iget4()
static void ufsd_read_inode2 (struct inode * inode, void* p)
{
	struct ufsd_iget4_param* param = (struct ufsd_iget4_param*)p;
	unsigned Status;
	PUFSD_FILE fh = NULL;
	int is_dir, is_system, is_readonly;
	int subdir_count;
	unsigned long long size;
	struct ufsd_mount_options* options;

	DebugTrace(+1, Dbg, ("ufsd_read_inode(%p, %s, %s)\n", inode, param->name,
						 param->create_new ? "<create>" :  "<open>"));

    ASSERT(kernel_locked());

    if (NULL == param) {
		make_bad_inode(inode);
		DebugTrace(-1, Dbg, ("ufsd_read_inode -> void (no param)\n"));
		return;
	}


	//
	// Next members are set at this point:
	//
	// inode->i_sb = sb;
	// inode->i_dev = sb->s_dev;
	// inode->i_blkbits = sb->s_blocksize_bits;
	// inode->i_ino = ino;
	// inode->i_flags = 0;
	//
	// The rest to be set in this routine
	// follows the attempt to open the file.
	//

	Status = UFSDAPI_FileOpen(UFSD_VOLUME(inode->i_sb),
							  param->parent, param->name, param->name_len,
							  param->create_new, S_ISDIR (param->mode_for_create),
							  (0 == (param->mode_for_create & S_IWUGO)),
							  &fh,
							  &is_dir, &is_system, &is_readonly,
							  &subdir_count,
							  &size,
							  &inode->i_atime, &inode->i_ctime, &inode->i_mtime);
	if (ERR_NOERROR != Status) {
		make_bad_inode(inode);
		DebugTrace(-1, Dbg, ("ufsd_read_inode -> void (open failed)\n"));
		return;
	}

	options = &UFSD_SB(inode->i_sb)->options;
	inode->i_uid = options->fs_uid;
	inode->i_gid = options->fs_gid;

	inode->i_version = ++event;
	inode->i_mode = S_IRWXUGO & ~options->fs_umask;

	if (is_dir) {

		inode->i_nlink = subdir_count;
		inode->i_size = inode->i_sb->s_blocksize; // fake. Required to be non-zero.
		inode->i_op = &ufsd_dir_inode_operations;
		inode->i_fop = &ufsd_dir_operations;
	}
	else {

		inode->i_nlink = 1;
		inode->i_size = size;
		inode->i_op = &ufsd_file_inode_operations;
		inode->i_fop = &ufsd_file_operations;
		// no way to mmap ;(
		//inode->i_mapping->a_ops = &ufsd_aops;
	}

	if (is_readonly)
		inode->i_mode &= ~S_IWUGO;

	if (is_dir) {
		inode->i_mode |= S_IFDIR;
	}
	else {

		if (options->showexec && !is_exec(param->name)) {
			inode->i_mode &= ~S_IXUGO;
		}

		inode->i_mode |= S_IFREG;
	}


	if(is_system && options->sys_immutable) {
		inode->i_flags |= S_IMMUTABLE;
	}

	inode->i_blksize = 1 << inode->i_sb->s_blocksize_bits;
	inode->i_blocks = ((inode->i_size + inode->i_blksize - 1)
					   & ~(inode->i_blksize - 1)) >> 9;


	//
	// Succeeded.
	// Set 'handle' value in inode object.
	//

	UFSD_FH(inode) = fh;

	DebugTrace(-1, Dbg, ("ufsd_read_inode -> void\n"));
	return;
}





/*
// delete_inode callback is used to
// dereference inode after it has been released last time.
static void ufsd_delete_inode(struct inode * inode)
{
	DebugTrace(+1, Dbg, ("ufsd_delete_inode(%p)\n", inode));
    lock_kernel();
    clear_inode(inode);
    if (NULL != UFSD_FH(inode)) {
        UFSDAPI_FileClose(UFSD_VOLUME(inode->i_sb), &UFSD_FH(inode), 0);
	}
	unlock_kernel();
	DebugTrace(-1, Dbg, ("ufsd_delete_inode -> void\n"));
}
*/

// put_inode callback is used to
// dereference inode after it has been released
// The only purpose for it to be used is to force subsequent 'delete inode' call.
static void ufsd_put_inode (struct inode * inode)
{
	DebugTrace(+1, Dbg, ("ufsd_put_inode(%p)\n", inode));
    lock_kernel();
    if (NULL != UFSD_FH(inode)) {
		write_inode_now(inode, 1);
        if (inode->i_data.nrpages)
            truncate_inode_pages(&inode->i_data, 0);
        UFSDAPI_FileClose(UFSD_VOLUME(inode->i_sb), &UFSD_FH(inode), 0);
	}
	unlock_kernel();
	DebugTrace(-1, Dbg, ("ufsd_put_inode -> void\n"));
	return;
}






// Drop the volume handle.
static void ufsd_put_super (struct super_block * sb)
{

	DebugTrace(+1, Dbg,
               ("ufsd_put_super(%p (%s))\n", sb, bdevname(sb->s_dev)));
    ASSERT(kernel_locked());
	UFSDAPI_VolumeFree(UFSD_VOLUME(sb));
	DebugTrace(-1, Dbg, ("ufsd_put_super -> void\n"));
	return;
}



// stat callback.
static int ufsd_statfs (struct super_block * sb, struct statfs * buf)
{
	DebugTrace(+1, Dbg,
			   ("ufsd_statfs(%p (%s), %p)\n", sb, bdevname(sb->s_dev), buf));
	UFSDAPI_QueryVolumeInfo (UFSD_VOLUME(sb),
							 &buf->f_type,
							 &buf->f_blocks,
							 &buf->f_bfree,
							 &buf->f_namelen);
	buf->f_bsize = sb->s_blocksize;
	buf->f_bavail = buf->f_bfree;
	buf->f_files = 0;	/* UNKNOWN */
	buf->f_ffree = 0;	/* UNKNOWN */
	DebugTrace(-1, Dbg, ("ufsd_statfs -> void\n"));
	return 0;
}



// remount callback.
static int ufsd_remount (struct super_block * sb, int * flags, char * data)
{
	if ((*flags & MS_RDONLY) == (sb->s_flags & MS_RDONLY))
		return 0;

	if (*flags & MS_RDONLY) {
	}
    else {
#if UFSD_READONLY
        printk(KERN_WARNING QUOTED_UFSD_DEVICE": No write support. Marking filesystem read-only\n");
		return -EINVAL;
#else //UFSD_READONLY
		sb->s_flags &= ~MS_RDONLY;
#endif //UFSD_READONLY
	}

	return 0;
}






//
// Volume operations.
// Pointer to this structure will be set in superblock
// for our volume.
//


static struct super_operations ufsd_sops = {
	read_inode2:   ufsd_read_inode2,
    //delete_inode: ufsd_delete_inode,
    put_inode:    ufsd_put_inode,
	put_super:    ufsd_put_super,
	statfs:       ufsd_statfs,
	remount_fs:   ufsd_remount,
};



//
// Parse options.
// taken from FAT
//

static int ufsd_parse_options(char *options, struct ufsd_mount_options *opts)
{
	char *this_char,*value,save,*savep;
	int ret = 1;

	opts->fs_uid = current->uid;
	opts->fs_gid = current->gid;
	opts->fs_umask = current->fs->umask;
	opts->quiet = opts->sys_immutable = opts->dotsOK = opts->showexec = 0;
	opts->codepage = 0;
	opts->nocase = 0;

	if (NULL == options)
		return 1;

	save = 0;
	savep = NULL;
    for (this_char = strtok(options,",");
         this_char;
	     this_char = strtok(NULL,",")) {
		if ((value = strchr(this_char,'=')) != NULL) {
			save = *value;
			savep = value;
			*value++ = 0;
        }

		if (!strcmp(this_char,"dots")) {
            opts->dotsOK = 1;
            DebugTrace(0, Dbg, ("ufsd_parse_options: dots=ON\n"));
		}
		else if (!strcmp(this_char,"nocase")) {
			opts->nocase = 1;
            DebugTrace(0, Dbg, ("ufsd_parse_options: nocase=ON\n"));
		}
		else if (!strcmp(this_char,"nodots")) {
			opts->dotsOK = 0;
            DebugTrace(0, Dbg, ("ufsd_parse_options: dots=OFF\n"));
		}
		else if (!strcmp(this_char,"showexec")) {
			opts->showexec = 1;
            DebugTrace(0, Dbg, ("ufsd_parse_options: showexec=ON\n"));
		}
		else if (!strcmp(this_char,"dotsOK") && value) {
			if (!strcmp(value,"yes")) opts->dotsOK = 1;
			else if (!strcmp(value,"no")) opts->dotsOK = 0;
            else ret = 0;
            if (ret) {
				DebugTrace(0, Dbg, ("ufsd_parse_options: dotsOK=%d\n", opts->dotsOK));
            }
		}
		else if (!strcmp(this_char,"uid")) {
			if (!value || !*value) ret = 0;
			else {
				opts->fs_uid = simple_strtoul(value,&value,0);
				DebugTrace(0, Dbg, ("ufsd_parse_options: uid=%d\n", opts->fs_uid));
				if (*value) ret = 0;
			}
		}
		else if (!strcmp(this_char,"gid")) {
			if (!value || !*value) ret= 0;
			else {
				opts->fs_gid = simple_strtoul(value,&value,0);
				DebugTrace(0, Dbg, ("ufsd_parse_options: gid=%d\n", opts->fs_gid));
				if (*value) ret = 0;
			}
		}
		else if (!strcmp(this_char,"umask")) {
			if (!value || !*value) ret = 0;
			else {
				opts->fs_umask = simple_strtoul(value,&value,8);
				DebugTrace(0, Dbg, ("ufsd_parse_options: umask=%d\n", opts->fs_umask));
				if (*value) ret = 0;
			}
		}
		else if (!strcmp(this_char,"sys_immutable")) {
			if (value) ret = 0;
			else opts->sys_immutable = 1;
		}
		else if (!strcmp(this_char,"codepage") && value) {
			opts->codepage = simple_strtoul(value,&value,0);
   			DebugTrace(0, Dbg, ("ufsd_parse_options: codepage=%d\n", opts->codepage));
			if (*value) ret = 0;
		}

		if (this_char != options) *(this_char-1) = ',';
		if (value) *savep = save;
		if (ret == 0)
			break;
	}

	return ret;
}




static struct super_block*
ufsd_read_super (
    IN OUT struct super_block * sb,
    IN void * data,
    IN int silent
   )
/*++

Routine Description:

    This routine is a callback used to recognize and
    initialize superblock using this filesystem driver.

Arguments:

    sb - Superblock structure. On entry sb->s_dev is set to device,
         sb->s_flags contains requested mount mode.
         On exit this structure should have initialized root directory
         inode and superblock callbacks.

    data - mount options in a string form.

    silent - non-zero if no messages shoule be displayed.

Return:

    struct super_block* - 'sb' on success, NULL on failure.

--*/
{
	PUFSD_VOLUME Volume = NULL;
	int Status;
	int blocksize;
	struct ufsd_iget4_param param;

	DebugTrace(+1, Dbg,
			   ("ufsd_read_super(%p (%s), %s, %s)\n",
				sb, bdevname(sb->s_dev), (char*)data,
				silent ? "silent" : "verbose"));

	__try {

		//
		// Parse options.
		//

		if (!ufsd_parse_options((char*)data, &UFSD_SB(sb)->options)) {
			DebugTrace(0, DEBUG_TRACE_DEBUG_HOOKS,
					   ("ufsd_read_super: failed to mount %s, bad options '%s'\n",
						bdevname(sb->s_dev), (char*)data));
			__leave;
		}


		//
		// First of all let's play with block size a bit.
		// This will ensure block size will be set
		// in superblock structure.
		//

		blocksize = get_hardsect_size(sb->s_dev);
		if(blocksize < BLOCK_SIZE )
			blocksize = BLOCK_SIZE;

		UFSD_BdSetBlockSize(sb, blocksize);

		//
		// At the entry I have
		// 'dev' member of superblock set to device in question.
		// At exit in case of filesystem been
		// successfully recognized next members of superblock should be set:
		// 's_magic' - filesystem magic nr
		// 's_maxbytes' - maximal file size for this filesystem.
		//
		// So first of all let's create OUR block device
		// on kdev passed to me by the caller.
		//

		Status = UFSDAPI_VolumeMount(sb,
									 UFSD_SB(sb)->options.codepage,
									 UFSD_SB(sb)->options.nocase,
									 &Volume);

		if (ERR_NOERROR != Status) {

			printk(KERN_ERR QUOTED_UFSD_DEVICE": failed to mount %s\n", bdevname(sb->s_dev));
			DebugTrace(0, DEBUG_TRACE_DEBUG_HOOKS,
					   ("ufsd_read_super: failed to mount %s\n",
						bdevname(sb->s_dev)));
			sb = NULL;
			__leave;
		}

		if (UFSDAPI_VolumeIsReadonly(Volume) &&
			!UFSD_BdIsReadonly(sb) ) {

			printk(KERN_WARNING QUOTED_UFSD_DEVICE": No write support. Marking filesystem read-only\n");
			sb->s_flags |= MS_RDONLY;
		}


		//
		// At this point filesystem has been recognized.
		// Let's query for it's capabilities.
		//

		UFSDAPI_QueryVolumeInfo (Volume,
								 &sb->s_magic,
								 NULL, NULL, NULL);

		//
		// At this point I know enough to allocate
		// my root.
		//

		sb->s_op = &ufsd_sops;
		UFSD_VOLUME(sb) = Volume;

		param.parent = NULL;
		param.name = NULL;
        param.name_len = 0;
        param.create_new = 0;
		sb->s_root = d_alloc_root(iget4(sb, UFSD_ROOT_INO, NULL, &param));
		ASSERT(NULL != sb->s_root);

		if (NULL == sb->s_root) {

			printk(KERN_ERR QUOTED_UFSD_DEVICE": failed to open root on %s\n",
				   bdevname(sb->s_dev));
			DebugTrace(0, DEBUG_TRACE_DEBUG_HOOKS,
					   ("ufsd_read_super: failed to open root on %s\n",
						bdevname(sb->s_dev)));
			sb = NULL;
			__leave;
		}

		//
		// Done.
		// Fill in the output and dereference
		// unwind pointers.
		//

		Volume = NULL;

	}
	__finally {
		DebugUnwind(ufsd_read_super);

		if (NULL != Volume) {
			UFSDAPI_VolumeFree(Volume);
		}
	}



	DebugTrace(-1, Dbg, ("ufsd_read_super -> %p\n", sb));
	return sb;
}

static DECLARE_FSTYPE_DEV(ufsd_fs_type, QUOTED_UFSD_DEVICE, ufsd_read_super);

#ifndef UFSD_TEST
static
#endif // UFSD_TEST
int __init ufsd_init (void)
{
	DEBUG_ONLY(TotalAllocs = 0);
	printk(KERN_NOTICE QUOTED_UFSD_DEVICE": driver loaded\n");
	return register_filesystem(&ufsd_fs_type);
}

#ifndef UFSD_TEST
static
#endif // UFSD_TEST
void __exit ufsd_exit(void)
{
	unregister_filesystem(&ufsd_fs_type);
	printk(KERN_NOTICE QUOTED_UFSD_DEVICE": driver unloaded\n");
	ASSERT(0 == TotalAllocs);
	DEBUG_ONLY(if (0 != TotalAllocs) DebugTrace(-1, Dbg, ("leak:  %d bytes\n", TotalAllocs)));
}

#ifndef UFSD_TEST
MODULE_DESCRIPTION("Paragon " QUOTED_UFSD_DEVICE " driver");
MODULE_AUTHOR("Andrey Shedel");
MODULE_LICENSE("Commertial product");
EXPORT_NO_SYMBOLS;

module_init(ufsd_init);
module_exit(ufsd_exit);
#endif //UFSD_TEST
